﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ivs/IVS_EXPORTS.h>
#include <aws/ivs/model/DestinationConfiguration.h>
#include <aws/ivs/model/RecordingConfigurationState.h>
#include <aws/ivs/model/RenditionConfiguration.h>
#include <aws/ivs/model/ThumbnailConfiguration.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IVS {
namespace Model {

/**
 * <p>An object representing a configuration to record a channel
 * stream.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/RecordingConfiguration">AWS
 * API Reference</a></p>
 */
class RecordingConfiguration {
 public:
  AWS_IVS_API RecordingConfiguration() = default;
  AWS_IVS_API RecordingConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_IVS_API RecordingConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IVS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Recording-configuration ARN.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  RecordingConfiguration& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A complex type that contains information about where recorded video will be
   * stored.</p>
   */
  inline const DestinationConfiguration& GetDestinationConfiguration() const { return m_destinationConfiguration; }
  inline bool DestinationConfigurationHasBeenSet() const { return m_destinationConfigurationHasBeenSet; }
  template <typename DestinationConfigurationT = DestinationConfiguration>
  void SetDestinationConfiguration(DestinationConfigurationT&& value) {
    m_destinationConfigurationHasBeenSet = true;
    m_destinationConfiguration = std::forward<DestinationConfigurationT>(value);
  }
  template <typename DestinationConfigurationT = DestinationConfiguration>
  RecordingConfiguration& WithDestinationConfiguration(DestinationConfigurationT&& value) {
    SetDestinationConfiguration(std::forward<DestinationConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Recording-configuration name. The value does not need to be unique.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  RecordingConfiguration& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If a broadcast disconnects and then reconnects within the specified interval,
   * the multiple streams will be considered a single broadcast and merged together.
   * Default: 0.</p>
   */
  inline int GetRecordingReconnectWindowSeconds() const { return m_recordingReconnectWindowSeconds; }
  inline bool RecordingReconnectWindowSecondsHasBeenSet() const { return m_recordingReconnectWindowSecondsHasBeenSet; }
  inline void SetRecordingReconnectWindowSeconds(int value) {
    m_recordingReconnectWindowSecondsHasBeenSet = true;
    m_recordingReconnectWindowSeconds = value;
  }
  inline RecordingConfiguration& WithRecordingReconnectWindowSeconds(int value) {
    SetRecordingReconnectWindowSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Object that describes which renditions should be recorded for a stream.</p>
   */
  inline const RenditionConfiguration& GetRenditionConfiguration() const { return m_renditionConfiguration; }
  inline bool RenditionConfigurationHasBeenSet() const { return m_renditionConfigurationHasBeenSet; }
  template <typename RenditionConfigurationT = RenditionConfiguration>
  void SetRenditionConfiguration(RenditionConfigurationT&& value) {
    m_renditionConfigurationHasBeenSet = true;
    m_renditionConfiguration = std::forward<RenditionConfigurationT>(value);
  }
  template <typename RenditionConfigurationT = RenditionConfiguration>
  RecordingConfiguration& WithRenditionConfiguration(RenditionConfigurationT&& value) {
    SetRenditionConfiguration(std::forward<RenditionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the current state of the recording configuration. When the state is
   * <code>ACTIVE</code>, the configuration is ready for recording a channel
   * stream.</p>
   */
  inline RecordingConfigurationState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(RecordingConfigurationState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline RecordingConfiguration& WithState(RecordingConfigurationState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags attached to the resource. Array of 1-50 maps, each of the form
   * <code>string:string (key:value)</code>. See <a
   * href="https://docs.aws.amazon.com/tag-editor/latest/userguide/best-practices-and-strats.html">Best
   * practices and strategies</a> in <i>Tagging Amazon Web Services Resources and Tag
   * Editor</i> for details, including restrictions that apply to tags and "Tag
   * naming limits and requirements"; Amazon IVS has no service-specific constraints
   * beyond what is documented there.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  RecordingConfiguration& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  RecordingConfiguration& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A complex type that allows you to enable/disable the recording of thumbnails
   * for a live session and modify the interval at which thumbnails are generated for
   * the live session.</p>
   */
  inline const ThumbnailConfiguration& GetThumbnailConfiguration() const { return m_thumbnailConfiguration; }
  inline bool ThumbnailConfigurationHasBeenSet() const { return m_thumbnailConfigurationHasBeenSet; }
  template <typename ThumbnailConfigurationT = ThumbnailConfiguration>
  void SetThumbnailConfiguration(ThumbnailConfigurationT&& value) {
    m_thumbnailConfigurationHasBeenSet = true;
    m_thumbnailConfiguration = std::forward<ThumbnailConfigurationT>(value);
  }
  template <typename ThumbnailConfigurationT = ThumbnailConfiguration>
  RecordingConfiguration& WithThumbnailConfiguration(ThumbnailConfigurationT&& value) {
    SetThumbnailConfiguration(std::forward<ThumbnailConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  DestinationConfiguration m_destinationConfiguration;
  bool m_destinationConfigurationHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  int m_recordingReconnectWindowSeconds{0};
  bool m_recordingReconnectWindowSecondsHasBeenSet = false;

  RenditionConfiguration m_renditionConfiguration;
  bool m_renditionConfigurationHasBeenSet = false;

  RecordingConfigurationState m_state{RecordingConfigurationState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  ThumbnailConfiguration m_thumbnailConfiguration;
  bool m_thumbnailConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace IVS
}  // namespace Aws
