﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/guardduty/GuardDutyRequest.h>
#include <aws/guardduty/GuardDuty_EXPORTS.h>
#include <aws/guardduty/model/StartMalwareScanConfiguration.h>

#include <utility>

namespace Aws {
namespace GuardDuty {
namespace Model {

/**
 */
class StartMalwareScanRequest : public GuardDutyRequest {
 public:
  AWS_GUARDDUTY_API StartMalwareScanRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartMalwareScan"; }

  AWS_GUARDDUTY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the resource for which you invoked the API.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  StartMalwareScanRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The idempotency token for the create request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartMalwareScanRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains information about the configuration to be used for the malware
   * scan.</p>
   */
  inline const StartMalwareScanConfiguration& GetScanConfiguration() const { return m_scanConfiguration; }
  inline bool ScanConfigurationHasBeenSet() const { return m_scanConfigurationHasBeenSet; }
  template <typename ScanConfigurationT = StartMalwareScanConfiguration>
  void SetScanConfiguration(ScanConfigurationT&& value) {
    m_scanConfigurationHasBeenSet = true;
    m_scanConfiguration = std::forward<ScanConfigurationT>(value);
  }
  template <typename ScanConfigurationT = StartMalwareScanConfiguration>
  StartMalwareScanRequest& WithScanConfiguration(ScanConfigurationT&& value) {
    SetScanConfiguration(std::forward<ScanConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceArn;
  bool m_resourceArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  StartMalwareScanConfiguration m_scanConfiguration;
  bool m_scanConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace GuardDuty
}  // namespace Aws
