﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancing/ElasticLoadBalancingRequest.h>
#include <aws/elasticloadbalancing/ElasticLoadBalancing_EXPORTS.h>
#include <aws/elasticloadbalancing/model/Listener.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancing {
namespace Model {

/**
 * <p>Contains the parameters for CreateLoadBalancerListeners.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticloadbalancing-2012-06-01/CreateLoadBalancerListenerInput">AWS
 * API Reference</a></p>
 */
class CreateLoadBalancerListenersRequest : public ElasticLoadBalancingRequest {
 public:
  AWS_ELASTICLOADBALANCING_API CreateLoadBalancerListenersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateLoadBalancerListeners"; }

  AWS_ELASTICLOADBALANCING_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCING_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the load balancer.</p>
   */
  inline const Aws::String& GetLoadBalancerName() const { return m_loadBalancerName; }
  inline bool LoadBalancerNameHasBeenSet() const { return m_loadBalancerNameHasBeenSet; }
  template <typename LoadBalancerNameT = Aws::String>
  void SetLoadBalancerName(LoadBalancerNameT&& value) {
    m_loadBalancerNameHasBeenSet = true;
    m_loadBalancerName = std::forward<LoadBalancerNameT>(value);
  }
  template <typename LoadBalancerNameT = Aws::String>
  CreateLoadBalancerListenersRequest& WithLoadBalancerName(LoadBalancerNameT&& value) {
    SetLoadBalancerName(std::forward<LoadBalancerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The listeners.</p>
   */
  inline const Aws::Vector<Listener>& GetListeners() const { return m_listeners; }
  inline bool ListenersHasBeenSet() const { return m_listenersHasBeenSet; }
  template <typename ListenersT = Aws::Vector<Listener>>
  void SetListeners(ListenersT&& value) {
    m_listenersHasBeenSet = true;
    m_listeners = std::forward<ListenersT>(value);
  }
  template <typename ListenersT = Aws::Vector<Listener>>
  CreateLoadBalancerListenersRequest& WithListeners(ListenersT&& value) {
    SetListeners(std::forward<ListenersT>(value));
    return *this;
  }
  template <typename ListenersT = Listener>
  CreateLoadBalancerListenersRequest& AddListeners(ListenersT&& value) {
    m_listenersHasBeenSet = true;
    m_listeners.emplace_back(std::forward<ListenersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_loadBalancerName;
  bool m_loadBalancerNameHasBeenSet = false;

  Aws::Vector<Listener> m_listeners;
  bool m_listenersHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancing
}  // namespace Aws
