/*
 * asmz80.c - Z80 Assembler-related utility functions.
 *
 * Written by
 *  Andreas Boose <viceteam@t-online.de>
 *  groepaz <groepaz@gmx.net>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include <stdlib.h>

#include "asm.h"
#include "archdep_exit.h"
#include "log.h"
#include "mon_assemble.h"
#include "mon_register.h"
#include "montypes.h"
#include "types.h"

/* NOTE: the value from this table is added to the total opcode size, so 0 may
         be valid, is this intended? */
static const int addr_mode_size[ASM_ADDR_MODE_LAST] = {
    /* (reused from) 6502 */
    [ASM_ADDR_MODE_IMPLIED] = 1,
    [ASM_ADDR_MODE_ACCUMULATOR] = 1,
    [ASM_ADDR_MODE_IMMEDIATE] = 2,
    [ASM_ADDR_MODE_ZERO_PAGE] = 2,        /* INA $12 */
    [ASM_ADDR_MODE_ABSOLUTE] = 3,
    [ASM_ADDR_MODE_ABS_INDIRECT] = 3,     /* LD IX, ($1234) */
    [ASM_ADDR_MODE_RELATIVE] = 2,
    /* z80 */
    [ASM_ADDR_MODE_ABSOLUTE_A] = 3,
    [ASM_ADDR_MODE_ABSOLUTE_HL] = 3,      /* LD ($1234), HL */
    [ASM_ADDR_MODE_ABSOLUTE_IX] = 3,
    [ASM_ADDR_MODE_ABSOLUTE_IY] = 3,
    [ASM_ADDR_MODE_Z80_ABSOLUTE_BC] = 3,
    [ASM_ADDR_MODE_Z80_ABSOLUTE_DE] = 3,
    [ASM_ADDR_MODE_Z80_ABSOLUTE_SP] = 3,
    [ASM_ADDR_MODE_ABS_INDIRECT_ZP] = 2,
    [ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT] = 3,
    [ASM_ADDR_MODE_IMMEDIATE_16] = 3,
    [ASM_ADDR_MODE_REG_B] = 1,
    [ASM_ADDR_MODE_REG_C] = 1,
    [ASM_ADDR_MODE_REG_D] = 1,
    [ASM_ADDR_MODE_REG_E] = 1,
    [ASM_ADDR_MODE_REG_H] = 1,
    [ASM_ADDR_MODE_REG_IXH] = 1,
    [ASM_ADDR_MODE_REG_IYH] = 1,
    [ASM_ADDR_MODE_REG_L] = 1,
    [ASM_ADDR_MODE_REG_IXL] = 1,
    [ASM_ADDR_MODE_REG_IYL] = 1,
    [ASM_ADDR_MODE_REG_AF] = 1,
    [ASM_ADDR_MODE_REG_BC] = 1,
    [ASM_ADDR_MODE_REG_DE] = 1,
    [ASM_ADDR_MODE_REG_HL] = 1,
    [ASM_ADDR_MODE_REG_IX] = 1,
    [ASM_ADDR_MODE_REG_IY] = 1,
    [ASM_ADDR_MODE_REG_SP] = 1,
    [ASM_ADDR_MODE_REG_IND_BC] = 1,
    [ASM_ADDR_MODE_REG_IND_DE] = 1,
    [ASM_ADDR_MODE_REG_IND_HL] = 1,
    [ASM_ADDR_MODE_REG_IND_IX] = 2,
    [ASM_ADDR_MODE_REG_IND_IY] = 2,
    [ASM_ADDR_MODE_REG_IND_SP] = 1,
    [ASM_ADDR_MODE_Z80_IND_IMMEDIATE] = 3,    /* LD (IX+$12), #$34 */
    [ASM_ADDR_MODE_Z80_IND_REG] = 2,
    [ASM_ADDR_MODE_IND_IX_REG] = 2,
    [ASM_ADDR_MODE_IND_IY_REG] = 2,
    /* CAUTION: when adding z80 modes also update: asm6809.c, asm65816.c, asmR6502.c */
};

/* Z80 instruction set table: https://clrhome.org/table/ */

static const asm_opcode_info_t opcode_list[] = {
    /* 00 */ { "NOP",        ASM_ADDR_MODE_IMPLIED },
    /* 01 */ { "LD BC,",     ASM_ADDR_MODE_IMMEDIATE_16 },
    /* 02 */ { "LD (BC),A",  ASM_ADDR_MODE_IMPLIED },
    /* 03 */ { "INC",        ASM_ADDR_MODE_REG_BC },
    /* 04 */ { "INC",        ASM_ADDR_MODE_REG_B },
    /* 05 */ { "DEC",        ASM_ADDR_MODE_REG_B },
    /* 06 */ { "LD B,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 07 */ { "RLCA",       ASM_ADDR_MODE_IMPLIED },
    /* 08 */ { "EX AF,AF'",  ASM_ADDR_MODE_IMPLIED },
    /* 09 */ { "ADD HL,BC",  ASM_ADDR_MODE_IMPLIED },
    /* 0a */ { "LD A,(BC)",  ASM_ADDR_MODE_IMPLIED },
    /* 0b */ { "DEC",        ASM_ADDR_MODE_REG_BC },
    /* 0c */ { "INC",        ASM_ADDR_MODE_REG_C },
    /* 0d */ { "DEC",        ASM_ADDR_MODE_REG_C },
    /* 0e */ { "LD C,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 0f */ { "RRCA",       ASM_ADDR_MODE_IMPLIED },
    /* 10 */ { "DJNZ",       ASM_ADDR_MODE_RELATIVE },
    /* 11 */ { "LD DE,",     ASM_ADDR_MODE_IMMEDIATE_16 },
    /* 12 */ { "LD (DE),A",  ASM_ADDR_MODE_IMPLIED },
    /* 13 */ { "INC",        ASM_ADDR_MODE_REG_DE },
    /* 14 */ { "INC",        ASM_ADDR_MODE_REG_D },
    /* 15 */ { "DEC",        ASM_ADDR_MODE_REG_D },
    /* 16 */ { "LD D,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 17 */ { "RLA",        ASM_ADDR_MODE_IMPLIED },
    /* 18 */ { "JR",         ASM_ADDR_MODE_RELATIVE },
    /* 19 */ { "ADD HL,DE",  ASM_ADDR_MODE_IMPLIED },
    /* 1a */ { "LD A,(DE)",  ASM_ADDR_MODE_IMPLIED },
    /* 1b */ { "DEC",        ASM_ADDR_MODE_REG_DE },
    /* 1c */ { "INC",        ASM_ADDR_MODE_REG_E },
    /* 1d */ { "DEC",        ASM_ADDR_MODE_REG_E },
    /* 1e */ { "LD E,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 1f */ { "RRA",        ASM_ADDR_MODE_IMPLIED },
    /* 20 */ { "JR NZ,",     ASM_ADDR_MODE_RELATIVE },
    /* 21 */ { "LD HL,",     ASM_ADDR_MODE_IMMEDIATE_16 },
    /* 22 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_HL },
    /* 23 */ { "INC",        ASM_ADDR_MODE_REG_HL },
    /* 24 */ { "INC",        ASM_ADDR_MODE_REG_H },
    /* 25 */ { "DEC",        ASM_ADDR_MODE_REG_H },
    /* 26 */ { "LD H,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 27 */ { "DAA",        ASM_ADDR_MODE_IMPLIED },
    /* 28 */ { "JR Z,",      ASM_ADDR_MODE_RELATIVE },
    /* 29 */ { "ADD HL,HL",  ASM_ADDR_MODE_IMPLIED },
    /* 2a */ { "LD HL,",     ASM_ADDR_MODE_ABS_INDIRECT },
    /* 2b */ { "DEC",        ASM_ADDR_MODE_REG_HL },
    /* 2c */ { "INC",        ASM_ADDR_MODE_REG_L },
    /* 2d */ { "DEC",        ASM_ADDR_MODE_REG_L },
    /* 2e */ { "LD L,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 2f */ { "CPL",        ASM_ADDR_MODE_IMPLIED },
    /* 30 */ { "JR NC,",     ASM_ADDR_MODE_RELATIVE },
    /* 31 */ { "LD SP,",     ASM_ADDR_MODE_IMMEDIATE_16 },
    /* 32 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_A },
    /* 33 */ { "INC",        ASM_ADDR_MODE_REG_SP },
    /* 34 */ { "INC",        ASM_ADDR_MODE_REG_IND_HL },
    /* 35 */ { "DEC",        ASM_ADDR_MODE_REG_IND_HL },
    /* 36 */ { "LD (HL),",   ASM_ADDR_MODE_IMMEDIATE },
    /* 37 */ { "SCF",        ASM_ADDR_MODE_IMPLIED },
    /* 38 */ { "JR C,",      ASM_ADDR_MODE_RELATIVE },
    /* 39 */ { "ADD HL,SP",  ASM_ADDR_MODE_IMPLIED },
    /* 3a */ { "LD A,",      ASM_ADDR_MODE_ABS_INDIRECT },
    /* 3b */ { "DEC",        ASM_ADDR_MODE_REG_SP },
    /* 3c */ { "INC",        ASM_ADDR_MODE_ACCUMULATOR },
    /* 3d */ { "DEC",        ASM_ADDR_MODE_ACCUMULATOR },
    /* 3e */ { "LD A,",      ASM_ADDR_MODE_IMMEDIATE },
    /* 3f */ { "CCF",        ASM_ADDR_MODE_IMPLIED },
    /* 40 */ { "LD B,B",     ASM_ADDR_MODE_IMPLIED },
    /* 41 */ { "LD B,C",     ASM_ADDR_MODE_IMPLIED },
    /* 42 */ { "LD B,D",     ASM_ADDR_MODE_IMPLIED },
    /* 43 */ { "LD B,E",     ASM_ADDR_MODE_IMPLIED },
    /* 44 */ { "LD B,H",     ASM_ADDR_MODE_IMPLIED },
    /* 45 */ { "LD B,L",     ASM_ADDR_MODE_IMPLIED },
    /* 46 */ { "LD B,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 47 */ { "LD B,A",     ASM_ADDR_MODE_IMPLIED },
    /* 48 */ { "LD C,B",     ASM_ADDR_MODE_IMPLIED },
    /* 49 */ { "LD C,C",     ASM_ADDR_MODE_IMPLIED },
    /* 4a */ { "LD C,D",     ASM_ADDR_MODE_IMPLIED },
    /* 4b */ { "LD C,E",     ASM_ADDR_MODE_IMPLIED },
    /* 4c */ { "LD C,H",     ASM_ADDR_MODE_IMPLIED },
    /* 4d */ { "LD C,L",     ASM_ADDR_MODE_IMPLIED },
    /* 4e */ { "LD C,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 4f */ { "LD C,A",     ASM_ADDR_MODE_IMPLIED },
    /* 50 */ { "LD D,B",     ASM_ADDR_MODE_IMPLIED },
    /* 51 */ { "LD D,C",     ASM_ADDR_MODE_IMPLIED },
    /* 52 */ { "LD D,D",     ASM_ADDR_MODE_IMPLIED },
    /* 53 */ { "LD D,E",     ASM_ADDR_MODE_IMPLIED },
    /* 54 */ { "LD D,H",     ASM_ADDR_MODE_IMPLIED },
    /* 55 */ { "LD D,L",     ASM_ADDR_MODE_IMPLIED },
    /* 56 */ { "LD D,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 57 */ { "LD D,A",     ASM_ADDR_MODE_IMPLIED },
    /* 58 */ { "LD E,B",     ASM_ADDR_MODE_IMPLIED },
    /* 59 */ { "LD E,C",     ASM_ADDR_MODE_IMPLIED },
    /* 5a */ { "LD E,D",     ASM_ADDR_MODE_IMPLIED },
    /* 5b */ { "LD E,E",     ASM_ADDR_MODE_IMPLIED },
    /* 5c */ { "LD E,H",     ASM_ADDR_MODE_IMPLIED },
    /* 5d */ { "LD E,L",     ASM_ADDR_MODE_IMPLIED },
    /* 5e */ { "LD E,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 5f */ { "LD E,A",     ASM_ADDR_MODE_IMPLIED },
    /* 60 */ { "LD H,B",     ASM_ADDR_MODE_IMPLIED },
    /* 61 */ { "LD H,C",     ASM_ADDR_MODE_IMPLIED },
    /* 62 */ { "LD H,D",     ASM_ADDR_MODE_IMPLIED },
    /* 63 */ { "LD H,E",     ASM_ADDR_MODE_IMPLIED },
    /* 64 */ { "LD H,H",     ASM_ADDR_MODE_IMPLIED },
    /* 65 */ { "LD H,L",     ASM_ADDR_MODE_IMPLIED },
    /* 66 */ { "LD H,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 67 */ { "LD H,A",     ASM_ADDR_MODE_IMPLIED },
    /* 68 */ { "LD L,B",     ASM_ADDR_MODE_IMPLIED },
    /* 69 */ { "LD L,C",     ASM_ADDR_MODE_IMPLIED },
    /* 6a */ { "LD L,D",     ASM_ADDR_MODE_IMPLIED },
    /* 6b */ { "LD L,E",     ASM_ADDR_MODE_IMPLIED },
    /* 6c */ { "LD L,H",     ASM_ADDR_MODE_IMPLIED },
    /* 6d */ { "LD L,L",     ASM_ADDR_MODE_IMPLIED },
    /* 6e */ { "LD L,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 6f */ { "LD L,A",     ASM_ADDR_MODE_IMPLIED },
    /* 70 */ { "LD (HL),B",  ASM_ADDR_MODE_IMPLIED },
    /* 71 */ { "LD (HL),C",  ASM_ADDR_MODE_IMPLIED },
    /* 72 */ { "LD (HL),D",  ASM_ADDR_MODE_IMPLIED },
    /* 73 */ { "LD (HL),E",  ASM_ADDR_MODE_IMPLIED },
    /* 74 */ { "LD (HL),H",  ASM_ADDR_MODE_IMPLIED },
    /* 75 */ { "LD (HL),L",  ASM_ADDR_MODE_IMPLIED },
    /* 76 */ { "HALT",       ASM_ADDR_MODE_IMPLIED },
    /* 77 */ { "LD (HL),A",  ASM_ADDR_MODE_IMPLIED },
    /* 78 */ { "LD A,B",     ASM_ADDR_MODE_IMPLIED },
    /* 79 */ { "LD A,C",     ASM_ADDR_MODE_IMPLIED },
    /* 7a */ { "LD A,D",     ASM_ADDR_MODE_IMPLIED },
    /* 7b */ { "LD A,E",     ASM_ADDR_MODE_IMPLIED },
    /* 7c */ { "LD A,H",     ASM_ADDR_MODE_IMPLIED },
    /* 7d */ { "LD A,L",     ASM_ADDR_MODE_IMPLIED },
    /* 7e */ { "LD A,(HL)",  ASM_ADDR_MODE_IMPLIED },
    /* 7f */ { "LD A,A",     ASM_ADDR_MODE_IMPLIED },
    /* 80 */ { "ADD",        ASM_ADDR_MODE_REG_B },
    /* 81 */ { "ADD",        ASM_ADDR_MODE_REG_C },
    /* 82 */ { "ADD",        ASM_ADDR_MODE_REG_D },
    /* 83 */ { "ADD",        ASM_ADDR_MODE_REG_E },
    /* 84 */ { "ADD",        ASM_ADDR_MODE_REG_H },
    /* 85 */ { "ADD",        ASM_ADDR_MODE_REG_L },
    /* 86 */ { "ADD",        ASM_ADDR_MODE_REG_IND_HL },
    /* 87 */ { "ADD",        ASM_ADDR_MODE_ACCUMULATOR },
    /* 88 */ { "ADC",        ASM_ADDR_MODE_REG_B },
    /* 89 */ { "ADC",        ASM_ADDR_MODE_REG_C },
    /* 8a */ { "ADC",        ASM_ADDR_MODE_REG_D },
    /* 8b */ { "ADC",        ASM_ADDR_MODE_REG_E },
    /* 8c */ { "ADC",        ASM_ADDR_MODE_REG_H },
    /* 8d */ { "ADC",        ASM_ADDR_MODE_REG_L },
    /* 8e */ { "ADC",        ASM_ADDR_MODE_REG_IND_HL },
    /* 8f */ { "ADC",        ASM_ADDR_MODE_ACCUMULATOR },
    /* 90 */ { "SUB",        ASM_ADDR_MODE_REG_B },
    /* 91 */ { "SUB",        ASM_ADDR_MODE_REG_C },
    /* 92 */ { "SUB",        ASM_ADDR_MODE_REG_D },
    /* 93 */ { "SUB",        ASM_ADDR_MODE_REG_E },
    /* 94 */ { "SUB",        ASM_ADDR_MODE_REG_H },
    /* 95 */ { "SUB",        ASM_ADDR_MODE_REG_L },
    /* 96 */ { "SUB",        ASM_ADDR_MODE_REG_IND_HL },
    /* 97 */ { "SUB",        ASM_ADDR_MODE_ACCUMULATOR },
    /* 98 */ { "SBC",        ASM_ADDR_MODE_REG_B },
    /* 99 */ { "SBC",        ASM_ADDR_MODE_REG_C },
    /* 9a */ { "SBC",        ASM_ADDR_MODE_REG_D },
    /* 9b */ { "SBC",        ASM_ADDR_MODE_REG_E },
    /* 9c */ { "SBC",        ASM_ADDR_MODE_REG_H },
    /* 9d */ { "SBC",        ASM_ADDR_MODE_REG_L },
    /* 9e */ { "SBC",        ASM_ADDR_MODE_REG_IND_HL },
    /* 9f */ { "SBC",        ASM_ADDR_MODE_ACCUMULATOR },
    /* a0 */ { "AND",        ASM_ADDR_MODE_REG_B },
    /* a1 */ { "AND",        ASM_ADDR_MODE_REG_C },
    /* a2 */ { "AND",        ASM_ADDR_MODE_REG_D },
    /* a3 */ { "AND",        ASM_ADDR_MODE_REG_E },
    /* a4 */ { "AND",        ASM_ADDR_MODE_REG_H },
    /* a5 */ { "AND",        ASM_ADDR_MODE_REG_L },
    /* a6 */ { "AND",        ASM_ADDR_MODE_REG_IND_HL },
    /* a7 */ { "AND",        ASM_ADDR_MODE_ACCUMULATOR },
    /* a8 */ { "XOR",        ASM_ADDR_MODE_REG_B },
    /* a9 */ { "XOR",        ASM_ADDR_MODE_REG_C },
    /* aa */ { "XOR",        ASM_ADDR_MODE_REG_D },
    /* ab */ { "XOR",        ASM_ADDR_MODE_REG_E },
    /* ac */ { "XOR",        ASM_ADDR_MODE_REG_H },
    /* ad */ { "XOR",        ASM_ADDR_MODE_REG_L },
    /* ae */ { "XOR",        ASM_ADDR_MODE_REG_IND_HL },
    /* af */ { "XOR",        ASM_ADDR_MODE_ACCUMULATOR },
    /* b0 */ { "OR",         ASM_ADDR_MODE_REG_B },
    /* b1 */ { "OR",         ASM_ADDR_MODE_REG_C },
    /* b2 */ { "OR",         ASM_ADDR_MODE_REG_D },
    /* b3 */ { "OR",         ASM_ADDR_MODE_REG_E },
    /* b4 */ { "OR",         ASM_ADDR_MODE_REG_H },
    /* b5 */ { "OR",         ASM_ADDR_MODE_REG_L },
    /* b6 */ { "OR",         ASM_ADDR_MODE_REG_IND_HL },
    /* b7 */ { "OR",         ASM_ADDR_MODE_ACCUMULATOR },
    /* b8 */ { "CP",         ASM_ADDR_MODE_REG_B },
    /* b9 */ { "CP",         ASM_ADDR_MODE_REG_C },
    /* ba */ { "CP",         ASM_ADDR_MODE_REG_D },
    /* bb */ { "CP",         ASM_ADDR_MODE_REG_E },
    /* bc */ { "CP",         ASM_ADDR_MODE_REG_H },
    /* bd */ { "CP",         ASM_ADDR_MODE_REG_L },
    /* be */ { "CP",         ASM_ADDR_MODE_REG_IND_HL },
    /* bf */ { "CP",         ASM_ADDR_MODE_ACCUMULATOR },
    /* c0 */ { "RET NZ",     ASM_ADDR_MODE_IMPLIED },
    /* c1 */ { "POP",        ASM_ADDR_MODE_REG_BC },
    /* c2 */ { "JP NZ,",     ASM_ADDR_MODE_ABSOLUTE },
    /* c3 */ { "JP",         ASM_ADDR_MODE_ABSOLUTE },
    /* c4 */ { "CALL NZ,",   ASM_ADDR_MODE_ABSOLUTE },
    /* c5 */ { "PUSH",       ASM_ADDR_MODE_REG_BC },
    /* c6 */ { "ADD",        ASM_ADDR_MODE_IMMEDIATE },
    /* c7 */ { "RST 00",     ASM_ADDR_MODE_IMPLIED },
    /* c8 */ { "RET Z",      ASM_ADDR_MODE_IMPLIED },
    /* c9 */ { "RET",        ASM_ADDR_MODE_IMPLIED },
    /* ca */ { "JP Z,",      ASM_ADDR_MODE_ABSOLUTE },
    /* cb */ { "PFX_CB",     ASM_ADDR_MODE_IMPLIED },   /* prefix for BIT instructions */
    /* cc */ { "CALL Z,",    ASM_ADDR_MODE_ABSOLUTE },
    /* cd */ { "CALL",       ASM_ADDR_MODE_ABSOLUTE },
    /* ce */ { "ADC",        ASM_ADDR_MODE_IMMEDIATE },
    /* cf */ { "RST 08",     ASM_ADDR_MODE_IMPLIED },
    /* d0 */ { "RET NC",     ASM_ADDR_MODE_IMPLIED },
    /* d1 */ { "POP",        ASM_ADDR_MODE_REG_DE },
    /* d2 */ { "JP NC,",     ASM_ADDR_MODE_ABSOLUTE },
    /* d3 */ { "OUTA",       ASM_ADDR_MODE_ZERO_PAGE },
    /* d4 */ { "CALL NC,",   ASM_ADDR_MODE_ABSOLUTE },
    /* d5 */ { "PUSH",       ASM_ADDR_MODE_REG_DE },
    /* d6 */ { "SUB",        ASM_ADDR_MODE_IMMEDIATE },
    /* d7 */ { "RST 10",     ASM_ADDR_MODE_IMPLIED },
    /* d8 */ { "RET C",      ASM_ADDR_MODE_IMPLIED },
    /* d9 */ { "EXX",        ASM_ADDR_MODE_IMPLIED },
    /* da */ { "JP C,",      ASM_ADDR_MODE_ABSOLUTE },
    /* db */ { "INA",        ASM_ADDR_MODE_ZERO_PAGE },
    /* dc */ { "CALL C,",    ASM_ADDR_MODE_ABSOLUTE },
    /* dd */ { "PFX_DD",     ASM_ADDR_MODE_IMPLIED },   /* prefix for IX instructions */
    /* de */ { "SBC",        ASM_ADDR_MODE_IMMEDIATE },
    /* df */ { "RST 18",     ASM_ADDR_MODE_IMPLIED },
    /* e0 */ { "RET PO",     ASM_ADDR_MODE_IMPLIED },
    /* e1 */ { "POP",        ASM_ADDR_MODE_REG_HL },
    /* e2 */ { "JP PO,",     ASM_ADDR_MODE_ABSOLUTE },
    /* e3 */ { "EX (SP),HL", ASM_ADDR_MODE_IMPLIED },
    /* e4 */ { "CALL PO,",   ASM_ADDR_MODE_ABSOLUTE },
    /* e5 */ { "PUSH",       ASM_ADDR_MODE_REG_HL },
    /* e6 */ { "AND",        ASM_ADDR_MODE_IMMEDIATE },
    /* e7 */ { "RST 20",     ASM_ADDR_MODE_IMPLIED },
    /* e8 */ { "RET PE",     ASM_ADDR_MODE_IMPLIED },
    /* e9 */ { "JP (HL)",    ASM_ADDR_MODE_IMPLIED },
    /* ea */ { "JP PE,",     ASM_ADDR_MODE_ABSOLUTE },
    /* eb */ { "EX DE,HL",   ASM_ADDR_MODE_IMPLIED },
    /* ec */ { "CALL PE,",   ASM_ADDR_MODE_ABSOLUTE },
    /* ed */ { "PFX_ED",     ASM_ADDR_MODE_IMPLIED },   /* prefix for extended instructions */
    /* ee */ { "XOR",        ASM_ADDR_MODE_IMMEDIATE },
    /* ef */ { "RST 28",     ASM_ADDR_MODE_IMPLIED },
    /* f0 */ { "RET P",      ASM_ADDR_MODE_IMPLIED },
    /* f1 */ { "POP",        ASM_ADDR_MODE_REG_AF },
    /* f2 */ { "JP P,",      ASM_ADDR_MODE_ABSOLUTE },
    /* f3 */ { "DI",         ASM_ADDR_MODE_IMPLIED },
    /* f4 */ { "CALL P,",    ASM_ADDR_MODE_ABSOLUTE },
    /* f5 */ { "PUSH",       ASM_ADDR_MODE_REG_AF },
    /* f6 */ { "OR",         ASM_ADDR_MODE_IMMEDIATE },
    /* f7 */ { "RST 30",     ASM_ADDR_MODE_IMPLIED },
    /* f8 */ { "RET M",      ASM_ADDR_MODE_IMPLIED },
    /* f9 */ { "LD SP,HL",   ASM_ADDR_MODE_IMPLIED },
    /* fa */ { "JP M,",      ASM_ADDR_MODE_ABSOLUTE },
    /* fb */ { "EI",         ASM_ADDR_MODE_IMPLIED },
    /* fc */ { "CALL M,",    ASM_ADDR_MODE_ABSOLUTE },
    /* fd */ { "PFX_FD",     ASM_ADDR_MODE_IMPLIED },   /* prefix for IY instructions */
    /* fe */ { "CP",         ASM_ADDR_MODE_IMMEDIATE },
    /* ff */ { "RST 38",     ASM_ADDR_MODE_IMPLIED }
};

/* (CB) BIT instructions */
static const asm_opcode_info_t opcode_list_cb[] = {
    /* cb 00 */ { "RLC",        ASM_ADDR_MODE_REG_B },
    /* cb 01 */ { "RLC",        ASM_ADDR_MODE_REG_C },
    /* cb 02 */ { "RLC",        ASM_ADDR_MODE_REG_D },
    /* cb 03 */ { "RLC",        ASM_ADDR_MODE_REG_E },
    /* cb 04 */ { "RLC",        ASM_ADDR_MODE_REG_H },
    /* cb 05 */ { "RLC",        ASM_ADDR_MODE_REG_L },
    /* cb 06 */ { "RLC",        ASM_ADDR_MODE_REG_IND_HL },
    /* cb 07 */ { "RLC",        ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 08 */ { "RRC",        ASM_ADDR_MODE_REG_B },
    /* cb 09 */ { "RRC",        ASM_ADDR_MODE_REG_C },
    /* cb 0a */ { "RRC",        ASM_ADDR_MODE_REG_D },
    /* cb 0b */ { "RRC",        ASM_ADDR_MODE_REG_E },
    /* cb 0c */ { "RRC",        ASM_ADDR_MODE_REG_H },
    /* cb 0d */ { "RRC",        ASM_ADDR_MODE_REG_L },
    /* cb 0e */ { "RRC",        ASM_ADDR_MODE_REG_IND_HL },
    /* cb 0f */ { "RRC",        ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 10 */ { "RL",         ASM_ADDR_MODE_REG_B },
    /* cb 11 */ { "RL",         ASM_ADDR_MODE_REG_C },
    /* cb 12 */ { "RL",         ASM_ADDR_MODE_REG_D },
    /* cb 13 */ { "RL",         ASM_ADDR_MODE_REG_E },
    /* cb 14 */ { "RL",         ASM_ADDR_MODE_REG_H },
    /* cb 15 */ { "RL",         ASM_ADDR_MODE_REG_L },
    /* cb 16 */ { "RL",         ASM_ADDR_MODE_REG_IND_HL },
    /* cb 17 */ { "RL",         ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 18 */ { "RR",         ASM_ADDR_MODE_REG_B },
    /* cb 19 */ { "RR",         ASM_ADDR_MODE_REG_C },
    /* cb 1a */ { "RR",         ASM_ADDR_MODE_REG_D },
    /* cb 1b */ { "RR",         ASM_ADDR_MODE_REG_E },
    /* cb 1c */ { "RR",         ASM_ADDR_MODE_REG_H },
    /* cb 1d */ { "RR",         ASM_ADDR_MODE_REG_L },
    /* cb 1e */ { "RR",         ASM_ADDR_MODE_REG_IND_HL },
    /* cb 1f */ { "RR",         ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 20 */ { "SLA",        ASM_ADDR_MODE_REG_B },
    /* cb 21 */ { "SLA",        ASM_ADDR_MODE_REG_C },
    /* cb 22 */ { "SLA",        ASM_ADDR_MODE_REG_D },
    /* cb 23 */ { "SLA",        ASM_ADDR_MODE_REG_E },
    /* cb 24 */ { "SLA",        ASM_ADDR_MODE_REG_H },
    /* cb 25 */ { "SLA",        ASM_ADDR_MODE_REG_L },
    /* cb 26 */ { "SLA",        ASM_ADDR_MODE_REG_IND_HL },
    /* cb 27 */ { "SLA",        ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 28 */ { "SRA",        ASM_ADDR_MODE_REG_B },
    /* cb 29 */ { "SRA",        ASM_ADDR_MODE_REG_C },
    /* cb 2a */ { "SRA",        ASM_ADDR_MODE_REG_D },
    /* cb 2b */ { "SRA",        ASM_ADDR_MODE_REG_E },
    /* cb 2c */ { "SRA",        ASM_ADDR_MODE_REG_H },
    /* cb 2d */ { "SRA",        ASM_ADDR_MODE_REG_L },
    /* cb 2e */ { "SRA",        ASM_ADDR_MODE_REG_IND_HL },
    /* cb 2f */ { "SRA",        ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 30 */ { "SLL",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* cb 31 */ { "SLL",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* cb 32 */ { "SLL",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* cb 33 */ { "SLL",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* cb 34 */ { "SLL",        ASM_ADDR_MODE_REG_H | ASM_ADDR_MODE_UNDOC },
    /* cb 35 */ { "SLL",        ASM_ADDR_MODE_REG_L | ASM_ADDR_MODE_UNDOC },
    /* cb 36 */ { "SLL",        ASM_ADDR_MODE_REG_IND_HL | ASM_ADDR_MODE_UNDOC },
    /* cb 37 */ { "SLL",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* cb 38 */ { "SRL",        ASM_ADDR_MODE_REG_B },
    /* cb 39 */ { "SRL",        ASM_ADDR_MODE_REG_C },
    /* cb 3a */ { "SRL",        ASM_ADDR_MODE_REG_D },
    /* cb 3b */ { "SRL",        ASM_ADDR_MODE_REG_E },
    /* cb 3c */ { "SRL",        ASM_ADDR_MODE_REG_H },
    /* cb 3d */ { "SRL",        ASM_ADDR_MODE_REG_L },
    /* cb 3e */ { "SRL",        ASM_ADDR_MODE_REG_IND_HL },
    /* cb 3f */ { "SRL",        ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 40 */ { "BIT 0,",     ASM_ADDR_MODE_REG_B },
    /* cb 41 */ { "BIT 0,",     ASM_ADDR_MODE_REG_C },
    /* cb 42 */ { "BIT 0,",     ASM_ADDR_MODE_REG_D },
    /* cb 43 */ { "BIT 0,",     ASM_ADDR_MODE_REG_E },
    /* cb 44 */ { "BIT 0,",     ASM_ADDR_MODE_REG_H },
    /* cb 45 */ { "BIT 0,",     ASM_ADDR_MODE_REG_L },
    /* cb 46 */ { "BIT 0,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 47 */ { "BIT 0,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 48 */ { "BIT 1,",     ASM_ADDR_MODE_REG_B },
    /* cb 49 */ { "BIT 1,",     ASM_ADDR_MODE_REG_C },
    /* cb 4a */ { "BIT 1,",     ASM_ADDR_MODE_REG_D },
    /* cb 4b */ { "BIT 1,",     ASM_ADDR_MODE_REG_E },
    /* cb 4c */ { "BIT 1,",     ASM_ADDR_MODE_REG_H },
    /* cb 4d */ { "BIT 1,",     ASM_ADDR_MODE_REG_L },
    /* cb 4e */ { "BIT 1,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 4f */ { "BIT 1,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 50 */ { "BIT 2,",     ASM_ADDR_MODE_REG_B },
    /* cb 51 */ { "BIT 2,",     ASM_ADDR_MODE_REG_C },
    /* cb 52 */ { "BIT 2,",     ASM_ADDR_MODE_REG_D },
    /* cb 53 */ { "BIT 2,",     ASM_ADDR_MODE_REG_E },
    /* cb 54 */ { "BIT 2,",     ASM_ADDR_MODE_REG_H },
    /* cb 55 */ { "BIT 2,",     ASM_ADDR_MODE_REG_L },
    /* cb 56 */ { "BIT 2,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 57 */ { "BIT 2,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 58 */ { "BIT 3,",     ASM_ADDR_MODE_REG_B },
    /* cb 59 */ { "BIT 3,",     ASM_ADDR_MODE_REG_C },
    /* cb 5a */ { "BIT 3,",     ASM_ADDR_MODE_REG_D },
    /* cb 5b */ { "BIT 3,",     ASM_ADDR_MODE_REG_E },
    /* cb 5c */ { "BIT 3,",     ASM_ADDR_MODE_REG_H },
    /* cb 5d */ { "BIT 3,",     ASM_ADDR_MODE_REG_L },
    /* cb 5e */ { "BIT 3,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 5f */ { "BIT 3,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 60 */ { "BIT 4,",     ASM_ADDR_MODE_REG_B },
    /* cb 61 */ { "BIT 4,",     ASM_ADDR_MODE_REG_C },
    /* cb 62 */ { "BIT 4,",     ASM_ADDR_MODE_REG_D },
    /* cb 63 */ { "BIT 4,",     ASM_ADDR_MODE_REG_E },
    /* cb 64 */ { "BIT 4,",     ASM_ADDR_MODE_REG_H },
    /* cb 65 */ { "BIT 4,",     ASM_ADDR_MODE_REG_L },
    /* cb 66 */ { "BIT 4,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 67 */ { "BIT 4,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 68 */ { "BIT 5,",     ASM_ADDR_MODE_REG_B },
    /* cb 69 */ { "BIT 5,",     ASM_ADDR_MODE_REG_C },
    /* cb 6a */ { "BIT 5,",     ASM_ADDR_MODE_REG_D },
    /* cb 6b */ { "BIT 5,",     ASM_ADDR_MODE_REG_E },
    /* cb 6c */ { "BIT 5,",     ASM_ADDR_MODE_REG_H },
    /* cb 6d */ { "BIT 5,",     ASM_ADDR_MODE_REG_L },
    /* cb 6e */ { "BIT 5,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 6f */ { "BIT 5,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 70 */ { "BIT 6,",     ASM_ADDR_MODE_REG_B },
    /* cb 71 */ { "BIT 6,",     ASM_ADDR_MODE_REG_C },
    /* cb 72 */ { "BIT 6,",     ASM_ADDR_MODE_REG_D },
    /* cb 73 */ { "BIT 6,",     ASM_ADDR_MODE_REG_E },
    /* cb 74 */ { "BIT 6,",     ASM_ADDR_MODE_REG_H },
    /* cb 75 */ { "BIT 6,",     ASM_ADDR_MODE_REG_L },
    /* cb 76 */ { "BIT 6,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 77 */ { "BIT 6,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 78 */ { "BIT 7,",     ASM_ADDR_MODE_REG_B },
    /* cb 79 */ { "BIT 7,",     ASM_ADDR_MODE_REG_C },
    /* cb 7a */ { "BIT 7,",     ASM_ADDR_MODE_REG_D },
    /* cb 7b */ { "BIT 7,",     ASM_ADDR_MODE_REG_E },
    /* cb 7c */ { "BIT 7,",     ASM_ADDR_MODE_REG_H },
    /* cb 7d */ { "BIT 7,",     ASM_ADDR_MODE_REG_L },
    /* cb 7e */ { "BIT 7,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 7f */ { "BIT 7,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 80 */ { "RES 0,",     ASM_ADDR_MODE_REG_B },
    /* cb 81 */ { "RES 0,",     ASM_ADDR_MODE_REG_C },
    /* cb 82 */ { "RES 0,",     ASM_ADDR_MODE_REG_D },
    /* cb 83 */ { "RES 0,",     ASM_ADDR_MODE_REG_E },
    /* cb 84 */ { "RES 0,",     ASM_ADDR_MODE_REG_H },
    /* cb 85 */ { "RES 0,",     ASM_ADDR_MODE_REG_L },
    /* cb 86 */ { "RES 0,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 87 */ { "RES 0,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 88 */ { "RES 1,",     ASM_ADDR_MODE_REG_B },
    /* cb 89 */ { "RES 1,",     ASM_ADDR_MODE_REG_C },
    /* cb 8a */ { "RES 1,",     ASM_ADDR_MODE_REG_D },
    /* cb 8b */ { "RES 1,",     ASM_ADDR_MODE_REG_E },
    /* cb 8c */ { "RES 1,",     ASM_ADDR_MODE_REG_H },
    /* cb 8d */ { "RES 1,",     ASM_ADDR_MODE_REG_L },
    /* cb 8e */ { "RES 1,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 8f */ { "RES 1,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 90 */ { "RES 2,",     ASM_ADDR_MODE_REG_B },
    /* cb 91 */ { "RES 2,",     ASM_ADDR_MODE_REG_C },
    /* cb 92 */ { "RES 2,",     ASM_ADDR_MODE_REG_D },
    /* cb 93 */ { "RES 2,",     ASM_ADDR_MODE_REG_E },
    /* cb 94 */ { "RES 2,",     ASM_ADDR_MODE_REG_H },
    /* cb 95 */ { "RES 2,",     ASM_ADDR_MODE_REG_L },
    /* cb 96 */ { "RES 2,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 97 */ { "RES 2,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb 98 */ { "RES 3,",     ASM_ADDR_MODE_REG_B },
    /* cb 99 */ { "RES 3,",     ASM_ADDR_MODE_REG_C },
    /* cb 9a */ { "RES 3,",     ASM_ADDR_MODE_REG_D },
    /* cb 9b */ { "RES 3,",     ASM_ADDR_MODE_REG_E },
    /* cb 9c */ { "RES 3,",     ASM_ADDR_MODE_REG_H },
    /* cb 9d */ { "RES 3,",     ASM_ADDR_MODE_REG_L },
    /* cb 9e */ { "RES 3,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb 9f */ { "RES 3,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb a0 */ { "RES 4,",     ASM_ADDR_MODE_REG_B },
    /* cb a1 */ { "RES 4,",     ASM_ADDR_MODE_REG_C },
    /* cb a2 */ { "RES 4,",     ASM_ADDR_MODE_REG_D },
    /* cb a3 */ { "RES 4,",     ASM_ADDR_MODE_REG_E },
    /* cb a4 */ { "RES 4,",     ASM_ADDR_MODE_REG_H },
    /* cb a5 */ { "RES 4,",     ASM_ADDR_MODE_REG_L },
    /* cb a6 */ { "RES 4,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb a7 */ { "RES 4,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb a8 */ { "RES 5,",     ASM_ADDR_MODE_REG_B },
    /* cb a9 */ { "RES 5,",     ASM_ADDR_MODE_REG_C },
    /* cb aa */ { "RES 5,",     ASM_ADDR_MODE_REG_D },
    /* cb ab */ { "RES 5,",     ASM_ADDR_MODE_REG_E },
    /* cb ac */ { "RES 5,",     ASM_ADDR_MODE_REG_H },
    /* cb ad */ { "RES 5,",     ASM_ADDR_MODE_REG_L },
    /* cb ae */ { "RES 5,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb af */ { "RES 5,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb b0 */ { "RES 6,",     ASM_ADDR_MODE_REG_B },
    /* cb b1 */ { "RES 6,",     ASM_ADDR_MODE_REG_C },
    /* cb b2 */ { "RES 6,",     ASM_ADDR_MODE_REG_D },
    /* cb b3 */ { "RES 6,",     ASM_ADDR_MODE_REG_E },
    /* cb b4 */ { "RES 6,",     ASM_ADDR_MODE_REG_H },
    /* cb b5 */ { "RES 6,",     ASM_ADDR_MODE_REG_L },
    /* cb b6 */ { "RES 6,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb b7 */ { "RES 6,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb b8 */ { "RES 7,",     ASM_ADDR_MODE_REG_B },
    /* cb b9 */ { "RES 7,",     ASM_ADDR_MODE_REG_C },
    /* cb ba */ { "RES 7,",     ASM_ADDR_MODE_REG_D },
    /* cb bb */ { "RES 7,",     ASM_ADDR_MODE_REG_E },
    /* cb bc */ { "RES 7,",     ASM_ADDR_MODE_REG_H },
    /* cb bd */ { "RES 7,",     ASM_ADDR_MODE_REG_L },
    /* cb be */ { "RES 7,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb bf */ { "RES 7,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb c0 */ { "SET 0,",     ASM_ADDR_MODE_REG_B },
    /* cb c1 */ { "SET 0,",     ASM_ADDR_MODE_REG_C },
    /* cb c2 */ { "SET 0,",     ASM_ADDR_MODE_REG_D },
    /* cb c3 */ { "SET 0,",     ASM_ADDR_MODE_REG_E },
    /* cb c4 */ { "SET 0,",     ASM_ADDR_MODE_REG_H },
    /* cb c5 */ { "SET 0,",     ASM_ADDR_MODE_REG_L },
    /* cb c6 */ { "SET 0,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb c7 */ { "SET 0,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb c8 */ { "SET 1,",     ASM_ADDR_MODE_REG_B },
    /* cb c9 */ { "SET 1,",     ASM_ADDR_MODE_REG_C },
    /* cb ca */ { "SET 1,",     ASM_ADDR_MODE_REG_D },
    /* cb cb */ { "SET 1,",     ASM_ADDR_MODE_REG_E },
    /* cb cc */ { "SET 1,",     ASM_ADDR_MODE_REG_H },
    /* cb cd */ { "SET 1,",     ASM_ADDR_MODE_REG_L },
    /* cb ce */ { "SET 1,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb cf */ { "SET 1,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb d0 */ { "SET 2,",     ASM_ADDR_MODE_REG_B },
    /* cb d1 */ { "SET 2,",     ASM_ADDR_MODE_REG_C },
    /* cb d2 */ { "SET 2,",     ASM_ADDR_MODE_REG_D },
    /* cb d3 */ { "SET 2,",     ASM_ADDR_MODE_REG_E },
    /* cb d4 */ { "SET 2,",     ASM_ADDR_MODE_REG_H },
    /* cb d5 */ { "SET 2,",     ASM_ADDR_MODE_REG_L },
    /* cb d6 */ { "SET 2,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb d7 */ { "SET 2,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb d8 */ { "SET 3,",     ASM_ADDR_MODE_REG_B },
    /* cb d9 */ { "SET 3,",     ASM_ADDR_MODE_REG_C },
    /* cb da */ { "SET 3,",     ASM_ADDR_MODE_REG_D },
    /* cb db */ { "SET 3,",     ASM_ADDR_MODE_REG_E },
    /* cb dc */ { "SET 3,",     ASM_ADDR_MODE_REG_H },
    /* cb dd */ { "SET 3,",     ASM_ADDR_MODE_REG_L },
    /* cb de */ { "SET 3,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb df */ { "SET 3,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb e0 */ { "SET 4,",     ASM_ADDR_MODE_REG_B },
    /* cb e1 */ { "SET 4,",     ASM_ADDR_MODE_REG_C },
    /* cb e2 */ { "SET 4,",     ASM_ADDR_MODE_REG_D },
    /* cb e3 */ { "SET 4,",     ASM_ADDR_MODE_REG_E },
    /* cb e4 */ { "SET 4,",     ASM_ADDR_MODE_REG_H },
    /* cb e5 */ { "SET 4,",     ASM_ADDR_MODE_REG_L },
    /* cb e6 */ { "SET 4,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb e7 */ { "SET 4,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb e8 */ { "SET 5,",     ASM_ADDR_MODE_REG_B },
    /* cb e9 */ { "SET 5,",     ASM_ADDR_MODE_REG_C },
    /* cb ea */ { "SET 5,",     ASM_ADDR_MODE_REG_D },
    /* cb eb */ { "SET 5,",     ASM_ADDR_MODE_REG_E },
    /* cb ec */ { "SET 5,",     ASM_ADDR_MODE_REG_H },
    /* cb ed */ { "SET 5,",     ASM_ADDR_MODE_REG_L },
    /* cb ee */ { "SET 5,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb ef */ { "SET 5,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb f0 */ { "SET 6,",     ASM_ADDR_MODE_REG_B },
    /* cb f1 */ { "SET 6,",     ASM_ADDR_MODE_REG_C },
    /* cb f2 */ { "SET 6,",     ASM_ADDR_MODE_REG_D },
    /* cb f3 */ { "SET 6,",     ASM_ADDR_MODE_REG_E },
    /* cb f4 */ { "SET 6,",     ASM_ADDR_MODE_REG_H },
    /* cb f5 */ { "SET 6,",     ASM_ADDR_MODE_REG_L },
    /* cb f6 */ { "SET 6,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb f7 */ { "SET 6,",     ASM_ADDR_MODE_ACCUMULATOR },
    /* cb f8 */ { "SET 7,",     ASM_ADDR_MODE_REG_B },
    /* cb f9 */ { "SET 7,",     ASM_ADDR_MODE_REG_C },
    /* cb fa */ { "SET 7,",     ASM_ADDR_MODE_REG_D },
    /* cb fb */ { "SET 7,",     ASM_ADDR_MODE_REG_E },
    /* cb fc */ { "SET 7,",     ASM_ADDR_MODE_REG_H },
    /* cb fd */ { "SET 7,",     ASM_ADDR_MODE_REG_L },
    /* cb fe */ { "SET 7,",     ASM_ADDR_MODE_REG_IND_HL },
    /* cb ff */ { "SET 7,",     ASM_ADDR_MODE_ACCUMULATOR }
};

/* (DD) IX instructions */
static const asm_opcode_info_t opcode_list_dd[] = {
    /* dd 00 */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 01 */ { "LD BC,",     ASM_ADDR_MODE_IMMEDIATE_16 | ASM_ADDR_MODE_UNDOC },
    /* dd 02 */ { "LD (BC),A",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 03 */ { "INC",        ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* dd 04 */ { "INC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd 05 */ { "DEC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd 06 */ { "LD B,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 07 */ { "RLCA",       ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 08 */ { "EX AF,AF'",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 09 */ { "ADD IX,BC",  ASM_ADDR_MODE_IMPLIED },
    /* dd 0a */ { "LD A,(BC)",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 0b */ { "DEC",        ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* dd 0c */ { "INC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd 0d */ { "DEC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd 0e */ { "LD C,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 0f */ { "RRCA",       ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 10 */ { "DJNZ",       ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* dd 11 */ { "LD DE,",     ASM_ADDR_MODE_IMMEDIATE_16 | ASM_ADDR_MODE_UNDOC },
    /* dd 12 */ { "LD (DE),A",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 13 */ { "INC",        ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* dd 14 */ { "INC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd 15 */ { "DEC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd 16 */ { "LD D,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 17 */ { "RLA",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 18 */ { "JR",         ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* dd 19 */ { "ADD IX,DE",  ASM_ADDR_MODE_IMPLIED },
    /* dd 1a */ { "LD A,(DE)",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 1b */ { "DEC",        ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* dd 1c */ { "INC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd 1d */ { "DEC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd 1e */ { "LD E,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 1f */ { "RRA",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 20 */ { "JR NZ,",     ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* dd 21 */ { "LD IX,",     ASM_ADDR_MODE_IMMEDIATE_16 },
    /* dd 22 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_IX },
    /* dd 23 */ { "INC",        ASM_ADDR_MODE_REG_IX },
    /* dd 24 */ { "INC",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd 25 */ { "DEC",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd 26 */ { "LD IXH,",    ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 27 */ { "DAA",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 28 */ { "JR Z,",      ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* dd 29 */ { "ADD IX,IX",  ASM_ADDR_MODE_IMPLIED },
    /* dd 2a */ { "LD IX,",     ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT },
    /* dd 2b */ { "DEC",        ASM_ADDR_MODE_REG_IX },
    /* dd 2c */ { "INC",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd 2d */ { "DEC",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd 2e */ { "LD IXL,",    ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 2f */ { "CPL",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 30 */ { "JR NC,",     ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* dd 31 */ { "LD SP,",     ASM_ADDR_MODE_IMMEDIATE_16 | ASM_ADDR_MODE_UNDOC },
    /* dd 32 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_A | ASM_ADDR_MODE_UNDOC },
    /* dd 33 */ { "INC",        ASM_ADDR_MODE_REG_SP | ASM_ADDR_MODE_UNDOC },
    /* dd 34 */ { "INC",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd 35 */ { "DEC",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd 36 */ { "LD (IX",     ASM_ADDR_MODE_Z80_IND_IMMEDIATE },
    /* dd 37 */ { "SCF",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 38 */ { "JR C,",      ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* dd 39 */ { "ADD IX,SP",  ASM_ADDR_MODE_IMPLIED },
    /* dd 3a */ { "LD A,",      ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT | ASM_ADDR_MODE_UNDOC },
    /* dd 3b */ { "DEC",        ASM_ADDR_MODE_REG_SP | ASM_ADDR_MODE_UNDOC },
    /* dd 3c */ { "INC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd 3d */ { "DEC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd 3e */ { "LD A,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd 3f */ { "CCF",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 40 */ { "LD B,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 41 */ { "LD B,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 42 */ { "LD B,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 43 */ { "LD B,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 44 */ { "LD B,IXH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 45 */ { "LD B,IXL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 46 */ { "LD B,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 47 */ { "LD B,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 48 */ { "LD C,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 49 */ { "LD C,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 4a */ { "LD C,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 4b */ { "LD C,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 4c */ { "LD C,IXH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 4d */ { "LD C,IXL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 4e */ { "LD C,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 4f */ { "LD C,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 50 */ { "LD D,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 51 */ { "LD D,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 52 */ { "LD D,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 53 */ { "LD D,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 54 */ { "LD D,IXH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 55 */ { "LD D,IXL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 56 */ { "LD D,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 57 */ { "LD D,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 58 */ { "LD E,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 59 */ { "LD E,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 5a */ { "LD E,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 5b */ { "LD E,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 5c */ { "LD E,IXH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 5d */ { "LD E,IXL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 5e */ { "LD E,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 5f */ { "LD E,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 60 */ { "LD IXH,B",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 61 */ { "LD IXH,C",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 62 */ { "LD IXH,D",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 63 */ { "LD IXH,E",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 64 */ { "LD IXH,IXH", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 65 */ { "LD IXH,IXL", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 66 */ { "LD H,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 67 */ { "LD IXH,A",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 68 */ { "LD IXL,B",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 69 */ { "LD IXL,C",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 6a */ { "LD IXL,D",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 6b */ { "LD IXL,E",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 6c */ { "LD IXL,IXH", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 6d */ { "LD IXL,IXL", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 6e */ { "LD L,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 6f */ { "LD IXL,A",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 70 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 71 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 72 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 73 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 74 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 75 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 76 */ { "HALT",       ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 77 */ { "LD (IX+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* dd 78 */ { "LD A,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 79 */ { "LD A,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 7a */ { "LD A,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 7b */ { "LD A,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 7c */ { "LD A,IXH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 7d */ { "LD A,IXL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 7e */ { "LD A,",      ASM_ADDR_MODE_REG_IND_IX },
    /* dd 7f */ { "LD A,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd 80 */ { "ADD",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd 81 */ { "ADD",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd 82 */ { "ADD",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd 83 */ { "ADD",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd 84 */ { "ADD",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd 85 */ { "ADD",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd 86 */ { "ADD",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd 87 */ { "ADD",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd 88 */ { "ADC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd 89 */ { "ADC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd 8a */ { "ADC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd 8b */ { "ADC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd 8c */ { "ADC",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd 8d */ { "ADC",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd 8e */ { "ADC",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd 8f */ { "ADC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd 90 */ { "SUB",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd 91 */ { "SUB",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd 92 */ { "SUB",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd 93 */ { "SUB",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd 94 */ { "SUB",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd 95 */ { "SUB",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd 96 */ { "SUB",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd 97 */ { "SUB",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd 98 */ { "SBC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd 99 */ { "SBC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd 9a */ { "SBC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd 9b */ { "SBC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd 9c */ { "SBC",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd 9d */ { "SBC",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd 9e */ { "SBC",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd 9f */ { "SBC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd a0 */ { "AND",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd a1 */ { "AND",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd a2 */ { "AND",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd a3 */ { "AND",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd a4 */ { "AND",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd a5 */ { "AND",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd a6 */ { "AND",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd a7 */ { "AND",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd a8 */ { "XOR",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd a9 */ { "XOR",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd aa */ { "XOR",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd ab */ { "XOR",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd ac */ { "XOR",        ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd ad */ { "XOR",        ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd ae */ { "XOR",        ASM_ADDR_MODE_REG_IND_IX },
    /* dd af */ { "XOR",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd b0 */ { "OR",         ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd b1 */ { "OR",         ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd b2 */ { "OR",         ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd b3 */ { "OR",         ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd b4 */ { "OR",         ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd b5 */ { "OR",         ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd b6 */ { "OR",         ASM_ADDR_MODE_REG_IND_IX },
    /* dd b7 */ { "OR",         ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd b8 */ { "CP",         ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* dd b9 */ { "CP",         ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* dd ba */ { "CP",         ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* dd bb */ { "CP",         ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* dd bc */ { "CP",         ASM_ADDR_MODE_REG_IXH | ASM_ADDR_MODE_UNDOC },
    /* dd bd */ { "CP",         ASM_ADDR_MODE_REG_IXL | ASM_ADDR_MODE_UNDOC },
    /* dd be */ { "CP",         ASM_ADDR_MODE_REG_IND_IX },
    /* dd bf */ { "CP",         ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* dd c0 */ { "RET NZ",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd c1 */ { "POP",        ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* dd c2 */ { "JP NZ,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd c3 */ { "JP",         ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd c4 */ { "CALL NZ,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd c5 */ { "PUSH",       ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* dd c6 */ { "ADD",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd c7 */ { "RST 00",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd c8 */ { "RET Z",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd c9 */ { "RET",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd ca */ { "JP Z,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd cb */ { "PFX_DDCB",   ASM_ADDR_MODE_IMPLIED },   /* prefix for IX BIT instructions */
    /* dd cc */ { "CALL Z,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd cd */ { "CALL",       ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd ce */ { "ADC",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd cf */ { "RST 08",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd d0 */ { "RET NC",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd d1 */ { "POP",        ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* dd d2 */ { "JP NC,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd d3 */ { "OUTA",       ASM_ADDR_MODE_ZERO_PAGE | ASM_ADDR_MODE_UNDOC },
    /* dd d4 */ { "CALL NC,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd d5 */ { "PUSH",       ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* dd d6 */ { "SUB",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd d7 */ { "RST 10",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd d8 */ { "RET C",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd d9 */ { "EXX",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd da */ { "JP C,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd db */ { "INA",        ASM_ADDR_MODE_ZERO_PAGE | ASM_ADDR_MODE_UNDOC },
    /* dd dc */ { "CALL C,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd dd */ { "NOP",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd de */ { "SBC",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd df */ { "RST 18",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd e0 */ { "RET PO",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd e1 */ { "POP",        ASM_ADDR_MODE_REG_IX },
    /* dd e2 */ { "JP PO,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd e3 */ { "EX (SP),IX", ASM_ADDR_MODE_IMPLIED },
    /* dd e4 */ { "CALL PO,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd e5 */ { "PUSH",       ASM_ADDR_MODE_REG_IX },
    /* dd e6 */ { "AND",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd e7 */ { "RST 20",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd e8 */ { "RET PE",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd e9 */ { "JP (IX)",    ASM_ADDR_MODE_IMPLIED },
    /* dd ea */ { "JP PE,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd eb */ { "EX DE,HL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd ec */ { "CALL PE,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd ed */ { "PFX_DDED",   ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd ee */ { "XOR",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd ef */ { "RST 28",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd f0 */ { "RET P",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd f1 */ { "POP",        ASM_ADDR_MODE_REG_AF | ASM_ADDR_MODE_UNDOC },
    /* dd f2 */ { "JP P,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd f3 */ { "DI",         ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC|ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd f4 */ { "CALL P,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd f5 */ { "PUSH",       ASM_ADDR_MODE_REG_AF | ASM_ADDR_MODE_UNDOC },
    /* dd f6 */ { "OR",         ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd f7 */ { "RST 30",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd f8 */ { "RET M",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd f9 */ { "LD SP,IX",   ASM_ADDR_MODE_IMPLIED },
    /* dd fa */ { "JP M,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd fb */ { "EI",         ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* dd fc */ { "CALL M,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* dd fd */ { "NOP",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd fe */ { "CP",         ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* dd ff */ { "RST 38",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC }
};

/* (DD CB) IX BIT instructions */
static const asm_opcode_info_t opcode_list_dd_cb[] = {
    /* dd cb .. 00 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 01 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 02 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 03 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 04 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 05 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 06 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 07 */ { "RLC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 08 */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 09 */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 0a */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 0b */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 0c */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 0d */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 0e */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 0f */ { "RRC",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 10 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 11 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 12 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 13 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 14 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 15 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 16 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 17 */ { "RL",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 18 */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 19 */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 1a */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 1b */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 1c */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 1d */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 1e */ { "RR",      ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 1f */ { "RR",      ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 20 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 21 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 22 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 23 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 24 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 25 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 26 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 27 */ { "SLA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 28 */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 29 */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 2a */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 2b */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 2c */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 2d */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 2e */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 2f */ { "SRA",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 30 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 31 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 32 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 33 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 34 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 35 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 36 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 37 */ { "SLL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 38 */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 39 */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 3a */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 3b */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 3c */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 3d */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 3e */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 3f */ { "SRL",     ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 40 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 41 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 42 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 43 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 44 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 45 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 46 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 47 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 48 */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 49 */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 4a */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 4b */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 4c */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 4d */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 4e */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 4f */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 50 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 51 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 52 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 53 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 54 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 55 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 56 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 57 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 58 */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 59 */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 5a */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 5b */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 5c */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 5d */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 5e */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 5f */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 60 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 61 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 62 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 63 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 64 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 65 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 66 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 67 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 68 */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 69 */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 6a */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 6b */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 6c */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 6d */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 6e */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 6f */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 70 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 71 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 72 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 73 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 74 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 75 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 76 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 77 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 78 */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 79 */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 7a */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 7b */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 7c */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 7d */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 7e */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX },
    /* dd cb .. 7f */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IX | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 80 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 81 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 82 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 83 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 84 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 85 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 86 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 87 */ { "RES 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 88 */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 89 */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 8a */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 8b */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 8c */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 8d */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 8e */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 8f */ { "RES 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 90 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 91 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 92 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 93 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 94 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 95 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 96 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 97 */ { "RES 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 98 */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 99 */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 9a */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 9b */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 9c */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 9d */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. 9e */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. 9f */ { "RES 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a0 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a1 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a2 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a3 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a4 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a5 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a6 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. a7 */ { "RES 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a8 */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. a9 */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. aa */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ab */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ac */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ad */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ae */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. af */ { "RES 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b0 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b1 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b2 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b3 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b4 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b5 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b6 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. b7 */ { "RES 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b8 */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. b9 */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ba */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. bb */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. bc */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. bd */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. be */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. bf */ { "RES 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c0 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c1 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c2 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c3 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c4 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c5 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c6 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. c7 */ { "SET 0,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c8 */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. c9 */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ca */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. cb */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. cc */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. cd */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ce */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. cf */ { "SET 1,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d0 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d1 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d2 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d3 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d4 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d5 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d6 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. d7 */ { "SET 2,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d8 */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. d9 */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. da */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. db */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. dc */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. dd */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. de */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. df */ { "SET 3,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e0 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e1 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e2 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e3 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e4 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e5 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e6 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. e7 */ { "SET 4,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e8 */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. e9 */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ea */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. eb */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ec */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ed */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. ee */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. ef */ { "SET 5,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f0 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f1 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f2 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f3 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f4 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f5 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f6 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. f7 */ { "SET 6,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f8 */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. f9 */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. fa */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. fb */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. fc */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. fd */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
    /* dd cb .. fe */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG },
    /* dd cb .. ff */ { "SET 7,",  ASM_ADDR_MODE_IND_IX_REG | ASM_ADDR_MODE_UNDOC },
};

/* (ED) extended instructions */
static const asm_opcode_info_t opcode_list_ed[] = {
    /* ed 00 - 3F are undefined */
    /* ed 00 */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 01 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 02 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 03 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 04 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 05 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 06 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 07 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 08 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 09 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 0a */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 0b */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 0c */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 0d */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 0e */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 0f */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 10 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 11 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 12 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 13 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 14 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 15 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 16 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 17 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 18 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 19 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 1a */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 1b */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 1c */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 1d */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 1e */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 1f */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 20 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 21 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 22 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 23 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 24 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 25 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 26 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 27 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 28 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 29 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 2a */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 2b */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 2c */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 2d */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 2e */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 2f */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 30 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 31 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 32 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 33 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 34 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 35 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 36 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 37 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 38 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 39 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 3a */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 3b */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 3c */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 3d */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 3e */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 3f */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },

    /* ed 40 - 7F are valid opcodes */
    /* ed 40 */ { "IN B,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 41 */ { "OUT (C),B",  ASM_ADDR_MODE_IMPLIED },
    /* ed 42 */ { "SBC HL,BC",  ASM_ADDR_MODE_IMPLIED },
    /* ed 43 */ { "LD",         ASM_ADDR_MODE_Z80_ABSOLUTE_BC },
    /* ed 44 */ { "NEG",        ASM_ADDR_MODE_IMPLIED },
    /* ed 45 */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 46 */ { "IM 0",       ASM_ADDR_MODE_IMPLIED },
    /* ed 47 */ { "LD I,A",     ASM_ADDR_MODE_IMPLIED },
    /* ed 48 */ { "IN C,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 49 */ { "OUT (C),C",  ASM_ADDR_MODE_IMPLIED },
    /* ed 4a */ { "ADC HL,BC",  ASM_ADDR_MODE_IMPLIED },
    /* ed 4b */ { "LD BC,",     ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT },
    /* ed 4c */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 4d */ { "RETI",       ASM_ADDR_MODE_IMPLIED },
    /* ed 4e */ { "IM 0/1",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 4f */ { "LD R,A",     ASM_ADDR_MODE_IMPLIED },
    /* ed 50 */ { "IN D,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 51 */ { "OUT (C),D",  ASM_ADDR_MODE_IMPLIED },
    /* ed 52 */ { "SBC HL,DE",  ASM_ADDR_MODE_IMPLIED },
    /* ed 53 */ { "LD",         ASM_ADDR_MODE_Z80_ABSOLUTE_DE },
    /* ed 54 */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 55 */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 56 */ { "IM 1",       ASM_ADDR_MODE_IMPLIED },
    /* ed 57 */ { "LD A,I",     ASM_ADDR_MODE_IMPLIED },
    /* ed 58 */ { "IN E,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 59 */ { "OUT (C),E",  ASM_ADDR_MODE_IMPLIED },
    /* ed 5a */ { "ADC HL,DE",  ASM_ADDR_MODE_IMPLIED },
    /* ed 5b */ { "LD DE,",     ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT },
    /* ed 5c */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 5d */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 5e */ { "IM 2",       ASM_ADDR_MODE_IMPLIED },
    /* ed 5f */ { "LD A,R",     ASM_ADDR_MODE_IMPLIED },
    /* ed 60 */ { "IN H,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 61 */ { "OUT (C),H",  ASM_ADDR_MODE_IMPLIED },
    /* ed 62 */ { "SBC HL,HL",  ASM_ADDR_MODE_IMPLIED },
    /* ed 63 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_HL | ASM_ADDR_MODE_UNDOC },
    /* ed 64 */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 65 */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 66 */ { "IM 0",       ASM_ADDR_MODE_IMPLIED },
    /* ed 67 */ { "RRD",        ASM_ADDR_MODE_IMPLIED },
    /* ed 68 */ { "IN L,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 69 */ { "OUT (C),L",  ASM_ADDR_MODE_IMPLIED },
    /* ed 6a */ { "ADC HL,HL",  ASM_ADDR_MODE_IMPLIED },
    /* ed 6b */ { "LD HL,",     ASM_ADDR_MODE_ABS_INDIRECT | ASM_ADDR_MODE_UNDOC },
    /* ed 6c */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 6d */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 6e */ { "IM 0/1",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 6f */ { "RLD",        ASM_ADDR_MODE_IMPLIED },
    /* ed 70 */ { "IN (C)",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 71 */ { "OUT (C), 0", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 72 */ { "SBC HL,SP",  ASM_ADDR_MODE_IMPLIED },
    /* ed 73 */ { "LD",         ASM_ADDR_MODE_Z80_ABSOLUTE_SP },
    /* ed 74 */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 75 */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 76 */ { "IM 1",       ASM_ADDR_MODE_IMPLIED },
    /* ed 77 */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 78 */ { "IN A,(C)",   ASM_ADDR_MODE_IMPLIED },
    /* ed 79 */ { "OUT (C),A",  ASM_ADDR_MODE_IMPLIED },
    /* ed 7a */ { "ADC HL,SP",  ASM_ADDR_MODE_IMPLIED },
    /* ed 7b */ { "LD SP,",     ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT },
    /* ed 7c */ { "NEG",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 7d */ { "RETN",       ASM_ADDR_MODE_IMPLIED },
    /* ed 7e */ { "IM 2",       ASM_ADDR_MODE_IMPLIED },
    /* ed 7f */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },

    /* ed 80 - 9F are not defined */
    /* ed 80 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 81 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 82 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 83 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 84 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 85 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 86 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 87 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 88 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 89 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 8a */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 8b */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 8c */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 8d */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 8e */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 8f */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 90 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 91 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 92 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 93 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 94 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 95 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 96 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 97 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 98 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 99 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 9a */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 9b */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 9c */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 9d */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 9e */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed 9f */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },

    /* ed A0 - BF are valid opcodes */
    /* ed a0 */ { "LDI",        ASM_ADDR_MODE_IMPLIED },
    /* ed a1 */ { "CPI",        ASM_ADDR_MODE_IMPLIED },
    /* ed a2 */ { "INI",        ASM_ADDR_MODE_IMPLIED },
    /* ed a3 */ { "OUTI",       ASM_ADDR_MODE_IMPLIED },
    /* ed a4 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed a5 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed a6 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed a7 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed a8 */ { "LDD",        ASM_ADDR_MODE_IMPLIED },
    /* ed a9 */ { "CPD",        ASM_ADDR_MODE_IMPLIED },
    /* ed aa */ { "IND",        ASM_ADDR_MODE_IMPLIED },
    /* ed ab */ { "OUTD",       ASM_ADDR_MODE_IMPLIED },
    /* ed ac */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ad */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ae */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed af */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed b0 */ { "LDIR",       ASM_ADDR_MODE_IMPLIED },
    /* ed b1 */ { "CPIR",       ASM_ADDR_MODE_IMPLIED },
    /* ed b2 */ { "INIR",       ASM_ADDR_MODE_IMPLIED },
    /* ed b3 */ { "OTIR",       ASM_ADDR_MODE_IMPLIED },
    /* ed b4 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed b5 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed b6 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed b7 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed b8 */ { "LDDR",       ASM_ADDR_MODE_IMPLIED },
    /* ed b9 */ { "CPDR",       ASM_ADDR_MODE_IMPLIED },
    /* ed ba */ { "INDR",       ASM_ADDR_MODE_IMPLIED },
    /* ed bb */ { "OTDR",       ASM_ADDR_MODE_IMPLIED },
    /* ed bc */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed bd */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed be */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed bf */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },

    /* ed C0 - FF are undefined */
    /* ed c0 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c1 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c2 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c3 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c4 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c5 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c6 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c7 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c8 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed c9 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ca */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed cb */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed cc */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed cd */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ce */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed cf */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d0 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d1 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d2 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d3 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d4 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d5 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d6 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d7 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d8 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed d9 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed da */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed db */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed dc */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed dd */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed de */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed df */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e0 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e1 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e2 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e3 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e4 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e5 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e6 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e7 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e8 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed e9 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ea */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed eb */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ec */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ed */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ee */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ef */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f0 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f1 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f2 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f3 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f4 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f5 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f6 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f7 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f8 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed f9 */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed fa */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed fb */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed fc */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed fd */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed fe */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* ed ff */ { "",           ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC }
};

/* (FD) IY instructions */
static const asm_opcode_info_t opcode_list_fd[] = {
    /* fd 00 */ { "NOP",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 01 */ { "LD BC,",     ASM_ADDR_MODE_IMMEDIATE_16 | ASM_ADDR_MODE_UNDOC },
    /* fd 02 */ { "LD (BC),A",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 03 */ { "INC",        ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* fd 04 */ { "INC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd 05 */ { "DEC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd 06 */ { "LD B,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 07 */ { "RLCA",       ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 08 */ { "EX AF,AF'",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 09 */ { "ADD IY,BC",  ASM_ADDR_MODE_IMPLIED },
    /* fd 0a */ { "LD A,(BC)",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 0b */ { "DEC",        ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* fd 0c */ { "INC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd 0d */ { "DEC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd 0e */ { "LD C,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 0f */ { "RRCA",       ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 10 */ { "DJNZ",       ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* fd 11 */ { "LD DE,",     ASM_ADDR_MODE_IMMEDIATE_16 | ASM_ADDR_MODE_UNDOC },
    /* fd 12 */ { "LD (DE),A",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 13 */ { "INC",        ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* fd 14 */ { "INC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd 15 */ { "DEC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd 16 */ { "LD D,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 17 */ { "RLA",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 18 */ { "JR",         ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* fd 19 */ { "ADD IY,DE",  ASM_ADDR_MODE_IMPLIED },
    /* fd 1a */ { "LD A,(DE)",  ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 1b */ { "DEC",        ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* fd 1c */ { "INC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd 1d */ { "DEC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd 1e */ { "LD E,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 1f */ { "RRA",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 20 */ { "JR NZ,",     ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* fd 21 */ { "LD IY,",     ASM_ADDR_MODE_IMMEDIATE_16 },
    /* fd 22 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_IY },
    /* fd 23 */ { "INC",        ASM_ADDR_MODE_REG_IY },
    /* fd 24 */ { "INC",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd 25 */ { "DEC",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd 26 */ { "LD IYH,",    ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 27 */ { "DAA",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 28 */ { "JR Z,",      ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* fd 29 */ { "ADD IY,IY",  ASM_ADDR_MODE_IMPLIED },
    /* fd 2a */ { "LD IY,",     ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT },
    /* fd 2b */ { "DEC",        ASM_ADDR_MODE_REG_IY },
    /* fd 2c */ { "INC",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd 2d */ { "DEC",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd 2e */ { "LD IYL,",    ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 2f */ { "CPL",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 30 */ { "JR NC,",     ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* fd 31 */ { "LD SP,",     ASM_ADDR_MODE_IMMEDIATE_16 | ASM_ADDR_MODE_UNDOC },
    /* fd 32 */ { "LD",         ASM_ADDR_MODE_ABSOLUTE_A | ASM_ADDR_MODE_UNDOC },
    /* fd 33 */ { "INC",        ASM_ADDR_MODE_REG_SP | ASM_ADDR_MODE_UNDOC },
    /* fd 34 */ { "INC",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd 35 */ { "DEC",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd 36 */ { "LD (IY",     ASM_ADDR_MODE_Z80_IND_IMMEDIATE },
    /* fd 37 */ { "SCF",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 38 */ { "JR C,",      ASM_ADDR_MODE_RELATIVE | ASM_ADDR_MODE_UNDOC },
    /* fd 39 */ { "ADD IY,SP",  ASM_ADDR_MODE_IMPLIED },
    /* fd 3a */ { "LD A,",      ASM_ADDR_MODE_Z80_ABS_INDIRECT_EXT | ASM_ADDR_MODE_UNDOC },
    /* fd 3b */ { "DEC",        ASM_ADDR_MODE_REG_SP | ASM_ADDR_MODE_UNDOC },
    /* fd 3c */ { "INC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd 3d */ { "DEC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd 3e */ { "LD A,",      ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd 3f */ { "CCF",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 40 */ { "LD B,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 41 */ { "LD B,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 42 */ { "LD B,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 43 */ { "LD B,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 44 */ { "LD B,IYH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 45 */ { "LD B,IYL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 46 */ { "LD B,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 47 */ { "LD B,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 48 */ { "LD C,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 49 */ { "LD C,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 4a */ { "LD C,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 4b */ { "LD C,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 4c */ { "LD C,IYH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 4d */ { "LD C,IYL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 4e */ { "LD C,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 4f */ { "LD C,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 50 */ { "LD D,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 51 */ { "LD D,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 52 */ { "LD D,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 53 */ { "LD D,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 54 */ { "LD D,IYH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 55 */ { "LD D,IY",    ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 56 */ { "LD D,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 57 */ { "LD D,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 58 */ { "LD E,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 59 */ { "LD E,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 5a */ { "LD E,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 5b */ { "LD E,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 5c */ { "LD E,IYH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 5d */ { "LD E,IYL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 5e */ { "LD E,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 5f */ { "LD E,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 60 */ { "LD IYH,B",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 61 */ { "LD IYH,C",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 62 */ { "LD IYH,D",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 63 */ { "LD IYH,E",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 64 */ { "LD IYH,IYH", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 65 */ { "LD IYH,IYL", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 66 */ { "LD H,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 67 */ { "LD IYH,A",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 68 */ { "LD IYL,B",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 69 */ { "LD IYL,C",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 6a */ { "LD IYL,D",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 6b */ { "LD IYL,E",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 6c */ { "LD IYL,IYH", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 6d */ { "LD IYL,IYL", ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 6e */ { "LD L,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 6f */ { "LD IYL,A",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 70 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 71 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 72 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 73 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 74 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 75 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 76 */ { "HALT",       ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 77 */ { "LD (IY+",    ASM_ADDR_MODE_Z80_IND_REG },
    /* fd 78 */ { "LD A,B",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 79 */ { "LD A,C",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 7a */ { "LD A,D",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 7b */ { "LD A,E",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 7c */ { "LD A,IYH",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 7d */ { "LD A,IYL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 7e */ { "LD A,",      ASM_ADDR_MODE_REG_IND_IY },
    /* fd 7f */ { "LD A,A",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd 80 */ { "ADD",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd 81 */ { "ADD",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd 82 */ { "ADD",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd 83 */ { "ADD",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd 84 */ { "ADD",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd 85 */ { "ADD",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd 86 */ { "ADD",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd 87 */ { "ADD",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd 88 */ { "ADC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd 89 */ { "ADC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd 8a */ { "ADC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd 8b */ { "ADC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd 8c */ { "ADC",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd 8d */ { "ADC",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd 8e */ { "ADC",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd 8f */ { "ADC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd 90 */ { "SUB",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd 91 */ { "SUB",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd 92 */ { "SUB",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd 93 */ { "SUB",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd 94 */ { "SUB",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd 95 */ { "SUB",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd 96 */ { "SUB",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd 97 */ { "SUB",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd 98 */ { "SBC",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd 99 */ { "SBC",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd 9a */ { "SBC",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd 9b */ { "SBC",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd 9c */ { "SBC",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd 9d */ { "SBC",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd 9e */ { "SBC",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd 9f */ { "SBC",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd a0 */ { "AND",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd a1 */ { "AND",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd a2 */ { "AND",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd a3 */ { "AND",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd a4 */ { "AND",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd a5 */ { "AND",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd a6 */ { "AND",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd a7 */ { "AND",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd a8 */ { "XOR",        ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd a9 */ { "XOR",        ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd aa */ { "XOR",        ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd ab */ { "XOR",        ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd ac */ { "XOR",        ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd ad */ { "XOR",        ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd ae */ { "XOR",        ASM_ADDR_MODE_REG_IND_IY },
    /* fd af */ { "XOR",        ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd b0 */ { "OR",         ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd b1 */ { "OR",         ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd b2 */ { "OR",         ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd b3 */ { "OR",         ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd b4 */ { "OR",         ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd b5 */ { "OR",         ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd b6 */ { "OR",         ASM_ADDR_MODE_REG_IND_IY },
    /* fd b7 */ { "OR",         ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd b8 */ { "CP",         ASM_ADDR_MODE_REG_B | ASM_ADDR_MODE_UNDOC },
    /* fd b9 */ { "CP",         ASM_ADDR_MODE_REG_C | ASM_ADDR_MODE_UNDOC },
    /* fd ba */ { "CP",         ASM_ADDR_MODE_REG_D | ASM_ADDR_MODE_UNDOC },
    /* fd bb */ { "CP",         ASM_ADDR_MODE_REG_E | ASM_ADDR_MODE_UNDOC },
    /* fd bc */ { "CP",         ASM_ADDR_MODE_REG_IYH | ASM_ADDR_MODE_UNDOC },
    /* fd bd */ { "CP",         ASM_ADDR_MODE_REG_IYL | ASM_ADDR_MODE_UNDOC },
    /* fd be */ { "CP",         ASM_ADDR_MODE_REG_IND_IY },
    /* fd bf */ { "CP",         ASM_ADDR_MODE_ACCUMULATOR | ASM_ADDR_MODE_UNDOC },
    /* fd c0 */ { "RET NZ",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd c1 */ { "POP",        ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* fd c2 */ { "JP NZ,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd c3 */ { "JP",         ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd c4 */ { "CALL NZ,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd c5 */ { "PUSH",       ASM_ADDR_MODE_REG_BC | ASM_ADDR_MODE_UNDOC },
    /* fd c6 */ { "ADD",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd c7 */ { "RST 00",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd c8 */ { "RET Z",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd c9 */ { "RET",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd ca */ { "JP Z,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd cb */ { "PFX_FDCB",   ASM_ADDR_MODE_IMPLIED },  /* prefix for IY BIT instructions */
    /* fd cc */ { "CALL Z,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd cd */ { "CALL",       ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd ce */ { "ADC",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd cf */ { "RST 08",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd d0 */ { "RET NC",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd d1 */ { "POP",        ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* fd d2 */ { "JP NC,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd d3 */ { "OUTA",       ASM_ADDR_MODE_ZERO_PAGE | ASM_ADDR_MODE_UNDOC },
    /* fd d4 */ { "CALL NC,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd d5 */ { "PUSH",       ASM_ADDR_MODE_REG_DE | ASM_ADDR_MODE_UNDOC },
    /* fd d6 */ { "SUB",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd d7 */ { "RST 10",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd d8 */ { "RET C",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd d9 */ { "EXX",        ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd da */ { "JP C,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd db */ { "INA",        ASM_ADDR_MODE_ZERO_PAGE | ASM_ADDR_MODE_UNDOC },
    /* fd dc */ { "CALL C,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd dd */ { "NOP",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd de */ { "SBC",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd df */ { "RST 18",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd e0 */ { "RET PO",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd e1 */ { "POP",        ASM_ADDR_MODE_REG_IY },
    /* fd e2 */ { "JP PO,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd e3 */ { "EX (SP),IY", ASM_ADDR_MODE_IMPLIED },
    /* fd e4 */ { "CALL PO,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd e5 */ { "PUSH",       ASM_ADDR_MODE_REG_IY },
    /* fd e6 */ { "AND",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd e7 */ { "RST 20",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC|ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd e8 */ { "RET PE",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd e9 */ { "JP (IY)",    ASM_ADDR_MODE_IMPLIED },
    /* fd ea */ { "JP PE,",     ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd eb */ { "EX DE,HL",   ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd ec */ { "CALL PE,",   ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd ed */ { "PFX_FDED",   ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd ee */ { "XOR",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd ef */ { "RST 28",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd f0 */ { "RET P",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd f1 */ { "POP",        ASM_ADDR_MODE_REG_AF | ASM_ADDR_MODE_UNDOC },
    /* fd f2 */ { "JP P,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd f3 */ { "DI",         ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd f4 */ { "CALL P,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd f5 */ { "PUSH",       ASM_ADDR_MODE_REG_AF | ASM_ADDR_MODE_UNDOC },
    /* fd f6 */ { "OR",         ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd f7 */ { "RST 30",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd f8 */ { "RET M",      ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd f9 */ { "LD SP,IY",   ASM_ADDR_MODE_IMPLIED },
    /* fd fa */ { "JP M,",      ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd fb */ { "EI",         ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC },
    /* fd fc */ { "CALL M,",    ASM_ADDR_MODE_ABSOLUTE | ASM_ADDR_MODE_UNDOC },
    /* fd fd */ { "NOP",        ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd fe */ { "CP",         ASM_ADDR_MODE_IMMEDIATE | ASM_ADDR_MODE_UNDOC },
    /* fd ff */ { "RST 38",     ASM_ADDR_MODE_IMPLIED | ASM_ADDR_MODE_UNDOC }
};

/* (FD CB) IY BIT instructions */
static const asm_opcode_info_t opcode_list_fd_cb[] = {
    /* fd cb .. 00 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 01 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 02 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 03 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 04 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 05 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 06 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 07 */ { "RLC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 08 */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 09 */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 0a */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 0b */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 0c */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 0d */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 0e */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 0f */ { "RRC",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 10 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 11 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 12 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 13 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 14 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 15 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 16 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 17 */ { "RL",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 18 */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 19 */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 1a */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 1b */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 1c */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 1d */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 1e */ { "RR",      ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 1f */ { "RR",      ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 20 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 21 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 22 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 23 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 24 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 25 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 26 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 27 */ { "SLA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 28 */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 29 */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 2a */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 2b */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 2c */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 2d */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 2e */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 2f */ { "SRA",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 30 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 31 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 32 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 33 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 34 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 35 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 36 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 37 */ { "SLL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 38 */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 39 */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 3a */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 3b */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 3c */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 3d */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 3e */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 3f */ { "SRL",     ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 40 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 41 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 42 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 43 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 44 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 45 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 46 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 47 */ { "BIT 0,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 48 */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 49 */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 4a */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 4b */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 4c */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 4d */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 4e */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 4f */ { "BIT 1,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 50 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 51 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 52 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 53 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 54 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 55 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 56 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 57 */ { "BIT 2,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 58 */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 59 */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 5a */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 5b */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 5c */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 5d */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 5e */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 5f */ { "BIT 3,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 60 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 61 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 62 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 63 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 64 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 65 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 66 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 67 */ { "BIT 4,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 68 */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 69 */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 6a */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 6b */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 6c */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 6d */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 6e */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 6f */ { "BIT 5,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 70 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 71 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 72 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 73 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 74 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 75 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 76 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 77 */ { "BIT 6,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 78 */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 79 */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 7a */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 7b */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 7c */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 7d */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 7e */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY },
    /* fd cb .. 7f */ { "BIT 7,",  ASM_ADDR_MODE_REG_IND_IY | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 80 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 81 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 82 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 83 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 84 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 85 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 86 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 87 */ { "RES 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 88 */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 89 */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 8a */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 8b */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 8c */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 8d */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 8e */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 8f */ { "RES 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 90 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 91 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 92 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 93 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 94 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 95 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 96 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 97 */ { "RES 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 98 */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 99 */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 9a */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 9b */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 9c */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 9d */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. 9e */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. 9f */ { "RES 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a0 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a1 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a2 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a3 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a4 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a5 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a6 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. a7 */ { "RES 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a8 */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. a9 */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. aa */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ab */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ac */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ad */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ae */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. af */ { "RES 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b0 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b1 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b2 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b3 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b4 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b5 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b6 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. b7 */ { "RES 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b8 */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. b9 */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ba */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. bb */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. bc */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. bd */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. be */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. bf */ { "RES 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c0 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c1 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c2 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c3 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c4 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c5 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c6 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. c7 */ { "SET 0,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c8 */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. c9 */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ca */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. cb */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. cc */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. cd */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ce */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. cf */ { "SET 1,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d0 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d1 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d2 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d3 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d4 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d5 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d6 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. d7 */ { "SET 2,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d8 */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. d9 */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. da */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. db */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. dc */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. dd */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. de */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. df */ { "SET 3,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e0 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e1 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e2 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e3 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e4 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e5 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e6 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. e7 */ { "SET 4,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e8 */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. e9 */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ea */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. eb */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ec */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ed */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. ee */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. ef */ { "SET 5,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f0 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f1 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f2 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f3 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f4 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f5 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f6 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. f7 */ { "SET 6,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f8 */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. f9 */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. fa */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. fb */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. fc */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. fd */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
    /* fd cb .. fe */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG },
    /* fd cb .. ff */ { "SET 7,",  ASM_ADDR_MODE_IND_IY_REG | ASM_ADDR_MODE_UNDOC },
};

static const asm_opcode_info_t *asm_opcode_info_get(unsigned int p0, unsigned int p1, unsigned int p2, unsigned int p3)
{
    if (p0 == 0xcb) {
        return opcode_list_cb + p1;
    }
    if (p0 == 0xdd) {       /* IX instructions */
        if (p1 == 0xcb) {   /* IX BIT instructions */
            return opcode_list_dd_cb + p3;
        }
        return opcode_list_dd + p1;
    }
    if (p0 == 0xed) {
        return opcode_list_ed + p1;
    }
    if (p0 == 0xfd) {       /* IY instructions */
        if (p1 == 0xcb) {   /* IY BIT instructions */
            return opcode_list_fd_cb + p3;
        }
        return opcode_list_fd + p1;
    }
    return opcode_list + p0;
}

/* must return a positive number (opcode length in bytes) */
static unsigned int asm_addr_mode_get_size(unsigned int mode, unsigned int p0, unsigned int p1, unsigned int p2, unsigned int p3)
{
#if 0
    if (addr_mode_size[mode] < 1) {
        log_error(LOG_DEFAULT, "internal error: asm_addr_mode_get_size size in table is < 1");
        archdep_vice_exit(EXIT_FAILURE);
    }
#endif
    if (p0 == 0xcb) {
        return addr_mode_size[mode] + 1;
    }
    if (p0 == 0xdd) {       /* IX instructions */
        if (p1 == 0xcb) {   /* IX BIT instructions */
            return 4; /* addr_mode_size[mode] + 2; */
        }
        return addr_mode_size[mode] + 1;
    }
    if (p0 == 0xed) {
        return addr_mode_size[mode] + 1;
    }
    if (p0 == 0xfd) {       /* IY instructions */
        if (p1 == 0xcb) {   /* IY BIT instructions */
            return 4; /* addr_mode_size[mode] + 2; */
        }
        return addr_mode_size[mode] + 1;
    }
    return addr_mode_size[mode];
}

void asmz80_init(monitor_cpu_type_t *monitor_cpu_type)
{
    monitor_cpu_type->cpu_type = CPU_Z80;
    monitor_cpu_type->asm_addr_mode_get_size = asm_addr_mode_get_size;
    monitor_cpu_type->asm_opcode_info_get = asm_opcode_info_get;

    /* Once we have a generic processor specific init, this will move.  */
    mon_assemblez80_init(monitor_cpu_type);
    mon_registerz80_init(monitor_cpu_type);
}
