use crate::gobject_models::GTag;
use crate::util::{GtkUtil, constants};
use glib::{Properties, clone, subclass::*};
use gtk4::{CompositeTemplate, DrawingArea, Image, Label, prelude::*, subclass::prelude::*};
use std::cell::RefCell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::PopoverTagRow)]
    #[template(file = "data/resources/ui_templates/tagging/row.blp")]
    pub struct PopoverTagRow {
        #[template_child]
        pub tag_title: TemplateChild<Label>,
        #[template_child]
        pub tag_color: TemplateChild<DrawingArea>,
        #[template_child]
        pub image: TemplateChild<Image>,

        #[property(get, set = Self::set_tag)]
        pub tag: RefCell<GTag>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for PopoverTagRow {
        const NAME: &'static str = "PopoverTagRow";
        type ParentType = gtk4::Box;
        type Type = super::PopoverTagRow;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for PopoverTagRow {
        fn constructed(&self) {
            self.tag_color.set_draw_func(clone!(
                #[weak(rename_to = imp)]
                self,
                move |_drawing_area, ctx, _width, _height| {
                    GtkUtil::draw_color_cirlce(
                        ctx,
                        imp.tag
                            .borrow()
                            .color()
                            .as_deref()
                            .unwrap_or(constants::TAG_DEFAULT_COLOR),
                        None,
                    );
                }
            ));
        }
    }

    impl WidgetImpl for PopoverTagRow {}

    impl BoxImpl for PopoverTagRow {}

    impl PopoverTagRow {
        fn set_tag(&self, tag: GTag) {
            self.tag.replace(tag);
            self.tag_color.queue_draw();
        }
    }
}

glib::wrapper! {
    pub struct PopoverTagRow(ObjectSubclass<imp::PopoverTagRow>)
        @extends gtk4::Widget, gtk4::Box;
}

impl Default for PopoverTagRow {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}
