mod feed_list;
mod list_revealer;
mod loader;
mod sidebar_selection;
mod special_sidebar_rows;
mod tag_list;

use crate::app::App;
use crate::content_page::{ArticleListColumn, ContentPage, SidebarColumn};
use crate::gobject_models::{GSidebarSelection, SidebarSelectionType};
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use crate::settings::GFeedOrder;
use crate::util::GtkUtil;
pub use feed_list::FeedList;
pub use feed_list::{FeedListItem, FeedListItemID, FeedListTree};
use glib::{ControlFlow, Properties, SourceId, clone, subclass};
use gtk4::{Box, CompositeTemplate, ScrolledWindow, TickCallbackId, Widget, prelude::*, subclass::prelude::*};
use list_revealer::ListRevealer;
pub use loader::SidebarLoader;
use news_flash::error::NewsFlashError;
pub use sidebar_selection::SidebarSelection;
use special_sidebar_rows::{SidebarSpecialRowType, SpecialSidebarRows};
use std::cell::{Cell, RefCell};
use std::time::Duration;
use tag_list::TagList;
pub use tag_list::TagListModel;

mod imp {
    use super::*;

    #[derive(Debug, CompositeTemplate, Default, Properties)]
    #[properties(wrapper_type = super::SideBar)]
    #[template(file = "data/resources/ui_templates/sidebar/sidebar.blp")]
    pub struct SideBar {
        #[template_child]
        pub sidebar_scroll: TemplateChild<ScrolledWindow>,
        #[template_child]
        pub special_rows: TemplateChild<SpecialSidebarRows>,
        #[template_child]
        pub feed_list: TemplateChild<FeedList>,
        #[template_child]
        pub tag_list: TemplateChild<TagList>,

        #[property(get, set, name = "feedlist-expanded")]
        feedlist_expanded: Cell<bool>,

        #[property(get, set, name = "taglist-expanded")]
        taglist_expanded: Cell<bool>,

        #[property(get, set, name = "taglist-visible")]
        taglist_visible: Cell<bool>,

        #[property(get, set = Self::set_selection)]
        selection: RefCell<GSidebarSelection>,

        pub delay_next_activation: Cell<bool>,
        pub delayed_selection: RefCell<Option<SourceId>>,

        pub drag_auto_scroll: RefCell<Option<TickCallbackId>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for SideBar {
        const NAME: &'static str = "SideBar";
        type ParentType = Box;
        type Type = super::SideBar;

        fn class_init(klass: &mut Self::Class) {
            ListRevealer::ensure_type();

            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for SideBar {}

    impl WidgetImpl for SideBar {}

    impl BoxImpl for SideBar {}

    #[gtk4::template_callbacks]
    impl SideBar {
        #[template_callback]
        fn on_list_activated(&self) {
            let article_list_column = ArticleListColumn::instance();
            article_list_column.set_show_sidebar(true);
            article_list_column.new_list();
        }

        #[template_callback]
        fn on_special_rows_selection_changed(&self, _special_rows: &SpecialSidebarRows, selection: GSidebarSelection) {
            self.obj().set_selection(selection);
        }

        #[template_callback]
        fn on_feed_list_selection_changed(&self, _feed_list: &FeedList, selection: GSidebarSelection) {
            self.obj().set_selection(selection);
        }

        #[template_callback]
        fn on_tag_list_selection_changed(&self, _tag_list: &TagList, selection: GSidebarSelection) {
            self.obj().set_selection(selection);
        }

        #[template_callback]
        fn on_drop_motion(&self, _x: f64, y: f64) {
            const AUTO_SCROLL_TICK: f64 = 5.0;

            if y < 60.0 {
                self.start_auto_scroll(-AUTO_SCROLL_TICK);
            } else if y > self.obj().height() as f64 - 60.0 {
                self.start_auto_scroll(AUTO_SCROLL_TICK);
            } else if let Some(source_id) = self.drag_auto_scroll.take() {
                source_id.remove();
            }
        }

        #[template_callback]
        fn on_drop_leave(&self) {
            if let Some(source_id) = self.drag_auto_scroll.take() {
                source_id.remove();
            };
        }

        #[template_callback]
        fn on_drag_end(&self) {
            if let Some(source_id) = self.drag_auto_scroll.take() {
                source_id.remove();
            };
        }

        fn set_selection(&self, selection: GSidebarSelection) {
            if self.selection.borrow().eq(&selection) {
                return;
            }

            self.selection.replace(selection.clone());

            match selection.selection_type() {
                SidebarSelectionType::None => {}
                SidebarSelectionType::All => {
                    self.feed_list.clear_selection();
                    self.tag_list.clear_selection();
                    self.special_rows.select(SidebarSpecialRowType::All);

                    self.maybe_delayed_activation(|imp| imp.special_rows.activate(SidebarSpecialRowType::All));
                }
                SidebarSelectionType::Today => {
                    self.feed_list.clear_selection();
                    self.tag_list.clear_selection();
                    self.special_rows.select(SidebarSpecialRowType::Today);

                    self.maybe_delayed_activation(|imp| imp.special_rows.activate(SidebarSpecialRowType::Today));
                }
                SidebarSelectionType::FeedList => {
                    self.obj().set_feedlist_expanded(true);

                    self.special_rows.clear_selection();
                    self.tag_list.clear_selection();
                    self.feed_list.set_selection(selection.position());

                    self.maybe_delayed_activation(|imp| imp.feed_list.activate());
                }
                SidebarSelectionType::TagList => {
                    self.obj().set_taglist_expanded(true);

                    self.special_rows.clear_selection();
                    self.feed_list.clear_selection();
                    self.tag_list.set_selection(selection.position());

                    self.maybe_delayed_activation(|imp| imp.tag_list.activate());
                }
            }
        }

        pub(super) fn delay_next_activation(&self) {
            GtkUtil::remove_source(self.delayed_selection.take());
            self.delay_next_activation.set(true);
        }

        fn maybe_delayed_activation<F: Fn(&Self) + 'static>(&self, f: F) {
            if self.delay_next_activation.get() {
                self.delay_next_activation.set(false);
                GtkUtil::remove_source(self.delayed_selection.take());

                let source_id = glib::timeout_add_local(
                    Duration::from_millis(300),
                    clone!(
                        #[weak(rename_to = imp)]
                        self,
                        #[upgrade_or_panic]
                        move || {
                            imp.delayed_selection.take();
                            f(&imp);
                            ControlFlow::Break
                        }
                    ),
                );

                self.delayed_selection.replace(Some(source_id));
            } else {
                f(self);
            }
        }

        fn start_auto_scroll(&self, diff: f64) {
            if self.drag_auto_scroll.borrow().is_none() {
                let tick_id = self.sidebar_scroll.add_tick_callback(move |scroll, _clock| {
                    let adjustment = scroll.vadjustment();
                    adjustment.set_value(adjustment.value() + diff);
                    ControlFlow::Continue
                });
                self.drag_auto_scroll.replace(Some(tick_id));
            }
        }
    }
}

glib::wrapper! {
    pub struct SideBar(ObjectSubclass<imp::SideBar>)
        @extends Widget, Box;
}

impl Default for SideBar {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl SideBar {
    pub fn instance() -> Self {
        SidebarColumn::instance().imp().sidebar.get()
    }

    pub fn update(
        &self,
        all_item_count: u32,
        today_count: u32,
        feed_list_model: FeedListTree,
        tag_list_model: TagListModel,
    ) {
        self.set_taglist_visible(!tag_list_model.is_empty());

        let imp = self.imp();
        imp.special_rows.set_all_item_count(all_item_count);
        imp.special_rows.set_today_item_count(today_count);
        imp.feed_list.update(feed_list_model);
        imp.tag_list.update(tag_list_model);
    }

    pub fn select_next_item(&self) {
        let imp = self.imp();
        let selection = self.selection();
        imp.delay_next_activation();

        let selection = match selection.selection_type() {
            SidebarSelectionType::None => None,
            SidebarSelectionType::All => Some(GSidebarSelection::today()),
            SidebarSelectionType::Today => imp.feed_list.first_item(),
            SidebarSelectionType::FeedList => {
                let feed_list_next = imp.feed_list.next_item();

                if feed_list_next.is_some() {
                    feed_list_next
                } else if self.taglist_visible() {
                    imp.tag_list.first_item()
                } else {
                    None
                }
            }
            SidebarSelectionType::TagList => imp.tag_list.next_item(),
        };

        let selection = selection.unwrap_or_else(GSidebarSelection::all);
        self.set_selection(selection);
    }

    pub fn select_prev_item(&self) {
        let imp = self.imp();
        let selection = self.selection();
        imp.delay_next_activation();

        let selection = match selection.selection_type() {
            SidebarSelectionType::None => None,
            SidebarSelectionType::All => {
                if self.taglist_visible() {
                    imp.tag_list.last_item()
                } else {
                    imp.feed_list.last_item()
                }
            }
            SidebarSelectionType::Today => Some(GSidebarSelection::all()),
            SidebarSelectionType::FeedList => {
                let feed_list_prev = imp.feed_list.prev_item();

                if feed_list_prev.is_some() {
                    feed_list_prev
                } else {
                    Some(GSidebarSelection::today())
                }
            }
            SidebarSelectionType::TagList => {
                let tag_list_prev = imp.tag_list.prev_item();

                if tag_list_prev.is_some() {
                    tag_list_prev
                } else {
                    imp.feed_list.last_item()
                }
            }
        };

        let selection = selection.unwrap_or_else(GSidebarSelection::all);
        self.set_selection(selection);
    }

    pub fn set_order(order: GFeedOrder) {
        TokioRuntime::execute_with_callback(
            move || async move {
                if order != GFeedOrder::Alphabetical {
                    return Ok(());
                }

                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash.sort_alphabetically().await
            },
            move |res| {
                if let Err(error) = res {
                    ContentPage::instance().newsflash_error(&i18n("Failed to set feed list order to manual"), error);
                    return;
                }

                ContentPage::instance().update_sidebar();
            },
        );
    }
}
