use super::theme_tile::ThemeTile;
use crate::app::App;
use crate::i18n::i18n;
use crate::settings::article_list::{GArticleOrder, GOrderBy};
use crate::settings::feed_list::GFeedOrder;
use crate::themes::NewsflashTheme;
use glib::{Properties, RustClosure, prelude::*, subclass};
use gtk4::pango::FontDescription;
use gtk4::{
    ClosureExpression, CompositeTemplate, FlowBox, FlowBoxChild, FontDialogButton, Widget, prelude::*,
    subclass::prelude::*,
};
use libadwaita::{
    ActionRow, ComboRow, EnumListItem, EnumListModel, PreferencesPage, SpinRow, SwitchRow, prelude::*,
    subclass::prelude::*,
};
use news_flash::models::{ArticleOrder, OrderBy};
use std::cell::Cell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::SettingsViewsPage)]
    #[template(file = "data/resources/ui_templates/settings/views.blp")]
    pub struct SettingsViewsPage {
        #[template_child]
        pub themes: TemplateChild<FlowBox>,
        #[template_child]
        pub feed_order_row: TemplateChild<ComboRow>,
        #[template_child]
        pub feed_list_filter_row: TemplateChild<SwitchRow>,
        #[template_child]
        pub article_order_row: TemplateChild<ComboRow>,
        #[template_child]
        pub article_order_by_row: TemplateChild<ComboRow>,
        #[template_child]
        pub article_list_hide_future_row: TemplateChild<SwitchRow>,
        #[template_child]
        pub article_list_thumbs_row: TemplateChild<SwitchRow>,
        #[template_child]
        pub content_width_spin_row: TemplateChild<SpinRow>,
        #[template_child]
        pub line_height_spin_row: TemplateChild<SpinRow>,
        #[template_child]
        pub font_row: TemplateChild<ActionRow>,
        #[template_child]
        pub font_button: TemplateChild<FontDialogButton>,

        #[property(get, set, name = "use-custom-font")]
        pub use_custom_font: Cell<bool>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for SettingsViewsPage {
        const NAME: &'static str = "SettingsViewsPage";
        type ParentType = PreferencesPage;
        type Type = super::SettingsViewsPage;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for SettingsViewsPage {
        fn constructed(&self) {
            let feed_list_settings = App::default().settings().feed_list();
            let article_list_settings = App::default().settings().article_list();
            let article_view_settings = App::default().settings().article_view();

            // -----------------------------------------------------------------------
            // theme tiles
            // -----------------------------------------------------------------------
            for theme in NewsflashTheme::iterator() {
                let theme_tile = ThemeTile::from(theme);
                self.themes.insert(&theme_tile, -1);
            }

            // -----------------------------------------------------------------------
            // feed list order
            // -----------------------------------------------------------------------
            let params: &[gtk4::Expression] = &[];
            let closure = RustClosure::new(|values| {
                let e = values[0].get::<EnumListItem>().unwrap();
                let feed_order = GFeedOrder::from(e.value());
                let order = feed_order.to_string().to_value();
                Some(order)
            });
            let feed_order_closure = ClosureExpression::new::<String>(params, closure);
            self.feed_order_row.set_expression(Some(&feed_order_closure));
            self.feed_order_row
                .set_model(Some(&EnumListModel::new(GFeedOrder::static_type())));
            self.feed_order_row
                .set_selected(App::default().settings().feed_list().order().into());
            self.feed_order_row.connect_selected_notify(|row| {
                let selected_index = row.selected();
                let new_order: GFeedOrder = selected_index.into();
                App::default().settings().feed_list().set_order(new_order);
            });

            // -----------------------------------------------------------------------
            // feed list show only relevant
            // -----------------------------------------------------------------------
            feed_list_settings
                .bind_property("only-show-relevant", &*self.feed_list_filter_row, "active")
                .bidirectional()
                .sync_create()
                .build();

            // -----------------------------------------------------------------------
            // article list order
            // -----------------------------------------------------------------------
            let params: &[gtk4::Expression] = &[];
            let closure = RustClosure::new(|values| {
                let e = values[0].get::<EnumListItem>().unwrap();
                let article_order: GArticleOrder = e.value().into();
                let article_order: ArticleOrder = article_order.into();
                let order = match article_order {
                    ArticleOrder::NewestFirst => i18n("Newest First"),
                    ArticleOrder::OldestFirst => i18n("Oldest First"),
                }
                .to_value();
                Some(order)
            });
            let article_order_closure = ClosureExpression::new::<String>(params, closure);
            self.article_order_row.set_expression(Some(&article_order_closure));
            self.article_order_row
                .set_model(Some(&EnumListModel::new(GArticleOrder::static_type())));
            self.article_order_row
                .set_selected(App::default().settings().article_list().order().into());
            self.article_order_row.connect_selected_notify(|row| {
                let selected_index = row.selected();
                let new_order: GArticleOrder = selected_index.into();
                App::default().settings().article_list().set_order(new_order);
            });

            // -----------------------------------------------------------------------
            // article list order by
            // -----------------------------------------------------------------------
            let params: &[gtk4::Expression] = &[];
            let closure = RustClosure::new(|values| {
                let e = values[0].get::<EnumListItem>().unwrap();
                let article_order_by: GOrderBy = e.value().into();
                let article_order_by: OrderBy = article_order_by.into();
                let order_by = match article_order_by {
                    OrderBy::Published => i18n("Published"),
                    OrderBy::Updated => i18n("Last Updated"),
                }
                .to_value();
                Some(order_by)
            });
            let article_order_by_closure = ClosureExpression::new::<String>(params, closure);
            self.article_order_by_row
                .set_expression(Some(&article_order_by_closure));
            self.article_order_by_row
                .set_model(Some(&EnumListModel::new(GOrderBy::static_type())));
            self.article_order_by_row
                .set_selected(App::default().settings().article_list().order_by().into());
            self.article_order_by_row.connect_selected_notify(|row| {
                let selected_index = row.selected();
                let new_order_by: GOrderBy = selected_index.into();
                App::default().settings().article_list().set_order_by(new_order_by);
            });

            // -----------------------------------------------------------------------
            // article list show thumbnails
            // -----------------------------------------------------------------------
            article_list_settings
                .bind_property("show-thumbnails", &*self.article_list_thumbs_row, "active")
                .bidirectional()
                .sync_create()
                .build();

            // -----------------------------------------------------------------------
            // article list hide future articles
            // -----------------------------------------------------------------------
            article_list_settings
                .bind_property("hide-future-articles", &*self.article_list_hide_future_row, "active")
                .bidirectional()
                .sync_create()
                .build();

            // -----------------------------------------------------------------------
            // article view content width
            // -----------------------------------------------------------------------
            article_view_settings
                .bind_property("content-width", &*self.content_width_spin_row, "value")
                .bidirectional()
                .sync_create()
                .build();

            // -----------------------------------------------------------------------
            // article view line height
            // -----------------------------------------------------------------------
            article_view_settings
                .bind_property("line-height", &*self.line_height_spin_row, "value")
                .bidirectional()
                .sync_create()
                .build();

            // -----------------------------------------------------------------------
            // article view fonts
            // -----------------------------------------------------------------------
            article_view_settings
                .bind_property("use-custom-font", &*self.obj(), "use-custom-font")
                .bidirectional()
                .sync_create()
                .build();

            article_view_settings
                .bind_property("font", &*self.font_button, "font-desc")
                .transform_from(|_binding, desc: FontDescription| Some(Some(desc.to_str().to_string())))
                .transform_to(|_binding, font: Option<String>| font.as_deref().map(FontDescription::from_string))
                .bidirectional()
                .sync_create()
                .build();
        }
    }

    impl WidgetImpl for SettingsViewsPage {}

    impl PreferencesPageImpl for SettingsViewsPage {}

    #[gtk4::template_callbacks]
    impl SettingsViewsPage {
        #[template_callback]
        fn themes_activated(&self, activated_item: &FlowBoxChild) {
            let mut child = self.themes.first_child();

            while let Some(widget) = child.and_downcast::<FlowBoxChild>() {
                if let Some(tile) = widget.child().and_downcast::<ThemeTile>() {
                    tile.set_selected(false);
                }
                child = widget.next_sibling();
            }

            if let Some(tile) = activated_item.child().and_downcast::<ThemeTile>() {
                tile.set_selected(true);
            }
        }
    }
}

glib::wrapper! {
    pub struct SettingsViewsPage(ObjectSubclass<imp::SettingsViewsPage>)
        @extends Widget, PreferencesPage;
}

impl Default for SettingsViewsPage {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}
