use chrono::{DateTime, Duration, Utc};
use gdk4::Texture;
use glib::Bytes;
use moka::future::Cache;
use news_flash::{NewsFlash, models::FeedID};
use once_cell::sync::OnceCell;
use reqwest::Client;
use std::sync::Arc;
use tokio::sync::RwLock;

#[derive(Debug, Clone)]
pub struct FaviconCacheEntry {
    pub texture: Option<Texture>,
    pub expires: DateTime<Utc>,
}

impl FaviconCacheEntry {
    pub fn is_expired(&self) -> bool {
        Utc::now() >= self.expires
    }
}

static INSTANCE: OnceCell<FaviconCache> = OnceCell::new();

#[derive(Debug, Clone)]
pub struct FaviconCache {
    cache: Cache<FeedID, FaviconCacheEntry>,
}

impl Default for FaviconCache {
    fn default() -> Self {
        Self {
            cache: Cache::new(Self::CACHE_SIZE),
        }
    }
}

impl FaviconCache {
    const CACHE_SIZE: u64 = 99999;
    const EXPIRES_AFTER_DAYS: i64 = 30;

    fn instance() -> &'static FaviconCache {
        INSTANCE.get_or_init(FaviconCache::default)
    }

    pub async fn get_icon(
        feed_id: &FeedID,
        news_flash: Arc<RwLock<Option<NewsFlash>>>,
        client: Client,
    ) -> Option<Texture> {
        Self::instance().get_icon_impl(feed_id, news_flash, client).await
    }

    async fn get_icon_impl(
        &self,
        feed_id: &FeedID,
        news_flash: Arc<RwLock<Option<NewsFlash>>>,
        client: Client,
    ) -> Option<Texture> {
        let init_future = Self::init_icon(feed_id, news_flash, client);
        let entry = self
            .cache
            .entry_by_ref(feed_id)
            .or_insert_with_if(init_future, FaviconCacheEntry::is_expired)
            .await;
        entry.into_value().texture
    }

    async fn init_icon(
        feed_id: &FeedID,
        news_flash: Arc<RwLock<Option<NewsFlash>>>,
        client: Client,
    ) -> FaviconCacheEntry {
        let favicon = if let Some(news_flash) = news_flash.read().await.as_ref() {
            let header_map = crate::app::App::default().settings().get_feed_header_map(feed_id);
            news_flash.get_icon(feed_id, &client, header_map).await.ok()
        } else {
            None
        };

        let expires = favicon
            .as_ref()
            .map(|favicon| favicon.expires)
            .unwrap_or(Utc::now() + Duration::try_days(Self::EXPIRES_AFTER_DAYS).unwrap());

        let bytes = favicon.and_then(|icon| icon.data).map(Bytes::from_owned);
        let texture = bytes.as_ref().and_then(|data| Texture::from_bytes(data).ok());

        FaviconCacheEntry { texture, expires }
    }
}
