package Win32API::Process;

use 5.006;
use strict;
use warnings;

require Exporter;
require DynaLoader;

our @ISA = qw(Exporter DynaLoader);

our $VERSION = '0.01';

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Win32API::Process ':All';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = (
	'Func' => [ qw(
		CloseProcess
		GetLastProcessError
		OpenProcess
		SetLastProcessError
		TerminateProcess
	) ],
	'PROCESS_' => [ qw(
		PROCESS_TERMINATE
		PROCESS_CREATE_THREAD
		PROCESS_SET_SESSIONID
		PROCESS_VM_OPERATION
		PROCESS_VM_READ
		PROCESS_VM_WRITE
		PROCESS_DUP_HANDLE
		PROCESS_CREATE_PROCESS
		PROCESS_SET_QUOTA
		PROCESS_SET_INFORMATION
		PROCESS_QUERY_INFORMATION
		PROCESS_SUSPEND_RESUME
		PROCESS_ALL_ACCESS
		STANDARD_RIGHTS_REQUIRED
		SYNCHRONIZE
	) ]
);

my @EXPORT_ALL = ();
foreach my $ref (values %EXPORT_TAGS) {
	push @EXPORT_ALL, @$ref;
}
$EXPORT_TAGS{'All'} = [ @EXPORT_ALL ];

our @EXPORT_OK = ( @{$EXPORT_TAGS{'All'}} );

our @EXPORT = qw();

bootstrap Win32API::Process $VERSION;

# Preloaded methods go here.

# generated by: perl -ne "print if /#define PROCESS_/;" <"%MSSdk%\Include\WinNT.h"
# and reformatted by: s/^#define\s+(\w+)\s+([^\s]+)[ \t]*.*$/sub \1\t{ \2 }/"

sub STANDARD_RIGHTS_REQUIRED	{ 0x000F0000 }
sub SYNCHRONIZE			{ 0x00100000 }

sub PROCESS_TERMINATE		{ 0x0001 }
sub PROCESS_CREATE_THREAD	{ 0x0002 }
sub PROCESS_SET_SESSIONID	{ 0x0004 }
sub PROCESS_VM_OPERATION	{ 0x0008 }
sub PROCESS_VM_READ		{ 0x0010 }
sub PROCESS_VM_WRITE		{ 0x0020 }
sub PROCESS_DUP_HANDLE		{ 0x0040 }
sub PROCESS_CREATE_PROCESS	{ (0x0080) }
sub PROCESS_SET_QUOTA		{ (0x0100) }
sub PROCESS_SET_INFORMATION	{ (0x0200) }
sub PROCESS_QUERY_INFORMATION	{ (0x0400) }
sub PROCESS_SUSPEND_RESUME	{ (0x0800) }
sub PROCESS_ALL_ACCESS		{ STANDARD_RIGHTS_REQUIRED | SYNCHRONIZE | 0xFFF }

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

=head1 NAME

Win32API::Process - Perl extension for handling the processes
                    using the plain Win32 API

=head1 SYNOPSIS

  use Win32API::Process ':All';

  # --- opens handle to the current process
  my $handle = OpenProcess(PROCESS_ALL_ACCESS, 0, $$);
  CloseProcess($handle);

=head1 DESCRIPTION

The I<Process> functions exposes the part of Win32 API handling the processes.

It contains just process opening an terminating functionality
in the meanwhile. It is supposed to be used
with the C<Win32API::ProcessStatus> module or modules that need handles
to processes on their interfaces.

(Note that much of the following documentation refers to the behavior
of the underlying Win32 API calls which may vary in its current and future
versions without any changes to this module. Therefore you should check the
Win32 API documentation in MSDN directly when needed.)

=head2 EXPORTS

Nothing is exported by default. The following tags can be used to have sets
of symbols exported:

=over

=item :Func

The basic function names: CloseProcess GetLastProcessError OpenProcess
SetLastProcessError TerminateProcess.

=back

=head2 CONSTANTS

The constants described here and exported by this module are used only
in its structures and functions. See the module C<Win32API::Const>
for the common Win32 constants.

C<PROCESS_> constants are used as flags in the process handlingopening functions.

=over

=item PROCESS_ALL_ACCESS

Specifies all possible access flags for the process object.

=item PROCESS_CREATE_PROCESS

Used internally.

=item PROCESS_CREATE_THREAD

Enables using the process handle in the C<CreateRemoteThread> function
to create a thread in the process.

=item PROCESS_DUP_HANDLE

Enables using the process handle as either the source or target process
in the C<DuplicateHandle> function to duplicate a handle.

=item PROCESS_QUERY_INFORMATION

Enables using the process handle in the C<GetExitCodeProcess>
and C<GetPriorityClass> functions to read information from the process object.

=item PROCESS_SET_QUOTA

Enables using the process handle in the C<AssignProcessToJobObject>
and C<SetProcessWorkingSetSize> functions to set memory limits.

=item PROCESS_SET_INFORMATION

Enables using the process handle in the C<SetPriorityClass> function
to set the priority class of the process.

=item PROCESS_TERMINATE

Enables using the process handle in the C<TerminateProcess> function
to terminate the process.

=item PROCESS_VM_OPERATION

Enables using the process handle in the C<VirtualProtectEx>
and C<WriteProcessMemory> functions to modify the virtual memory
of the process.

=item PROCESS_VM_READ

Enables using the process handle in the C<ReadProcessMemory> function
to read from the virtual memory of the process.

=item PROCESS_VM_WRITE

Enables using the process handle in the C<WriteProcessMemory> function
to write to the virtual memory of the process.

=item STANDARD_RIGHTS_REQUIRED

Combines C<DELETE>, C<READ_CONTROL>, C<WRITE_DAC>, and C<WRITE_OWNER> access.

=item SYNCHRONIZE

Windows NT/2000/XP: Enables using the process handle in any of the wait
functions to wait for the process to terminate.

=back

=head2 FUNCTIONS

I<Process> functions return either a boolean status of the function's
result or a handle to a process. To retrieve an extended information
about the error if it occurs use the C<GetLastProcessError> function.
If no error happens C<GetLastProcessError> still returns the last occured
error code (successful calls do not modify the last stored error code).
You can set or reset the internally stored error code explicitely
by the function C<SetLastProcessError>.

To use something more convenient than numbers for comparisons of return
values and error codes see the module C<Win32API::Const>.

=over

=item CloseProcess($hProcess)

Closes an open process handle.

=over

=item hProcess [IN]

Handle to the open process.

=item [RETVAL]

If the function succeeds, the return value is nonzero. If the function fails,
the return value is zero. To get extended error information, call
C<GetLastProcessError>.

Windows NT/2000/XP: Closing an invalid handle raises an exception
when the application is running under a debugger. This includes closing
a handle twice, for example.

=back

It invalidates the specified process handle, decrements the object's
handle count, and performs object retention checks. After the last handle
to an object is closed, the object is removed from the system.

=item GetLastProcessError()

Retrieves the last-error code value of the I<Process> functions.
The last-error code is stored if a function fails and remembered until
another function calls when it is overwritten by the new error code.
Successful calls do not modify this internally stored last-error code value.

=over

=item [RETVAL]

The return value is the last-error code value. Functions set this value
by calling the C<SetLastProcessError> function if they fail.

=back

To obtain an error string for system error codes, use
the C<FormatMessage> function. For a complete list of error codes, see
the System Error Codes section in MSDN. There are pre-defined constants
for the Win32 system error codes in the module <Win32API::Const>.

You should call the C<GetLastProcessError> function immediately when
a function's return value indicates that such a call will return useful data.
A subsequent call to another I<Process> function could fail as well
and C<GetLastProcessError> would return its error code instead
of the former one.

Function failure is typically indicated by a return value such as zero,
undefined, or 1 (0xffffffff).

Error codes returned are 32-bit values with the most significant bit set
to 1 (bit 31 is the most significant bit). Zero code is C<ERROR_SUCCESS>.

=item OpenProcess($dwDesiredAccess, $bInheritHandle, $dwProcessId)

Takes a snapshot of the processes and the heaps, modules, and threads used
by the processes.

=over

=item dwDesiredAccess [IN]

Specifies the access to the process object. For operating systems that support
security checking, this access is checked against any security descriptor
for the target process. The values available are listed in the description of
C<PROCESS__> constants.

=item bInheritHandle [IN]

Specifies whether the returned handle can be inherited by a new process
created by the current process. If TRUE, the handle is inheritable.

=item dwProcessId [IN]

Specifies the identifier of the process to open.

=item [RETVAL]

If the function succeeds, the return value is an open handle to the specified
process. If the function fails, the return value is NULL. To get extended
error information, call C<GetLastProcessError>.

=back

The handle returned by the C<OpenProcess> function can be used in any function
that requires a handle to a process, such as the wait functions, provided
the appropriate access rights were requested.

When you are finished with the handle, be sure to close it using
the C<CloseProcess> or C<CloseHandle> function.

=item SetLastProcessError($dwError)

Sets the last-error code value of the I<Process> functions.

=over

=item dwError [IN]

Specifies the last-error code.

=back

Error codes returned are 32-bit values with the most significant bit set
to 1 (bit 31 is the most significant bit). Zero code is C<ERROR_SUCCESS>.

Applications can retrieve the value saved by this function by using
the C<GetLastProcessError> function. The use of C<GetLastProcessError>
is optional; an application can call it to find out the specific reason
for a function failure.

=item TerminateProcess($hProcess, $uExitCode)

Terminates the specified process and all of its threads.

=over

=item hProcess [IN]

Handle to the process to terminate. 

=item uExitCode [IN]

Specifies the exit code for the process and for all threads terminated
as a result of this call. Use the C<GetExitCodeProcess> function to retrieve
the process's exit value. Use the C<GetExitCodeThread> function to retrieve
a thread's exit value.

=item [RETVAL]

If the function succeeds, the return value is nonzero. If the function fails,
the return value is zero. To get extended error information, call
C<GetLastProcessError>.

=back

The C<TerminateProcess> function is used to unconditionally cause a process
to exit. Use it only in extreme circumstances. The state of global data
maintained by dynamic-link libraries (DLLs) may be compromised if
C<TerminateProcess> is used rather than C<ExitProcess>.

C<TerminateProcess> causes all threads within a process to terminate,
and causes a process to exit, but DLLs attached to the process are
not notified that the process is terminating.

Terminating a process causes the following:

=over

All of the object handles opened by the process are closed.

All of the threads in the process terminate their execution.

The state of the process object becomes signaled, satisfying any threads
that had been waiting for the process to terminate.

The states of all threads of the process become signaled, satisfying
any threads that had been waiting for the threads to terminate.

The termination status of the process changes from C<STILL_ACTIVE> to the exit
value of the process.

=back

Terminating a process does not cause child processes to be terminated.

Terminating a process does not necessarily remove the process object
from the system. A process object is deleted when the last handle
to the process is closed.

Terminating a process does not generate notifications for C<WH_CBT> hook
procedures.

=back

=head1 AUTHOR

Ferdinand Prantl E<lt>F<prantl@host.sk>E<gt>

See F<http://prantl.host.sk/perl/modules/Win32API/Process>
for the most recent version.

=head1 COPYRIGHT

Copyright (c) 2002, Ferdinand Prantl. All rights reserved.

Permission to use, copy, modify, distribute and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation. Author makes no representations
about the suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty.

=head1 SEE ALSO

L<Win32API::ProcessStatus>, L<Win32::Process>, L<Win32::Job>
and L<Win32API::Const>.

=cut
