#!/usr/bin/env perl

#ABSTRACT: script to send ossec alerts through jabber as an active response
#PODNAME: ossec-jabber.pl
use strict;
use warnings;
use File::Basename;
use OSSEC;
use Try::Tiny;
use XML::LibXML;
use Pod::Usage;

if($ARGV[0] eq "--help" || $ARGV[0] eq "-h")
{
  pod2usage(-verbose => 2);
  exit(1);
}

my $ALERTID = $ARGV[3];

my $ossec  = OSSEC->new();
my $log    = $ossec->arLog();
my $mysql  = $ossec->mysql();

#
# load configuration file if exist
#
if (! -e $ossec->ossecPath() . "/etc/jabber.conf")
{
  $log->fatal("ossec-jabber","no \"" .  $ossec->ossecPath() . "/etc/jabber.conf\" found");
}


#
# at the moment we rely on the alerts being in a mysql database
#
if (!$ossec->mysqlAvail())
{
  $log->fatal("ossec-jabber","no mysql database configuration found");
}

open(my $fh, '<', $ossec->ossecPath() ."/etc/jabber.conf");
binmode $fh;
my $config = XML::LibXML->load_xml(IO => $fh);
close $fh;

#fetch options
my $options="";
my @options_nodes = $config->getElementsByTagName("options");
if (@options_nodes)
{
  $options = $options_nodes[0]->textContent;
}

$options=~s/^\s+//;
$options=~s/\n//;

#fetch $recipients
my @recipient_nodes = $config->getElementsByTagName("recipient");
my @recipients;
if (@recipient_nodes)
{
  my $r = $recipient_nodes[0]->textContent;
  $r =~s/\n//g;
  push(@recipients,$r);
}

#wait for mysql update from ossec
#sometimes updating the mysql database requires some time because of OSSEC internals
sleep 3;

my $alert;
try {
  $alert  = $mysql->searchAlert($ALERTID);
}catch {
  $log->fatal("ossec-jabber"," alert $ALERTID not found");
};

my $message .= $alert->{timestamp_string} . " on agent " . $alert->{"agent"} . ": " . $alert->{"description"};

for my $r (@recipients)
{
  $log->info("ossec-jabber", "send to $r alert " . $ALERTID . " Description: " . $alert->{description});
  readpipe("echo \" $message \" | sendxmpp $options $r");
}

__END__

=pod

=encoding UTF-8

=head1 NAME

ossec-jabber.pl - script to send ossec alerts through jabber as an active response

=head1 VERSION

version 0.1

=head1 SYNOPSIS

ossec-jabber [action] [username] [srcip] [alertid] [ruleid] [agent name] [filename]
 action:
   -help | -h       brief help message

 please read OSSEC documentation for all the parameters

=head1 DESCRIPTION

B<This program> will search for the given alertid in the OSSEC mysql database and
send it via B<sendxmpp> through a jabber server.

=head1 NAME

ossec-jabber - send OSSEC alerts through jabber (XMPP) using sendxmpp commandline tool

=head1 DEPENDENCIES

=head2 sendxmpp

Please install B<sendxmpp> and provide a working sendxmpp configuration file.

=head1 CONFIGURATION FILE

The configuration file has to be placed under B<OSSEC_BASE>/etc/jabber.conf

=head2 Example

  <config>
    <options>-t</options>
    <recipients>
      <recipient>byterazor@federationhq.de</recipient>
    </recipients>
</config>

=head1 AUTHOR

Domink Meyer <dmeyer@federationhq.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2019 by Dominik Meyer.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
