<?php
/**
 * Zend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @category   Zend
 * @package    Zend_Format
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2005-2014 Zend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @version    $Id$
 */

/**
 * Zend_Locale_Format
 */
require_once 'Zend/Locale/Math.php';
require_once 'Zend/Locale/Math/PhpMath.php';


/**
 * @category   Zend
 * @package    Zend_Locale
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2005-2014 Zend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @group      Zend_Locale
 */
class Zend_Locale_MathTest extends PHPUnit_Framework_TestCase
{
    private static $savedLocale = 'C';

    /**
     * setup for tests (BCMath is not designed to normalize localized numbers)
     */
    public function setUp()
    {
        self::$savedLocale = setlocale(LC_NUMERIC, '0');
        if (self::$savedLocale != 'C') {
            setlocale(LC_NUMERIC, 'C');
        }
    }

    /**
     * teardown for tests (restore whatever setlocale was previously in place)
     */
    public function tearDown()
    {
        if (self::$savedLocale != 'C') {
            setlocale(LC_NUMERIC, self::$savedLocale);
        }
    }

    /*
     * Note: All other aspects of Zend_Locale_Math receive extensive testing
     * via unit tests in Zend_Date and Zend_Measure*.
     */

    /**
     * test round()
     * expect string when BCMath extension is enabled
     */
    public function testRound()
    {
        $this->assertEquals(               '3', Zend_Locale_Math::round('3.4'));
        $this->assertEquals(        round(3.4), Zend_Locale_Math::round('3.4'));
        $this->assertEquals(               '4', Zend_Locale_Math::round('3.5'));
        $this->assertEquals(        round(3.5), Zend_Locale_Math::round('3.5'));
        $this->assertEquals(               '4', Zend_Locale_Math::round('3.6'));
        $this->assertEquals(        round(3.6), Zend_Locale_Math::round('3.6'));
        $this->assertEquals(               '4', Zend_Locale_Math::round('3.6', 0));
        $this->assertEquals(      round(3.6,0), Zend_Locale_Math::round('3.6', 0));
        $this->assertEquals(            '1.96', Zend_Locale_Math::round('1.95583', 2), '', 0.02);
        $this->assertEquals(  round(1.95583,2), Zend_Locale_Math::round('1.95583', 2), '', 0.02);
        $this->assertEquals(           1242000, Zend_Locale_Math::round('1241757', -3), '', 250);
        $this->assertEquals(round(1241757, -3), Zend_Locale_Math::round('1241757', -3), '', 250);
        $this->assertEquals(              5.05, Zend_Locale_Math::round('5.045', 2), '', 0.02);
        $this->assertEquals(   round(5.045, 2), Zend_Locale_Math::round('5.045', 2), '', 0.02);
        $this->assertEquals(              5.06, Zend_Locale_Math::round('5.055', 2), '', 0.02);
        $this->assertEquals(   round(5.055, 2), Zend_Locale_Math::round('5.055', 2), '', 0.02);
    }

    /**
     * test round()
     * expect string when BCMath extension is enabled
     */
    public function testRound2()
    {
        if (!defined('TESTS_ZEND_I18N_EXTENDED_COVERAGE') || TESTS_ZEND_I18N_EXTENDED_COVERAGE == false) {
            $this->markTestSkipped('Extended I18N test skipped');
            return;
        }

        $rounder = array(
            '0.477051'      => -12, '513695.36425'  =>   1, '89.85'         =>   4, '533.506'       =>   3,
            '0.376139'      =>  -2, '1784212419'    =>  -9, '447878.429296' => -11, '79.31'         =>  -2,
            '941.543'       =>  -5, '0.396850'      => -10, '425271509'     =>  10, '220820.93233'  =>  -2,
            '13.36'         =>   4, '138.281'       =>  -5, '0.289925'      =>   2, '1519168597'    =>  12,
            '23387.294481'  =>  -3, '88.100'        =>   1, '19.899'        =>  -1, '0.60083'       =>  11,
            '1807269979'    =>   4, '30808.137760'  =>  -5, '16.63'         =>   0, '682.33'        =>   5,
            '0.127627'      =>  -8, '416684519'     =>   2, '508773.125117' =>  11, '89.39'         =>  -3,
            '803.184'       =>   4, '0.398620'      =>   6, '849182084'     =>  -5, '320473.238127' =>   3,
            '99.3'          =>  -4, '946.410'       =>   4, '0.73955'       =>  10, '1075981461'    =>   4,
            '316913.247175' =>  -4, '65.46'         =>  -3, '299.308'       =>   3, '0.20681'       =>  12,
            '625539733'     => -11, '351290.473192' =>  -2, '0.88'          =>   0, '756.511'       =>   3,
            '0.351428'      =>  -9, '1709881316'    =>   4, '517636.210076' =>  -8, '52.29'         =>   1,
            '96.318'        =>  -5, '0.46457'       =>  -4, '856460724'     =>  -4, '415364.36099'  =>  -3,
            '60.28'         =>   0, '19.312'        =>  -4, '0.84798'       =>   5, '714201053'     =>  12,
            '80610.167713'  =>  11, '88.88'         =>   3, '900.179'       =>   4, '0.120498'      =>  -7,
            '1234420216'    =>  -6, '339890.192448' =>   3, '4.30'          =>   4, '470.236'       =>   1,
            '0.78949'       =>  10, '1930996112'    =>  -3, '186792.27756'  =>   0, '85.69'         =>   2,
            '584.490'       =>   5, '0.419941'      => -10, '819232149'     =>   6, '182934.15610'  =>  -3,
            '97.79'         =>   2, '396.818'       =>  -4, '0.163940'      =>  -2, '1745535129'    =>  12,
            '236683.88660'  =>  -7, '89.11'         =>  -2, '47.522'        =>   0, '0.84264'       =>  12,
            '1473190621'    =>  -4, '366353.18312'  =>   1, '8.53'          =>  -1, '711.629'       =>  -1,
            '0.65511'       =>  -6, '461264361'     =>  11, '80405.349297'  =>  11, '92.100'        =>  -1,
            '466.580'       =>   4, '0.226480'      =>  -3, '1512143953'    =>   6, '148397.211241' => -10,
            '85.23'         =>   3, '663.98'        =>  -5, '0.476893'      =>   5, '1390893395'    =>  -7,
            '84243.419979'  =>  -9, '22.71'         =>  -2, '571.725'       =>   3, '0.498972'      =>  -4,
            '156080671'     =>  -9, '144016.186502' =>   4, '21.91'         =>  -2, '981.325'       =>   1,
            '0.360149'      =>   5, '1973782312'    =>  -7, '188196.41835'  =>  -4, '35.35'         =>  -3,
            '106.431'       =>   1, '0.229880'      =>  -4, '1046448903'    =>  -3, '20467.399497'  =>   0,
            '19.7'          =>  -4, '676.982'       =>   4, '0.195086'      =>  -8, '882554724'     =>   4,
            '1780.403663'   =>  -5, '92.72'         =>  -3, '324.601'       =>   3, '0.343310'      =>  -7,
            '1277914818'    => -12, '288455.332458' =>  -9, '73.9'          =>  -4, '946.170'       =>   3,
            '0.63718'       =>   5, '823941985'     =>   1, '437187.202938' =>   1, '53.22'         =>  -3,
            '2.247'         =>  -5, '0.112341'      =>   3, '1416720336'    =>  10, '23273.110046'  =>   8,
            '63.17'         =>   4, '909.988'       =>  -4, '0.326807'      =>  -5, '1792810450'    => -10,
            '336225.350597' => -12, '88.92'         =>   4, '941.168'       =>   3, '0.294118'      => -11,
            '2032796853'    =>  -3, '100492.519561' =>  -7, '94.5'          =>   1, '179.719'       =>   5,
            '0.247127'      =>  12, '166770266'     =>  -4, '183142.376940' =>  -5, '66.85'         =>   2,
            '645.604'       =>   4, '0.361570'      =>  12, '813314439'     =>  -8, '202182.299055' =>   3,
            '20.57'         =>   3, '548.53'        =>  -5, '0.271132'      =>   6, '166087023'     =>   1,
            '435424.223691' =>   2, '17.98'         =>   3, '805.299'       =>   4, '0.420066'      =>  -2,
            '1294019654'    =>  12, '327747.518105' =>  -8, '81.96'         =>  -2, '972.367'       =>   2,
            '0.370667'      =>   6, '548609247'     =>  10, '251267.45074'  =>   7, '90.38'         =>   3,
            '214.574'       =>   0, '0.342039'      =>   9, '1789816836'    =>  -3, '291202.240426' =>  -6,
            '71.4'          =>  -2, '726.233'       =>   0, '0.239589'      =>  10, '1678730037'    =>  11,
            '380074.136825' =>   4, '5.69'          =>   4, '235.532'       =>   4, '0.41770'       =>  -3,
            '1660444329'    =>  -8, '230203.172296' =>  10, '64.8'          =>   2, '613.456'       =>   4,
            '0.267306'      =>  -9, '150903984'     =>  10, '327734.416916' =>  -5, '73.61'         =>  -1,
            '430.348'       =>  -5, '0.433954'      =>   2, '1640566986'    =>  -2, '178906.106444' =>   3,
            '89.56'         =>   1, '460.408'       =>   1, '0.309152'      => -11, '2111984712'    =>  10,
            '109862.319068' =>   4, '3.18'          =>   2, '234.799'       =>  -4, '0.426879'      =>   0,
            '1979964791'    =>   3, '119122.138007' =>   2, '98.67'         =>  -3, '281.180'       =>   3,
            '0.184761'      =>  -8, '210096321'     =>   7, '225587.161155' =>  -4, '7.98'          =>   3,
            '122.306'       =>  -1, '0.322883'      =>  -7, '1853238528'    =>  -9, '366172.47285'  =>   1,
            '24.78'         =>   0, '549.428'       =>   5, '0.382104'      =>  -9, '1972761165'    =>   7,
            '60782.182930'  =>   8, '23.20'         =>  -4, '642.159'       =>  -2, '0.460457'      =>   3,
            '1572631121'    =>   7, '163894.225827' =>   6, '91.7'          =>  -3, '46.440'        =>  -5,
            '0.101855'      =>   0, '567173959'     =>  -8, '521830.199253' => -10, '44.70'         =>  -3,
            '2.975'         =>   0, '0.322017'      =>   3, '882661445'     =>   9, '311363.379387' =>  -4,
            '84.69'         =>  -1, '744.550'       =>   3, '0.119550'      =>  -8, '723180551'     =>   6,
            '449221.174100' =>   0, '46.31'         =>   3, '790.16'        =>   3, '0.223467'      =>   7,
            '1334151870'    =>   3, '512572.112795' =>   8, '64.40'         =>   2, '126.186'       =>  -4,
            '0.154390'      =>   5, '939837920'     =>  -2, '475325.154386' =>   3, '61.42'         =>   3,
            '219.882'       =>   1, '0.517762'      =>   4, '592690249'     =>  -5, '75942.132984'  =>   3,
            '0.46'          =>   4, '609.230'       =>  -1, '0.158459'      =>   3, '111563587'     =>   1,
            '81793.502563'  =>   6, '74.65'         =>  -3, '275.763'       =>   4, '0.482465'      =>   5,
            '38064166'      =>   6, '188392.85235'  =>  -8, '87.46'         =>  -3, '910.256'       =>   5,
            '0.286740'      =>  -7, '1561805584'    =>   1, '156399.463093' =>   2, '0.29'          =>   2,
            '123.255'       =>   1, '0.437082'      =>  -5, '129042096'     =>  -8, '195012.219896' =>   9,
            '60.89'         =>   0, '855.189'       =>  -3, '0.29867'       =>   0, '139602988'     =>   6,
            '175877.190481' =>   7, '70.67'         =>   4, '216.311'       =>  -2, '0.275937'      =>  11,
            '230515001'     =>  -9, '522334.251290' =>   2, '30.61'         =>  -4, '859.747'       =>   1,
            '0.191629'      =>   7, '1090159168'    =>   8, '308377.442029' =>  -2, '74.49'         =>  -2,
            '166.142'       =>   5, '0.51124'       =>  -7, '815518430'     =>   4, '133830.197147' =>  -7,
            '39.99'         =>  -2, '567.729'       =>  -2, '0.177050'      => -12, '1781759241'    => -10,
            '275013.219089' =>   9, '68.42'         =>  -2, '348.975'       =>   1, '0.305648'      =>   6,
            '882222353'     => -11, '144939.349216' =>  12, '93.73'         =>   4, '3.513'         =>   3,
            '0.15497'       =>   1, '1330004472'    =>   9, '364678.75433'  =>   9, '46.96'         =>  -1,
            '550.156'       =>  -3, '0.153132'      =>  -6, '678841651'     =>   6, '368190.310672' =>   5,
            '55.29'         =>  -4, '92.770'        =>   0, '0.324112'      => -12, '1850715381'    =>   7,
            '345338.292225' =>  -7, '95.85'         =>   4, '114.437'       =>   3, '0.188012'      =>  10,
            '1387145934'    =>   6, '226297.182561' =>   4, '39.82'         =>   3, '319.189'       =>  -4,
            '0.173574'      =>  11, '1556035741'    =>   8, '23778.200941'  =>  -8, '32.69'         =>  -1,
            '441.384'       =>  -3, '0.182397'      =>   4, '796735165'     =>   6, '219063.420812' =>  12,
            '60.73'         =>   2, '887.825'       =>   2, '0.425583'      =>  10, '1497714427'    =>  -9,
            '182965.389431' =>   1, '17.24'         =>   4, '444.232'       =>  -1, '0.45383'       =>   8,
            '1218890312'    =>   2, '300990.516644' =>  -1, '67.87'         =>   4, '563.534'       =>  -4,
            '0.237510'      => -12, '337619655'     =>  -4, '462448.265391' =>  -9, '61.6'          =>   3,
            '503.485'       =>  -1, '0.171549'      =>   3, '448643907'     =>   2, '15061.410523'  =>  10,
            '70.55'         =>  -2, '727.716'       =>  -2, '0.381646'      =>  10, '485956258'     =>  -8,
            '167378.56802'  =>  10, '91.100'        =>  -3, '393.778'       =>   5, '0.11387'       =>  -2,
            '12977184'      => -10, '524273.18229'  =>  -7, '32.71'         =>   3, '59.157'        =>   0,
            '0.492791'      =>  -8, '774557136'     =>  10, '404139.356479' =>  -8, '90.18'         =>  -3,
            '134.552'       =>  -4, '0.298695'      =>  11, '109079452'     =>   2, '421718.26616'  =>   2,
            '94.94'         =>   3, '410.590'       =>  -2, '0.315720'      =>   0, '870440545'     =>   1,
            '365067.92361'  =>  12, '34.54'         =>   4, '703.290'       =>   0, '0.339982'      => -11,
            '1337929149'    =>   9, '228439.224073' =>   5, '24.68'         =>  -4, '42.391'        =>   3,
            '0.294921'      =>  -3, '1925153138'    =>  12, '501140.310787' =>  12, '56.48'         =>   1,
            '980.837'       =>   2, '0.15171'       => -12, '1866705287'    =>   0, '96693.159889'  =>   4,
            '35.32'         =>   4, '101.658'       =>  -2, '0.245009'      =>   6, '1536424140'    =>  -7,
            '98922.351956'  =>  -3, '27.52'         =>  -2, '239.795'       =>  -1, '0.130324'      =>  -6,
            '1558395489'    =>   8, '249345.477161' => -10, '85.78'         =>   0, '105.981'       =>   0,
            '0.434193'      => -10, '728314599'     =>  -1, '425578.276733' =>   1, '86.48'         =>   2,
            '889.528'       =>   2, '0.67736'       =>  -1, '631126221'     =>   7, '468815.403429' =>   9,
            '32.61'         =>  -2, '69.196'        =>  -3, '0.98505'       =>  -8, '1358601844'    => -12,
            '298028.232980' =>   1, '21.92'         =>   3, '591.157'       =>   0, '0.32092'       => -12,
            '1729465213'    =>  -5, '73919.366759'  =>  -4, '1.45'          =>   4, '653.585'       =>  -4,
            '0.439245'      =>  -5, '706568548'     =>   7, '157303.470530' =>  -1, '68.12'         =>  -1,
            '629.862'       =>   5, '0.344800'      => -10, '841981456'     =>  11, '145508.279481' =>  -1,
            '44.18'         =>  -1, '286.504'       =>   5, '0.309732'      =>   1, '671541317'     =>  -4,
            '425949.321253' =>  10, '97.50'         =>   2, '447.483'       =>  -3, '0.289988'      => -10,
            '46304377'      =>  -9, '316706.156813' =>  11, '70.92'         =>   2, '872.232'       =>   1,
            '0.215806'      => -11, '855811504'     =>   8, '389447.110599' =>   6, '94.30'         =>   1,
            '543.505'       =>   3, '0.336457'      =>  -6, '896430071'     => -11, '1879.11273'    =>   4,
            '79.67'         =>   0, '216.271'       =>   3, '0.142853'      =>  12, '1694493658'    =>   5,
            '213385.278854' =>   4, '9.82'          =>   2, '613.591'       =>  -4, '0.466328'      =>   4,
            '1532833750'    =>  -7, '469990.376106' =>  -5, '20.11'         =>   3, '851.739'       =>   1,
            '0.427988'      =>   9, '1185585018'    => -12, '357491.502835' =>   3, '71.23'         =>   1,
            '379.998'       =>   3, '0.18177'       =>  -4, '1237929480'    =>  10, '101383.247931' =>   6,
            '33.67'         =>   0, '263.489'       =>   4, '0.69620'       =>   2, '1295904718'    =>  -5,
            '477234.149586' =>   5, '85.35'         =>  -4, '99.656'        =>  -3, '0.231926'      =>  -7,
            '250380028'     =>   5, '230223.162511' => -12, '84.83'         =>   4, '678.834'       =>   0,
            '0.141665'      =>  11, '565658786'     =>  12, '341314.91046'  =>   9, '22.0'          =>  -2,
            '241.741'       =>   0, '0.247073'      =>  -9, '1813614696'    =>  -8, '106060.148712' =>   2,
            '19.73'         =>   2, '199.195'       =>   3, '0.82872'       =>   2, '1198351802'    =>  -1,
            '177699.109593' =>  -2, '50.82'         =>   0, '314.747'       =>  -4, '0.245384'      =>  10,
            '151706729'     =>  -7, '422706.143097' =>  -5, '543.633'       =>   0, '0.70901'       =>   4,
            '310817463'     =>   7, '111502.253582' =>   5, '99.41'         =>   3, '552.939'       =>   0,
            '0.230433'      =>  -7, '973071638'     =>  -2, '376895.135985' =>   0, '97.75'         =>  -2,
            '234.643'       =>  -1, '0.479141'      => -12, '1763890293'    =>   7, '247314.17852'  => -11,
            '85.52'         =>  -1, '11.332'        =>   0, '0.129061'      =>  -2, '1740619639'    =>   7,
            '148015.277563' =>   5, '85.93'         =>  -4, '319.49'        =>   5, '0.183134'      =>   2,
            '786094623'     =>  12, '11745.439232'  =>  -8, '21.4'          =>   1, '513.100'       =>   5,
            '0.480654'      =>  -4, '208016871'     =>  -4, '23840.198801'  =>   6, '44.10'         =>  -4,
            '45.245'        =>  -3, '0.331976'      =>  -5, '587815390'     =>   2, '401253.155255' =>  12,
            '86.4'          =>   4, '40.529'        =>   5, '0.191665'      => -11, '849540417'     =>  -1,
            '90847.231247'  =>  -1, '44.28'         =>  -1, '546.577'       =>  -1, '0.448907'      =>   3,
            '454030751'     =>  -7, '109715.512100' =>  -9, '95.28'         =>  -4, '236.136'       =>   5,
            '0.149792'      =>   0, '2015639148'    =>   0, '505388.58658'  =>   5, '54.57'         =>   3,
            '939.638'       =>  -5, '0.291381'      =>   4, '1846723436'    =>   0, '327283.36287'  =>   7,
            '11.69'         =>  -4, '654.574'       =>  -3, '0.76996'       =>  -4, '1821131976'    =>   9,
            '283387.425713' => -10, '62.74'         =>   1, '813.169'       =>   2, '0.242363'      =>  -4,
            '1807590204'    =>   6, '258169.244302' =>  -4, '52.51'         =>  -2, '718.581'       =>   4,
            '0.33564'       =>  -2, '614654309'     =>   6, '426594.433449' =>  10, '49.34'         =>  -4,
            '50.563'        =>   2, '0.208860'      =>  10, '1320256804'    => -12, '181967.56210'  =>  -6,
            '34.22'         =>   2, '75.963'        =>  -3, '0.263642'      =>   9, '1246068123'    => -12,
            '492311.206522' =>   7, '40.10'         =>   1, '826.55'        =>  -4, '0.374927'      =>   5,
            '1594733132'    =>  -1, '363878.47018'  =>  -4, '83.78'         =>  -1, '51.699'        =>  -5,
            '0.473667'      => -12, '136424545'     =>  -7, '519175.1330'   =>   0, '67.28'         =>   3,
            '438.537'       =>   2, '0.81496'       =>   6, '385982976'     =>  -6, '305189.458112' =>  10,
            '27.82'         =>  -4, '890.852'       =>   4, '0.485550'      =>   2, '1709031568'    =>   6,
            '125027.412132' =>  12, '39.66'         =>  -3, '328.837'       =>  -4, '0.28185'       =>   4,
            '1917570138'    =>  12, '466503.249058' =>  -7, '49.38'         =>   3, '277.583'       =>  -2,
            '0.457270'      =>  -9, '1933373722'    =>   0, '261480.72754'  =>   4, '25.85'         =>   0,
            '950.477'       =>   2, '0.309669'      =>   0, '1033967919'    =>   1, '400503.194648' =>   0,
            '25.45'         =>   3, '747.530'       =>   5, '0.467982'      =>   1, '73608455'      =>  -2,
            '36720.279450'  =>   7, '2.18'          =>   0, '840.320'       =>  -4, '0.172859'      =>   4,
            '385764681'     =>  10, '206826.494684' =>   8, '82.43'         =>  -1, '373.553'       =>  -5,
            '0.471131'      =>   2, '2145228020'    =>  -1, '125096.36169'  =>  11, '59.49'         =>  -3,
            '663.196'       =>   5, '0.163393'      =>  -5, '417958569'     =>   1, '177016.308867' =>   0,
            '95.25'         =>   0, '316.325'       =>   3, '0.478939'      =>  -5, '1604251943'    =>  -7,
            '81908.516759'  =>  10, '15.58'         =>   1, '538.775'       =>   0, '0.430828'      =>   5,
            '506589845'     => -10, '302671.300696' => -12, '9.55'          =>  -4, '296.578'       =>  -5,
            '0.313157'      =>  -3, '1273404167'    =>   0, '211916.392798' =>  -4, '37.49'         =>  -1,
            '999.498'       =>   0, '0.371216'      =>  -6, '738010455'     =>  -5, '176328.482849' =>  11,
            '94.61'         =>   4, '656.786'       =>  -3, '0.13435'       =>   7, '568477995'     =>  -3,
            '381224.350704' =>   3, '5.74'          =>  -4, '222.542'       =>  -5, '0.251694'      =>   0,
            '1788665792'    =>  -6, '101486.88726'  =>   8, '11.68'         =>   2, '873.979'       =>   1,
            '0.338419'      =>  -3, '1651069858'    => -12, '326287.260029' =>  10, '55.64'         =>   4,
            '649.952'       =>  -4, '0.84611'       =>   3, '1905458609'    =>  -2, '133958.42398'  =>  12,
            '79.27'         =>  -4, '876.551'       =>  -4, '0.136933'      => -10, '205928650'     =>  -8,
            '55664.376562'  => -12, '71.7'          =>  -4, '175.968'       =>  -1, '0.426143'      =>  -3,
            '1763415445'    =>  -9, '64523.40192'   =>   5, '7.57'          =>  -2, '687.436'       =>   3,
            '0.408456'      =>  -5, '1076514101'    =>   5, '381416.318485' =>   9, '29.30'         =>  -2,
            '739.508'       =>   1, '0.67442'       =>   2, '1520147758'    =>  -4, '10794.435652'  =>  11,
            '11.10'         =>   3, '859.61'        =>   2, '0.85705'       => -10, '1744771167'    =>   9,
            '375072.283098' =>   4, '80.57'         =>  -2, '176.349'       =>  -4, '0.396252'      =>   3,
            '2049042840'    =>  -1, '224324.511048' =>  -1, '78.41'         =>   0, '90.416'        =>  -4,
            '0.107923'      =>  -7, '112842944'     =>   2, '127931.402622' => -12, '7.35'          =>   0,
            '478.258'       =>   2, '0.58406'       => -12, '1104295611'    => -12, '250270.493928' => -11,
            '40.82'         =>   2, '968.275'       =>   5, '0.94102'       =>  -3, '2038106571'    =>   0,
            '261684.101227' =>  -7, '763.887'       =>   3, '0.419104'      => -11, '133351571'     =>  12,
            '33468.282826'  =>  -5, '31.65'         =>  -4, '606.889'       =>   0, '0.511690'      =>   7,
            '539608460'     => -11, '157104.393425' =>  11, '81.83'         =>  -2, '618.139'       =>  -3,
            '0.349358'      => -10, '219266259'     =>  11, '132301.87000'  =>   6, '27.58'         =>   0,
            '74.767'        =>  -3, '0.443174'      => -11, '314832140'     =>   6, '349866.233967' => -10,
            '75.66'         =>   3, '872.231'       =>  -4, '0.511472'      =>  -6, '429750945'     =>  10,
            '327987.237221' => -11, '97.51'         =>   4, '705.548'       =>  -4, '0.410789'      =>  10,
            '1351303037'    =>  -3, '256611.155486' =>  11, '8.12'          =>  -1, '244.863'       =>   2,
            '0.265113'      =>  11, '1766929095'    =>   8, '480312.235078' =>   5, '34.11'         =>  -3,
            '239.378'       =>   3, '0.66263'       =>  -3, '1621617180'    =>  -8, '48016.128226'  =>   6,
            '58.13'         =>   0, '829.835'       =>   1, '0.395683'      =>  -2, '587408316'     =>  -4,
            '24096.99434'   =>   2, '35.37'         =>  -3, '182.562'       =>  -5, '0.285531'      =>  -2,
            '1207286666'    =>   5, '232993.342764' =>  -5, '12.12'         =>   0, '698.460'       =>   2,
            '0.69479'       =>   1, '1904090278'    =>  -7, '415270.488962' =>  -2, '13.95'         =>   3,
            '968.368'       =>   0, '0.193252'      => -12, '1526701023'    =>  10, '111348.81435'  =>  11,
            '65.71'         =>   2, '418.580'       =>  -2, '0.505803'      =>   6, '2055498202'    =>   5,
            '61865.392813'  =>   9, '57.93'         =>   2, '381.627'       =>   3, '0.211185'      =>  -9,
            '71999054'      =>  12, '485521.128926' => -10, '0.93'          =>   0, '203.635'       =>   4,
            '0.496141'      =>  -1, '377814566'     =>   2, '236623.154105' =>  10, '0.22'          =>  -2,
            '973.537'       =>   2, '0.49074'       =>  -5, '619293446'     =>  11, '27988.112428'  =>  -9,
            '73.37'         =>  -2, '260.447'       =>   5, '0.379680'      =>   2, '380236521'     =>  -3,
            '155680.329454' =>  10, '87.20'         =>   4, '868.142'       =>  -5, '0.84534'       =>   1,
            '1066107351'    =>  -7, '190309.288268' =>  -1, '88.39'         =>   2, '331.372'       =>  -4,
            '0.476173'      =>   2, '2053624162'    => -10, '522268.132766' => -10, '20.58'         =>  -2,
            '597.260'       =>  -1, '0.80816'       =>   0, '1611741955'    =>  -2, '456586.59512'  =>   4,
            '85.7'          =>   0, '188.255'       =>   5, '0.393854'      =>  -4, '921402161'     =>  -8,
            '130362.222931' =>   5, '62.16'         =>   4, '536.137'       =>  -5, '0.26585'       =>  -6,
            '1478673742'    =>  -5, '493785.293303' =>   8, '9.77'          =>  -3, '726.403'       =>  -5,
            '0.30668'       =>  -6, '938214830'     =>  -9, '451958.359419' =>   0, '65.53'         =>  -4,
            '536.522'       =>  -3, '0.420885'      =>  -4, '1322660686'    =>   2, '196287.292412' =>   0,
            '70.14'         =>  -2, '163.779'       =>   5, '0.229146'      =>  -5, '1015206824'    =>   8,
            '170356.175523' =>   9, '15.20'         =>  -2, '395.301'       =>  -4, '0.377290'      =>   4,
            '514496980'     => -10, '312183.321896' =>  -4, '91.59'         =>   1, '508.955'       =>   5,
            '0.433249'      =>   6, '1230001569'    =>  -3, '139399.470649' =>  -5, '13.79'         =>   3,
            '936.428'       =>   4, '0.315546'      =>  -7, '658623891'     =>  -9, '253142.472980' =>   9,
            '55.36'         =>  -1, '315.477'       =>  -2, '0.26433'       =>  -5, '1474209829'    =>  -5,
            '316116.499313' =>   3, '78.66'         =>  -2, '558.964'       =>   3, '0.398515'      =>  -7,
            '827700634'     =>   2, '213389.455217' => -11, '24.52'         =>  -2, '857.536'       =>  -3,
            '0.79445'       =>   5, '2136820837'    =>   2, '109867.313513' =>   8, '84.57'         =>   2,
            '378.945'       =>   4, '0.323336'      =>   8, '842604404'     =>   5, '404292.419103' => -12,
            '66.11'         =>   2, '328.114'       =>   4, '0.10305'       =>  -2, '10843546'      =>  -6,
            '54463.112515'  => -12, '95.45'         =>  -4, '310.841'       =>  -2, '0.72016'       =>  -6,
            '2105123959'    =>  -1, '241075.393951' =>   1, '49.100'        =>  -2, '874.774'       =>   3,
            '0.153247'      =>   9, '277109581'     =>   7, '423515.122116' =>  -5, '12.46'         =>  -1,
            '789.178'       =>   1, '0.33907'       =>  -2, '1907352136'    =>  -5, '443331.182449' => -11,
            '68.29'         =>  -3, '36.39'         =>  -4, '0.471768'      =>  10, '1481206297'    =>  -4,
            '329700.260849' => -12, '92.39'         =>  -3, '979.386'       =>   3, '0.214445'      =>  12,
            '601901375'     =>  -1, '260120.65992'  =>  -5, '30.32'         =>  -2, '447.617'       =>   0,
            '0.179959'      =>   0, '1797183025'    =>   0, '286086.244178' => -12, '61.45'         =>   3,
            '234.175'       =>  -3, '0.104259'      =>   2, '90282551'      =>   2, '461979.282162' =>  -1,
            '83.91'         =>   0, '372.296'       =>  -1, '0.459854'      => -12, '1169863468'    =>   3,
            '352639.47409'  =>   5, '90.79'         =>  -3, '991.886'       =>  -4, '0.304837'      =>  -6,
            '1832184648'    => -10, '295734.385002' =>   3, '26.100'        =>  -3, '403.446'       =>  -4,
            '0.226114'      =>  -2, '99751977'      =>  -4, '19645.376992'  =>  12, '93.97'         =>  -3,
            '58.995'        =>   1, '0.171894'      =>  -1, '1169970566'    =>   2, '104292.57083'  =>  -8,
            '28.17'         =>  -1, '286.756'       =>  -2, '0.363514'      =>   7, '263041008'     =>  -8,
            '85743.83864'   =>   6, '38.38'         =>   4, '955.253'       =>  -1, '0.217441'      =>  -3,
            '2040200029'    =>   2, '399450.78100'  =>  11, '51.34'         =>   2, '699.139'       =>   4,
            '0.261091'      =>   3, '1731247537'    =>   8, '300414.508411' =>   3, '20.15'         =>  -4,
            '538.368'       =>  -1, '0.478543'      =>   4, '489449991'     =>   7, '389036.518944' =>   9,
            '67.52'         =>  -2, '384.856'       =>   0, '0.1266'        =>   4, '1059654290'    =>  -4,
            '510813.34831'  =>   8, '9.98'          =>   2, '372.450'       =>   0, '0.10054'       =>   9,
            '1420362962'    =>  -3, '118494.211516' =>  -6, '424.142'       =>  -1, '0.44364'       =>  -7,
            '1246325687'    =>  -9, '500072.290803' =>   2, '31.75'         =>   2, '254.82'        =>  -3,
            '0.69673'       =>   4, '76040964'      => -10, '182733.137059' =>   7, '50.41'         =>   3,
            '783.914'       =>  -3, '0.4661'        =>  -2, '1173122397'    =>  12, '302235.262191' =>   2,
            '23.45'         =>  -3, '675.217'       =>   5, '0.182047'      => -12, '270733451'     =>  -4,
            '75972.248830'  =>  -3, '6.14'          =>  -1, '401.537'       =>   1, '0.420766'      =>  -1,
            '991403896'     =>   1, '70651.19989'   =>  -4, '26.8'          =>  -2, '23.407'        =>  -3,
            '0.25229'       =>  -1, '1699002511'    =>   6, '71777.490768'  =>  -8, '28.71'         =>   3,
            '258.326'       =>  -5, '0.378300'      =>   2, '247142413'     =>  -9, '250124.130989' =>  -8,
            '53.48'         =>  -2, '703.431'       =>  -3, '0.75403'       =>   4, '905866320'     =>  -1,
            '342322.292936' =>  11, '34.15'         =>   4, '182.269'       =>  -3, '0.394428'      =>   2,
            '1541570625'    =>  12, '91033.102197'  =>   9, '52.65'         =>  -4, '907.579'       =>   5,
            '0.300857'      =>  -4, '1798276914'    =>   6, '344701.257066' =>  12, '79.90'         =>   0,
            '158.861'       =>   3, '0.391954'      => -11, '132938371'     =>   3, '252029.123488' =>  -1,
            '63.18'         =>   1, '109.150'       =>   0, '0.245355'      => -10, '1733130068'    =>   6,
            '377224.243540' =>  -5, '100.59'        =>  -2, '390.326'       =>   2, '0.242125'      =>   1,
            '312684996'     => -11, '282603.328369' =>   1, '22.39'         =>  -3, '896.490'       =>   0,
            '0.513923'      =>   3, '733998996'     =>  11, '138071.32135'  =>   7, '33.24'         =>  -4,
            '917.37'        =>   0, '0.235102'      =>   8, '1938630497'    =>   0, '11864.231613'  =>   2,
            '64.44'         =>   3, '913.67'        =>  -5, '0.278156'      =>   0, '1008954246'    => -11,
            '437691.384397' =>   9, '80.27'         =>  -4, '772.247'       =>  -1, '0.315186'      =>  -1,
            '2093728765'    =>   5, '152063.523028' =>   0, '78.59'         =>   0, '609.868'       =>   5,
            '0.63193'       => -12, '141731881'     =>  -3, '141788.472294' =>   2, '99.11'         =>  -3,
            '555.245'       =>  -2, '0.16140'       =>   2, '563758848'     =>   9, '183489.289699' => -12,
            '6.79'          =>  -2, '734.491'       =>  -5, '0.394487'      =>  10, '608996993'     =>   3,
            '105951.290469' =>   8, '76.71'         =>   0, '25.196'        =>  -3, '0.310455'      =>  -1,
            '691374395'     =>  -7, '121976.352282' =>   9, '97.47'         =>  -2, '488.753'       =>   4,
            '0.195856'      =>  -4, '2046570908'    =>   0, '169693.81314'  =>   4, '68.97'         =>  -4,
            '200.776'       =>  -2, '0.354494'      =>   0, '1600135021'    =>  -3, '116435.512634' =>  12,
            '21.64'         =>   1, '488.588'       =>  -3, '0.434626'      =>  12, '1053704434'    =>  12,
            '324155.426945' =>   7, '86.62'         =>   3, '940.999'       =>   1, '0.240201'      =>  -3,
            '1609688531'    =>   4, '352967.509426' =>  -7, '60.39'         =>  -1, '628.469'       =>   2,
            '0.321316'      =>  -5, '534968446'     =>  -5, '41696.454762'  => -10, '70.78'         =>  -4,
            '382.574'       =>   0, '0.400636'      =>   0, '364465487'     => -12, '123613.441948' =>   0,
            '18.8'          =>  -1, '787.439'       =>  -4, '0.39197'       =>   3, '2003904224'    => -11,
            '81505.6642'    =>  -4, '2.89'          =>  -4, '384.230'       =>   2, '0.460468'      =>  -7,
            '581064832'     =>  12, '266505.265475' => -11, '93.7'          =>   4, '244.867'       =>   3,
            '0.461266'      =>  -8, '484896921'     =>  11, '139172.199888' =>   8, '45.86'         =>   0,
            '519.729'       =>  -2, '0.396722'      =>   1, '1133993792'    =>   3, '343304.19071'  => -12,
            '3.46'          =>   0, '191.867'       =>  -4, '0.191609'      =>  12, '1972439949'    =>   6,
            '293502.96437'  =>   2, '64.43'         =>  -1, '730.759'       =>   5, '0.138194'      =>   5,
            '688996186'     =>   0, '284626.511516' =>   7, '52.86'         =>  -4, '969.725'       =>  -4,
            '0.495512'      => -12, '1628264257'    => -10, '357366.166739' =>   9, '71.69'         =>   1,
            '674.812'       =>   5, '0.203279'      =>   5, '468804690'     => -11, '454343.399080' =>   5,
            '84.20'         =>   2, '350.521'       =>  -1, '0.189374'      => -10, '399928338'     =>  -1,
            '172813.455004' => -10, '52.14'         =>  -1, '84.739'        =>   1, '0.412801'      =>  -2,
            '1051319402'    =>  -9, '437960.186725' => -12, '5.61'          =>   0, '976.75'        =>   1,
            '0.37429'       =>  -9, '1267813370'    =>  -2, '291306.482338' =>  -6, '30.65'         =>   4,
            '250.218'       =>  -2, '0.348818'      => -10, '490363689'     => -12, '6116.33390'    =>  -9,
            '58.56'         =>   3, '165.358'       =>  -2, '0.151838'      =>  -3, '1760007733'    =>  -2,
            '360487.196707' =>   9, '37.9'          =>  -3, '460.711'       =>   0, '0.294223'      =>   6,
            '1561003209'    =>  -3, '278621.387220' =>   7, '52.44'         =>  -4, '745.770'       =>   4,
            '0.67210'       =>  -6, '1478844494'    => -11, '492351.197245' =>  11, '19.69'         =>   2,
            '193.296'       =>   2, '0.479387'      =>  -3, '1583916869'    =>   9, '338957.141031' =>   4,
            '31.90'         =>   0, '797.681'       =>   5, '0.29207'       =>   5, '1598371296'    =>   8,
            '396866.358290' =>   6, '42.31'         =>  -4, '960.965'       =>  -3, '0.168106'      =>   1,
            '883436959'     =>   9, '272743.30352'  => -11, '56.64'         =>   4, '554.594'       =>   0,
            '0.125592'      =>  -8, '2113137303'    =>   4, '206580.388481' =>  12, '82.9'          =>   0,
            '348.784'       =>   1, '0.473266'      =>   1, '935392589'     => -10, '85395.501110'  =>  -5,
            '86.93'         =>   0, '423.683'       =>   2, '0.313619'      =>  -6, '1925010508'    =>  11,
            '475261.152265' =>   2, '81.14'         =>   1, '736.11'        =>   1, '0.153447'      =>  -2,
            '1112693289'    =>  10, '119762.357049' =>   6, '17.54'         =>   1, '192.954'       =>  -2,
            '0.230363'      =>  12, '1944695430'    =>   1, '70103.425752'  =>  -4, '38.43'         =>   0,
            '60.337'        =>  -5, '0.248570'      => -11, '2094265461'    =>   3, '187646.106769' =>  12,
            '81.85'         =>  -4, '144.775'       =>  -2, '0.66653'       =>  -5, '1801456305'    =>  -7,
            '288179.509912' => -12, '43.23'         =>  -3, '280.668'       =>  -5, '0.178560'      =>  -3,
            '2141024139'    =>  -9, '70304.186069'  =>  -5, '40.32'         =>  -1, '658.796'       =>   5,
            '0.495359'      =>   6, '2076726873'    =>  11, '501097.270904' =>   7, '69.18'         =>   3,
            '0.305884'      =>  12, '234602643'     =>   4, '507072.127580' =>   2, '5.33'          =>   0,
            '311.468'       =>   2, '0.33579'       =>   0, '905015881'     =>  -2, '102801.197760' => -12,
            '540.354'       =>  -3, '0.266050'      =>   0, '367126157'     =>  11, '271589.72414'  =>   3,
            '63.91'         =>   0, '613.96'        =>  -1, '0.58478'       =>  10, '675945896'     =>  -8,
            '100416.267827' =>   1, '46.8'          =>   2, '296.520'       =>   1, '0.427965'      =>  11,
            '1917770866'    => -12, '337603.215506' =>  12, '87.75'         =>   0, '185.27'        =>   2,
            '0.45752'       =>   6, '1947135822'    =>  -9, '352524.51503'  =>  -2, '84.37'         =>   1,
            '978.968'       =>   3, '0.489039'      =>  -4, '782344762'     =>   7, '404044.4317'   =>   8,
            '51.10'         =>   4, '751.831'       =>   5, '0.260782'      =>  11, '202303187'     =>  10,
            '167916.401915' =>  -2, '35.34'         =>  -2, '157.587'       =>  -5, '0.259152'      =>  12,
            '1653896744'    =>   0, '180937.283539' =>   9, '44.38'         =>   1, '306.434'       =>  -5,
            '0.136697'      =>  10, '1786993865'    =>  11, '176069.79906'  =>  -4, '38.47'         =>   3,
            '490.335'       =>   5, '0.250091'      =>   2, '216053612'     =>   8, '360814.233684' =>  -6,
            '2.22'          =>   0, '308.240'       =>   2, '0.116174'      =>  -9, '165138471'     =>   0,
            '448624.216386' => -11, '25.59'         =>   4, '587.939'       =>  -1, '0.87944'       =>   7,
            '1275347970'    =>  -2, '347256.147890' =>  -6, '84.74'         =>   0, '453.983'       =>   3,
            '0.313413'      => -12, '1099410498'    =>  -7, '135325.192746' =>   4, '34.34'         =>   0,
            '768.643'       =>   3, '0.278694'      =>   6, '313676060'     =>   8, '41131.423837'  =>  -9,
            '26.24'         =>   4, '15.987'        =>   5, '0.14174'       =>  10, '259911263'     =>   2,
            '135969.198780' =>   1, '85.25'         =>  -1, '44.452'        =>   4, '0.406375'      =>   1,
            '755430284'     =>  12, '57336.225562'  =>  11, '22.99'         =>   0, '923.778'       =>   3,
            '0.439752'      =>   6, '705503397'     =>   8, '180033.308211' =>  11, '94.18'         =>  -2,
            '81.546'        =>   1, '0.326787'      => -11, '120116605'     =>  -4, '126852.86661'  =>  12,
            '68.63'         =>   1, '385.882'       =>  -1, '0.67589'       =>  -7, '54702997'      =>   6,
            '356776.193388' =>  -1, '62.98'         =>  -4, '600.283'       =>  -4, '0.352127'      =>   4,
            '2119195643'    =>  10, '462290.119946' =>   6, '18.300'        =>  -4, '0.125739'      => -11,
            '1638029921'    =>  -2, '464779.232397' =>  -6, '46.25'         =>   4, '373.511'       =>   2,
            '0.20672'       =>  11, '1835034875'    =>  -8, '477385.386009' => -12, '2.45'          =>   4,
            '616.397'       =>   5, '0.344731'      =>  -7, '1433625609'    =>   3, '473078.290498' =>  -2,
            '49.32'         =>   0, '991.163'       =>  -1, '0.492057'      =>  10, '197460706'     =>  -7,
            '119850.1305'   =>   7, '81.55'         =>   0, '491.554'       =>  -3, '0.382472'      =>   2,
            '469326455'     =>  -7, '331476.63372'  =>  -3, '66.44'         =>   0, '835.813'       =>  -1,
            '0.389892'      =>   3, '1744041622'    =>   6, '285370.21353'  =>   8, '829.117'       =>   0,
            '0.214876'      => -12, '493303082'     =>   9, '127794.451912' =>   3, '36.41'         =>  -3,
            '854.467'       =>  -4, '0.242904'      =>  12, '1597315518'    =>  -7, '24088.151051'  =>  -6,
            '36.84'         =>   3, '304.356'       =>   5, '0.168618'      => -10, '2043698383'    =>   7,
            '206882.102456' =>   1, '39.77'         =>   2, '951.404'       =>  -5, '0.494594'      =>   0,
            '1501805237'    =>   9, '74836.390739'  =>   5, '63.34'         =>   0, '325.603'       =>  -4,
            '0.225366'      =>   0, '601135821'     =>  11, '249678.353643' =>  -3, '82.97'         =>   4,
            '990.611'       =>  -1, '0.250930'      =>   7, '1374761348'    =>   1, '126822.410471' =>  -8,
            '24.55'         =>  -3, '710.879'       =>   0, '0.117245'      =>  -9, '149079574'     =>  -5,
            '361948.286074' =>   0, '43.100'        =>  -2, '365.224'       =>  -4, '0.72458'       =>   5,
            '562135537'     =>   5, '461771.264062' =>   7, '72.81'         =>  -1, '344.66'        =>  -2,
            '0.256308'      =>  -9, '993879384'     =>  -5, '416083.80306'  =>   1, '42.54'         =>   4,
            '635.333'       =>   2, '0.179203'      =>   9, '1250027646'    =>  -1, '131106.242666' =>   3,
            '26.83'         =>   3, '103.959'       =>   1, '0.134753'      =>   5, '2104213542'    =>  -1,
            '3125.405519'   =>   5, '75.29'         =>  -4, '949.284'       =>   2, '0.434020'      => -11,
            '1113433931'    =>   3, '120118.402940' =>  -2, '58.14'         =>  -1, '763.267'       =>  -3,
            '0.237367'      =>  -9, '1290277516'    =>   1, '128085.318134' =>  -2, '72.96'         =>  -3,
            '665.359'       =>  -1, '0.372937'      =>   4, '590423356'     =>  -8, '313048.264264' =>  -7,
            '84.58'         =>   0, '773.372'       =>  -5, '0.485373'      =>  -6, '1710904821'    =>   8,
            '135132.21498'  =>   1, '15.85'         =>   0, '215.79'        =>   1, '0.453586'      =>  11,
            '248315533'     =>   7, '10839.373672'  =>  -9, '18.48'         =>  -4, '489.370'       =>   3,
            '0.399257'      =>  -8, '2125156608'    =>  -5, '431114.129681' =>   1, '71.76'         =>   4,
            '326.21'        =>  -4, '0.133750'      =>  -7, '2140171244'    =>  10, '14027.9054'    =>  -3,
            '3.26'          =>   2, '58.645'        =>   1, '0.116675'      => -12, '642436874'     =>  -2,
            '505643.63671'  =>  -7, '7.17'          =>   1, '297.648'       =>   4, '0.263478'      =>  10,
            '1698877745'    =>  10, '143508.428792' =>   0, '49.55'         =>  -1, '186.682'       =>   2,
            '0.99495'       =>  -5, '1492587467'    =>  10, '60820.345756'  =>   0, '22.69'         =>   1,
            '570.43'        =>   0, '0.248952'      =>  -8, '204474556'     =>   1, '23410.193428'  =>  -6,
            '73.57'         =>   2, '216.300'       =>  -2, '0.265031'      => -12, '2006906410'    =>   0,
            '513155.26500'  =>  -7, '63.8'          =>   1, '628.847'       =>  -1, '0.429799'      =>  -2,
            '2063015688'    =>  -4, '308193.2788'   =>  10, '57.1'          =>   2, '359.461'       =>   2,
            '0.204513'      =>  12, '1009417979'    =>  12, '87506.235307'  =>  -4, '3.0'           =>  -4,
            '373.738'       =>  -2, '0.413427'      =>  -8, '1038395873'    =>   1, '247684.37420'  =>   6,
            '75.32'         =>   2, '1.125'         =>   5, '0.514472'      =>  -1, '889887830'     =>   0,
            '440661.304764' =>  -1, '61.94'         =>   4, '252.590'       =>   2, '0.217230'      =>  -1,
            '28792678'      =>  -3, '67123.254713'  =>  10, '24.71'         =>   3, '700.204'       =>  -5,
            '0.77310'       =>  -9, '1396409395'    =>  -8, '334049.257293' =>   2, '71.75'         =>   0,
            '470.728'       =>   5, '0.485319'      =>   0, '1916276590'    =>  -9, '21114.10676'   =>  11,
            '91.67'         =>  -4, '676.421'       =>  -4, '0.424692'      =>  -5, '1066814307'    =>   2,
            '68423.70214'   =>  -1, '67.66'         =>  -2, '581.495'       =>  -4, '0.49407'       =>   9,
            '36610481'      =>  -2, '188456.30052'  =>  12, '6.91'          =>   3, '16.127'        =>   3,
            '0.171556'      =>  -7, '1671792944'    =>  11, '208369.476576' =>   8, '18.63'         =>   4,
            '776.587'       =>  -3, '0.340977'      =>   5, '263522931'     =>   5, '349927.252792' =>   4,
            '26.79'         =>   2, '580.601'       =>   2, '0.414198'      =>   9, '1679807647'    =>  11,
            '394111.94190'  =>   2, '53.66'         =>   2, '868.114'       =>   0, '0.298914'      =>  -9,
            '1721264409'    =>   7, '226615.245870' =>   5, '14.44'         =>  -3, '159.529'       =>   3,
            '0.2404'        =>  -4, '549275787'     =>  -5, '465184.3924'   =>   7, '72.49'         =>  -1,
            '28.893'        =>  -5, '0.78272'       =>  -8, '2008199383'    =>  -1, '81631.192610'  =>  11,
            '52.56'         =>  -2, '290.932'       =>   0, '0.324392'      => -12, '688007865'     =>  11,
            '379138.108867' =>   9, '57.60'         =>   2, '655.213'       =>   1, '0.432680'      =>   0,
            '2046378555'    =>  -3, '164070.56947'  =>   7, '44.65'         =>   2, '125.587'       =>  -3,
            '0.66391'       =>  -9, '171104136'     =>  -7, '41802.420912'  =>  -1, '49.22'         =>   4,
            '825.215'       =>   2, '0.164887'      =>  10, '540281514'     => -11, '420198.295975' =>  -6,
            '30.82'         =>   4, '26.707'        =>   1, '0.89275'       =>  -8, '826339363'     =>   8,
            '146818.243545' =>  -6, '43.55'         =>   4, '222.987'       =>   5, '0.58368'       =>  -7,
            '523766319'     =>  -5, '297859.23782'  =>  -3, '98.22'         =>   4, '818.787'       =>   1,
            '0.522888'      =>  -6, '204898719'     =>  11, '108373.196842' =>  -3, '31.27'         =>  -2,
            '260.26'        =>   0, '0.242649'      =>  -1, '1629568497'    =>  -9, '262573.17145'  =>   4,
            '27.73'         =>  -1, '282.471'       =>  -5, '0.285776'      =>   8, '1661847712'    =>  -1,
            '445735.514098' =>   4, '24.32'         =>   2, '798.229'       =>  -5, '0.134278'      => -11,
            '449337646'     =>   5, '209310.372274' => -12, '13.61'         =>  -3, '644.160'       =>   3,
            '0.241562'      =>  -6, '1567747899'    =>   5, '181534.304198' =>  -7, '27.45'         =>  -1,
            '500.486'       =>  -3, '0.299538'      =>  -1, '362861591'     =>   6, '397051.297899' =>   7,
            '72.63'         =>  -4, '420.356'       =>   1, '0.357673'      =>   6, '996312202'     =>   8,
            '26839.424774'  =>  -2, '17.27'         =>  -2, '516.221'       =>   4, '0.520470'      =>  -4,
            '191718894'     =>   7, '300654.443857' =>  -8, '46.113'        =>   2, '0.422092'      => -12,
            '1783356979'    =>   7, '108889.462229' =>  -6, '14.25'         =>  -3, '112.986'       =>   4,
            '0.240723'      => -10, '1898433085'    =>  12, '73569.239851'  =>   1, '52.34'         =>   3,
            '999.366'       =>  -3, '0.6640'        =>  -4, '264666875'     =>  -8, '230998.173505' =>   9,
            '53.88'         =>   0, '743.884'       =>  -5, '0.436960'      =>  12, '2072927211'    =>  -4,
            '467248.55366'  =>   5, '98.70'         =>  -2, '911.626'       =>  -2, '0.125236'      =>   4,
            '1789898610'    =>   3, '252502.143697' =>  -3, '80.22'         =>  -3, '225.822'       =>   1,
            '0.110900'      =>  -1, '1700212496'    =>   4, '195178.358051' =>  -8, '97.26'         =>   0,
            '554.154'       =>   1, '0.113013'      =>  -7, '1847934343'    =>  -3, '396236.179370' => -11,
            '55.11'         =>   0, '682.898'       =>   1, '0.68808'       =>  -1, '222761296'     => -10,
            '150027.249563' =>  11, '2.58'          =>   1, '171.380'       =>  -3, '0.206157'      =>  -2,
            '1750969343'    =>   2, '338182.299431' =>  -6, '2.14'          =>   4, '772.768'       =>   2,
            '0.115753'      =>   2, '1503347218'    =>  10, '430797.517055' =>  -4, '56.83'         =>  -1,
            '90.173'        =>  -1, '0.268636'      =>   7, '768770225'     =>   2, '430315.1582'   =>  -4,
            '98.92'         =>  -1, '612.333'       =>  -5, '0.106103'      =>   8, '1551619043'    =>   1,
            '460368.285322' =>  10, '98.44'         =>   2, '825.593'       =>   5, '0.307738'      =>  -9,
            '1535989049'    => -10, '227090.281024' =>  11, '96.92'         =>  -2, '976.863'       =>   0,
            '0.419062'      =>   9, '858784789'     => -11, '507128.145744' =>   1, '96.3'          =>   3,
            '12.124'        =>   0, '0.84111'       => -11, '337884600'     =>   5, '344032.309581' =>   9,
            '93.36'         =>   3, '878.499'       =>  -4, '0.379685'      => -12, '216021134'     =>  -5,
            '429868.35580'  =>  -9, '818.936'       =>  -3, '0.455889'      =>  -4, '125812719'     => -10,
            '509654.374747' => -12, '32.54'         =>  -1, '713.626'       =>   4, '0.392622'      =>  -4,
            '1372458626'    =>  -4, '248639.240653' =>   9, '34.84'         =>   3, '64.52'         =>  -3,
            '0.215750'      =>  -7, '646614975'     =>  11, '497553.143231' =>  -1, '66.50'         =>   1,
            '346.820'       =>  -1, '0.360217'      => -12, '818557945'     =>   2, '218973.448482' =>  -4,
            '73.13'         =>  -1, '399.240'       =>   5, '0.316427'      => -12, '2021733384'    =>   4,
            '63883.466852'  =>  10, '46.84'         =>  -2, '240.156'       =>   1, '0.142031'      =>  -3,
            '2041302265'    =>   9, '89018.193049'  =>   4, '71.24'         =>   4, '206.484'       =>   4,
            '0.118101'      => -10, '1656259082'    => -10, '520147.468243' =>  -8, '20.51'         =>  -3,
            '77.244'        =>  -5, '0.240837'      =>  -8, '22661163'      =>   0, '87869.94550'   =>  -2,
            '13.48'         =>   2, '360.120'       =>  -3, '0.251709'      =>   5, '1700311736'    =>  -4,
            '268547.410974' =>  -6, '96.34'         =>   1, '354.160'       =>   5, '0.273448'      =>  -7,
            '1543387549'    =>   3, '15528.464673'  =>   1, '56.85'         =>   3, '68.650'        =>   0,
            '0.404985'      =>   2, '2021731540'    =>  -9, '199815.237845' =>   2, '2.30'          =>   0,
            '260.224'       =>  -4, '0.243067'      =>  -6, '1903548584'    =>   6, '145844.480262' =>  -8,
            '73.63'         =>   4, '105.907'       =>  -5, '0.367839'      =>   4, '1701400136'    =>   4,
            '127503.90908'  =>   4, '70.12'         =>   1, '226.153'       =>   1, '0.258361'      =>   7,
            '1158338032'    =>   3, '419193.428642' =>  -3, '81.1'          =>   3, '31.604'        =>   2,
            '0.364043'      =>   1, '498427374'     =>  -5, '497254.249190' =>   0, '6.36'          =>  -4,
            '928.391'       =>   5, '0.378096'      => -10, '1568974566'    =>  10, '137841.277955' =>  -6,
            '40.89'         =>   1, '533.609'       =>   4, '0.45110'       =>   0, '1730110007'    =>  -5,
            '414170.395357' =>  -5, '80.86'         =>   0, '820.737'       =>   5, '0.489231'      =>  -4,
            '512635777'     =>  -1, '376723.262996' =>   4, '6.59'          =>   3, '489.216'       =>   2,
            '0.518471'      =>  -6, '1179397656'    =>  -5, '325108.177821' =>   7, '79.35'         =>  -1,
            '312.93'        =>   1, '0.349657'      =>  -4, '1702896027'    => -12, '361851.268181' =>  10,
            '31.47'         =>   3, '104.965'       =>   5, '0.183382'      =>  -7, '1093187607'    =>  11,
            '392383.67712'  =>  10, '54.2'          =>  -2, '626.957'       =>   3, '0.503551'      =>  10,
            '2009920130'    =>   3, '277542.328266' =>   6, '51.57'         =>   4, '564.508'       =>   5,
            '0.404436'      =>  11, '1944137870'    =>   9, '396520.342738' =>  -5, '96.96'         =>   4,
            '769.647'       =>   1, '0.355473'      =>   1, '378439008'     =>   3, '189308.369935' =>   3,
            '64.75'         =>   1, '737.764'       =>  -5, '0.360824'      =>   1, '882094495'     =>   1,
            '427451.87587'  =>   2, '16.19'         =>  -2, '794.946'       =>  -3, '0.181493'      =>   8,
            '940850155'     =>   3, '124604.419008' =>   1, '42.87'         =>   0, '917.965'       =>   1,
            '0.246162'      =>  -3, '1981331646'    =>   4, '129251.386887' =>  10, '97.21'         =>   0,
            '392.485'       =>  -4, '0.116678'      => -10, '1523513817'    =>   8, '185621.496556' =>   4,
            '42.89'         =>   1, '509.368'       =>   2, '0.470392'      =>   3, '581189047'     =>   8,
            '376510.271143' =>   8, '1.52'          =>   2, '661.366'       =>   2, '0.391309'      =>  -8,
            '42659426'      =>   2, '130865.196036' =>  -1, '96.83'         =>   2, '481.104'       =>  -1,
            '0.51511'       =>   7, '1319418097'    => -12, '430432.174345' =>  -3, '9.90'          =>  -1,
            '600.560'       =>  -5, '0.413110'      =>  -2, '59388480'      =>  -5, '399584.145364' =>   8,
            '99.13'         =>  -3, '546.473'       =>   1, '0.169033'      =>   5, '1473953001'    =>   2,
            '493126.265996' =>  11, '89.37'         =>   3, '440.323'       =>   3, '0.460527'      =>  -4,
            '1399637678'    => -10, '481870.217004' =>  10, '58.17'         =>  -4, '752.785'       =>   1,
            '0.228851'      => -11, '1324738282'    => -10, '22838.292260'  =>  -9, '38.52'         =>  -2,
            '302.120'       =>  -4, '0.339296'      =>  -2, '1164571066'    =>  12, '474147.241901' =>  -5,
            '56.29'         =>   2, '864.8'         =>   3, '0.476450'      =>   2, '1485601243'    =>  -1,
            '297976.385533' =>   1, '4.47'          =>  -3, '575.16'        =>   3, '0.3051'        =>   9,
            '1818437869'    =>   4, '504714.393814' =>   1, '372.831'       =>  -4, '0.501972'      => -10,
            '1183210712'    =>   6, '139047.62558'  =>   4, '32.43'         =>  -1, '161.390'       =>  -5,
            '0.5134'        => -11, '1531338350'    =>  -3, '495022.354288' =>   9, '35.45'         =>   4,
            '543.898'       =>  -2, '0.340010'      =>  -9, '1908876622'    =>   9, '264483.80794'  =>  -8,
            '50.64'         =>   0, '670.483'       =>  -2, '0.376835'      =>   5, '223636098'     =>   0,
            '183271.25333'  =>  -2, '778.947'       =>   0, '0.485825'      =>  -2, '602261949'     =>   2,
            '416722.411520' =>   8, '68.47'         =>  -3, '790.420'       =>   5, '0.261060'      =>  -1,
            '2087347292'    =>  -9, '299036.168589' =>   3, '28.65'         =>   2, '179.915'       =>   2,
            '0.317202'      =>   3, '482739664'     =>   9, '344080.10290'  =>  10, '13.34'         =>   1,
            '340.888'       =>   4, '0.429461'      =>  -5, '948997291'     =>  -3, '517894.6437'   =>   7,
            '69.39'         =>   1, '74.470'        =>   4, '0.368892'      =>   6, '1249602202'    => -11,
            '503777.124871' =>  11, '87.8'          =>   1, '965.316'       =>   5, '0.140672'      =>   9,
            '505868829'     =>  -9, '485478.117109' =>   5, '87.41'         =>  -4, '240.504'       =>  -4,
            '0.514275'      =>   3, '1498883502'    =>  -7, '71485.345428'  =>   2, '29.67'         =>  -4,
            '627.130'       =>  -3, '0.250039'      =>   2, '1538141308'    =>  -3, '57220.336712'  =>   5,
            '68.80'         =>   3, '950.484'       =>  -4, '0.299959'      =>   3, '1768189933'    =>  -1,
            '14596.503172'  =>  -6, '4.22'          =>   3, '485.121'       =>  -3, '0.34688'       =>  -4,
            '1587642883'    =>   2, '162705.444828' =>   7, '68.10'         =>  -2, '216.994'       =>   5,
            '0.436235'      =>  -2, '1876146837'    =>   2, '437984.472640' =>   4, '73.66'         =>   2,
            '869.468'       =>  -3, '0.109800'      =>  -4, '2025005363'    =>   1, '166063.132803' =>  11,
            '50.48'         =>   2, '402.63'        =>   0, '0.424841'      =>   1, '218027228'     =>  10,
            '288899.491214' =>   5, '70.31'         =>  -3, '540.695'       =>   2, '0.464008'      =>  -9,
            '1475639948'    =>  10, '447032.2038'   =>   8, '75.58'         =>   1, '123.529'       =>   1,
            '0.346077'      =>   4, '1184019054'    =>  -4, '12103.53678'   =>   7, '27.40'         =>   1,
            '84.464'        =>  -4, '0.124402'      =>   7, '1486963062'    =>  -4, '92925.285772'  =>  11,
            '19.68'         =>  -2, '266.964'       =>  -2, '0.483152'      =>   9, '407763412'     =>   3,
            '201784.111655' =>  -1, '82.85'         =>  -4, '938.568'       =>   3, '0.380829'      =>  -7,
            '1201056638'    =>   8, '382395.386152' =>  -2, '100.16'        =>   1, '481.597'       =>   4,
            '0.173792'      =>   7, '903673626'     =>   0, '519828.422408' =>   1, '301.348'       =>  -1,
            '0.268273'      =>  11, '1599608892'    => -11, '258818.248637' => -11, '13.81'         =>   0,
            '637.97'        =>   0, '0.227801'      => -11, '1243064059'    =>  -2, '497988.299022' =>   9,
            '13.55'         =>   2, '923.542'       =>   1, '0.452093'      =>  10, '2085377358'    => -11,
            '65182.243656'  =>   3, '93.21'         =>   4, '677.371'       =>   0, '0.386274'      =>   3,
            '137444033'     =>   8, '346631.7256'   =>  -5, '70.2'          =>  -2, '571.795'       =>   5,
            '0.244844'      =>  -1, '1144655586'    =>   4, '138428.344639' =>  -2, '59.83'         =>  -1,
            '363.235'       =>  -2, '0.523837'      =>   5, '1075393121'    =>  12, '150813.84890'  => -11,
            '48.78'         =>  -3, '713.768'       =>  -2, '0.101016'      =>  -9, '1734432063'    =>  -7,
            '352601.37585'  =>   6, '42.98'         =>   1, '296.764'       =>   5, '0.519424'      =>  -7,
            '393507553'     =>   7, '433235.246884' =>   0, '65.13'         =>   4, '494.391'       =>   4,
            '0.337903'      =>   3, '1810438038'    =>   2, '162358.270314' =>  -1, '9.95'          =>   2,
            '917.961'       =>   0, '0.67151'       =>  -6, '39971366'      =>  11, '232235.442994' =>  -5,
            '75.43'         =>  -4, '268.728'       =>  -1, '0.459522'      =>   8, '862954172'     =>  -3,
            '432102.373040' =>   5, '818.206'       =>  -5, '0.38191'       => -12, '2107478350'    =>  -4,
            '338317.222468' => -10, '52.69'         =>  -1, '111.122'       =>   3, '0.487109'      =>   0,
            '1834013192'    =>  12, '376481.355572' =>  -4, '516.96'        =>  -2, '0.286139'      =>   1,
            '1836245739'    =>  -8, '136816.262331' =>   1, '93.47'         =>  -3, '226.994'       =>   3,
            '0.371545'      =>  10, '1938609029'    =>  11, '132782.325486' =>   2, '57.26'         =>  -2,
            '582.258'       =>   4, '0.54228'       =>  11, '256390013'     =>   8, '173622.199412' =>  11,
            '58.39'         =>  -2, '168.982'       =>   0, '0.38263'       =>  -1, '805954544'     =>   5,
            '132532.329548' =>  12, '93.17'         =>   0, '23.343'        =>   2, '0.510482'      =>  11,
            '1565401625'    =>  -1, '212368.31512'  => -10, '34.5'          =>   3, '511.419'       =>  -4,
            '0.505317'      => -11, '1651440077'    =>   0, '245443.11428'  =>  -4, '52.90'         =>  -4,
            '559.259'       =>  -2, '0.253523'      =>   3, '141653050'     =>  -7, '170770.246951' =>  -7,
            '33.62'         =>   3, '713.361'       =>   1, '0.415109'      => -12, '1068283538'    =>  -4,
            '348137.506254' =>   2, '3.9'           =>   1, '374.82'        =>  -3, '0.520999'      =>   1,
            '1625983934'    =>   2, '408082.43451'  => -11, '89.87'         =>  -4, '451.379'       =>   3,
            '0.239213'      => -10, '898144505'     =>  -8, '383605.43123'  =>  11, '72.95'         =>  -2,
            '215.787'       =>  -3, '0.387769'      =>  -5, '1902150834'    =>   7, '440416.348186' =>   0,
            '72.46'         =>  -1, '194.688'       =>  -1, '0.150586'      =>   7, '2000212172'    =>  -9,
            '51177.347650'  =>   5, '98.50'         =>  -4, '740.181'       =>  -1, '0.31027'       =>  -8,
            '2054196733'    =>  11, '219176.417641' =>  11, '16.32'         =>   4, '706.653'       =>  -1,
            '0.244336'      => -10, '505184436'     =>  11, '338639.174513' =>  -3, '35.3'          =>  -1,
            '427.675'       =>  -2, '0.316516'      =>   0, '900424870'     =>   6, '355808.439006' =>   2,
            '84.52'         =>  -2, '928.317'       =>  -2, '0.10098'       =>  11, '1907071721'    =>   7,
            '94808.279944'  =>  -6, '97.94'         =>  -2, '253.707'       =>   4, '0.394018'      =>   2,
            '668015293'     =>  -5, '456502.518898' =>  -8, '28.85'         =>  -2, '137.680'       =>   4,
            '0.44647'       =>  -7, '2125950054'    =>   5, '499868.89551'  =>  12, '41.47'         =>   0,
            '377.732'       =>   2, '0.500638'      =>  -3, '1521738398'    =>   0, '342789.303733' =>  -6,
            '81.53'         =>   1, '422.643'       =>   1, '0.329752'      =>  -7, '2012653703'    =>   6,
            '367576.466950' =>   3, '43.20'         =>  -3, '894.60'        =>   5, '0.152840'      =>   4,
            '265938546'     => -12, '147516.407715' => -12, '44.39'         =>   0, '945.493'       =>   5,
            '0.96094'       =>   2, '677379668'     =>  -9, '388175.8664'   =>  -9, '22.44'         =>   0,
            '352.626'       =>   4, '0.520504'      =>  12, '2112410311'    =>  10, '480790.138953' => -10,
            '81.33'         =>   0, '279.828'       =>   4, '0.454172'      =>  -3, '1044105482'    => -12,
            '69007.118795'  =>  -8, '76.42'         =>  -1, '331.938'       =>   0, '0.156771'      => -10,
            '66590711'      =>  11, '24469.497047'  =>  -5, '44.35'         =>   4, '222.554'       =>  -2,
            '0.307636'      =>  -9, '357205628'     =>  -2, '315383.156215' => -11, '44.5'          =>   2,
            '545.554'       =>  -5, '0.345555'      =>  12, '1918484847'    =>   9, '70896.492849'  =>   5,
            '88.8'          =>   2, '410.255'       =>  -4, '0.296371'      =>   5, '277649008'     =>   9,
            '479467.383169' =>   6, '53.47'         =>  -3, '496.816'       =>   3, '0.240991'      =>  -5,
            '342371195'     =>  -1, '86247.154483'  =>  -4, '98.17'         =>   3, '330.295'       =>  -5,
            '0.20620'       =>  -6, '229042716'     =>   9, '310539.11098'  =>  -8, '23.43'         =>   1,
            '183.589'       =>   0, '0.257068'      =>  -4, '1707817704'    => -11, '210345.503195' =>   4,
            '77.93'         =>   2, '933.198'       =>   2, '0.103522'      =>  11, '1726236577'    =>  -7,
            '212934.207695' =>  -5, '1.56'          =>  -2, '66.920'        =>  -4, '0.202865'      =>   5,
            '1604010685'    =>  -2, '31015.77661'   =>  -8, '85.91'         =>  -4, '552.858'       =>   2,
            '0.259677'      =>  10, '314385167'     => -11, '188461.289689' =>  -5, '86.72'         =>   2,
            '824.311'       =>   1, '0.273825'      =>   0, '1980601777'    =>   5, '461612.514560' =>  -6,
            '99.26'         =>  -4, '971.479'       =>   5, '0.460995'      =>   7, '1769133465'    =>  12,
            '143330.96090'  => -12, '87.44'         =>   4, '30.506'        =>  -4, '0.284701'      =>  -2,
            '263816881'     => -12, '155717.1733'   =>  -1, '2.43'          =>   2, '342.945'       =>   5,
            '0.58090'       =>  -2, '871310040'     =>   8, '445339.356053' =>  11, '65.99'         =>  -1,
            '99.895'        =>  -2, '0.274606'      =>   8, '1853484234'    =>  -2, '473150.83940'  =>   4,
            '88.68'         =>  -1, '628.123'       =>   4, '0.31292'       =>   7, '1324370935'    =>  -7,
            '88469.244383'  =>  -1, '90.80'         =>  -2, '122.372'       =>   3, '0.501946'      =>   4,
            '191147575'     =>  10, '284224.519817' =>   8, '34.86'         =>  -3, '704.496'       =>  -4,
            '0.260311'      =>  -6, '1028953176'    =>   1, '11010.339678'  => -10, '94.47'         =>  -3,
            '231.80'        =>  -3, '0.462785'      =>   2, '1018518157'    => -12, '449749.8598'   =>  12,
            '21.30'         =>   3, '425.397'       =>  -5, '0.351579'      =>   2, '274041092'     =>  10,
            '3516.77914'    =>  -9, '31.73'         =>   1, '940.290'       =>  -3, '0.271575'      =>  -4,
            '836051772'     =>  -8, '75690.129575'  =>  12, '8.21'          =>  -4, '293.837'       =>   3,
            '0.453657'      =>  -8, '149238976'     =>   0, '344615.39951'  =>  12, '66.46'         =>   0,
            '728.126'       =>   2, '0.41754'       => -12, '395187825'     =>   4, '333714.172171' =>  -1,
            '44.80'         =>   1, '581.155'       =>   4, '0.396718'      =>   5, '412133738'     => -12,
            '7195.445233'   =>   8, '30.57'         =>  -3, '286.338'       =>   0, '0.155907'      =>   3,
            '561921734'     =>  -2, '465120.470902' =>   0, '54.24'         =>   1, '93.986'        =>  -1,
            '0.407591'      =>   2, '1249904732'    =>   6, '298033.312347' =>  12, '75.12'         =>   0,
            '614.321'       =>  -2, '0.121261'      =>   5, '254708221'     =>   3, '52547.3017'    =>   0,
            '340.249'       =>  -5, '0.472562'      =>  12, '1592989317'    => -12, '521959.162658' =>   0,
            '78.83'         =>   3, '192.551'       =>   3, '0.467873'      =>  -9, '1625277402'    =>   7,
            '144121.449343' => -12, '77.63'         =>   1, '640.941'       =>   5, '0.335126'      =>  -8,
            '424836979'     =>   9, '170582.101391' =>  -3, '0.47'          =>  -1, '289.427'       =>   5,
            '0.223789'      =>   0, '594562283'     =>  -5, '173007.289277' =>  -8, '21.81'         =>  -1,
            '731.612'       =>   4, '0.473011'      =>   6, '1290435715'    =>  -6, '331557.485629' => -10,
            '33.30'         =>  -2, '245.84'        =>   1, '0.388037'      =>  12, '667701863'     =>   0,
            '364550.336021' => -12, '99.79'         =>   2, '605.274'       =>   5, '0.190688'      =>   2,
            '1769022812'    =>  -8, '145053.239160' =>  12, '78.22'         =>   1, '787.116'       =>   1,
            '0.408386'      =>   4, '369832012'     =>   9, '457861.454841' => -10, '60.72'         =>   4,
            '679.834'       =>  -4, '0.131573'      =>  -2, '1285317653'    =>  -8, '84761.458851'  => -11,
            '44.31'         =>  -4, '887.61'        =>  -2, '0.286789'      =>  -2, '620138832'     =>   9,
            '487575.84974'  =>   5, '61.19'         =>   1, '338.57'        =>   4, '0.398293'      =>   4,
            '1853436729'    =>  -5, '201032.12972'  =>  -4, '7.68'          =>   4, '995.111'       =>  -2,
            '0.224083'      =>  -2, '542844334'     =>   4, '198015.95817'  =>   5, '85.48'         =>   4,
            '315.493'       =>  -3, '0.514361'      =>  -5, '1265192214'    =>  -6, '320077.509917' =>   9,
            '89.95'         =>   3, '297.518'       =>  -4, '0.372414'      =>  -7, '1893726623'    =>  11,
            '346604.136063' =>  12, '212.337'       =>  -1, '0.258198'      =>   9, '265602359'     =>  -5,
            '84922.384921'  => -10, '58.95'         =>  -4, '53.274'        =>  -4, '0.144928'      =>   3,
            '1209193268'    =>  12, '497093.117529' =>  -9, '77.64'         =>  -4, '20.295'        =>   3,
            '0.467622'      =>  -9, '944575541'     =>   7, '455870.315531' => -12, '63.33'         =>   4,
            '428.918'       =>   1, '0.24464'       =>  10, '1867128508'    =>  -8, '184864.428647' =>   3,
            '92.60'         =>  -2, '383.433'       =>   5, '0.264602'      =>  -5, '167980317'     =>  -4,
            '178358.496880' =>  11, '88.18'         =>  -4, '789.847'       =>   3, '0.352955'      =>   4,
            '1822214055'    =>  10, '263576.105452' =>   1, '98.27'         =>  -1, '267.317'       =>  -1,
            '0.292079'      =>   8, '302626626'     =>   8, '401589.252241' =>   3, '57.45'         =>  -2,
            '190.280'       =>   5, '0.443014'      =>   8, '1563902948'    =>  11, '260072.121101' =>   5,
            '57.87'         =>   3, '250.395'       =>   0, '0.34687'       =>   8, '1374092842'    =>  11,
            '307900.212773' =>  -7, '72.26'         =>  -3, '248.816'       =>  -3, '0.43752'       =>   0,
            '951652948'     =>   4, '185645.492410' =>   8, '80.61'         =>  -2, '112.365'       =>   5,
            '0.487811'      =>  -5, '580676605'     =>  -2, '334490.449666' =>   2, '2.70'          =>  -2,
            '904.834'       =>   2, '0.206374'      =>  -1, '276934366'     =>  10, '382111.253256' =>   8,
            '495.494'       =>   4, '0.426504'      =>  -7, '1379458592'    =>  -2, '521413.146984' => -10,
            '61.97'         =>   4, '622.569'       =>  -3, '0.47569'       => -12, '2024625708'    =>  12,
            '419016.352116' =>  -7, '44.43'         =>   1, '281.642'       =>   4, '0.258311'      =>  -5,
            '1595476777'    =>  -1, '282630.386646' =>  -9, '71.2'          =>   0, '214.72'        =>  -1,
            '0.125624'      =>  11, '1941944613'    =>  -8, '200779.368835' =>  -8, '28.30'         =>  -4,
            '122.802'       =>  -1, '0.231087'      => -11, '1121413816'    =>  -5, '225023.32125'  =>  -8,
            '54.63'         =>   3, '2.80'          =>   5, '0.495029'      => -11, '489427620'     =>  -4,
            '263445.320268' =>   9, '70.83'         =>  -1, '256.776'       =>  -3, '0.169346'      => -12,
            '670845041'     =>  -7, '209785.388803' =>   9, '59.45'         =>  -2, '440.953'       =>  -1,
            '0.264834'      =>   1, '1832382586'    =>  10, '390924.186516' =>   6, '257.384'       =>  -1,
            '0.144430'      =>  -9, '768655589'     =>   5, '289651.397445' =>   2, '8.92'          =>   0,
            '368.348'       =>  -5, '0.472663'      =>  12, '1890009262'    =>   9, '307229.328064' =>   6,
            '93.83'         =>   4, '188.227'       =>  -5, '0.173105'      =>  -2, '307398131'     =>   0,
            '47681.364699'  =>   5, '15.53'         =>  -1, '556.705'       =>  -5, '0.282339'      =>  -2,
            '564058794'     =>   3, '189116.444939' =>  -2, '78.97'         =>   0, '19.483'        =>   3,
            '0.238656'      =>  12, '1029908858'    =>   8, '262670.299124' =>   2, '100.45'        =>   4,
            '510.28'        =>  -2, '0.482393'      => -12, '1368468749'    =>   2, '516658.523215' =>  -8,
            '47.81'         =>  -4, '559.16'        =>   3, '0.280692'      =>  -6, '126513357'     =>  -1,
            '36896.293557'  => -12, '51.58'         =>  -1, '904.323'       =>   5, '0.490065'      => -12,
            '1234720396'    =>  -6, '223380.293815' =>   2, '59.3'          =>   0, '803.827'       =>   1,
            '0.25400'       =>   3, '164456613'     =>  -3, '197469.77046'  => -10, '51.17'         =>   4,
            '154.442'       =>  -2, '0.85115'       =>   4, '1912643615'    =>  -3, '497374.166046' =>   6,
            '78.89'         =>   0, '476.360'       =>  -5, '0.42048'       =>   9, '493031165'     =>  -2,
            '270519.317838' =>  -9, '57.82'         =>  -3, '139.512'       =>  -4, '0.412335'      =>   1,
            '1987751232'    =>   4, '215733.458377' =>   4, '810.72'        =>   5, '0.355213'      =>  -5,
            '1544886400'    =>  -9, '162401.123400' =>  10, '42.8'          =>   1, '188.568'       =>   3,
            '0.386704'      =>  -5, '2066402694'    =>  -4, '16370.195938'  =>   4, '8.93'          =>  -4,
            '881.603'       =>   2, '0.100530'      =>   6, '1945153853'    =>  -5, '98182.113004'  =>  10,
            '83.57'         =>   2, '341.971'       =>  -3, '0.330009'      =>   8, '385060738'     =>   5,
            '194901.110379' =>   2, '23.4'          =>   4, '729.721'       =>  -5, '0.252272'      =>   3,
            '1957296793'    =>  10, '431722.51750'  =>  -3, '80.43'         =>   4, '689.877'       =>   1,
            '0.269623'      =>   1, '560754700'     =>   0, '51164.331804'  =>   5, '54.14'         =>  -4,
            '168.840'       =>  -1, '0.419207'      => -10, '31234167'      =>  -4, '371097.439348' =>   5,
            '81.61'         =>  -2, '190.431'       =>  -2, '0.389657'      =>  -7, '1557769604'    =>  11,
            '334885.431479' => -12, '89.18'         =>   0, '561.851'       =>   5, '0.347717'      =>   8,
            '338970112'     =>   4, '189490.453853' =>   4, '17.32'         =>  -3, '667.971'       =>   5,
            '0.464637'      =>  -4, '631080581'     =>   5, '376623.488958' =>   3, '48.32'         =>  -1,
            '318.95'        =>  -4, '0.81195'       => -12, '1415732807'    =>   6, '229014.10840'  =>  -5,
            '19.35'         =>   4, '272.943'       =>   5, '0.325493'      =>  11, '1410487228'    =>   8,
            '231128.196692' =>  10, '73.32'         =>   3, '481.514'       =>  -3, '0.256699'      =>  -3,
            '258648844'     =>   4, '514580.292161' =>  -9, '51.36'         =>  -3, '333.787'       =>   3,
            '0.144779'      =>   6, '1198953939'    =>  -2, '168742.523841' =>  11, '88.3'          =>  -3,
            '304.973'       =>  -5, '0.349067'      =>   2, '543917380'     =>  -1, '320446.123084' =>   7,
            '39.86'         =>   3, '640.475'       =>   4, '0.193186'      =>   5, '219655785'     =>   6,
            '358909.222369' =>  11, '59.80'         =>   4, '871.336'       =>  -5, '0.238886'      => -11,
            '1117326039'    => -12, '413109.68943'  =>   5, '92.34'         =>  -3, '247.624'       =>  -5,
            '0.494013'      =>   1, '604388635'     =>  12, '482024.506465' => -11, '96.6'          =>   3,
            '147.517'       =>   5, '0.107571'      =>  -4, '703997261'     =>  12, '357315.60695'  =>   6,
            '24.77'         =>   1, '824.210'       =>  -3, '0.185287'      =>   9, '1822528982'    =>  -9,
            '103514.402689' =>   3, '61.56'         =>   1, '550.876'       =>  -3, '0.459206'      =>   7,
            '1608014464'    => -10, '422553.225608' =>  -7, '5.57'          =>   1, '799.255'       =>  -5,
            '0.343237'      =>  -7, '821588891'     =>   3, '127917.304097' =>   0, '24.89'         =>   3,
            '144.955'       =>   5, '0.482459'      =>   9, '1997193157'    =>   8, '17663.385861'  =>   4,
            '40.15'         =>  -1, '174.719'       =>   4, '0.211591'      => -10, '1494458060'    =>  -3,
            '459503.492775' =>   7, '79.61'         =>  -3, '587.176'       =>  -3, '0.228626'      =>  -9,
            '2090041883'    => -10, '401566.3639'   =>  -3, '36.50'         =>   3, '332.194'       =>  -2,
            '0.219563'      =>  12, '797537412'     =>   2, '201185.129926' => -10, '39.84'         =>   2,
            '28.949'        =>  -1, '0.90017'       =>   3, '1507444163'    =>  -3, '393697.245306' =>  -9,
            '33.51'         =>   3, '792.686'       =>  -2, '0.414281'      =>   1, '310944923'     => -10,
            '115040.277099' =>  -8, '28.81'         =>  -2, '233.141'       =>   2, '0.439580'      =>   2,
            '2092212441'    =>  -5, '240345.380203' =>   1, '58.34'         =>   1, '695.915'       =>   5,
            '0.126809'      =>   3, '1552125996'    =>   5, '453985.493977' =>  -3, '21.19'         =>  -2,
            '835.314'       =>  -2, '0.222151'      =>  -1, '1685322602'    =>  -9, '363098.127513' =>  -3,
            '86.70'         =>   2, '634.737'       =>   5, '0.131834'      =>  10, '1276312123'    =>  12,
            '191584.241297' =>  12, '74.13'         =>  -2, '462.308'       =>   3, '0.489398'      =>  -9,
            '605075305'     =>   6, '68278.510821'  =>   5, '68.82'         =>  -2, '791.588'       =>   2,
            '0.356921'      =>  -1, '929919464'     =>   8, '426669.418615' =>  -6, '7.47'          =>   2,
            '287.325'       =>   1, '0.218276'      =>   9, '1275754734'    =>  -6, '512048.379741' =>   5,
            '98.35'         =>   4, '65.544'        =>   5, '0.272214'      =>   1, '773380702'     =>  -7,
            '262015.91195'  =>  11, '91.83'         =>  -2, '383.993'       =>   3, '0.128372'      => -10,
            '752359425'     =>  -2, '56159.171441'  =>   8, '26.54'         =>   3, '473.192'       =>  -5,
            '0.521075'      =>   2, '1689479013'    =>   5, '347839.150197' =>  -4, '52.43'         =>  -1,
            '199.627'       =>  -3, '0.152574'      =>  -7, '1351329243'    =>   9, '372598.386073' =>   4,
            '52.23'         =>  -2, '251.288'       =>   2, '0.426629'      =>   8, '1172920955'    =>  -5,
            '333459.109908' =>   3, '63.41'         =>  -3, '636.795'       =>   4, '0.448941'      =>   0,
            '786612913'     => -10, '381432.40354'  =>   0, '35.74'         =>   2, '381.181'       =>  -4,
            '0.113118'      =>  -1, '1580207428'    =>  12, '312052.194964' =>   7, '54.60'         =>   0,
            '463.527'       =>  -4, '0.502017'      =>  11, '28421197'      =>  -3, '455359.388371' =>  -4,
            '44.75'         =>  -4, '918.209'       =>  -1, '0.194597'      =>   1, '1163350661'    =>  -2,
            '97917.71785'   => -10, '62.1'          =>  -3, '146.216'       =>   0, '0.38631'       => -12,
            '900569107'     =>  -8, '261095.150936' => -10, '38.26'         =>   4, '630.568'       =>  -3,
            '0.92956'       =>  -2, '1804971260'    =>  11, '24094.14296'   =>   5, '18.47'         =>   0,
            '985.686'       =>   5, '0.11043'       =>  -9, '708948475'     =>   6, '443943.434178' =>  -1,
            '73.10'         =>   3, '313.569'       =>  -1, '0.382185'      =>  11, '1898592770'    =>   6,
            '243998.487618' =>   2, '45.79'         =>  -2, '722.781'       =>   5, '0.460099'      =>   5,
            '945769609'     =>   8, '127141.150556' =>  -7, '97.77'         =>   4, '140.391'       =>   4,
            '0.44704'       =>  -1, '463388026'     => -10, '361326.357130' =>  12, '0.35'          =>  -2,
            '349.790'       =>   3, '0.15362'       =>   2, '820198660'     =>   4, '458826.327385' =>  -1,
            '619.814'       =>   2, '0.46267'       => -11, '393398368'     =>  10, '455329.457371' =>  -5,
            '15.99'         =>   3, '787.471'       =>  -3, '0.192102'      => -12, '1468529827'    =>   0,
            '508533.293066' =>   6, '47.97'         =>   2, '641.64'        =>  -4, '0.369226'      =>  11,
            '1839566271'    =>  -9, '239288.380153' =>  10, '49.92'         =>  -1, '144.216'       =>   1,
            '0.84409'       => -10, '1259217342'    =>   2, '173336.291671' =>   9, '28.88'         =>   2,
            '17.312'        =>   3, '0.511154'      =>   7, '2029269530'    => -12, '473889.210427' =>   8,
            '85.5'          =>   1, '665.727'       =>  -3, '0.393714'      =>   0, '151233890'     =>  10,
            '383876.210258' =>  -4, '10.17'         =>  -3, '906.27'        =>   3, '0.365852'      => -10,
            '224382563'     => -12, '437109.4381'   =>   4, '67.89'         =>  -1, '159.104'       =>  -3,
            '0.339486'      =>  12, '1862914143'    =>   9, '101239.314401' =>   5, '74.22'         =>  -4,
            '1000.168'      =>   3, '0.45778'       =>   7, '1663878468'    =>   9, '267035.319042' =>  -8,
            '100.88'        =>   0, '126.600'       =>  -1, '0.55631'       =>   1, '857937515'     =>  -6,
            '506127.310697' =>  12, '68.31'         =>  -4, '54.307'        =>   5, '0.433604'      => -10,
            '2100668704'    =>  -3, '307303.255605' =>   3, '4.21'          =>   3, '648.785'       =>   0,
            '0.100605'      => -10, '706157416'     =>  10, '453071.154241' =>   2, '38.87'         =>  -1,
            '807.801'       =>   2, '0.481221'      =>  12, '832432926'     =>   3, '120398.510533' =>  10,
            '906.297'       =>  -1, '0.517726'      =>  -7, '200952545'     =>   0, '42316.502132'  =>  -5,
            '675.935'       =>  -4, '0.350550'      =>  11, '1224345436'    =>   9, '392278.419310' => -11,
            '24.79'         =>  -3, '208.462'       =>   4, '0.215814'      =>  -8, '1451226453'    =>  -1,
            '401728.396619' =>   0, '2.86'          =>  -1, '1000.41'       =>  -4, '0.493887'      => -10,
            '2009460819'    =>   5, '401594.358581' =>   0, '82.55'         =>   3, '441.795'       =>  -1,
            '0.328834'      =>   8, '610865731'     =>  -7, '373058.26577'  =>  -5, '43.8'          =>  -3,
            '230.32'        =>   3, '0.164383'      =>   8, '1168333968'    =>  12, '255056.162543' =>  -6,
            '10.75'         =>   3, '121.791'       =>   2, '0.500997'      =>   0, '1488546354'    =>  -2,
            '378017.212184' =>  -6, '58.72'         =>   3, '511.91'        =>   1, '0.181246'      =>   4,
            '1328228167'    =>  -2, '342848.55043'  =>  10, '91.30'         =>   1, '998.457'       =>  -3,
            '0.254136'      => -11, '992870908'     =>  -2, '342217.96130'  =>  -8, '97.16'         =>   4,
            '774.173'       =>   4, '0.219060'      =>  10, '688916804'     =>   0, '214139.511040' => -12,
            '56.4'          =>   1, '298.912'       =>   4, '0.179565'      =>  -4, '1648744440'    =>  -3,
            '301225.220454' =>   8, '36.96'         =>  -3, '777.92'        =>  -5, '0.363420'      =>  -8,
            '1540254552'    =>   0, '286658.65890'  => -12, '74.55'         =>   2, '448.30'        =>   2,
            '0.421306'      =>   1, '2054049169'    =>  -3, '93475.278413'  =>   2, '69.35'         =>  -3,
            '381.896'       =>  -5, '0.304011'      => -10, '1310287670'    =>   2, '315980.82265'  =>  -5,
            '85.15'         =>   1, '805.591'       =>   1, '0.191051'      =>   3, '2109206142'    =>  -3,
            '506142.84130'  =>  -4, '33.54'         =>   4, '547.694'       =>  -1, '0.339179'      =>  -4,
            '1996535939'    =>  -6, '299486.279127' =>  -6, '12.26'         =>   4, '837.956'       =>  -3,
            '0.233547'      =>   9, '537470584'     =>  -5, '335813.113073' =>  -2, '17.92'         =>   0,
            '360.580'       =>   0, '0.367785'      =>  -8, '975514236'     =>  -7, '253630.13361'  =>   3,
            '1.29'          =>   4, '553.994'       =>  -4, '0.214617'      =>  -9, '955561142'     =>  -2,
            '454321.44816'  =>   4, '58.4'          =>   0, '534.730'       =>   1, '0.374423'      =>  -8,
            '895237981'     =>  -2, '32973.472194'  =>  -8, '80.21'         =>  -2, '383.67'        =>   5,
            '0.266785'      =>  11, '832156768'     =>  -1, '71293.133197'  =>  11, '71.62'         =>  -4,
            '227.594'       =>  -5, '0.219087'      =>   8, '515978774'     =>  -5, '249592.158944' =>   8,
            '53.53'         =>   2, '489.267'       =>  -5, '0.229370'      => -10, '1260314162'    =>   7,
            '382476.378986' =>   8, '74.79'         =>   2, '867.614'       =>   3, '0.353751'      =>  -3,
            '1491936863'    => -12, '454214.89546'  =>  10, '84.75'         =>   1, '792.625'       =>   2,
            '0.472840'      =>  12, '1646314706'    => -11, '464997.260120' =>   0, '26.95'         =>   4,
            '740.620'       =>   2, '0.66293'       =>  -6, '636315921'     =>  -5, '109877.85276'  =>   8,
            '77.84'         =>  -1, '557.967'       =>   4, '0.287182'      =>   1, '1258982012'    =>   9,
            '187442.248077' =>  -5, '44.7'          =>   4, '727.254'       =>  -3, '0.519671'      =>   8,
            '230346146'     =>  -3, '105834.166114' =>  -5, '75.21'         =>  -3, '114.583'       =>   5,
            '0.334914'      =>  10, '1623042297'    =>  -9, '233591.59404'  =>  -7, '68.13'         =>  -2,
            '330.58'        =>  -3, '0.74911'       =>  -1, '1787196074'    => -11, '176538.17873'  =>  -6,
            '23.31'         =>   3, '231.115'       =>  -3, '0.62236'       =>   8, '2081067983'    =>   6,
            '513664.217376' =>   9, '67.67'         =>   2, '868.145'       =>  -5, '0.167298'      => -11,
            '235063451'     =>  -5, '282355.233927' =>  -9, '23.41'         =>  -4, '668.649'       =>  -3,
            '0.265170'      =>  -4, '225569705'     =>   7, '336582.44447'  =>  -1, '77.56'         =>   1,
            '562.167'       =>   1, '0.326419'      =>   6, '25752866'      =>   0, '334487.288642' =>   6,
            '66.41'         =>   3, '646.905'       =>  -5, '0.522260'      =>   6, '1051048962'    =>  -2,
            '464112.68898'  => -10, '76.95'         =>   2, '527.898'       =>  -5, '0.131253'      =>  -7,
            '1536377413'    =>   0, '132027.185291' =>  -6, '17.47'         =>  -1, '542.487'       =>  -1,
            '0.138450'      =>  -5, '1564553396'    =>   0, '306411.321795' =>   6, '39.65'         =>   2,
            '708.246'       =>   2, '0.490782'      =>   3, '301106405'     =>   0, '397681.205539' =>   4,
            '83.18'         =>   1, '595.583'       =>   2, '0.478934'      =>  -9, '1361154939'    =>   9,
            '314604.114436' =>   4, '99.65'         =>   0, '367.839'       =>   2, '0.517998'      =>  11,
            '965803321'     =>   7, '223362.109185' =>  -9, '78.34'         =>  -1, '5.208'         =>   4,
            '0.81985'       =>  -2, '534815524'     =>  -9, '400342.445174' =>   7, '33.100'        =>  -4,
            '705.31'        =>   0, '0.345202'      =>  -5, '167708690'     => -12, '77320.264306'  => -10,
            '94.9'          =>   2, '916.15'        =>   1, '0.167018'      =>   6, '1884242836'    =>  -8,
            '504611.336074' =>   2, '66.75'         =>   4, '216.673'       =>  -2, '0.272397'      =>  -4,
            '804534093'     =>  -4, '230442.273740' =>   2, '19.3'          =>  -2, '585.768'       =>  -3,
            '0.178557'      =>  -7, '833374022'     => -10, '343018.183783' =>  -7, '32.52'         =>  -4,
            '844.37'        =>  -2, '0.104299'      =>  12, '631164853'     =>   1, '81230.384535'  =>   2,
            '49.28'         =>  -4, '485.879'       =>  -2, '0.367677'      => -10, '421212287'     =>  -7,
            '229033.445853' => -10, '93.76'         =>  -2, '51.659'        =>   3, '0.7081'        =>   7,
            '726125328'     =>   9, '226146.258507' =>  -5, '75.53'         =>   0, '872.298'       =>  -5,
            '0.514402'      =>  11, '301733263'     =>  12, '46503.302698'  =>  -8, '88.95'         =>   0,
            '692.341'       =>   2, '0.247996'      =>  -4, '440384341'     =>  -4, '347016.333662' =>   8,
            '16.5'          =>   2, '97.104'        =>   5, '0.9960'        =>  -6, '587145931'     => -10,
            '137925.189849' =>  -9, '89.61'         =>  -2, '298.932'       =>  -4, '0.338602'      =>   1,
            '1881453802'    =>  -2, '113315.282068' =>   9, '24.65'         =>   0, '627.672'       =>   0,
            '0.472303'      =>   2, '1426268683'    =>   5, '445735.486135' =>  -8, '22.43'         =>   1,
            '608.496'       =>   0, '0.67854'       =>  -3, '1706716161'    =>  -9, '520276.5705'   => -12,
            '53.79'         =>   0, '258.757'       =>  -4, '0.438510'      =>   7, '1004776687'    =>   4,
            '436364.166754' =>   0, '91.65'         =>  -1, '103.403'       =>  -4, '0.258641'      =>   2,
            '1587482095'    =>   1, '282317.383557' =>  -5, '21.83'         =>  -4, '936.124'       =>  -2,
            '0.373783'      => -10, '1625426089'    =>  12, '241840.308909' =>  -8, '96.40'         =>   4,
            '506.639'       =>   3, '0.54486'       =>  -9, '1481879732'    =>  -9, '220079.119816' =>   9,
            '31.79'         =>   2, '127.485'       =>  -4, '0.118745'      =>  -5, '562468606'     =>  -8,
            '198378.379161' =>  12, '61.53'         =>   3, '284.788'       =>  -4, '0.231304'      =>   7,
            '1969544595'    =>  10, '385061.176637' =>  -9, '5.67'          =>  -2, '848.223'       =>   1,
            '0.77794'       => -10, '615691954'     =>  -1, '226840.348693' =>  -2, '54.75'         =>  -4,
            '893.820'       =>   4, '0.355600'      =>   3, '1078526925'    =>  -6, '205610.124085' =>   2,
            '55.99'         =>   2, '593.269'       =>   2, '0.357361'      =>   9, '672306509'     =>  -2,
            '434174.390977' =>  12, '75.17'         =>   4, '1000.600'      =>  -2, '0.319940'      => -10,
            '788933660'     =>  -6, '458134.398221' => -10, '51.90'         =>   3, '536.386'       =>   5,
            '0.209606'      =>  -7, '1333578221'    => -11, '19114.235467'  =>   5, '41.21'         =>   0,
            '496.845'       =>  -4, '0.315981'      =>   5, '2140386144'    =>  -1, '297586.456401' =>  12,
            '68.24'         =>   4, '175.348'       =>   0, '0.198181'      =>   9, '1987222384'    =>   6,
            '241378.504276' =>  12, '63.86'         =>   4, '977.625'       =>   0, '0.364493'      =>  -4,
            '1669268745'    =>  -7, '103573.180834' =>   5, '710.343'       =>  -5, '0.297275'      =>  -2,
            '362208538'     =>   6, '377524.329808' =>   0, '55.8'          =>  -3, '563.3'         =>   2,
            '0.464184'      =>  -5, '1135770879'    =>   5, '205055.380861' =>   4, '66.27'         =>   4,
            '342.903'       =>   4, '0.400438'      =>  -8, '1902658601'    =>  -1, '504249.317753' =>  11,
            '78.11'         =>  -1, '786.65'        =>   3, '0.45893'       =>   8, '1083352314'    =>  11,
            '321446.469545' =>   3, '85.100'        =>   1, '819.232'       =>  -5, '0.515486'      =>  11,
            '888046360'     =>  12, '486962.196769' =>   9, '79.25'         =>   0, '50.386'        =>  -2,
            '0.448836'      =>  -4, '662933581'     =>  -1, '47159.483295'  =>  -3, '54.10'         =>  -1,
            '147.247'       =>  -1, '0.47789'       =>   4, '1420380237'    =>  -7, '52410.309447'  =>   0,
            '8.67'          =>   1, '965.876'       =>   2, '0.169213'      => -11, '1159443878'    =>   6,
            '48355.330226'  =>   3, '6.9'           =>  -1, '528.601'       =>   0, '0.101339'      =>   6,
            '2083533883'    =>   5, '103030.36798'  =>   9, '206.712'       =>   0, '0.142726'      =>  -4,
            '166291914'     => -12, '349423.88954'  =>   8, '52.50'         =>  -3, '997.844'       =>   5,
            '0.385892'      =>   2, '567429324'     => -12, '467533.241563' =>   5, '54.42'         =>  -1,
            '892.68'        =>  -2, '0.124688'      =>  12, '69447445'      =>  -4, '79880.366378'  =>  -5,
            '92.0'          =>  -1, '975.95'        =>  -3, '0.286829'      => -11, '246664446'     =>   8,
            '281997.3466'   =>   8, '34.81'         =>  -2, '94.628'        =>  -3, '0.48059'       =>  -6,
            '1317798191'    =>   9, '95061.401608'  =>  -9, '36.74'         =>   1, '976.644'       =>   0,
            '0.27918'       =>   1, '830145843'     =>  -8, '1207.484669'   =>  -8, '724.178'       =>   5,
            '0.522648'      =>   3, '1007597492'    =>  -6, '206406.341057' =>   3, '31.74'         =>  -3,
            '196.934'       =>   5, '0.393853'      =>   3, '1762818530'    =>  12, '81088.431583'  =>  -9,
            '9.30'          =>  -2, '276.571'       =>  -4, '0.460856'      =>  -4, '1350580999'    =>   2,
            '93449.11849'   =>  -4, '24.31'         =>   0, '301.488'       =>  -3, '0.488295'      =>  -9,
            '250903215'     =>  -7, '197613.142344' =>   6, '60.67'         =>  -1, '855.15'        =>  -1,
            '0.111877'      => -12, '1260264928'    =>  10, '127501.401131' =>   9, '32.10'         =>  -2,
            '301.859'       =>   1, '0.226112'      => -10, '727090804'     =>   7, '284397.375125' =>   1,
            '15.42'         =>  -3, '869.645'       =>   4, '0.465527'      =>  10, '1730135295'    =>   9,
            '367342.25609'  =>  -1, '29.56'         =>   1, '171.971'       =>   4, '0.141928'      =>   5,
            '1067855866'    =>  -2, '498043.20816'  =>   3, '15.48'         =>   2, '374.58'        =>   5,
            '0.149323'      =>   2, '969946418'     =>  11, '221008.79857'  => -10, '72.43'         =>   1,
            '278.392'       =>  -1, '0.516575'      =>  -4, '1631715279'    =>   4, '142822.372123' =>   0,
            '48.68'         =>   1, '628.451'       =>   2, '0.116123'      =>   6, '1449499030'    =>   6,
            '433070.50602'  =>   4, '76.54'         =>   4, '931.674'       =>   5, '0.151400'      =>  -3,
            '74388227'      =>   4, '74188.160983'  => -11, '91.24'         =>   1, '907.972'       =>   5,
            '0.333071'      =>  -2, '358908368'     =>  -8, '454233.520694' =>  -5, '96.16'         =>   2,
            '143.166'       =>  -2, '0.270170'      => -12, '1474258062'    =>  -1, '263898.434114' =>   1,
            '49.76'         =>  -2, '742.406'       =>   4, '0.91057'       =>   8, '640336'        => -11,
            '184561.454389' =>  -4, '646.67'        =>   3, '0.353302'      =>  11, '890011967'     =>   2,
            '69500.481186'  =>  -3, '18.68'         =>   4, '515.493'       =>   2, '0.169110'      => -10,
            '273070130'     =>   4, '3876.251228'   =>  -4, '24.75'         =>  -1, '542.280'       =>  -4,
            '0.262921'      =>  -7, '1742471373'    =>  11, '164807.494908' =>  10, '94.1'          =>   0,
            '692.699'       =>   2, '0.407652'      =>  -8, '1812106610'    => -12, '180231.446285' =>  10,
            '19.62'         =>   0, '784.823'       =>   4, '0.6358'        => -10, '129737024'     =>  -4,
            '126119.196481' =>   8, '88.70'         =>  -4, '385.780'       =>  -5, '0.294964'      =>   6,
            '1674263192'    =>   1, '242150.64699'  =>  -2, '66.45'         =>   3, '276.251'       =>   5,
            '0.200806'      =>   7, '184299841'     =>  -6, '45397.171114'  =>  -1, '40.70'         =>   3,
            '756.941'       =>   0, '0.269884'      =>  -6, '1771565159'    =>   0, '476831.150373' =>  -5,
            '13.7'          =>  -3, '12.894'        =>   1, '0.405670'      =>   9, '245118583'     =>  -6,
            '377666.105240' =>   9, '5.59'          =>   0, '74.991'        =>   1, '0.179576'      =>   1,
            '114722185'     =>   0, '419113.504839' =>  -8, '13.59'         =>  -3, '187.304'       =>   3,
            '0.19958'       =>   4, '1971862231'    => -11, '491572.334790' =>  -9, '17.41'         =>  -3,
            '17.785'        =>  -1, '0.297809'      =>  -2, '1946685997'    =>  -9, '154595.370090' =>   3,
            '93.72'         =>   2, '462.523'       =>  -3, '0.54041'       =>   0, '1282241176'    =>   2,
            '373945.280331' => -12, '66.84'         =>  -4, '248.492'       =>  -1, '0.340641'      =>  -9,
            '602753459'     => -11, '357078.301751' =>   1, '32.18'         =>  -2, '400.442'       =>   2,
            '0.481857'      =>  -8, '1599180253'    =>   1, '296730.240082' =>   0, '52.71'         =>  -2,
            '77.311'        =>   5, '0.120984'      =>   0, '1179534686'    =>  12, '271019.120762' =>   0,
            '72.364'        =>   0, '0.134213'      =>  10, '991808417'     =>  -5, '419154.14583'  => -12,
            '25.99'         =>   3, '981.637'       =>  -2, '0.246613'      =>   3, '1262950106'    =>   9,
            '198756.55068'  => -12, '89.75'         =>   0, '221.951'       =>   4, '0.72611'       =>  -7,
            '1064876135'    =>  -7, '108406.154846' =>   3, '14.89'         =>   1, '198.629'       =>   2,
            '0.432856'      =>  -4, '450209192'     => -10, '62292.308670'  =>  -2, '5.40'          =>  -1,
            '251.336'       =>   5, '0.243109'      =>  10, '190677574'     =>  -1, '39160.154958'  =>  11,
            '97.40'         =>   2, '105.783'       =>   1, '0.234700'      =>  -7, '723572270'     =>   3,
            '11631.238945'  =>   4, '83.30'         =>   0, '233.195'       =>   5, '0.60889'       =>  -6,
            '1609060270'    =>   4, '324092.431997' =>  -7, '53.46'         =>   3, '70.379'        =>  -4,
            '0.161560'      =>  10, '1079758822'    =>   8, '253900.275244' =>   4, '6.93'          =>   2,
            '865.264'       =>  -5, '0.73080'       =>  -7, '1853635520'    =>  -3, '307076.252351' =>  10,
            '35.24'         =>  -4, '351.618'       =>   1, '0.128587'      => -12, '1791732556'    =>  11,
            '299489.167047' =>   6, '4.34'          =>   2, '349.462'       =>  -5, '0.290050'      =>  -4,
            '1558058269'    =>   3, '256893.163173' =>  -2, '40.25'         =>  -3, '881.793'       =>   5,
            '0.465314'      =>   2, '1050433131'    => -12, '412265.31655'  =>   6, '169.747'       =>  -2,
            '0.269635'      =>   7, '844311209'     =>  -5, '375718.463024' =>  -7, '8.95'          =>   1,
            '894.747'       =>   0, '0.239334'      =>  -4, '771235551'     =>   3, '186636.76267'  =>  11,
            '20.74'         =>  -1, '68.656'        =>   1, '0.451009'      =>  -4, '1396818915'    =>  -8,
            '214698.192451' =>  -6, '99.22'         =>   4, '750.803'       =>   4, '0.37243'       =>   2,
            '490909289'     =>  -6, '107286.306487' =>  -2, '99.78'         =>  -3, '384.17'        =>   0,
            '0.384777'      =>  -9, '627042627'     =>   9, '68753.367784'  =>  -3, '66.74'         =>  -2,
            '581.240'       =>   1, '0.91844'       =>  -3, '1358390352'    =>  -5, '388694.438925' =>  -2,
            '4.97'          =>  -2, '436.481'       =>   5, '0.292863'      => -12, '1915994911'    =>  -7,
            '314286.12237'  =>   9, '29.21'         =>  -1, '785.665'       =>   3, '0.83220'       => -12,
            '713828588'     =>   8, '127598.38681'  =>   7, '39.68'         =>  -2, '673.638'       =>   0,
            '0.360560'      =>  12, '396425761'     =>  -8, '24423.411070'  =>   2, '13.90'         =>   1,
            '675.598'       =>  -3, '0.365168'      =>  10, '1551896366'    =>   6, '287288.506478' => -12,
            '19.49'         =>  -3, '982.526'       =>  -1, '0.498741'      => -12, '498842326'     =>  -7,
            '416076.146210' =>  -9, '88.5'          =>  -1, '273.967'       =>   2, '0.82083'       =>  -5,
            '96507945'      =>   4, '362738.310849' =>   9, '77.4'          =>   3, '442.270'       =>   1,
            '0.250705'      =>  -9, '776729985'     =>  -7, '187149.81419'  =>   5, '55.87'         =>  -2,
            '557.605'       =>   4, '0.443261'      =>   8, '1015175741'    =>   7, '336085.86296'  =>  10,
            '96.25'         =>   0, '875.972'       =>  -3, '0.11882'       =>   0, '1585025947'    =>  -1,
            '83806.49831'   =>  -3, '936.573'       =>   2, '0.390534'      =>  -5, '151911157'     =>  -2,
            '162070.373172' =>  -7, '20.34'         =>   3, '589.920'       =>   3, '0.55338'       =>   5,
            '317649471'     =>  11, '171289.161357' =>   9, '14.894'        =>   0, '0.169460'      => -11,
            '998696332'     =>   3, '448682.405892' =>  -3, '57.30'         =>  -2, '31.168'        =>   1,
            '0.382562'      =>   1, '1017199412'    =>  -6, '61394.419628'  =>   9, '67.64'         =>  -4,
            '115.381'       =>   1, '0.84776'       =>   0, '335831848'     =>   5, '51310.6384'    =>   3,
            '25.97'         =>  -3, '579.539'       =>   4, '0.57923'       =>  12, '581829645'     =>  -6,
            '76440.203443'  =>   2, '676.88'        =>   3, '0.101913'      =>  -8, '1921580350'    =>  -6,
            '459965.520446' =>  -4, '19.61'         =>  -2, '443.825'       =>   2, '0.230763'      =>  -6,
            '1096029721'    =>  11, '272402.34425'  =>   5, '99.54'         =>   4, '353.632'       =>   5,
            '0.285236'      => -11, '644786723'     =>   8, '92638.93095'   =>  -2, '46.63'         =>   0,
            '331.596'       =>   2, '0.300748'      =>   1, '992008167'     => -12, '391472.514592' => -10,
            '89.73'         =>   3, '689.742'       =>   4, '0.386946'      =>  -1, '640884548'     =>  -5,
            '379991.249104' => -12, '1.67'          =>  -3, '373.115'       =>  -4, '0.471745'      =>   7,
            '1714925717'    =>   5, '405376.285867' =>  -3, '39.23'         =>   0, '69.930'        =>   3,
            '0.284324'      =>   1, '1734248167'    =>   4, '369934.279103' =>   9, '69.33'         =>   4,
            '498.967'       =>   0, '0.136923'      =>   2, '595889189'     =>   9, '119482.26568'  => -11,
            '56.100'        =>   1, '776.870'       =>   3, '0.155946'      =>   6, '862253778'     =>  -2,
            '132540.56157'  => -10, '2.32'          =>   1, '799.397'       =>   1, '0.202929'      =>   2,
            '975505082'     =>   4, '376640.357642' =>   4, '25.28'         =>  -2, '84.95'         =>  -5,
            '0.429104'      =>  11, '719501086'     =>  -3, '260910.308200' =>  -7, '15.35'         =>  -3,
            '978.260'       =>  -5, '0.296448'      =>  -1, '240455431'     =>   4, '163252.435345' => -12,
            '15.96'         =>  -4, '8.390'         =>   0, '0.488281'      =>   2, '2063595465'    =>  -7,
            '224670.240429' =>  -6, '59.1'          =>   2, '66.88'         =>  -5, '0.272565'      =>  -5,
            '2064240053'    =>   4, '353224.142929' =>   3, '30.62'         =>  -1, '89.339'        =>   1,
            '0.344641'      =>  12, '437521483'     =>  -6, '281381.331487' =>   3, '39.67'         =>  -1,
            '725.468'       =>   0, '0.20699'       =>   2, '519210309'     =>  12, '325015.479985' => -12,
            '29.57'         =>  -4, '177.787'       =>   1, '0.90600'       =>  12, '496186991'     =>  -1,
            '32893.402520'  =>  -3, '16.40'         =>  -3, '896.535'       =>   1, '0.253518'      => -12,
            '35522617'      =>   2, '301654.333687' =>   6, '72.94'         =>   3, '554.599'       =>  -1,
            '0.269657'      =>  -7, '1391964796'    =>   7, '77705.372729'  =>   6, '85.67'         =>  -3,
            '232.449'       =>   2, '0.136599'      =>  -8, '1654464260'    =>  11, '345178.181287' =>  -6,
            '19.63'         =>  -2, '48.785'        =>   1, '0.132377'      =>  -7, '2117933516'    =>  -5,
            '23278.70491'   =>  10, '29.46'         =>   1, '122.461'       =>   4, '0.167001'      =>  -5,
            '489022220'     =>   6, '523772.464568' => -10, '245.594'       =>  -1, '0.240900'      =>  -7,
            '1073876604'    =>   9, '391436.285454' =>  -1, '10.88'         =>  -1, '717.944'       =>  -2,
            '0.6014'        =>   7, '1105176714'    =>  -6, '183800.269302' =>  -7, '43.31'         =>   3,
            '727.660'       =>   5, '0.486536'      =>   2, '881050931'     => -10, '296020.82249'  =>  -2,
            '52.7'          =>  -3, '406.733'       =>  -2, '0.104091'      =>  -6, '1159938485'    =>   6,
            '515845.466988' =>   5, '68.65'         =>  -4, '624.146'       =>  -5, '0.98714'       =>   6,
            '1865766716'    => -11, '263588.227241' =>   2, '58.27'         =>   0, '606.956'       =>   1,
            '0.463708'      =>  -4, '155865207'     =>  -9, '454735.29610'  =>  -2, '16.100'        =>  -4,
            '81.371'        =>   1, '0.436074'      =>   6, '1732624335'    =>   1, '82834.162304'  =>   5,
            '79.71'         =>   4, '307.790'       =>  -2, '0.332778'      =>  -5, '954404878'     =>   4,
            '51864.163456'  =>   5, '40.93'         =>  -1, '515.487'       =>  -4, '0.129485'      =>   1,
            '179865788'     =>  -5, '10314.126747'  => -11, '15.14'         =>  -3, '483.172'       =>  -2,
            '0.405335'      =>  -9, '1725134882'    =>   8, '269876.473040' =>  -7, '31.94'         =>  -2,
            '88.723'        =>   2, '0.321245'      =>  -7, '1134932876'    =>  12, '12366.287397'  =>   5,
            '70.9'          =>  -1, '432.18'        =>   1, '0.294402'      =>   6, '1176528959'    =>   7,
            '4544.32826'    =>  -3, '34.58'         =>   0, '305.562'       =>   1, '0.270004'      =>  -6,
            '519119598'     =>  -8, '456581.139104' => -11, '65.29'         =>   3, '862.335'       =>  -3,
            '0.315414'      =>  -7, '2140092333'    =>  10, '137059.2740'   =>  -9, '54.55'         =>   0,
            '102.418'       =>  -5, '0.196226'      =>   2, '1615928903'    =>  11, '511813.326807' =>   3,
            '43.53'         =>   1, '721.723'       =>  -2, '0.485730'      =>   4, '1216632514'    =>  -4,
            '248307.434089' =>  11, '36.47'         =>   2, '488.341'       =>   0, '0.34604'       =>  10,
            '1758910663'    =>   5, '173818.416947' =>   8, '36.88'         =>  -3, '209.467'       =>  -3,
            '0.460928'      =>  11, '582273698'     =>  -3, '166175.390463' =>  -1, '43.63'         =>  -2,
            '98.204'        =>   2, '0.516467'      => -12, '1559365444'    =>  -6, '46679.30234'   =>   8,
            '27.97'         =>   0, '187.614'       =>  -5, '0.70257'       =>   0, '1886354324'    =>   0,
            '492363.102423' =>  -2, '44.58'         =>  -3, '249.942'       =>  -4, '0.136477'      =>  -3,
            '1357889353'    =>  11, '383185.378195' =>  12, '25.31'         =>  -2, '331.480'       =>   2,
            '0.427866'      =>  -6, '1642764933'    =>   3, '9938.369141'   =>   5, '92.33'         =>  -3,
            '488.999'       =>   3, '0.445313'      =>   5, '1348858011'    =>  -6, '128791.188208' =>   0,
            '69.45'         =>  -2, '112.936'       =>   5, '0.201364'      =>  -1, '162495054'     =>   6,
            '234154.49610'  =>  -9, '26.77'         =>  -1, '530.154'       =>  -5, '0.121155'      => -12,
            '581488470'     => -12, '254555.270756' =>   9, '81.91'         =>   2, '890.256'       =>  -1,
            '0.184203'      =>  -4, '212067454'     =>   4, '413890.285928' =>  -2, '89.79'         =>   1,
            '798.943'       =>   1, '0.424338'      =>   5, '1518296227'    =>  -8, '17044.100945'  => -10,
            '88.40'         =>  -2, '814.47'        =>  -5, '0.83877'       =>  -5, '1979121881'    =>  -3,
            '433033.372786' =>  -6, '35.69'         =>  -1, '234.931'       =>   3, '0.491675'      =>   5,
            '1893353723'    =>   6, '429908.479289' =>   4, '16.72'         =>  -1, '744.633'       =>  -2,
            '0.12874'       =>  -3, '1652500606'    =>   7, '26939.312188'  =>  11, '95.1'          =>   3,
            '777.776'       =>  -5, '0.254411'      => -12, '1602139214'    =>   5, '188034.296767' =>  -3,
            '99.45'         =>  -3, '769.658'       =>   3, '0.73128'       =>  -2, '487927630'     =>   5,
            '329514.146062' =>  10, '13.45'         =>  -2, '523.73'        =>   2, '0.277560'      =>  -1,
            '355146425'     =>   1, '485252.274739' =>  -6, '72.83'         =>  -3, '626.421'       =>   1,
            '0.31743'       =>  -5, '2092908655'    =>   2, '246392.316190' =>  -5, '89.70'         =>   4,
            '376.946'       =>  -3, '0.433827'      =>  -6, '1755052572'    =>  -1, '393176.389443' =>  12,
            '95.23'         =>   1, '872.41'        =>   1, '0.85765'       => -12, '1255127494'    => -10,
            '204365.28532'  =>  -9, '2.63'          =>   4, '816.32'        =>  -1, '0.38750'       =>  -3,
            '1868771565'    =>   9, '447108.325131' =>  10, '21.58'         =>  -1, '185.478'       =>   1,
            '0.109531'      =>   4, '1654866032'    =>   4, '408753.84097'  =>   7, '11.88'         =>   0,
            '327.967'       =>   3, '0.379086'      =>   8, '1429295707'    => -11, '488263.271769' =>   2,
            '89.42'         =>   1, '52.334'        =>   0, '0.377803'      =>  12, '1318328164'    =>   1,
            '176363.206322' =>   0, '92.77'         =>  -3, '541.388'       =>  -5, '0.181873'      => -12,
            '153912328'     => -10, '369317.1551'   => -12, '57.42'         =>   4, '914.395'       =>   2,
            '0.464426'      => -11, '1275606463'    => -10, '318052.487790' =>   0, '11.81'         =>  -2,
            '873.568'       =>  -4, '0.474255'      =>   2, '1448766956'    =>  10, '486044.198732' =>  12,
            '29.55'         =>   0, '264.790'       =>   2, '0.165316'      => -12, '321934385'     =>   4,
            '92094.396649'  =>  10, '1.96'          =>  -4, '129.750'       =>  -5, '0.363401'      =>   3,
            '1124207770'    =>   4, '330513.236221' =>  -3, '91.4'          =>   1, '684.421'       =>   3,
            '0.376355'      =>   2, '539466993'     =>   6, '338682.223800' =>   9, '680.570'       =>  -1,
            '0.151635'      =>  -9, '1666663786'    =>  -6, '330571.213126' =>   8, '1.17'          =>  -1,
            '957.104'       =>  -4, '0.278996'      =>   2, '711170694'     =>   8, '361748.512307' =>   7,
            '46.71'         =>  -2, '78.634'        =>   4, '0.104999'      =>  -1, '1578756048'    =>  11,
            '385689.191722' =>  12, '75.40'         =>   4, '572.127'       =>  -4, '0.87360'       =>  -6,
            '194691078'     =>   4, '232264.409280' =>   0, '9.94'          =>   1, '841.437'       =>  -5,
            '0.164100'      =>   5, '1299079227'    =>   3, '322013.178559' => -11, '321.969'       =>   0,
            '0.308985'      =>   3, '638463866'     =>  -9, '331670.388139' =>   5, '39.56'         =>  -4,
            '604.886'       =>   3, '0.149885'      =>  -8, '1778392533'    =>  -8, '129537.231903' =>  10,
            '44.57'         =>   1, '81.138'        =>   4, '0.376795'      =>   8, '2015329247'    =>   6,
            '213926.299406' => -12, '68.15'         =>   3, '903.28'        =>   2, '0.48715'       =>   3,
            '1736838292'    =>  -5, '107381.29281'  =>  11, '638.88'        =>   0, '0.231656'      =>   4,
            '861705318'     =>   8, '431381.424303' =>  -3, '354.632'       =>  -4, '0.505326'      =>  -9,
            '954188025'     =>   6, '338440.340337' =>   7, '8.55'          =>  -4, '79.450'        =>  -4,
            '0.394793'      =>  -8, '2120370310'    =>  10, '142616.424762' =>  -9, '39.64'         =>  -2,
            '744.207'       =>  -4, '0.458267'      =>   6, '1018468383'    =>  -8, '494834.62802'  =>  -5,
            '12.5'          =>   2, '145.397'       =>   5, '0.170869'      =>  -3, '605361750'     =>  -9,
            '159681.290410' =>  -5, '78.51'         =>  -2, '631.654'       =>   1, '0.199611'      =>   8,
            '19995477'      =>   1, '443496.499715' =>   9, '43.11'         =>  -5, '0.227522'      =>  -1,
            '1268123310'    =>  -4, '341246.469282' =>   5, '16.4'          =>   2, '678.918'       =>  -5,
            '0.270488'      =>   6, '545052123'     =>  -5, '80589.52277'   =>   1, '92.20'         =>  -3,
            '246.275'       =>   3, '0.360145'      =>   7, '276565302'     =>   5, '223106.408766' =>  -6,
            '55.33'         =>   2, '409.420'       =>   4, '0.70752'       =>   6, '124029482'     =>  11,
            '241885.110870' =>  -3, '34.40'         =>   4, '181.390'       =>   1, '0.509781'      =>   9,
            '1196793147'    => -11, '185890.515292' => -10, '68.43'         =>   0, '276.271'       =>  -1,
            '0.18264'       =>   9, '1668393853'    =>  -2, '416898.124919' => -10, '17.65'         =>  -2,
            '216.867'       =>  -2, '0.33306'       =>   5, '1693925058'    =>   8, '492757.75158'  =>   3,
            '703.531'       =>   0, '0.284773'      =>   8, '453410602'     =>   2, '345315.3306'   =>  -4,
            '29.74'         =>   4, '57.823'        =>   5, '0.403424'      =>   9, '1461090654'    =>  10,
            '265804.325180' =>   1, '44.37'         =>  -4, '906.94'        =>   2, '0.372266'      =>  -7,
            '1704091169'    =>   3, '502320.237065' =>  11, '192.310'       =>  -5, '0.17973'       =>   9,
            '1568440410'    => -10, '334644.124436' =>   5, '35.78'         =>  -1, '472.617'       =>   3,
            '0.366749'      =>   8, '1016580040'    =>  -4, '130283.226220' =>   0, '95.36'         =>  -3,
            '382.586'       =>  -3, '0.118419'      =>   0, '1269661192'    =>   3, '175634.120332' =>  11,
            '57.54'         =>   2, '282.918'       =>  -2, '0.61553'       =>  10, '485269598'     =>   3,
            '57798.248757'  =>   2, '65.97'         =>  -4, '171.797'       =>  -5, '0.357599'      =>   3,
            '537426709'     => -12, '513433.306841' =>  -6, '25.7'          =>  -3, '353.997'       =>   2,
            '0.132257'      =>   3, '506054940'     => -10, '369460.181347' =>  -1, '40.60'         =>   2,
            '647.868'       =>   3, '0.145186'      =>   3, '1368273320'    =>   1, '431677.323196' =>   1,
            '79.37'         =>  -2, '68.752'        =>   0, '0.360084'      =>   0, '1318911357'    => -12,
            '252973.167172' =>  11, '4.67'          =>  -1, '487.686'       =>   3, '0.52781'       =>   3,
            '345767518'     =>  12, '256618.516093' =>   6, '31.14'         =>   1, '648.37'        =>   2,
            '0.77394'       =>  -1, '999084791'     =>   6, '339257.496890' =>   7, '38.12'         =>   4,
            '870.71'        =>  -1, '0.251847'      => -10, '148064284'     =>  -8, '477213.292766' =>   4,
            '16.94'         =>  -2, '645.55'        =>  -4, '0.55673'       =>  -8, '1916008879'    =>  -7,
            '505851.282744' =>   2, '45.91'         =>  -4, '323.41'        =>  -5, '0.225493'      => -10,
            '609795303'     =>   1, '85495.101800'  =>   0, '46.21'         =>   3, '347.370'       =>   4,
            '0.270065'      =>  11, '268889639'     =>  -1, '25129.47210'   =>  -2, '13.40'         =>   1,
            '441.793'       =>  -2, '0.289805'      =>  -4, '1869358233'    =>  -7, '378422.17593'  =>   8,
            '83.49'         =>   3, '356.344'       =>   1, '0.145979'      =>  10, '846724011'     =>   9,
            '139953.231849' =>  -2, '0.63'          =>  -3, '937.56'        =>   0, '0.149658'      =>  -6,
            '1060686659'    =>   3, '233601.113091' =>  -9, '100.1'         =>   0, '559.322'       =>   5,
            '0.250137'      =>  -4, '421031725'     =>   3, '2274.242744'   =>   9, '68.32'         =>   0,
            '802.63'        =>  -2, '0.38081'       =>  -4, '2053778662'    =>  -4, '264919.210724' =>   2,
            '88.78'         =>   2, '769.266'       =>   4, '0.55346'       =>  -8, '1647797972'    =>   6,
            '322154.404569' =>  -5, '27.44'         =>   4, '733.52'        =>  -1, '0.42017'       =>  -1,
            '1739861744'    =>  -9, '362615.165402' => -12, '1.31'          =>   0, '161.20'        =>   2,
            '0.171853'      =>  12, '1559012266'    =>  -9, '224083.178484' => -10, '27.13'         =>  -3,
            '533.366'       =>   0, '0.521246'      =>   2, '1496801953'    =>  -1, '252953.203757' =>   8,
            '87.13'         =>   4, '847.399'       =>  -4, '0.435135'      =>  -9, '557542360'     =>   6,
            '430835.360202' =>   1, '88.16'         =>   2, '685.358'       =>   0, '0.134357'      =>   9,
            '695881783'     =>  -4, '82260.422846'  =>  11, '73.20'         =>  -2, '318.254'       =>   5,
            '0.236925'      =>  -3, '194660861'     => -10, '335458.478360' =>   5, '790.385'       =>  -1,
            '0.350204'      =>   7, '293539387'     =>   4, '311913.153924' =>  12, '57.53'         =>   4,
            '491.459'       =>   5, '0.448296'      =>  -7, '676702556'     =>  10, '326500.500669' =>  11,
            '49.46'         =>  -4, '885.88'        =>  -3, '0.350773'      =>   2, '470837535'     =>   6,
            '392572.426864' =>   8, '75.67'         =>   0, '647.564'       =>   4, '0.460393'      =>  -4,
            '564421582'     =>  -2, '191396.464298' =>  -8, '91.64'         =>  -2, '832.567'       =>  -3,
            '0.214503'      =>  -5, '110339611'     =>  -7, '21932.419510'  =>   3, '35.19'         =>  -3,
            '85.458'        =>  -1, '0.229354'      =>  12, '868322234'     =>  -4, '277420.403389' =>   5,
            '100.49'        =>   2, '699.148'       =>   2, '0.520712'      =>  -6, '1026256589'    =>   1,
            '97377.272483'  => -11, '66.19'         =>   1, '613.908'       =>   0, '0.313438'      =>  11,
            '569881040'     =>  -4, '23839.416551'  => -12, '47.77'         =>   1, '439.823'       =>   3,
            '0.356877'      =>  11, '1061041043'    =>   8, '451095.356420' =>  -9, '31.34'         =>   2,
            '811.510'       =>  -4, '0.388167'      => -12, '822243100'     => -10, '57744.224582'  => -12,
            '6.27'          =>   3, '60.84'         =>  -3, '0.522899'      => -12, '1765824910'    =>   7,
            '484461.357917' =>   0, '39.48'         =>   1, '609.753'       =>  -2, '0.339243'      =>   7,
            '1939477099'    =>   0, '261367.6961'   => -12, '90.52'         =>  -2, '702.632'       =>   0,
            '0.371310'      =>   3, '839896957'     =>   3, '63058.165226'  =>   1, '886.380'       =>   4,
            '0.357139'      =>   3, '223717296'     => -12, '30014.315986'  =>   2, '91.49'         =>   0,
            '33.736'        =>   0, '0.345573'      => -10, '1573364975'    =>  -1, '510675.447180' => -11,
            '14.28'         =>  -3, '141.674'       =>   5, '0.394923'      =>  -5, '693454176'     =>  -4,
            '492663.199315' =>   9, '90.36'         =>   0, '834.63'        =>   5, '0.491651'      =>   4,
            '232172289'     =>   6, '405139.43069'  =>   4, '27.67'         =>  -3, '583.563'       =>  -5,
            '0.460724'      =>   6, '186884682'     =>  -5, '90980.14001'   =>  -7, '407.830'       =>   1,
            '0.37423'       =>   4, '815554087'     =>   7, '229295.79961'  =>  -8, '29.53'         =>   4,
            '589.211'       =>   2, '0.167863'      =>  -6, '1268234610'    => -10, '153127.400608' =>  -2,
            '40.100'        =>   3, '592.759'       =>  -1, '0.130577'      =>  -7, '2124172267'    =>   8,
            '4789.223603'   =>   3, '39.7'          =>  -2, '784.390'       =>  -5, '0.30447'       =>  -7,
            '682184859'     =>   5, '61682.319676'  =>   4, '48.81'         =>   0, '203.141'       =>   4,
            '0.223201'      =>   2, '908398046'     =>  -3, '10030.226566'  =>  -2, '70.23'         =>   3,
            '244.656'       =>   0, '0.247178'      =>  10, '1300114146'    =>   5, '190199.379092' =>   7,
            '27.89'         =>   2, '718.790'       =>   4, '0.163032'      =>  12, '763697413'     =>  -7,
            '330370.196479' =>   6, '17.87'         =>   0, '374.945'       =>  -5, '0.140864'      =>   5,
            '897384185'     =>  -2, '107363.409287' =>   9, '56.88'         =>   1, '696.203'       =>  -4,
            '0.346641'      =>  11, '315672920'     =>   0, '130368.407439' => -12, '25.57'         =>  -1,
            '615.658'       =>   5, '0.166828'      =>  10, '662500778'     =>  11, '419572.269106' =>   6,
            '11.7'          =>  -1, '215.966'       =>   1, '0.76981'       =>   8, '690359917'     =>   3,
            '337115.298913' =>  -5, '99.32'         =>  -4, '268.215'       =>  -3, '0.84442'       =>  11,
            '1830461013'    =>   1, '141583.342174' =>  -1, '5.10'          =>   3, '703.74'        =>  -2,
            '0.267173'      =>  -5, '1955300868'    =>  -7, '270237.290195' =>   8, '59.38'         =>   1,
            '655.174'       =>  -3, '0.304600'      =>  -2, '2016725871'    =>  -4, '421952.109660' =>  -2,
            '100.13'        =>   3, '455.590'       =>   4, '0.387782'      =>  11, '326443094'     => -10,
            '488051.349935' =>   4, '25.79'         =>  -2, '982.373'       =>  -4, '0.202479'      =>  -4,
            '1437185531'    =>  11, '203056.248539' =>  -4, '26.53'         =>   0, '386.64'        =>  -4,
            '0.165902'      =>   6, '1802089567'    =>   8, '238281.403726' =>  -1, '10.39'         =>   1,
            '550.431'       =>   4, '0.463228'      =>  -8, '316156841'     =>   8, '74656.280243'  =>   1,
            '438.593'       =>   1, '0.92801'       =>   5, '1044193208'    =>  10, '198279.493211' => -12,
            '709.453'       =>  -5, '0.472872'      =>  -1, '637081832'     =>  -4, '466448.230194' => -11,
            '11.59'         =>   4, '208.643'       =>  -2, '0.466927'      =>  10, '1774366832'    =>   8,
            '438243.107186' =>  -3, '72.92'         =>   1, '966.219'       =>   1, '0.233141'      => -11,
            '886071886'     =>  -9, '89226.158486'  =>   6, '22.29'         =>   2, '620.554'       =>  -1,
            '0.281044'      => -11, '1903262225'    =>  -1, '430692.378619' =>  -3, '34.28'         =>  -3,
            '908.20'        =>   4, '0.515546'      => -11, '418400225'     =>  -1, '108313.191374' =>  -3,
            '12.92'         =>  -1, '341.303'       =>   0, '0.213615'      => -12, '994161677'     =>   8,
            '105500.149119' =>  -5, '42.36'         =>   2, '215.961'       =>   0, '0.151327'      =>   0,
            '2071739529'    =>  11, '178325.89821'  =>   0, '56.97'         =>  -4, '432.296'       =>  -3,
            '0.235877'      =>   1, '823459397'     =>   5, '522933.306540' =>  10, '35.62'         =>  -4,
            '99.588'        =>  -1, '0.314118'      =>  -3, '667770832'     =>   9, '188929.341355' =>   2,
            '69.29'         =>  -3, '535.608'       =>   2, '0.41781'       =>  11, '1384659835'    => -11,
            '508848.336697' =>  -1, '66.40'         =>  -3, '807.889'       =>  -5, '0.90774'       =>   3,
            '1060976301'    =>   0, '26839.447957'  =>   0, '6.83'          =>   3, '469.615'       =>  -3,
            '0.215229'      =>   9, '1943408411'    =>   5, '82774.459025'  => -11, '4.35'          =>  -3,
            '382.412'       =>   4, '0.523800'      =>   6, '445411727'     =>   2, '36591.135582'  =>   0,
            '4.41'          =>  -3, '650.499'       =>  -4, '0.260833'      =>   6, '933383067'     =>  10,
            '505854.310651' =>  -2, '80.32'         =>   1, '153.397'       =>  -1, '0.457857'      =>   8,
            '864716751'     =>  -3, '466315.247704' => -10, '68.38'         =>  -2, '509.298'       =>   2,
            '0.130576'      =>  -7, '248206334'     =>   7, '103785.42163'  =>   5, '75.7'          =>   0,
            '479.951'       =>   3, '0.155093'      =>  -9, '1184600251'    =>   8, '474211.231236' =>   5,
            '68.85'         =>  -2, '555.137'       =>  -2, '0.399205'      =>   3, '900285806'     =>  -3,
            '36693.323581'  =>   9, '9.70'          =>  -1, '458.900'       =>   4, '0.321701'      =>   9,
            '1427377228'    =>  -8, '468605.298404' =>  -8, '82.28'         =>  -4, '610.468'       =>  -1,
            '0.122567'      =>   2, '2071388762'    =>  -7, '56055.1825'    =>   3, '38.36'         =>  -4,
            '253.223'       =>   3, '0.49607'       =>   5, '1924595010'    =>   8, '849.414189'    =>   6,
            '86.14'         =>   1, '583.502'       =>   3, '0.84266'       =>  -7, '1001248538'    =>   6,
            '203942.300501' =>  -9, '64.54'         =>   1, '342.276'       =>  -1, '0.22490'       =>   3,
            '990502631'     =>  -3, '68229.242671'  =>  11, '7.58'          =>   0, '352.182'       =>   5,
            '0.308290'      =>  -1, '431852030'     =>  -4, '245610.309375' => -11, '25.26'         =>   1,
            '667.511'       =>  -3, '0.152985'      =>   1, '1973200073'    =>   1, '258426.25679'  =>   6,
            '51.28'         =>  -3, '249.157'       =>  -4, '0.374691'      =>  -5, '1359777331'    =>   6,
            '419342.53299'  =>  10, '73.76'         =>   2, '26.359'        =>   4, '0.301405'      =>  -2,
            '2069275744'    =>   4, '211798.239332' =>   0, '68.5'          =>   1, '158.156'       =>   0,
            '0.234814'      =>   4, '832155923'     =>   9, '412347.98217'  => -10, '62.92'         =>  -2,
            '812.74'        =>  -2, '0.116050'      =>   4, '680424040'     =>   4, '74953.377917'  =>  -8,
            '34.32'         =>   2, '805.894'       =>  -3, '0.233956'      =>  -2, '588372584'     =>  -4,
            '221640.31705'  =>   8, '35.64'         =>   1, '943.330'       =>  -1, '0.318948'      =>  -1,
            '265975454'     =>   1, '385887.139888' =>  11, '96.99'         =>  -1, '234.502'       =>   2,
            '0.334051'      =>   2, '1984107482'    =>  11, '409812.181753' =>  10, '96.32'         =>   3,
            '725.381'       =>   0, '0.104498'      =>   8, '754079843'     =>  12, '479393.45700'  =>  -4,
            '46.54'         =>   1, '231.781'       =>  -4, '0.426330'      =>   4, '1313355869'    =>  -2,
            '172630.206167' =>   6, '506.350'       =>   1, '0.160569'      => -12, '9374862'       =>   1,
            '457426.481681' =>   4, '48.21'         =>   0, '371.236'       =>  -5, '0.18262'       =>   8,
            '483859935'     => -11, '412138.290760' =>   4, '86.23'         =>   3, '870.412'       =>   0,
            '0.464076'      =>   8, '816125629'     =>   5, '196928.132387' =>  10, '72.29'         =>  -3,
            '313.701'       =>   2, '0.63312'       =>  11, '10216986'      =>   6, '200164.414633' =>  -4,
            '66.76'         =>  -1, '237.527'       =>   1, '0.35438'       =>  12, '1495884017'    =>  -2,
            '158165.37846'  =>  -1, '19.41'         =>  -4, '135.711'       =>   0, '0.47582'       =>  -7,
            '2056545177'    =>   8, '93175.177962'  => -10, '76.55'         =>  -4, '113.71'        =>  -1,
            '0.48159'       =>  11, '766776723'     =>  -9, '307529.345367' =>   7, '77.5'          =>  -2,
            '575.837'       =>  -4, '0.417865'      =>   9, '1521506019'    =>  11, '12107.464637'  =>   6,
            '97.86'         =>   4, '273.570'       =>   3, '0.114173'      =>   9, '20096671'      =>   4,
            '236082.312435' =>   2, '89.0'          =>  -2, '168.939'       =>   1, '0.23631'       =>  -2,
            '749143830'     =>   6, '257069.195003' =>   6, '28.61'         =>   1, '403.712'       =>   3,
            '0.143607'      =>  -5, '2082566916'    =>  11, '264692.220233' =>  -5, '346.909'       =>  -4,
            '0.402410'      => -12, '1651766620'    =>  11, '370943.136044' => -10, '85.77'         =>  -1,
            '508.355'       =>  -2, '0.427580'      =>  12, '1987714863'    =>   3, '478356.225686' =>   7,
            '17.94'         =>   3, '31.900'        =>   5, '0.33942'       =>   8, '1658612685'    =>   4,
            '410829.251590' =>   8, '83.94'         =>  -2, '220.549'       =>   3, '0.113337'      =>  -9,
            '1639086102'    =>  -3, '78713.354235'  =>  -9, '47.74'         =>  -1, '651.323'       =>  -5,
            '0.245932'      =>  11, '1261711612'    =>  -3, '116253.194576' =>  -6, '64.55'         =>  -2,
            '327.473'       =>   2, '0.241987'      =>  -7, '1285865121'    =>   5, '435672.392645' =>  -7,
            '10.37'         =>  -1, '388.795'       =>  -1, '0.161827'      =>   5, '230244765'     =>  -6,
            '265038.172466' =>  10, '65.91'         =>  -2, '595.102'       =>   4, '0.434638'      =>  -3,
            '126129840'     =>   2, '413719.466465' =>  -2, '29.28'         =>   2, '678.446'       =>  -1,
            '0.208034'      =>  -7, '1065944072'    =>   6, '341032.990'    =>   0, '23.48'         =>   4,
            '373.423'       =>  -5, '0.389585'      => -10, '1441107220'    =>  -6, '356509.241264' =>   8,
            '422.816'       =>  -3, '0.342884'      =>   4, '873838463'     =>  -7, '375403.30084'  =>  -9,
            '98.38'         =>   0, '385.399'       =>   3, '0.257833'      =>  -7, '1028034561'    =>  -9,
            '495168.83206'  =>  -8, '1.78'          =>   1, '980.42'        =>  -4, '0.331573'      =>  12,
            '387462598'     =>   5, '443818.469998' => -12, '876.765'       =>   0, '0.53328'       =>   9,
            '29407914'      =>  -6, '219873.502347' =>   2, '60.76'         =>  -4, '908.795'       =>  -3,
            '0.459345'      =>  -6, '2046247729'    =>  -2, '244695.419102' => -11, '80.40'         =>   1,
            '870.511'       =>  -4, '0.388478'      =>  11, '427727793'     =>   3, '101019.324298' =>   6,
            '66.31'         =>   4, '522.594'       =>   0, '0.408007'      =>  -6, '1499912745'    =>  -1,
            '276166.86597'  =>   6, '73.33'         =>   0, '506.748'       =>  -1, '0.235478'      =>   9,
            '1045213811'    =>   0, '127791.356198' =>   4, '61.93'         =>   4, '404.785'       =>  -2,
            '0.353893'      =>  10, '776947132'     =>   2, '154285.465851' =>  -3, '37.26'         =>   3,
            '631.660'       =>  -3, '0.265100'      =>  12, '1038752775'    => -10, '397025.381393' =>  12,
            '69.28'         =>  -4, '581.356'       =>   5, '0.247522'      =>   6, '635005176'     => -12,
            '211983.309316' =>   8, '41.9'          =>   4, '337.842'       =>   1, '0.173992'      =>  -7,
            '723221456'     =>  -4, '177913.49305'  =>   8, '11.22'         =>  -4, '174.875'       =>  -5,
            '0.475533'      =>  -2, '981140567'     =>  -4, '94672.451519'  =>  -7, '81.20'         =>   4,
            '776.331'       =>  -5, '0.1686'        =>   9, '435229915'     =>   4, '250840.284170' =>  -4,
            '55.55'         =>  -4, '622.423'       =>   4, '0.25344'       =>   6, '146835656'     =>  -4,
            '279294.130521' =>   0, '15.17'         =>  -2, '185.308'       =>  -4, '0.21281'       =>  -1,
            '299802106'     =>  -5, '341504.324034' =>  -6, '64.34'         =>  -4, '185.260'       =>   4,
            '0.475653'      =>  -5, '1376528125'    =>  -5, '426514.91072'  =>  12, '50.38'         =>  -3,
            '522.535'       =>  -5, '0.523764'      =>  -4, '1398866761'    =>   2, '429538.158736' =>  -9,
            '39.40'         =>   1, '284.128'       =>  -5, '0.312023'      =>  -5, '1275671412'    => -12,
            '5929.213670'   =>  -7, '59.91'         =>  -2, '941.134'       =>   5, '0.147235'      =>  -1,
            '106169053'     =>  -9, '142750.455458' =>  11, '47.62'         =>  -4, '649.709'       =>  -1,
            '0.490043'      =>   1, '1246299209'    =>  -5, '283381.310201' =>   1, '83.52'         =>   4,
            '462.800'       =>  -1, '0.490291'      =>   2, '1520998937'    => -10, '6895.514087'   =>  12,
            '0.32'          =>  -2, '743.520'       =>  -1, '0.150701'      =>  -9, '962438429'     =>  -1,
            '515717.518352' =>   9, '94.28'         =>   4, '727.906'       =>   3, '0.158088'      =>   2,
            '1458866058'    =>   9, '432610.363063' => -12, '95.100'        =>   2, '434.748'       =>   3,
            '0.303128'      =>  -4, '1690353126'    => -12, '473459.404113' => -12, '69.96'         =>  -1,
            '753.376'       =>  -4, '0.183909'      => -12, '1890384328'    =>   4, '360580.369842' => -12,
            '38.66'         =>   2, '475.112'       =>  -1, '0.423296'      =>  -9, '369343490'     =>   5,
            '375810.39343'  =>   2, '190.438'       =>  -5, '0.120623'      =>  -1, '303710229'     =>   7,
            '429459.434728' => -10, '51.20'         =>   0, '589.680'       =>  -3, '0.380255'      =>  -5,
            '2121572252'    =>   0, '43009.369484'  =>  10, '44.41'         =>   0, '497.694'       =>  -4,
            '0.503517'      =>   2, '750917862'     =>  11, '17861.88501'   => -10, '71.44'         =>   3,
            '624.638'       =>   5, '0.482829'      =>   2, '178657038'     =>  -3, '136615.86626'  =>  10,
            '61.5'          =>  -2, '172.139'       =>  -4, '0.384425'      =>   8, '1574595991'    => -10,
            '105825.402284' => -10, '62.97'         =>  -3, '878.194'       =>   2, '0.248212'      =>  -5,
            '1434390361'    => -10, '12755.486808'  =>   1, '16.33'         =>   1, '450.861'       =>   3,
            '0.136195'      =>  11, '94135708'      =>   2, '354832.128808' =>  12, '44.769'        =>   5,
            '0.182803'      =>  -9, '557554733'     =>  -6, '418674.148877' => -12, '93.10'         =>   0,
            '911.305'       =>   1, '0.447984'      =>  -1, '138482844'     =>   2, '292468.388641' =>  10,
            '472.740'       =>   1, '0.323103'      =>  -4, '1226149205'    => -10, '54987.193739'  =>   9,
            '89.8'          =>  -4, '692.67'        =>   3, '0.85097'       => -11, '1319943123'    =>  11,
            '429890.90431'  =>  -7, '49.64'         =>   4, '198.959'       =>  -3, '0.276511'      =>  11,
            '30177090'      =>   2, '236303.62355'  => -10, '17.36'         =>   1, '102.384'       =>  -2,
            '0.79893'       =>  -3, '674979790'     =>   1, '405348.70392'  =>   7, '9.43'          =>   2,
            '616.578'       =>   1, '0.284218'      =>   1, '253346959'     => -10, '106539.298155' =>   1,
            '18.85'         =>   2, '169.131'       =>   4, '0.294728'      =>  -6, '1722210543'    =>  -5,
            '26708.301522'  =>   9, '31.11'         =>   4, '694.36'        =>   2, '0.124103'      =>  -6,
            '810995233'     =>   2, '70871.304536'  =>  -4, '25.10'         =>   3, '975.824'       =>  -2,
            '0.120505'      =>  -2, '1495393117'    =>  -3, '137570.391794' => -12, '74.93'         =>  -2,
            '619.75'        =>  -4, '0.467372'      =>  -8, '1662762076'    =>  -5, '349689.476819' =>  -6,
            '72.10'         =>   1, '383.667'       =>  -1, '0.416946'      =>  -8, '1253560206'    =>  10,
            '486994.443615' =>   0, '12.98'         =>   0, '135.77'        =>   0, '0.173148'      =>   2,
            '1909493874'    =>  12, '124884.291587' => -10, '14.74'         =>  -4, '107.573'       =>   4,
            '0.160192'      =>   0, '867063434'     =>  -3, '450761.174392' =>  10, '87.86'         =>  -4,
            '356.210'       =>  -2, '0.489071'      =>  -2, '803631773'     =>   1, '338212.321083' =>   8,
            '70.52'         =>   1, '364.501'       =>   4, '0.456586'      =>  -1, '1785975209'    =>   8,
            '364576.362502' =>  -6, '79.48'         =>  -1, '803.642'       =>   4, '0.109176'      => -11,
            '1323336416'    =>   2, '207735.137004' =>   2, '92.90'         =>   2, '786.663'       =>  -2,
            '0.130848'      =>   8, '1331158018'    =>   1, '424006.165278' => -10, '75.56'         =>  -1,
            '428.695'       =>  -5, '0.264127'      =>   4, '530578942'     =>  10, '254527.337271' =>  -9,
            '42.30'         =>   2, '615.362'       =>   2, '0.224771'      =>  -3, '1962960365'    =>   3,
            '363152.378956' => -11, '14.11'         =>   0, '141.456'       =>  -5, '0.426860'      =>   6,
            '1349110264'    =>  11, '41910.59612'   => -11, '80.12'         =>   0, '839.378'       =>  -1,
            '0.114787'      =>   6, '736831764'     =>   8, '448394.18754'  =>  -9, '50.100'        =>  -3,
            '25.470'        =>   0, '0.393347'      =>  12, '2029781516'    =>  -5, '181227.13175'  =>  -3,
            '656.11'        =>  -3, '0.203296'      =>   3, '931979866'     =>  -7, '190494.151640' =>   6,
            '51.27'         =>  -3, '797.239'       =>   4, '0.410048'      =>  -5, '1271106666'    =>  12,
            '192689.491556' =>   6, '99.24'         =>   3, '488.802'       =>  -4, '0.60574'       =>  -8,
            '863081313'     =>  -8, '482679.401208' =>   1, '99.35'         =>   2, '184.752'       =>  -3,
            '0.257832'      =>  -2, '914827407'     =>   1, '136969.416036' => -10, '65.0'          =>  -4,
            '169.741'       =>   3, '0.178673'      =>   3, '1698545278'    =>   9, '212734.373075' =>   6,
            '40.49'         =>  -2, '691.776'       =>  -2, '0.66820'       => -10, '1373687051'    =>  -6,
            '184194.472342' =>   1, '800.551'       =>  -4, '0.210012'      =>  10, '1640724691'    =>  10
        );
        if (!Zend_Locale_Math::isBcmathDisabled()) {
            // If no BCMath extension, the round below falls through to PHP's round(),
            // which actually produces incorrect results, and fails these tests.
            // There are many such examples ...
            $this->assertEquals('56055.1812', Zend_Locale_Math::round('56055.18115', 4));
            $this->assertEquals(    '639.80', Zend_Locale_Math::round('639.795',     2));
            $this->assertEquals(    '267.84', Zend_Locale_Math::round('267.835',     2));
            $this->assertEquals(   '0.30252', Zend_Locale_Math::round('0.302515',    5));
            $this->assertEquals(    '0.3667', Zend_Locale_Math::round('0.36665',     4));
        }

        foreach ($rounder as $val => $precision) {
//            $this->assertEquals((string) round($val, $precision), (string)Zend_Locale_Math::round($val, $precision));
        }
    }

    public function testAdd()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(3, Zend_Locale_Math_PhpMath::Add(1, 2   ));
        $this->assertEquals(2, Zend_Locale_Math_PhpMath::Add(null, 2));
        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        //try {
        //    $this->assertEquals(9E+200, Zend_Locale_Math_PhpMath::Add(9E+100, 9E+200));
        //    $this->fail("exception expected");
        //} catch (Zend_Locale_Math_Exception $e) {
        //    $this->assertEquals(array(9E+100, 9E+200, 9E+200), $e->getResults());
        //    // success
        //}
        $this->assertEquals(15,  Zend_Locale_Math_PhpMath::Add( 10.4444,  4.5556, 2));
        $this->assertEquals(15,  Zend_Locale_Math_PhpMath::Add( 10.4444,  4.5556, 0));
        $this->assertEquals(-15, Zend_Locale_Math_PhpMath::Add(-10.4444, -4.5556, 0));
    }

    public function testSub()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(-1, Zend_Locale_Math_PhpMath::Sub(   1, 2));
        $this->assertEquals(-2, Zend_Locale_Math_PhpMath::Sub(null, 2));
        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        // $this->assertEquals(9E+300, Zend_Locale_Math_PhpMath::Sub(-9E+100, -9E+300));
        $this->assertEquals( 5.89, Zend_Locale_Math_PhpMath::Sub( 10.4444,  4.5556, 2));
        $this->assertEquals( 6,    Zend_Locale_Math_PhpMath::Sub( 10.4444,  4.5556, 0));
        $this->assertEquals(-6,    Zend_Locale_Math_PhpMath::Sub(-10.4444, -4.5556, 0));
        $this->assertEquals(-1,    Zend_Locale_Math_PhpMath::Sub( 10,      11,      2));
    }

    public function testPow()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(1, Zend_Locale_Math_PhpMath::Pow(   1, 2));
        $this->assertEquals(0, Zend_Locale_Math_PhpMath::Pow(null, 2));
        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        //try {
        //    $this->assertEquals(0, Zend_Locale_Math_PhpMath::Pow(9E+300, 9E+200));
        //    $this->fail("exception expected");
        //} catch (Zend_Locale_Math_Exception $e) {
        //    // success
        //}
        $this->assertEquals(    11899.64, Zend_Locale_Math_PhpMath::Pow( 10.4444, 4.5556, 2));
        $this->assertEquals(       11900, Zend_Locale_Math_PhpMath::Pow( 10.4444, 4.5556, 0));
        $this->assertEquals(       11900, Zend_Locale_Math_PhpMath::Pow(-10.4444, 4,      0));
        $this->assertEquals(100000000000, Zend_Locale_Math_PhpMath::Pow( 10,     11,      2));
    }

    public function testMul()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(2, Zend_Locale_Math_PhpMath::Mul(   1, 2));
        $this->assertEquals(0, Zend_Locale_Math_PhpMath::Mul(null, 2));
        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        //try {
        //    $this->assertEquals(0, Zend_Locale_Math_PhpMath::Mul(9E+300, 9E+200));
        //    $this->fail("exception expected");
        //} catch (Zend_Locale_Math_Exception $e) {
        //    // success
        //}
        $this->assertEquals( 47.58, Zend_Locale_Math_PhpMath::Mul( 10.4444, 4.5556, 2));
        $this->assertEquals( 48,    Zend_Locale_Math_PhpMath::Mul( 10.4444, 4.5556, 0));
        $this->assertEquals(-42,    Zend_Locale_Math_PhpMath::Mul(-10.4444, 4,      0));
        $this->assertEquals(110,    Zend_Locale_Math_PhpMath::Mul( 10,     11,      2));
    }

    public function testDiv()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(1, Zend_Locale_Math_PhpMath::Div(   1, 2));
        $this->assertEquals(0, Zend_Locale_Math_PhpMath::Div(null, 2));
        try {
            $this->assertEquals(0, Zend_Locale_Math_PhpMath::Div(10, null));
            $this->fail("exception expected");
        } catch (Zend_Locale_Math_Exception $e) {
            // success
        }

        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        // $this->assertEquals(0, Zend_Locale_Math_PhpMath::Div(9E-300, 9E+200));
        $this->assertEquals( 2.29, Zend_Locale_Math_PhpMath::Div( 10.4444, 4.5556, 2));
        $this->assertEquals( 2,    Zend_Locale_Math_PhpMath::Div( 10.4444, 4.5556, 0));
        $this->assertEquals(-3,    Zend_Locale_Math_PhpMath::Div(-10.4444, 4,      0));
        $this->assertEquals( 0.91, Zend_Locale_Math_PhpMath::Div( 10,     11,      2));
    }

    public function testComp()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(-1, Zend_Locale_Math_PhpMath::Comp(       1,  2        ));
        $this->assertEquals(-1, Zend_Locale_Math_PhpMath::Comp(    null,  2        ));
        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        // $this->assertEquals(-1, Zend_Locale_Math_PhpMath::Comp(  9E+100, 9E+200    ));
        $this->assertEquals( 1, Zend_Locale_Math_PhpMath::Comp( 10.5556, 10.4444, 2));
        $this->assertEquals( 0, Zend_Locale_Math_PhpMath::Comp( 10.5556, 10.4444, 0));
        $this->assertEquals(-1, Zend_Locale_Math_PhpMath::Comp(-10.4444, -4.5556, 0));
    }

    public function testSqrt()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(1,    Zend_Locale_Math_PhpMath::Sqrt(     1));
        $this->assertEquals(0,    Zend_Locale_Math_PhpMath::Sqrt(  null));
        $this->assertEquals(3.25, Zend_Locale_Math_PhpMath::Sqrt(10.5556, 2));
        $this->assertEquals(3,    Zend_Locale_Math_PhpMath::Sqrt(10.5556, 0));
        $this->assertEquals(NULL, Zend_Locale_Math_PhpMath::Sqrt(-10.4444));
    }

    public function testMod()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertEquals(1, Zend_Locale_Math_PhpMath::Mod(   1, 2));
        $this->assertEquals(0, Zend_Locale_Math_PhpMath::Mod(null, 2));
        $this->assertEquals(null, Zend_Locale_Math_PhpMath::Mod(10, null));
        /**
         * BCMath extension doesn't actually operatest with a scientific notation (e.g. 1.2e+100)
         * So we shouldn't test numbers such as -9E+100, but probably should care about correct
         * float => string conversion
         *
         * @todo provide correct behavior
         */
        //$this->assertEquals(0, Zend_Locale_Math_PhpMath::Mod(9E-300, 9E+200));
        $this->assertEquals(2,  Zend_Locale_Math_PhpMath::Mod( 10.4444, 4.5556));
        $this->assertEquals(2,  Zend_Locale_Math_PhpMath::Mod( 10.4444, 4.5556));
        $this->assertEquals(-2, Zend_Locale_Math_PhpMath::Mod(-10.4444, 4     ));
        $this->assertEquals(10, Zend_Locale_Math_PhpMath::Mod( 10,     11     ));
    }

    public function testScale()
    {
        Zend_Locale_Math_PhpMath::disable();
        $this->assertTrue(Zend_Locale_Math_PhpMath::Scale(3));
        try {
            $this->assertTrue(Zend_Locale_Math_PhpMath::Scale(10));
            $this->fail("exception expected");
        } catch (Zend_Locale_Math_Exception $e) {
            // success
        }
        $this->assertEquals(1, Zend_Locale_Math_PhpMath::Comp(10.5556, 10.4444));
        $this->assertTrue(Zend_Locale_Math_PhpMath::Scale(0));
        $this->assertEquals(0, Zend_Locale_Math_PhpMath::Comp(10.5556, 10.4444));
    }

    public function testExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('1000', Zend_Locale_Math::exponent('1e3'));
        $this->assertEquals('10320', Zend_Locale_Math::exponent('1.032e4'));
        $this->assertEquals('10320', Zend_Locale_Math::exponent('10.32e3'));
        $this->assertEquals('1000', Zend_Locale_Math::exponent('1e+3'));
        $this->assertEquals('0.001', Zend_Locale_Math::exponent('1e-3', 3));
        $this->assertEquals('0.0001032', Zend_Locale_Math::exponent('1.032e-4', 7));
        $this->assertEquals('0.01032', Zend_Locale_Math::exponent('10.32e-3', 5));
    }

    public function testAddingWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('1300', Zend_Locale_Math::Add('1e3', 300));
        $this->assertEquals('1300', Zend_Locale_Math::Add(300, '1e3'));
    }

    public function testSubbingWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('700', Zend_Locale_Math::Sub(1000, '0.3e3'));
    }

    public function testPowerWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('1000000', Zend_Locale_Math::Pow('1e3', 2));
    }

    public function testMultiplyingWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('2000', Zend_Locale_Math::Mul('1e3', 2));
        $this->assertEquals('2000', Zend_Locale_Math::Mul(2, '1e3'));
    }

    public function testDivisionWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('500', Zend_Locale_Math::Div('1e3', 2));
    }

    public function testSqrtWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('31.62', Zend_Locale_Math::Sqrt('1e3', 2));
    }

    public function testModuloWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('0', Zend_Locale_Math::Mod('1e3', 2));
    }

    public function testComparisonWithBCMathAndExponent()
    {
        if (!extension_loaded('bcmath')) {
            $this->markTestSkipped('BCMath extension not loaded, test skipped');
            return;
        }

        $this->assertEquals('1', Zend_Locale_Math::Comp('1e3', 2));
        $this->assertEquals('-1', Zend_Locale_Math::Comp(2, '1e3'));
        $this->assertEquals('0', Zend_Locale_Math::Comp('1e3', '1e3'));
    }

    public function testNegativeRounding()
    {
        $this->assertEquals(               '-3', Zend_Locale_Math::round('-3.4'));
        $this->assertEquals(        round(-3.4), Zend_Locale_Math::round('-3.4'));
        $this->assertEquals(               '-4', Zend_Locale_Math::round('-3.5'));
        $this->assertEquals(        round(-3.5), Zend_Locale_Math::round('-3.5'));
        $this->assertEquals(               '-4', Zend_Locale_Math::round('-3.6'));
        $this->assertEquals(        round(-3.6), Zend_Locale_Math::round('-3.6'));
        $this->assertEquals(               '-4', Zend_Locale_Math::round('-3.6', 0));
        $this->assertEquals(      round(-3.6,0), Zend_Locale_Math::round('-3.6', 0));
        $this->assertEquals(            '-1.96', Zend_Locale_Math::round('-1.95583', 2), '', 0.02);
        $this->assertEquals(  round(-1.95583,2), Zend_Locale_Math::round('-1.95583', 2), '', 0.02);
        $this->assertEquals(           -1242000, Zend_Locale_Math::round('-1241757', -3), '', 250);
        $this->assertEquals(round(-1241757, -3), Zend_Locale_Math::round('-1241757', -3), '', 250);
        $this->assertEquals(              -5.05, Zend_Locale_Math::round('-5.045', 2), '', 0.02);
        $this->assertEquals(   round(-5.045, 2), Zend_Locale_Math::round('-5.045', 2), '', 0.02);
        $this->assertEquals(              -5.06, Zend_Locale_Math::round('-5.055', 2), '', 0.02);
        $this->assertEquals(   round(-5.055, 2), Zend_Locale_Math::round('-5.055', 2), '', 0.02);
    }
}
