/*
 *   XPP - The X Printing Panel
 *   --------------------------
 *
 *   Definition of the "Input_Slider" class which is a widget to enter
 *   numerical values by both an input field and a slider.
 *
 *   Copyright 2000 by Till Kamppeter
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of the
 *   License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *   02111-1307  USA
 *
 */

#ifndef Input_Slider_H
#define Input_Slider_H

#include <FL/Fl_Group.H>
#include <FL/Fl_Slider.H>
#include <FL/Fl_Value_Input.H>
#include <FL/Fl_Valuator.H>
#include <stdio.h>

class Input_Slider : public Fl_Group {
    Fl_Slider *slider_;
    Fl_Value_Input *inputfield_;
public:
    FL_EXPORT Input_Slider(int x,int y,int w,int h, const char *l = 0);
    // values which have to be kept in sync in both the input field and
    // the scroll bar
    double minimum() const {return inputfield_->minimum();}
    void minimum(double s) {
      slider_->minimum(s);
      inputfield_->minimum(s);
    }
    double maximum() const {return inputfield_->maximum();}
    void maximum(double s) {
      slider_->maximum(s);
      inputfield_->maximum(s);
    }
    void range(double s, double t) {
      slider_->range(s, t);
      inputfield_->range(s, t);
    }
    double value() const {return inputfield_->value();}
    void value(double s) {
      slider_->value(inputfield_->clamp(s));
      inputfield_->value(inputfield_->clamp(s));
    }
    // The step width is only applied to the input field
    double step() const {return inputfield_->step();}
    void step(double s) {
      slider_->step(s);
    }
    void step(int a, int b) {
      slider_->step(a, b);
    }
    void step(int s) {
      slider_->step(s);
    } 
    // Access to the "clamp" function of the input field
    double clamp(double s) {return inputfield_->clamp(s);}
    // Basic widget elements have to be set in both sub-widgets
    void when(Fl_When s) {
      Fl_Widget::when(s);
      slider_->when(Fl_Widget::when());
      inputfield_->when(Fl_Widget::when());
    }
    void box(Fl_Boxtype s) {
      Fl_Widget::box(s);
      slider_->box(Fl_Widget::box());
    }
    void input_box(Fl_Boxtype s) {
      inputfield_->box(Fl_Widget::box());
    }
    void color(Fl_Color s) {
      Fl_Widget::color(s);
      slider_->color(Fl_Widget::color());
    }
    void input_color(Fl_Color s) {
      inputfield_->color(Fl_Widget::color());
    }
    void selection_color(Fl_Color s) {
      Fl_Widget::selection_color(s);
      slider_->selection_color(Fl_Widget::selection_color());
    }
    void input_selection_color(Fl_Color s) {
      inputfield_->selection_color(Fl_Widget::selection_color());
    }
    // Manipulate the parameters for the slider
    int horizontal() const {return slider_->type()&1;}
    double slider_size() const {return slider_->slider_size();}
    void slider_size(double s) {slider_->slider_size(s);}
    Fl_Boxtype slider() const {return slider_->slider();}
    void slider(Fl_Boxtype s) {slider_->slider(s);}
    uchar type() const {return slider_->type();}
    void type(uchar s) {
      slider_->type(s);
      int sxx = x(), syy = y(), sww = w(), shh = h();
      int ixx = x(), iyy = y(), iww = w(), ihh = h();
      if (s&1) {
        iww = 40; sxx += 45; sww -= 45;
      } else {
        syy += 30; ihh = 25; shh -= 30;
      }
      slider_->resize(sxx,syy,sww,shh);
      inputfield_->resize(ixx,iyy,iww,ihh);
    }
    // Manipulate the parameters for the input field
    Fl_Font textfont() const {return (Fl_Font)(inputfield_->textfont());}
    void textfont(uchar s) {inputfield_->textfont(s);}
    uchar textsize() const {return inputfield_->textsize();}
    void textsize(uchar s) {inputfield_->textsize(s);}
    Fl_Color textcolor() const {
      return (Fl_Color)(inputfield_->textcolor());
    }
    void textcolor(uchar s) {inputfield_->textcolor(s);}
    Fl_Color cursor_color() const {
      return (Fl_Color)(inputfield_->cursor_color());
    }
    void cursor_color(uchar s) {inputfield_->cursor_color(s);}
private:
    // callbacks to keep the input field and the slider in sync
    inline void slider_cbi(Fl_Slider* o, void* v);
    static void slider_cb(Fl_Slider* o, void* v);
    inline void inputfield_cbi(Fl_Value_Input* o, void* v);
    static void inputfield_cb(Fl_Value_Input* o, void* v);
};

#endif
