/*
 * cpufreq plugin for Xfce4.
 *
 * Copyright (c) 2005 Joshua Kwan <joshk@triplehelix.org>
 *
 * Some code was used as a starter, but is mostly gone. Nevertheless, this code
 * was:
 * 
 * Copyright (c) 2003 Nicholas Penwarden <toth64@yahoo.com>
 * Copyright (c) 2003 Benedikt Meurer <benedikt.meurer@unix-ag.uni-siegen.de>
 * Copyright (c) 2003 edscott wilson garcia <edscott@users.sourceforge.net>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define GETTEXT_PACKAGE "xfce4-cpufreq"
#define PACKAGE_LOCALE_DIR "/usr/share/locale"

#include <ctype.h>
#include <string.h>

#include <cpufreq.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include <libxfce4panel/xfce-panel-plugin.h>

#define COLOR_LOW_DEFAULT "#ff0000"
#define COLOR_HI_DEFAULT "#00ff00"

struct cpufreq_monitor
{
	GtkWidget* pbar;
	GtkWidget* label;
	GtkWidget* vbox;
	GtkWidget* ebox;

	unsigned long min, max;

	GtkOrientation orientation;
	int nr_cpus;
	int cpu;
	int timeoutid;

	gboolean invert_colors;
	gboolean showed_warning;

	GdkColor colorLow;
	GdkColor colorHi;

	char** cpustrings;

	XfcePanelPlugin *plugin;

	struct cpufreq_available_frequencies* freqs;
};

static int
get_nr_cpus (void)
{
	FILE *f = fopen("/proc/stat", "r");
	char buf[512];
	int nr_cpus = 0;

	if (!f)
		return -1;

	while (fgets(buf, 512, f) != NULL)
	{
		if (strncmp(buf, "cpu", 3) == 0)
		{
			if (isdigit(buf[3]))
				nr_cpus++;
		}
	}

	fclose(f);

	return nr_cpus;
}

static char*
get_cpu_name (int cpu)
{
	FILE *f = fopen("/proc/cpuinfo", "r");
	int cur_cpu = -1;
	char buf[512];

	if (!f)
		return strdup(_("Processor model unknown"));
	
	while (fgets(buf, 512, f) != NULL)
	{
		if (strncmp(buf, "processor", 9) == 0)
		{
			cur_cpu++;
		}
		else if (cur_cpu == cpu && strncmp(buf, "model name", 10) == 0)
		{
			char* p = strchr(buf, ':');

			if (p)
			{
				char *q = NULL;
				fclose (f);
				
				if ((q = strchr(buf, '\n')) != NULL)
					*q = '\0';
				
				return strdup(p + 2);
			}
		}
	}

	fclose(f);
	
	return strdup(_("Processor model unknown"));
}

static gboolean
update_cpufreq_status (struct cpufreq_monitor *c)
{
	unsigned int cur = cpufreq_get_freq_kernel(c->cpu);
	char buf[256];

	if (cur == 0) /* workaround for Transmeta Crusoe Longrun, maybe others as well */
		cur = cpufreq_get_freq_hardware(c->cpu);

	if (cur == 0) /* error */
	{
		if (c->showed_warning) return TRUE;

		GtkWindow* win = GTK_WINDOW(gtk_widget_get_toplevel(c->vbox));
		
		if (c->cpu == 0) /* MUST work, unless cpufreq support is not available */
		{
			GtkWidget* d = gtk_message_dialog_new(win, 0,
				GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
				_("CPUFreq support is not present. The CPUFreq Monitor will not function this session."));
			c->showed_warning = (gtk_dialog_run(GTK_DIALOG(d)) == GTK_RESPONSE_OK);
			gtk_widget_destroy(d);

			gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(c->pbar), 0);
			gtk_label_set_markup(GTK_LABEL(c->label), _("N/A"));

			return TRUE; /* not an indicator of success */
		}
		else /* some other error, fall back to 0 */
		{
			GtkWidget* d = gtk_message_dialog_new(win, 0,
				GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
				_("CPU %d is not present. The CPUFreq Monitor will monitor CPU 0 for this session."), c->cpu);
			c->showed_warning = (gtk_dialog_run(GTK_DIALOG(d)) == GTK_RESPONSE_OK);
			gtk_widget_destroy(d);

			c->cpu = 0;
			return TRUE; /* to get to the top of error-handling */
		}
	}
	
	if (cur < 1000000)
		snprintf(buf, 256, "<span size=\"x-small\">%d</span><span size=\"small\"> </span><span size=\"x-small\">MHz</span>", cur / 1000);
	else
		snprintf(buf, 256, "<span size=\"small\">%.1f GHz</span>", (gdouble)cur / 1000000);

	if ((cur < c->max && !c->invert_colors) || 
					(cur == c->max && c->invert_colors))
		gtk_widget_modify_bg(c->pbar, GTK_STATE_PRELIGHT, &(c->colorLow));
	else
		gtk_widget_modify_bg(c->pbar, GTK_STATE_PRELIGHT, &(c->colorHi));

	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(c->pbar), (gdouble)cur / c->max);
	gtk_label_set_markup(GTK_LABEL(c->label), buf);

	return TRUE;
}

static void
cpufreq_free (struct cpufreq_monitor *c)
{
	int i;
	
	if (c->timeoutid != 0)
	{
		g_source_remove(c->timeoutid);
		c->timeoutid = 0;
	}

	for (i = 0; i < c->nr_cpus; i++)
		g_free(c->cpustrings[i]);
	g_free(c->cpustrings);

	g_free(c);
}

static void
setup_cpufreq (XfcePanelPlugin *plugin, struct cpufreq_monitor *c)
{
	c->pbar = gtk_progress_bar_new();
	c->label = gtk_label_new("");

	gtk_label_set_use_markup (GTK_LABEL(c->label), TRUE);

	if (!c->ebox)
	{
		c->ebox = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(plugin), c->ebox);
		gtk_widget_show(c->ebox);
	}

	if (c->vbox)
		gtk_widget_destroy(c->vbox);
	
	if (c->orientation == GTK_ORIENTATION_HORIZONTAL) {
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(c->pbar),
			GTK_PROGRESS_BOTTOM_TO_TOP);
		c->vbox = gtk_hbox_new(FALSE, 0);
		gtk_widget_set_size_request(GTK_WIDGET(c->pbar), 8, -1);
	} else if (c->orientation == GTK_ORIENTATION_VERTICAL) {
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(c->pbar),
			GTK_PROGRESS_LEFT_TO_RIGHT);
		c->vbox = gtk_vbox_new(FALSE, 0);
		gtk_widget_set_size_request(GTK_WIDGET(c->pbar), -1, 8);
	}

	gtk_box_pack_start(GTK_BOX(c->vbox), c->pbar, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(c->vbox), c->label, FALSE, FALSE, 0);
	
	gtk_container_add(GTK_CONTAINER(c->ebox), c->vbox);
	gtk_container_set_border_width(GTK_CONTAINER(c->vbox), 4);

	gtk_widget_show(c->vbox);
	gtk_widget_show(c->label);
	gtk_widget_show(c->pbar);

	update_cpufreq_status(c);
	
	c->timeoutid = g_timeout_add(1000, (GSourceFunc) update_cpufreq_status, c);
	
	return;
}

static void
cpufreq_set_orientation (XfcePanelPlugin *plugin, GtkOrientation orientation,
	struct cpufreq_monitor *c)
{
	if (c->orientation == orientation)
		return;
	
	c->orientation = orientation;

	if (c->timeoutid > 0)
	{
		g_source_remove(c->timeoutid);
		c->timeoutid = 0;
	}

	gtk_container_remove(GTK_CONTAINER(plugin), c->vbox);

	setup_cpufreq(plugin, c);
}

static void
cpufreq_read_config (XfcePanelPlugin *plugin, struct cpufreq_monitor *c)
{
	char *file;
	const gchar *str;
	XfceRc *rc;

	if ((file = xfce_panel_plugin_lookup_rc_file(plugin)) == NULL)
	{
		c->cpu = 0;
		c->invert_colors = FALSE;
		return;
	}
	
	rc = xfce_rc_simple_open(file, FALSE);

	g_free(file);

	if (!rc || (str = xfce_rc_read_entry(rc, "NumCPUs", NULL)) == NULL)
	{
		c->cpu = 0;
		c->invert_colors = FALSE;
		return;
	}

	c->cpu = atoi(str);
	c->invert_colors = xfce_rc_read_bool_entry(rc, "InvColors", FALSE);
	xfce_rc_close (rc);
}

static void
cpufreq_write_config (XfcePanelPlugin *plugin, struct cpufreq_monitor *c)
{
	char *file, buf[32];
	XfceRc *rc;

	if (!(file = xfce_panel_plugin_save_location(plugin, TRUE)))
		return;

	rc = xfce_rc_simple_open(file, FALSE);

	g_free(file);

	if (!rc)
		return;

	snprintf(buf, 32, "%d", c->cpu);
	xfce_rc_write_entry(rc, "NumCPUs", buf);
	xfce_rc_write_bool_entry(rc, "InvColors", c->invert_colors);
	xfce_rc_close (rc);
}

static void
cpufreq_set_size (XfcePanelPlugin *plugin, int size, struct cpufreq_monitor *c)
{
	if (c->orientation == GTK_ORIENTATION_VERTICAL)
		gtk_widget_set_size_request(GTK_WIDGET(plugin), size, -1);
	else
		gtk_widget_set_size_request(GTK_WIDGET(plugin), -1, size);
}

static void
invert_colors_toggled (GtkToggleButton *button, gpointer data)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)data;
	c->invert_colors = gtk_toggle_button_get_active (button);

	update_cpufreq_status (c);
}

static void
combo_changed_cb (GtkComboBox *cb, gpointer data)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)data;
	int val = (int)gtk_combo_box_get_active(cb);
	
	if (c->cpu == val) return;

	c->cpu = val;

	cpufreq_get_hardware_limits(c->cpu, &c->min, &c->max);
}

static void
cpufreq_dialog_response (GtkWidget *dlg, int response,
	struct cpufreq_monitor *c)
{
	gtk_widget_destroy(dlg);
	cpufreq_write_config(c->plugin, c);
}


static void
cpufreq_create_options(XfcePanelPlugin *plugin, struct cpufreq_monitor *c)
{
	GtkWidget *dlg = NULL, *vbox = NULL;
	GtkWidget *header = xfce_create_header(NULL, _("CPUFreq Plugin Options"));
	GtkWidget *label = gtk_label_new(_("Choose the CPU to monitor:"));
	GtkWidget *cpucombo = gtk_combo_box_new_text();
	int i;
	
	/* populate combo */
	for (i = 0; i < c->nr_cpus; i++)
		gtk_combo_box_append_text(GTK_COMBO_BOX(cpucombo), c->cpustrings[i]);

	gtk_combo_box_set_active(GTK_COMBO_BOX(cpucombo), c->cpu);

	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
	
	g_signal_connect(cpucombo, "changed", G_CALLBACK(combo_changed_cb), c);

	dlg = gtk_dialog_new_with_buttons (_("Properties"),
		GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
		GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CLOSE,
		GTK_RESPONSE_OK,
		NULL);

	g_signal_connect(dlg, "response", G_CALLBACK(cpufreq_dialog_response), c);
	
	vbox = GTK_DIALOG(dlg)->vbox;

	GtkWidget *options = gtk_frame_new (NULL);
	gtk_widget_show (options);
	gtk_container_set_border_width (GTK_CONTAINER (options), 2);

	GtkWidget *options_label = gtk_label_new (_("Options"));
	gtk_widget_show (options_label);
	gtk_frame_set_label_widget (GTK_FRAME (options), options_label);

	GtkWidget *invert = gtk_check_button_new_with_mnemonic
							(_("Invert colors"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(invert), c->invert_colors);
	gtk_widget_show(invert);
	gtk_container_add (GTK_CONTAINER (options), invert);

	g_signal_connect(invert, "toggled", 
				G_CALLBACK (invert_colors_toggled), c);
	
	gtk_box_pack_start(GTK_BOX(vbox), header, FALSE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), cpucombo, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), options, FALSE, FALSE, 10);
	
	g_object_set_data(G_OBJECT(plugin), "dialog", dlg);
	gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_CENTER);

	gtk_widget_show(header);
	gtk_widget_show(label);
	gtk_widget_show(cpucombo);
	gtk_widget_show(dlg);
}

static struct cpufreq_monitor* cpufreq_new (XfcePanelPlugin *plugin)
{
	struct cpufreq_monitor* c = calloc(1, sizeof(struct cpufreq_monitor));
	int i;

	c->plugin = plugin;
	c->freqs = NULL;
	c->orientation = xfce_panel_plugin_get_orientation(plugin);
	c->nr_cpus = get_nr_cpus();
	c->showed_warning = FALSE;
	c->invert_colors = FALSE;
	
	/* determine min/max */
	cpufreq_get_hardware_limits(c->cpu, &c->min, &c->max);

	/* possibly feed config in here */
	cpufreq_read_config(plugin, c);

	c->cpustrings = malloc(c->nr_cpus * sizeof(char*));
	
	/* populate list of CPU ids */
	for (i = 0; i < c->nr_cpus; i++)
		c->cpustrings[i] = get_cpu_name(i); 
	
	gdk_color_parse(COLOR_LOW_DEFAULT, &(c->colorLow));
	gdk_color_parse(COLOR_HI_DEFAULT, &(c->colorHi));

	return c;
}

void cpufreq_construct(XfcePanelPlugin *plugin)
{
#ifdef ENABLE_NLS
    /* This is required for UTF-8 at least - Please don't remove it */
    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);
#endif

	struct cpufreq_monitor *mon = cpufreq_new(plugin);
	//cc->create_control = (CreateControlFunc)cpufreq_control_new;

	g_signal_connect(plugin, "free-data", G_CALLBACK(cpufreq_free), mon);
	g_signal_connect(plugin, "size-changed", G_CALLBACK(cpufreq_set_size), mon);
	g_signal_connect(plugin, "orientation-changed", G_CALLBACK(cpufreq_set_orientation), mon);
	xfce_panel_plugin_menu_show_configure(plugin);
	g_signal_connect(plugin, "configure-plugin", G_CALLBACK(cpufreq_create_options), mon);
	g_signal_connect(plugin, "save", G_CALLBACK(cpufreq_write_config), mon);
	
	setup_cpufreq(plugin, mon);
}

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL(cpufreq_construct);
