/* copyop.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: copyop.cc,v 1.31 2002/03/22 23:12:50 ralf Exp $ */

#include "copyop.h"
#include "listermode.h"
#include "normalmode.h"
#include "worker.h"

const char *CopyOp::name="CopyOp";

CopyOp::CopyOp(Worker *tworker):FunctionProto(tworker)
{
  follow_symlinks=false;
  move=false;
  do_rename=false;
  same_dir=false;
  request_dest=false;
  request_flags=false;
  overwrite=COPYOP_OVERWRITE_NORMAL;
  fastcopy=COPYOP_NORMALCOPY;
  cp_com[0]=dupstring("");
  cp_com[1]=dupstring("");
  preserve_attr=true;
}

CopyOp::~CopyOp()
{
  _freesafe(cp_com[0]);
  _freesafe(cp_com[1]);
}

CopyOp*
CopyOp::duplicate()
{
  CopyOp *ta=new CopyOp(worker);
  ta->follow_symlinks=follow_symlinks;
  ta->move=move;
  ta->do_rename=do_rename;
  ta->same_dir=same_dir;
  ta->request_dest=request_dest;
  ta->request_flags=request_flags;
  ta->overwrite=overwrite;
  ta->fastcopy=fastcopy;
  _freesafe(ta->cp_com[0]);
  _freesafe(ta->cp_com[1]);
  ta->cp_com[0]=dupstring(cp_com[0]);
  ta->cp_com[1]=dupstring(cp_com[1]);
  ta->preserve_attr=preserve_attr;
  return ta;
}

bool
CopyOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *
CopyOp::getName()
{
  return name;
}

int
CopyOp::run(ActionMessage *msg)
{
  ListerMode *lm1;
  am=msg;
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    Lister *l1=worker->getActiveLister();
    if(l1!=NULL) {
      startlister=l1;
      endlister=worker->getOtherLister(startlister);
      lm1=startlister->getActiveMode();
      if(lm1!=NULL)
        if(lm1->isType("NormalMode")==true)
          normalmodecopy();
    }
  } else {
    normalmodecopy();
  }
  return 0;
}

bool
CopyOp::save(Datei *fh)
{
  int size=9*Datei::getUCharSize()+2*Datei::getIntSize();
  size+=strlen(cp_com[0]);
  size+=strlen(cp_com[1]);
  fh->putInt(size);

  fh->putUChar((follow_symlinks==true)?1:0);
  fh->putUChar((move==true)?1:0);
  fh->putUChar((do_rename==true)?1:0);
  fh->putUChar((same_dir==true)?1:0);
  fh->putUChar((request_dest==true)?1:0);
  fh->putUChar((request_flags==true)?1:0);
  switch(overwrite) {
    case COPYOP_OVERWRITE_ALWAYS:
      fh->putUChar(1);
      break;
    case COPYOP_OVERWRITE_NEVER:
      fh->putUChar(2);
      break;
    default:
      fh->putUChar(0);
  }
  switch(fastcopy) {
    case COPYOP_FASTCOPY:
      fh->putUChar(1);
      break;
    case COPYOP_FASTESTCOPY:
      fh->putUChar(2);
      break;
    default:
      fh->putUChar(0);
  }
  fh->putInt(strlen(cp_com[0]));
  fh->putString(cp_com[0]);
  fh->putInt(strlen(cp_com[1]));
  fh->putString(cp_com[1]);
  fh->putUChar((preserve_attr==true)?1:0);
  return true;
}

int
CopyOp::load(Datei *fh)
{
  int chunksize=fh->getInt();
  int cs=Datei::getUCharSize(),
      is=Datei::getIntSize(),
      len;
  unsigned char c1;
  if(chunksize>=(8*cs+2*is)) {
    if(fh->getUChar()==1) follow_symlinks=true; else follow_symlinks=false;
    if(fh->getUChar()==1) move=true; else move=false;
    if(fh->getUChar()==1) do_rename=true; else do_rename=false;
    if(fh->getUChar()==1) same_dir=true; else same_dir=false;
    if(fh->getUChar()==1) request_dest=true; else request_dest=false;
    if(fh->getUChar()==1) request_flags=true; else request_flags=false;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        overwrite=COPYOP_OVERWRITE_ALWAYS;
        break;
      case 2:
        overwrite=COPYOP_OVERWRITE_NEVER;
        break;
      default:
        overwrite=COPYOP_OVERWRITE_NORMAL;
    }
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        fastcopy=COPYOP_FASTCOPY;
        break;
      case 2:
        fastcopy=COPYOP_FASTESTCOPY;
        break;
      default:
        fastcopy=COPYOP_NORMALCOPY;
    }
    chunksize-=8*cs;

    len=fh->getInt();
    chunksize-=is;
    _freesafe(cp_com[0]);
    cp_com[0]=fh->getString(len);
    if(cp_com[0]==NULL) cp_com[0]=dupstring("");
    chunksize-=len;

    len=fh->getInt();
    chunksize-=is;
    _freesafe(cp_com[1]);
    cp_com[1]=fh->getString(len);
    if(cp_com[1]==NULL) cp_com[1]=dupstring("");
    chunksize-=len;
    
    if(chunksize>=cs) {
      if(fh->getUChar()==1) preserve_attr=true; else preserve_attr=false;
      chunksize-=cs;
    }
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return true;
}

const char *
CopyOp::getDescription()
{
  return catalog.getLocaleCom(8);
}

int
CopyOp::normalmodecopy()
{
  struct NM_copyorder copyorder;
  NormalMode *nm1=NULL,
             *nm2=NULL;
  ListerMode *lm1=NULL;
  struct NM_specialsource *specialsource=NULL;
  bool do_request,cont=true;
  char *tstr,*tstr2,*dest=NULL;
  
  if(am->mode==am->AM_MODE_DNDACTION) {
    lm1=((DNDMsg*)am->extraInfo)->getSourceMode();
    if(lm1==NULL) return 1;
    if(lm1->isType("NormalMode")!=true) return 1;
    nm1=(NormalMode*)lm1;
    lm1=((DNDMsg*)am->extraInfo)->getDestMode();
    if(lm1!=NULL)
      if(lm1->isType("NormalMode")==true)
        nm2=(NormalMode*)lm1;
  } else {
    if(startlister==NULL) return 1;
    lm1=startlister->getActiveMode();
    if(lm1==NULL) return 1;
    if(lm1->isType("NormalMode")!=true) return 1;
    nm1=(NormalMode*)lm1;
    if(endlister!=NULL) {
      lm1=endlister->getActiveMode();
      if(lm1!=NULL)
        if(lm1->isType("NormalMode")==true)
          nm2=(NormalMode*)lm1;
    }
  }
  
  if(request_flags==true) {
    if(doconfigure(1)!=0) cont=false;
  } else {
    // set values in t* variables
    tfollow_symlinks=follow_symlinks;
    tmove=move;
    trename=do_rename;
    tsame_dir=same_dir;
    trequest_dest=request_dest;
    toverwrite=overwrite;
    tfastcopy=fastcopy;
    tcp_com[0]=dupstring(cp_com[0]);
    tcp_com[1]=dupstring(cp_com[1]);
    tpreserve_attr=preserve_attr;
  }
  
  if(cont==true) {
    // detect what to copy
    if(am->mode==am->AM_MODE_ONLYACTIVE)
      copyorder.source=copyorder.NM_ONLYACTIVE;
    else if(am->mode==am->AM_MODE_DNDACTION) {
      // insert DND-element into list
      copyorder.source=copyorder.NM_SPECIAL;
      copyorder.sources=new List;
      specialsource=(struct NM_specialsource*)_allocsafe(sizeof(struct NM_specialsource));
      specialsource->entry=((DNDMsg*)am->extraInfo)->getFE();
      specialsource->row=-1;
      copyorder.sources->addElement(specialsource);
    } else if(am->mode==am->AM_MODE_SPECIAL) {
      copyorder.source=copyorder.NM_SPECIAL;
      copyorder.sources=new List;
      specialsource=(struct NM_specialsource*)_allocsafe(sizeof(struct NM_specialsource));
      specialsource->entry=(FileEntry*)am->extraInfo;
      specialsource->row=-1;
      copyorder.sources->addElement(specialsource);
    } else
      copyorder.source=copyorder.NM_ALLENTRIES;

    /* now source defined
       next find destination
       
       Priority:
        1.Flag for requesting destination
        2.Flag for same dir
        3.When DNDAction destination of DND
        4.nonactive lister
       
       when no destination then also requesting
    */

    do_request=false;
    if(trequest_dest==true)
      do_request=true;
    else if(tsame_dir==true) {
      tstr=nm1->getActualDir();
      if(tstr!=NULL)
        dest=dupstring(tstr);
    } else if(am->mode==am->AM_MODE_DNDACTION) {
      const char *ddstr=((DNDMsg*)am->extraInfo)->getDestDir();
      if(ddstr!=NULL) dest=dupstring(ddstr);
      else dest=NULL;
    } else {
      if(nm2!=NULL) {
        tstr=nm2->getActualDir();
        if(tstr!=NULL)
          dest=dupstring(tstr);
        else
          do_request=true;
      } else
        do_request=true;
    }

    if(do_request==true) {
      if(dest!=NULL) _freesafe(dest);
      if(requestdest(nm1->getActualDir(),&dest)!=0) {
        _freesafe(dest);
        dest=NULL;
      }
    }
    /* if dest==null nothing to do
       otherwise: */
    if(dest!=NULL) {
      /* if dest starts with no slash then take it as local dir
         so add actualDir */
      if(dest[0]!='/') {
        tstr=catstring(nm1->getActualDir(),"/");
        tstr2=catstring(tstr,dest);
        _freesafe(tstr);
        _freesafe(dest);
        dest=tstr2;
      }
      /* now also destination determined
         next describing what to do */
      copyorder.move=tmove;
      copyorder.follow_symlinks=tfollow_symlinks;
      copyorder.preserve_attr=tpreserve_attr;
      copyorder.do_rename=trename;
      switch(toverwrite) {
        case COPYOP_OVERWRITE_ALWAYS:
          copyorder.overwrite=copyorder.NM_OVERWRITE_ALWAYS;
          break;
        case COPYOP_OVERWRITE_NEVER:
          copyorder.overwrite=copyorder.NM_OVERWRITE_NEVER;
          break;
        default:
          copyorder.overwrite=copyorder.NM_OVERWRITE_NORMAL;
      }
      switch(tfastcopy) {
        case COPYOP_FASTCOPY:
          copyorder.copymode=copyorder.NM_COPYMODE_NODATABASE;
          break;
        default:
          copyorder.copymode=copyorder.NM_COPYMODE_NORMAL;
          break;
      }
      copyorder.destdir=dest;
      // now start copy process
      copyorder.cowin=new CopyOpWin(worker->getAGUIX());
      copyorder.cp_com[0]=NULL;
      copyorder.cp_com[1]=NULL;
      if(strlen(tcp_com[0])>0) copyorder.cp_com[0]=tcp_com[0];
      if(strlen(tcp_com[1])>0) copyorder.cp_com[1]=tcp_com[1];
      nm1->copy(&copyorder);
      delete copyorder.cowin;
    }
    if(dest!=NULL) _freesafe(dest);
    if(copyorder.source==copyorder.NM_SPECIAL) {
      if(specialsource!=NULL) _freesafe(specialsource);
      delete copyorder.sources;
    }
    _freesafe(tcp_com[0]);
    _freesafe(tcp_com[1]);
  }
  return 0;
}

int
CopyOp::doconfigure(int mode)
{
  AGUIX *aguix=worker->getAGUIX();
  Button *okb,*cb;
  StringGadget *ccsg[2];
  AWindow *win;
  ChooseButton *fscb,*mcb,*rcb,*sdcb,*rfcb=NULL,*rdcb,*pacb;
//  CycleButton *ocyb,*fccyb;
  CycleButton *fccyb;
  Text *ttext;
  int tw,w,h,x,y,t1,t2;
  AGMessage *msg;
  int endmode=-1;
  Requester *req;
  char *textstr,*buttonstr,*tstr;
  int erg,i;
  
  req=new Requester(aguix);
  w=h=10;
  x=y=5;
  win=new AWindow(aguix);
  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(8))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(8));
  win->create(NULL,10,10,w,h,0,tstr);
  _freesafe(tstr);

  fscb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(follow_symlinks==true)?1:0,
                                                catalog.getLocale(298),LABEL_RIGHT,1,0));
  y+=fscb->getHeight()+5;
  tw=fscb->getWidth()+10;
  if(tw>w) w=tw;

  mcb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(move==true)?1:0,
                                               catalog.getLocale(299),LABEL_RIGHT,1,0));
  y+=mcb->getHeight()+5;
  tw=mcb->getWidth()+10;
  if(tw>w) w=tw;

  rcb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(do_rename==true)?1:0,
                                               catalog.getLocale(300),LABEL_RIGHT,1,0));
  y+=rcb->getHeight()+5;
  tw=rcb->getWidth()+10;
  if(tw>w) w=tw;

  sdcb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(same_dir==true)?1:0,
                                                catalog.getLocale(301),LABEL_RIGHT,1,0));
  y+=sdcb->getHeight()+5;
  tw=sdcb->getWidth()+10;
  if(tw>w) w=tw;

  rdcb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(request_dest==true)?1:0,
                                                catalog.getLocale(302),LABEL_RIGHT,1,0));
  y+=rdcb->getHeight()+5;
  tw=rdcb->getWidth()+10;
  if(tw>w) w=tw;

  pacb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(preserve_attr==true)?1:0,
                                                catalog.getLocale(150),LABEL_RIGHT,1,0));
  y+=pacb->getHeight()+5;
  tw=pacb->getWidth()+10;
  if(tw>w) w=tw;

/*  ttext=(Text*)win->add(new Text(aguix,x,y,catalog.getLocale(303),1));
  x+=ttext->getWidth()+5;
  ocyb=(CycleButton*)win->add(new CycleButton(aguix,x,y,100,1,0,0));
  ocyb->addOption(catalog.getLocale(304));
  ocyb->addOption(catalog.getLocale(305));
  ocyb->addOption(catalog.getLocale(306));
  ocyb->resize(ocyb->getMaxSize(),ocyb->getHeight());
  switch(overwrite) {
    case COPYOP_OVERWRITE_ALWAYS:
      ocyb->setOption(1);
      break;
    case COPYOP_OVERWRITE_NEVER:
      ocyb->setOption(2);
      break;
    default:
      ocyb->setOption(0);
      break;
  }
  
  y+=ocyb->getHeight()+5;
  tw=x+ocyb->getWidth()+5;
  if(tw>w) w=tw;
  x=5;*/

  ttext=(Text*)win->add(new Text(aguix,x,y,catalog.getLocale(307),1));
  x+=ttext->getWidth()+5;
  fccyb=(CycleButton*)win->add(new CycleButton(aguix,x,y,100,1,0,0));
  fccyb->addOption(catalog.getLocale(308));
  fccyb->addOption(catalog.getLocale(309));
//  fccyb->addOption("schnellster Modus");
  fccyb->resize(fccyb->getMaxSize(),fccyb->getHeight());
  switch(fastcopy) {
    case COPYOP_FASTCOPY:
      fccyb->setOption(1);
      break;
/*    case COPYOP_FASTESTCOPY:
      fccyb->setOption(2);
      break;*/
    default:
      fccyb->setOption(0);
      break;
  }
  
  y+=fccyb->getHeight()+5;
  tw=x+fccyb->getWidth()+5;
  if(tw>w) w=tw;
  x=5;

  ttext=(Text*)win->add(new Text(aguix,x,y,catalog.getLocale(310),1));
  x+=ttext->getWidth()+5;
  tw=w-x-5;
  if(tw<50) tw=50;
  ccsg[0]=(StringGadget*)win->add(new StringGadget(aguix,x,y,tw,cp_com[0],0));
  y+=ccsg[0]->getHeight()+5;
  tw=ccsg[0]->getWidth()+5+x;
  if(tw>w) w=tw;
  x=5;
  ttext=(Text*)win->add(new Text(aguix,x,y,catalog.getLocale(311),1));
  x+=ttext->getWidth()+5;
  tw=w-x-5;
  if(tw<50) tw=50;
  ccsg[1]=(StringGadget*)win->add(new StringGadget(aguix,x,y,tw,cp_com[1],0));
  y+=ccsg[1]->getHeight()+5;
  tw=ccsg[1]->getWidth()+5+x;
  if(tw>w) w=tw;
  x=5;

  if(mode==0) {
    rfcb=(ChooseButton*)win->add(new ChooseButton(aguix,x,y,20,20,(request_flags==true)?1:0,
                                                  catalog.getLocale(294),LABEL_RIGHT,1,0));
    y+=rfcb->getHeight()+5;
    tw=rfcb->getWidth()+10;
    if(tw>w) w=tw;
  }

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>w) w=tw;
  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   y,
                                   t1,
                                   catalog.getLocale(11),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  w-5-t2,
                                  y,
                                  t2,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));
  y+=okb->getHeight()+5;
  
  ccsg[0]->resize(w-5-ccsg[0]->getX(),ccsg[0]->getHeight());
  ccsg[1]->resize(w-5-ccsg[1]->getX(),ccsg[1]->getHeight());

  h=y;
  win->resize(w,h);
  win->setMaxSize(w,h);
  win->setMinSize(w,h);
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) {
            i=0;
            for(int j=0;j<2;j++) {
              if(NormalMode::isCorrectCPProg(ccsg[j]->getText())==false) {
                // show requester
                textstr=catalog.getLocale(312);
                buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(313))+1+
                                            strlen(catalog.getLocale(314))+1);
                sprintf(buttonstr,"%s|%s",catalog.getLocale(313),
                                          catalog.getLocale(314));
                erg=req->request(catalog.getLocale(125),textstr,buttonstr);
                _freesafe(buttonstr);
                if(erg==0) {
                  ccsg[(j==0)?1:0]->deactivate();
                  ccsg[j]->activate();
                } else {
                  ccsg[j]->setText("");
                }
                break;
              } else i++;
            }
            if(i==2) endmode=0;
          } else if(msg->button.button==cb) endmode=1;
          break;
	case AG_STRINGGADGET_DEACTIVATE:
          if((msg->stringgadget.sg==ccsg[0])||
             (msg->stringgadget.sg==ccsg[1])) {
            if(NormalMode::isCorrectCPProg(msg->stringgadget.sg->getText())==false) {
              // show requester
              textstr=catalog.getLocale(312);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(313))+1+
                                          strlen(catalog.getLocale(314))+1);
              sprintf(buttonstr,"%s|%s",catalog.getLocale(313),
                                        catalog.getLocale(314));
              erg=req->request(catalog.getLocale(125),textstr,buttonstr);
              _freesafe(buttonstr);
              if(erg==0) {
                if(msg->stringgadget.sg==ccsg[0]) ccsg[1]->deactivate();
                else ccsg[0]->deactivate();
                msg->stringgadget.sg->activate();
              } else {
                msg->stringgadget.sg->setText("");
              }
            }
          }
	  break;
        case AG_KEYRELEASED:
          if(win->isParent(msg->key.window,false)==true) {
            switch(msg->key.key) {
              case XK_1:
                fscb->setState((fscb->getState()==1)?0:1);
                break;
              case XK_2:
                mcb->setState((mcb->getState()==1)?0:1);
                break;
              case XK_3:
                rcb->setState((rcb->getState()==1)?0:1);
                break;
              case XK_4:
                sdcb->setState((sdcb->getState()==1)?0:1);
                break;
              case XK_5:
                rdcb->setState((rdcb->getState()==1)?0:1);
                break;
              case XK_6:
/*                i=ocyb->getSelectedOption()+1;
                if(i>=3) i=0;
                ocyb->setOption(i);
                break;
              case XK_7:*/
                i=fccyb->getSelectedOption()+1;
                if(i>=2) i=0;
                fccyb->setOption(i);
                break;
              case XK_Return:
                i=0;
                for(int j=0;j<2;j++) {
                  if(NormalMode::isCorrectCPProg(ccsg[j]->getText())==false) {
                    // show requester
                    textstr=catalog.getLocale(312);
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(313))+1+
                                                strlen(catalog.getLocale(314))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(313),
                                              catalog.getLocale(314));
                    erg=req->request(catalog.getLocale(125),textstr,buttonstr);
                    _freesafe(buttonstr);
                    if(erg==0) {
                      ccsg[(j==0)?1:0]->deactivate();
                      ccsg[j]->activate();
                    } else {
                      ccsg[j]->setText("");
                    }
                    break;
                  } else i++;
                }
                if(i==2) endmode=0;
                break;
              case XK_Escape:
                endmode=1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    if(mode==1) {
      // store in t-variables
      tfollow_symlinks=(fscb->getState()==1)?true:false;
      tmove=(mcb->getState()==1)?true:false;
      trename=(rcb->getState()==1)?true:false;
      tsame_dir=(sdcb->getState()==1)?true:false;
      trequest_dest=(rdcb->getState()==1)?true:false;
      tpreserve_attr=(pacb->getState()==1)?true:false;
      tcp_com[0]=dupstring(ccsg[0]->getText());
      tcp_com[1]=dupstring(ccsg[1]->getText());
      toverwrite=COPYOP_OVERWRITE_NORMAL;
/*      switch(ocyb->getSelectedOption()) {
        case 1:
          toverwrite=COPYOP_OVERWRITE_ALWAYS;
          break;
        case 2:
          toverwrite=COPYOP_OVERWRITE_NEVER;
          break;
        default:
          toverwrite=COPYOP_OVERWRITE_NORMAL;
          break;
      }*/
      switch(fccyb->getSelectedOption()) {
        case 1:
          tfastcopy=COPYOP_FASTCOPY;
          break;
/*        case 2:
          tfastcopy=COPYOP_FASTESTCOPY;
          break;*/
        default:
          tfastcopy=COPYOP_NORMALCOPY;
          break;
      }
    } else {
      // store in normal variables
      follow_symlinks=(fscb->getState()==1)?true:false;
      move=(mcb->getState()==1)?true:false;
      do_rename=(rcb->getState()==1)?true:false;
      same_dir=(sdcb->getState()==1)?true:false;
      request_dest=(rdcb->getState()==1)?true:false;
      request_flags=(rfcb->getState()==1)?true:false;
      preserve_attr=(pacb->getState()==1)?true:false;
      _freesafe(cp_com[0]);
      _freesafe(cp_com[1]);
      cp_com[0]=dupstring(ccsg[0]->getText());
      cp_com[1]=dupstring(ccsg[1]->getText());
      overwrite=COPYOP_OVERWRITE_NORMAL;
/*      switch(ocyb->getSelectedOption()) {
        case 1:
          overwrite=COPYOP_OVERWRITE_ALWAYS;
          break;
        case 2:
          overwrite=COPYOP_OVERWRITE_NEVER;
          break;
        default:
          overwrite=COPYOP_OVERWRITE_NORMAL;
          break;
      }*/
      switch(fccyb->getSelectedOption()) {
        case 1:
          fastcopy=COPYOP_FASTCOPY;
          break;
/*        case 2:
          fastcopy=COPYOP_FASTESTCOPY;
          break;*/
        default:
          fastcopy=COPYOP_NORMALCOPY;
          break;
      }
    }
  }
  
  win->close();
  delete win;
  delete req;

  return endmode;
}

int
CopyOp::configure()
{
  return doconfigure(0);
}

int
CopyOp::requestdest(char *defaultstr,char **dest)
{
  Requester *req=new Requester(worker->getAGUIX());
  char *textstr,*buttonstr;
  int erg;
  
  textstr=catalog.getLocale(212);
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocale(123),textstr,(defaultstr!=NULL)?defaultstr:"",buttonstr,dest);
  _freesafe(buttonstr);
  delete req;
  return erg;
}

void CopyOp::setFollowSymlinks(bool nv)
{
  follow_symlinks=nv;
}

void CopyOp::setMove(bool nv)
{
  move=nv;
}

void CopyOp::setRename(bool nv)
{
  do_rename=nv;
}

void CopyOp::setSameDir(bool nv)
{
  same_dir=nv;
}

void CopyOp::setRequestDest(bool nv)
{
  request_dest=nv;
}

void CopyOp::setRequestFlags(bool nv)
{
  request_flags=nv;
}

void CopyOp::setOverwrite(overwrite_t nv)
{
  overwrite=nv;
}

void CopyOp::setPreserveAttr(bool nv)
{
  preserve_attr=nv;
}

/*******************
 * Class CopyOpWin *
 *******************/

CopyOpWin::CopyOpWin(AGUIX*taguix)
{
  this->aguix=taguix;
  source=dupstring("");
  dest=dupstring("");
  copied_bytes_actfile=0;
  bytes_actfile=0;
  copied_bytes=0;
  bytes=0;
  copied_files=0;
  files=0;
  copied_dirs=0;
  dirs=0;
  slim=false;
  
  window=NULL;
  melw = 0;
  filenamespace[0] = 0;
  filenamespace[1] = 0;
}

CopyOpWin::~CopyOpWin()
{
  if(source!=NULL) _freesafe(source);
  if(dest!=NULL) _freesafe(dest);
  if(window!=NULL) close();
}

int
CopyOpWin::open(bool tslim)
{
  int w,h;
  int tx,ty;
  Text *ttext;

  close();
  this->slim=tslim;
  window=new AWindow(aguix);
  w=400;
  h=10;
  window->create(NULL,10,10,w,h,0,catalog.getLocaleCom(8));
  tx=5;
  ty=5;
  sourcetext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  ty+=sourcetext->getHeight()+5;
  desttext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  ty+=desttext->getHeight()+5;
  fsb=(SolidButton*)window->add(new SolidButton(aguix,tx+1,ty+1,w-2*(tx+1),"",2,3,0));
  sbw=fsb->getWidth();
  bb1=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,fsb->getHeight()+2,1));
  fsb->toFront();
  ty+=bb1->getHeight()+5;
  tx=10;
  filetext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=filetext->getHeight()+5;
  bb2=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,2,0));
  ty+=bb2->getHeight()+5;

  ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale((slim==true)?284:118),1));
  tx+=ttext->getWidth()+5;
  files2gotext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=ttext->getHeight()+5;
  ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale((slim==true)?285:119),1));
  tx+=ttext->getWidth()+5;
  dirs2gotext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=ttext->getHeight()+5;
  if(slim==false) {
    gsb=(SolidButton*)window->add(new SolidButton(aguix,tx+1,ty+1,w-2*(tx+1),"",2,3,0));
    bb3=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,gsb->getHeight()+2,1));
    gsb->toFront();
    ty+=bb3->getHeight()+5;
  }
  tx=10;
  ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale(120),1));
  tx+=ttext->getWidth()+5;
  globtext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=globtext->getHeight()+5;
  if(slim==false) {
    ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale(121),1));
    tx+=ttext->getWidth()+5;
    timetext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
    ty+=timetext->getHeight()+5;
  }
  tx=5;
  bb4=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,2,0));
  ty+=bb4->getHeight()+5;
  cb=(Button*)window->add(new Button(aguix,tx,ty,
                                     (strlen(catalog.getLocale(8))+2)*aguix->getCharWidth(),
                                     catalog.getLocale(8),1,0,0));
  cb->move(w/2-cb->getWidth()/2,cb->getY());
  ty+=cb->getHeight()+5;
  h=ty;
  window->resize(w,h);
  window->setMinSize(w,h);
  //  window->setMaxSize(w,h);
  window->centerScreen();
  window->show();

  filenamespace[0] = ( w - window->getBorderWidth() - sourcetext->getX() ) / aguix->getCharWidth();
  filenamespace[0] -= strlen(catalog.getLocale(116)) - 2;
  filenamespace[1] = ( w - window->getBorderWidth() - desttext->getX() ) / aguix->getCharWidth();
  filenamespace[1] -= strlen(catalog.getLocale(117)) - 2;

  return 0;
}

void
CopyOpWin::close()
{
  if(window!=NULL) {
    window->close();
    delete window;
  }
  window=NULL;
}

void
CopyOpWin::starttimer()
{
  starttime=time(NULL);
}

void
CopyOpWin::set_files_to_copy(long nfiles)
{
  if(nfiles>0) files=nfiles;
  else files=0;
  update_files2gotext=true;
}

void
CopyOpWin::set_dirs_to_copy(long ndirs)
{
  if(ndirs>0) dirs=ndirs;
  else dirs=0;
  update_dirs2gotext=true;
}

void
CopyOpWin::set_bytes_to_copy(loff_t nbytes)
{
  if(nbytes>0) bytes=nbytes;
  else bytes=0;
  update_globtext=true;
}

void
CopyOpWin::set_bytes_to_copy_actfile(loff_t nbytes)
{
  if(nbytes>0) bytes_actfile=nbytes;
  else bytes_actfile=nbytes;
  update_filetext=true;
}

void
CopyOpWin::dir_finished()
{
  copied_dirs++;
  update_dirs2gotext=true;
}

void
CopyOpWin::file_finished()
{
  // inc counter
  copied_files++;
  update_files2gotext=true;
}

void
CopyOpWin::add_actbytes_copied(loff_t nbytes)
{
  if(nbytes>0) {
    copied_bytes_actfile+=nbytes;
    copied_bytes+=nbytes;
  }
  update_filetext=true;
  update_globtext=true;
}

void
CopyOpWin::newfile(char *name,char *ndest)
{
  if(source!=NULL) _freesafe(source);
  if(name!=NULL)
    source=dupstring(name);
  else
    source=dupstring("");
  if(dest!=NULL) _freesafe(dest);
  if(ndest!=NULL)
    dest=dupstring(ndest);
  else
    dest=dupstring("");
  filestarttime=time(NULL);
  update_sourcetext=true;
  update_desttext=true;
  copied_bytes_actfile=0;
  update_filetext=true;
}

int
CopyOpWin::redraw()
{ /* this functions will redraw the window AND process events for this window
   * returns values:
   *  0 for regular exit
   *  1 when clicked Cancel-Button or closed window
   *  other not implemented */
  int returnvalue=0;
  time_t acttime=0,difftime2,rtime;
  char tstr[1024],buf1[64],buf2[64],*tstr2;
  float rate;
  int tm;
  bool doresize = false;
  
  char *tstr3;

  aguix->Flush();
  if ( update_sourcetext == true ) {
    if ( source != NULL ) {
      tstr3 = Datei::shrinkFilename( source, filenamespace[0] );
      if ( tstr3 == NULL)
        tstr3 = dupstring( source );
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 116 ) ) + strlen( tstr3 ) + 1 );
      sprintf( tstr2, catalog.getLocale( 116 ), tstr3 );
      _freesafe( tstr3 );
    } else {
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 116 ) ) + 1 );
      sprintf( tstr2, catalog.getLocale( 116 ), "" );
    }
    sourcetext->setText(tstr2);
    _freesafe(tstr2);
    /* no need for explicit redraw */
    update_sourcetext=false;
  }
  if ( update_desttext == true ) {
    if ( dest != NULL ) {
      tstr3 = Datei::shrinkFilename( dest, filenamespace[1] );
      if ( tstr3 == NULL)
        tstr3 = dupstring( dest );
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 117 ) ) + strlen( tstr3 ) + 1 );
      sprintf( tstr2, catalog.getLocale( 117 ), tstr3 );
      _freesafe(tstr3);
    } else {
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 117 ) ) + 1 );
      sprintf(tstr2,catalog.getLocale(117),"");
    }
    desttext->setText(tstr2);
    _freesafe(tstr2);
    // no need for explicit redraw
    update_desttext=false;
  }
  if(update_filetext==true) {
    if(acttime==0) acttime=time(NULL);
    difftime2=acttime-filestarttime;
    if(difftime2>0) {
      rate=(float)copied_bytes_actfile/(float)difftime2;
    } else rate=0.0;
//    sprintf(tstr,"%ld / %ld @ %.2f KB/s",copied_bytes_actfile,bytes_actfile,rate/1024);
    buf1[MakeLong2NiceStr(buf1,copied_bytes_actfile)]='\0';
    buf2[MakeLong2NiceStr(buf2,bytes_actfile)]='\0';
    sprintf(tstr,"%s / %s @ %.2f KB/s",buf1,buf2,rate/1024);
    filetext->setText(tstr);
    tm = filetext->getX() + filetext->getWidth() + window->getBorderWidth();
    if ( tm > melw ) {
      melw = tm;
      doresize = true;
    }
    
    if(bytes_actfile>0)
      rate=(float)copied_bytes_actfile/(float)bytes_actfile;
    else
      rate=0.0;
    if ( rate < 0.01 )
      rate = 0.0;
    sprintf(tstr,"%.4g %%",rate*100);
    fsb->setText(tstr);
    fsb->resize( w_max( (int)( sbw * rate ), 2 ), fsb->getHeight() );
    update_filetext=false;
  }
  if(update_globtext==true) {
    if(acttime==0) acttime=time(NULL);
    difftime2=acttime-starttime;
    if(difftime2>0) {
      rate=(float)copied_bytes/(float)difftime2;
    } else rate=0.0;
//    sprintf(tstr,"%ld / %ld @ %.2f KB/s",copied_bytes,bytes,rate/1024);
    buf1[MakeLong2NiceStr(buf1,copied_bytes)]='\0';
    if(slim==true) {
      sprintf(tstr,"%s @ %.2f KB/s",buf1,rate/1024);
    } else {
      buf2[MakeLong2NiceStr(buf2,bytes)]='\0';
      sprintf(tstr,"%s / %s @ %.2f KB/s",buf1,buf2,rate/1024);
    }
    
    globtext->setText(tstr);
    tm = globtext->getX() + globtext->getWidth() + window->getBorderWidth();
    if ( tm > melw ) {
      melw = tm;
      doresize = true;
    }
    
    if(slim==false) {
      if(bytes>0)
        rate=(float)copied_bytes/(float)bytes;
      else
        rate=0.0;
      if ( rate < 0.01 )
        rate = 0.0;
      sprintf(tstr,"%.4g %%",rate*100);
      gsb->setText(tstr);
      gsb->resize( w_max( (int)( sbw * rate ), 2 ), gsb->getHeight() );
      if(rate>0) {
        rtime=(time_t)((float)difftime2/rate);
        rtime-=difftime2;
      } else
        rtime = 700000;  // no rate available => no real time calculatable
      if ( rtime > 604800 ) // more then 7 days
        sprintf( tstr, "--:--" );
      else 
        sprintf( tstr, "%ld:%02ld", rtime / 60, rtime % 60 );
      timetext->setText(tstr);
      tm = timetext->getX() + timetext->getWidth() + window->getBorderWidth();
      if ( tm > melw ) {
        melw = tm;
        doresize = true;
      }
    }
    update_globtext=false;
  }
  if(update_files2gotext==true) {
    if(slim==true) sprintf(tstr,"%ld",copied_files);
    else sprintf(tstr,"%ld",files-copied_files);
    files2gotext->setText(tstr);
    tm = files2gotext->getX() + files2gotext->getWidth() + window->getBorderWidth();
    if ( tm > melw ) {
      melw = tm;
      doresize = true;
    }
    
    update_files2gotext=false;
  }
  if(update_dirs2gotext==true) {
    if(slim==true) sprintf(tstr,"%ld",copied_dirs);
    else sprintf(tstr,"%ld",dirs-copied_dirs);
    dirs2gotext->setText(tstr);
    tm = dirs2gotext->getX() + dirs2gotext->getWidth() + window->getBorderWidth();
    if ( tm > melw ) {
      melw = tm;
      doresize = true;
    }
    
    update_dirs2gotext=false;
  }
  if ( doresize == true ) {
    // only resize if window is smaller
    if ( window->getWidth() < melw ) {
      window->resize( melw, window->getHeight() );
    }
    // anyway the min size has changed so set it again
    window->setMinSize( melw, bb4->getY() + bb4->getHeight() + 5 +cb->getHeight() + window->getBorderWidth() );
  }

  /* update complete
     now check for X-messages */
  AGMessage *msg;
  do {
    msg=aguix->GetMessage(NULL);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==window->getWindow()) returnvalue=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==cb) returnvalue=1;
          break;
        case AG_SIZECHANGED:
          bb1->resize( msg->size.neww - 2 * window->getBorderWidth(), bb1->getHeight() );
          sbw = bb1->getWidth() - 2;
          bb2->resize( msg->size.neww - 2 * window->getBorderWidth(), bb2->getHeight() );
          if ( slim == false ) {
            bb3->resize( msg->size.neww - 2 * window->getBorderWidth(), bb3->getHeight() );
          }
          bb4->resize( msg->size.neww - 2 * window->getBorderWidth(), bb4->getHeight() );
          cb->move( msg->size.neww / 2 - cb->getWidth() / 2, msg->size.newh - window->getBorderWidth() - cb->getHeight() );
          
          filenamespace[0] = ( msg->size.neww - window->getBorderWidth() - sourcetext->getX() ) / aguix->getCharWidth();
          filenamespace[0] -= strlen(catalog.getLocale(116)) - 2;
          filenamespace[1] = ( msg->size.neww - window->getBorderWidth() - desttext->getX() ) / aguix->getCharWidth();
          filenamespace[1] -= strlen(catalog.getLocale(117)) - 2;

          // now gsb and fsb are not scaled for new width
          // but next call of redraw they will so this shouldn't hurt anybody
          // but a solution would be to first react for messages and then redraw
          
          break;
      }
      aguix->ReplyMessage(msg);
    }
  } while(msg!=NULL);
  return returnvalue;
}

void
CopyOpWin::setmessage(char *msg,int line)
{
  switch(line) {
    case 0:
      if(msg!=NULL) sourcetext->setText(msg);
      else sourcetext->setText("");
      break;
    case 1:
      if(msg!=NULL) desttext->setText(msg);
      else desttext->setText("");
      break;
  }
}

void
CopyOpWin::forceredraw()
{
  window->redraw();
}

void
CopyOpWin::dec_file_counter(unsigned long f)
{
  files-=f;
  update_files2gotext=true;
}

void
CopyOpWin::dec_dir_counter(unsigned long d)
{
  dirs-=d;
  update_dirs2gotext=true;
}

void
CopyOpWin::dec_byte_counter(loff_t b)
{
  bytes-=b;
  update_globtext=true;
}

void
CopyOpWin::stoptimer()
{
  stoptime=time(NULL);  
}

void
CopyOpWin::conttimer()
{
  double difft=difftime(time(NULL),stoptime);
  starttime+=(time_t)difft;
  filestarttime+=(time_t)difft;
}

