/*
 *    wmmaiload - A dockapp to monitor mails numbers
 *    Copyright (C) 2002  Thomas Nemeth <tnemeth@free.fr>
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002  Mark Staggs <me@markstaggs.net>

 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.

 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.

 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "defines.h"

#include <pwd.h>
#include <signal.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <utime.h>
#include <time.h>
#include <math.h>
#ifdef HAVE_POP3
# include "pop3client.h"
#endif
#ifdef HAVE_IMAP
# include "imapclient.h"
#endif
#include "dockapp.h"
#include "backlight_on.xpm"
#include "backlight_off.xpm"
#include "parts.xpm"



#define FREE(data) {if (data) free(data); data = NULL;}
#define SET_STRING(str, val) {if (str) free(str); str = xstrdup(val);}

#define DEFAULT_CFGFILE ".wmmailoadrc"
#define SIZE            58
#define MAXSTRLEN       512
#define WINDOWED_BG     ". c #AEAAAE"
#define MAX_HISTORY     16
#define CPUNUM_NONE     -1

#ifdef HAVE_MH
# define MH_PROFILE      ".mh_profile"
# define FCLOSE(data) {if (data) fclose(data); data = NULL;}
# define DCLOSE(data) {if (data) closedir(data); data = NULL;}
# define DIRTRIM(data) dirtrim(data)
#endif

#define MTIME 1
#define ATIME 2
#define CTIME 3


typedef enum { LIGHTON, LIGHTOFF } light;
typedef enum { NOALARM, NEWGLOB, NEWHERE, NEWBOTH } MailAlarm;
typedef enum { MBOX, MAILDIR, MH, POP3, HOTMAIL, IMAP } BoxType;
#ifdef HAVE_MH
typedef enum { CWD=1, HOME } BaseDir;
#endif

typedef struct MailBox {
    char           *entry;
    time_t         time;
    off_t          size;
    int            new;
    int            total;
    int            updated;
    BoxType        type;
    char           *username;
    char           *password;
    char           *folder;
    int            port;
    struct MailBox *next;
} MailBox;


Pixmap pixmap;
Pixmap backdrop_on;
Pixmap backdrop_off;
Pixmap parts;
Pixmap mask;
static char	 *display_name     = "";
static char	 *light_color      = NULL;	/* back-light color */
static unsigned  update_interval   = 1;
static int       boxnum            = 0;
static light     backlight         = LIGHTOFF;
static unsigned  new_mails         = 0;
static unsigned  total_mails       = 0;
static Bool      switch_authorized = True;
static Bool      display_new       = True;
static Bool      run_once          = False;
static Bool      test_size         = False;
static char      *command          = NULL;
static char      *notif_cmd        = NULL;
static char      *config_file      = NULL;
static time_t    config_mtime      = 0;
static off_t     config_fsize      = 0;
static MailBox   *mboxes           = NULL;
static MailAlarm alarm_type        = NOALARM;
static time_t    check_delay        = 30;

/* prototypes */
static void install_pixmaps();
static void update();
static void switch_light();
static void switch_leds();
static void draw_dockapp();
static void draw_boxdigit(int num);
static void draw_newdigit(int num);
static void draw_totaldigit(int num);
static void parse_arguments(int argc, char **argv);
static void print_help(char *prog);
static void mail_getnumbers(int *new, int *total, MailAlarm *bell);
static Bool update_mailboxes();
/* FORMAT DEPENDENT FUNCTIONS */
#ifdef HAVE_MBOX
static Bool check_mbox(MailBox *box, time_t now);
#endif
#ifdef HAVE_MAILDIR
static Bool list_dir(const char *dirname, int *seen, int *total);
static Bool check_maildir(MailBox *box, time_t now);
#endif
#ifdef HAVE_POP3
static Bool check_pop3(MailBox *box, time_t now);
#endif
#ifdef HAVE_MH
static Bool check_mh(MailBox *box, time_t now);
#endif
#ifdef HAVE_IMAP
static Bool check_imap(MailBox *box, time_t now);
#endif
/* should add functions for mh, maildir and so long... */
/***/
static int  get_globnew();
static Bool fexist(const char *filename);
static Bool filetime(const char *filename, time_t *time, int mode);
static Bool filesize(const char *filename, off_t *fsize);
static int  my_system(char *cmd, char *param);
static void mbox_add(MailBox **list, const char *value);
static BoxType getboxtype(const char *t);
static void free_mbox(MailBox **list);
static int nb_mbox(MailBox *list);
void *xmalloc(size_t size);
char *xstrdup(const char *string);
static Bool getbool(char *value);
static Bool load_cfgfile();
static char *robust_home();
#ifdef HAVE_MH
static Bool isnumber(const char*);
static Bool isdir(const char*);
static Bool isfile(const char *);
static char *absname(const char*, int);
static char *dirfileconcat(const char *dir, const char *file);
static char *get_homedir(void);
static void dirtrim(char *s);
static char *mh_get_profile(void);
static char *mh_context_get(const char *);
static int mh_getseqcnt(const char*);
static int mh_getnmcnt(const char*);
static int mh_getnewmails(const char* );
static int mh_getmails(const char *mh_box);
#endif

#ifndef _GNU_SOURCE
int isblank(int c) {
    return (c == ' ' || c == '\t');
}
#endif


int main(int argc, char **argv) {
    XEvent event;
    struct sigaction sa;

    sa.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
    sa.sa_flags = SA_NOCLDWAIT;
#else
    sa.sa_flags = 0;
#endif
    sigemptyset(&sa.sa_mask);
    sigaction(SIGCHLD, &sa, NULL);

    /* Load default configuration */
    if (! config_file) {
        char *Home = robust_home();
        config_file = xmalloc(strlen(Home) + strlen(DEFAULT_CFGFILE) + 2);
        sprintf(config_file, "%s/%s", Home, DEFAULT_CFGFILE);
    }
    if (fexist(config_file)) load_cfgfile();
    FREE(config_file);

    /* Parse CommandLine */
    parse_arguments(argc, argv);

    if (! config_file) {
        char *Home = robust_home();
        config_file = xmalloc(strlen(Home) + strlen(DEFAULT_CFGFILE) + 2);
        sprintf(config_file, "%s/%s", Home, DEFAULT_CFGFILE);
    } else {
        load_cfgfile();
    }

    /* Initialize Application */
    mail_getnumbers(&new_mails, &total_mails, &alarm_type);
    dockapp_open_window(display_name, PACKAGE, SIZE, SIZE, argc, argv);
    dockapp_set_eventmask(ButtonPressMask);

    install_pixmaps();
    draw_dockapp();

    /* Main loop */
    while (1) {
        if (dockapp_nextevent_or_timeout(&event, update_interval * 1000)) {
            /* Next Event */
            switch (event.type) {
                case ButtonPress:
                    switch (event.xbutton.button) {
                        case 1:
                            if ( (event.xbutton.x >= 13) && (event.xbutton.x <= 17) &&
                                 (event.xbutton.y >= 22) && (event.xbutton.y <= 26) ) {
                                if (nb_mbox(mboxes) > 1) boxnum++;
                                if (boxnum > nb_mbox(mboxes) ) boxnum = 0;
                            } else if ( (event.xbutton.x >= 7) && (event.xbutton.x <= 11) &&
                                        (event.xbutton.y >= 22) && (event.xbutton.y <= 26) ) {
                                if (nb_mbox(mboxes) > 1) boxnum--;
                                if (boxnum < 0 ) boxnum = nb_mbox(mboxes);
                            } else {
                                switch_light();
                            }
                            draw_dockapp();
                            break;
                        case 2:
                            if ((event.xbutton.state & ControlMask) == ControlMask)
                                my_system("wmmaiload-config -f ", config_file);
                            else
                                if (command) my_system(command, NULL);
                            break;
                        case 3:
                            if ( (event.xbutton.x >= 13) && (event.xbutton.x <= 17) &&
                                 (event.xbutton.y >= 22) && (event.xbutton.y <= 26) ) {
                                if (nb_mbox(mboxes) > 1) boxnum += 10;
                                if (boxnum > nb_mbox(mboxes) ) boxnum = 0;
                            } else if ( (event.xbutton.x >= 7) && (event.xbutton.x <= 11) &&
                                        (event.xbutton.y >= 22) && (event.xbutton.y <= 26) ) {
                                if (nb_mbox(mboxes) > 1) boxnum -= 10;
                                if (boxnum < 0 ) boxnum = nb_mbox(mboxes);
                            } else {
                                switch_authorized = !switch_authorized;
                            }
                            break;
                        default:
                            break;
                    }
                    break;
                default: /* make gcc happy */
                    break;
            }
        } else {
            /* Time Out */
            update();
        }
    }

    return 0;
}


static void install_pixmaps() {
    XpmColorSymbol colors[2] = { {"Back0", NULL, 0}, {"Back1", NULL, 0} };
    int ncolor = 0;

    if (light_color) {
        colors[0].pixel = dockapp_getcolor(light_color);
        colors[1].pixel = dockapp_blendedcolor(light_color, -24, -24, -24, 1.0);
        ncolor = 2;
    }

    /* change raw xpm data to pixmap */
    if (dockapp_iswindowed) {
        backlight_on_xpm[1]  = WINDOWED_BG;
        backlight_off_xpm[1] = WINDOWED_BG;
    }

    if (backdrop_on) XFreePixmap(display, backdrop_on);
    if (!dockapp_xpm2pixmap(backlight_on_xpm, &backdrop_on, &mask, colors, ncolor)) {
        fprintf(stderr, "Error initializing backlit background image.\n");
        exit(1);
    }
    if (backdrop_off) XFreePixmap(display, backdrop_off);
    if (!dockapp_xpm2pixmap(backlight_off_xpm, &backdrop_off, NULL, NULL, 0)) {
        fprintf(stderr, "Error initializing background image.\n");
        exit(1);
    }
    if (parts) XFreePixmap(display, parts);
    if (!dockapp_xpm2pixmap(parts_xpm, &parts, NULL, colors, ncolor)) {
        fprintf(stderr, "Error initializing parts image.\n");
        exit(1);
    }

    /* shape window */
    if (!dockapp_iswindowed) dockapp_setshape(mask, 0, 0);
    if (mask) XFreePixmap(display, mask);

    /* pixmap : draw area */
    if (pixmap) XFreePixmap(display, pixmap);
    pixmap = dockapp_XCreatePixmap(SIZE, SIZE);

    /* Initialize pixmap */
    if (backlight == LIGHTON) 
        dockapp_copyarea(backdrop_on, pixmap, 0, 0, SIZE, SIZE, 0, 0);
    else
        dockapp_copyarea(backdrop_off, pixmap, 0, 0, SIZE, SIZE, 0, 0);

    dockapp_set_background(pixmap);
    dockapp_show();
}


/* called by timer */
static void update() {
    static light pre_backlight;
    static MailAlarm in_alarm_mode = NOALARM;

    /* check if the config file has been modified */
    if (load_cfgfile()) {
        install_pixmaps();
        mail_getnumbers(&new_mails, &total_mails, &alarm_type);
    }

    /* get current mailbox' mails numbers */
    mail_getnumbers(&new_mails, &total_mails, &alarm_type);

    /* alarm mode */
    if (alarm_type != NOALARM) {
        if (in_alarm_mode == NOALARM) {
            in_alarm_mode = alarm_type;
            pre_backlight = backlight;
        }
        if ( (alarm_type == NEWGLOB) || (alarm_type == NEWBOTH) ) {
            if ( (switch_authorized) ||
                 ( (! switch_authorized) && (backlight != pre_backlight) ) ) {
                switch_light();
            }
            if (alarm_type == NEWGLOB) display_new = True;
        }
        if ( (alarm_type == NEWHERE) || (alarm_type == NEWBOTH) ) {
            if (switch_authorized) switch_leds();
            if (alarm_type == NEWHERE) backlight = pre_backlight;
        }
    } else {
        if (in_alarm_mode != NOALARM) {
            in_alarm_mode = NOALARM;
            if (backlight != pre_backlight) switch_light();
            display_new = True;
        }
    }

    draw_dockapp();
}


/* called when mouse button pressed */
static void switch_light() {
    switch (backlight) {
        case LIGHTOFF: backlight = LIGHTON;  break;
        case LIGHTON:  backlight = LIGHTOFF; break;
    }
}


static void switch_leds() {
    display_new = !display_new;
}


static void draw_dockapp() {
    /* all clear */
    if (backlight == LIGHTON)
        dockapp_copyarea(backdrop_on, pixmap, 0, 0, 58, 58, 0, 0);
    else
        dockapp_copyarea(backdrop_off, pixmap, 0, 0, 58, 58, 0, 0);

    /* draw digit */
    draw_boxdigit(boxnum);
    if (display_new) draw_newdigit(new_mails);
    draw_totaldigit(total_mails);

    /* show */
    dockapp_copy2window(pixmap);
}


static void draw_boxdigit(int num) {
    int x = 0, y = 31;

    if (backlight == LIGHTON) {
        x = 50;
        y = 40;
    }

    /* draw digits */
    if (num != 0) {
        dockapp_copyarea(parts, pixmap, (num % 10) * 5 + x, 40, 5, 9, 13, 12);
        dockapp_copyarea(parts, pixmap, (num / 10) * 5 + x, 40, 5, 9,  7, 12);
    } else {
        dockapp_copyarea(parts, pixmap, 100, y, 5, 9, 13, 12);
        dockapp_copyarea(parts, pixmap, 100, y, 5, 9,  7, 12);
    }
}


static void draw_newdigit(int num) {
    int v1000, v100, v10, v1;
    int y = 0;

    if (num < 0) num = 0;

    v1000 = num / 1000;
    v100  = (num - v1000 * 1000) / 100;
    v10   = (num - v1000 * 1000 - v100 * 100) / 10;
    v1    = (num - v1000 * 1000 - v100 * 100 - v10 * 10);

    if (backlight == LIGHTON) y = 20;

    /* draw digits */
    dockapp_copyarea(parts, pixmap, v1 * 10, y, 10, 20, 45, 7);
    if ( (v10 != 0) || ( (v10 == 0) && (v100 != 0) ) )
        dockapp_copyarea(parts, pixmap, v10 * 10, y, 10, 20, 33, 7);
    if (v100 != 0)
        dockapp_copyarea(parts, pixmap, v100 * 10, y, 10, 20, 21, 7);
}


static void draw_totaldigit(int num) {
    int v10k, v1000, v100, v10, v1;
    int y = 0;

    if (num < 0) num = 0;

    v10k  = num / 10000;
    v1000 = (num - v10k * 10000) / 1000;
    v100  = (num - v10k * 10000 - v1000 * 1000) / 100;
    v10   = (num - v10k * 10000 - v1000 * 1000 - v100 * 100) / 10;
    v1    = (num - v10k * 10000 - v1000 * 1000 - v100 * 100 - v10 * 10);

    if (backlight == LIGHTON) y = 20;

    /* draw digits */
    dockapp_copyarea(parts, pixmap, v1 * 10, y, 10, 20, 45, 34);
    if ( (v10 != 0) || ( (v10 == 0) && (v100 != 0) ) || ( (v10 == 0) && (v100 == 0) && (v1000 != 0) ) )
        dockapp_copyarea(parts, pixmap, v10 * 10, y, 10, 20, 33, 34);
    if ( (v100 != 0) || ( (v100 == 0) && (v1000 != 0) ) )
        dockapp_copyarea(parts, pixmap, v100 * 10, y, 10, 20, 21, 34);
    if (v1000 != 0)
        dockapp_copyarea(parts, pixmap, v1000 * 10, y, 10, 20, 9, 34);
}


static void parse_arguments(int argc, char **argv) {
    int i;
    int integer;
    for (i = 1; i < argc; i++) {
        if (!strcmp(argv[i], "--help") || !strcmp(argv[i], "-h")) {
            print_help(argv[0]), exit(0);
        } else if (!strcmp(argv[i], "--version") || !strcmp(argv[i], "-v")) {
            printf("%s version %s\n", PACKAGE, VERSION), exit(0);
        } else if (!strcmp(argv[i], "--display") || !strcmp(argv[i], "-d")) {
            display_name = argv[i + 1];
            i++;
        } else if (!strcmp(argv[i], "--backlight") || !strcmp(argv[i], "-bl")) {
            backlight = LIGHTON;
        } else if (!strcmp(argv[i], "--light-color") || !strcmp(argv[i], "-lc")) {
            SET_STRING(light_color, argv[i + 1]);
            i++;
        } else if (!strcmp(argv[i], "--interval") || !strcmp(argv[i], "-i")) {
            if (argc == i + 1)
                fprintf(stderr, "%s: error parsing argument for option %s\n",
                        argv[0], argv[i]), exit(1);
            if (sscanf(argv[i + 1], "%i", &integer) != 1)
                fprintf(stderr, "%s: error parsing argument for option %s\n",
                        argv[0], argv[i]), exit(1);
            if (integer < 1)
                fprintf(stderr, "%s: argument %s must be >=1\n",
                        argv[0], argv[i]), exit(1);
            update_interval = integer;
            i++;
        } else if (!strcmp(argv[i], "--windowed") || !strcmp(argv[i], "-w")) {
            dockapp_iswindowed = True;
        } else if (!strcmp(argv[i], "--broken-wm") || !strcmp(argv[i], "-bw")) {
            dockapp_isbrokenwm = True;
        } else if (!strcmp(argv[i], "--no-blink") || !strcmp(argv[i], "-nb")) {
            switch_authorized = False;
        } else if (!strcmp(argv[i], "--mailbox") || !strcmp(argv[i], "-m")) {
            mbox_add(&mboxes, argv[i + 1]);
            i++;
        } else if (!strcmp(argv[i], "--command") || !strcmp(argv[i], "-c")) {
            SET_STRING(command, argv[i + 1]);
            i++;
        } else if (!strcmp(argv[i], "--notify") || !strcmp(argv[i], "-n")) {
            SET_STRING(notif_cmd, argv[i + 1]);
            i++;
        } else if (!strcmp(argv[i], "--cfgfile") || !strcmp(argv[i], "-f")) {
            SET_STRING(config_file, argv[i + 1]);
            i++;
        } else if (!strcmp(argv[i], "--box-num") || !strcmp(argv[i], "-bn")) {
            if (argc == i + 1)
                fprintf(stderr, "%s: error parsing argument for option %s\n",
                        argv[0], argv[i]), exit(1);
            if (sscanf(argv[i + 1], "%i", &integer) != 1)
                fprintf(stderr, "%s: error parsing argument for option %s\n",
                        argv[0], argv[i]), exit(1);
            if ( (integer < 0) || (integer > nb_mbox(mboxes) ) )
                fprintf(stderr, "%s: argument %s must be >=0 and <=%d\n",
                        argv[0], argv[i], nb_mbox(mboxes)), exit(1);
            boxnum = integer;
            i++;
        } else if (!strcmp(argv[i], "--check-delay") || !strcmp(argv[i], "-cd")) {
            if (argc == i + 1)
                fprintf(stderr, "%s: error parsing argument for option %s\n",
                        argv[0], argv[i]), exit(1);
            if (sscanf(argv[i + 1], "%i", &integer) != 1)
                fprintf(stderr, "%s: error parsing argument for option %s\n",
                        argv[0], argv[i]), exit(1);
            if (integer < 15)
                fprintf(stderr, "%s: argument %s must be >=15\n",
                        argv[0], argv[i]), exit(1);
            check_delay = integer;
            i++;
        } else if (!strcmp(argv[i], "--run-once") || !strcmp(argv[i], "-r1")) {
            run_once = True;
        } else if (!strcmp(argv[i], "--size-check") || !strcmp(argv[i], "-s")) {
            test_size = True;
        } else {
            fprintf(stderr, "%s: unrecognized option '%s'\n", argv[0], argv[i]);
            print_help(argv[0]), exit(1);
        }
    }
}


static void print_help(char *prog)
{
    printf("Usage : %s [OPTIONS]\n", prog);
    printf("wmmaiload - Window Maker mails monitor dockapp\n"
           "  -d,  --display <string>      display to use\n"
           "  -bl, --backlight             turn on back-light\n"
           "  -lc, --light-color <string>  back-light color(rgb:6E/C6/3B is default)\n"
           "  -i,  --interval <number>     number of secs between updates (1 is default)\n"
           "  -h,  --help                  show this help text and exit\n"
           "  -v,  --version               show program version and exit\n"
           "  -w,  --windowed              run the application in windowed mode\n"
           "  -bw, --broken-wm             activate broken window manager fix\n"
           "  -m,  --mailbox <string>      use that mboxfile (may be used more than once)\n"
           "  -c,  --command <string>      command to launch with button 2\n"
           "  -n,  --notify <string>       command to launch each time there is new mail\n"
           "  -nb, --no-blink              disable blinking when there is new mail\n"
           "  -bn, --box-num <number>      display specified box infos directly (def. 0 for total)\n"
           "  -cb, --check-delay <number>  pop3, hotmail and imap boxes verification delay (seconds)\n"
           "  -r1, --run-once              run command (see -n) only once\n"
           "  -f,  --cfgfile               load configuration file\n"
           "  -s,  --size-check            use files' size to check for new mails\n");
    printf("\nCompiled features : ");
#ifdef HAVE_MBOX
    printf("MBOX ");
#endif
#ifdef HAVE_MAILDIR
    printf("MAILDIR ");
#endif
#ifdef HAVE_POP3
    printf("POP3/HOTMAIL ");
#endif
#ifdef HAVE_MH
    printf("MH ");
#endif
#ifdef HAVE_IMAP
    printf("IMAP ");
#endif
    printf("\n");
}


static void mail_getnumbers(int *new, int *total, MailAlarm *bell) {
    static int bnum = 0;
    int  newmails = 0, nbmails = 0, globnew = 0, globtot = 0, i;
    Bool changed = False;
    MailAlarm alrm;
    MailBox *box = mboxes;

    changed = update_mailboxes();
    if (changed || bnum != boxnum) {
        bnum = boxnum;
        for (i = 0 ; box ; i++) {
            globnew += box->new;
            globtot += box->total;
            box = box->next;
        }
        if (boxnum != 0) {
            i   = 1;
            box = mboxes;
            while (box && (i != boxnum) ) {
                box = box->next;
                i++;
            }
            if (box) {
                newmails = box->new;
                nbmails  = box->total;
            }
            alrm = NOALARM;
            if (globnew > 0)
                alrm = (globnew - newmails == globnew) ? NEWGLOB :
                       (globnew - newmails > 0)        ? NEWBOTH :
                       (globnew - newmails == 0)       ? NEWHERE :
                       NOALARM;
        } else {
            newmails = globnew;
            nbmails  = globtot;
            alrm = (globnew > 0) ? NEWGLOB : NOALARM;
        }
        *new   = newmails;
        *total = nbmails;
        *bell  = alrm;
    }
}


static Bool update_mailboxes() {
    int     i, globnew, prevnew;
    Bool    changed  = False;
    MailBox *mailbox = mboxes;
    time_t  now;

    prevnew = get_globnew();
    time(&now);
    for (i = 0 ; mailbox ; i++) {
        Bool modif = False;
        switch (mailbox->type) {
#ifdef HAVE_MBOX
            case MBOX:    modif = check_mbox(mailbox, now);    break;
#else
            case MBOX:    break;
#endif
#ifdef HAVE_MAILDIR
            case MAILDIR: modif = check_maildir(mailbox, now); break;
#else
            case MAILDIR: break;
#endif
#ifdef HAVE_MH
            case MH:      modif = check_mh(mailbox, now);      break;
#else
            case MH:      break;
#endif
#ifdef HAVE_POP3
            case POP3:    modif = check_pop3(mailbox, now);    break;
            case HOTMAIL: modif = check_pop3(mailbox, now);    break;
#else
            case POP3:    break;
            case HOTMAIL: break;
#endif
#ifdef HAVE_IMAP
            case IMAP:    modif = check_imap(mailbox, now);    break;
#else
            case IMAP:    break;
#endif
            default:      break;
        }
        if (modif) changed = True;
        mailbox  = mailbox->next;
    }
    globnew = get_globnew();

    if ( (changed) && (notif_cmd) &&
        (
          ( (globnew > prevnew) && (!run_once) ) ||
          ( (globnew > 0) && (prevnew == 0) && (run_once) )
        )
       )
        my_system(notif_cmd, NULL);

    return changed;
}


#ifdef HAVE_MBOX
static Bool check_mbox(MailBox *box, time_t now) {
    FILE *file;
    /*struct utimbuf u;*/
    char line[MAXSTRLEN + 1];
    int  nbmails     = 0;
    int  nbreadmails = 0;
    int  longline    = 0;
    int  inheaders   = 0;
    int  force       = 0;

    force = ((now - box->time > 10) && (!box->updated));
    if (force) {
        box->time = 0;
        box->size = 0;
    }
    if (!fexist(box->entry) )                                     return False;
    if (!test_size && !filetime(box->entry, &(box->time), CTIME)) return False;
    if (test_size && !filesize(box->entry, &(box->size)))         return False;
    if ((file = fopen(box->entry, "r")) == NULL)                  return False;
    while (! feof(file) ) {
        fgets(line, MAXSTRLEN, file);
        if (! longline) {
            if (strncmp(line, "From ", 5) == 0) {
                nbmails++;
                inheaders = 1;
            } else if (inheaders == 1) {
                if (strcmp(line, "\n") == 0) {
                    inheaders = 0;
                } else if ( (strncmp(line, "Status:", 7) == 0) &&
                        (strchr(line, 'R') != NULL) ) {
                    nbreadmails++;
                }
            }
        }
        longline = (strchr(line, '\n') == NULL);
    }
    fclose (file);
    box->new     = nbmails - nbreadmails;
    box->total   = nbmails;
    box->updated = force;
    /*
    u.actime  = box->atime;
    u.modtime = box->mtime;
    utime(box->entry, &u);
    */

    return True;
}
#endif


#ifdef HAVE_MAILDIR
static Bool list_dir(const char *dirname, int *seen, int *total) {
    DIR           *dir;
    struct dirent *dir_ent;

    if ((dir = opendir(dirname)) == NULL) return False;
    while ((dir_ent = readdir(dir)) != NULL) {
        if (dir_ent->d_name[0] != '.') {
            char *lastpart;
            (*total)++;
            if ((lastpart = strstr(dir_ent->d_name, ":2")) != NULL) {
                if (strstr(lastpart, "S") != NULL) (*seen)++;
            }
        }
    }
    closedir(dir);
    return True;
}


static Bool check_maildir(MailBox *box, time_t now) {
    char   cur[MAXSTRLEN + 1], new[MAXSTRLEN + 1];
    int    nbmails     = 0;
    int    nbreadmails = 0;
    int    force       = 0;
    time_t btime1, btime2;

    force = ((now - box->time > 10) && (!box->updated));
    if (force) {
        box->time = 0;
        box->size = 0;
    }
    btime1 = box->time;
    btime2 = box->time;
    bzero(cur, MAXSTRLEN + 1);
    bzero(new, MAXSTRLEN + 1);
    strcpy(cur, box->entry);
    strcpy(new, box->entry);
    strcat(cur, "/cur");
    strcat(new, "/new");
    if ((!fexist(cur)) || (!fexist(new)))       return False;
    if ((!filetime(cur, &btime1, CTIME)) &&
        (!filetime(new, &btime2, CTIME)))       return False;
    if (!list_dir(cur, &nbreadmails, &nbmails)) return False;
    if (!list_dir(new, &nbreadmails, &nbmails)) return False;
    box->new     = nbmails - nbreadmails;
    box->total   = nbmails;
    box->updated = force;
    if (btime1 != box->time)
        box->time = btime1;
    else
        box->time = btime2;

    return True;
}
#endif


#ifdef HAVE_POP3
static Bool check_pop3(MailBox *box, time_t now) {
    Pop3 p3 = NULL;

    if ((now - box->time) > check_delay) {
        int port;
        p3        = initPop3();
        box->time = time(0);
        switch(box->type) {
            case POP3:
                port = box->port != 0 ? box->port : 110;
                break;
            case HOTMAIL:
                port = box->port != 0 ? box->port : 1100;
                break;
            default:
		break;
        }
        if ( ((pop3Connect(p3, box->entry, port)) == -1) ||
             ((pop3Login(p3, box->username, box->password)) == -1) ||
             ((pop3CheckMail(p3)) == -1) ) {
            free_pop3(&p3);
            return False;
        }
        pop3Quit(p3);
        box->new     = pop3GetUnreadMess(p3);
        box->total   = pop3GetTotalMess(p3);
        box->updated = True;
        box->time    = time(0);
        free_pop3(&p3);
        return True;
    }
    return False;
}
#endif


#ifdef HAVE_IMAP
static Bool check_imap(MailBox *box, time_t now) {
	Imap imap;

	if((now - box->time) > check_delay) {
		imap = initImap();
		box->time = time(0);
		if((imapConnect(imap, box->entry, 143)) == -1)
			return False;
		if((imapLogin(imap, box->username, box->password)) == -1)
			return False;
		if((imapCheckMail(imap, box->folder)) == -1)
			return False;
		imapQuit(imap);
		free(imap);
		box->new	= imapGetUnreadMess(imap);
		box->total	= imapGetTotalMess(imap);
		box->updated	= True;
		box->time	= time(0);
		return True;
	}
	return False;
}
#endif


static int get_globnew() {
    int     globnew = 0;
    MailBox *box    = mboxes;

    while (box) {
        globnew += box->new;
        box = box->next;
    }
    return globnew;
}


static Bool fexist(const char *filename) {
    /*
    FILE           *file;
    struct utimbuf u;
    */
    struct stat    s;
    

    if (stat(filename, &s) == -1) return False;
    /*
    if ( (file = fopen(filename, "r") ) == NULL) return False;
    fclose(file);

    u.actime  = s.st_atime;
    u.modtime = s.st_mtime;
    utime(filename, &u);
    */
    return True;
}


static Bool filetime(const char *filename, time_t *time, int mode) {
    struct stat s;
    time_t      t = 0;

    if (stat(filename, &s) == -1) return False;
    switch (mode) {
        case MTIME: t = s.st_mtime; break;
        case ATIME: t = s.st_atime; break;
        case CTIME: t = s.st_ctime; break;
        default: break;
    }
    if (t == *time) {
        return False;
    } else {
        *time = t;
        return True;
    }
}


static Bool filesize(const char *filename, off_t *fsize) {
    struct stat s;

    if (stat(filename, &s) == -1) return False;
    if (s.st_size == *fsize) {
        return False;
    } else {
        *fsize = s.st_size;
        return True;
    }
    return False; /* Should not pass here ! */
}


static int my_system(char *cmd, char *param) {
    int           pid;
    extern char **environ;

    if (cmd == 0) return 1;
    pid = fork();
    if (pid == -1) return -1;
    if (pid == 0) {
        pid = fork();
        if (pid == 0) {
            char *argv[4];
            char fcmd[2048];
            setsid();
            strncpy(fcmd, cmd, 2047);
            if (param) strncat(fcmd, param, 2045 - strlen(cmd));
            argv[0] = "sh";
            argv[1] = "-c";
            argv[2] = fcmd;
            argv[3] = 0;
            execve("/bin/sh", argv, environ);
            exit(0);
        }
        exit(0);
    }
    return 0;
}


static void mbox_add(MailBox **list, const char *value) {
    MailBox *lst = *list;
    Bool     ok  = True;
    char     box[100];
    char     user[25];
    char     pass[25];
    char     btype[10];
    char     folder[100];
    char    *pstr;
    int count, port = 0;

    if (! value) return;
    bzero(box, 100);
    bzero(btype, 10);
    bzero(user, 25);
    bzero(pass, 25);
    bzero(folder, 100);
    count = sscanf(value, "%s %s %s %s %s", btype, box, user, pass, folder);

    pstr = strchr(box, ':');
    if (pstr != NULL) {
        *pstr = 0;
        pstr++;
        port = atoi(pstr);
    }

    if (((getboxtype(btype) == MBOX) ||
         (getboxtype(btype) == MAILDIR) || 
         (getboxtype(btype) == MH)) &&
        count < 2) {
        fprintf(stderr, "Invalid Mail box parameters !\n");
        return;
    }

    if ( ((getboxtype(btype) == MBOX) ||
          (getboxtype(btype) == MAILDIR) || 
          (getboxtype(btype) == MH) ) &&
         (! fexist(box))) {
        fprintf(stderr, "Mail box '%s' does not exist !\n", box);
        return;
    }
    if (! lst) {
        lst   = xmalloc(sizeof(MailBox));
        *list = lst;
    } else {
        if (strcmp(value, lst->entry) == 0) ok = False;
        while ( (lst->next) && ok) {
            lst = lst->next;
            if (strcmp(value, lst->entry) == 0) ok = False;
        }
        if (! ok) return;
        lst->next = xmalloc(sizeof(MailBox));
        lst = lst->next;
    }
    lst->entry    = xstrdup(box);
    lst->time     = 0;
    lst->size     = 0;
    lst->new      = 0;
    lst->total    = 0;
    lst->port     = port;
    lst->type     = getboxtype(btype);
    lst->username = user ? xstrdup(user) : NULL;
    lst->password = pass ? xstrdup(pass) : NULL;
    lst->folder   = folder ? xstrdup(folder) : NULL;
    lst->next     = NULL;
}


static BoxType getboxtype(const char *t) {
    if ((strcmp(t, "MBOX")) == 0)    return MBOX;
    if ((strcmp(t, "MAILDIR")) == 0) return MAILDIR;
    if ((strcmp(t, "MH")) == 0)      return MH;
    if ((strcmp(t, "HOTMAIL")) == 0) return HOTMAIL;
    if ((strcmp(t, "POP3")) == 0)    return POP3;
    if ((strcmp(t, "IMAP")) == 0)    return IMAP;
    return MBOX;
}
 

static void free_mbox(MailBox **list) {
    MailBox *lst = *list, *next;
    while (lst) {
        next = lst->next;
        FREE(lst->entry);
        FREE(lst->username);
        FREE(lst->password);
        FREE(lst->folder);
        free(lst);
        lst = next;
    }
    *list = NULL;
}


static int nb_mbox(MailBox *list) {
    MailBox *box = list;
    int      n = 0;
    while (box) {
        n++;
        box = box->next;
    }
    return n;
}


void *xmalloc(size_t size) {
    void *ret = malloc(size);
    if (ret == NULL) {
        perror("malloc() ");
        exit(-1);
    } else
        return ret;
}


char *xstrdup(const char *string) {
    char *ret;
    if(!string || string[0] == '\0')
        return NULL;
    ret = strdup(string);
    if (ret == NULL) {
        perror("strdup() ");
        exit(-1);
    } else
        return ret;
}


static Bool getbool(char *value) {
    int i;
    for (i = 0 ; value[i] ; i++) value[i] = tolower(value[i]);
    if (strcmp(value, "0") == 0) return False;
    if (strcmp(value, "1") == 0) return True;
    if (strcmp(value, "true") == 0) return True;
    if (strcmp(value, "false") == 0) return False;
    if (strcmp(value, "yes") == 0) return True;
    if (strcmp(value, "no") == 0) return False;
    if (strcmp(value, "on") == 0) return True;
    if (strcmp(value, "off") == 0) return False;
    fprintf(stderr, "Error in converting \"%s\" to boolean value.\n", value);
    return False;
}


static Bool load_cfgfile() {
    FILE *file;
    int  i = 0;
    char line[MAXSTRLEN + 1];
    char *value;

    if ( (file = fopen(config_file, "r")) == NULL) {
        if (strstr(config_file, "/"DEFAULT_CFGFILE) == NULL)
            fprintf(stderr, "Unable to open configuration file \"%s\".\n",
                    config_file);
        return False;
    }
    if ( (!test_size && !filetime(config_file, &config_mtime, MTIME)) ||
         (test_size && !filesize(config_file, &config_fsize)) ) {
        fclose(file);
        return False;
    }
    if (mboxes) free_mbox(&mboxes);
    while (! feof(file)) {
        bzero(line, MAXSTRLEN + 1);
        fgets(line, MAXSTRLEN, file);
        i++;
        if (line[strlen(line) - 1] == '\n') line[strlen(line) - 1] = 0;
        if ( (line[0] == '#') || (line[0] == 0) ) continue;
        value = strchr(line, '=') + 1;
        while ( (value[0] == ' ') && (value[0] != 0) ) value++;
        if (value[0] == 0) continue;

        if (strncmp(line, "Backlight", 9) == 0)
            backlight = getbool(value) ? LIGHTON : LIGHTOFF;
        else if (strncmp(line, "Color", 5) == 0) {
            SET_STRING(light_color, value);
        } else if (strncmp(line, "Interval", 8) == 0)
            update_interval = atoi(value);
        else if (strncmp(line, "Mailbox", 7) == 0)
            mbox_add(&mboxes, value);
        else if (strncmp(line, "Command", 7) == 0) {
            SET_STRING(command, value);
        } else if (strncmp(line, "Notify", 6) == 0) {
            SET_STRING(notif_cmd, value);
        } else if (strncmp(line, "Blink", 5) == 0)
            switch_authorized = getbool(value);
        else if (strncmp(line, "DefaultBox", 10) == 0)
            boxnum = atoi(value);
        else if (strncmp(line, "CheckDelay", 9) == 0) {
            check_delay = atoi(value);
            if (check_delay < 15) check_delay = 15;
        } else if (strncmp(line, "RunOnce", 7) == 0)
            run_once = getbool(value);
        else if (strncmp(line, "CheckSize", 9) == 0)
            test_size = getbool(value);
        else
            fprintf(stderr, "Error in %s at line %d :\n[%s].\n",
                    config_file, i, line);
    }
    return True;
}


static char *robust_home() {
    if (getenv("HOME"))
        return getenv("HOME");
    else if (getenv("USER") && getpwnam(getenv("USER")))
        return getpwnam(getenv("USER"))->pw_dir;
    else if (getenv("LOGNAME") && getpwnam(getenv("LOGNAME")))
        return getpwnam(getenv("LOGNAME"))->pw_dir;
    else if ((getuid() != -1) && (getpwuid(getuid())))
        return getpwuid(getuid())->pw_dir;
    else
        return "/";
}


#ifdef HAVE_MH
static Bool isnumber(const char *s) {
    while (*s != '\0') {
        if (!isdigit(*s)) return False;
        s++;
    }

    return True;
}


static Bool isdir(const char *d) {
    struct stat dir_stats;

    if ( (stat(d, &dir_stats) != 0) || !S_ISDIR(dir_stats.st_mode))
        return False;
    return True;
}


static Bool isfile(const char *f) {
    struct stat file_stats;

    if((stat(f, &file_stats) != 0) || !S_ISREG(file_stats.st_mode))
        return False;
    return True;
}


static char* absname(const char *name, int rt) {
    char *retval = NULL;
    char tmp1[MAXSTRLEN];
    char *tmp2 = NULL;
    Bool rc = False;

    if (!name || !strlen(name)) goto bailout;

    if (name[0] == '/') {
        char *cp = NULL; 
        
        retval = xstrdup(name);
        cp = retval + strlen(retval) - 1;
        if (*cp == '/') *cp = 0;
        rc = True;
        goto bailout;
    }

    switch (rt) {
    case CWD:
        if (!getcwd(tmp1, MAXSTRLEN)) goto bailout;
        if (!(retval = dirfileconcat(tmp1, name))) goto bailout;
        break;
    case HOME:
        if (!(tmp2 = get_homedir())) goto bailout;
        retval = dirfileconcat(tmp2, name);
        FREE(tmp2);
        if (!retval) goto bailout;
        break;
    default:
        goto bailout;
        break;
    }

    DIRTRIM(retval);

    rc =True;
  bailout:
    if (!rc) FREE(retval);
    return retval;
}


static char *dirfileconcat(const char *dir, const char *file) {
    char *tmp = NULL;

    if (!dir || !file) goto bailout;

    tmp = (char*)xmalloc(strlen(dir) + strlen(file) + 2);
    if (!tmp) goto bailout;

    strcpy(tmp, dir);
    DIRTRIM(tmp);
    strcat(tmp, "/");
    strcat(tmp, file);
    
  bailout:
    return tmp;
}


static char *get_homedir(void) {
    char          *mypath = NULL;
    char          *cp = NULL;
    struct passwd *pw;

    /* XXX is it usefull ??? */
    mypath = xstrdup(robust_home());
    if (mypath[0] != '\0') DIRTRIM(mypath);
}

            
static char *mh_get_profile(void) {
    char *cp = NULL;
    Bool rc = False;
    char *profile = NULL;

    if ((cp = getenv("MH"))) {
        if (*cp |= '/')
            profile = absname(cp, CWD);
        else
            profile = xstrdup(cp);
    } else
        profile = absname(MH_PROFILE, HOME);

    if (!profile) goto bailout;
    
    if (!isfile(profile)) goto bailout;

    rc = True;
  bailout:
    FREE(cp);
    if (!rc) FREE(profile);
    return profile;
}


static char *mh_context_get(const char *str) {
    char *home;
    char *retval = NULL;
    char *profile = NULL;
    char line[MAXSTRLEN];
    FILE *fp = NULL;
    char *cp = NULL;

    if (!(home = get_homedir())) goto bailout;

    if (!(profile = mh_get_profile())) goto bailout;

    if (!(fp = fopen(profile,"r"))) goto bailout;

    while (fgets(line, MAXSTRLEN, fp)) {
        cp = line;
        while (isblank(*cp)) cp++;
        if (strncasecmp(str, cp, strlen(str))) continue;

        cp += strlen(str);
        while (isblank(*cp)) cp++;
        if (*cp != ':') continue;

        cp++;
        while (isblank(*cp)) cp++;
        if (!(retval = xstrdup(cp))) continue;

        DIRTRIM(retval);
        break;
    }

    FCLOSE(fp);
  bailout:
    
    FREE(home);
    FREE(profile);
    FCLOSE(fp);

    return retval;
}


static int mh_getseqcnt(const char *word) {
    int retval = 0;

    if (isnumber(word))
        retval = 1;
    else {
        int start,end;

        if ((sscanf(word, "%d-%d", &start, &end) == 2) && (end > start))
            retval = end-start+1;
    }

    return retval;
}


static int mh_getnmcnt(const char *cp) {
    char word[MAXSTRLEN];
    int  i = 0;
    int  retval = 0;

    while (*cp != '\0') {
        if (i == MAXSTRLEN) goto bailout;

        if (isspace(*cp)) {
            if (i) {
                word[i] = '\0';
                retval += mh_getseqcnt(word);
                i = 0;
            }
        }
        else
            word[i++] = *cp;

        cp++;
    }

    if (i) {
        word[i] = '\0';
        retval += mh_getseqcnt(word);
        i = 0;
    }

  bailout:
    return retval;
}


static int mh_getnewmails(const char *mh_box) {
    int  nbunread = 0;
    char *tmp = NULL;
    char line[MAXSTRLEN];
    char *seqfile;
    char *unseen;
    FILE *fp;

    if (!mh_box) goto bailout;

    if (!isdir(mh_box)) goto bailout;

    if (!(tmp = mh_context_get("mh-sequences")) &&
        !(tmp = xstrdup(".mh_sequences")))
        goto bailout;

    seqfile = dirfileconcat(mh_box, tmp);
    FREE(tmp);
    if (!seqfile) goto bailout;

    if (!isfile(seqfile)) goto bailout;

    if (!(unseen = mh_context_get("unseen-sequence")) &&
        !(unseen = xstrdup("unseen")))
        goto bailout;

    if (!(fp = fopen(seqfile, "r"))) goto bailout;

    while (fgets(line, MAXSTRLEN, fp)) {
        char *cp = line;

        while (isblank(*cp)) cp++;

        if (strncasecmp(unseen, cp, strlen(unseen))) continue;
        cp += strlen(unseen);

        while (isblank(*cp)) cp++;
        if (*cp != ':') continue;

        cp++;
        while (isblank(*cp)) cp++;
        nbunread = mh_getnmcnt(cp);
        break;
    }

    FCLOSE(fp);

  bailout:
    FREE(unseen);
    FREE(seqfile);
    return nbunread;
}


static int mh_getmails(const char *mh_box) {
    int           nbmails = 0;
    DIR           *dir;
    struct dirent *d_entry;

    if (!mh_box) goto bailout;
    if (!isdir(mh_box)) goto bailout;
    if (!(dir=opendir(mh_box))) goto bailout;

    while((d_entry=readdir(dir))) {
        if(isnumber(d_entry->d_name)) {
            char *tmp = NULL;

            if (!(tmp = dirfileconcat(mh_box, d_entry-> d_name)))
                goto bailout;

            DIRTRIM(tmp);

            if (!isfile(tmp)) continue;
            
            nbmails++;
            FREE(tmp);
        }
    }
    
  bailout:
        DCLOSE(dir);

    return nbmails;
}


static Bool check_mh(MailBox *box, time_t now) {
    Bool retval = False;
    int  force = 0;
    char *tmp = NULL;
    char *mh_box = NULL;
    int  nbmails = 0;
    int  nbunread = 0;
    char *path;

    if (!box) goto bailout;

    force = ((now - box->time > 10) && (!box->updated));
    if (force) {
        box->time = 0;
        box->size = 0;
    }

    if (!(tmp = mh_context_get("path"))) goto bailout;

    path = absname(tmp, HOME);
    FREE(tmp);
    if (!path) goto bailout;
    if (!isdir(path)) goto bailout;
    if (!box->entry) goto bailout;

    DIRTRIM(box->entry);

    if (!(mh_box=dirfileconcat(path, box->entry))) goto bailout;
    if (!isdir(mh_box)) goto bailout;

    nbmails  = mh_getmails(mh_box);
    nbunread = mh_getnewmails(mh_box);

    box->new     = nbunread;
    box->total   = nbmails;
    box->updated = force;
    retval       = True;
  bailout:
    FREE(mh_box);
    FREE(path);

    return retval;
}


static void dirtrim(char *s) {
    char *cp;

    cp = s + strlen(s) - 1;
    while (isspace(*cp)) cp--;
    if (*cp == '/')
        *cp = '\0';
    else
        *(cp + 1) = '\0';

    return;
}
#endif /* HAVE_MH */

