/***************************************************************************
 *   Copyright (C) 2005 by Pawel Nawrocki                                  *
 *   pnawrocki@interia.pl                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "wlassistant.h"
#include "netlistviewitem.h"
#include "waconfig.h"
#include "watools.h"
#include "ui_netparamswizard.h"
#include "ui_netparamsedit.h"

#include <iostream>

#include <qregexp.h>
#include <qlabel.h>
#include <qprocess.h>
#include <qcursor.h>
#include <qeventloop.h>
#include <qtimer.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qwidgetstack.h>
#include <qtooltip.h>

#include <kpushbutton.h>
#include <kcombobox.h>
#include <klistview.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <klocale.h>

WirelessAssistant::WirelessAssistant(QWidget* parent, const char* name, bool modal, WFlags fl)
                : mainWindow(parent,name, modal,fl)
{
        buttonScan->setIconSet( SmallIconSet("reload") );
        buttonConnect->setIconSet( SmallIconSet("connect_creating") );
        buttonOptions->setIconSet( SmallIconSet("configure") );
        buttonClose->setIconSet( SmallIconSet("fileclose") );

        netList->setAllColumnsShowFocus(1);
        netList->setItemMargin(8);
        frameDevice->hide();

        /// Network List Widget
        connect( buttonScan, SIGNAL(clicked()),
                 this, SLOT(netScan()) );

        connect( buttonConnect, SIGNAL(clicked()),
                 this, SLOT(itemAction()) );

        connect( buttonClose, SIGNAL(clicked()),
                 this, SLOT(close()) );

        connect( devCombo, SIGNAL(activated( const QString & )),
                 this, SLOT(setDev( const QString & )) );

        connect( netList, SIGNAL(rightButtonPressed( QListViewItem*, const QPoint&, int )),
                 SLOT(showItemContextMenu( QListViewItem*, const QPoint&, int )) );

        /// Settings Widget
        connect( buttonOptions, SIGNAL(toggled(bool)),
                 this, SLOT(togglePage(bool)) );

        connect( buttonEnableAllMessages, SIGNAL(clicked()),
                 this, SLOT(enableAllMessages()) );

        /// Global KDE Options
        connect( KApplication::kApplication(), SIGNAL(settingsChanged(int)),
                 this, SLOT(updateConfiguration(int)) );

        setMouseBehaviour();

        QTimer::singleShot(10, this, SLOT(init()) ); //WAIT FOR THE UI TO BE READY BEFORE FURTHER SETUP (msec)
}

WirelessAssistant::~WirelessAssistant()
{}

/*$SPECIALIZATION$*/


void WirelessAssistant::init()
{
        statusLabel->setText(i18n("Initializing..."));
        statusLabel->repaint();

        ////////////////////////////////////////
        ///// CHECK FOR SYSFS (KERNEL 2.6) /////
        if ( !QFile::exists("/sys") ) {
                std::cout << "Sysfs not present. Exiting." << std::endl;
                KMessageBox::error( 0, i18n("Kernel 2.6 or later not present.\nWireless Assistant will now quit.") );
                close();
                return;
        }

        /////////////////////////////////////////////////////
        ///// LOAD CONFIG FILE INCL. ALL NET PARAMETERS /////
        WAConfig::self()->setCurrentGroup("Global Options");
        WAConfig::self()->addItemBool("Auto Quit", autoQuit);
        WAConfig::self()->addItemInt("DHCP Client Timeout", DhcpTimeout);
        WAConfig::self()->addItemString("Interface", NetParams.iface);

        WAConfig::self()->setCurrentGroup("Paths");
        // Commented out cos no longer needed. Paths are detected when necessary.
        /*WAConfig::self()->addItemString("DHCP Info (dhcpcd)", dhcpcdInfoPath);
        WAConfig::self()->addItemString("DHCP PID File (dhcpcd)", dhcpcdPidPath);
        WAConfig::self()->addItemString("DHCP Info (dhclient)", dhclientInfoPath);
        WAConfig::self()->addItemString("DHCP PID File (dhclient)", dhclientPidPath);*/

        WAConfig::self()->setCurrentGroup("Network Parameters");
        WAConfig::self()->addItemStringList("NetParamsList", NetParamsList );
        WAConfig::self()->readConfig();
        checkAutoQuit->setChecked(autoQuit);
        if (!DhcpTimeout)
                DhcpTimeout = spinDhcpTimeout->value();
        else
                spinDhcpTimeout->setValue(DhcpTimeout);
        std::cout << "Loaded application options." << std::endl;

        ///////////////////////////////////
        ///// DETECT WIRELESS DEVICES /////

        Commands.init();
        QStringList devList = interfaceList();
        if ( devList.count()==0 ) {
                std::cout << "No wireless interfaces found. Exiting." << std::endl;
                KMessageBox::error(0, i18n("No usable wireless devices found.\nWireless Assistant will now quit."));
                close();
                return;
        }
        std::cout << "Wireless interface(s): " << devList.join(", ") << std::endl;
        devCombo->insertStringList(devList);

        if (devCombo->count() > 1) { //check if last used (saved) interface is available (only if more that 1 interface present).
                for (int i=0; i<devCombo->count(); i++) {
                        if ( devCombo->text(i)==NetParams.iface ) { //select matching interface.
                                devCombo->setCurrentItem( i );
                                break;
                        }
                }
                frameDevice->show(); //only if more than 1 wireless device.
        }
        NetParams.iface = devCombo->currentText();	// set interface name
        WATools::setInterface( NetParams.iface );	// set fallback interface for WATools

        //////////////////////////////////
        ///// CHECK FILE PERMISSIONS /////
        if (!QFileInfo("/etc/resolv.conf").isWritable()) {
                std::cout << "warning: /etc/resolv.conf not writable" << std::endl;
                KMessageBox::information(0, i18n("<qt><p>You might have insufficient permissions for Wireless Assistant to function properly.</p><p>Did you run it using '<tt>sudo</tt>'?</p></qt>") );
        }
        std::cout << "Permissions checked." << std::endl;

        ///////////////////////////////////////
        ///// INITIALIZE GLOBAL VARIABLES /////
        connectedItem = 0;
        timerGui = new QTimer();
        timerConnectionCheck = new QTimer();
        timerFastConnectionCheck = new QTimer();
        connect( timerGui, SIGNAL(timeout()), SLOT(updateConnectedItem()) );
        connect( timerConnectionCheck, SIGNAL(timeout()), SLOT(checkConnectionStatus()) );
        connect( timerFastConnectionCheck, SIGNAL(timeout()), SLOT(checkConnectionStatus()) );

        ////////////////////////
        ///// DETECT PATHS /////
        if (!Commands.allFound) {	//all ok or ONLY dhcpcd not found (i.e. dhclient present).
                std::cout << "Missing executables (" << Commands.notFound.join("', '") << "). Exiting." << std::endl;
                KMessageBox::error(0, i18n("Executable(s) '%1' could not be found.\nWireless Assistant will now quit.").arg(Commands.notFound.join("', '")) );
                close();
                return;
        }

        ///////////////////////////////////////
        ///// SCAN FOR AVAILABLE NETWORKS /////
        netScan();
}

void WirelessAssistant::checkConnectionStatus()
{
        timerConnectionCheck->stop(); //in case check takes too long.
        if (connectedItem) {	//connectedItem exists
                //std::cout << "Checking connection status." << std::endl;
                bool isOK = isConnected( true );	// can internet be reached?
                if (isOK == connectedItem->isInternet()) {	// nothing changed - return;
                        timerConnectionCheck->start(10000); //resume interval checking.
                        return;
                }
                // status changed? Continue:
                connectedItem->setInternet( isOK );	// set isInternet accordingly
                if (!isOK) {	// internet cannot be reached?
                        std::cout << "Internet no longer accessible." << std::endl;
                        //netList->setSelected( connectedItem, true );
                        setNetParamsFromList( connectedItem );
                        if ( WATools::quality()==0 || !isConnected()) {	// if link quality=0 or cannot reach gateway -> connection is lost. Tell user:
                                setConnectedItemByAp( 0 );
                                if ( KMessageBox::questionYesNo(0, i18n("Connection to '%1' has been lost!\nWould you like to reconnect?").arg(NetParams.essid), i18n("Connection Lost") , KStdGuiItem::yes(), KStdGuiItem::no(), "AutoReconnect" ) == KMessageBox::Yes ) {
                                        netDisconnect( true );
                                        netConnect();
                                } // goes to end where timer is resumed.
                        }
                } else
                        std::cout << "Internet now accessible." << std::endl; // (isOK)

        } else {// no connectedItem
                std::cout << "Checking for active connection." << std::endl;
                if (WATools::ap()=="00:00:00:00:00:00")
                        return;
                if ( isConnected() ) {
                        std::cout << "Active connection found." << std::endl;
                        setConnectedItemByAp( WATools::ap() );
                        timerFastConnectionCheck->start(850, true); //don't wait 10s for automatic check, do it now.
                }
        }
        timerConnectionCheck->start(10000); //resume interval checking.
}

void WirelessAssistant::removeNetParams()
{
        NetListViewItem *nvi = static_cast<NetListViewItem*>(netList->selectedItem());
        QString ap = nvi->ap();
        for (QStringList::Iterator nps = NetParamsList.begin(); nps != NetParamsList.end(); nps++) {
                if ( (*nps).section(",",2,2)==ap ) {
                        if ( KMessageBox::warningContinueCancel(0, i18n("<qt><p>Settings for network '<b>%1</b>' are about to be deleted.</p><p>Would you like to continue?</p></qt>").arg(matchEssidForAp(ap))) == KMessageBox::Continue ) {
                                if (nvi->hidden())	// hiddenEssid = 1
                                        nvi->setEssid("<hidden>");
                                NetParamsList.remove(nps);
                                WAConfig::self()->writeConfig();
                                statusLabel->setText( i18n("Settings deleted.") );
                        }
                        break;
                }
        }
}


void WirelessAssistant::setDNS( const WANetParams & np )
{
        QFile f("/etc/resolv.conf");
        if (f.open( IO_WriteOnly | IO_Truncate )) {
                QTextStream s( &f );
                if (!np.domain.isEmpty()) {
                        s << QString("domain " + np.domain + "\n");
                        std::cout << "resolv.conf: domain " << np.domain << std::endl;
                }
                if (!np.dns1.isEmpty()) {
                        s << QString("nameserver " + np.dns1 + "\n");
                        std::cout << "resolv.conf: nameserver " << np.dns1 << std::endl;
                }
                if (!np.dns2.isEmpty()) {
                        s << QString("nameserver " + np.dns2 + "\n");
                        std::cout << "resolv.conf: nameserver " << np.dns2 << std::endl;
                }
                f.close();
        } else {
                std::cout << "dns setup error: " << f.name() << " is not writeable." << std::endl;
                KMessageBox::error(0, i18n("<qt><p>File '<i>%1</i>' could not be opened for writing.</p><p>Nameserver(s) and/or domain are not set.</p></qt>").arg(f.name()) );
        }
}

void WirelessAssistant::netScan()
{
        timerConnectionCheck->stop(); //stop while scanning.
        timerFastConnectionCheck->stop(); //if fast update scheduled - cancel it.
        netScan( NetParams );
        if (netList->childCount() > 0)
                timerConnectionCheck->start(10000); //check connection status every 10seconds
}

void WirelessAssistant::netScan( const WANetParams & np )
{
        if (!radioEnabled()) {
                statusLabel->setText("Radio off. Scanning aborted.");
                std::cout << "Radio is off!" << std::endl;
                setUi(1);
                return;
        }

        setUi(0);

        bool wasConnected = false;
        if (connectedItem) {
                wasConnected = true;
                setConnectedItemByAp( 0 );
        }

        if ( !(runCommand( Commands.cmd("ifconfig_status", np) ).find("UP ")>-1) ) {
                statusLabel->setText(i18n("Bringing interface %1 up...").arg(np.iface));
                runCommand( Commands.cmd("ifup",np) );
                usleep(150*1000); //sleep 150ms to make sure that scan results are ready.
        }

        statusLabel->setText(i18n("Scanning..."));
        statusLabel->repaint();

        netList->clear();

        QString result;
        statusLabel->setText(i18n("Scanning..."));
        result = runCommand( Commands.cmd("scan",np) );

        parseScan( result );

        if (netList->childCount() > 0) {
                std::cout << "Networks found: " << QString::number( netList->childCount() ) << std::endl;
                if (wasConnected)
                        setConnectedItemByAp( WATools::ap() );	//mark item as connected.
                timerFastConnectionCheck->start(850, true);	// check for active conneciton.
                statusLabel->setText( i18n("Done.") );
        } else {
                //Workaround for cards overusing cache - bringing if down seems to solve it.
                //runCommand( Commands.cmd("ifdown", NetParams) ); //Commented out b/c it seems to cause more problems than it solves. (like no scan results)
                std::cout << "No networks found!" << std::endl;
                statusLabel->setText( i18n("No networks found.") );
                if ( result.find("Resource temporarily unavailable")>-1 ) {
                        std::cout << "Radio switch seems to be off." << std::endl;
                        KMessageBox::information(0, i18n("Radio of your wireless card seems to be turned off using an external switch on your computer.\nYou need turn it on to be able to use wireless networks.") );
                }
        }
        setNetListColumns();
}

void WirelessAssistant::parseScan( const QString & output )
{
        QString essid;
        QString channel;
        QString mode;
        int qualInt;
        bool enc; //default to false
        bool hidden; //default to false
        QString ap;

        bool ok_channel = true;	//does iwlist return channel?
        QString section;

        netList->setUpdatesEnabled( false ); //do not redraw while adding items to avoid flicker.

        for (int i=1; (!output.section("Cell ",i,i).isEmpty()); i++ ) {
                section = output.section("Cell ",i,i);

                // GET ESSID VALUE
                essid = getVal(section, "ESSID\\W+\"(.+)\"");

                // GET CHANNEL NUMBER
                channel = getVal(section, "Channel\\D+(\\d+)" );
                if (channel.isEmpty()) {
                        channel = getVal(section, "Frequency\\D+(\\d.+)Hz");
                        ok_channel = false;
                }

                // GET MODE VALUE
                mode = getVal(section, "Mode:(\\w)"); //get 1st letter of mode.
                if (mode.upper()!="M") //this covers both Managed and Master. Other are unsupported.
                        continue;

                // GET AP
                ap = getVal(section, "Address\\W+(\\S+)");

                if (essid.isEmpty()) {
                        if (!ap.isEmpty()) //older wireless-tools report "", not "<hidden>"
                                essid = "<hidden>";
                        else
                                continue;	//some cards report one '' essid even when no network's present. Workaround.
                }

                if (essid=="<hidden>") {
                        hidden = true;
                        essid = matchEssidForAp( ap );
                } else
                        hidden=false;

                // GET QUALITY
                int wsignal;
                //check if driver reports quality directly
                qualInt = getVal(section, "Quality\\D+(\\d+)").toInt();

                if (qualInt == 0) { //noise not reported? estimate.
                        wsignal = getVal(section, "Signal level\\D+(\\d+)" ).toInt();
                        qualInt = 100-wsignal;
                }
                qualInt = (100*qualInt)/50; //adjust and normalize quality (0-100). 50 is the best (6 stars) noise/signal difference
        	if (qualInt > 100) qualInt = 100;


                // GET ENCRYPTION
                if (getVal(section, "Encryption key\\W+(\\w+)" ).upper()=="OFF")
                        enc = false;
                else
                        enc = true;

                new NetListViewItem( netList, essid, channel, qualInt, enc, ap, hidden );
        }

        if (!ok_channel)
                netList->setColumnText( 1, i18n("Freq (Hz)") );

        /// @fixme HACK: Test item for the network list.
        /// new NetListViewItem( netList, "Test Net", "9", 76, 1, "00:00:00:00:00:11", 0 );


        netList->setUpdatesEnabled( true );
        setUi(1);
}

bool WirelessAssistant::radioEnabled()
{
        bool r;
        if ( WATools::txpower()==-1 ) {
                if (KMessageBox::questionYesNo(0, i18n("Radio of your wireless card is off.\nWould you like to turn it on?") )== KMessageBox::Yes) {
                        runCommand( Commands.cmd("radio_on", NetParams) );
                        r = true;
                } else {
                        r = false;
                }
        } else
                r = true;

        return r;
}

void WirelessAssistant::setNetParamsFromList( QListViewItem* lvi )
{
        NetListViewItem *nvi = static_cast<NetListViewItem*>(lvi);
        NetParams.essid = nvi->essid();
        NetParams.hiddenEssid = nvi->hidden();
        //NetParams.mode = nvi->mode();
        NetParams.channel = nvi->channel();
        NetParams.ap = nvi->ap();
        NetParams.wep = nvi->enc();
}

bool WirelessAssistant::setNetParamsFromConfig( const QString & ap )
{
        for (QStringList::Iterator nps = NetParamsList.begin(); nps != NetParamsList.end(); nps++) {
                if ( (*nps).section(",",2,2)==ap ) {
                        NetParams.loadNetParamsString( *nps );
                        return 1;
                }
        }
        return 0;
}

void WirelessAssistant::itemAction()
{
        QListViewItem* lvi = netList->selectedItem();
        if (!lvi)
                return;

        NetListViewItem* nvi = static_cast<NetListViewItem*>(lvi);
        ///////////////////
        ///// ACTIONS /////
        if (nvi->isConnected()) {
                std::cout << "ACTION: DISCONNECT." << std::endl;
                netDisconnect();
                return;
        } else {
                std::cout << "ACTION: CONNECT." << std::endl;
                netConnect();
                return;
        }
}

void WirelessAssistant::netConnect()
{
        setNetParamsFromList( netList->selectedItem() );

        if ( (NetParams.essid=="<hidden>") || (!setNetParamsFromConfig( NetParams.ap )) ) {
                ui_NetParamsWizard *netwiz = new ui_NetParamsWizard;
                if (!NetParams.hiddenEssid)
                        netwiz->setCaption( i18n("%1 - First Connection Wizard").arg(NetParams.essid) );
                netwiz->setEssidEnabled( NetParams.hiddenEssid );
                netwiz->setWepEnabled( NetParams.wep );
                netwiz->exec();
                if (netwiz->result()==QDialog::Rejected) {
                        delete netwiz;
                        return;
                } else {
                        NetParams = netwiz->readNetParams( NetParams );
                        NetParams.wasHiddenEssid = NetParams.hiddenEssid; //first time values.
                        NetParams.wasWep = NetParams.wep;
                        NetParamsList << NetParams.netParamsString();
                        if (NetParams.hiddenEssid)
                                static_cast<NetListViewItem*>(netList->selectedItem())->setEssid( NetParams.essid );
                        WAConfig::self()->writeConfig();
                        delete netwiz;
                }
        }

        if (NetParams.review())
                editNetParams();
        updateNetParams();
        timerConnectionCheck->stop(); // stop while connecting.
        netConnect( NetParams );
        timerConnectionCheck->start(10000); //check connection status every 10seconds
}

void WirelessAssistant::updateNetParams()
{
        for (QStringList::Iterator nps = NetParamsList.begin(); nps != NetParamsList.end(); nps++) {
                if ( (*nps).section(",",2,2)==NetParams.ap ) {
                        QString newNps = NetParams.netParamsString();
                        if ( newNps!=(*nps) ) {
                                (*nps) = newNps;
                                WAConfig::self()->writeConfig();
                                std::cout << "Network settings updated." << std::endl;
                                statusLabel->setText( i18n("Network settings updated.") );
                                break;
                        }
                }
        }
}

QString WirelessAssistant::matchEssidForAp( const QString & ap )
{
        for (QStringList::Iterator nps = NetParamsList.begin(); nps != NetParamsList.end(); nps++) {
                if ( (*nps).section(",",2,2)==ap ) {
                        return (*nps).section(",",1,1); //essid
                }
        }
        return "<hidden>";
}

void WirelessAssistant::netConnect( const WANetParams & np )
{
        setUi(0);

        if (connectedItem)
                netDisconnect( true );
        else if ( dhcpClientRunning() )
                runCommand( Commands.cmd("kill_dhcp", np) ); //kill any stale DHCP client running

        statusLabel->setText( i18n("Connecting to '%1'...").arg(np.essid) );
        statusLabel->repaint();
        runCommand( Commands.cmd("ifup", np) );
        runCommand( Commands.cmd("iwconfig_set", np) );
        runCommand( Commands.cmd("iwconfig_ap", np) );
        if (np.dhcp) { //DHCP config
                QString dhcp_out = runCommand( Commands.cmd("ifconfig_dhcp", np), DhcpTimeout );
                if (dhcp_out.find("::ERR::")>-1) {
                        if ( dhcpClientRunning() )
                                runCommand( Commands.cmd("kill_dhcp", np) ); //kill any stale DHCP client running (seems it's dhclient only)
                        setUi(1);
                        std::cout << "CONNECTION FAILED." << std::endl;
                        statusLabel->setText( i18n("Connection failed.") );
                        return;
                }
        } else { //manual config
                runCommand( Commands.cmd("ifconfig_manual", np) );
                setDNS( np );
                runCommand( Commands.cmd("route", np) );
        }
        //////////////////////
        ///// CHECK CONNECTION
        //usleep(150*1000);	//sleep 150ms to make sure DHCP client sets all parameters.
        statusLabel->setText(i18n("Testing connection..."));
        statusLabel->repaint();
        if (isConnected()) {	//connected to gateway?
                if (autoQuit)
                        this->close();
                setConnectedItemByAp( np.ap );
                statusLabel->setText( i18n("Successfully connected to '%1'.").arg(np.essid) );
                setUi(1);
                timerFastConnectionCheck->start(850, true);
                //QTimer::singleShot(850, this, SLOT(checkConnectionStatus()) ); //check for internet in quickly, don't wait 10s for first automatic check.
        } else {
                statusLabel->setText(i18n("Connection failed."));
                setConnectedItemByAp( 0 );
                setUi(1);
                if (KMessageBox::questionYesNo(0, i18n("Connection failed.\nWould you like to review settings for this network?"), i18n("Review Settings?") , KStdGuiItem::yes(), KStdGuiItem::no(), "ReviewSettings" ) == KMessageBox::Yes)
                        editNetParams();
        }
}

void WirelessAssistant::updateConnectedItem()
{
        connectedItem->setQuality( WATools::quality() );
}

void WirelessAssistant::setConnectedItemByAp( const QString & ap )
{
        timerConnectionCheck->stop(); //stop timer while changing currentItem
        if (connectedItem) {
                timerGui->stop();
                connectedItem->setConnected( false );
                connectedItem = 0;
        }
        if (!ap.isEmpty()) {
                QListViewItem* lvi = getItemByAp( ap );
                if (lvi) {
                        if (WATools::quality()==0)
                                return;	// don't set as connected if link quality is 0
                        NetListViewItem* nvi = static_cast<NetListViewItem*>(lvi);
                        nvi->setConnected( true );
                        connectedItem = nvi;
                        netList->sort();	// sort to make sure new connectedItem is 1st.
                }
        }

        if (connectedItem) {
                timerGui->start(2500); //update quality indicator every 2.5seconds
        }
        updateConnectButton( netList->selectedItem() );
        timerConnectionCheck->start(10000); //check connection status every 10seconds
}

void WirelessAssistant::netDisconnect( const bool & quiet )
{
        if ( (quiet) || (KMessageBox::warningContinueCancel(0, i18n("<qt><p>You are about to disconnect from '<b>%1</b>'.</p><p>Would you like to continue?<p></qt>").arg(connectedItem->essid()) )== KMessageBox::Continue ) ) {
                timerConnectionCheck->stop(); //stop while disconnecting.
                timerFastConnectionCheck->stop(); //if update scheduled - cancel it.
                statusLabel->setText( i18n("Disconnecting...") );
                statusLabel->repaint();
                setConnectedItemByAp( 0 );
                if ( dhcpClientRunning() ) {
                        runCommand( Commands.cmd( "kill_dhcp", NetParams ) );
                        statusLabel->setText( i18n("Waiting for DHCP client to shut down...") );
                        statusLabel->repaint();
                        QTimer* tmr = new QTimer();
                        tmr->start(1500, true);	//wait 1.5sec for dhcp client to really shutdown, single shot.
                        while ( tmr->isActive() ) {
                                KApplication::eventLoop()->processEvents( QEventLoop::AllEvents );
                                usleep(75*1000); //75msec on Linux
                        }
                        delete tmr;
                }
                runCommand( Commands.cmd( "disconnect", NetParams ) );
                runCommand( Commands.cmd( "ifdown", NetParams ) );

                std::cout << "DISCONNECTED." << std::endl;
                statusLabel->setText( i18n("Done.") );
                timerConnectionCheck->start(10000); //check connection status every 10seconds
        } else {
                statusLabel->setText( i18n("Cancelled.") );
        }
}

QListViewItem* WirelessAssistant::getItemByAp( const QString & ap )
{
        QListViewItem* lvi = netList->firstChild();
        while (lvi) {
                if ( static_cast<NetListViewItem*>(lvi)->
                                ap() == ap ) {
                        break;
                }
                lvi = lvi->nextSibling();
        }
        return lvi;
}

void WirelessAssistant::updateConfiguration(int category)
{
        if (category == KApplication::SETTINGS_MOUSE) {
                setMouseBehaviour();
                return;
        }
        if (category == -1) {
                autoQuit = checkAutoQuit->isChecked();
                DhcpTimeout = spinDhcpTimeout->value();
        }
}

void WirelessAssistant::togglePage(bool options)
{
        buttonScan->setDisabled(options);
        buttonConnect->setDisabled(options);
        if (options) {
                if (WAConfig::self()->config()->groupList().contains("Notification Messages")>0)
                        buttonEnableAllMessages->setEnabled(true);
                else
                        buttonEnableAllMessages->setEnabled(false);
                widgetStack->raiseWidget(optionsPage);
        } else {
                widgetStack->raiseWidget(netPage);
                updateConfiguration(-1);
        }
}

void WirelessAssistant::enableAllMessages()
{
        KMessageBox::enableAllMessages();
        buttonEnableAllMessages->setEnabled( false );
}

void WirelessAssistant::setMouseBehaviour()
{
        if ( KGlobalSettings::singleClick() ) {
                disconnect( netList, SIGNAL(selectionChanged(QListViewItem*)),
                            this, SLOT(updateConnectButton(QListViewItem*)) );
                disconnect( netList, SIGNAL(doubleClicked(QListViewItem*, const QPoint &, int)),
                            this, SLOT(itemAction()) );
                connect( netList, SIGNAL(clicked(QListViewItem*, const QPoint &, int)),
                         this, SLOT(itemAction()) );
                buttonConnect->hide();
        } else {
                disconnect( netList, SIGNAL(clicked(QListViewItem*, const QPoint &, int)),
                            this, SLOT(itemAction()) );

                connect( netList, SIGNAL(selectionChanged(QListViewItem*)),
                         this, SLOT(updateConnectButton(QListViewItem*)) );
                connect( netList, SIGNAL(doubleClicked(QListViewItem*, const QPoint &, int)),
                         this, SLOT(itemAction()) );
                buttonConnect->show();
        }
}

void WirelessAssistant::updateConnectButton(QListViewItem* lvi)
{
        QToolTip::remove
                (buttonConnect);
        if ( lvi == connectedItem ) {
                buttonConnect->setText( i18n("&Disconnect") );
                QToolTip::add
                        ( buttonConnect, i18n("Disconnect from the selected network") );

        } else {
                buttonConnect->setText( i18n("&Connect") );
                QToolTip::add
                        ( buttonConnect, i18n("Connect to the selected network") );

        }
}

void WirelessAssistant::setDev( const QString & ifname)
{
        NetParams.iface = ifname;
        WATools::setInterface( ifname );
        std::cout << "Selected interface: " << ifname << std::endl;
        netScan();
}

QString WirelessAssistant::runCommand( const QStringList & cmd, int timeout )
{
        QTimer* timerProc = new QTimer(); //timeout timer
        if (cmd.isEmpty())
                return QString::null;

        QProcess* p = new QProcess( this );
        p->setArguments( cmd );

        if (timeout>0) {
                connect( timerProc, SIGNAL(timeout()), p, SLOT(kill()) );
                timerProc->start(timeout*1000); //convert sec to msec
        }

        connect(buttonClose, SIGNAL(clicked()),
                p, SLOT(kill()) );

        int i = 0;
        p->start();
        while ( p->isRunning() ) { // PROCESS USER EVENTS
                KApplication::eventLoop()->processEvents( QEventLoop::AllEvents );
                usleep(75*1000); //75msec on Linux (75000msec on Windows...)
                if (i==27) { // ca 2sec have passed and the process is still running. Replace the 'Close' button with 'Stop'.
                        disconnect(buttonClose, SIGNAL(clicked()),
                                   this, SLOT(close()) );
                        buttonClose->setIconSet( SmallIconSet("stop") );
                        buttonClose->setText( i18n("&Stop") );
                        QToolTip::remove
                                (buttonClose);
                        QToolTip::add
                                ( buttonClose, i18n("Terminate current process\n(%1)").arg( p->arguments().join(" ") ) );
                }
                i++;
        }

        disconnect(buttonClose, SIGNAL(clicked()),
                   p, SLOT(kill()) );
        if (i>27) {//set 'stop' back to 'close' if needed
                connect(buttonClose, SIGNAL(clicked()),
                        this, SLOT(close()) );
                buttonClose->setIconSet( SmallIconSet("fileclose") );
                buttonClose->setText( i18n("&Quit") );
                QToolTip::remove
                        (buttonClose);
                QToolTip::add
                        ( buttonClose, i18n("Quit the application") );
        }

        if (timerProc->isActive())
                timerProc->stop();
        delete timerProc;
        QString e = QString( p->readStderr() );
        QString o = QString( p->readStdout() );
        if (!p->normalExit()) {
                o.append("::ERR::killed");
                //std::cout << "Process terminated (timed out)." << std::endl; //too much output when checking for internet when it's not available.
        }
        delete p;
        if (!e.isEmpty()) {
                std::cout << "==>stderr: " << e;// << std::endl;
                o.append("::ERR::");
                o.append(e);
        }

        return o;
}

void WirelessAssistant::setUi(int uiState)
{

        if (uiState==0) {
                devCombo->setEnabled( false );
                buttonScan->setEnabled( false );
                buttonConnect->setEnabled( false );
                buttonOptions->setEnabled( false );
                KApplication::setOverrideCursor( QCursor(Qt::BusyCursor) );
        } else {
                if (devCombo->count() > 0) {
                        devCombo->setEnabled( true );
                        buttonScan->setEnabled( true );
                }
                if (netList->childCount() > 0)
                        buttonConnect->setEnabled( true );
                buttonOptions->setEnabled( true );
                KApplication::restoreOverrideCursor();
        }
}

void WirelessAssistant::showItemContextMenu( QListViewItem* i, const QPoint& p, int c )
{
        if (!i)
                return;

        NetListViewItem *nvi = static_cast<NetListViewItem*>(i);
        QString ap = nvi->ap();
        bool isConfigured = setNetParamsFromConfig(ap);
        KPopupMenu *icm = new KPopupMenu();
        icm->insertTitle(nvi->essid());
        if (isConfigured) {
                if (nvi->isConnected()) {
                        icm->insertItem( SmallIcon("connect_no"), i18n("Disconnect..."), this, SLOT(netDisconnect()) );
                        //icm->insertItem( SmallIcon("reload"), i18n("Reconnect"), this, SLOT(netConnect()) );
                } else
                        icm->insertItem( SmallIcon("connect_creating"), i18n("Connect"), this, SLOT(netConnect()) );
                icm->insertSeparator();
                icm->insertItem(i18n("Forget Settings..."), this, SLOT(removeNetParams()) );
                icm->insertItem(i18n("Edit Settings..."), this, SLOT(editNetParams()) );
        } else {
                if (nvi->isConnected()) {
                        icm->insertItem( SmallIcon("connect_no"), i18n("Disconnect..."), this, SLOT(netDisconnect()) );
                        //icm->insertItem( SmallIcon("reload"), i18n("Configure and Reconnect..."), this, SLOT(netConnect()) );
                } else
                        icm->insertItem( SmallIcon("connect_creating"), i18n("Configure and Connect..."), this, SLOT(netConnect()) );
        }
        icm->exec( QCursor::pos() );
}

void WirelessAssistant::editNetParams()
{
        setNetParamsFromList( netList->selectedItem() );	//prepate NetParams
        setNetParamsFromConfig( NetParams.ap );	//prepate NetParams

        ui_NetParamsEdit *netedit = new ui_NetParamsEdit();
        netedit->setValues( NetParams );
        netedit->setCaption( i18n("%1 Settings").arg(NetParams.essid) );
        netedit->exec();
        if (netedit->result() == QDialog::Rejected) {
                delete netedit;
                return;
        } else {	//accepted
                NetParams = netedit->readNetParams( NetParams );
                updateNetParams();
        }
}

void WirelessAssistant::setNetListColumns()
{
        int realWidth = netList->viewportSize( netList->contentsWidth(), netList->contentsHeight() ).width(); //calculate actual width taking scrollbars into account
        int essidWidth = realWidth - netList->columnWidth(1) - netList->columnWidth(2) - netList->columnWidth(3);

        netList->setColumnWidth(0, essidWidth);
        netList->triggerUpdate();
}

bool WirelessAssistant::dhcpClientRunning()
{
        QStringList pidPaths;
        QString pidFile;
        pidPaths << "/etc/dhcpc/" << "/var/run/";
        if ( Commands.dhcpClient=="dhcpcd" )
                pidFile = QString("dhcpcd-%1.pid").arg(NetParams.iface);
        else
                pidFile = QString("dhclient.pid");

        for ( QStringList::Iterator it = pidPaths.begin(); it != pidPaths.end(); ++it ) {
                if ( QFile( QString(*it).append(pidFile) ).exists() )
                        std::cout << "Running DHCP client found." << std::endl;
                return true;
        }
        std::cout << "No DHCP client running." << std::endl;
        return false;
}

QString WirelessAssistant::getGateway()
{
        std::cout << "Trying to get gateway address..." << std::endl;

        QString fs, rxs, gw;
        QStringList leasePaths;
        leasePaths << "/etc/dhcpc/" << "/var/state/dhcp/" << "/var/lib/dhcpc/" << "/var/lib/dhcp3/" << "/var/lib/dhcp/";

        if ( dhcpClientRunning() ) {
                if ( Commands.dhcpClient=="dhcpcd" ) { //dhcpcd
                        fs = QString("dhcpcd-%1.info").arg(NetParams.iface);
                        rxs = "GATEWAY=(\\d+\\.\\d+\\.\\d+\\.\\d+)";
                } else { //dhclient
                        fs = "dhclient.leases";
                        rxs = "option routers\\W+(\\d+\\.\\d+\\.\\d+\\.\\d+)";
                }

                QFile f;
                for ( QStringList::Iterator it = leasePaths.begin(); it != leasePaths.end(); ++it ) {
                        f.setName( QString(*it).append(fs) );
                        if ( f.exists() && f.open( IO_ReadOnly ) ) {
                                std::cout << "...from " << f.name() << std::endl;
                                QString fout = f.readAll();
                                if (Commands.dhcpClient=="dhclient") { //get only the last lease in the file.
                                        fout = fout.right( fout.length() - fout.findRev( QString("interface \"%1\"").arg(NetParams.iface)) );
                                }
                                gw = getVal( fout, rxs );
                                f.close();
				break;
                        }
                }
        }
        if ( gw.isEmpty() ) { /// no DHCP client running or no gateway info available - try to get gw address from 'route'.
                std::cout << "...from 'route'" << std::endl;
                QString rt = runCommand( Commands.route );
                gw = getVal( rt, QString("default\\W+(\\d+\\.\\d+\\.\\d+\\.\\d+)\\W+.*%1").arg(NetParams.iface) );
        }
        return gw;
}

bool WirelessAssistant::isConnected( const bool & internet )
{
        QString result;
        if (internet)
                result = runCommand( Commands.cmd( "ping_internet", NetParams, true ), 3 ); //3sec timeout for pinging internet
        else {
                if ( NetParams.gateway.isEmpty() ) { // gateway field is empty?
                        NetParams.gateway = getGateway();
                        if ( NetParams.gateway.isEmpty() ) { //no gateway found?
                                std::cout << "No default gateway." << std::endl;
                                return false;
                        }
                        std::cout << "Gateway address: " << NetParams.gateway << std::endl;
                }
                result = runCommand( Commands.cmd( "ping_gw", NetParams, true ), 2 ); //2sec timeout for pinging gateway (true = quiet)
        }
        if ( (result.find("::ERR::")>-1) || (result.find("unknown host")>-1) || (result.find("unreachable")>-1) || (result.find("100%")>-1) )
                return false;
        else
                return true;
}

QStringList WirelessAssistant::interfaceList()
{
	WANetParams empty;
	QString o = runCommand( Commands.cmd( "iwconfig_status", empty ) );
	QStringList ol = QStringList::split("\n\n", o);
	int c = ol.count();
	QStringList ifList = QStringList();
	for (int i = 0; i < c; i++) {
		o = ol[i];
		if ( o.find("no wireless extensions")==-1 ) //string not found
			ifList << o.left(o.find("  "));
	}
	return ifList;
}

QString WirelessAssistant::getVal(const QString & str, const QString & rxs)
{
        QRegExp rx(rxs);
        rx.search(str);
        return rx.cap(1).stripWhiteSpace();
}

bool WirelessAssistant::close()
{
        updateConfiguration(-1);	//read values from setingsPage;
        WAConfig::self()->writeConfig();
        std::cout << "Application options saved." << std::endl;
        WATools::cleanup();
        std::cout << "Kernel socket closed." << std::endl;
        return QWidget::close();
}


#include "wlassistant.moc"
