#!/usr/bin/perl
#
# $ID: whereami.pl,v 1.2 2001/11/23 12:18:14 andrew Exp $
#
# by Andrew McMillan, Catalyst IT Ltd, (c) 2001 licensed
# for use under the Gnu GPL version 2
#

use strict;

my $debug = 0;
my $syslogging = 0;
my $noactions = 0;
my $locking = 1;
my $basedir = "/etc/whereami";
my $lockdir = "/var/run";
my $scriptbase = "/usr/share/whereami";
my $caller   = "";
my $run_from = "";
my $default = "none";
my $last_locations = "";
my $now_locations = "";
my $hint_locations = "";
my %whereiam;
my %whereiwas;
$ENV{'PATH'} = "$scriptbase/tests:$scriptbase:/bin:/usr/bin:/sbin:/usr/sbin:$basedir/tests";


# Argument parsing.
for ( my $i=0; $i<=$#ARGV; $i++ ) {
  print "$ARGV[$i]\n" if ( $debug );
  if ($ARGV[$i] eq "--debug") {
    $debug = 1;
    $ENV{'DEBUGWHEREAMI'} = "1";
  }
  elsif ($ARGV[$i] eq "--scriptdebug" ) {
    $ENV{'DEBUGWHEREAMI'} = "1";
  }
  elsif ($ARGV[$i] eq "--noactions" ) {
    $noactions = 1;
  }
  elsif ($ARGV[$i] eq "--nolocking" ) {
    $locking = 0;
  }
  elsif ($ARGV[$i] eq "-h" || $ARGV[$i] eq "--help" ) {
    print "
whereami [options] [location]

Options are:
    --debug       Turn on debugging
    --scriptdebug  Turn on debugging in all test scripts, but run normally
    --noactions   Don't actually do anything
    --nolocking   Allow multiple copies to run concurrently
    --basedir     Specify a different directory for the configuration files
    --from        Specify the location we are moving from
    --run_from    Indicate what program is calling us now
    --syslog      Log actions to syslog

Whereami will detect the location your computer is currently in, according to
/etc/whereami/detect.conf, and then flexibly reconfigure everything through
a script configured in /etc/whereami/whereami.conf.

";
    exit 0;
  }
  elsif ($ARGV[$i] eq "-b" || $ARGV[$i] eq "--basedir" ) {
    $basedir = "$ARGV[++$i]";
  }
  elsif ($ARGV[$i] eq "-f" || $ARGV[$i] eq "--from" ) {
    $last_locations = "$ARGV[++$i]";
  }
  elsif ($ARGV[$i] eq "-r" || $ARGV[$i] eq "--run_from" ) {
    $caller = "$ARGV[++$i]";
  }
  elsif ( $ARGV[$i] eq "--hint" ) {
    $hint_locations = "$ARGV[++$i]";
  }
  elsif ($ARGV[$i] eq "-s" || $ARGV[$i] eq "--syslog" ) {
    $syslogging = 1; # "$ARGV[++$i]";
    use Sys::Syslog;
    openlog( 'whereami', 'pid', 'user');
  }
  elsif ( $ARGV[$i] =~ /^-/ ) {
    print "Unknown option \"$ARGV[$i]\"\n";
  }
  else {
    $now_locations = $ARGV[$i];
  }
}
my $detectconf = "$basedir/detect.conf";
my $locateconf = "$basedir/whereami.conf";

if ( $locking && -f "$lockdir/whereami.started" ) {
  # Could do checks for age of lock here.  Say 60 seconds then smash it.
  syslog "warning", "It seems that another whereami is already running.  Exiting to avoid conflict." if ( $syslogging );
  die "whereami is already running";
}
open ( STARTTIME, "> $lockdir/whereami.started" ) || die "Can't open timestamp file";
close( STARTTIME );

print "BaseDir: $basedir\n" if ( $debug );
# Set the BASEDIR and LOCKDIR environment for scripts to use
$ENV{'BASEDIR'} =$basedir;
$ENV{'LOCKDIR'} =$lockdir;

sub detect_location {
  open( DETECT, "< $detectconf" ) || die "Can't open detect configuration file: $detectconf";
  my $state = "looking";
  my $startstate = "looking";
  my $dhcp = "";
  my $lno=0;
  my @splitup;
  my $testname;
  my $params;
  my $locations;

  print "Line State     : Script\n" if ( $debug );
  print "---- ----------:----------------------------------------------------\n" if ( $debug );

  syslog "debug", "Line State     : Script\n" if ( $syslogging );
  syslog "debug", "---- ----------:----------------------------------------------------\n" if ( $syslogging );

  while( <DETECT> ) {
    $lno++;
    chomp;
    next if ( /^\s*#/ );
    next if ( /^\s*$/ );
    $startstate = $state;

    # Trim leading blanks
    s/^\s+//;

    # The following three blocks implement the 'if ... [else] ... fi' functionality
    # for the config file.  Note that the [else] is optional, but we can't handle
    # nesting at this stage.
    /^else/i && do {
      # Note the test for 'success', in case the 'if' clause was successful. That
      # would imply that 'if' was true before that, and that we should still skip
      # the else clause contents...
      if ( $state =~ /^(if)|(success)/ ) {
        $state = "!else";
      }
      elsif ( $state eq "!if" ) {
        $state = "else";
      }
      else {
        die "$detectconf ($lno): 'else' without prior 'if'";
      }
      next;
    };

    # Deal to a 'fi ...' line
    /^fi/i && do {
      if ( $state =~ /^!?(if)|(else)|(success)/ ) {
        $state = "looking";
      }
      else {
        die "$detectconf ($lno): 'fi' without prior 'if' or 'else'";
      }
      next;
    };

    # Deal to an 'if location[,...]' line
    /^if (.*)$/i && do {
      if ( $state =~ /^!?(if)|(else)/ ) {
        die "$detectconf ($lno): 'if' may not be nested at this time";
      }
      $state = "!if";
      foreach( split( /[\s,]/ , $1) ) {
        if ( exists( $whereiam{$_} ) ) {
          $state = "if";
          last;
        }
      }
      next;
    };

    # So we just go to the next line if we are inside a non-processing chunk
    next if ( $state =~ /^[\!#]/ );
    # Skip to the next if we have been successful and we don't always do this.
    next if ( $state =~ /^success/ && $_ !~ /^always/i );

    # Shouldn't really have any whitespace lines here, but ignore anyway
    next if ( /^\s*$/ );
    # Skip to the next if this is a comment
    next if ( /^\s*#/ );

    printf ( "%#4d %-10.10s %s\n", $lno, $state, $_) if ( $debug );
    syslog ( "debug",  "%#4d %-10.10s: %s\n", $lno, $state, $_ ) if ( $syslogging );

    # Strip the always, if it is there.
    s/^always\s//i;

    ($testname, $params, $locations) = split(/\s+/, $_, 3);

    # In future we might implement some of the commoner tests directly
    # in here.  Especially DHCP, where we can do one pass to resolve
    # all possibilities...
    # For now we will run a script for each test we want to be available
    # which makes things more easily extensible, especially by end users.
    # The script needs to take one parameter, and give a zero exit
    # result on success.
    if ( $testname =~ /^set$/i ) {
      # The "set name value" syntax doesn't change the processing state
      $ENV{$params} = $locations;
    }
    elsif ( $testname =~ /^default$/i ) {
      # The "default name" syntax doesn't change the processing state
      $default = $params;
    }
    elsif ( 0 eq system( "$testname $params" ) ) {
      $state = "success";
      printf ( "****************> Test successful - adding locations: %s\n", $locations) if ( $debug );
      syslog ( "notice", "****************> Test successful - adding locations: %s\n", $locations) if ( $syslogging );

      foreach( split( /[\s,]+/, $locations ) ) {
        printf ( "*******location: %s\n", $_) if ( $debug );
        $whereiam{$_} = "1";
      }
    }
  }

  $now_locations = "";
  my ($k, $v);
  while( ($k, $v) = each( %whereiam ) ) {
    print "$k\n" if( $debug );
    $now_locations .= $k . ",";
  }
  chop $now_locations;
  if ( "$now_locations" eq "" ) {
    $now_locations = $default;
    $whereiam{$default} = 1;
  }
}
# end sub "detect_location"



sub process_location {

  my ($k, $v);
  my ( $condition, $script_line );
  my $found;
  #
  #
  if ( $debug ) {
    open ( WHEREIAM, "> ./whereiam.sh" ) || die "Can't open \"$basedir/whereiam.sh\" for output: ";
  }
  else {
    open ( WHEREIAM, "> $basedir/whereiam.sh" ) || die "Can't open \"$basedir/whereiam.sh\" for output: ";
  }
  open ( LOCN_CONFIG, "< $locateconf" ) || die "Can't open \"$locateconf\" for input: ";
  printf ( WHEREIAM "#!/bin/sh\n[ \"\$DEBUGWHEREAMI\" = \"1\" ] && set -o xtrace\nLASTLOCN=%s\nLOCATION=%s\n\n", $last_locations, $now_locations );
  my $start_with = "+";
  if ( $now_locations eq $last_locations ) {
    printf( "Continuing at %s\n", $now_locations);
    syslog( "notice", "Continuing at %s\n", $now_locations) if ( $syslogging ) ;
    $start_with = "="
  }
  else {
    printf( "Moving from %s to %s\n", $last_locations, $now_locations);
    syslog ( "notice", "Moving from %s to %s", $last_locations, $now_locations ) if ( $syslogging );
  }
  while ( <LOCN_CONFIG> ) {
    /^\s*$/ && next;
    /^\s*#/ && next;
    s/^\s+//;
    ($condition, $script_line) = split( /\s+/, $_, 2);
    if ( $start_with eq "=" ) {
      $found = 0;
      while( ($k, $v) = each( %whereiam ) ) {
        next if ( $found );
        $condition  =~ /=$k$/ && do {
          print "=>$condition<=|=>$k<|>$script_line" if ( $debug );
          print WHEREIAM "$script_line";
          syslog( "debug", "$condition $script_line") if ( $syslogging );
          $found = 1;
        };
      }
    }
    else {
      $found = 0;
      while( ($k, $v) = each( %whereiwas ) ) {
        next if ( $found );
        $condition =~ /-$k$/ && do {
          print "->$condition<-|->$k<|>$script_line" if ( $debug );
          print WHEREIAM "$script_line";
          syslog( "debug", "$condition $script_line") if ( $syslogging );
          $found = 1;
        };
      }
      $found = 0;
      while( ($k, $v) = each( %whereiam ) ) {
        next if ( $found );
        $condition =~ /[+=]$k$/ && do {
          print "+>$condition<+|+>$k<|>$script_line" if ( $debug );
          print WHEREIAM "$script_line";
          syslog( "debug", "$condition $script_line") if ( $syslogging );
          $found = 1;
        };
      }
    }
  }
  close WHEREIAM;
  close LOCN_CONFIG;
}
# end sub "process_location"



if ( "" eq "$last_locations" ) {
  # Read where we last were from where we wrote it last time
  open ( LASTLOCN, "< $basedir/iam" ) && do {
    while ( <LASTLOCN> ) {
      chomp;
      $whereiwas{$_} = 1;
      $last_locations .= $_ . ",";
    }
    close LASTLOCN;
    chop $last_locations;
  };
}

print "Last: $last_locations\n" if ( $debug );

if ( "" eq "$now_locations" ) {
  if ( "" ne "$hint_locations" ) {
    # Add any hint locations into the starting mix
    foreach( split( /[\s,]+/, $hint_locations ) ) {
      printf ( "*hint**location: %s\n", $_) if ( $debug );
      $whereiam{$_} = "1";
    }
  }
  detect_location();
}
else {
  foreach( split( /[\s,]+/, $now_locations ) ) {
    printf ( "*******location: %s\n", $_) if ( $debug );
    $whereiam{$_} = "1";
  }
}


if ( $now_locations ne $last_locations ) {
  #
  # Rotate saved 'where we were' files.
  for ( my $i=5; $i >= 0; $i-- ) {
    rename "$basedir/iwas.$i", "$basedir/iwas." . ($i + 1);
  }
  rename "$basedir/iwas", "$basedir/iwas.0";
  rename "$basedir/iam", "$basedir/iwas";
  #
  # Write file of these locations for next time...
  open ( LASTLOCN, "> $basedir/iam" ) && do {
    while( my ($k, $v) = each( %whereiam ) ) {
      print LASTLOCN "$k\n";
    }
    close LASTLOCN;
  };
}


if ( $noactions ) {
  my $k; my $v;
  while( ($k, $v) = each( %whereiam ) ) {
    print "$k\n";
  }
}
else {
  #
  $whereiam{'any'} = 1;
  process_location("$basedir/whereami.conf", "$basedir/whereiam.sh");

  if ( $debug ) {
    system( "cat $basedir/whereiam.sh" );
  }
  else {
    system( "sh -c \". $basedir/whereiam.sh\"" ) if ( ! $noactions );
  }
}

# Delete our locking file / timestamp
unlink( "$lockdir/whereami.started" );

closelog if ( $syslogging);

exit 0;
