/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/* This icmp code is largely inspired from iptables, by Paul "Rusty" Russell.
   Of course, he would be the one to blame for potential bugs. ;-) */

using namespace std;

#include <iostream>

#include "wfprotocol.h"
#include "list1.h"
#include "defs.h"

struct icmp_names {
  const char* name;
  uint8_t type;
  uint8_t code;
  /* flag is:
     0    if type without code
     0xFF if type with codes
     1    if code
     2    if alias
  */
  uint8_t flag;
};

static const struct icmp_names icmp_codes[] = {
  { "echo-reply", 0, 0, 0 },	/* ICMP_ECHOREPLY */
  /* Alias */ { "pong", 0, 0, 2 },

  { "destination-unreachable", 3, 0, 0xFF },	/* ICMP_DEST_UNREACH */
  {   "network-unreachable", 3, 0, 1 },	/* code: ICMP_NET_UNREACH */
  {   "host-unreachable", 3, 1, 1 },	/* code: ICMP_HOST_UNREACH */
  {   "protocol-unreachable", 3, 2, 1 },	/* code: ICMP_PROT_UNREACH */
  {   "port-unreachable", 3, 3, 1 },	/* code: ICMP_PORT_UNREACH */
  {   "fragmentation-needed", 3, 4, 1 },	/* code: ICMP_FRAG_NEEDED */
  {   "source-route-failed", 3, 5, 1 },	/* code: ICMP_SR_FAILED */
  {   "network-unknown", 3, 6, 1 },	/* code: ICMP_NET_UNKNOWN */
  {   "host-unknown", 3, 7, 1 },	/* code: ICMP_HOST_UNKNOWN */
  {   "host-isolated", 3, 8, 1 },	/* code: ICMP_HOST_ISOLATED */
  {   "network-prohibited", 3, 9, 1 },	/* code: ICMP_NET_ANO */
  {   "host-prohibited", 3, 10, 1 },	/* code: ICMP_HOST_ANO */
  {   "TOS-network-unreachable", 3, 11, 1 },	/* code: ICMP_NET_UNR_TOS */
  {   "TOS-host-unreachable", 3, 12, 1 },	/* code: ICMP_HOST_UNR_TOS */
  {   "communication-prohibited", 3, 13, 1 },	/* code: ICMP_PKT_FILTERED */
  {   "host-precedence-violation", 3, 14, 1 },	/* code: ICMP_PREC_VIOLATION */
  {   "precedence-cutoff", 3, 15, 1 },	/* code: ICMP_PREC_CUTOFF */

  { "source-quench", 4, 0, 0 },	/* ICMP_SOURCE_QUENCH */

  { "redirect", 5, 0, 0xFF },	/* ICMP_REDIRECT */
  {   "network-redirect", 5, 0, 1 },	/* code: ICMP_REDIR_NET */
  {   "host-redirect", 5, 1, 1 },	/* code: ICMP_REDIR_HOST */
  {   "TOS-network-redirect", 5, 2, 1 },	/* code: ICMP_REDIR_NETTOS */
  {   "TOS-host-redirect", 5, 3, 1 },	/* code: ICMP_REDIR_HOSTTOS */

  { "echo-request", 8, 0, 0 },	/* ICMP_ECHO */
  /* Alias */ { "ping", 8, 0, 2 },

  { "router-advertisement", 9, 0, 0 },	/* ... */

  { "router-solicitation", 10, 0, 0 },	/* ... */

  { "time-exceeded", 11, 0, 0xFF },	/* ICMP_TIME_EXCEEDED */
  /* Alias */ { "ttl-exceeded", 11, 0, 2 },
  {   "ttl-zero-during-transit", 11, 0, 1 },	/* code: ICMP_EXC_TTL */
  {   "ttl-zero-during-reassembly", 11, 1, 1 },	/* code: ICMP_EXC_FRAGTIME */

  { "parameter-problem", 12, 0, 0xFF },	/* ICMP_PARAMETERPROB */
  {   "ip-header-bad", 12, 0, 1 },	/* ... */
  {   "required-option-missing", 12, 1, 1 },	/* ... */

  { "timestamp-request", 13, 0, 0 },	/* ICMP_TIMESTAMP */

  { "timestamp-reply", 14, 0, 0 },	/* ICMP_TIMESTAMPREPLY */

  { "info-request", 15, 0, 0 },	/* ICMP_INFO_REQUEST */

  { "info-reply", 16, 0, 0 },	/* ICMP_INFO_REPLY */

  { "address-mask-request", 17, 0, 0 },	/* ICMP_ADDRESS */

  { "address-mask-reply", 18, 0, 0 }	/* ICMP_ADDRESSREPLY */
};


wf_protocol_icmp::wf_protocol_icmp() :
  wf_protocol(IPPROTO_ICMP),
  _type(-1),
  neg(false)
{}

wf_protocol_icmp::wf_protocol_icmp(const string& name) :
  wf_protocol(IPPROTO_ICMP),
  neg(false)
{
  set(name);
}

wf_protocol_icmp::wf_protocol_icmp(uint8_t type) :
  wf_protocol(IPPROTO_ICMP),
  neg(false)
{
  set(type);
}

wf_protocol_icmp::wf_protocol_icmp(uint8_t type, int8_t code) :
  wf_protocol(IPPROTO_ICMP),
  neg(false)
{
  set(type, code);
}


bool
wf_protocol_icmp::set(const string& name) {
  unsigned int i;
  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (icmp_codes[i].name == name) {
      _type = icmp_codes[i].type;
      if (icmp_codes[i].flag == 1) /* it is a type with a code */
	_code = icmp_codes[i].code;
      else
	_code = -1;
      return true;
    }
  }
  return false; /* not found */
}

bool
wf_protocol_icmp::set(uint8_t type) {
  if (type >
      icmp_codes[sizeof(icmp_codes) / sizeof(struct icmp_names) - 1].type)
    return false;

  _type = type;
  _code = -1;
  return true;
}

bool
wf_protocol_icmp::set(uint8_t type, int8_t code) {
  if (code == -1)
    return set(type);

  unsigned int i;
  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (icmp_codes[i].type == type &&
	icmp_codes[i].flag != 0xFF &&
	icmp_codes[i].code == code) {
      _type = type;
      _code = code;
      return true;
    }
  }
  return false;
}

string
wf_protocol_icmp::type_tostr() const {
  if (_type != -1)
    return wf_protocol_icmp_type_tostr(_type, _code);
  return "";
}


ostream&
wf_protocol_icmp_types_print(ostream& os) {
  unsigned int i;

  os << _("Valid ICMP Types:") << endl;

  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (i && icmp_codes[i].type == icmp_codes[i-1].type) {
      if (icmp_codes[i].flag == 2) /* alias */
	os << " (" << icmp_codes[i].name << ')';
      else /* code */
	os << endl << "   " << icmp_codes[i].name;
    }
    else
      os << endl << icmp_codes[i].name;
  }
  return os << endl;
}

string
wf_protocol_icmp_type_tostr(uint8_t type) {
  unsigned int i;
  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (icmp_codes[i].type == type)
      return icmp_codes[i].name;
  }
  return ""; /* not found */
}

string
wf_protocol_icmp_type_tostr(uint8_t type, int8_t code) {
  if (code == -1)
    return wf_protocol_icmp_type_tostr(type);

  unsigned int i;
  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (icmp_codes[i].type == type &&
	icmp_codes[i].flag != 0xFF &&
	icmp_codes[i].code == code)
      return icmp_codes[i].name;
  }
  return ""; /* not found */
}

bool
wf_protocol_icmp_totype(const string& name, int* type, int* code) {
  unsigned int i;
  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (icmp_codes[i].name == name) {
      *type = icmp_codes[i].type;
      if (icmp_codes[i].flag == 1) /* it is a type with a code */
	*code = icmp_codes[i].code;
      else
	*code = -1;
      return true;
    }
  }
  return false; /* not found */
}

bool
wf_protocol_icmp_type_hascode(uint8_t type) {
  unsigned int i;
  for (i = 0; i < sizeof(icmp_codes) / sizeof(struct icmp_names); i++) {
    if (icmp_codes[i].type == type &&
	icmp_codes[i].flag == 0xFF)
      return true;
  }
  return false;
}
