/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>
#include <algorithm>

#include <stdio.h>
#include <stdlib.h>
/* <<< guess */
#include <unistd.h>
#include <sys/param.h> /* for MAXHOSTNAMELEN */
#include <sys/utsname.h> /* for uname */
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <net/if.h>
#ifdef HAVE_SYS_SOCKIO_H
#include <sys/sockio.h>
#endif
#include <sys/ioctl.h>
#include <errno.h>
/* guess >>> */

#include "checks.h"
#include "wfhost.h"
#include "list1.h"
#include "defs.h"

/* Warning: this must match enum order. */
static const char* wf_osnames[] = {
  "Unknown", "Linux", "FreeBSD", "NetBSD", "OpenBSD", "Solaris", "SunOS",
  "Irix", "HP-UX" };

/* Warning: this must match enum order. */
static const char* wf_lindistrib[] = {
  "Unknown", "Debian", "Mandrake", "Redhat"
}; /* supported */
/*
  "Unknown", "Debian", "Slackware", "Mandrake", "Redhat", "SuSe", "TurboLinux"
*/

wf_host::wf_host() :
  name(),
  fqdn(),
  comment(),
  ipaddr(),
  ifaces(),
  services(),
  os(WF_OS_UNKNOWN),
  os_version_major(-1),
  os_version_minor(-1),
  os_version_micro(-1),
  lindistrib(WF_LINDISTRIB_UNKNOWN),
  lindistrib_version_major(-1),
  lindistrib_version_minor(-1)
{
}

wf_host::~wf_host() {}

// default wf_host::wf_host(const wf_host& host)


bool
wf_host::iface_isdefined(const string& name) const {
  list<wf_iface>::const_iterator first = ifaces.begin(), last = ifaces.end();
  for (; first != last; ++first)
    if (first->name == name)
      return true;
  return false;
}

bool
wf_host::iface_add(const wf_iface& iface) {
  if (iface_isdefined(iface.name))
    return false;  /* interface with the same name already defined */
  ifaces.push_back(iface);
  return true;
}

void
wf_host::iface_del(const wf_iface& iface) {
  ifaces.remove(iface);
}

void
wf_host::iface_del(const string& name) {
  list<wf_iface>::iterator first = ifaces.begin(), last = ifaces.end();
  for (; first != last; ++first) {
    if (first->name == name) {
      ifaces.erase(first);
      return; /* we infer there cannot be 2 interfaces with the same name */
    }
  }
}


bool
wf_host::service_add(const wf_service& service) {
  if (find(services.begin(), services.end(), service) != services.end())
    return false;  /* service already present in the list */
  
  services.push_back(service);
  return true;
}

void
wf_host::service_del(const wf_service& service) {
  services.remove(service);
}

bool
wf_host::os_version_guess() {
  struct utsname utsname;
  if (uname(&utsname))
    return false;

  int major, minor, micro;
  if (sscanf(utsname.release, "%i.%i.%i", &major, &minor, &micro) != 3) {
    if (sscanf(utsname.release, "%i.%i", &major, &minor) != 2)
      return false;
  }
  else
    os_version_micro = micro;

  os_version_major = major;
  os_version_minor = minor;
  return true;
}

bool
wf_host::os_guess() {
  struct utsname utsname;
  if (uname(&utsname))
    return false;
  /*
    cerr <<
    "sysname: " << utsname.sysname << endl <<
    "release: " << utsname.release << endl <<
    "machine: " << utsname.machine << endl;
  */
  unsigned int i;
  for (i = 1 /* skip "Unknown" */;
       i < sizeof(wf_osnames) / sizeof(char*);
       i++) {
    if (!strcmp(utsname.sysname, wf_osnames[i])) {
      os = (enum wf_os)i;
      break;
    }
  }
#if 0 // always return true for the moment ALL@@3
  if (i == sizeof(wf_osnames) / sizeof(char*))
    return false; /* not found */
#endif

  os_version_guess();

  return true;
}

string
wf_host::os_tostr() const {
  if (os < 0 || (unsigned)os >= (sizeof(wf_osnames) / sizeof(char*)))
    return "Invalid";
  return wf_osnames[os];
}

string
wf_host::os_version_tostr() const {
  if (os == WF_OS_UNKNOWN)
    return "";

  ostringstream outs;

  if (os_version_major != -1) {
    outs << os_version_major;
    if (os_version_minor != -1) {
      outs << '.' << os_version_minor;
      if (os_version_micro != -1)
	outs << '.' << os_version_micro;
    }
  }

  return outs.str();
}

#ifdef linux
/* We must figure it at runtime because configure time probing would
   prevent from shipping generic executables. */
bool
wf_host::linux_distribution_guess() {
  FILE* file;
  int major, minor;

  if ((file = fopen("/etc/debian_version", "r")) != NULL) {
    char line[32];
    lindistrib = WF_LINDISTRIB_DEBIAN;
    if (fgets(line, sizeof(line), file) == NULL)
      return false;
    if (sscanf(line, "%i.%i", &major, &minor) != 2)
      return false;
  }
  else if ((file = fopen("/etc/mandrake-release", "r")) != NULL) {
    lindistrib = WF_LINDISTRIB_MANDRAKE;
    if (fscanf(file, "Linux Mandrake release %i.%i", &major, &minor) != 2)
      return false;
  }
  else if ((file = fopen("/etc/redhat-release", "r")) != NULL) {
    lindistrib = WF_LINDISTRIB_REDHAT;
    if (fscanf(file, "Linux Redhat release %i.%i", &major, &minor) != 2)
      return false;
  }
  else
    return false;

  lindistrib_version_major = major;
  lindistrib_version_minor = minor;
  return true;
}

#endif /* linux */

string
wf_host::linux_distribution_tostr() const {
  if (os != WF_OS_LINUX ||
      lindistrib < 0 ||
      (unsigned)lindistrib >= sizeof(wf_lindistrib) / sizeof(char*))
    return "Invalid";
  return wf_lindistrib[lindistrib];
}

string
wf_host::linux_distribution_version_tostr() const {
  ostringstream outs;

  if (os != WF_OS_LINUX || lindistrib == WF_LINDISTRIB_UNKNOWN)
    return "";

  if (lindistrib_version_major != -1) {
    outs << lindistrib_version_major;
    if (lindistrib_version_minor != -1)
      outs << '.' << lindistrib_version_minor;
  }

  return outs.str();
}


bool
wf_host::name_guess() {
  char hostname[MAXHOSTNAMELEN + 1];

  if (gethostname(hostname, MAXHOSTNAMELEN) == 0)
    name = hostname;
  return true;
}

static string
replace_colon(const string& str) {
  string res;
  string::const_iterator first = str.begin(), last = str.end();
  for ( ; first != last; ++first) {
    switch (*first) {
    case ':':
      res += '_';
      break;
    default:
      res += *first;
    }
  }
  return res;
}

/* We consider that no field is already defined, except name. */
bool
wf_host::ifaces_guess(bool skip_useless) {
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
  struct ifaddrs *ifap, *ifa;
  if (getifaddrs(&ifap) < 0) {
    perror("getifaddrs");
    return false;
  }
  
  for (ifa = ifap; ifa; ifa = ifa->ifa_next) {
    if (iface_isdefined(ifa->ifa_name))
      continue;
    wf_iface iface;
    if (iface.guess(ifa->ifa_name, skip_useless))
      iface_add(iface);
  }
  freeifaddrs(ifap);
#else
  int sock;
  char iflist[2048];
  struct ifconf ifc;
  struct ifreq* ifr;
  int n;
  
  sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
  if (sock < 0) {
    perror("socket creation");
    return false;
  }
  /* get interface list */
  ifc.ifc_buf = iflist;
  ifc.ifc_len = sizeof(iflist);
  if (ioctl(sock, SIOCGIFCONF, (char*)&ifc) < 0) {
    perror("ioctl IFCONF");
    close(sock);
    return false;
  }

  ifr = ifc.ifc_req;
  for (n = ifc.ifc_len / sizeof(struct ifreq); --n >= 0; ++ifr) {
    wf_iface iface;
    if (iface.guess(ifr, sock, skip_useless))
      iface_add(iface);
  }
  close(sock);
#endif

  /* complete ifaces' names and comments */
  list<wf_iface>::iterator first = ifaces.begin(), last = ifaces.end();
  for (; first != last; ++first) {
    wf_iface& iface = *first;

    wf_network& network = iface.network;
    if (network.network.isdefined()) {
      struct netent* net;
      net = getnetbyaddr(ntohl(network.network.get()), AF_INET);
      if (net != NULL)
	network.name = net->n_name;
      else
	network.name = name + "_"  + replace_colon(iface.name) + "_net";
    }
    else
      network.name = name + "_"  + replace_colon(iface.name) + "_net";
    
    char* str;
    {
      char tmp[256];
      str = _("%s %s network");
      snprintf(tmp, sizeof(tmp), str, name.c_str(), iface.name.c_str());
      network.comment = tmp;
    }

    string devname;
    if (iface.flags & IFF_LOOPBACK)
      devname = "Loopback";
    else if (iface.flags & IFF_POINTOPOINT)
      devname = "PPP";
    else
      devname = iface.name;

    {
      char tmp[256];
      str = _("%s interface");
      snprintf(tmp, sizeof(tmp), str, devname.c_str());
      iface.comment = tmp;
    }


  }

  return true;
}

bool
wf_host::check() const {
  if (name.empty()) {
    fprintf(stderr, _("Host: no name is defined.\n"));
    return false;
  }

  if (wf_object_name_check(name) == false) {
    fprintf(stderr, _("Host `%s': invalid name.\n"), name.c_str());
    return false;
  }
  if (fqdn.empty() == false) {
    if (wf_hostname_check(fqdn) == false) {
      fprintf(stderr, _("Host `%s': invalid fqdn.\n"), fqdn.c_str());
      return false;
    }
  }
  /* check ifaces */
  list<wf_iface>::const_iterator first = ifaces.begin(), last = ifaces.end();
  for (; first != last; ++first) {
    if (first->check() == false) {
      fprintf(stderr, _("Host `%s': wrong interface.\n"), name.c_str());
      return false;
    }
  }

  // check services?? ALL@@2
  return true;
}

/* We consider that no field is already defined. */
bool
wf_host::guess(bool skip_useless) {
  name_guess();

  if (os_guess() == false || ifaces_guess(skip_useless) == false)
    return false;
  
  /* We do not return false if distribution guess fails. */
#ifdef linux
  if (os == WF_OS_LINUX)
    linux_distribution_guess();
#endif

  return true;
}

template<class T>
struct debugprint_class {
  debugprint_class(ostream& __os) : os(__os) {}
  void operator()(T x) { x.debugprint(os); }
  ostream& os;
};

ostream&
wf_host::debugprint(ostream& os) const {
  os << _("Host:") << endl;

  os << _("comment:\t") << comment << endl;

  os << _("name:\t\t") ;
  if (name.empty() == false)
    os << name;
  else
    os << _("(undefined)");
  os << endl;

  os << _("fqdn:\t\t");
  if (fqdn.empty() == false)
    os << fqdn;
  else
    os << _("(undefined)");
  os << endl;

  os << _("address:\t") << ipaddr << endl;

  for_each(ifaces.begin(), ifaces.end(), debugprint_class<wf_iface>(os));
  os << endl;

  for_each(services.begin(), services.end(), debugprint_class<wf_service>(os));

  return os;
}
