/*
   Terminal attributes functions
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

#if defined(VCHE_RAW)
int current_color; /* The current set color */
#endif

void term_init( void)
{
 /* 
  * Select null mapping - straight to character ROM
  *
  * See: man 4 console_codes
  *
  */

 printf( "\033(U"); 
}

#if defined(VCHE_VC) || defined(VCHE_RAW)

void term_noraw(void)
{
 struct termios tattr;

 if( tcgetattr( fileno( stdin), &tattr) < 0) {
     fprintf( stderr, "%s\n", strerror( errno));
     exit( 1);
 }

 tattr.c_lflag |= ICANON;
 tattr.c_lflag |= ECHO;
 tattr.c_iflag |= IXON;
 tattr.c_oflag |= OPOST;

 if( tcsetattr( fileno( stdin), TCSANOW,  &tattr) < 0) {
     fprintf( stderr, "%s\n", strerror( errno));
     exit( 1);
 }

}

void term_raw(void)
{
 struct termios tattr;

 if( tcgetattr( fileno( stdin), &tattr) < 0) {
     fprintf( stderr, "%s\n", strerror( errno));
     exit( 1);
 }

 tattr.c_lflag &= ~( ICANON);
 tattr.c_lflag &= ~( ECHO);
 tattr.c_iflag &= ~( INPCK | ISTRIP | IXON);
 tattr.c_oflag &= ~( OPOST);
 tattr.c_cc[  VMIN] = 1;
 tattr.c_cc[ VTIME] = 0;

 if( tcsetattr( fileno( stdin), TCSANOW,  &tattr) < 0) {
     fprintf( stderr, "%s\n", strerror( errno));
     exit( 1);
 }

}

#endif

void term_beep( void)
{
 if( flags.bell) {
#if defined(VCHE_VC) || defined(VCHE_RAW)
     putchar( '\007');
     fflush( stdout);
#elif defined(VCHE_NC)
     beep();
#endif
 }
}

#if defined(VCHE_VC) || defined(VCHE_RAW)

void term_color( int color)
{
#if defined(VCHE_RAW)
 current_color = color;
#endif

 switch( color) {
         case VC_COLOR_BLACK       : printf( "\033[0;40;30m"); break;
         case VC_COLOR_BLUE        : printf( "\033[0;40;34m"); break;
         case VC_COLOR_GREEN       : printf( "\033[0;40;32m"); break;
         case VC_COLOR_CYAN        : printf( "\033[0;40;36m"); break;
         case VC_COLOR_RED         : printf( "\033[0;40;31m"); break;
         case VC_COLOR_MAGENTA     : printf( "\033[0;40;35m"); break;
         case VC_COLOR_BROWN       : printf( "\033[0;40;33m"); break;
         case VC_COLOR_LIGHTGRAY   : printf( "\033[0;40;37m"); break;
         case VC_COLOR_DARKGRAY    : printf( "\033[1;40;30m"); break;
         case VC_COLOR_LIGHTBLUE   : printf( "\033[1;40;34m"); break;
         case VC_COLOR_LIGHTGREEN  : printf( "\033[1;40;32m"); break;
         case VC_COLOR_LIGHTCYAN   : printf( "\033[1;40;36m"); break;
         case VC_COLOR_LIGHTRED    : printf( "\033[1;40;31m"); break;
         case VC_COLOR_LIGHTMAGENTA: printf( "\033[1;40;35m"); break;
         case VC_COLOR_YELLOW      : printf( "\033[1;40;33m"); break;
         case VC_COLOR_WHITE       : printf( "\033[1;40;37m"); break;

         case VC_COLOR_BG_LIGHTGRAY: printf( "\033[0;47;30m"); break;

         case VC_COLOR_ATTENTION   : printf( "\033[1;41;37m"); break;
         case VC_COLOR_BACKCYAN    : printf( "\033[0;46;30m"); break;
         case VC_COLOR_BACKBLUE    : printf( "\033[1;44;36m"); break;

         case VC_COLOR_OFF         : printf( "\033[0m"      ); break;
 }

 fflush( stdout);
}
#endif

void term_unhide_cursor( void)
{
#if defined(VCHE_VC) || defined(VCHE_RAW)
 fputs( "\033[?25h", stdout);
 fflush( stdout);
#elif defined(VCHE_NC)
 curs_set( 1); /* Cursor normal */
#endif
}

void term_hide_cursor( void)
{
#if defined(VCHE_VC) || defined(VCHE_RAW)
 fputs( "\033[?25l", stdout);
 fflush( stdout);
#elif defined(VCHE_NC)
 curs_set( 0); /* Cursor invisible */
#endif
}

void term_goto( int x, int y)
{
#if defined(VCHE_VC) || defined(VCHE_RAW)
 printf( "\033[%d;%dH", y + 1, x + 1);
 fflush( stdout);
#elif defined(VCHE_NC)
 move( y, x);
 refresh();
#endif
}

#if defined(VCHE_NC)
void term_wgoto( WINDOW *w, int x, int y)
{
 wmove( w, y, x);
 wrefresh( w);
}
#endif

#if defined(VCHE_RAW)
void term_enable_graf_map( void)
{
 printf( "\033)0");  /* Start sequence for G1 set to vt100 graphics mapping */
 printf( "\016");    /* activates the G1 character set                      */
 fflush( stdout);
}

void term_disable_graf_map( void)
{
 printf( "\017");    /* activates the G0 character set */
 fflush( stdout);
}
#endif

/*

  If you don't know how to find more information on these "\033..."
  stuff, read the kernel sources for linux 2.x.x:

  linux/drivers/char/console.c
  linux/drivers/char/consolemap.c
  linux/drivers/char/consolemap.h
  ....

  man 4 console_codes

  and the sources of these packages:

  ncurses.x.x.x.tar.gz
  termcap.tar.gz (the /etc/termcap database, the |linux| entry)

*/

