/*
   Initialization functions
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

extern char *hbuffer;
extern char *hex_buffer;
extern char *abuffer;
#if defined(VCHE_VC)
extern char *zbuffer;
#endif
extern char *offset_buffer;
extern char *trans_buffer;

static char *token;
static const char delimiters[] = "=\t\n";

/*
 * Function prototypes
 */
#if defined(VCHE_VC)
static void init_read_vcss( void);
#endif
#if defined(VCHE_RAW)
static void init_screen_size( void);
#endif
static void init_check_config_file( void);
static void init_process_config( char *buf);
static int init_get_token( void);

#if defined(VCHE_VC)
int vcsafd;              /* File descriptor of the /dev/vcsa* file   */
int vcsfd;               /* File descriptor of the /dev/vcs* file    */
char **vcs_save_buffer;  /* Buffer for saving and restoring screens  */
int vcs_save_index;      /* Index for the save/restore screen buffer */
#endif

#if defined(VCHE_RAW)
char *restore_buffer;  /* Save/restore screen data */
int stop_restore;      /* Do not save restore data */
#endif

void init( void)
{
 vc_case_sensitive = VC_CASE_SENSITIVE;
 space_buffer = NULL;
 fpos = 0;

#if defined(VCHE_RAW)
 stop_restore = 0;
 init_screen_size();
#elif defined(VCHE_NC)
 init_ncurses();
#endif

 signals_init();

 term_init();

#if defined(VCHE_VC)
 init_read_vcss();
#endif

 term_hide_cursor();

 init_buffers();

 flags.bell     = 0;
 flags.edit     = 0;
 flags.nohex    = 0;
 flags.readonly = 0;
#if defined(VCHE_VC)
 flags.mask     = 0;
 flags.zero     = 0;
#endif

 init_check_config_file();

 if( flags.readonly && flags.edit) {
     fprintf( stderr, "\nConfig file error: Flags \"readonly\" and \"edit\" cannot be set to \"on\"\nat the same time.\n\n");
     do_exit( 1);
 }

#if defined(VCHE_VC)
 vcs_save_buffer = ( char **) allocate_mem( MAX_VCS_SAVE * sizeof( char *));
#elif defined(VCHE_RAW)
 raw_fill_screen( VC_COLOR_BACKBLUE);
#endif

 draw_main_screen();

 set_flag( FLAG_BELL);
 set_flag( FLAG_EDIT);
 set_flag( FLAG_NOHEX);
 set_flag( FLAG_READONLY);
#if defined(VCHE_VC)
 set_flag( FLAG_MASK);
 set_flag( FLAG_ZERO);
 vcs_save_index = 0;
#endif

 edit_mask = 0;

 memset( &srch, 0, 63);

 set_hexmode_variables();

 lib_fill_obuffer();
}

#if defined(VCHE_VC)

static void init_read_vcss( void)
{
 char vcsadevname[ 256];
 char vcsdevname[ 256];
 char *mytty;
 int vcnum;

 if( (mytty = ttyname( fileno( stdin))) == NULL) {
     perror( "stdin"); 
     exit( 1);
 }
 
 if( sscanf( mytty, "/dev/tty%d", &vcnum) != 1) { 
     fprintf( stderr, "This program only runs in a system console!\n");
     fprintf( stderr, "Run vche-nc or vche-raw instead.\n");
     exit(1);
 }

 sprintf( vcsadevname, "/dev/vcsa%d", vcnum);

 if( (vcsafd = open( vcsadevname, O_RDWR)) == -1) {
     fprintf( stderr, "Can't open \"%s\" : %s\n", strerror( errno), vcsadevname);
     exit( 1);
 }

 read( vcsafd, &scrn, 4);

 sprintf( vcsdevname, "/dev/vcs%d", vcnum);

 if( (vcsfd = open( vcsdevname, O_RDWR)) == -1) {
     fprintf( stderr, "Can't open \"%s\" : %s\n", strerror( errno), vcsdevname);
     exit( 1);
 }

}

#endif


#if defined(VCHE_NC)

void init_ncurses( void)
{
 initscr();     /* Init the ncurses structures */

 start_color(); /* We will use colors          */

 cbreak();                 /* cbreak mode                           */
 noecho();                 /* Do no echo characters                 */
 scrollok( stdscr, FALSE); /* Don't scroll                          */
 idlok( stdscr, TRUE);     /* Insert/Delete line on                 */
 keypad( stdscr, TRUE);    /* We will use the keypad, so turn it on */

 /* Max interval between chars in function keys, in millisecs. 
    It's the time to wait after each ESC char. I thought it was
    useless (for vche), but you may have some problems if you
    use some delay (i.e. in edit()). */
 ESCDELAY = 0;

 init_pair( 20, COLOR_WHITE, COLOR_BLACK );
 init_pair( 21, COLOR_WHITE, COLOR_BLACK );
 init_pair( 22, COLOR_BLACK, COLOR_WHITE );
 init_pair( 23, COLOR_WHITE, COLOR_RED   );
 init_pair( 24, COLOR_BLACK, COLOR_CYAN  );
 init_pair( 25, COLOR_CYAN , COLOR_BLUE  );
 init_pair( 26, COLOR_BLACK, COLOR_WHITE );
}

#endif

static void init_check_config_file( void)
{
 char buf[ 8192];
 FILE *fp;

 /* If the config file does not exist return silently */
 if( access( VCHE_CONFIG_FILE, F_OK) == -1) return;

 if( (fp = fopen( VCHE_CONFIG_FILE, "r")) == NULL) {
     fprintf( stderr, "Can't open \"%s\" : %s\n", VCHE_CONFIG_FILE, strerror( errno));
     do_exit( 1);
 }

 while( !feof( fp) && fgets( buf, sizeof( buf) - 1, fp) != NULL) {
        if( buf[ 0] == '#' || buf[ 0] == '\n') continue;
        init_process_config( buf);
 }

 fclose( fp);
}

static void init_process_config( char *buf)
{
 token = strtok( buf, delimiters);

 if( token == NULL) return;

 if( !strcasecmp( token, "bell"    )) flags.bell     = init_get_token();
 if( !strcasecmp( token, "edit"    )) flags.edit     = init_get_token();
 if( !strcasecmp( token, "nohex"   )) flags.nohex    = init_get_token();
 if( !strcasecmp( token, "readonly")) flags.readonly = init_get_token();
#if defined(VCHE_VC)
 if( !strcasecmp( token, "mask"    )) flags.mask     = init_get_token();
 if( !strcasecmp( token, "zero"    )) flags.zero     = init_get_token();
#endif
}

static int init_get_token( void)
{
 token = strtok( NULL, delimiters);

 if( token == NULL) return 0;

 if( !strcasecmp( token,  "on")) return 1;
 if( !strcasecmp( token, "off")) return 0;

 return 0;
}

void init_buffers( void)
{
 hex_buffer = ( char *) allocate_mem( vc_cols);
 memset( hex_buffer, ' ', vc_cols);

 hbuffer    = ( char *) allocate_mem( vc_cols *     ( vc_lines - 2));
 obuffer    = ( char *) allocate_mem( vc_cols * 2 * ( vc_lines - 2));

#if defined(VCHE_RAW)
 restore_buffer = ( char *) allocate_mem( vc_cols * 2 * vc_lines );
#endif

 space_buffer = ( char *) allocate_mem( vc_cols);
 memset( space_buffer, ' ', vc_cols);

 abuffer = ( char *) allocate_mem( vc_cols);

#if defined(VCHE_VC)
 zbuffer = ( char *) allocate_mem( vc_cols);
#endif

 offset_buffer = ( char *) allocate_mem( vc_cols);
 trans_buffer  = ( char *) allocate_mem( vc_cols);
}

#if defined(VCHE_RAW)

static void init_screen_size( void)
{
 struct winsize wsize;

 /* Defaults, if something goes wrong */
 wsize.ws_row    = 25;
 wsize.ws_col    = 80;
 wsize.ws_xpixel =  0;
 wsize.ws_ypixel =  0;

 /* If we are on a terminal, get windows size */
 if( isatty( fileno( stdin)))
     ioctl( fileno( stdin), TIOCGWINSZ, &wsize);

 vc_lines = wsize.ws_row;
 vc_cols  = wsize.ws_col;
 scrn.xp  = wsize.ws_xpixel;
 scrn.yp  = wsize.ws_xpixel;
}

#endif

