/* valalocalvariable.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaLocalVariablePrivate {
	ValaExpression* _initializer;
	ValaDataType* _variable_type;
};

#define VALA_LOCAL_VARIABLE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariablePrivate))
enum  {
	VALA_LOCAL_VARIABLE_DUMMY_PROPERTY,
	VALA_LOCAL_VARIABLE_INITIALIZER,
	VALA_LOCAL_VARIABLE_VARIABLE_TYPE
};
static void vala_local_variable_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_local_variable_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_local_variable_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_local_variable_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_local_variable_parent_class = NULL;
static void vala_local_variable_dispose (GObject * obj);



/**
 * Creates a new local variable.
 *
 * @param name   name of the variable
 * @param init   optional initializer expression
 * @param source reference to source code
 * @return       newly created variable declarator
 */
ValaLocalVariable* vala_local_variable_new (ValaDataType* variable_type, const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaLocalVariable * self;
	g_return_val_if_fail (variable_type == NULL || VALA_IS_DATA_TYPE (variable_type), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_LOCAL_VARIABLE, 0, NULL);
	vala_local_variable_set_variable_type (self, variable_type);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_local_variable_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_local_variable_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_local_variable (visitor, self);
}


static void vala_local_variable_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (vala_local_variable_get_initializer (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_local_variable_get_initializer (self)), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_local_variable_get_initializer (self));
	}
	if (vala_local_variable_get_variable_type (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_local_variable_get_variable_type (self)), visitor);
	}
}


static void vala_local_variable_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	if (vala_local_variable_get_initializer (self) == old_node) {
		vala_local_variable_set_initializer (self, new_node);
	}
}


static void vala_local_variable_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_local_variable_get_variable_type (self) == old_type) {
		vala_local_variable_set_variable_type (self, new_type);
	}
}


ValaExpression* vala_local_variable_get_initializer (ValaLocalVariable* self) {
	g_return_val_if_fail (VALA_IS_LOCAL_VARIABLE (self), NULL);
	return self->priv->_initializer;
}


void vala_local_variable_set_initializer (ValaLocalVariable* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_LOCAL_VARIABLE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL))), _tmp2);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_initializer), VALA_CODE_NODE (self));
	}
}


ValaDataType* vala_local_variable_get_variable_type (ValaLocalVariable* self) {
	g_return_val_if_fail (VALA_IS_LOCAL_VARIABLE (self), NULL);
	return self->priv->_variable_type;
}


void vala_local_variable_set_variable_type (ValaLocalVariable* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_LOCAL_VARIABLE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_variable_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_variable_type == NULL ? NULL : (self->priv->_variable_type = (g_object_unref (self->priv->_variable_type), NULL))), _tmp2);
	if (self->priv->_variable_type != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_variable_type), VALA_CODE_NODE (self));
	}
}


static void vala_local_variable_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (object);
	switch (property_id) {
		case VALA_LOCAL_VARIABLE_INITIALIZER:
		g_value_set_object (value, vala_local_variable_get_initializer (self));
		break;
		case VALA_LOCAL_VARIABLE_VARIABLE_TYPE:
		g_value_set_object (value, vala_local_variable_get_variable_type (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_local_variable_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (object);
	switch (property_id) {
		case VALA_LOCAL_VARIABLE_INITIALIZER:
		vala_local_variable_set_initializer (self, g_value_get_object (value));
		break;
		case VALA_LOCAL_VARIABLE_VARIABLE_TYPE:
		vala_local_variable_set_variable_type (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_local_variable_class_init (ValaLocalVariableClass * klass) {
	vala_local_variable_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaLocalVariablePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_local_variable_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_local_variable_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_local_variable_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_local_variable_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_local_variable_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_local_variable_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_local_variable_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_LOCAL_VARIABLE_INITIALIZER, g_param_spec_object ("initializer", "initializer", "initializer", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_LOCAL_VARIABLE_VARIABLE_TYPE, g_param_spec_object ("variable-type", "variable-type", "variable-type", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_local_variable_instance_init (ValaLocalVariable * self) {
	self->priv = VALA_LOCAL_VARIABLE_GET_PRIVATE (self);
}


static void vala_local_variable_dispose (GObject * obj) {
	ValaLocalVariable * self;
	self = VALA_LOCAL_VARIABLE (obj);
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL)));
	(self->priv->_variable_type == NULL ? NULL : (self->priv->_variable_type = (g_object_unref (self->priv->_variable_type), NULL)));
	G_OBJECT_CLASS (vala_local_variable_parent_class)->dispose (obj);
}


GType vala_local_variable_get_type (void) {
	static GType vala_local_variable_type_id = 0;
	if (G_UNLIKELY (vala_local_variable_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaLocalVariableClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_local_variable_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaLocalVariable), 0, (GInstanceInitFunc) vala_local_variable_instance_init };
		vala_local_variable_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaLocalVariable", &g_define_type_info, 0);
	}
	return vala_local_variable_type_id;
}




