/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_pairs.h"
#include "pairsModeling/externalPairsExtensions.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>

#include "visu_tools.h"
#include "visu_object.h"
#include "visu_nodes.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/text.h"
#include "visu_extension.h"
#include "visu_configFile.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/dataNode.h"

/**
 * SECTION:visu_pairs
 * @short_description: V_Sim can draw link between nodes. This part
 * defines a pair object and interface to draw pairs.
 *
 * <para>The visu_pairs.c defines only general methods to draw
 * pairs. It introduces a new object called #VisuPairData. This stores
 * some characteristics on links between two #VisuElement. The main
 * characteristic is that pairs are drawn only if the length between
 * two nodes is in a specific range. Use visu_pair_data_setDistance() and
 * visu_pair_data_getDistance() to tune this range.</para>
 *
 * <para>This file does not draw any pairs. But it gives some
 * interface to create rendering capabilities. To create a new pair
 * rendering module, called #VisuPairExtension, use
 * visu_pair_extension_new(). Basically, a #VisuPairExtension is characterized
 * by it drawing method. But it can have other methods that are called
 * in different cases. See VisuPairDrawFunc() and
 * VisuPairStartEndFunc() prototypes to have more informations.</para>
 */

#define PAIRS_ONOFF_DEFAULT 0

/**
 * _Property:
 * @name: a pointer to a string, only used when DEBUG is on ;
 * @data: a pointer to an allocated memory area ;
 * @freeFunc: a destroy functon (can be NULL).
 *
 * This container can be used in hashtable to store whatever kind of
 * data that requires specific free functions.
 */
struct _Property
{
  /* A convenient debug pointer on the name. */
  const gchar *name;

  /* This is a pointer to the data stored. */
  gpointer data;
  
  /* This method is called when the table is freed. */
  GDestroyNotify freeFunc;
};
typedef struct _Property Property;

/* This structure is made to store pairs information between two
   elements. */
struct _VisuPair
{
  VisuElement *ele1;
  VisuElement *ele2;

  /* This is a GList of link (VisuPairData). */
  GList *pairs;

  /* A list of properties. */
  GHashTable *properties;
};

struct foreachPairsData_struct
{
  VisuPairForeachFunc func;
  gpointer userData;
};

#define BONDHISTOGRAM_ID   "bondDistribution_data"
#define BONDHISTOGRAM_STEP 0.1f
#define BONDHISTOGRAM_MIN  0.f
#define BONDHISTOGRAM_MAX  10.f

static VisuExtension* extensionPairs;
static int openGlListPairsId;

/* This hashtable as VisuElement* as keys and pointer to other hashtable as value.
   The main idea is to have an hashtable of 2 dimension, each two keys are
   VisuElement* and the final values are PairsData. Nevertheless, this hashtable
   must not be access directly but through the get and put methods. */
static GHashTable *DminDmax;

static GList *availableVisuPairExtensions;
static VisuPairExtension *defaultPairMethod;
static gboolean rebuildPairsNeeded;
static guint sensitiveToFacette;

/* Internal variables. */
#define BONDNUMBER_ID "bondNumber_data"
static VisuDataNode *dataNode;

/* Parameters */
static ToolColor *defaultPairColor;

/* Local methods. */
static void freeData(gpointer obj, gpointer data);
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data);
static int* getBond(VisuData *dataObj, VisuNode *node);
static void freePair(gpointer ele);
static void freePairData(VisuPairData *data);
static void freeHistoData(gpointer data);
static VisuPairData* newPairData(float minMax[2]);
static VisuPair* newPair(VisuElement *ele1, VisuElement *ele2);


/* Callbacks. */
static void onNewData(GObject* obj, VisuData *dataObj, gpointer data);
static void onDataReady(GObject* obj, VisuData *dataObj, gpointer data);
static void onNodesReady(VisuData *dataObj, VisuNodeArray *nodes, gpointer data);
static void rebuildPairs(VisuData *dataObj);
static void facetteChanged(VisuData* obj, gpointer data);
static void createPairsOnNodeRenderedChanged(VisuData *dataObj, gpointer data);
static void createPairsOnNodePositionChanged(VisuData *dataObj, gpointer data);
static void createPairsOnNodePopulationChanged(VisuData *dataObj,
					       int *nodes, gpointer data);
static void createPairsOnResources(GObject *obj, VisuData *dataObj, gpointer bool);

/* This function details how to read what is needed by PairsData. */
static gboolean readPairsData(gchar **lines, int nbLines, int position,
			      VisuData *dataObj, GError **error);
static gboolean readPairLink(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);
static gboolean readPairsAreOn(gchar **lines, int nbLines, int position,
			       VisuData *dataObj, GError **error);
static gboolean readFavPairsMethod(gchar **lines, int nbLines, int position,
				   VisuData *dataObj, GError **error);
/* This function save the resources. */
static void exportResourcesPairs(GString *data, VisuData *dataObj);
#define FLAG_RESOURCES_PAIRS      "pairs_are_on"
#define DESC_RESOURCES_PAIRS      "Ask the opengl engine to draw pairs between elements ; boolean 0 or 1"
#define RESOURCES_PAIRS_DEFAULT FALSE

#define FLAG_RESOURCES_PAIRS_DATA "pair_data"
#define DESC_RESOURCES_PAIRS_DATA "Draw pairs between [ele1] [ele2] [0. <= dmin] [0. <= dmax] [0. <= RGB <= 1.]x3"
#define FLAG_RESOURCES_PAIR_LINK "pair_link"
#define DESC1_RESOURCES_PAIR_LINK "Draw a link between [ele1] [ele2] [0. <= dmin] [0. <= dmax]"
#define DESC2_RESOURCES_PAIR_LINK "                    [0. <= RGB <= 1.]x3 [bool: drawn] [bool: printLength] [string: method]"

#define FLAG_RESOURCES_FAV_PAIRS  "pairs_favoriteMethod"
#define DESC_RESOURCES_FAV_PAIRS  "Favorite method used to render files ; chain"



int initPairsModule()
{
  char *name = "Pairs";
  char *description = _("Draw pairs between elements with a criterion of distance.");
  int i, res;
  VisuPairExtension *extension;
  float rgbOfPairs[4] = {1.0, 0.6, 0.2, 1.};
  VisuConfigFileEntry *resourceEntry, *oldEntry;

  openGlListPairsId = visu_openGL_objectList_new(1);
  extensionPairs = visu_extension_new(name, _(name), description,
				       openGlListPairsId, rebuildPairs);
  visu_extension_setPriority(extensionPairs, VISU_EXTENSION_PRIORITY_LOW);
  visu_extension_setSensitiveToRenderingMode(extensionPairs, TRUE);
  extensionPairs->used = PAIRS_ONOFF_DEFAULT;
  visuExtensions_add(extensionPairs);

  /* Initialise the pairs extensions. */
  availableVisuPairExtensions = (GList*)0;
  res = 1;
  for (i = 0; listInitPairsFunc[i]; i++)
    {
      extension = listInitPairsFunc[i]();
      if (!extension)
	res = 0;
      visu_pair_extension_add(extension);
    }
  if (!res)
    g_warning("Some pairs extensions can't initialse.\n");

  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_PAIRS,
					  DESC_RESOURCES_PAIRS,
					  1, readPairsAreOn);
  oldEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				     FLAG_RESOURCES_PAIRS_DATA,
				     DESC_RESOURCES_PAIRS_DATA,
				     1, readPairsData);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_PAIR_LINK,
					  DESC1_RESOURCES_PAIR_LINK,
					  2, readPairLink);
  visu_configFile_entry_setVersion(resourceEntry, 3.4f);
  visu_configFile_entry_setReplace(resourceEntry, oldEntry);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_FAV_PAIRS,
					  DESC_RESOURCES_FAV_PAIRS,
					  1, readFavPairsMethod);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesPairs);

  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReady), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onNewData), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
		   G_CALLBACK(createPairsOnResources), (gpointer)0);

  DminDmax = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, freePair);
  
  defaultPairColor = tool_color_new(rgbOfPairs);
  tool_color_addColor(defaultPairColor);

  rebuildPairsNeeded = TRUE;
  
  if (availableVisuPairExtensions)
    defaultPairMethod = (VisuPairExtension*)availableVisuPairExtensions->data;
  else
    defaultPairMethod = (VisuPairExtension*)0;
  sensitiveToFacette = (defaultPairMethod &&
                        defaultPairMethod->sensitiveToFacette)?1:0;

  /* Register a new NodeData. */
  dataNode = VISU_DATA_NODE(visu_data_node_new(BONDNUMBER_ID, G_TYPE_INT));
  visu_data_node_setLabel(dataNode, _("Bonds"));

  return res;
}

static void freePair(gpointer ele)
{
  VisuPair *pair;
  GList *tmpLst;

  pair = (VisuPair*)ele;
  DBG_fprintf(stderr, "Visu Pairs: freeing pair between '%s' and '%s'.\n",
	      pair->ele1->name, pair->ele2->name);
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    freePairData((VisuPairData*)tmpLst->data);
  g_list_free(pair->pairs);
  g_hash_table_destroy(pair->properties);
}
static void freePairData(VisuPairData *data)
{
  DBG_fprintf(stderr, "Visu Pairs: freeing link.\n");
  if (data->method && data->method->sensitiveToFacette)
    sensitiveToFacette -= 1;  
  g_hash_table_destroy(data->properties);
}

static VisuPairData* newPairData(float minMax[2])
{
  VisuPairData *data;
  ToolColor *color;

  /* We create one link. */
  data = g_malloc(sizeof(VisuPairData));
  data->minMax[VISU_PAIR_DISTANCE_MIN] = minMax[VISU_PAIR_DISTANCE_MIN];
  data->minMax[VISU_PAIR_DISTANCE_MAX] = minMax[VISU_PAIR_DISTANCE_MAX];
  data->drawn             = TRUE;
  data->printLength       = FALSE;
  data->method            = (VisuPairExtension*)0;
  data->properties        = g_hash_table_new_full(g_str_hash, g_str_equal,
						  NULL, g_free);
  g_return_val_if_fail(data->properties, (VisuPairData*)0);

  /* We create the color property . */
  color = g_malloc(sizeof(ToolColor));
  tool_color_copy(color, defaultPairColor);
  g_hash_table_insert(data->properties, (gpointer)"color", (gpointer)color);

  return data;
}

/* Create a new PairsData structure with default values.
   The newly created structure can be freed by a call to free. */
static VisuPair* newPair(VisuElement *ele1, VisuElement *ele2)
{
  VisuPair *pair;

  g_return_val_if_fail(ele1 && ele2, (VisuPair*)0);

  DBG_fprintf(stderr, "Visu Pairs: create a new pair between '%s' and '%s'.\n",
	      ele1->name, ele2->name);

  pair        = g_malloc(sizeof(VisuPair));
  pair->ele1  = ele1;
  pair->ele2  = ele2;
  pair->pairs = (GList*)0;
  pair->properties = g_hash_table_new_full(g_str_hash, g_str_equal,
					   NULL, g_free);
  g_return_val_if_fail(pair->properties, (VisuPair*)0);

/*   pair->pairs = g_list_prepend(pair->pairs, (gpointer)newPairData(minMax)); */

  return pair;
}

/**
 * visu_pair_setStatus:
 * @onOff : a boolean 0 or 1.
 * 
 * Turn on or off the pairs.
 *
 * Returns: TRUE if visu_pair_build() should be called and then the 'OpenGLAskForReDraw'
 *          signal be emitted.
 */
gboolean visu_pair_setStatus(gboolean onOff)
{
  GList *allObj;

  if (onOff == extensionPairs->used)
    return FALSE;

  extensionPairs->used = onOff;
  DBG_fprintf(stderr, "Visu Pairs: change pairs on/off status (rebuild"
	      " flag is %d).\n", rebuildPairsNeeded);

  /* Change the status of the bond count. */
  allObj = visu_data_getAllObjects();
  if (allObj)
    visu_data_node_setUsed(dataNode, VISU_DATA(allObj->data), (onOff)?1:0);
/*   visu_pair_build(); */
  return TRUE;
}
/**
 * visu_pair_setOutOfDate:
 *
 * Use this method to change the internal flag that pairs should be
 * rebuilt. It is useful when an extension of pairs has one of its
 * parameters that has changed.
 */
void visu_pair_setOutOfDate()
{
  rebuildPairsNeeded = TRUE;
}
/**
 * visu_pair_data_setDrawn:
 * @data: a #VisuPairData object ;
 * @drawn: a boolean.
 *
 * A pair can or cannot be drawn, use this method to tune it.
 *
 * Returns: TRUE if visu_pair_build() should be called.
 */
gboolean visu_pair_data_setDrawn(VisuPairData *data, gboolean drawn)
{
  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set drawn status %d (%d) for %p.\n",
	      (int)drawn, (int)data->drawn, (gpointer)data);
  if (data->drawn == drawn)
    return FALSE;
  data->drawn = drawn;

  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}
/**
 * visu_pair_data_setPrintLength:
 * @data: a #VisuPairData object ;
 * @status: TRUE to print length near pairs.
 *
 * Set the attribute that controls if the length of pairs are drawn near pairs.
 *
 * Returns: TRUE if visu_pair_build() should be called or not.
 */
gboolean visu_pair_data_setPrintLength(VisuPairData *data, gboolean status)
{
  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set print length status %d (%d) for %p.\n",
	      (int)status, (int)data->printLength, (gpointer)data);
  if (data->printLength == status)
    return FALSE;

  data->printLength = status;
  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}
/**
 * visu_pair_data_setDrawMethod:
 * @data: a #VisuPairData object.
 * @ext: a #VisuPairExtension object.
 * 
 * Set the drawing method of a pair.
 *
 * Since: 3.6
 *
 * Returns: TRUE if drawing method is changed.
 */
gboolean visu_pair_data_setDrawMethod(VisuPairData *data, VisuPairExtension *ext)
{
  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set method %p (%p) for %p.\n",
	      (gpointer)ext, (gpointer)data->method, (gpointer)data);
  if (data->method == ext)
    return FALSE;

  if (data->method && data->method->sensitiveToFacette)
    sensitiveToFacette -= 1;
  data->method = ext;
  if (data->method && data->method->sensitiveToFacette)
    sensitiveToFacette += 1;
  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}

/**
 * visu_pair_data_getDrawn:
 * @data: a #VisuPairData object ;
 *
 * A pair can or cannot be drawn, use this method to retrieve its state.
 *
 * Returns: TRUE if pairs can be drawn.
 */
gboolean visu_pair_data_getDrawn(VisuPairData *data)
{
  g_return_val_if_fail(data, FALSE);
  return data->drawn;
}
/**
 * visu_pair_data_setColor:
 * @data: a #VisuPairData object ;
 * @destColor: a #ToolColor object.
 *
 * Set the color of the given pair.
 *
 * Returns: TRUE if visu_pair_build() should be called or not.
 */
gboolean visu_pair_data_setColor(VisuPairData *data, ToolColor* destColor)
{
  ToolColor *color;

  g_return_val_if_fail(data && destColor, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set color [%g;%g;%g] for %p.\n",
	      destColor->rgba[0], destColor->rgba[1], destColor->rgba[2],
	      (gpointer)data);

  color = (ToolColor*)g_hash_table_lookup(data->properties, (gpointer)"color");
  if (!color)
    {
      color = g_malloc(sizeof(ToolColor));
      g_hash_table_insert(data->properties, (gpointer)"color", (gpointer)color);
    }
  else
    if (tool_color_equal(color, destColor))
      return FALSE;

  /* Copy values of dest to current color. */
  tool_color_copy(color, destColor);

  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}
/**
 * visu_pair_data_getColor:
 * @data: a #VisuPairData object.
 *
 * Look for the properties of the pair @data to find if a colour has
 * been defined. If none, the default colour is returned instead.
 *
 * Returns: a colour (don't free it).
 */
ToolColor* visu_pair_data_getColor(VisuPairData *data)
{
  ToolColor *color;

  g_return_val_if_fail(data, defaultPairColor);

  color = (ToolColor*)g_hash_table_lookup(data->properties, (gpointer)"color");
  if (!color)
    color = defaultPairColor;
/*   DBG_fprintf(stderr, "Visu Pairs: get color [%g;%g;%g] for %p.\n", */
/* 	      color->rgba[0], color->rgba[1], color->rgba[2], */
/* 	      (gpointer)data); */
  return color;
}
/**
 * visu_pair_data_setDistance:
 * @val: a floating point value ;
 * @data: a #VisuPairData object ;
 * @minOrMax: #VISU_PAIR_DISTANCE_MAX or #VISU_PAIR_DISTANCE_MIN.
 *
 * Set the minimum or the maximum length for the given pair.
 *
 * Returns: TRUE if visu_pair_build() should be called or not.
 */
gboolean visu_pair_data_setDistance(VisuPairData *data, float val, int minOrMax)
{
  g_return_val_if_fail(data && (minOrMax == VISU_PAIR_DISTANCE_MIN ||
				minOrMax == VISU_PAIR_DISTANCE_MAX), FALSE);

  if (data->minMax[minOrMax] == val)
    return FALSE;

  data->minMax[minOrMax] = val;
  rebuildPairsNeeded = TRUE;

  return rebuildPairsNeeded && extensionPairs->used;
}
/**
 * visu_pair_data_setProperty:
 * @data: a #VisuPairData object ;
 * @key: a static string ;
 * @value: a pointer to some allocated data.
 *
 * Each pair can store some informations that can be retrieve by a string key.
 * This method is used to registered a new value associated to the key @key.
 */
void visu_pair_data_setProperty(VisuPairData *data, const gchar* key, gpointer value)
{
  if (data && key)
    g_hash_table_insert(data->properties, (gpointer)key, (gpointer)value);

  rebuildPairsNeeded = TRUE;
}
/**
 * visu_pair_setProperty:
 * @pair: a #VisuPair object ;
 * @key: a static string ;
 * @value: a pointer to some allocated data ;
 * @freeFunc: a destroying method (can be NULL).
 *
 * Each element/element can have associated data.
 */
void visu_pair_setProperty(VisuPair *pair, const gchar* key,
			  gpointer value, GDestroyNotify freeFunc)
{
  Property *prop;

  g_return_if_fail(pair && key && *key);

  prop           = g_malloc(sizeof(Property));
  prop->name     = key;
  prop->data     = value;
  prop->freeFunc = freeFunc;
  g_hash_table_insert(pair->properties, (gpointer)key, (gpointer)prop);
}
/**
 * visu_pair_getProperty:
 * @pair: a #VisuPair object ;
 * @key: a string.
 *
 * Retrieve the property associated to the @key or NULL if none exist.
 *
 * Returns: the associated data.
 */
gpointer visu_pair_getProperty(VisuPair *pair, const gchar* key)
{
  Property *prop;

  g_return_val_if_fail(pair, (gpointer)0);

  prop = (Property*)g_hash_table_lookup(pair->properties, (gpointer)key);
  if (prop)
    return prop->data;
  else
    return (gpointer)0;
}

/**
 * visu_pair_getStatus:
 * 
 * Get the status of pairs, drawn or not.
 *
 * Returns: TRUE if pairs are drawn.
 */
gboolean visu_pair_getStatus()
{
  if(!extensionPairs)
    initPairsModule();

  return (gboolean)extensionPairs->used;
}
/**
 * visu_pair_data_getPrintLength:
 * @data: a #VisuPairData object.
 * 
 * Get the print length parameter of a pair. This parameter is used to tell if
 * length should be drawn near pairs of this kind.
 *
 * Returns: TRUE if length are printed.
 */
gboolean visu_pair_data_getPrintLength(VisuPairData *data)
{
  g_return_val_if_fail(data, FALSE);
  return data->printLength;
}
/**
 * visu_pair_data_getDrawMethod:
 * @data: a #VisuPairData object.
 * 
 * Get the drawing method of a pair.
 *
 * Since: 3.6
 *
 * Returns: a drawing method.
 */
VisuPairExtension* visu_pair_data_getDrawMethod(VisuPairData *data)
{
  g_return_val_if_fail(data, FALSE);
  return (data->method)?data->method:defaultPairMethod;
}
/**
 * visu_pair_data_getDistance:
 * @data: a #VisuPairData object ;
 * @minOrMax: #VISU_PAIR_DISTANCE_MIN or #VISU_PAIR_DISTANCE_MAX.
 *
 * A pair between @ele1 and @ele2 is drawn only if its length is between
 * a minimum and a maximum value. This method can get these values.
 *
 * Returns: the minimum or the maximum value for the pair between @ele1 and @ele2.
 */
float visu_pair_data_getDistance(VisuPairData *data, int minOrMax)
{
  g_return_val_if_fail(data, 0.);
  g_return_val_if_fail(minOrMax == VISU_PAIR_DISTANCE_MIN || minOrMax == VISU_PAIR_DISTANCE_MAX, 0.);

  return data->minMax[minOrMax];
}
/**
 * visu_pair_data_getProperty:
 * @data: a #VisuPairData object ;
 * @key: a static string.
 *
 * Each pair can store some informations that can be retrieve by a string key.
 * This method is used to retrieve a stored value associated to the key @key.
 *
 * Returns: a found value, or NULL if nothing is associated to the @key.
 *          If something is returned it should not be freed.
 */
gpointer visu_pair_data_getProperty(VisuPairData *data, const gchar* key)
{
  if (data)
    return g_hash_table_lookup(data->properties, (gpointer)key);
  else
    return (gpointer)0;
}

/**
 * visu_pair_getPair:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object.
 *
 * The object #VisuPair is used to characterized links between two elements.
 *
 * Returns: the #VisuPair object associated to the given two elements. If
 *          none exists it is created. The returned value should not be freed.
 */
VisuPair* visu_pair_getPair(VisuElement *ele1, VisuElement *ele2)
{
  VisuPair *pair;
  gchar *key;

  g_return_val_if_fail(ele1 && ele2, (VisuPair*)0);

  if (strcmp(ele1->name, ele2->name) < 0)
    key = g_strdup_printf("%s %s", ele1->name, ele2->name);
  else
    key = g_strdup_printf("%s %s", ele2->name, ele1->name);
  pair = (VisuPair*)g_hash_table_lookup(DminDmax, (gpointer)key);
  DBG_fprintf(stderr, "Visu Pairs: test key '%s' -> %p.\n", key, (gpointer)pair);
  if (!pair)
    {
      /* Ok, create one if none found. */
      pair = newPair(ele1, ele2);
      g_hash_table_insert(DminDmax, (gpointer)key, (gpointer)pair);
    }
  else
    g_free(key);

  return pair;
}
/**
 * visu_pair_getAllPairData:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object.
 *
 * There can be one or several links between elements, retrieve them
 * with this routine.
 *
 * Returns: a list of #VisuPairData. The list is owned by V_Sim and
 * should not be freed.
 */
GList* visu_pair_getAllPairData(VisuElement *ele1, VisuElement *ele2)
{
  VisuPair *pair;
  float minMax[2] = {0.f, 0.f};

  pair = visu_pair_getPair(ele1, ele2);
  DBG_fprintf(stderr, "Visu Pairs: links for key '%s %s' -> %p.\n",
	      ele1->name, ele2->name, (gpointer)pair->pairs);
  if (!pair->pairs)
    pair->pairs = g_list_append(pair->pairs, (gpointer)newPairData(minMax));
  return pair->pairs;
}
/**
 * visu_pair_getPairData:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @minMax: the two min and max distances.
 *
 * A link between two elements is characterized by its boundary distances.
 *
 * Returns: the #VisuPairData object associated to the given two
 *          elements and distances. If none exists it is created. The
 *          returned value should not be freed.
 */
VisuPairData* visu_pair_getPairData(VisuElement *ele1, VisuElement *ele2, float minMax[2])
{
  VisuPair *pair;
  GList *tmpLst;
  VisuPairData *data;

  g_return_val_if_fail(minMax, (VisuPairData*)0);

  pair = visu_pair_getPair(ele1, ele2);
  g_return_val_if_fail(pair, (VisuPairData*)0);
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      data = (VisuPairData*)tmpLst->data;
      DBG_fprintf(stderr, " | test %p (%g %g).\n", tmpLst->data,
		  data->minMax[0], data->minMax[1]);
      if (data->minMax[0] == minMax[0] && data->minMax[1] == minMax[1])
	return data;
    }
  data = newPairData(minMax);
  pair->pairs = g_list_append(pair->pairs, (gpointer)data);
  DBG_fprintf(stderr, " | new %p (%g %g).\n", (gpointer)data,
	      data->minMax[0], data->minMax[1]);
  rebuildPairsNeeded = TRUE;

  return data;
}
/**
 * visu_pair_removePairData:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a link object.
 *
 * Delete the given link.
 *
 * Returns: TRUE if the link exists and has been successfully removed.
 */
gboolean visu_pair_removePairData(VisuElement *ele1, VisuElement *ele2, VisuPairData *data)
{
  VisuPair *pair;
  GList *tmpLst;

  g_return_val_if_fail(data, FALSE);

  pair = visu_pair_getPair(ele1, ele2);
  g_return_val_if_fail(pair, FALSE);
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      if ((VisuPairData*)tmpLst->data == data)
	{
	  freePairData(data);
	  if (tmpLst->prev)
	    tmpLst->prev->next = tmpLst->next;
	  if (tmpLst->next)
	    tmpLst->next->prev = tmpLst->prev;
	  if (pair->pairs == tmpLst)
	    pair->pairs = tmpLst->next;
	  g_list_free_1(tmpLst);
	  
	  rebuildPairsNeeded = TRUE;
	  return TRUE;
	}
    }
  return FALSE;
}
/**
 * visu_pair_getPairDataFromId:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @pos: the position in the list of links.
 *
 * A link can also be retrieved by its position.
 *
 * Returns: the #VisuPairData object associated to the given two
 *          elements and distances. If none exists NULL is returned.
 */
VisuPairData* visu_pair_getPairDataFromId(VisuElement *ele1, VisuElement *ele2, guint pos)
{
  VisuPair *pair;

  pair = visu_pair_getPair(ele1, ele2);
  g_return_val_if_fail(pair->pairs, (VisuPairData*)0);
  return (VisuPairData*)(g_list_nth(pair->pairs, pos)->data);
}


static void createPairsOnNodeRenderedChanged(VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs: catch 'NodeRenderedChanged' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visu_pair_build(dataObj);
}
static void createPairsOnNodePositionChanged(VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs: catch 'NodePositionChanged' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visu_pair_build(dataObj);
}
static void createPairsOnNodePopulationChanged(VisuData *dataObj,
					       int *nodes _U_, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs: catch 'NodePopulation[In|De]crease' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visu_pair_build(dataObj);
}
static void createPairsOnResources(GObject *obj _U_, VisuData *dataObj _U_,
				   gpointer data _U_)
{
  rebuildPairsNeeded = TRUE;
}

static void freeHistoData(gpointer data)
{
  DBG_fprintf(stderr, "Visu Pairs: free '%s' data.\n", BONDHISTOGRAM_ID);
  g_free(((VisuPairDistribution*)data)->histo);
  g_free(data);
}

/**
 * visu_pair_getDistanceDistribution:
 * @pair: a #VisuPair ;
 * @dataObj: a #VisuData ;
 * @step: a float for the distance mesh (negative value to use
 * built-in default) ;
 * @min: a float for the minimum scanning value (negative value to use
 * built-in default).
 * @max: a float for the maximum scanning value (negative value to use
 * built-in default).
 * 
 * This will compute the distnace distribution of nodes for the given
 * @pair.
 *
 * Returns: a structure defining the distance distribution. This
 * structure is private and should not be freed.
 */
VisuPairDistribution* visu_pair_getDistanceDistribution(VisuPair *pair,
						       VisuData *dataObj,
						       float step,
						       float min, float max)
{
  VisuPairDistribution *dd;
  guint nRef, i;
  int *iEle, iEle1, iEle2;
  VisuDataIter iter1, iter2;
  float d2, inv;
  float xyz1[3], xyz2[3];
#if DEBUG == 1
  GTimer *timer;
  gulong fractionTimer;
#endif

  g_return_val_if_fail(pair && IS_VISU_DATA_TYPE(dataObj), (VisuPairDistribution*)0);

#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif

  /* We create the storage structure. */
  dd = (VisuPairDistribution*)
    visu_pair_getProperty(pair, BONDHISTOGRAM_ID);
  if (dd)
    g_free(dd->histo);
  else
    {
      dd = g_malloc(sizeof(VisuPairDistribution));
      visu_pair_setProperty(pair, BONDHISTOGRAM_ID,
			   (gpointer)dd, freeHistoData);
    }
  dd->ele1 = pair->ele1;
  dd->ele2 = pair->ele2;
  dd->nNodesEle1 = 0;
  dd->nNodesEle2 = 0;
  dd->stepValue = (step > 0.f)?step:BONDHISTOGRAM_STEP;
  dd->initValue = (min > 0.f)?min:BONDHISTOGRAM_MIN;
  dd->nValues   = (int)((((max > 0.f)?max:BONDHISTOGRAM_MAX) -
			 dd->initValue) / dd->stepValue) + 1;
  dd->histo     = g_malloc0(sizeof(int) * dd->nValues);

  DBG_fprintf(stderr, "Visu Pairs: compute distance distribution (%p %g %d).\n",
	      (gpointer)dd, dd->stepValue, dd->nValues);

  /* We compute the distribution. */
  visu_data_iterNew(dataObj, &iter1);
  iEle = (int*)g_hash_table_lookup(dataObj->fromVisuElementToInt, pair->ele1);
  g_return_val_if_fail(iEle, (VisuPairDistribution*)0);
  iEle1 = *iEle;
  iEle = (int*)g_hash_table_lookup(dataObj->fromVisuElementToInt, pair->ele2);
  g_return_val_if_fail(iEle, (VisuPairDistribution*)0);
  iEle2 = *iEle;
  inv = 1.f / dd->stepValue;
  iter1.iElement = iEle1;
  iter1.element = pair->ele1;
  for(visu_data_iterRestartNode(dataObj, &iter1); iter1.node;
      visu_data_iterNextNodeOriginal(dataObj, &iter1))
    {
      if (!iter1.node->rendered)
	continue;
      dd->nNodesEle1 += 1;
      visu_data_getNodePosition(dataObj, iter1.node, xyz1);

      visu_data_iterNew(dataObj, &iter2);
      iter2.iElement = iEle2;
      iter2.element = pair->ele2;
/*       fprintf(stderr, "## %d\n", dd->histo[177]); */
      for(visu_data_iterRestartNode(dataObj, &iter2); iter2.node;
	  visu_data_iterNextNode(dataObj, &iter2))
	{
	  if (!iter2.node->rendered)
	    continue;
	  /* Don't count the inter element pairs two times. */
	  if (iter1.element == iter2.element &&
	      iter2.node == iter1.node)
	    continue;

	  visu_data_getNodePosition(dataObj, iter2.node, xyz2);
	  d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
	    (xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
	    (xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
	  /* We put the distance into the histogram. */
	  dd->histo[MIN((guint)((sqrt(d2) - dd->initValue) * inv), dd->nValues - 1)] += 1;
/* 	  fprintf(stderr, "%d-%d %d\n", iter1.node->number, iter2.node->number, dd->histo[177]); */
	}
/*       fprintf(stderr, "-> %d\n", dd->histo[177]); */
    }
  for(visu_data_iterRestartNode(dataObj, &iter2); iter2.node;
      visu_data_iterNextNode(dataObj, &iter2))
    if (iter2.node->rendered)
      dd->nNodesEle2 += 1;
  if (iter1.element == iter2.element)
    for (i = 0; i < dd->nValues; i++)
      dd->histo[i] /= 2;

#if DEBUG == 1
  g_timer_stop(timer);
#endif

  if (DEBUG)
    for (nRef = 0; nRef < dd->nValues; nRef++)
      fprintf(stderr, " | %03d -> %6.3f, %5d\n", nRef,
	      dd->initValue + dd->stepValue * nRef, dd->histo[nRef]);

#if DEBUG == 1
  fprintf(stderr, "Visu Pairs: distances analysed in %g micro-s.\n",
	  g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif

  return dd;
}

/**
 * visu_pair_distribution_getNextPick:
 * @dd: a #VisuPairDistribution object.
 * @startStopId: two ids.
 * @integral: a location for a guint value, can be NULL.
 *
 * Try to find the next pick in the distribution. A pick is a group of
 * consecutive non-null values, with a significant integral. On enter,
 * @startStopId contains the span to look into for the pick, and on
 * output, it contains the span of the pick itself.
 * 
 * Since: 3.6
 * 
 * Returns: TRUE if a pick is found.
 */
gboolean visu_pair_distribution_getNextPick(VisuPairDistribution *dd,
                                            guint startStopId[2], guint *integral,
                                            guint *max, guint *posMax)
{
  float min, start, stop;
  guint i, iStart, iStop, sum, _posMax, _max;

  g_return_val_if_fail(dd, FALSE);
  g_return_val_if_fail(startStopId[1] < dd->nValues, FALSE);

  iStart = startStopId[0];
  iStop  = startStopId[1];
  _max = 0;
  _posMax = 0;
  min = 1.5f * MIN(dd->nNodesEle1, dd->nNodesEle2);
  DBG_fprintf(stderr, "Visu Pairs: look for one pick in %d-%d.\n",
              startStopId[0], startStopId[1]);
  do
    {
      min  *= 0.5f;
      start = -1.f;
      stop  = -1.f;
      sum   = 0;
      for (i = startStopId[0] ; i < startStopId[1]; i++)
        {
          if (start < 0.f && dd->histo[i] > 0)
            {
              start = dd->stepValue * i + dd->initValue;
              sum = dd->histo[i];
              iStart = i;
              _max = dd->histo[i];
              _posMax = i;
            }
          else if (start > 0.f)
            {
              if (dd->histo[i] == 0)
                {
                  if (sum >= min)
                    {
                      stop = dd->stepValue * i + dd->initValue;
                      iStop = i;
                      break;
                    }
                  else
                    start = -1.f;
                }
              else
                {
                  sum += dd->histo[i];
                  if (dd->histo[i] > _max)
                    {
                      _max = dd->histo[i];
                      _posMax = i;
                    }
                }
            }
        }
      DBG_fprintf(stderr, "Visu Pairs: found one pick at %d-%d (%d).\n",
                  iStart, iStop, sum);
    }
  while (start < 0.f && min > 0.1f * MIN(dd->nNodesEle1, dd->nNodesEle2));
  DBG_fprintf(stderr, "Visu Pairs: set start and stop at %f, %f (%f %d).\n",
              start, stop, min, sum);
  if (start <= 0.f || stop <= 0.f)
    return FALSE;

  startStopId[0] = iStart;
  startStopId[1] = iStop;
  if (integral)
    *integral = sum;
  if (max)
    *max = _max;
  if (posMax)
    *posMax = _posMax;

  return TRUE;
}

/**
 * visu_pair_build:
 * @dataObj: the #VisuData object to create pairs for.
 *
 * This methods recreates the OpenGL list of the #VisuExtension associated
 * to the pairs.
 * Thus it is the heart of the pairs drawing. When called, if a
 * valid #VisuPairExtension has been set with setPairsMethod(), it takes all the
 * nodes and compute all the distances between two different nodes. If this distance
 * is compatible with the distance of drawn pairs for the two elements it calls
 * the VisuPairDrawFunc() for these two nodes. WARNING! this method is very expensive
 * in computing cost.
 *
 * Returns: TRUE if the OpenGLAskForReDraw signal show be emitted or not.
 */
gboolean visu_pair_build(VisuData *dataObj)
{
  VisuDataIter iter1, iter2;
  float d2, d2min, d2max, d2min_buffered, d2max_buffered, l;
  VisuPair *pair;
  VisuPairData *data;
  float xyz1[3], xyz2[3], alpha;
  VisuOpenGLView *view;
  char distStr[8];
  int *bond1, *bond2;
  GList *tmpLst;
  gboolean useBond;
  VisuPairExtension *ext;

  if (!dataObj)
    {
      glDeleteLists(openGlListPairsId, 1);
      rebuildPairsNeeded = TRUE;
      return TRUE;
    }

  if (!extensionPairs->used)
    {
      DBG_fprintf(stderr, "Visu Pairs: aborting creating pairs, extension is not used.\n");
      return FALSE;
    }
  if (!rebuildPairsNeeded)
    {
      DBG_fprintf(stderr, "Visu Pairs: aborting creating pairs, pairs are up to date.\n");
      return TRUE;
    }

  DBG_fprintf(stderr, "Visu Pairs: creating pairs between elements (%s).\n",
	      defaultPairMethod->name);

  view = visu_data_getOpenGLView(dataObj);

  glDeleteLists(openGlListPairsId, 1);
  g_return_val_if_fail(defaultPairMethod, FALSE);

  rebuildPairsNeeded = FALSE;
  glNewList(openGlListPairsId, GL_COMPILE);

  /* We get the counting array for bonds. */
  if (visu_node_array_getProperty(visu_data_getNodeArray(dataObj),
			   (const gchar*)BONDNUMBER_ID))
    useBond = TRUE;
  else
    useBond = FALSE;

  visu_data_iterNew(dataObj, &iter1);
  visu_data_iterNew(dataObj, &iter2);
  for(visu_data_iterStart(dataObj, &iter1); iter1.element;
      visu_data_iterNextElement(dataObj, &iter1))
    {
      if (!visu_element_getRendered(iter1.element))
	continue;

      /* Initialise the bond count. */
      if (useBond)
	for(visu_data_iterRestartNode(dataObj, &iter1); iter1.node;
	    visu_data_iterNextNode(dataObj, &iter1))
	  {
	    bond1 = getBond(dataObj, iter1.node);
	    *bond1 = 0;
	  }

      for(visu_data_iterStart(dataObj, &iter2);
	  iter2.element && iter2.iElement <= iter1.iElement ;
	  visu_data_iterNextElement(dataObj, &iter2))
	{
	  if (!visu_element_getRendered(iter2.element))
	    continue;
	  
	  DBG_fprintf(stderr, "Visu Pairs: draw pairs between '%s' and '%s'.\n",
		      iter1.element->name, iter2.element->name);
	  pair = visu_pair_getPair(iter1.element, iter2.element);
	  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
	    {
	      data = (VisuPairData*)tmpLst->data;
	      if (!data->drawn)
		continue;
	      d2min = data->minMax[VISU_PAIR_DISTANCE_MIN] * data->minMax[VISU_PAIR_DISTANCE_MIN];
	      d2max = data->minMax[VISU_PAIR_DISTANCE_MAX] * data->minMax[VISU_PAIR_DISTANCE_MAX];
	      if(d2min >= d2max || d2max <= 0.)
		continue;

              ext = (data->method)?data->method:defaultPairMethod;
	      l = data->minMax[VISU_PAIR_DISTANCE_MAX] - data->minMax[VISU_PAIR_DISTANCE_MIN];
	      d2min_buffered = (data->minMax[VISU_PAIR_DISTANCE_MIN] - 0.15 * l);
	      d2min_buffered *= d2min_buffered;
	      d2max_buffered = (data->minMax[VISU_PAIR_DISTANCE_MAX] + 0.15 * l);
	      d2max_buffered *= d2max_buffered;
	      if (ext->beginDrawingPairs)
		ext->beginDrawingPairs(iter1.element, iter2.element, data);

	      for(visu_data_iterRestartNode(dataObj, &iter1); iter1.node;
		  visu_data_iterNextNode(dataObj, &iter1))
		{
		  if (!iter1.node->rendered)
		    continue;

		  bond1 = (useBond)?getBond(dataObj, iter1.node):(int*)0;
		  for(visu_data_iterRestartNode(dataObj, &iter2); iter2.node;
		      visu_data_iterNextNode(dataObj, &iter2))
		    {
		      if (!iter2.node->rendered)
			continue;
		      /* Don't draw the inter element pairs two times. */
		      if (iter1.element == iter2.element &&
			  iter2.node >= iter1.node)
			break;

		      visu_data_getNodePosition(dataObj, iter1.node, xyz1);
		      visu_data_getNodePosition(dataObj, iter2.node, xyz2);
		      d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
			(xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
			(xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
		      if(d2 <= 0. || d2 < d2min_buffered || d2 > d2max_buffered)
			continue;

		      if (d2 < d2min)
			alpha = (d2 - d2min_buffered) /
			  (d2min - d2min_buffered);
		      else if (d2 > d2max)
			alpha = (d2max_buffered - d2) /
			  (d2max_buffered - d2max);
		      else
			{
			  alpha = 1.f;
			  /* Update bond count. */
			  if (bond1)
			    (*bond1) += 1;
			  bond2 = (useBond)?getBond(dataObj, iter2.node):(int*)0;
			  if (bond2)
			    (*bond2) += 1;
			}

		      ext->drawPairs(iter1.element, iter2.element, data, view,
                                     xyz1[0], xyz1[1], xyz1[2],
                                     xyz2[0], xyz2[1], xyz2[2],
                                     d2, alpha);
		      if (data->printLength)
			{
			  glRasterPos3f((xyz1[0] + xyz2[0]) / 2.,
					(xyz1[1] + xyz2[1]) / 2.,
					(xyz1[2] + xyz2[2]) / 2.);
			  sprintf(distStr, "%7.3f", sqrt(d2));
			  openGLText_drawChars(distStr, TEXT_NORMAL);
			}
		    }
		}
	      if (ext->endDrawingPairs)
		ext->endDrawingPairs(iter1.element, iter2.element, data);
	    }
	}
    }

  glEndList();

  /* Emit the valueChanged signal of dataNode. */
  visu_data_node_emitValueChanged(dataNode, dataObj);
  
  return TRUE;
}

static void rebuildPairs(VisuData *dataObj)
{
  if (dataObj && extensionPairs->used)
    {
      openGLText_rebuildFontList();
      /* Force to recreate pairs. */
      rebuildPairsNeeded = TRUE;
      visu_pair_build(dataObj);
    }
}

static int* getBond(VisuData *dataObj, VisuNode *node)
{
  int *bond;
  GValue val = {0, {{0}, {0}}};

  g_value_init(&val, G_TYPE_POINTER);
  visu_node_array_getPropertyValue(visu_data_getNodeArray(dataObj), node,
			    BONDNUMBER_ID, &val);
  bond = (int*)g_value_get_pointer(&val);

  if (!bond)
    {
      bond = (int*)newOrCopyData((gconstpointer)0, (gpointer)0);
      g_value_set_pointer(&val, (gpointer)bond);
      visu_node_setpropertyValue(visu_data_getNodeArray(dataObj), node,
				BONDNUMBER_ID, &val);
    }
  return bond;
}

static void freeData(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(int), obj);
#else
  g_free(obj);
#endif
}
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data _U_)
{
  int* data;

#if GLIB_MINOR_VERSION > 9
  data = g_slice_alloc(sizeof(int));
#else
  data = g_malloc(sizeof(int));
#endif
  if (orig)
    *data = *(int*)orig;
  else
    *data = 0;
    
  return (gpointer)data;
}

static void onNewData(GObject* obj _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "NodeRenderedChanged",
		   G_CALLBACK(createPairsOnNodeRenderedChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePositionChanged",
		   G_CALLBACK(createPairsOnNodePositionChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
                   G_CALLBACK(createPairsOnNodePopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationIncrease",
                   G_CALLBACK(createPairsOnNodePopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDefined",
		   G_CALLBACK(onNodesReady), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLFacetteChanged",
		   G_CALLBACK(facetteChanged), (gpointer)0);
}

static void onNodesReady(VisuData *dataObj, VisuNodeArray *nodes, gpointer data _U_)
{
  if (nodes)
    {
      /* Create the Node property to count the number of bonds. */
      DBG_fprintf(stderr, "Visu Pairs: create the bond node property.\n");
      visu_node_property_newPointer(nodes, BONDNUMBER_ID, freeData,
				  newOrCopyData, (gpointer)0);
      visu_data_node_setUsed(dataNode, dataObj, (extensionPairs->used)?1:0);
    }    
}

static void onDataReady(GObject* obj _U_, VisuData *dataObj, gpointer data _U_)
{
  rebuildPairsNeeded = TRUE;

  if (extensionPairs->used)
    visu_pair_build(dataObj);
}

static void facetteChanged(VisuData* obj, gpointer data _U_)
{
  if (extensionPairs->used && sensitiveToFacette)
    visu_pair_build(obj);
}




/*****************************************/
/* Methods to organize pairs extensions. */
/*****************************************/

/**
 * visu_pair_extension_new:
 * @name: name of the extension (must be non null) ;
 * @printName: a string to label the method that can be safely translated ;
 * @description: a brief description of the extension (can be null) ;
 * @sensitive: a boolean 0 or 1 ;
 * @start: a VisuPairStartEndFunc() method or NULL ;
 * @end: a VisuPairStartEndFunc() method or NULL ;
 * @draw: a VisuPairDrawFunc() method (not NULL).
 *
 * This creates a new pairs extension. Such an extension describes how to draw
 * links (called pairs) between elements. The @sensitive argument is to inform if
 * pairs must be redrawn when the OpenGL engine sends the OpenGLFacetteChanged signal.
 *
 * Returns: the new VisuPairExtension or null if something wrong happens.
 */
VisuPairExtension* visu_pair_extension_new(const char* name, const char* printName,
                                        const char* description, gboolean sensitive,
                                        VisuPairStartEndFunc start,
                                        VisuPairStartEndFunc end,
                                        VisuPairDrawFunc draw)
{
  VisuPairExtension *extension;

  extension = g_malloc(sizeof(VisuPairExtension));
  extension->name        = g_strdup(name);
  extension->printName   = g_strdup(printName);
  extension->description = g_strdup(description);
  
  extension->sensitiveToFacette = sensitive;

  extension->beginDrawingPairs = start;
  extension->endDrawingPairs   = end;
  extension->drawPairs         = draw;

  return extension;
}
/**
 * visu_pair_extension_free:
 * @extension: the extension to delete.
 *
 * Free all the allocated attributes of the specified method.
 */
void visu_pair_extension_free(VisuPairExtension* extension)
{
  if (!extension)
    return;
  if (extension->name)
    g_free(extension->name);
  if (extension->printName)
    g_free(extension->printName);
  if (extension->description)
    g_free(extension->description);
  g_free(extension);
}
/**
 * visu_pair_extension_add:
 * @extension: an extension.
 *
 * A method used by user to registered a new extension.
 */
void visu_pair_extension_add(VisuPairExtension *extension)
{
  DBG_fprintf(stderr, "Visu Pairs: registering a new pairs extension ... ");
  g_return_if_fail(extension && extension->drawPairs && extension->name);

  availableVisuPairExtensions = g_list_append(availableVisuPairExtensions, (gpointer)extension);
  DBG_fprintf(stderr, "'%s' (%d).\n", extension->name, GPOINTER_TO_INT(extension));
}
/**
 * visu_pair_extension_setDefault:
 * @extension: a #VisuPairExtension object.
 *
 * Choose the method used to draw pairs. If necessary, visu_pair_build() are called.
 *
 * Returns: TRUE if the OpenGLAskForReDraw signal show be emitted or not.
 */
gboolean visu_pair_extension_setDefault(VisuPairExtension *extension)
{
  DBG_fprintf(stderr, "Visu Pairs: set pairs drawing method to '%s'"
	      " (previous '%s').\n", extension->name, defaultPairMethod->name);

  if (extension == defaultPairMethod)
    return FALSE;
  
  if (defaultPairMethod &&
      defaultPairMethod->sensitiveToFacette)
    sensitiveToFacette -= 1;
  defaultPairMethod = extension;
  if (defaultPairMethod &&
      defaultPairMethod->sensitiveToFacette)
    sensitiveToFacette += 1;

  rebuildPairsNeeded = TRUE;

  if (!extensionPairs->used)
    return FALSE;
  return TRUE;
}
/**
 * visu_pair_extension_getDefault:
 *
 * If some process need to know the current #VisuPairExtension. Such extension has
 * been set with setPairsMethod().
 *
 * Returns: the current #VisuPairExtension, NULL if none has been set.
 */
VisuPairExtension* visu_pair_extension_getDefault()
{
  return defaultPairMethod;
}
/**
 * visu_pair_extension_getAllMethods:
 * 
 * Useful to know all #VisuPairExtension.
 *
 * Returns: a list of all the known #VisuPairExtension. This list should
 *          be considered read-only.
 */
GList* visu_pair_extension_getAllMethods()
{
  return availableVisuPairExtensions;
}
/**
 * visu_pair_extension_getByName:
 * @name: a method name (untranslated).
 * 
 * Useful to know a #VisuPairExtension when knowing its name.
 *
 * Returns: a #VisuPairExtension, or NULL if not any matchs.
 *
 * Since: 3.6
 */
VisuPairExtension* visu_pair_extension_getByName(const gchar *name)
{
  GList *pairsMethods;
  gchar *name_;

  name_ = g_strstrip(g_strdup(name));
  pairsMethods = availableVisuPairExtensions;
  while (pairsMethods &&
         g_ascii_strcasecmp(((VisuPairExtension*)pairsMethods->data)->name, name_))
    pairsMethods = g_list_next(pairsMethods);
  g_free(name_);
  return (pairsMethods)?(VisuPairExtension*)pairsMethods->data:(VisuPairExtension*)0;
}




static void foreachLevel2(gpointer key _U_, gpointer value, gpointer userData)
{
  VisuPair *pair;
  GList *tmpLst;
  struct foreachPairsData_struct *storage;

  pair = (VisuPair*)value;
  storage = (struct foreachPairsData_struct *)userData;
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    storage->func(pair->ele1, pair->ele2,
		  (VisuPairData*)tmpLst->data, storage->userData);
}
void visu_pair_foreach(VisuPairForeachFunc whatToDo, gpointer userData)
{
  struct foreachPairsData_struct storage;

  storage.func = whatToDo;
  storage.userData = userData;
  g_hash_table_foreach(DminDmax, (GHFunc)foreachLevel2, (gpointer)(&storage));
}


/*****************************************/
/* Dealing with parameters and resources */
/*****************************************/

gboolean visu_pair_readLinkFromTokens(gchar **tokens, int *index, VisuPairData **data,
				     int lineId, GError **error)
{
  VisuElement *ele[2];
  float minMax[2];
  gboolean res;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(data, FALSE);
  g_return_val_if_fail(tokens && index, FALSE);
  
  /* Read the two elements. */
  res = tool_configFile_readElementFromTokens(tokens, index, ele, 2, lineId, error);
  if (!res)
    return FALSE;
  /* Read the two distances. */
  res = tool_configFile_readFloatFromTokens(tokens, index, minMax, 2, lineId, error);
  if (!res)
    return FALSE;
  
  /* Check the values. */
  res = FALSE;
  res = res || tool_configFile_clampFloat(&minMax[0], minMax[0], 0., -1.);
  res = res || tool_configFile_clampFloat(&minMax[1], minMax[1], 0., -1.);
  if (res)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d, given distance"
			     " are out of bounds (should be positive).\n"), lineId);
      return FALSE;
    }

  /* Set the values. */
  *data = visu_pair_getPairData(ele[0], ele[1], minMax);
  g_return_val_if_fail(*data, FALSE);

  return TRUE;
}

static gboolean readPairLink(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  int id;
  gboolean flags[2], res;
  VisuPairData *data;
  float rgb[4];
  ToolColor *color;
  gchar *methodName;
  VisuPairExtension *method;

  g_return_val_if_fail(nbLines == 2, FALSE);

  tokens = g_strsplit_set(lines[0], " \t\n", MAX_LINE_LENGTH);
  id = 0;
  if (!visu_pair_readLinkFromTokens(tokens, &id, &data, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  /* Read second line with basic data. */
  tokens = g_strsplit_set(lines[1], " \t\n", MAX_LINE_LENGTH);
  id = 0;
  /* Read the color. */
  if (!tool_configFile_readFloatFromTokens(tokens, &id, rgb, 3, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  /* Read the flags drawn and printLength. */
  if (!tool_configFile_readBooleanFromTokens(tokens, &id, flags, 2, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  /* Read the specific method, if any. */
  methodName = g_strjoinv(" ", tokens + id);
  if (methodName && methodName[0])
    {
      method = visu_pair_extension_getByName(methodName);
      if (!method)
	{
	  *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			       _("Parse error at line %d, the method"
				 " '%s' is unknown.\n"), position + 1, methodName);
          g_free(methodName);
          g_strfreev(tokens);
	  return FALSE;
	}
    }
  else
    method = (VisuPairExtension*)0;
  if (methodName)
    g_free(methodName);
  /* Global free. */
  g_strfreev(tokens);

  /* Check the values. */
  res = tool_configFile_clampFloat(&rgb[0], rgb[0], 0., 1.) ||
    tool_configFile_clampFloat(&rgb[1], rgb[1], 0., 1.) ||
    tool_configFile_clampFloat(&rgb[2], rgb[2], 0., 1.);
  if (res)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d, 3 floating points"
			     "(0 <= v <= 1) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCES_PAIRS_DATA);
      return FALSE;
    }


  /* Set the values. */
  rgb[3] = 1.f;
  color = tool_color_getByValues((int*)0, rgb[0], rgb[1], rgb[2], rgb[3]);
  if (!color)
    color = tool_color_addFloatRGBA(rgb, (int*)0);
  visu_pair_data_setColor(data, color);
  visu_pair_data_setDrawn(data, flags[0]);
  visu_pair_data_setPrintLength(data, flags[1]);
  visu_pair_data_setDrawMethod(data, method);

  return TRUE;
}

/* This function details how to read the parameter FLAG_PARAMETER_PAIRS. */
static gboolean readPairsAreOn(gchar **lines, int nbLines, int position,
			       VisuData *dataObj _U_, GError **error)
{
  gboolean val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &val, 1, error))
    return FALSE;
  visu_pair_setStatus(val);

  return TRUE;
}
static gboolean readFavPairsMethod(gchar **lines, int nbLines, int position,
				   VisuData *dataObj _U_, GError **error)
{
  VisuPairExtension *method;

  g_return_val_if_fail(nbLines == 1, FALSE);

  lines[0] = g_strstrip(lines[0]);

  if (!lines[0][0])
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d, 1 string value must appear"
			     " after the %s markup.\n"), position,
			   FLAG_RESOURCES_FAV_PAIRS);
      return FALSE;
    }
  else
    {
      method = visu_pair_extension_getByName(lines[0]);
      if (!method)
	{
	  *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			       _("Parse error at line %d, the method"
				 " '%s' is unknown.\n"), position, lines[0]);
	  return FALSE;
	}
      visu_pair_extension_setDefault(method);
    }

  return TRUE;
}
/* These functions save the resources. */
static void exportLevel1(gpointer key, gpointer value, gpointer user_data)
{
  VisuPair *pair;
  VisuPairData *data;
  struct foreachFuncExport_struct *str;
  GList *tmpLst;
  ToolColor *color;
  
  pair = (VisuPair*)value;
  str = (struct foreachFuncExport_struct*)user_data;
  
  /* We export only if the two elements of the pair are used in the given dataObj. */
  if (str->dataObj)
    {
      /* We test the first element. */
      if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt,
			       (gpointer)pair->ele1))
	return;
      /* We test the second element. */
      if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt,
			       (gpointer)pair->ele2))
	return;
    }

  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      data = (VisuPairData*)tmpLst->data;
      color = g_hash_table_lookup(data->properties, (gpointer)"color");
      g_return_if_fail(color);
      g_string_append_printf(str->data, "%s:\n", FLAG_RESOURCES_PAIR_LINK);
      g_string_append_printf(str->data, "    %s %4.3f %4.3f\n"
			     "    %4.3f %4.3f %4.3f  %d  %d  %s\n", (gchar*)key,
			     data->minMax[VISU_PAIR_DISTANCE_MIN], data->minMax[VISU_PAIR_DISTANCE_MAX],
			     color->rgba[0], color->rgba[1], color->rgba[2],
			     data->drawn, data->printLength,
                             (data->method)?data->method->name:
                             defaultPairMethod->name);
    }  
}
static void exportResourcesPairs(GString *data, VisuData *dataObj)
{
  struct foreachFuncExport_struct str;
  GList *pairsMeth;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_PAIRS);
  g_string_append_printf(data, "%s:\n    %i\n", FLAG_RESOURCES_PAIRS,
			 (int)extensionPairs->used);

  if (defaultPairMethod)
    {
      g_string_append_printf(data, "# %s (", DESC_RESOURCES_FAV_PAIRS);
      pairsMeth = availableVisuPairExtensions;
      while (pairsMeth)
	{
	  g_string_append_printf(data, "'%s'", ((VisuPairExtension*)pairsMeth->data)->name);
	  if (pairsMeth->next)
	    g_string_append_printf(data, ", ");
	  pairsMeth = g_list_next(pairsMeth);
	}
      g_string_append_printf(data, ")\n");
      g_string_append_printf(data, "%s:\n    %s\n", FLAG_RESOURCES_FAV_PAIRS,
			     defaultPairMethod->name);
    }

  g_string_append_printf(data, "# %s\n", DESC1_RESOURCES_PAIR_LINK);
  g_string_append_printf(data, "# %s\n", DESC2_RESOURCES_PAIR_LINK);
  str.data = data;
  str.dataObj = dataObj;
  g_hash_table_foreach(DminDmax, (GHFunc)exportLevel1, (gpointer)&str);

  g_string_append_printf(data, "\n");
}






/* OBSOLETE function, keep for backward compatibility. */
static gboolean readPairsData(gchar **lines, int nbLines, int position,
			      VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele[2];
  int res, token;
  float rgb[4];
  float minMax[2];
  VisuPairData *data;
  gchar **tokens;
  ToolColor *color;

  /* Tokenize the line of values. */
  tokens = g_strsplit_set(g_strchug(lines[0]), " \n", MAX_LINE_LENGTH);
  token = 0;

  /* Get the two elements. */
  if (!tool_configFile_readElementFromTokens(tokens, &token, ele, 2, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }

  /* Read 5 floats. */
  if (!tool_configFile_readFloatFromTokens(tokens, &token, minMax, 2, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  if (!tool_configFile_readFloatFromTokens(tokens, &token, rgb, 3, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  res = tool_configFile_clampFloat(&rgb[0], rgb[0], 0., 1.) ||
    tool_configFile_clampFloat(&rgb[1], rgb[1], 0., 1.) ||
    tool_configFile_clampFloat(&rgb[2], rgb[2], 0., 1.) ||
    tool_configFile_clampFloat(&minMax[0], minMax[0], 0., -1.) ||
    tool_configFile_clampFloat(&minMax[1], minMax[1], 0., -1.);
  if (res)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d, 5 floating points"
			     " must appear after the %s markup.\n"),
			   position, FLAG_RESOURCES_PAIRS_DATA);
      return FALSE;
    }
  data = visu_pair_getPairData(ele[0], ele[1], minMax);
  g_return_val_if_fail(data, FALSE);

  rgb[3] = 1.f;
  color = tool_color_getByValues((int*)0, rgb[0], rgb[1], rgb[2], rgb[3]);
  if (!color)
    color = tool_color_addFloatRGBA(rgb, (int*)0);
  visu_pair_data_setColor(data, color);
  
  return TRUE;
}
