/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "visu_basic.h"
#include "visu_object.h"
#include "visu_rendering.h"
#include "visu_extension.h"
#include "visu_data.h"
#include "visu_elements.h"
#include "visu_commandLine.h"
#include "opengl.h"
#include "visu_dump.h"
#include "visu_pairs.h"
#include "visu_tools.h"
#include "visu_configFile.h"
#include "extraFunctions/dataFile.h"
#include "extraFunctions/plane.h"
#include "extraFunctions/pot2surf.h"
#include "extraFunctions/extraNode.h"
#include "extraFunctions/map.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/text.h"
#include "openGLFunctions/interactive.h"
#include "OSOpenGL/visu_openGL.h"
#include "extensions/fogAndBGColor.h"
#include "extensions/scale.h"
#include "extensions/marks.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/geometry.h"
#include "renderingMethods/renderingAtomic.h"
#include "renderingMethods/renderingSpin.h"

#include <stdio.h>
#include <locale.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */

/* For the GdkPixbuf, to be removed later. */
#include <gtk/gtk.h>

/**
 * SECTION:visu_basic
 * @short_description: Main functions of V_Sim (except graphical
 * ones).
 *
 * <para>There are here the main functions of V_Sim (except for
 * graphical methods) such as open file.</para>
 */

/* Valeurs par d�faut pour les chemins utiles dans V_Sim. */
#if SYSTEM_X11 == 1
#define V_SIM_DATA_DIR_DEFAULT    "/usr/local/share/v_sim"
#define V_SIM_LEGAL_DIR_DEFAULT   "/usr/local/share/v_sim"
#define V_SIM_PIXMAPS_DIR_DEFAULT "/usr/local/share/v_sim/pixmaps"
#define V_SIM_ICONS_DIR_DEFAULT   "/usr/local/share/icons"
#define V_SIM_PLUGINS_DIR_DEFAULT "/usr/local/lib/v_sim/plug-ins"
#define V_SIM_LOCALE_DIR_DEFAULT  "/usr/local/share/locale"
#endif
#if SYSTEM_WIN32 == 1
#define V_SIM_DATA_DIR_DEFAULT    "C:\\PROGRAM FILES\\V_Sim\\share\\v_sim"
#define V_SIM_LEGAL_DIR_DEFAULT   "C:\\PROGRAM FILES\\V_Sim\\share\\doc\\v_sim"
#define V_SIM_PIXMAPS_DIR_DEFAULT "C:\\PROGRAM FILES\\V_Sim\\share\\v_sim\\pixmaps"
#define V_SIM_ICONS_DIR_DEFAULT   "C:\\PROGRAM FILES\\V_Sim\\share\\icons"
#define V_SIM_PLUGINS_DIR_DEFAULT "C:\\PROGRAM FILES\\V_Sim\\lib\\v_sim\\plug-ins"
#define V_SIM_LOCALE_DIR_DEFAULT  "C:\\PROGRAM FILES\\V_Sim\\share\\locale"
#endif
static gchar *exeLocation = NULL;
static gchar *v_sim_data_dir = NULL;
static gchar *v_sim_legal_dir = NULL;
static gchar *v_sim_pixmaps_dir = NULL;
static gchar *v_sim_icons_dir = NULL;
static gchar *v_sim_local_conf_dir = NULL;
static gchar *v_sim_old_local_conf_dir = NULL;
static gchar *v_sim_plugins_dir = NULL;
static gchar *v_sim_locale_dir = NULL;
static ToolUnits preferedUnit;
static gboolean loadCancel, loadCurrent = FALSE;
static GFunc loadMessage = NULL;
static gpointer loadData = NULL;

/* Local methods. */
static gboolean dumpData(gpointer data);
static void freeExtension(gpointer data);
static void onNewData(GObject* obj, VisuData *dataObj, gpointer data);
static void onNodesSet(VisuData *data, VisuNodeArray *nodes, gpointer user_data);
static void setToolFileFormatOption(gpointer key, gpointer value, gpointer data);

#define FLAG_PARAMETER_UNIT   "main_unit"
#define DESC_PARAMETER_UNIT   "Define the prefered unit to display files ; string"
static gboolean readUnit(gchar **lines, int nbLines, int position,
			 VisuData *dataObj, GError **error);
static void exportParameters(GString *data, VisuData *dataObj);

/**
 * visu_basic_getLoadCancel:
 *
 * V_Sim implements cancelation load process by setting a flag that can
 * be check by running routine. Use this method to check the flag
 * value (not thread safe). See visu_basic_setLoadCancel() to set the
 * cancelation flag.
 *
 * Since: 3.6
 *
 * Returns: the cancelation status.
 */
gboolean visu_basic_getLoadCancel(void)
{
  return loadCancel;
}
/**
 * visu_basic_setLoadCancel:
 * @status: a boolean.
 *
 * V_Sim implements cancelation load process by setting a flag that can
 * be check by running routine. Use this method to set the flag
 * value (not thread safe). See visu_basic_getLoadCancel() to check the
 * cancelation flag.
 *
 * Since: 3.6
 */
void visu_basic_setLoadCancel(gboolean status)
{
  loadCancel = status;
}
/**
 * visu_basic_setLoadMessageFunc:
 * @func: (scope call): a function to print a message.
 * @data: user data.
 *
 * When a load process is running, on can defined a message function
 * that may be called to output message to the user using
 * visu_basic_setLoadMessage().
 *
 * Since: 3.6
 */
void visu_basic_setLoadMessageFunc(GFunc func, gpointer data)
{
  loadMessage = func;
  loadData = data;
}
/**
 * visu_basic_setLoadMessage:
 * @mess: a string.
 *
 * If a message function on load action has been set by
 * visu_basic_setLoadMessageFunc(), then the given @mess is given as
 * argument to this function.
 *
 * Since: 3.6
 */
void visu_basic_setLoadMessage(const gchar *mess)
{
  if (loadMessage)
    loadMessage((gpointer)mess, loadData);
}

/**
 * visu_basic_loadData:
 * @data: a #VisuData object ;
 * @format: (allow-none): a pointer on a format (can be NULL if format is to be
 * guessed) ;
 * @nSet: an integer ;
 * @error: a pointer to store a possible error, location must be initialized to (GError*)0.
 *
 * This calls the load method of the current rendering
 * method. Some informations may be store in @error->message if the returned
 * value is FALSE.
 * The file(s) which is(are) opened is(are) stored in the
 * #VisuData. The @nSet argument is used to load a specific set of
 * nodes if the input format supports it. If @nSet is 0, then the
 * default set of nodes is loaded.
 *
 * Returns: TRUE if everithing is OK, if FALSE, the @error is set and should be freed
 *          with g_error_free().
 */
gboolean visu_basic_loadData(VisuData *data, ToolFileFormat *format,
                             int nSet, GError **error)
{
  VisuRendering *method;
  gboolean res;
  ToolUnits unit;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(!loadCurrent, FALSE);

  DBG_fprintf(stderr, "##### Load process #####\n");
  loadCurrent = TRUE;

  method = visu_object_getRendering(VISU_INSTANCE);
  g_return_val_if_fail(method, FALSE);

#if DEBUG == 1
  g_mem_profile();
#endif
  DBG_fprintf(stderr, "Visu Basic: load dataset %d.\n", nSet);
  visu_basic_setLoadCancel(FALSE);
  res = visu_rendering_load(method, data, format, nSet, error);
  DBG_fprintf(stderr, "Visu Basic: load OK from rendering method,"
	      " continue with basic parts.\n");
#if DEBUG == 1
  g_mem_profile();
#endif

  /* Set flags... */
  if (res)
    {
      visu_data_setISubset(data, nSet);
      visu_data_setChangeElementFlag(data, FALSE);
      unit = visu_data_getUnit(data);
      if (preferedUnit != unit_undefined && unit != unit_undefined &&
	  unit != preferedUnit)
	visu_data_setUnit(data, preferedUnit);
    }

  DBG_fprintf(stderr, "##### Post-load signal emission #####\n");
  DBG_fprintf(stderr, "Visu Basic: emitting 'dataLoaded' signal for %p (%d).\n",
	      (gpointer)data, res);
  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATALOADED_SIGNAL],
		0, (res)?(gpointer)data:(gpointer)0, NULL);
  DBG_fprintf(stderr, "Visu Basic: emittion done.\n");

  loadCurrent = FALSE;
  return res;
}

/**
 * initVisuBasic:
 *
 * A call to this method is done at startup after having probe the locale of the file
 * system and having initialized the rendering window. It makes the following actions :
 * create the visu object to store the signals, initialize the module part (parameters,
 * and resources), initialize the dump part, the OpenGL part and its extensions, the
 * storage of elements and the 'colorize with data' part.
 */
void initVisuBasic(void)
{
  int res;
  VisuConfigFileEntry *entry;

  DBG_fprintf(stderr,"--- Initialising variables ---\n");
  /* We want to read . as floating point separator : in french it is , */
  setlocale(LC_NUMERIC, "C");

  /* Force the creation of the main object class. */
  g_type_class_ref(visuObject_get_type());
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onNewData), (gpointer)0);

  preferedUnit = unit_undefined;

  /* initialise OpenGL. */
  DBG_fprintf(stderr,"--- Initialising OpenGL ---\n");
  initOpengl();

  /* initialise les ajouts OpenGL. */
  DBG_fprintf(stderr,"--- Initialising OpenGL extensions ---\n");
  res = initVisuExtensions();
  if (!res)
    exit(1);

  /* initialise la gestion des liaisons. */
  DBG_fprintf(stderr,"--- Initialising pairs modules ---\n");
  res = initPairsModule();
  if (!res)
    exit(1);

  DBG_fprintf(stderr,"--- Initialising dataFile modules ---\n");
  res = initDataFileModule();
  if (!res)
    exit(1);

  /* initialise the scalar field handler. */
  DBG_fprintf(stderr,"--- Initialising scalar field module ---\n");
  scalarFieldInit();

  /* Initialise extra functions. */
  DBG_fprintf(stderr,"--- Initialising extra functions ---\n");
  isosurfacesInit();
  initToolMatrix();
  geometryInit();
  g_type_class_ref(VISU_MARKS_TYPE);

  /* Force the creation of some internal classes. */
  DBG_fprintf(stderr,"--- Initialising internal classes ---\n");
  g_type_class_ref(VISU_RENDERING_TYPE);
  g_type_class_ref(VISU_DATA_TYPE);
  g_type_class_ref(VISU_GLEXT_SCALE_TYPE);

  entry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
				  FLAG_PARAMETER_UNIT,
				  DESC_PARAMETER_UNIT,
				  1, readUnit);
  visu_configFile_entry_setVersion(entry, 3.5f);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);
}

/**
 * visu_basic_parseConfigFiles:
 * 
 * Parse the parameter and the resource file. Used once at startup.
 *
 * Returns: a newly allocated #GString if some error occured.
 */
gchar* visu_basic_parseConfigFiles(void)
{
  GString *message;
  char* path;
  GError *error;
  gboolean res;
  char *resFile;

  message = (GString*)0;
  DBG_fprintf(stderr,"--- Initialising the parameters ---\n");
  /* Look for parameters file */
  path = visu_configFile_getValidPath(VISU_CONFIGFILE_PARAMETER, R_OK, 0);
  if (path)
    {
      error = (GError*)0;
      res = visu_configFile_load(VISU_CONFIGFILE_PARAMETER, path, (VisuData*)0, &error);
      if (error)
	{
	  message = g_string_new("");
	  g_string_append_printf(message,
				 _("While parsing parameter file '%s':\n\n%s"),
				 path, error->message);
	  g_error_free(error);
	}
      g_free(path);
    }
  else
    message = g_string_new("Unable to find a valid parameters file."
			   " Starting with defaults.\n");

  DBG_fprintf(stderr,"--- Initialising resources ---\n");
  /* Look for resources file */
  resFile = commandLineGet_resourcesFile();
  if (!resFile)
    path = visu_configFile_getValidPath(VISU_CONFIGFILE_RESOURCE, R_OK, 0);
  else
    path = g_strdup(resFile);
  if (path)
    {
      error = (GError*)0;
      res = visu_configFile_load(VISU_CONFIGFILE_RESOURCE, path, (VisuData*)0, &error);
      if (error)
	{
	  if (message)
	    g_string_append(message, "\n\n");
	  else
	    message = g_string_new("");
	  g_string_append_printf(message,
				 _("While parsing resource file '%s':\n\n%s"),
				 path, error->message);
	  g_error_free(error);
	}
      g_free(path);
    }

  if (message)
    return g_string_free(message, FALSE);
  else
    return (gchar*)0;
}

/**
 * visu_basic_setDataFromCommandLine:
 * 
 * Read the command line option and set the filenames for a new
 * #VisuData. The object is not loaded (files are not parsed), just prepared.
 *
 * Returns: (transfer full): a newly allocated #VisuData if required.
 */
VisuData* visu_basic_setDataFromCommandLine(void)
{
  char* filename, *spin_filename;
  VisuData *newData;

  newData = (VisuData*)0;

  filename = commandLineGet_ArgFilename();
  spin_filename = commandLineGet_ArgSpinFileName();
  if (filename && !spin_filename)
    {
      newData = visu_data_new();
      if (!newData)
	return (VisuData*)0;

      visu_object_setRendering(VISU_INSTANCE,
                               visu_rendering_getByName(VISU_RENDERING_ATOMIC_NAME));

      visu_data_addFile(newData, filename, FILE_KIND_POSITION, (ToolFileFormat*)0);
    }
  else if(filename && spin_filename)
    {
      newData = visu_data_new();
      if (!newData)
	return (VisuData*)0;

      visu_object_setRendering(VISU_INSTANCE, visu_rendering_getByName(VISU_RENDERING_SPIN_NAME));

      visu_data_addFile(newData, filename, FILE_KIND_POSITION, (ToolFileFormat*)0);
      visu_data_addFile(newData, spin_filename, FILE_KIND_SPIN, (ToolFileFormat*)0);
    }
  return newData;
}

static void onNewData(GObject* obj _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDefined",
		   G_CALLBACK(onNodesSet), (gpointer)0);
}
static void onNodesSet(VisuData *data, VisuNodeArray *nodes, gpointer user_data _U_)
{
  /* We add the label node property to each VisuData with new nodes. */
  if (nodes)
    extraNodeAdd_label(data);
}

struct _dump
{
  VisuData *data;
  VisuDump *format;
  gchar *exportFileName;
  int width, height;

  /* Bitmap export only. */
  VisuPixmapContext *dumpData;
  GMainLoop *loop;

  /* Return. */
  int status;
};

/**
 * visu_basic_mainExport:
 *
 * This method is called when V_Sim is in export mode from the command line. 
 *
 * Returns: 0 if everything is normal, 1 if an error occured.
 */
int visu_basic_mainExport(void)
{
  gchar *message;
  VisuData *newData;
  GError *error;
  gboolean res;
  struct _dump dt;
  GList *pnt;
  GHashTable *opts;
  VisuBasicCLISet *set;
  ToolOption *id;

  dt.exportFileName = commandLineGet_ExportFileName();
  if (!dt.exportFileName)
    {
      g_error("This method should be called with"
	      " an argument that is the file name to export to.\n");
    }
  
  opts = commandLineGet_options();
  id = (ToolOption*)g_hash_table_lookup(opts, "fileFormatId");
  pnt = visu_dump_getAllModules();
  if (!id)
    while (pnt && !tool_file_format_match( ((VisuDump*)pnt->data)->fileType,
                                           dt.exportFileName))
      pnt = g_list_next(pnt);
  else
    pnt = g_list_nth(pnt, g_value_get_int(tool_option_getValue(id)) - 1);
  if (!pnt)
    {
      g_warning(_("The format can't be found from the"
                  " filename '%s' entered.\n"), dt.exportFileName);
      g_print(_("Use -o fileFormatId=id to specify a file format"
                " when the autodetection fails. Get a list of ids"
                " with option -o list:\n\n"));
      visu_basic_showOptionHelp(TRUE);
      return 1;
    }
  dt.format = (VisuDump*)pnt->data;

  /* We transfer some options to file format. */
  if (opts)
    g_hash_table_foreach(opts, setToolFileFormatOption,
                         (gpointer)dt.format->fileType);

  /* Start the elements related to the main loop. */
  dt.loop = g_main_loop_new(NULL, FALSE);
  g_type_init();

  initVisuBasic();

  commandLineGet_XWindowGeometry((int*)&(dt.width), (int*)&(dt.height));
  if (dt.format->bitmap || dt.format->glRequired)
    {
      initOpenglGraphics();

      /* We need to initialise the OpenGL surface as soon as possible to
	 avoid crash in some OpenGL implementations when OpenGL routines
	 may be called while parsing files. */
      dt.dumpData = visu_pixmap_context_new((guint)dt.width, (guint)dt.height);
      if (!dt.dumpData)
	{
	  g_error("can't create off-screen rendering. Command line"
		  " exportation is not available.");
	}
      /* We set the glViewport of this new context. */
      glViewport(0, 0, dt.width, dt.height);
      /* We call the given draw method. */
      openGLInit_context();

      message = visu_basic_parseConfigFiles();
      if (message)
	{
	  g_warning("%s", message);
	  g_free(message);
	}
    }

  newData = visu_basic_setDataFromCommandLine();
  if (!newData)
    {
      if (dt.format->bitmap || dt.format->glRequired)
	visu_pixmap_context_free(dt.dumpData);
      g_error(_("a file to render is mandatory with the '--export' option."));
    }
  dt.data = newData;
  
  error = (GError*)0;
  res = visu_basic_loadData(newData, (ToolFileFormat*)0, 0, &error);
  if (!res)
    {
      if (dt.format->bitmap || dt.format->glRequired)
	visu_pixmap_context_free(dt.dumpData);
      g_object_unref(newData);
      g_error("%s", (error)?error->message:"No error message!");
    }

  error = (GError*)0;
  dt.status = 0;
  if (dt.format->bitmap || dt.format->glRequired)
    {
      /* Run a main loop to load everything. */
      DBG_fprintf(stderr, "Visu Basic: emitting the "
		  "'dataReadyForRendering' signal.\n");
      g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATAREADYFORRENDERING_SIGNAL],
		    0, (gpointer)newData, NULL);
      DBG_fprintf(stderr, "Visu Basic: emitting done.\n");
      
      DBG_fprintf(stderr, "Visu Basic: adding the load call back in the queue.\n");
      g_idle_add(dumpData, (gpointer)&dt);
  
      g_main_loop_run(dt.loop);
    }
  else
    {
      /* Direct export. */
      set = g_malloc0(sizeof(VisuBasicCLISet));
      if (!visu_basic_applyCommandLine(dt.data, set, &error))
	{
	  g_warning("%s", error->message);
	  g_error_free(error);
	  dt.status = 1;
	}
      else if (!dt.format->writeFunc(dt.format->fileType, dt.exportFileName,
				     dt.width, dt.height,
				     dt.data, (guchar*)0, &error,
				     (ToolVoidDataFunc)0, (gpointer)0) && error)
	{
	  g_warning("%s", error->message);
	  g_error_free(error);
	  dt.status = 1;
	}
      freeExtension(set);
    }

  g_object_unref(G_OBJECT(dt.data));

  return dt.status;
}

/**
 * visu_basic_setExtInfos:
 * @dataObj: a #VisuData object ;
 * @method: where to draw information ;
 * @nodes: a possible list of selected nodes, or NULL ;
 * @data: the information to draw.
 *
 * Create and draw possible information on nodes.  This method
 * is mainly used when V_Sim run without the command panel.
 */
void visu_basic_setExtInfos(VisuData *dataObj, VisuGlExtInfosDrawMethod method,
                            int *nodes, VisuDataNode *data)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(dataObj));
  g_return_if_fail((method == EXT_DRAW_METH_OTHER && IS_VISU_DATA_NODE_TYPE(data)) ||
		   method != EXT_DRAW_METH_OTHER);

  DBG_fprintf(stderr, "VisuBasic: set the extension infos with method %d.\n", method);
  switch (method)
    {
    case EXT_DRAW_METH_NONE:
      visu_glExt_infos_setOn(dataObj, FALSE);
      break;
    case EXT_DRAW_METH_ID:
      visu_glExt_infos_setOn(dataObj, TRUE);
      visu_glExt_infos_drawIds(dataObj, nodes);
      break;
    case EXT_DRAW_METH_TYPE:
      visu_glExt_infos_setOn(dataObj, TRUE);
      visu_glExt_infos_drawElements(dataObj, nodes);
      break;
    default:
      visu_glExt_infos_setOn(dataObj, TRUE);
      visu_glExt_infos_drawData(dataObj, data, nodes);
    }
}

static int planesId, surfId, mapId;

static void setBgImage(const gchar *filename)
{
  GError *error;
  GdkPixbuf *pixbuf;
  gchar *title;
  gboolean fit;

  DBG_fprintf(stderr, "Visu Basic: set the background image to '%s'.\n",
	      filename);
  error = (GError*)0;
  pixbuf = gdk_pixbuf_new_from_file(filename, &error);
  if (!pixbuf)
    {
      g_warning("%s", error->message);
      g_error_free(error);
      return;
    }

  fit = TRUE;
  title = g_path_get_basename(filename);
  if (!strcmp(title, "logo_grey.png"))
    {
      fit = FALSE;
      g_free(title);
      title = (gchar*)0;
    }
  visu_glExt_bg_setImage(gdk_pixbuf_get_pixels(pixbuf),
	      gdk_pixbuf_get_width(pixbuf),
	      gdk_pixbuf_get_height(pixbuf),
	      gdk_pixbuf_get_has_alpha(pixbuf),
	      title, fit);
  g_object_unref(pixbuf);
  g_free(title);
}

static void rebuildPlanes(VisuData *dataObj)
{
  VisuBasicCLISet *set;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);

  DBG_fprintf(stderr, "Visu Basic: create OpenGL list for planes.\n");
  planesDraw_list(set->planesList, planesId);
  DBG_fprintf(stderr, " | OK\n");
}
static void rebuildSurf(VisuData *dataObj)
{
  SurfacesOrder *surfaceOrder;
  VisuBasicCLISet *set;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);
  /* Create the surface list. */
  surfaceOrder = isosurfacesOrder_new();
  isosurfacesDraw_surfaces(surfId + 1, set->surfsList, surfaceOrder);
  isosurfacesDuplicate(surfId, surfId + 1, dataObj, TRUE);
  isosurfacesOrder_free(surfaceOrder);
}
static float drawnMinMax[2];
static void rebuildMap(VisuData *dataObj)
{
  VisuBasicCLISet *set;
  int i;
  float *extens, ext[3], *mM;
  VisuOpenGLView *view;
  VisuMap *map;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);

  extens = visu_data_getBoxExtens(dataObj);
  view = visu_data_getOpenGLView(dataObj);
  visu_data_getExtension(dataObj, ext);
  drawnMinMax[0] = + G_MAXFLOAT;
  drawnMinMax[1] = - G_MAXFLOAT;
  /* Create the map list. */
  glNewList(mapId, GL_COMPILE);
  for (i = 1; i <= set->mapPlaneId[0]; i++)
    {
      map = visu_map_newFromPlane(set->planesList[set->mapPlaneId[i]]);
      if (map)
        {
          visu_map_setLevel(map, view->camera->gross, extens[0]);
          visu_map_setField(map, (ScalarField*)set->fieldsList->data,
                            set->logScale, commandLineGet_mapMinMax(), ext);
          visu_map_compute(map);
          visu_map_setLines(map, set->nIsolines, visu_map_getScaledMinMax(map));
          visu_map_draw(map, (float)commandLineGet_mapPrecision(),
                        set->shade, set->isoLinesColor, FALSE, TRUE);

          mM = visu_map_getFieldMinMax(map);
          drawnMinMax[0] = MIN(drawnMinMax[0], mM[0]);
          drawnMinMax[1] = MAX(drawnMinMax[1], mM[1]);
          visu_map_free(map);
        }
    }
  glEndList();
}
static void rebuildLegend(VisuData *dataObj)
{
  double minMax[2];
  float fMinMax[2];
  VisuBasicCLISet *set;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);
  /* Create the map list. */
  scalarFieldGet_minMax((ScalarField*)set->fieldsList->data, minMax);
  fMinMax[0] = minMax[0];
  fMinMax[1] = minMax[1];
  
  openGLText_rebuildFontList();

  glNewList(mapId + 1, GL_COMPILE);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_ALPHA_TEST);
  shadeDraw_legend(set->shade, .05, -1., fMinMax,
		   drawnMinMax, 2, set->logScale);
  glEndList();
}

/**
 * visu_basic_createExtensions:
 * @data: a #VisuData object ;
 * @set: a table of options ;
 * @rebuild: a boolean.
 *
 * Create all extensions (planes, surfaces...) for the given data. Use
 * static values read from visu_basic_applyCommandLine(). This method
 * is mainly used when V_Sim run without the command panel.
 */
void visu_basic_createExtensions(VisuData *data, VisuBasicCLISet *set, gboolean rebuild)
{
  VisuExtension *plane, *surfExt, *map;

  /* We create the plane extension if necessary. */
  if (set->planesList)
    {
      DBG_fprintf(stderr, "Visu Basic: create plane extension.\n");
      planesId = visu_openGL_objectList_new(1);
      plane = visu_extension_new("Planes", _("Planes"), (char*)0,
				  planesId, rebuildPlanes);
      plane->used = 1;
      visu_extension_setSensitiveToRenderingMode(plane, TRUE);
      visu_extension_setPriority(plane, VISU_EXTENSION_PRIORITY_LOW + 1);
      visuExtensions_add(plane);
      if (rebuild)
	rebuildPlanes(data);
    }

  /* We create the surface extension. */
  if (set->surfsList)
    {
      DBG_fprintf(stderr, "Visu Basic: create surface extension.\n");
      surfId = visu_openGL_objectList_new(2);
      surfExt = visu_extension_new("Isosurfaces", _("Isosurfaces"), (char*)0,
				    surfId, rebuildSurf);
      surfExt->used = 1;
      visu_extension_setSensitiveToRenderingMode(surfExt, TRUE);
      visu_extension_setSaveOpenGLState(surfExt, TRUE);
      visu_extension_setPriority(surfExt, VISU_EXTENSION_PRIORITY_LOW + 2); 
      visuExtensions_add(surfExt);
      if (rebuild)
	rebuildSurf(data);
    }

  /* We create the map extension. */
  if (set->mapPlaneId)
    {
      DBG_fprintf(stderr, "Visu Basic: create map extension.\n");
      mapId = visu_openGL_objectList_new(2);
      map = visu_extension_new("Map", _("Colored map"), (char*)0,
				mapId, rebuildMap);
      map->used = 1;
      visuExtensions_add(map);
      if (rebuild)
	rebuildMap(data);

      map = visu_extension_new("MapLegend", _("Map legend"), (char*)0,
				mapId + 1, rebuildLegend);
      visu_extension_setPriority(map, VISU_EXTENSION_PRIORITY_LAST);
      visu_extension_setSaveOpenGLState(map, TRUE);
      map->used = 1;
      visuExtensions_add(map);
      if (rebuild)
	rebuildLegend(data);
    }

  /* We set the background image. */
  if (set->bgImage)
    setBgImage(set->bgImage);
}

static gboolean dumpData(gpointer data)
{
  GMainContext *context;
  GError *error;
  VisuBasicCLISet *set;
  struct _dump *dt;
  GArray* image;
  VisuOpenGLView *view;
  guchar *imgData;
  float length0;

  dt = (struct _dump*)data;

  /* Here the dump is bitmap only. */
  g_return_val_if_fail(dt->format->bitmap || dt->format->glRequired, FALSE);

  DBG_fprintf(stderr, "Visu Basic: begin dump exportation.\n");
  set = g_malloc0(sizeof(VisuBasicCLISet));
  g_object_set_data_full(G_OBJECT(dt->data), "optionSet", set, freeExtension);
  error = (GError*)0;
  if (!visu_basic_applyCommandLine(dt->data, set, &error))
    {
      g_warning("%s", error->message);
      g_error_free(error);
    }
  else
    {
      visu_basic_createExtensions(dt->data, set, FALSE);

      context = g_main_loop_get_context(dt->loop);
      while(g_main_context_pending(context))
	g_main_context_iteration(context, FALSE);

      view = visu_data_getOpenGLView(dt->data);
      length0 = openGLCameraGet_refLength(view->camera, (ToolUnits*)0);
      if (length0 <= 0.f)
        {
          openGLCameraSet_refLength(view->camera, visu_data_getBoxExtens(dt->data)[0],
                                    visu_data_getUnit(dt->data));
          visu_data_createAllElements(dt->data);
        }
      openGLModelize(view->camera);
      if (!visu_data_setSizeOfView(dt->data, (guint)dt->width, (guint)dt->height))
        openGLProject(view->window, view->camera,
                      visu_data_getBoxExtens(dt->data)[1]);

      visuExtensions_rebuildAllLists(dt->data);

      if (dt->format->bitmap)
        {
          openGL_reDraw((const gchar**)0, dt->data);
          /* We copy the pixmap into generic data. */
          image = visuOpenGLGet_pixmapData((guint)dt->width, (guint)dt->height,
                                           dt->format->hasAlpha);
          imgData = (guchar*)g_array_free(image, FALSE);
        }
      else
        imgData = (guchar*)0;

      DBG_fprintf(stderr, "Visu Basic: calling exportation routine.\n");
      error = (GError*)0;
      if (!dt->format->writeFunc(dt->format->fileType, dt->exportFileName,
				 dt->width, dt->height,
				 dt->data, imgData, &error,
				 (ToolVoidDataFunc)0, (gpointer)0) && error)
	{
	  g_warning("%s", error->message);
	  g_error_free(error);
	  dt->status = 1;
	}
      else
	dt->status = 0;

      if (imgData)
        g_free(imgData);
    }

  /* We free the pixmap context. */
  DBG_fprintf(stderr, "Visu Basic: going to free the context.\n");
  visu_pixmap_context_free(dt->dumpData);

  DBG_fprintf(stderr, "Visu Basic: stopping the main loop.\n");
  g_main_loop_quit(dt->loop);

  return FALSE;
}

/**
 * visu_basic_applyCommandLine:
 * @data: a #VisuData object to apply the options on ;
 * @set: a table of options ;
 * @error: a location for error report.
 *
 * Call all the get methods on the command line options to tune the
 * given @data.
 *
 * Returns: TRUE if complete without error.
 */
gboolean visu_basic_applyCommandLine(VisuData *visuData, VisuBasicCLISet *set,
				    GError **error)
{
  gchar *planesFile, *surfFile, *fieldFile, *valueFile;
  int i, j, presetShade, nb, surfNb;
  gboolean somethingIsLoaded, res, new;
  int *colUsed;
  float *translations, *extension;
  float *values;
  gchar **names;
  GList *tmplst, *list;
  GHashTable *table;
  int *nodes;
  VisuGlExtInfosDrawId mode;
  guint info;
  VisuDataNode *dt;
  VisuGlExtInfosDrawMethod method;
  float vertices[8][3];

  g_return_val_if_fail(visuData && set, FALSE);
  g_return_val_if_fail(error && !*error, FALSE);

  set->colorFile     = (gchar*)0;
  set->planesList    = (Plane**)0;
  set->surfsList     = (Surfaces**)0;
  set->shade         = (Shade*)0;
  set->mapPlaneId    = (gint*)0;
  set->logScale      = linear;
  set->nIsolines     = 0;
  set->isoLinesColor = (float*)0;
  set->fieldsList    = (GList*)0;
  set->bgImage       = (gchar*)0;

  surfNb = 0;

/*   commandLineGet_XWindowGeometry((int*)&width, (int*)&height); */
/*   visu_data_setSizeOfView(visuData, width, height); */

  /* The value File, maybe used later. */
  valueFile = commandLineGet_valueFile();
  /* translate argument */
  translations = commandLineGet_translation();
  if (translations)
    {
      visu_data_setXYZtranslation(visuData, translations);
      visu_data_constrainedInTheBox(visuData);
    }
  /* expand argument */
  extension = commandLineGet_extension();
  if (extension)
    {
      if (!translations)
	visu_data_constrainedInTheBox(visuData);
      visu_data_replicate(visuData, extension, (gboolean*)0);
    }

  /* The shade option. */
  presetShade = commandLineGet_presetColor();
  list        = toolShadeBuild_presetList();
  if (presetShade >= 0)
    {
      if (presetShade >= (int)g_list_length(list))
	{
	  g_warning(_("unknown shade id (max is %d)."), g_list_length(list) - 1);
	  set->shade = (Shade*)g_list_nth_data(list, 0);
	}
      else
	set->shade = (Shade*)g_list_nth_data(list, presetShade);
      DBG_fprintf(stderr, "Visu Basic: load a shade from command line %p.\n",
		  (gpointer)set->shade);
    }
  else
    set->shade = (Shade*)g_list_nth_data(list, 0);
  /* colorize argument */
  set->colorFile = commandLineGet_colorizeFileName();
  if (set->colorFile)
    {
      DBG_fprintf(stderr, "Visu basic : loading a new data file '%s'.\n",
		  set->colorFile);
      somethingIsLoaded = dataFileSet_file(visuData, set->colorFile, &new, error);

      /* Print errors if necessary. */
      if (*error)
	{
	  g_warning("%s", (*error)->message);
	  g_error_free(*error);
	  *error = (GError*)0;
	}

      if (somethingIsLoaded)
	{
	  colUsed = commandLineGet_colorizeColUsed();
	  if (colUsed)
	    for (i = 0; i < 3; i++)
	      dataFileSet_colUsed(visuData, colUsed[i] - 1, i);
	  dataFileSet_shade(visuData, set->shade);
	  dataFileSet_used(visuData, TRUE);
	}
    }
  /* Planes argument. */
  planesFile = (valueFile)?valueFile:commandLineGet_planesFileName();
  if (planesFile)
    {
      somethingIsLoaded = planesParse_XMLFile(planesFile, &set->planesList, error);
      if (!somethingIsLoaded)
	return FALSE;
      else
	{
	  visu_data_getBoxVertices(visuData, vertices, TRUE);
	  for (i = 0; set->planesList[i]; i++)
	    planeSet_box(set->planesList[i], vertices);
	  /* Hide nodes if required. */
	  res = planeShowHide_all(visuData, set->planesList);
	  if (res)
	    visu_data_emitNodeRenderedChange(visuData);
	}
    }
  /* Iso-surfaces arguments. */
  surfFile = commandLineGet_isoSurfacesFileName();
  if (surfFile)
    {
      surfNb = 1;
      set->surfsList = g_malloc(sizeof(Surfaces*) * (surfNb + 1));
      somethingIsLoaded = isosurfacesLoad_file(surfFile, set->surfsList, error);
      if (!somethingIsLoaded)
	{
	  surfNb = 0;
	  return FALSE;
	}
      else if (commandLineGet_fitToBox())
	isoSurfacesSet_fitToBox(visuData, set->surfsList[0]);
      isosurfacesSet_showAll(set->surfsList[0], TRUE);
      set->surfsList[surfNb] = (Surfaces*)0;
    }
  /* Scalar-field arguments. */
  fieldFile = commandLineGet_scalarFieldFileName();
  if (fieldFile)
    {
      /* Create an option table for possible spin or complex
	 modifiers. */
      table = commandLineGet_options();
      /* Call the load method. */
      somethingIsLoaded = scalarFieldLoad_fromFile(fieldFile, &set->fieldsList,
						   error, table);
      if (!somethingIsLoaded)
	{
	  surfNb = 0;
	  return FALSE;
	}
      else
	{
	  if (commandLineGet_fitToBox())
	    {
	      tmplst = set->fieldsList;
	      while(tmplst)
		{
		  scalarFieldSet_fitToBox(visuData, (ScalarField*)tmplst->data);
		  tmplst = g_list_next(tmplst);
		}
	    }

	  nb             = g_list_length(set->fieldsList);
	  set->surfsList = g_malloc0(sizeof(Surfaces*) * (nb + 1));
	  surfNb         = 0;
	  values         = commandLineGet_isoValues(&nb);
	  names          = commandLineGet_isoNames(&nb);
	  if (values)
	    {
	      tmplst = set->fieldsList;
	      while(tmplst)
		{
		  for (i = 0; i < nb; i++)
		    res = pot2surfCreate(set->surfsList, (ScalarField*)tmplst->data,
					 values[i], i, names[i]);
		  isosurfacesSet_showAll(set->surfsList[surfNb++], TRUE);
		  tmplst = g_list_next(tmplst);
		}
	    }
	  
	  /* We may add the iso-values from a file. */
	  if (valueFile)
	    {
	      somethingIsLoaded =
		pot2surfParse_XMLFile(valueFile, set->surfsList,
				      (ScalarField*)set->fieldsList->data, error);
	      if (!somethingIsLoaded)
		return FALSE;
	    }
	}
    }
  /* We apply the masking effect of planes, if necessary. */
  if (set->surfsList && set->planesList)
    for (i = 0; set->surfsList[i]; i++)
      isosurfacesHide(set->surfsList[i], set->planesList);
  /* The coloured map argument. */
  set->mapPlaneId = (gint*)commandLineGet_coloredMap();
  if (set->mapPlaneId)
    {
      if (!set->planesList)
	g_warning(_("option '--build-map' has been given but"
		    " no plane is available (use '--planes')."));
      if (!set->fieldsList)
	g_warning(_("option '--build-map' has been given but"
		    " no scalar field is available (use '--scalar-field')."));
      if (!set->shade)
	g_warning(_("option '--build-map' has been given but"
		    " no shade is available (use '--color-preset')."));
      set->logScale = commandLineGet_logScale();
      set->nIsolines = commandLineGet_nIsoLines();
      set->isoLinesColor = commandLineGet_isoLinesColor();
      if (set->planesList && set->fieldsList && set->shade)
	{
	  /* We change the visibility of the selected planes. */
	  for (i = 0; set->planesList[i]; i++)
	    for (j = 1; j <= set->mapPlaneId[0]; j++)
	      if (i == set->mapPlaneId[j])
		{
		  planeSet_rendered(set->planesList[i], FALSE);
		  break;
		}
	}
    }
  /* The bg image. */
  set->bgImage = commandLineGet_bgImage();
  /* The pick information. */
  if (valueFile)
    {
      if (!visu_marks_parseXMLFile((VisuMarks*)0, valueFile, &list, &mode, &info, error))
	return FALSE;
      nb = g_list_length(list) + 1;
      if (mode == DRAW_SELECTED)
	{
	  nodes = g_malloc(sizeof(int) * nb);
	  nodes[nb] = -1;
	  i = 0;
	  for (tmplst = list; tmplst; tmplst = g_list_next(tmplst))
	    {
	      if (GPOINTER_TO_INT(tmplst->data) == PICK_SELECTED)
		{
		  tmplst = g_list_next(tmplst);
		  nodes[i++] = GPOINTER_TO_INT(tmplst->data) - 1;
		}
	      else if (GPOINTER_TO_INT(tmplst->data) == PICK_REFERENCE_1)
		{
		  tmplst = g_list_next(tmplst);
		  /* We add the last selection. */
		  tmplst = g_list_next(tmplst);
		  nodes[i++] = GPOINTER_TO_INT(tmplst->data) - 1;
		}
	      else if (GPOINTER_TO_INT(tmplst->data) == PICK_HIGHLIGHT)
		tmplst = g_list_next(tmplst);
	    }
	  g_list_free(list);
	  nodes[i] = -1;
	}
      else
	nodes = (int*)0;
      if (mode != DRAW_NEVER)
	{
	  dt = (VisuDataNode*)0;
	  switch (info)
	    {
	    case 0:
	      method = EXT_DRAW_METH_ID; break;
	    case 1:
	      method = EXT_DRAW_METH_TYPE; break;
	    default:
	      method = EXT_DRAW_METH_OTHER;
	      i = 2;
	      for (tmplst = nodeDataGet_list(); tmplst; tmplst = g_list_next(tmplst))
		{
		  if (visu_data_node_getUsed(VISU_DATA_NODE(tmplst->data), visuData))
		    {
		      if (i == (int)info)
			break;
		      i += 1;
		    }
		}
	      if (tmplst)
		dt = VISU_DATA_NODE(tmplst->data);
	      break;
	    }
	  visu_basic_setExtInfos(visuData, method, nodes, dt);
	}
    }

  return TRUE;
}
static void freeExtension(gpointer data)
{
  int i;
  GList *tmplst;
  VisuBasicCLISet *set;

  set = (VisuBasicCLISet*)data;
  /* Free allocated memory. */
  if (set->fieldsList)
    {
      tmplst = set->fieldsList;
      while(tmplst)
	{
	  scalarFieldFree((ScalarField*)tmplst->data);
	  tmplst = g_list_next(tmplst);
	}
	g_list_free(set->fieldsList);
    }
  if (set->planesList)
    {
      for (i = 0; set->planesList[i]; i++)
	g_object_unref(G_OBJECT(set->planesList[i]));
      g_free(set->planesList);
    }
  if (set->surfsList)
    {
      for (i = 0; set->surfsList[i]; i++)
	isosurfacesFree(set->surfsList[i]);
      g_free(set->surfsList);
    }
  g_free(set);
}

static void setToolFileFormatOption(gpointer key, gpointer value, gpointer data)
{
  ToolOption *prop;

  DBG_fprintf(stderr, "Visu Basic: transfer option '%s' to file format.\n",
              (gchar*)key);
  prop = tool_file_format_getPropertyByName((ToolFileFormat*)data, (const gchar*)key);
  if (!prop)
    return;

  fprintf(stderr, "%p %p\n", (gpointer)tool_option_getValue((ToolOption*)value), (gpointer)tool_option_getValue(prop));
  g_value_copy(tool_option_getValue((ToolOption*)value), tool_option_getValue(prop));
}
/**
 * visu_basic_showOptionHelp:
 * @force: a boolean.
 *
 * Display a small help for some options. The output is different from
 * the -h command line options, here some details about running time
 * options is displayed like the available file format for
 * exportation... If @force is TRUE, all possible values are output,
 * otherwise only those relevant to the user provided command line
 * options.
 *
 * Since: 3.6
 *
 * Returns: TRUE if something is displayed.
 */
gboolean visu_basic_showOptionHelp(gboolean force)
{
  const GList *lst;
  ToolOption *prop;
  GValue *val;
  GList *pnt;
  GHashTable *opts;
  ToolFileFormat *format;
  guint i;

  if (!force)
    {
      opts = commandLineGet_options();
      if (!opts || !g_hash_table_lookup(opts, "list"))
        return FALSE;
    }

  i = 1;
  for (pnt = visu_dump_getAllModules(); pnt; pnt = g_list_next(pnt))
    {
      format = ((VisuDump*)pnt->data)->fileType;
      fprintf(stdout, _("\n#%2d - file format '%s':\n"),
              i++, tool_file_format_getName(format));
      for (lst = tool_file_format_getPropertyList(format);
           lst; lst = g_list_next(lst))
	{
	  prop = (ToolOption*)lst->data;
	  fprintf(stdout, " - '%25s'", tool_option_getName(prop));
          val = tool_option_getValue(prop);
	  switch (tool_option_getType(prop))
	    {
	    case G_TYPE_INT:
	      fprintf(stdout, " %10s (%5d): ", _("integer"), g_value_get_int(val));
	      break;
	    case G_TYPE_BOOLEAN:
	      fprintf(stdout, " %10s (%5d): ", _("boolean"), g_value_get_boolean(val));
	      break;
	    case G_TYPE_STRING:
	      fprintf(stdout, " %10s: ", _("string"));
	      break;
	    default:
              g_warning("Unknown type for file format property.");
	      break;
	    }
	  fprintf(stdout, "%s.\n", tool_option_getLabel(prop));
	}      
      if (!tool_file_format_getPropertyList(format))
	fprintf(stdout, _("No option for this file format.\n"));
    }

  return TRUE;
}

static gchar* setDir(const gchar* const *sysDirs, const gchar *prefix,
		     const gchar* subDir, const gchar* defaultDir)
{
  gchar *dir;
  int i;

  dir = g_build_filename(prefix, subDir, NULL);
  if (!g_file_test(dir, G_FILE_TEST_IS_DIR))
    {
      g_free(dir);
      dir = (gchar*)0;
      /* We try the XDG stuff. */
      for (i = 0; sysDirs[i]; i++)
	{
	  dir = g_build_filename(sysDirs[i], subDir, NULL);
	  if (g_file_test(dir, G_FILE_TEST_IS_DIR))
	    break;
	  else
	    g_free(dir);
	  dir = (gchar*)0;
	}
    }
  if (!dir)
    dir = g_strdup(defaultDir);
  return dir;
}

/**
 * visu_basic_setExePath:
 * @exePath: a path where the V_Sim executable is running in.
 *
 * This method is used to tell V_Sim where V_Sim is running (usually
 * reading from argv[0]. This makes it possible to relocate everything
 * on the fly. @exePath is copied.
 *
 * Since: 3.6
 */
void visu_basic_setExePath(const gchar *exePath)
{
  if (exeLocation)
    g_free(exeLocation);

  exeLocation = g_strdup(exePath);
}
/**
 * setVisuPathes:
 * 
 * This method sets the paths. On Unix systems, this method sets the paths
 * from macros defined by configure. On Win32 systems, it reads paths in
 * a v_sim.ini file found in the current directory or in the C:\windows.
 */
static void setVisuPathes(void)
{
#if SYSTEM_WIN32 == 1
  #define V_SIM_INI_FILE PACKAGE".ini"
  GIOChannel *iniFile;
  gchar *iniPath, *tmp, *me, *prefix;
  gchar *buffer, *line;
  GIOStatus res;
  GError *err;
  gchar **tokens;
  gsize length;

  if (!exeLocation)
    exeLocation = g_strdup(PACKAGE_TARNAME);

  v_sim_data_dir    = g_strdup(V_SIM_DATA_DIR_DEFAULT);
  v_sim_legal_dir   = g_strdup(V_SIM_LEGAL_DIR_DEFAULT);
  v_sim_pixmaps_dir = g_strdup(V_SIM_PIXMAPS_DIR_DEFAULT);
  v_sim_icons_dir   = g_strdup(V_SIM_ICONS_DIR_DEFAULT);
  v_sim_plugins_dir = g_strdup(V_SIM_PLUGINS_DIR_DEFAULT);
  v_sim_locale_dir  = g_strdup(V_SIM_LOCALE_DIR_DEFAULT);
  iniPath = g_strdup(V_SIM_INI_FILE);
  /* Try to find the INI file from cwd. */
  prefix = (gchar*)0;
  iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
  if (iniFile)
    prefix = g_get_current_dir();
  /* Try to find the INI file from the name of the executable. */
  if (!iniFile)
    {
      g_free(iniPath);

      if (g_file_test(exeLocation, G_FILE_TEST_IS_SYMLINK))
	tmp = g_file_read_link(exeLocation, (GError**)0);
      else
	tmp = g_strdup(exeLocation);
      me = normalize_path(tmp);
      g_free(tmp);
    
      DBG_fprintf(stderr, "Visu Basic: running program is '%s'.\n", me);
      /* From the location of the executable, we take the base dir. */
      prefix = g_path_get_dirname(me);
      g_free(me);

      iniPath = g_build_filename(prefix, V_SIM_INI_FILE, NULL);
      iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
    }  
  /* Try to find the INI file in the Windows directory. */
  if (!iniFile)
    {
      g_free(iniPath);
      prefix = g_strdup("C:\\WINDOWS");
      iniPath = g_build_filename(prefix, V_SIM_INI_FILE, NULL);
      iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
    }
  if (iniFile)
    {
      buffer = (gchar*)0;
      err = (GError*)0;
      do
	{
	  res = g_io_channel_read_line(iniFile, &line, &length, NULL, &err);
	  if (line && res == G_IO_STATUS_NORMAL)
	    {
	      tokens = g_strsplit(line, "=", 2);
	      if (!strcmp(g_strstrip(tokens[0]), "data_dir") && tokens[1])
		{
		  g_free(v_sim_data_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_data_dir = g_strdup(tmp);
		  else
		    v_sim_data_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "legal_dir") && tokens[1])
		{
		  g_free(v_sim_legal_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_legal_dir = g_strdup(tmp);
		  else
		    v_sim_legal_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "pixmaps_dir") && tokens[1])
		{
		  g_free(v_sim_pixmaps_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_pixmaps_dir = g_strdup(tmp);
		  else
		    v_sim_pixmaps_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "icons_dir") && tokens[1])
		{
		  g_free(v_sim_icons_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_icons_dir = g_strdup(tmp);
		  else
		    v_sim_icons_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "plugins_dir") && tokens[1])
		{
		  g_free(v_sim_plugins_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_plugins_dir = g_strdup(tmp);
		  else
		    v_sim_plugins_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "locale_dir") && tokens[1])
		{
		  g_free(v_sim_locale_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_locale_dir = g_strdup(tmp);
		  else
		    v_sim_locale_dir = g_build_filename(prefix, tmp, NULL);
		}
	      g_strfreev(tokens);
	      g_free(line);
	    }
	}
      while (res != G_IO_STATUS_EOF);
      g_io_channel_shutdown (iniFile, FALSE, (GError**)0);
      g_io_channel_unref (iniFile);
    }
  g_free(iniPath);
#endif

#if SYSTEM_X11 == 1
  const gchar* const *sysDirs;
  gchar *me, *prefix, *tmp;
#if DEBUG == 1
  int i;
#endif

  /* We try to get the dirs from XDG specs, otherwise we fall back to
     hard coded values at compilation time. */
#if GLIB_MINOR_VERSION > 5
  sysDirs = g_get_system_data_dirs();
#if DEBUG == 1
  fprintf(stderr, "Visu Basic: available data dirs:\n");
  for (i = 0; sysDirs[i]; i++)
    fprintf(stderr, " | '%s'.\n", sysDirs[i]);
#endif
#else
  sysDirs = g_malloc(sizeof(const gchar*));
  sysDirs[0] = (const gchar*)0;
#endif

  if (!exeLocation)
    exeLocation = g_strdup(PACKAGE_TARNAME);
  if (g_file_test(exeLocation, G_FILE_TEST_IS_SYMLINK))
    tmp = g_file_read_link(exeLocation, (GError**)0);
  else
    tmp = g_strdup(exeLocation);
  me = normalize_path(tmp);
  g_free(tmp);
    
  DBG_fprintf(stderr, "Visu Basic: running program is '%s'.\n", me);
  /* From the location of the executable, we take the base dir
     or its parents if the basedir is bin. */
  prefix = g_path_get_dirname(me);
  g_free(me);
  me = g_path_get_basename(prefix);
  if (!strcmp(me, "bin"))
    {
      g_free(me);
      me = prefix;
      prefix = g_path_get_dirname(me);
    }
  g_free(me);
  DBG_fprintf(stderr, " | prefix is '%s'.\n", prefix);

  v_sim_data_dir = setDir(sysDirs, prefix, "share/" PACKAGE, DATA_DIR);
  v_sim_legal_dir = setDir(sysDirs, prefix, "share/doc/" PACKAGE, LEGAL_DIR);
  v_sim_pixmaps_dir = setDir(sysDirs, prefix, "share/" PACKAGE "/pixmaps", PIXMAPS_DIR);
  v_sim_icons_dir = setDir(sysDirs, prefix, "share/icons", ICONS_DIR);
  v_sim_plugins_dir = setDir(sysDirs, prefix, "lib/" PACKAGE "/plug-ins", PLUGINS_DIR);
  v_sim_locale_dir = setDir(sysDirs, prefix, "share/locale", LOCALE_DIR);

  g_free(prefix);
#endif

  /* Create the local dirs. */
#if GLIB_MINOR_VERSION > 5
  v_sim_local_conf_dir = g_build_filename(g_get_user_config_dir(), "v_sim", NULL);
#else
  v_sim_local_conf_dir = g_build_filename(g_get_home_dir(), ".config/v_sim", NULL);
#endif
  if (!v_sim_local_conf_dir)
    g_warning("WARNING! Impossible to get the default"
	      " path $XDG_CONFIG_HOME/v_sim.\n");
  v_sim_old_local_conf_dir = g_build_filename(g_get_home_dir(), ".v_sim", NULL);

  DBG_fprintf(stderr, "Visu Basic: data directory      : '%s'.\n", v_sim_data_dir);
  DBG_fprintf(stderr, "Visu Basic: local conf directory: '%s'.\n", v_sim_local_conf_dir);
  DBG_fprintf(stderr, "Visu Basic: legal directory     : '%s'.\n", v_sim_legal_dir);
  DBG_fprintf(stderr, "Visu Basic: pixmaps directory   : '%s'.\n", v_sim_pixmaps_dir);
  DBG_fprintf(stderr, "Visu Basic: icons directory     : '%s'.\n", v_sim_icons_dir);
  DBG_fprintf(stderr, "Visu Basic: plug-ins directory  : '%s'.\n", v_sim_plugins_dir);
  DBG_fprintf(stderr, "Visu Basic: locale directory    : '%s'.\n", v_sim_locale_dir);
}
/**
 * visu_basic_getDataDir:
 *
 * Get the static string where V_Sim looks for its data files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getDataDir(void)
{
  if (!v_sim_data_dir)
    setVisuPathes();

  return v_sim_data_dir;
}
/**
 * visu_basic_getLegalDir:
 *
 * Get the static string where V_Sim looks for its legal files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getLegalDir(void)
{
  if (!v_sim_legal_dir)
    setVisuPathes();

  return v_sim_legal_dir;
}
/**
 * visu_basic_getPixmapsDir:
 *
 * Get the static string where V_Sim looks for its pixmap files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getPixmapsDir(void)
{
  if (!v_sim_pixmaps_dir)
    setVisuPathes();

  return v_sim_pixmaps_dir;
}
/**
 * visu_basic_getIconsDir:
 *
 * Get the static string where V_Sim looks for its icon files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getIconsDir(void)
{
  if (!v_sim_icons_dir)
    setVisuPathes();

  return v_sim_icons_dir;
}
/**
 * visu_basic_getLocalDir:
 *
 * Get the static string where V_Sim looks for its user configuration files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getLocalDir(void)
{
  if (!v_sim_local_conf_dir)
    setVisuPathes();

  return v_sim_local_conf_dir;
}
/**
 * visu_basic_getOldLocalDir:
 *
 * Get the static string where V_Sim looks for its user configuration
 * files (old location).
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getOldLocalDir(void)
{
  if (!v_sim_old_local_conf_dir)
    setVisuPathes();

  return v_sim_old_local_conf_dir;
}
/**
 * visu_basic_getPluginsDir:
 *
 * Get the static string where V_Sim looks for its plug-in files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getPluginsDir(void)
{
  if (!v_sim_plugins_dir)
    setVisuPathes();

  return v_sim_plugins_dir;
}
/**
 * visu_basic_getLocaleDir:
 *
 * Get the static string where V_Sim looks for its localisation files.
 *
 * Since: 3.4
 *
 * Returns: (transfer none): a string owned by V_Sim.
 */
const gchar* visu_basic_getLocaleDir(void)
{
  if (!v_sim_locale_dir)
    setVisuPathes();

  return v_sim_locale_dir;
}
/**
 * visu_basic_freeAll:
 *
 * This routine is called by V_Sim when quiting and it frees the memory
 * used by visu_basic.
 *
 * Since: 3.5
 */
void visu_basic_freeAll(void)
{
  DBG_fprintf(stderr, "Visu Basic: free all.\n - the paths\n");
  g_free(v_sim_data_dir);
  g_free(v_sim_legal_dir);
  g_free(v_sim_pixmaps_dir);
  g_free(v_sim_plugins_dir);
  g_free(v_sim_locale_dir);
  g_free(v_sim_local_conf_dir);
  g_free(v_sim_old_local_conf_dir);

  DBG_fprintf(stderr, " - the color storage\n");
  tool_color_freeAll();
}
/**
 * visu_basic_getMainContext:
 *
 * Even without GUI, V_Sim requires to run a main loop. This method is
 * to get the main loop.
 *
 * Since: 3.6
 *
 * Returns: (transfer none): the main loop, as defined in GLib.
 */
GMainContext* visu_basic_getMainContext(void)
{
  return g_main_context_default();
}


/* Resources. */
/**
 * visu_basic_getPreferedUnit:
 *
 * By setting the prefered unit, when a file is load, V_Sim tries to
 * render it in this prefered unit.
 *
 * Since: 3.5
 *
 * Returns: the prefered unit set by the user (default is
 * #unit_undefined).
 */
ToolUnits visu_basic_getPreferedUnit(void)
{
  return preferedUnit;
}
/**
 * visu_basic_setPreferedUnit:
 * @unit: a #ToolUnits value.
 *
 * By setting the prefered unit, when a file is load, V_Sim tries to
 * render it in this prefered unit.
 *
 * Since: 3.5
 *
 * Returns: TRUE if the prefered unit is actually changed.
 */
gboolean visu_basic_setPreferedUnit(ToolUnits unit)
{
  if (unit == preferedUnit)
    return FALSE;

  preferedUnit = unit;
  return TRUE;
}
static gboolean readUnit(gchar **lines, int nbLines, int position,
			 VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  ToolUnits unit;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readString(lines[0], position, &tokens, 1, FALSE, error))
    return FALSE;

  unit = tool_physic_getUnitFromName(tokens[0]);
  if (unit == unit_undefined)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: unit '%s' is unknown.\n"),
			   position, tokens[0]);
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  preferedUnit = unit;
  return TRUE;
}
static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  const gchar **units;

  if (preferedUnit != unit_undefined)
    {
      units = tool_physic_getUnitNames();

      g_string_append_printf(data, "# %s\n", DESC_PARAMETER_UNIT);
      g_string_append_printf(data, "%s: %s\n\n", FLAG_PARAMETER_UNIT,
			     units[preferedUnit]);
    }
}
