/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "panelGeometry.h"
#include "panelBrowser.h"

#include <support.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_data.h>
#include <visu_basic.h>
#include <visu_extension.h>
#include <visu_gtk.h>
#include <gtk_renderingWindowWidget.h>
#include <extensions/box.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolPhysic.h>
#include <extraGtkFunctions/gtk_stippleComboBoxWidget.h>
#include <extraGtkFunctions/gtk_shadeComboBoxWidget.h>
#include <extraFunctions/geometry.h>

/**
 * SECTION: panelGeometry
 * @short_description: This tab gathers the geometry operation on a
 * #VisuData, like periodic translation, physical units, ...
 */

/* Local objects. */
static GtkWidget *panelGeometry;
static GtkWidget *stippleExpandBox;
static GtkWidget *vboxPeriodic;
static GtkWidget *checkAllowTranslations;
static GtkWidget *spinTransXYZ[3];
static GtkWidget *checkAllowExpand;
static GtkWidget *spinExpandXYZ[3];
static GtkWidget *imagePeriodic;
static GtkWidget *labelPeriodic;
static gulong     transId;
static gboolean   transStatus = FALSE;
static GtkWidget *comboUnit;
static GtkWidget *checkDiff, *checkAdjust, *checkOrdering;
static GtkWidget *checkVisuPathes, *hboxVisuPathes, *togglePathSave, *labelVisuPathes;
static GtkWidget *hboxIOVisuPathes, *hboxVisuPathes2, *checkColorisePath, *comboShadePath;

/* Local variables. */
static gboolean disableCallbacks;
static gboolean widgetsNotBuilt;
static VisuExtension *extVisuPathes = NULL;
static VisuPathes *pathes;
static gchar *exportPathFile = NULL;

#define NO_PATH _("<span size=\"small\"><i>No stored path</i></span>")
#define PATH _("<span size=\"small\">Path has %d step(s)</span>")

/* Local routines. */
static GtkWidget *createInteriorBox();
static void updateValues(VisuData *dataObj);
static void updatePeriodic(VisuData *dataObj);
static void applyTranslation(gboolean setFree);
static void applyExpansion(float expansion[3]);
static void createPathExtension();
static void rebuildPathExtension(VisuData *data);
static void getCartesianTranslation(float trans[3]);

/* Local callbacks. */
static void onEnter(ToolPanel *toolPanel, gpointer data);
static void onVisuDataChanged(GObject *obj, VisuData* visuData, gpointer data);
static void onVisuDataReady(GObject *obj, VisuData* visuData, gpointer data);
static void onTranslationChanged(GtkSpinButton* button, gpointer data);
static void onTranslationChecked(GtkToggleButton* button, gpointer data);
static void onExpandChanged(GtkSpinButton* button, gpointer data);
static void onExpandChecked(GtkToggleButton* button, gpointer data);
static gboolean onElementRenderChanged(GSignalInvocationHint *ihint,
                                       guint nVals, const GValue *vals, gpointer data);
static void onNodePositionChanged(VisuData *dataObj, gpointer data);
static void onStippleExpandChanged(StippleComboBox *combo, gint stipple, gpointer data);
static void onUnitChanged(GtkComboBox *combo, gpointer data);
static void onDiffChanged(GtkToggleButton *toggle, gpointer data);
static void onAdjustChanged(GtkToggleButton *toggle, gpointer data);
static void onPathToggled(GtkToggleButton *toggle, gpointer data);
static void onPathSaveToggled(GtkToggleButton *toggle, gpointer data);
static void onPathClearClicked(GtkButton *bt, gpointer data);
static void onColorisePathToggled(GtkToggleButton *bt, gpointer data);
static void onShadePathSelected(ShadeComboBox *combo, Shade *shade, gpointer data);
static void onSavePathClicked(GtkButton *bt, gpointer data);
static void onLoadPathClicked(GtkButton *bt, gpointer data);
static void onDirBrowsed(VisuObject *obj, VisuGtkDirType type, gpointer user);

/**
 * panelGeometry_init: (skip)
 *
 * Should be used in the list declared in externalModules.h to be loaded by
 * V_Sim on start-up. This routine will create the #ToolPanel where the box
 * stuff can be tuned, such as the bounding box, its colour, and the actions linked
 * to the periodicity (translation, dupplication...).
 *
 * Returns: a newly created #ToolPanel object.
 */
ToolPanel* panelGeometry_init()
{
  int i;

  panelGeometry = toolPanelNew_withIconFromPath("Panel_geometry",
						_("Geometry operations"),
						_("Geometry"),
						"stock-geometry_20.png");
  if (!panelGeometry)
    return (ToolPanel*)0;

  toolPanelSet_dockable(TOOL_PANEL(panelGeometry), TRUE);

  /* Create the widgets that are needed even without the GTK interface be built. */
  imagePeriodic = gtk_image_new_from_stock(GTK_STOCK_DIALOG_WARNING,
					   GTK_ICON_SIZE_MENU);
  labelPeriodic = gtk_label_new("");
  vboxPeriodic = gtk_vbox_new(FALSE, 0);
  checkAllowTranslations =
    gtk_check_button_new_with_mnemonic(_("_Translations"));
  checkAllowExpand =
    gtk_check_button_new_with_mnemonic(_("_Expand nodes"));
  for (i = 0; i < 3; i++)
    {
      /* For the translation. */
      spinTransXYZ[i] = gtk_spin_button_new_with_range(-1, 1, 0.05);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinTransXYZ[i]), 0);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spinTransXYZ[i]), TRUE);
      /* For the expansion. */
      spinExpandXYZ[i] = gtk_spin_button_new_with_range(0, 5, 0.05);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spinExpandXYZ[i]), TRUE);
    }
  checkDiff = gtk_check_button_new_with_mnemonic(_("Show node _displacements"));
  checkOrdering = gtk_check_button_new_with_mnemonic(_("with re_ordering"));
  comboUnit = gtk_combo_box_text_new();
  checkVisuPathes = gtk_check_button_new_with_mnemonic(_("Use _pathes"));
  togglePathSave = gtk_toggle_button_new();
  stippleExpandBox = stippleComboBox_new();
  checkAdjust = gtk_check_button_new_with_mnemonic(_("Automatic zoom _adjustment on file loading"));

  /* Create the callbacks of all the sensitive widgets. */
  g_signal_connect(G_OBJECT(panelGeometry), "page-entered",
		   G_CALLBACK(onEnter), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataLoaded",
		   G_CALLBACK(onVisuDataChanged), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onVisuDataReady), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "DirectoryChanged",
		   G_CALLBACK(onDirBrowsed), (gpointer)0);
  g_signal_add_emission_hook(g_signal_lookup("ElementRenderedChanged", VISU_ELEMENT_TYPE),
                             0, onElementRenderChanged, (gpointer)0, (GDestroyNotify)0);

  /* Private parameters. */
  disableCallbacks = FALSE;
  widgetsNotBuilt  = TRUE;
  pathes           = (VisuPathes*)0;

  return TOOL_PANEL(panelGeometry);
}

static GtkWidget *createInteriorBox()
{
  GtkWidget *vbox, *hbox, *bt, *wd;
  GtkWidget *label, *align;
  int i;
#define X_LABEL _("dx:")
#define Y_LABEL _("dy:")
#define Z_LABEL _("dz:")
  char *xyz[3];
  const gchar **units;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();
#endif

  vbox = gtk_vbox_new(FALSE, 0);

  /**************************/
  /* The periodicity stuff. */
  /**************************/
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("<b>Periodic operations</b>"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

  gtk_box_pack_end(GTK_BOX(hbox), labelPeriodic, FALSE, FALSE, 2);
  gtk_label_set_use_markup(GTK_LABEL(labelPeriodic), TRUE);
  gtk_misc_set_alignment(GTK_MISC(labelPeriodic), 0.5, 0.5);
  gtk_box_pack_end(GTK_BOX(hbox), imagePeriodic, FALSE, FALSE, 0);

  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_container_add(GTK_CONTAINER(align), vboxPeriodic);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);
  /* The translations. */
  gtk_widget_set_tooltip_text(checkAllowTranslations,
			      _("Translations are given in box coordinates and nodes are "
				"automatically translated back into the bounding box."));
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), checkAllowTranslations, FALSE, FALSE, 0);
  transId = g_signal_connect(G_OBJECT(checkAllowTranslations), "toggled",
			     G_CALLBACK(onTranslationChecked), (gpointer)0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);

  xyz[0] = X_LABEL;
  xyz[1] = Y_LABEL;
  xyz[2] = Z_LABEL;
  for (i = 0; i < 3; i++)
    {
      label = gtk_label_new(xyz[i]);
      gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 3);
      gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

      g_signal_connect(G_OBJECT(spinTransXYZ[i]), "value-changed",
		       G_CALLBACK(onTranslationChanged), (gpointer)0);
      gtk_box_pack_start(GTK_BOX(hbox), spinTransXYZ[i], FALSE, FALSE, 0);
    }

  /* The replication. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), hbox, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(checkAllowExpand,
			      _("The size of the expansion is given in box coordinates."
				" Nodes are automatically translated back into the new"
				" defined area. The drawn bounding box is kept to the"
				" original size."));
  gtk_box_pack_start(GTK_BOX(hbox), checkAllowExpand, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(checkAllowExpand), "toggled",
		   G_CALLBACK(onExpandChecked), (gpointer)0);
  /* The stipple for expand. */
  label = gtk_label_new(_("line:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  /* The stipple pattern. */
  gtk_box_pack_start(GTK_BOX(hbox), stippleExpandBox, FALSE, FALSE, 3);
  g_signal_connect(G_OBJECT(stippleExpandBox), "stipple-selected",
		   G_CALLBACK(onStippleExpandChanged), (gpointer)0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);

  xyz[0] = X_LABEL;
  xyz[1] = Y_LABEL;
  xyz[2] = Z_LABEL;
  for (i = 0; i < 3; i++)
    {
      label = gtk_label_new(xyz[i]);
      gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 3);
      gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

      g_signal_connect(G_OBJECT(spinExpandXYZ[i]), "value-changed",
		       G_CALLBACK(onExpandChanged), (gpointer)0);
      gtk_box_pack_start(GTK_BOX(hbox), spinExpandXYZ[i], FALSE, FALSE, 0);
    }

  /********************/
  /* The units stuff. */
  /********************/
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("<b>Units and lengths</b>"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAdjust),
			       (gboolean)visu_rendering_window_getAutoAdjust());
  g_signal_connect(G_OBJECT(checkAdjust), "toggled",
		   G_CALLBACK(onAdjustChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(vbox), checkAdjust, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Set the unit of the file:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  gtk_widget_set_sensitive(comboUnit, FALSE);
  units = tool_physic_getUnitNames();
  for (i = 0; units[i]; i++)
    gtk_combo_box_text_append(GTK_COMBO_BOX_TEXT(comboUnit), (const gchar*)0, units[i]);
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboUnit), 0);
  g_signal_connect(G_OBJECT(comboUnit), "changed",
		   G_CALLBACK(onUnitChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox), comboUnit, FALSE, FALSE, 0);

  /************************/
  /* The multifile stuff. */
  /************************/
  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 15, 0, 1, 0);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);

  label = gtk_label_new(_("<b>Multi file actions</b>"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_set_name(label, "label_head");
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_container_add(GTK_CONTAINER(align), label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(checkOrdering,
			      _("On load of a new file, reorder the nodes to"
                                " minimize displacements."));
  gtk_box_pack_end(GTK_BOX(hbox), checkOrdering, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(checkDiff,
			      _("When a new file is loaded, draw arrows on  nodes that"
				" represent their displacements with respect to their"
				" previous positions."));
  g_signal_connect(G_OBJECT(checkDiff), "toggled",
		   G_CALLBACK(onDiffChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox), checkDiff, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(checkVisuPathes), "toggled",
		   G_CALLBACK(onPathToggled), (gpointer)0);
  gtk_widget_set_tooltip_text(checkVisuPathes,
			      _("Store differences between files and plot"
				" them as lines."));
  gtk_box_pack_start(GTK_BOX(hbox), checkVisuPathes, TRUE, TRUE, 0);
  hboxVisuPathes = gtk_hbox_new(FALSE, 2);
  gtk_widget_set_sensitive(hboxVisuPathes, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), hboxVisuPathes, FALSE, FALSE, 3);
  labelVisuPathes = gtk_label_new(NO_PATH);
  gtk_label_set_use_markup(GTK_LABEL(labelVisuPathes), TRUE);
  gtk_box_pack_start(GTK_BOX(hboxVisuPathes), labelVisuPathes, FALSE, FALSE, 3);
  gtk_widget_set_tooltip_text(togglePathSave, _("When toggled, store differences"
						" between files as pathes"
						" through nodes.."));
  gtk_container_add(GTK_CONTAINER(togglePathSave),
		    gtk_image_new_from_stock(GTK_STOCK_MEDIA_RECORD,
					     GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(togglePathSave), "toggled",
		   G_CALLBACK(onPathSaveToggled), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hboxVisuPathes), togglePathSave, FALSE, FALSE, 0);
  bt = gtk_button_new();
  gtk_widget_set_tooltip_text(bt, _("Remove all stored pathes."));
  gtk_container_add(GTK_CONTAINER(bt),
		    gtk_image_new_from_stock(GTK_STOCK_CLEAR,
					     GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(bt), "clicked",
		   G_CALLBACK(onPathClearClicked), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hboxVisuPathes), bt, FALSE, FALSE, 0);

  hboxVisuPathes2 = gtk_hbox_new(FALSE, 0);
  gtk_widget_set_sensitive(hboxVisuPathes2, FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), hboxVisuPathes2, FALSE, FALSE, 0);

  align = gtk_alignment_new(1., 0.5, 0, 1);
  gtk_box_pack_start(GTK_BOX(hboxVisuPathes2), align, TRUE, TRUE, 0);
  checkColorisePath = bt = gtk_check_button_new_with_mnemonic(_("colourise with: "));
  gtk_widget_set_tooltip_text(bt, _("If energy information was present"
				    " when loading file, colourise the pathes"
				    " with shading colours."));
  gtk_container_add(GTK_CONTAINER(align), bt);
  comboShadePath = wd = shadeComboBox_new(FALSE, FALSE);
  gtk_widget_set_sensitive(wd, FALSE);
  g_signal_connect(G_OBJECT(bt), "toggled",
		   G_CALLBACK(onColorisePathToggled), (gpointer)wd);
  g_signal_connect(G_OBJECT(wd), "shade-selected",
		   G_CALLBACK(onShadePathSelected), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hboxVisuPathes2), wd, FALSE, FALSE, 0);
  hboxIOVisuPathes = gtk_hbox_new(FALSE, 2);
/*   gtk_widget_set_sensitive(hboxIOVisuPathes, FALSE); */
  gtk_box_pack_start(GTK_BOX(hboxVisuPathes2), hboxIOVisuPathes, FALSE, FALSE, 3);
  bt = gtk_button_new();
  gtk_widget_set_tooltip_text(bt, _("Read a set of pathes from a file and"
				    " add them to the current set."));
  gtk_container_add(GTK_CONTAINER(bt),
		    gtk_image_new_from_stock(GTK_STOCK_OPEN,
					     GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(bt), "clicked",
		   G_CALLBACK(onLoadPathClicked), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hboxIOVisuPathes), bt, FALSE, FALSE, 0);
  bt = gtk_button_new();
  gtk_widget_set_tooltip_text(bt, _("Save the current set of pathes"
				    " to an XML file."));
  gtk_container_add(GTK_CONTAINER(bt),
		    gtk_image_new_from_stock(GTK_STOCK_SAVE,
					     GTK_ICON_SIZE_MENU));
  g_signal_connect(G_OBJECT(bt), "clicked",
		   G_CALLBACK(onSavePathClicked), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hboxIOVisuPathes), bt, FALSE, FALSE, 0);

  gtk_widget_show_all(vbox);
  return vbox;
}
static void updateSensitive(VisuData *dataObj)
{
  gboolean per[3], setTrans, setExpand;
  VisuDataBoxBoundaries bc;

  setExpand = TRUE;
  setTrans  = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowExpand));
  if (dataObj)
    {
      bc = visu_data_getBoundaryConditions(dataObj);
      per[0] = (bc != VISU_DATA_BOX_FREE && bc != VISU_DATA_BOX_SURFACE_YZ);
      per[1] = (bc != VISU_DATA_BOX_FREE && bc != VISU_DATA_BOX_SURFACE_ZX);
      per[2] = (bc != VISU_DATA_BOX_FREE && bc != VISU_DATA_BOX_SURFACE_XY);
      if (bc == VISU_DATA_BOX_FREE)
        {
          setTrans = setExpand = FALSE;
          gtk_widget_show(imagePeriodic);
        }
      else
        gtk_widget_hide(imagePeriodic);
    }
  else
    {
      gtk_widget_hide(imagePeriodic);
      setTrans = setExpand = FALSE;
      per[0] = FALSE;
      per[1] = FALSE;
      per[2] = FALSE;
    }
  gtk_widget_set_sensitive(checkAllowTranslations, setTrans);
  gtk_widget_set_sensitive(spinTransXYZ[0], setTrans && per[0]);
  gtk_widget_set_sensitive(spinTransXYZ[1], setTrans && per[1]);
  gtk_widget_set_sensitive(spinTransXYZ[2], setTrans && per[2]);
  gtk_widget_set_sensitive(checkAllowExpand, setExpand);
  gtk_widget_set_sensitive(spinExpandXYZ[0], setExpand && per[0]);
  gtk_widget_set_sensitive(spinExpandXYZ[1], setExpand && per[1]);
  gtk_widget_set_sensitive(spinExpandXYZ[2], setExpand && per[2]);
}
static void updateValues(VisuData *dataObj)
{
  int i;
  float *transXYZ, transBox[3], expand[3];
  gboolean set;
  guint16 stipple;

  disableCallbacks = TRUE;
  DBG_fprintf(stderr, "Panel Geometry: update inside values.\n");

  stipple = visu_glExt_box_getExpandStipple();
  if (!stippleComboBoxSet_selectionByStipple(STIPPLE_COMBOX(stippleExpandBox), stipple))
    {
      stippleComboBoxAdd_pattern(STIPPLE_COMBOX(stippleExpandBox), stipple);
      stippleComboBoxSet_selectionByStipple(STIPPLE_COMBOX(stippleExpandBox), stipple);
    }

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAdjust),
                               visu_rendering_window_getAutoAdjust());
  
  /* Values dependent on a visuData. */
  DBG_fprintf(stderr, "Panel Geometry: update values for object %p.\n",
	      (gpointer)dataObj);
  if (dataObj)
    {
      transXYZ = visu_data_getXYZtranslation(dataObj);
      visu_data_convertXYZtoBoxCoordinates(dataObj, transBox, transXYZ);
      set = visu_data_getTranslationStatus(dataObj);
      if (set)
        for (i = 0; i < 3; i++)
          gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinTransXYZ[i]),
                                    (double)transBox[i]);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowTranslations),
                                   set);
      g_free(transXYZ);
      if (visu_data_getBoundaryConditions(dataObj) != VISU_DATA_BOX_FREE)
	{
	  visu_data_getExtension(dataObj, expand);
	  set = (expand[0] != 0. || expand[1] != 0. || expand[2] != 0.);
	  if (set)
	    for (i = 0; i < 3; i++)
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]),
					(double)expand[i]);
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowExpand), set);
	}
      gtk_combo_box_set_active(GTK_COMBO_BOX(comboUnit),
			       visu_data_getUnit(dataObj));
      gtk_widget_set_sensitive(comboUnit, TRUE);
    }
  else
    {
      for (i = 0; i < 3; i++)
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinTransXYZ[i]), 0.);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowTranslations), FALSE);
      for (i = 0; i < 3; i++)
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]), 0.);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowExpand), FALSE);
      gtk_widget_set_sensitive(comboUnit, FALSE);
    }
  disableCallbacks = FALSE;
  
  /* Set the sensitivity of the translation. */
  updateSensitive(dataObj);
}
static void updatePeriodic(VisuData *dataObj)
{
  VisuDataBoxBoundaries bc;
  gchar *lbl;

  if (dataObj)
    {
      bc = visu_data_getBoundaryConditions(dataObj);
      switch (bc)
        {
        case VISU_DATA_BOX_FREE:
          lbl = g_markup_printf_escaped("<i>%s</i>", _("non periodic data"));
          break;
        case VISU_DATA_BOX_SURFACE_XY:
          lbl = g_strdup(_("(surface XY)"));
          break;
        case VISU_DATA_BOX_SURFACE_YZ:
          lbl = g_strdup(_("(surface YZ)"));
          break;
        case VISU_DATA_BOX_SURFACE_ZX:
          lbl = g_strdup(_("(surface ZX)"));
          break;
        case VISU_DATA_BOX_PERIODIC:
          lbl = g_strdup(_("(periodic)"));
          break;
        default:
          lbl = g_strdup("");
        }
    }
  else
    lbl = g_strdup("");
  gtk_label_set_markup(GTK_LABEL(labelPeriodic), lbl);
  g_free(lbl);
}
static void updateGeometry(VisuData *visuData)
{
  VisuData *dataRef;
  gboolean res, new;
  gchar *text;

  dataRef = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  DBG_fprintf(stderr, "Panel Geometry: updating geometry between %p and %p.\n",
              (gpointer)dataRef, (gpointer)visuData);
  if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDiff)) ||
       gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglePathSave)))&& dataRef)
    {
      res = visu_geodiff_new(dataRef, visuData,
                         gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkOrdering)));
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDiff)))
	visu_geodiff_rebuildList(visuData);
      if (res && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglePathSave)))
	{
	  g_return_if_fail(pathes);

	  new = visu_pathes_addFromDiff(pathes, visuData);
	  if (new)
	    visu_pathes_constrainInBox(pathes, visuData);
	  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkVisuPathes)))
	    rebuildPathExtension((VisuData*)0);
	  text = g_strdup_printf(PATH, visu_pathes_getLength(pathes));
	  gtk_label_set_markup(GTK_LABEL(labelVisuPathes), text);
	  g_free(text);
	}
    }
}
static void applyTranslation(gboolean setFree)
{
  VisuData *data;
  float cartCoord[3];
  gboolean rebuild;

  data = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  if (!data)
    return;

  if (setFree)
    {
      cartCoord[0] = 0.f;
      cartCoord[1] = 0.f;
      cartCoord[2] = 0.f;
      rebuild = (gboolean)visu_data_setXYZtranslation(data, cartCoord);
      rebuild = visu_data_constrainedFree(data) || rebuild;
    }
  else
    {
      getCartesianTranslation(cartCoord);
      rebuild = (gboolean)visu_data_setXYZtranslation(data, cartCoord);
      rebuild = visu_data_constrainedInTheBox(data) || rebuild;
    }

  /* Update the pathes if necessary. */
  if (pathes)
    {
      visu_pathes_setTranslation(pathes, cartCoord);
      visu_pathes_constrainInBox(pathes, data);
      rebuildPathExtension((VisuData*)0);
    }

  if (rebuild)
    {
      visu_data_emitNodePositionChanged(data);
      visu_data_createAllNodes(data);
      VISU_ADD_REDRAW;
    }
}
static void applyExpansion(float expansion[3])
{
  VisuData *data;
  VisuOpenGLView *view;
  gboolean rebuild, redraw;
  float *ext;

  data = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  if (!data)
    return;

  redraw = visu_data_replicate(data, expansion, &rebuild);
  if (rebuild)
    visu_data_createAllNodes(data);
  if (redraw)
    {
      view = visu_data_getOpenGLView(data);
      ext = visu_data_getBoxExtens(data);
      openGLProject(view->window, view->camera, ext[1]);
      VISU_ADD_REDRAW;
    }
}
static void createPathExtension()
{
  int id;

  /* Get an OpenGL identifier to store the arrows and labels. */
  id        = visu_openGL_objectList_new(1);
  extVisuPathes = visu_extension_new("pathes", "pathes",
				  "Representation of pathes.",
				  id, rebuildPathExtension);
  visu_extension_setPriority(extVisuPathes, VISU_EXTENSION_PRIORITY_LAST - 1);
  extVisuPathes->used = 0;
  visuExtensions_add(extVisuPathes);
}
static void rebuildPathExtension(VisuData *data _U_)
{
  glNewList(extVisuPathes->objectListId, GL_COMPILE);
  if (pathes)
    visu_pathes_draw(pathes);
  glEndList();
}
static void getCartesianTranslation(float trans[3])
{
  float transBox[3];
  VisuData *data;

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
    {
      transBox[0] = (float)gtk_spin_button_get_value
	(GTK_SPIN_BUTTON(spinTransXYZ[0]));
      transBox[1] = (float)gtk_spin_button_get_value
	(GTK_SPIN_BUTTON(spinTransXYZ[1]));
      transBox[2] = (float)gtk_spin_button_get_value
	(GTK_SPIN_BUTTON(spinTransXYZ[2]));
      data = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
      visu_data_convertBoxCoordinatestoXYZ(data, trans, transBox);
    }
  else
    {
      trans[0] = 0.f;
      trans[1] = 0.f;
      trans[2] = 0.f;
    }
}

/*************/
/* Callbacks */
/*************/
static void onEnter(ToolPanel *toolPanel _U_, gpointer data _U_)
{
  VisuData *dataObj;

  if (widgetsNotBuilt)
    {
      DBG_fprintf(stderr, "Panel Geometry: first build on enter.\n");
      gtk_container_add(GTK_CONTAINER(panelGeometry), createInteriorBox());
      widgetsNotBuilt = FALSE;
    }
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  updateValues(dataObj);
  updatePeriodic(dataObj);
  updateSensitive(dataObj);
}
static void onStippleExpandChanged(StippleComboBox *combo _U_, gint stipple,
				   gpointer data _U_)
{
  VisuData *dataObj;

  if (disableCallbacks)
    return;

  if (visu_glExt_box_setExpandStipple(stipple))
    {
      dataObj = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
      if (!dataObj) 
	return;
      visu_glExt_box_draw(dataObj);
      VISU_ADD_REDRAW;
    }
}
static void onVisuDataChanged(GObject *obj _U_, VisuData* visuData, gpointer data _U_)
{
  float cartCoord[3];
  float transBox[3];
  float expand[3];
  int i, val;
  VisuData *dataRef;
  
  DBG_fprintf(stderr, "Panel Geometry: Catch 'dataLoaded' signal, update.\n");

  /* Update the sensitivity of the periodic stuff. */
  updatePeriodic(visuData);
  updateSensitive(visuData);

  if (!visuData)
    return;

  dataRef = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  /* Apply the periodic stuffs. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)) &&
      (visu_data_getBoundaryConditions(visuData) != VISU_DATA_BOX_FREE &&
       dataRef && visu_data_getBoundaryConditions(dataRef) != VISU_DATA_BOX_FREE))
    {
      DBG_fprintf(stderr, "Panel Geometry: Catch 'dataLoaded' signal,"
		  " applying current translations.\n");
      for (i = 0; i < 3; i++)
	transBox[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinTransXYZ[i]));
      visu_data_convertBoxCoordinatestoXYZ(visuData, cartCoord, transBox);
      visu_data_setXYZtranslation(visuData, cartCoord);
      visu_data_constrainedInTheBox(visuData);
    }
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowExpand)) &&
      visu_data_getBoundaryConditions(visuData) != VISU_DATA_BOX_FREE)
    {
      DBG_fprintf(stderr, "Panel Geometry: Catch 'dataLoaded' signal,"
		  " applying current expansion.\n");
      for (i = 0; i < 3; i++)
	expand[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]));
      visu_data_replicate(visuData, expand, (gboolean*)0);
    }

  /* Update the units if necessary. */
  if (visu_basic_getPreferedUnit() == unit_undefined)
    {
      val = gtk_combo_box_get_active(GTK_COMBO_BOX(comboUnit));
      if (val > 0 && visu_data_getUnit(visuData) != unit_undefined)
	visu_data_setUnit(visuData, (ToolUnits)val);
    }

  /* Make a geometry diff. */
  updateGeometry(visuData);

  g_signal_connect(G_OBJECT(visuData), "NodePositionChanged",
		   G_CALLBACK(onNodePositionChanged), (gpointer)0);
}
static void onVisuDataReady(GObject *obj _U_, VisuData* visuData, gpointer data _U_)
{
  DBG_fprintf(stderr, "Panel Geometry: Catch 'dataReadyForRendering' signal,"
	      " update values.\n");
  if (visuData)
    updateValues(visuData);
}
static void onTranslationChanged(GtkSpinButton* button _U_, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
    return;

  applyTranslation(FALSE);
}
static void onTranslationChecked(GtkToggleButton* button, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (gtk_toggle_button_get_active(button))
    applyTranslation(FALSE);
  else
    applyTranslation(TRUE);
}
static gboolean onElementRenderChanged(GSignalInvocationHint *ihint _U_,
                                       guint nVals _U_, const GValue *vals,
                                       gpointer data _U_)
{
  VisuData *dataObj;
  VisuElement *element;
				   
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
    return TRUE;

  element = VISU_ELEMENT(g_value_get_object(vals + 0));
  if (element->rendered)
    {
      DBG_fprintf(stderr, "Panel Geometry: caught 'ElementRenderedChanged' applying"
		  " translation for element '%s'.\n", element->name);
      dataObj = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
      if (visu_data_constrainedElementInTheBox(dataObj, element))
	visu_data_emitNodePositionChanged(dataObj);
    }
  else
    DBG_fprintf(stderr, "Panel Geometry: caught 'ElementRenderedChanged' but"
		" do not apply translation since element '%s' is masked.\n",
		element->name);

  return TRUE;
}
static void onExpandChanged(GtkSpinButton* button _U_, gpointer data _U_)
{
  float expansion[3];
  int i;

  if (disableCallbacks)
    return;

  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowExpand)))
    return;

  for (i = 0; i < 3; i++)
    expansion[i] =
      (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]));
  applyExpansion(expansion);
}
static void onExpandChecked(GtkToggleButton* button, gpointer data _U_)
{
  float expansion[3], *ext;
  float transBox[3], cartCoord[3];
  float transZero[3] = { 0.f, 0.f, 0.f};
  int i;
  gboolean set;
  VisuData *dataObj;
  VisuOpenGLView *view;
  gboolean rebuildTrans, rebuildExp, redraw;

  if (disableCallbacks)
    return;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  if (!dataObj)
    return;

  redraw = rebuildTrans = FALSE;
  if (gtk_toggle_button_get_active(button))
    {
      for (i = 0; i < 3; i++)
	expansion[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]));

      /* Before applying expansion we put everything into the box
	 if it is not there yet. */
      if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
	{
	  g_signal_handler_block(G_OBJECT(checkAllowTranslations), transId);
	  gtk_toggle_button_set_active
	    (GTK_TOGGLE_BUTTON(checkAllowTranslations), TRUE);
	  g_signal_handler_unblock(G_OBJECT(checkAllowTranslations), transId);
	  for (i = 0; i < 3; i++)
	    transBox[i] =
	      (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinTransXYZ[i]));
	  visu_data_convertBoxCoordinatestoXYZ(dataObj, cartCoord, transBox);
	  rebuildTrans = (gboolean)visu_data_setXYZtranslation(dataObj, cartCoord);
	  rebuildTrans = visu_data_constrainedInTheBox(dataObj) || rebuildTrans;
	  if (rebuildTrans)
	    visu_data_emitNodePositionChanged(dataObj);
	  transStatus = FALSE;
	}
      else
	  transStatus = TRUE;

      redraw = visu_data_replicate(dataObj, expansion, &rebuildExp);
    }
  else
    {
      redraw = rebuildExp = visu_data_restore(dataObj);

      /* We release the translation if not checked. */
      if (!transStatus)
	{
	  g_signal_handler_block(G_OBJECT(checkAllowTranslations), transId);
	  gtk_toggle_button_set_active
	    (GTK_TOGGLE_BUTTON(checkAllowTranslations), FALSE);
	  g_signal_handler_unblock(G_OBJECT(checkAllowTranslations), transId);
	  rebuildTrans = (gboolean)visu_data_setXYZtranslation(dataObj, transZero);
	  rebuildTrans = visu_data_constrainedFree(dataObj) || rebuildTrans;
	  if (rebuildTrans)
	    visu_data_emitNodePositionChanged(dataObj);
	}
    }
  if (redraw)
    {
      view = visu_data_getOpenGLView(dataObj);
      ext = visu_data_getBoxExtens(dataObj);
      openGLProject(view->window, view->camera, ext[1]);
    }
  if (rebuildTrans || rebuildExp)
    visu_data_createAllNodes(dataObj);
  if (rebuildTrans || redraw)
    VISU_ADD_REDRAW;

  /* Set the sensitivity of the translation. */
  set = gtk_toggle_button_get_active(button);
  updateSensitive(dataObj);
}
static void onUnitChanged(GtkComboBox *combo, gpointer data _U_)
{
  gint val;
  ToolUnits unit;
  VisuData *dataObj;

  val = gtk_combo_box_get_active(combo);
  g_return_if_fail(val >= 0);

  unit = (ToolUnits)val;
  
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  g_return_if_fail(dataObj);

  if (visu_data_setUnit(dataObj, unit))
    {
      visu_data_createAllNodes(dataObj);

      openGLModelize(visu_data_getOpenGLView(dataObj)->camera);

      VISU_ADD_REDRAW;
    }
}
static void onAdjustChanged(GtkToggleButton *toggle, gpointer data _U_)
{
  visu_rendering_window_setAutoAdjust(gtk_toggle_button_get_active(toggle));
}
static void onDiffChanged(GtkToggleButton *toggle, gpointer data _U_)
{
  visu_geodiff_setActive(gtk_toggle_button_get_active(toggle));
  VISU_ADD_REDRAW;
}
static void onNodePositionChanged(VisuData *dataObj, gpointer data _U_)
{
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDiff)))
    visu_geodiff_rebuildList(dataObj);
}
static void onPathToggled(GtkToggleButton *toggle, gpointer data _U_)
{
  gboolean used;
  float t[3];

  used = gtk_toggle_button_get_active(toggle);
  gtk_widget_set_sensitive(hboxVisuPathes, used);
  gtk_widget_set_sensitive(hboxVisuPathes2, used);

  if (!pathes)
    {
      getCartesianTranslation(t);
      pathes = visu_pathes_new(t);
    }

  if (!extVisuPathes)
    createPathExtension();

  extVisuPathes->used = (int)used;
  VISU_ADD_REDRAW;
}
static void pinPath()
{
  VisuData *dataObj;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelGeometry));
  if (dataObj && pathes)
    visu_pathes_pinPositions(pathes, dataObj);
  panelBrowserSet_infoMessage(_("Recording pathes"), GTK_MESSAGE_INFO);
}
static void onPathSaveToggled(GtkToggleButton *toggle, gpointer data _U_)
{
  if (gtk_toggle_button_get_active(toggle))
    pinPath();
  else
    panelBrowserSet_infoMessage((const gchar*)0, GTK_MESSAGE_INFO);
}
static void onDirBrowsed(VisuObject *obj _U_, VisuGtkDirType type, gpointer user _U_)
{
  if (type == VISU_DIR_BROWSER)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(togglePathSave), FALSE);
}
static void onPathClearClicked(GtkButton *bt _U_, gpointer data _U_)
{
  if (pathes)
    visu_pathes_empty(pathes);
  rebuildPathExtension((VisuData*)0);
  
  gtk_label_set_markup(GTK_LABEL(labelVisuPathes), NO_PATH);

  VISU_ADD_REDRAW;
}
static void onColorisePathToggled(GtkToggleButton *bt, gpointer data)
{
  gboolean set;
  Shade *shade;

  set = gtk_toggle_button_get_active(bt);

  gtk_widget_set_sensitive(GTK_WIDGET(data), set);
  shade = (set)?shadeComboBoxGet_selectedShade(SHADE_COMBOX(data)):(Shade*)0;
  if (visu_pathes_setShade(pathes, shade))
    {
      rebuildPathExtension((VisuData*)0);
      VISU_ADD_REDRAW;
    }
}
static void onShadePathSelected(ShadeComboBox *combo _U_, Shade *shade,
				gpointer data _U_)
{
  if (visu_pathes_setShade(pathes, shade))
    {
      rebuildPathExtension((VisuData*)0);
      VISU_ADD_REDRAW;
    }
}
static void onSavePathClicked(GtkButton *bt _U_, gpointer data _U_)
{
  GtkWidget *wd;
  gint response;
  GError *error;
  gchar *base;

  if (visu_pathes_getLength(pathes) < 1)
    return;

  wd = gtk_file_chooser_dialog_new(_("Export current set of pathes."), (GtkWindow*)0,
				   GTK_FILE_CHOOSER_ACTION_SAVE,
				   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				   GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				   NULL);
  if (!exportPathFile)
    exportPathFile = g_build_filename(g_get_current_dir(), _("pathes.xml"), NULL);
  base = g_path_get_basename(exportPathFile);
  gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(wd), base);
  g_free(base);
#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 7
  gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(wd), TRUE);
#endif
  do
    {
      response = gtk_dialog_run(GTK_DIALOG(wd));
      if (exportPathFile)
	g_free(exportPathFile);
      exportPathFile = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(wd));
      switch (response)
	{
	case GTK_RESPONSE_ACCEPT:
	  error = (GError*)0;
	  if (!visu_pathes_exportToXML(pathes, exportPathFile, &error))
	    {
	      visuGtkRaise_warning(_("Export current set of pathes."),
				   error->message, GTK_WINDOW(wd));
	      g_error_free(error);
	      response = GTK_RESPONSE_NONE;
	    }
	  break;
	default:
	  response = GTK_RESPONSE_ACCEPT;
	  break;
	}
    }
  while (response != GTK_RESPONSE_ACCEPT);
  gtk_widget_destroy(wd);
}
static void onLoadPathClicked(GtkButton *bt _U_, gpointer data _U_)
{
  GtkWidget *wd;
  gint response;
  GError *error;
  gchar *text, *directory;

  wd = gtk_file_chooser_dialog_new(_("Load a set of pathes."), (GtkWindow*)0,
				   GTK_FILE_CHOOSER_ACTION_OPEN,
				   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				   GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				   NULL);
  directory = visuGtkGet_lastOpenDirectory();
  if (directory)
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(wd), directory);
  do
    {
      response = gtk_dialog_run(GTK_DIALOG(wd));
      if (exportPathFile)
	g_free(exportPathFile);
      exportPathFile = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(wd));
      switch (response)
	{
	case GTK_RESPONSE_ACCEPT:
	  error = (GError*)0;
	  if (!visu_pathes_parseFromXML(exportPathFile, pathes, &error))
	    {
	      visuGtkRaise_warning(_("Load a set of pathes."),
				   error->message, GTK_WINDOW(wd));
	      g_error_free(error);
	      response = GTK_RESPONSE_NONE;
	    }
	  break;
	default:
	  response = GTK_RESPONSE_ACCEPT;
	  break;
	}
    }
  while (response != GTK_RESPONSE_ACCEPT);
  gtk_widget_destroy(wd);

  rebuildPathExtension((VisuData*)0);
  
  text = g_strdup_printf(PATH, visu_pathes_getLength(pathes));
  gtk_label_set_markup(GTK_LABEL(labelVisuPathes), text);
  g_free(text);

  VISU_ADD_REDRAW;
}
