/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "fogAndBGColor.h"

#include <opengl.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_configFile.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/text.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolConfigFile.h>

#include <GL/gl.h>
#include <GL/glu.h>

/**
 * SECTION:fogAndBGColor
 * @short_description: Handle the background colour and the fog.
 *
 * This module is used to support a background colour and to tune the
 * fog. This last one can be turn on or off and its colour can be
 * either a user defined one or the one of the background. The fog is
 * a linear blending into the fog colour. It starts at a given z
 * position (in the camera basis set) and ends at a lower z.
 */

static VisuExtension* extensionFog;

/* Parameters & resources*/
/* This is a boolean to control is the axes is render or not. */
#define FLAG_RESOURCE_FOG_USED   "fog_is_on"
#define DESC_RESOURCE_FOG_USED   "Control if the fog is used ; boolean (0 or 1)"
#define RESOURCE_FOG_USED_DEFAULT 0
static gboolean readFogIsOn(gchar **lines, int nbLines,
			    int position, VisuData *dataObj, GError **error);
#define FLAG_RESOURCE_FOG_SPECIFIC   "fog_color_is_specific"
#define DESC_RESOURCE_FOG_SPECIFIC   "Control if the fog uses a specific color ; boolean (0 or 1)"
#define RESOURCE_FOG_SPECIFIC_DEFAULT 0
static int fogColorSpecific;
static gboolean readFogSpecific(gchar **lines, int nbLines,
				int position, VisuData *dataObj, GError **error);
/* A resource to control the color used to render the lines of the axes. */
#define FLAG_RESOURCE_FOG_COLOR   "fog_specific_color"
#define DESC_RESOURCE_FOG_COLOR   "Define the color of the fog ; four floating point values (0. <= v <= 1.)"
static float fogRGB[4];
static gboolean readFogColor(gchar **lines, int nbLines,
			     int position, VisuData *dataObj, GError **error);
#define FLAG_RESOURCE_FOG_STARTEND "fog_start_end"
#define DESC_RESOURCE_FOG_STARTEND "Define the position of the fog ; two floating point values (0. <= v <= 1.)"
static float fog_start, fog_end;
static gboolean readFogStartEnd(gchar **lines, int nbLines,
				int position, VisuData *dataObj, GError **error);
/* A resource to control the color of the background. */
#define FLAG_RESOURCE_BG_COLOR   "backgroundColor_color"
#define DESC_RESOURCE_BG_COLOR   "Set the background of the background ; four floating point values (0. <= v <= 1.)"
static float bgRGB[4];
static gboolean readBgColor(gchar **lines, int nbLines,
			    int position, VisuData *dataObj, GError **error);

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static void exportResourcesFog(GString *data, VisuData *dataObj);

static float fogRGBDefault[4] = {0., 0., 0., 1.};
static float fogStartEndDefault[2] = {0.3, 0.7};
static float bgRGBDefault[4] = {0., 0., 0., 0.};

static guchar *bgImage;
static gboolean bgImageAlpha, bgImageFit;
static guint bgImageW, bgImageH;
static gchar *bgImageTitle;

static GLuint texName = 0;

static gboolean fogIsOn;
static gboolean fogHasBeenBuilt;

/* Local callbacks */
static void rebuildOpenGLValues(VisuData *dataObj);
static void onDataReadySignal(GObject *visu, VisuData *dataObj, gpointer data);
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data);
static void onFogParameterChanged(VisuData *dataObj, VisuOpenGLView *view, gpointer data);
static void onBgImageRescale(VisuData *dataObj, VisuOpenGLView *view, gpointer data);
static void rebuildFogOnResources(GObject *obj, VisuData *dataObj, gpointer data);

/* Local methods */
static void refreshBgRGB();
static void createBgImage();
static void createBgChess();

VisuExtension* initExtFogAndBG()
{
  char *description = _("Draw a nice fog around the elements and set the color of the background.");
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr,"Initialising the fog & bg OpenGL extension...\n");
  extensionFog = visu_extension_new(VISU_GLEXT_FOG_AND_BG_ID,
				     _("Fog and background color"),
				     description, visu_openGL_objectList_new(3),
				     rebuildOpenGLValues);
  visu_extension_setPriority(extensionFog, VISU_EXTENSION_PRIORITY_BACKGROUND);
  visu_extension_setSaveOpenGLState(extensionFog, TRUE);

  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_FOG_USED,
					  DESC_RESOURCE_FOG_USED,
					  1, readFogIsOn);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_FOG_SPECIFIC,
					  DESC_RESOURCE_FOG_SPECIFIC,
					  1, readFogSpecific);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_FOG_COLOR,
					  DESC_RESOURCE_FOG_COLOR,
					  1, readFogColor);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_FOG_STARTEND,
					  DESC_RESOURCE_FOG_STARTEND,
					  1, readFogStartEnd);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BG_COLOR,
					  DESC_RESOURCE_BG_COLOR,
					  1, readBgColor);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesFog);

  /* Initialisation des valeurs par dfaut. */
  extensionFog->used = 1;
  fogIsOn            = FALSE;
  fogHasBeenBuilt    = FALSE;
  fogColorSpecific   = RESOURCE_FOG_SPECIFIC_DEFAULT;
  memcpy(fogRGB, fogRGBDefault, 4 * sizeof(float));
  fog_start = fogStartEndDefault[0];
  fog_end   = fogStartEndDefault[1];
  memcpy(bgRGB,  bgRGBDefault,  4 * sizeof(float));
  bgImage      = (guchar*)0;
  bgImageTitle = (gchar*)0;

  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
		   G_CALLBACK(rebuildFogOnResources), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReadySignal), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onDataNewSignal), (gpointer)0);

  return extensionFog;
}

/* Method used to change the value of the parameter fog. */
int visu_glExt_fog_setValues(float rgba[4], int mask)
{
  int diff = 0;
  
  if (mask & TOOL_COLOR_MASK_R && fogRGB[0] != rgba[0])
    {
      fogRGB[0] = rgba[0];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_G && fogRGB[1] != rgba[1])
    {
      fogRGB[1] = rgba[1];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_B && fogRGB[2] != rgba[2])
    {
      fogRGB[2] = rgba[2];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_A && fogRGB[3] != rgba[3])
    {
      fogRGB[3] = rgba[3];
      diff = 1;
    }
  if (!diff)
    return 0;

  /* The method which calls the set method should ask for redraw. */
  if (fogColorSpecific)
    glFogfv(GL_FOG_COLOR, fogRGB);

  return (fogColorSpecific && fogIsOn);
}

/* Get methods. */
void visu_glExt_fog_getValues(float rgba[4])
{
  memcpy(rgba, fogRGB, sizeof(float) * 4);
}
/* Method used to change the value of the parameter axes_is_on. */
gboolean visu_glExt_fog_setOn(gboolean value)
{
  if (value == fogIsOn)
    return FALSE;

  fogIsOn = value;
  if (value)
    {
      glEnable(GL_FOG);
      glFogi(GL_FOG_MODE, GL_LINEAR);
/*       glFogi(GL_FOG_MODE, GL_EXP);  */
/*       glFogf(GL_FOG_DENSITY, 0.03f);  */
    }
  else
    glDisable(GL_FOG);
    
  return (value && !fogHasBeenBuilt);
}
gboolean visu_glExt_fog_getOn()
{
  return fogIsOn;
}
gboolean visu_glExt_fog_setUseSpecificColor(gboolean value)
{
  if (value == fogColorSpecific)
    return FALSE;

  fogColorSpecific = value;
  visu_glExt_fog_create_color();
    
  return fogIsOn;
}
gboolean visu_glExt_fog_getUseSpecificColor()
{
  return fogColorSpecific;
}
gboolean visu_glExt_fog_setStartEndValues(float startEnd[2], int mask)
{
  int diff = 0;
  
  if (mask & VISU_GLEXT_FOG_MASK_START && fog_start != startEnd[0])
    {
      fog_start = CLAMP(startEnd[0], 0., 1.);
      if (mask & VISU_GLEXT_FOG_MASK_END)
	{
	  if (fog_start >= startEnd[1])
	    fog_start = startEnd[1] - 0.001;
	}
      else
	if (fog_start >= fog_end)
	  fog_start = fog_end - 0.001;
      diff = 1;
    }
  if (mask & VISU_GLEXT_FOG_MASK_END && fog_end != startEnd[1])
    {
      fog_end = CLAMP(startEnd[1], 0., 1.);
      if (fog_end <= fog_start)
	fog_end = fog_start + 0.001;
      diff = 1;
    }
  if (!diff)
    return FALSE;
  
  /* The method which calls the set method should ask for redraw. */
/*   visu_glExt_fog_create((GObject*)0, (gpointer)0); */
  fogHasBeenBuilt = FALSE;
  
  return (fogIsOn);
}
float visu_glExt_fog_getStart()
{
  return fog_start;
}
float visu_glExt_fog_getEnd()
{
  return fog_end;
}

/* Method used to change the value of the parameter backgroundColor. */
int visu_glExt_bg_setValues(float rgba[4], int mask)
{
  int diff = 0, i, j, c;
  GLubyte chessboard[32][32][3];

  if (mask & TOOL_COLOR_MASK_R && bgRGB[0] != rgba[0])
    {
      bgRGB[0] = rgba[0];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_G && bgRGB[1] != rgba[1])
    {
      bgRGB[1] = rgba[1];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_B && bgRGB[2] != rgba[2])
    {
      bgRGB[2] = rgba[2];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_A && bgRGB[3] != rgba[3])
    {
      bgRGB[3] = rgba[3];
      diff = 1;
      if (bgRGB[3] < 1.f)
	{
	  DBG_fprintf(stderr, "Extension bg: recreate the chessborad.\n");
	  /* We create again the chessboard texture. */
	  for (i = 0; i < 32; i++)
	    for (j = 0; j < 32; j++)
	      {
		c = 128 + ( ((i&0x10)==0) ^ ((j&0x10) == 0) ) * 64;
		chessboard[i][j][0] = (GLubyte)(255.f * bgRGB[0] * bgRGB[3] + (1.f - bgRGB[3]) * c);
		chessboard[i][j][1] = (GLubyte)(255.f * bgRGB[1] * bgRGB[3] + (1.f - bgRGB[3]) * c);
		chessboard[i][j][2] = (GLubyte)(255.f * bgRGB[2] * bgRGB[3] + (1.f - bgRGB[3]) * c);
	      }
	  /* We bind the texture. */
	  if (texName == 0)
	    glGenTextures(1, &texName);
	  glBindTexture(GL_TEXTURE_2D, texName);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 32, 32, 0,
		       GL_RGB, GL_UNSIGNED_BYTE, chessboard);
	  createBgChess();
	}
      else
	glDeleteLists(extensionFog->objectListId + 1, 1);
    }
  if (!diff)
    return 0;

  /* The method which calls the set method should ask for redraw. */
  glClearColor(bgRGB[0], bgRGB[1], bgRGB[2], bgRGB[3]);

  /* Update the fog color */
  visu_glExt_fog_create_color();
  return 1;
}

/* Get methods. */
void visu_glExt_bg_getValues(float rgba[4])
{
  memcpy(rgba, bgRGB, sizeof(float) * 4);
}

void visu_glExt_bg_setImage(const guchar *imageData, guint width, guint height,
		 gboolean alpha, const gchar *title, gboolean fit)
{
  guint n;

  if (bgImage)
    g_free(bgImage);
  if (bgImageTitle)
    g_free(bgImageTitle);
  bgImageTitle = (gchar*)0;

  if (!imageData)
    {
      glDeleteLists(extensionFog->objectListId + 2, 1);
      bgImage = (guchar*)0;
      return;
    }

  DBG_fprintf(stderr, "Extension bg: copy image to memory buffer.\n");
  /* We copy the image to some correct size buffer. */
  bgImageW = width;
  bgImageH = height;
  n = (alpha)?4:3;
  bgImage = g_malloc(sizeof(guchar) * bgImageW * bgImageH * n);
  memcpy(bgImage, imageData, sizeof(guchar) * bgImageW * bgImageH * n);
  bgImageAlpha = alpha;
  if (title)
    bgImageTitle = g_strdup_printf(_("Background: %s"), title);
  bgImageFit = fit;

  createBgImage();
}


/****************/
/* Private part */
/****************/

static void onDataReadySignal(GObject *visu _U_, VisuData *dataObj,
			      gpointer data _U_)
{
  if (!dataObj)
    return;

  refreshBgRGB();
  visu_glExt_fog_create(dataObj);
}
static void onDataNewSignal(GObject *visu _U_, VisuData *dataObj,
			    gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLNearFar",
		   G_CALLBACK(onFogParameterChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLWidthHeight",
		   G_CALLBACK(onBgImageRescale), (gpointer)0);
}
static void onFogParameterChanged(VisuData *dataObj, VisuOpenGLView *view _U_,
				  gpointer data _U_)
{
  DBG_fprintf(stderr, "Extension Fog: caught the 'OpenGLNearFar' signal, rebuilding"
	      " fog.\n");
  visu_glExt_fog_create(dataObj);
}
static void rebuildFogOnResources(GObject *obj _U_, VisuData *dataObj _U_,
				  gpointer data _U_)
{
  DBG_fprintf(stderr, "Extension Fog: caught the 'resourcesLoaded' signal, rebuilding"
	      " fog.\n");
}
static void onBgImageRescale(VisuData *dataObj _U_, VisuOpenGLView *view _U_,
			     gpointer data _U_)
{
  DBG_fprintf(stderr, "Extension Bg: caught the 'OpenGLWidthHeight' signal.\n");

  if (bgImage)
    createBgImage();
}

void visu_glExt_fog_create(VisuData *data)
{
  float start, stop;
  VisuOpenGLView *view;

  g_return_if_fail(data);

  fogHasBeenBuilt = TRUE;

  view = visu_data_getOpenGLView(data);
  start = (float)(view->window->near +
		  (view->window->far - view->window->near) * fog_start);
  stop = (float)(view->window->near +
		 (view->window->far - view->window->near) * fog_end);

/*   start = visuBox->extens * visuCamera->d_red * (1. - fog_start); */
/*   stop = visuBox->extens * visuCamera->d_red * (1. + fog_end); */
/*   start = visuBox->extens * visuCamera->d_red * (1. - 1 / 1.1); */
/*   stop = visuBox->extens * visuCamera->d_red * (1. + 1 / 1.1); */
/*   fprintf(stderr, "----------> %f %f %f %f\n", (float)(view->window->near + */
/* 						       (view->window->far - view->window->near) * fog_start), (float)(view->window->near + */
/* 														  (view->window->far - view->window->near) * fog_end), start, stop); */
  glFogf(GL_FOG_START, start);
  glFogf(GL_FOG_END, stop);
}

void visu_glExt_fog_create_color()
{
  if (fogColorSpecific)
    glFogfv(GL_FOG_COLOR, fogRGB);
  else
    glFogfv(GL_FOG_COLOR, bgRGB);
}
static void refreshBgRGB()
{
  DBG_fprintf(stderr, "Extension bg: set background color.\n");
  glClearColor(bgRGB[0], bgRGB[1], bgRGB[2], bgRGB[3]);
}
static void createBgImage()
{
  int viewport[4];
  float x, y;
  float zoom;

  DBG_fprintf(stderr, "Extension bg: set background image.\n");

  g_return_if_fail(bgImage);

  openGLText_initFontList();
  
  glGetIntegerv(GL_VIEWPORT, viewport);
  if (bgImageFit)
    {
      x = (float)viewport[2] / (float)bgImageW;
      y = (float)viewport[3] / (float)bgImageH;
    }
  else
    {
      x = y = 1.f;
    }
  zoom = MIN(x, y);
  x = ((float)viewport[2] - zoom * (float)bgImageW) / 2.f;
  y = ((float)viewport[3] - zoom * (float)bgImageH) / 2.f;

  glNewList(extensionFog->objectListId + 2, GL_COMPILE);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0., (float)viewport[2], 0., (float)viewport[3]);   
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glDepthMask(0);
  glRasterPos2i(x, viewport[3] - y);
  glPixelZoom(zoom, -zoom);
  if (bgImageAlpha)
    glDrawPixels(bgImageW, bgImageH, GL_RGBA, GL_UNSIGNED_BYTE, bgImage);
  else
    glDrawPixels(bgImageW, bgImageH, GL_RGB, GL_UNSIGNED_BYTE, bgImage);
  glPixelZoom(1., 1.);

  if (bgImageTitle)
    {
      glDisable(GL_LIGHTING);
      glColor4f(1.f - bgRGB[0], 1.f - bgRGB[1], 1.f - bgRGB[2], 1.f);
      glRasterPos2f(5.f, 5.f);
      openGLText_drawChars(bgImageTitle, TEXT_NORMAL); 
    }
  glDepthMask(1);

  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  glEndList();

  glNewList(extensionFog->objectListId, GL_COMPILE);
  glCallList(extensionFog->objectListId + 1);
  glCallList(extensionFog->objectListId + 2);
  glEndList();

}
static void createBgChess()
{
  int viewport[4];

  DBG_fprintf(stderr, "Extension bg: set background chess board.\n");

  glGetIntegerv(GL_VIEWPORT, viewport);

  glNewList(extensionFog->objectListId + 1, GL_COMPILE);
  glDisable(GL_CULL_FACE);
  glDisable(GL_LIGHTING);

  glEnable(GL_TEXTURE_2D);
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
  glBindTexture(GL_TEXTURE_2D, texName);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0., (float)viewport[2], 0., (float)viewport[3]);   
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glDepthMask(0);
  glBegin(GL_QUADS);
  glTexCoord2f(0.0, 0.0);
  glVertex3f(0.0, 0.0, 0.0);
  glTexCoord2f(0.0, (float)viewport[3] / 32.f);
  glVertex3f(0.0, (float)viewport[3], 0.0);
  glTexCoord2f((float)viewport[2] / 32.f, (float)viewport[3] / 32.f);
  glVertex3f((float)viewport[2], (float)viewport[3], 0.0);
  glTexCoord2f((float)viewport[2] / 32.f, 0.0);
  glVertex3f((float)viewport[2], 0.0, 0.0);
  glEnd();
  glDepthMask(1);

  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  glDisable(GL_TEXTURE_2D);
  glEndList();

  glNewList(extensionFog->objectListId, GL_COMPILE);
  glCallList(extensionFog->objectListId + 1);
  glCallList(extensionFog->objectListId + 2);
  glEndList();
}

static void rebuildOpenGLValues(VisuData *dataObj)
{
  DBG_fprintf(stderr, "Fog & Bg: rebuild extension.\n");
  if (fogIsOn)
    {
      glEnable(GL_FOG);
      glFogi(GL_FOG_MODE, GL_LINEAR); 
      visu_glExt_fog_create_color();
      visu_glExt_fog_create(dataObj);
    }
  refreshBgRGB();
  if (bgImage)
    createBgImage();
}

/* This is a boolean to control is the fog is render or not. */
static gboolean readFogIsOn(gchar **lines, int nbLines,
			    int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &val, 1, error))
    return FALSE;
  visu_glExt_fog_setOn(val);

  return TRUE;
}
/* This is a boolean to control the color used by the fog : a
   specific one or the background color. */
static gboolean readFogSpecific(gchar **lines, int nbLines,
				int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &val, 1, error))
    return FALSE;
  visu_glExt_fog_setUseSpecificColor(val);

  return TRUE;
}
/* A resource to control the color used for the background. */
static gboolean readFogColor(gchar **lines, int nbLines,
			     int position, VisuData *dataObj _U_, GError **error)
{
  float val[4];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, val, 4, error))
    {
      if (*error)
	g_error_free(*error);
      *error = (GError*)0;
      /* Read old 3 values. */
      if (!tool_configFile_readFloat(lines[0], position, val, 3, error))
	return FALSE;
      val[3] = 0.f;
    }
  visu_glExt_fog_setValues(val, TOOL_COLOR_MASK_RGBA);
  return TRUE;
}
/* A resource to control the color used for the background. */
static gboolean readFogStartEnd(gchar **lines, int nbLines,
				int position, VisuData *dataObj _U_, GError **error)
{
  float val[2];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, val, 2, error))
    return FALSE;
  visu_glExt_fog_setStartEndValues(val, VISU_GLEXT_FOG_MASK_START | VISU_GLEXT_FOG_MASK_END);
  return TRUE;
}
/* A resource to control the color used for the background. */
static gboolean readBgColor(gchar **lines, int nbLines,
			    int position, VisuData *dataObj _U_, GError **error)
{
  float val[4];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, val, 4, error))
    {
      if (*error)
	g_error_free(*error);
      *error = (GError*)0;
      /* Read old 3 values. */
      if (!tool_configFile_readFloat(lines[0], position, val, 3, error))
	return FALSE;
      val[3] = 1.f;
    }
  visu_glExt_bg_setValues(val, TOOL_COLOR_MASK_RGBA);
  return TRUE;
}

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static void exportResourcesFog(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_FOG_USED);
  g_string_append_printf(data, "%s:\n    %d\n", FLAG_RESOURCE_FOG_USED,
			 fogIsOn);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_FOG_SPECIFIC);
  g_string_append_printf(data, "%s:\n    %d\n", FLAG_RESOURCE_FOG_SPECIFIC,
			 fogColorSpecific);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_FOG_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_FOG_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f %4.3f\n",
			 fogRGB[0], fogRGB[1], fogRGB[2], fogRGB[3]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_FOG_STARTEND);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_FOG_STARTEND);
  g_string_append_printf(data, "    %4.3f %4.3f\n",
			 fog_start, fog_end);
  g_string_append_printf(data, "\n");
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BG_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BG_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f %4.3f\n",
			 bgRGB[0], bgRGB[1], bgRGB[2], bgRGB[3]);
  g_string_append_printf(data, "\n");
}
