/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dumpToSVG.h"

#ifdef HAVE_CAIRO

#include <math.h>
#include <string.h>
#include <glib.h>
#include <cairo.h>
#include <GL/gl.h>

#include <visu_tools.h>
#include <visu_object.h>
#include <visu_rendering.h>
#include <visu_pairs.h>
#include <opengl.h>
#include <openGLFunctions/light.h>
#include <openGLFunctions/view.h>
#include <extensions/box.h>
#include <extensions/fogAndBGColor.h>
#include <extensions/axes.h>
#include <extensions/legend.h>

#if CAIRO_VERSION_MINOR > 1
#include <cairo-svg.h>
#include <cairo-pdf.h>

#define DEBUG_USE_FOG 0

/**
 * SECTION:dumpToSVG
 * @short_description: add an export capability into SVG files.
 * @include: extensions/box.h, extensions/axes.h, visu_pairs.h and visu_data.h
 *
 * This provides a write routine to export V_Sim views into SVG
 * files. Currently, this is an experimental feature. Not all V_Sim
 * elements are rendered, only the nodes, the box, the pairs and the
 * axes. All the characteristics are not used (no line stipple for
 * instance). In spin mode, nodes are only atomic.
 *
 * Since: 3.4
 */

#define FONT_SIZE 16.
#define FONT_SMALL 12.

#define NVERT 2
#define NVALS 8
#define NPASS 1
#define NBUFF (NVERT * NVALS + NPASS * 2)

#define PAIRS_NVERT 2
#define PAIRS_NVALS 4
#define PAIRS_XBUFF (PAIRS_NVERT * PAIRS_NVALS)
#define PAIRS_NPASS 6
#define PAIRS_NBUFF (PAIRS_XBUFF + PAIRS_NPASS * 2)
#define PAIRS_NCMPT (4 + 1 + PAIRS_NPASS)

struct _pairs
{
  VisuPairData *data;
  VisuNode *node1, *node2;
  VisuElement *ele1, *ele2;
  float alpha;
};

static gboolean writeViewInSvgFormat(ToolFileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData, GError **error,
				     ToolVoidDataFunc functionWait, gpointer data);
static gboolean writeViewInPdfFormat(ToolFileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData, GError **error,
				     ToolVoidDataFunc functionWait, gpointer data);

static void svgGet_fogRGBA(float rgba[4]);
static cairo_pattern_t** svgSetup_patterns(VisuData *dataObj, gboolean flat);
static GList* svgSetup_pairs(VisuData *dataObj);
static void svgSetup_scale(cairo_t *cr, VisuData *dataObj);
static GLfloat* svgCompute_pairs(VisuData *dataObj, int *nPairs_out);
static GLfloat* svgCompute_coordinates(VisuData *dataObj, int *nNodes_out);
static GLfloat* svgCompute_box(int *nBox_out);
static void svgDraw_line(cairo_t *cr, float x0, float y0, float x1, float y1,
			 float z0, float z1, float *rgb, float *fog);
static void svgDraw_boxBack(cairo_t *cr, GLfloat *box, int nValuesBox, GLfloat val);
static void svgDraw_boxFront(cairo_t *cr, GLfloat *box, int nValuesBox, GLfloat val);
static void svgDraw_nodesAndPairs(cairo_t *cr, GLfloat *coordinates, int nNodes,
				  GLfloat *pairs, int nPairs, cairo_pattern_t **pat,
				  gboolean flat);
static void svgDraw_legend(cairo_t *cr, VisuData *dataObj, cairo_pattern_t **pat);
static void svgDraw_axes(cairo_t *cr);
static void svgDraw_pairs(cairo_t *cr, GLfloat *pairs, int nPairs,
			  int *iPairs, GLfloat val);

VisuDump* dumpToCairoSVG_init()
{
  VisuDump *svg;
  char *typeSVG[] = {"*.svg", (char*)0};
#define descrSVG _("Scalar Vector Graphic (SVG) file")
  ToolFileFormat* fmt;

  svg = g_malloc(sizeof(VisuDump));
  fmt = tool_file_format_new(descrSVG, typeSVG);
  if (!fmt)
    {
      g_error("Can't initialize the SVG dump module, aborting.\n");
    }

  svg->bitmap     = FALSE;
  svg->hasAlpha   = FALSE;
  svg->glRequired = TRUE;
  svg->fileType   = fmt;
  svg->writeFunc  = writeViewInSvgFormat;

  tool_file_format_addPropertyBoolean(fmt, "use_flat_rendering",
                                      _("Use flat colours for scheme rendering"),
                                      FALSE);
  
  return svg;
}
VisuDump* dumpToCairoPDF_init()
{
  VisuDump *pdf;
  char *typePDF[] = {"*.pdf", (char*)0};
#define descrPDF _("Portable Document Format (PDF) file")
  ToolFileFormat* fmt;

  pdf = g_malloc(sizeof(VisuDump));
  fmt = tool_file_format_new(descrPDF, typePDF);
  if (!fmt)
    {
      g_error("Can't initialize the PDF dump module, aborting.\n");
    }

  pdf->bitmap     = FALSE;
  pdf->hasAlpha   = FALSE;
  pdf->glRequired = TRUE;
  pdf->fileType   = fmt;
  pdf->writeFunc  = writeViewInPdfFormat;

  tool_file_format_addPropertyBoolean(fmt, "use_flat_rendering",
                                      _("Use flat colours for scheme rendering"),
                                      FALSE);
  
  return pdf;
}

static void sort_by_z(float *coordinates, float *buffer,
		      int n, int z, int begin, int end)
{
  int i;
  int middle;

  if( begin >= end ) return;
  
  memcpy(buffer, coordinates + begin * n, sizeof(float) * n);
  memcpy(coordinates + begin * n, coordinates + (end + begin) / 2 * n, sizeof(float) * n);
  memcpy(coordinates + (end + begin) / 2 * n, buffer, sizeof(float) * n);

  middle = begin;
  for(i = begin + 1; i <= end; i++)
    {
      if ( coordinates[i * n + z] > coordinates[begin * n + z] )
	{
	  middle += 1;
	  memcpy(buffer, coordinates + i * n, sizeof(float) * n);
	  memcpy(coordinates + i * n, coordinates + middle * n, sizeof(float) * n);
	  memcpy(coordinates + middle * n, buffer, sizeof(float) * n);
	}
    }
  memcpy(buffer, coordinates + begin * n, sizeof(float) * n);
  memcpy(coordinates + begin * n, coordinates + middle * n, sizeof(float) * n);
  memcpy(coordinates + middle * n, buffer, sizeof(float) * n);
  sort_by_z(coordinates, buffer, n, z, begin, middle - 1);
  sort_by_z(coordinates, buffer, n, z, middle + 1, end);
}

gboolean writeDataToCairoSurface(cairo_surface_t *cairo_surf, ToolFileFormat *format,
				 VisuData *dataObj, GError **error,
				 ToolVoidDataFunc functionWait _U_,
				 gpointer user_data _U_)
{
  cairo_t *cr;
  cairo_status_t status;
  cairo_pattern_t **pat;
  guint i;
  int nNodes, nPairs;
  float rgbaBg[4], centre[3];
  GLfloat *coordinates, *box, *pairs;
  int nValuesBox;
  ToolOption *prop;
  gboolean flat;

  DBG_fprintf(stderr, "Dump SVG: begin OpenGL buffer writing.\n");

  /* We get the properties related to SVG output. */
  prop = tool_file_format_getPropertyByName(format, "use_flat_rendering");
  flat = g_value_get_boolean(tool_option_getValue(prop));
  /* We setup the cairo output. */
  cr = cairo_create(cairo_surf);
  status = cairo_status(cr);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   "%s", cairo_status_to_string(status));
      cairo_destroy(cr);
      return FALSE;
    }

  cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);
  svgSetup_scale(cr, dataObj);
  pat = svgSetup_patterns(dataObj, flat);

  glPushMatrix();
  visu_data_getBoxCentre(dataObj, centre);
  glTranslated(-centre[0], -centre[1], -centre[2]);

  /* We calculate the node positions. */
  coordinates = svgCompute_coordinates(dataObj, &nNodes);

  /* Calculate the pair positions. */
  nPairs = -1;
  pairs = (GLfloat*)0;
  if (visu_pair_getStatus())
    pairs = svgCompute_pairs(dataObj, &nPairs);

  /* We draw the background colour. */
  visu_glExt_bg_getValues(rgbaBg);
  cairo_set_source_rgba(cr, rgbaBg[0], rgbaBg[1], rgbaBg[2], rgbaBg[3]);
  cairo_paint(cr);

  /* Draw the back box. */
  nValuesBox = -1;
  box = (GLfloat*)0;
  if (visu_glExt_box_getOn())
    {
      box = svgCompute_box(&nValuesBox);
      svgDraw_boxBack(cr, box, nValuesBox, coordinates[nNodes - NBUFF + 3]);
    }

  DBG_fprintf(stderr, "Dump SVG: begin main SVG exportation.\n");

  /* We draw nodes and pairs. */
  svgDraw_nodesAndPairs(cr, coordinates, nNodes, pairs, nPairs,
			(!visu_data_hasUserColorFunc(dataObj))?
			pat:(cairo_pattern_t**)0, flat);
  if (pairs)
    g_free(pairs);

  /* Draw the front box. */
  if (visu_glExt_box_getOn())
    svgDraw_boxFront(cr, box, nValuesBox, coordinates[nNodes - NBUFF + 3]);
  if (box)
    g_free(box);

  g_free(coordinates);

  /* We draw the axes. */
  if (visu_glExt_axes_getOn())
    svgDraw_axes(cr);

  glPopMatrix();

  /* We draw a legend, if required. */
  if (visu_glExt_legend_getOn())
    svgDraw_legend(cr, dataObj, pat);

  for (i = 0; i < dataObj->ntype; i++)
    cairo_pattern_destroy(pat[i]);
  g_free(pat);

  cairo_show_page(cr);
  cairo_destroy(cr);

  return TRUE;
}

static gboolean writeViewInSvgFormat(ToolFileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData _U_, GError **error,
				     ToolVoidDataFunc functionWait, gpointer data)
{
  cairo_surface_t *svg_surface;
  cairo_status_t status;
  guint old_width, old_height;
  VisuOpenGLView *view;

  g_return_val_if_fail(error && !*error, FALSE);

  DBG_fprintf(stderr, "Dump Cairo: begin export to SVG.\n");

  svg_surface = cairo_svg_surface_create(filename, (double)width, (double)height);
  status = cairo_surface_status(svg_surface);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   "%s", cairo_status_to_string(status));
      cairo_surface_destroy(svg_surface);
      return FALSE;
    }

  /* Change the view port to match width and height. */
  view = visu_data_getOpenGLView(dataObj);
  old_width = view->window->width;
  old_height = view->window->height;
  visu_data_setSizeOfView(dataObj, width, height);

  writeDataToCairoSurface(svg_surface, format, dataObj, error, functionWait, data);
  cairo_surface_destroy(svg_surface);

  /* Set back the viewport. */
  visu_data_setSizeOfView(dataObj, old_width, old_height);

  return TRUE;
}

static gboolean writeViewInPdfFormat(ToolFileFormat *format, const char* filename,
				     int width, int height, VisuData *dataObj,
				     guchar* imageData _U_, GError **error,
				     ToolVoidDataFunc functionWait, gpointer data)
{
  cairo_surface_t *pdf_surface;
  cairo_status_t status;
  guint old_width, old_height;
  VisuOpenGLView *view;

  g_return_val_if_fail(error && !*error, FALSE);

  DBG_fprintf(stderr, "Dump Cairo: begin export to PDF.\n");

  pdf_surface = cairo_pdf_surface_create(filename, (double)width, (double)height);
  status = cairo_surface_status(pdf_surface);
  if (status != CAIRO_STATUS_SUCCESS)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   "%s", cairo_status_to_string(status));
      cairo_surface_destroy(pdf_surface);
      return FALSE;
    }

  /* Change the view port to match width and height. */
  view = visu_data_getOpenGLView(dataObj);
  old_width = view->window->width;
  old_height = view->window->height;
  visu_data_setSizeOfView(dataObj, width, height);

  writeDataToCairoSurface(pdf_surface, format, dataObj, error, functionWait, data);
  cairo_surface_destroy(pdf_surface);

  /* Set back the viewport. */
  visu_data_setSizeOfView(dataObj, old_width, old_height);

  return TRUE;
}

static void svgDraw_legend(cairo_t *cr, VisuData *dataObj, cairo_pattern_t **pat)
{
  cairo_matrix_t scaleFont = {FONT_SIZE, 0., 0., -FONT_SIZE, 0., 0.};
  GLfloat *legend;
  GLint nValuesLegend;
  float lgWidth, lgHeight, max;
  int i, nEle;
  VisuNodeArray *array;
  VisuElement *ele;
  gchar *lbl;
  float radius;
  VisuRendering *method;

  method = visu_object_getRendering(VISU_INSTANCE);
  legend = g_malloc(sizeof(GLfloat) * 70000);
  glFeedbackBuffer(70000, GL_3D, legend);
  glRenderMode(GL_FEEDBACK);
  visuExtensions_callList(VISU_GLEXT_LEGEND_ID, TRUE);
  nValuesLegend = glRenderMode(GL_RENDER);

  cairo_select_font_face(cr, "Serif", CAIRO_FONT_SLANT_NORMAL,
			 CAIRO_FONT_WEIGHT_NORMAL);
  cairo_set_font_size(cr, 12.0);
  /* The two first polygons are the white frame. */
  if (nValuesLegend >= 22)
    {
      cairo_set_source_rgba(cr, 1.f, 1.f, 1.f, 0.4f);
      lgWidth  = legend[5] - legend[2];
      lgHeight = legend[9] - legend[3];
      cairo_rectangle(cr, legend[2], legend[3], lgWidth, lgHeight);
      cairo_fill(cr);
      /* Run to the names and the elements... */
      nEle = 0;
      array = visu_data_getNodeArray(dataObj);
      max = visu_data_getAllElementExtens(dataObj);
      for (i = 0; i < nValuesLegend; i++)
	if (legend[i] == GL_BITMAP_TOKEN)
	  {
	    ele = dataObj->fromIntToVisuElement[nEle];

	    /* Write the name. */
	    cairo_move_to(cr, legend[i + 1], legend[i + 2]);
	    lbl = g_strdup_printf("%s (%d)", ele->name,
				  array->numberOfStoredNodes[nEle]);
	    cairo_set_source_rgb(cr, 0.f, 0.f, 0.f);
	    cairo_set_font_matrix(cr, &scaleFont);
	    cairo_show_text(cr, lbl);
	    g_free(lbl);

	    /* Draw the element. */
	    radius = 0.4f * lgHeight *
	      visu_rendering_getSizeOfElement(method, ele) / max;
	    cairo_new_path(cr);
	    cairo_save(cr);
	    cairo_translate(cr, legend[i + 1] - 0.6f * lgHeight,
			    legend[3] + 0.5f * lgHeight);
	    cairo_arc(cr, 0.f, 0.f, radius, 0., 2 * G_PI);
	    cairo_save(cr);
	    cairo_scale(cr, radius, radius);
	    cairo_set_source(cr, pat[nEle]);
	    cairo_fill_preserve(cr);
	    cairo_restore(cr);
	    cairo_set_source_rgb(cr, 0.f, 0.f, 0.f);
	    cairo_stroke(cr);
	    cairo_restore(cr);

	    nEle += 1;
	    do
	      i += 4;
	    while (legend[i] == GL_BITMAP_TOKEN && i < nValuesLegend);
	  }
    }
  g_free(legend);
}

static void svgDraw_axes(cairo_t *cr)
{
  cairo_matrix_t scaleFont = {FONT_SIZE, 0., 0., -FONT_SIZE, 0., 0.};
  GLfloat *axes;
  GLint nValuesAxes;
  float *rgb;

  /* We create the feedback for the axes. */
  axes = g_malloc(sizeof(GLfloat) * 7000);
  glFeedbackBuffer(7000, GL_3D, axes);
  glRenderMode(GL_FEEDBACK);
  visuExtensions_callList(VISU_GLEXT_AXES_ID, TRUE);
  nValuesAxes = glRenderMode(GL_RENDER);
  DBG_fprintf(stderr, "Dump Cairo: found %d axes output.\n", nValuesAxes);

  cairo_set_line_width(cr, (double)visu_glExt_axes_getLineWidth());
  rgb = visu_glExt_axes_getRGBValues();
  cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
  cairo_select_font_face(cr, "Serif", CAIRO_FONT_SLANT_NORMAL,
			 CAIRO_FONT_WEIGHT_NORMAL);
  cairo_set_font_size(cr, 12.0);
  /* We draw the 3 lines of the axes. */
  if (nValuesAxes >= 21 && axes[0] == GL_LINE_RESET_TOKEN &&
      axes[7] == GL_LINE_RESET_TOKEN && axes[14] == GL_LINE_RESET_TOKEN)
    {
      DBG_fprintf(stderr, "Dump Cairo: output axes.\n");
      cairo_move_to(cr, axes[0 + 1], axes[0 + 2]);
      cairo_line_to(cr, axes[0 + 4], axes[0 + 5]);
      cairo_stroke(cr);
      cairo_move_to(cr, axes[7 + 1], axes[7 + 2]);
      cairo_line_to(cr, axes[7 + 4], axes[7 + 5]);
      cairo_stroke(cr);
      cairo_move_to(cr, axes[14 + 1], axes[14 + 2]);
      cairo_line_to(cr, axes[14 + 4], axes[14 + 5]);
      cairo_stroke(cr);
    }
  cairo_set_source_rgb(cr, 1.f - rgb[0], 1.f - rgb[1], 1.f - rgb[2]);
  cairo_set_font_matrix(cr, &scaleFont);
  if (nValuesAxes >= 33 && axes[21] == GL_BITMAP_TOKEN &&
      axes[25] == GL_BITMAP_TOKEN && axes[29] == GL_BITMAP_TOKEN)
    {
      DBG_fprintf(stderr, "Dump Cairo: output axes names.\n");
      cairo_move_to(cr, axes[21 + 1], axes[21 + 2]);
      cairo_show_text(cr, "x");
      cairo_move_to(cr, axes[25 + 1], axes[25 + 2]);
      cairo_show_text(cr, "y");
      cairo_move_to(cr, axes[29 + 1], axes[29 + 2]);
      cairo_show_text(cr, "z");
    }
  g_free(axes);
}

static GList* svgSetup_pairs(VisuData *dataObj)
{
  GList *pairsLst;
  VisuDataIter iter1, iter2;
  GList *tmpLst;
  struct _pairs *pairData;
  VisuPairData *data;
  float d2, d2min, d2max, d2min_buffered, d2max_buffered, l;
  float xyz1[3], xyz2[3], alpha;

  pairsLst = (GList*)0;

  visu_data_iterNew(dataObj, &iter1);
  visu_data_iterNew(dataObj, &iter2);
  for(visu_data_iterStart(dataObj, &iter1); iter1.element;
      visu_data_iterNextElement(dataObj, &iter1))
    {
      if (!visu_element_getRendered(iter1.element))
	continue;

      for(visu_data_iterStart(dataObj, &iter2);
	  iter2.element && iter2.iElement <= iter1.iElement ;
	  visu_data_iterNextElement(dataObj, &iter2))
	{
	  if (!visu_element_getRendered(iter2.element))
	    continue;
	  
	  for (tmpLst = visu_pair_getAllPairData(iter1.element, iter2.element);
	       tmpLst; tmpLst = g_list_next(tmpLst))
	    {
	      data = (VisuPairData*)tmpLst->data;
	      if (!data->drawn)
		continue;
	      d2min = data->minMax[VISU_PAIR_DISTANCE_MIN] * data->minMax[VISU_PAIR_DISTANCE_MIN];
	      d2max = data->minMax[VISU_PAIR_DISTANCE_MAX] * data->minMax[VISU_PAIR_DISTANCE_MAX];
	      if(d2min >= d2max || d2max <= 0.)
		continue;

	      l = data->minMax[VISU_PAIR_DISTANCE_MAX] - data->minMax[VISU_PAIR_DISTANCE_MIN];
	      d2min_buffered = (data->minMax[VISU_PAIR_DISTANCE_MIN] - 0.15 * l);
	      d2min_buffered *= d2min_buffered;
	      d2max_buffered = (data->minMax[VISU_PAIR_DISTANCE_MAX] + 0.15 * l);
	      d2max_buffered *= d2max_buffered;

	      for(visu_data_iterRestartNode(dataObj, &iter1); iter1.node;
		  visu_data_iterNextNode(dataObj, &iter1))
		{
		  if (!iter1.node->rendered)
		    continue;

		  for(visu_data_iterRestartNode(dataObj, &iter2); iter2.node;
		      visu_data_iterNextNode(dataObj, &iter2))
		    {
		      if (!iter2.node->rendered)
			continue;
		      /* Don't draw the inter element pairs two times. */
		      if (iter1.element == iter2.element &&
			  iter2.node >= iter1.node)
			break;

		      visu_data_getNodePosition(dataObj, iter1.node, xyz1);
		      visu_data_getNodePosition(dataObj, iter2.node, xyz2);
		      d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
			(xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
			(xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
		      if(d2 <= 0. || d2 < d2min_buffered || d2 > d2max_buffered)
			continue;

		      if (d2 < d2min)
			alpha = (d2 - d2min_buffered) /
			  (d2min - d2min_buffered);
		      else if (d2 > d2max)
			alpha = (d2max_buffered - d2) /
			  (d2max_buffered - d2max);
		      else
			alpha = 1.f;

#if GLIB_MINOR_VERSION > 9
		      pairData = g_slice_alloc(sizeof(struct _pairs));
#else
		      pairData = g_malloc(sizeof(struct _pairs));
#endif
		      pairData->data  = data;
		      pairData->node1 = iter1.node;
		      pairData->node2 = iter2.node;
		      pairData->ele1  = iter1.element;
		      pairData->ele2  = iter2.element;
		      pairData->alpha = alpha;
		      pairsLst = g_list_prepend(pairsLst, pairData);
		    }
		}
	    }
	}
    }
  return pairsLst;
}

static GLfloat* svgCompute_pairs(VisuData *dataObj, int *nPairs_out)
{
  int nPairs, i, *width;
  GLfloat *pairs;
  GLint nValues;
  GList *tmpLst;
  struct _pairs *pairData;
  float xyz1[3], xyz2[3], u[3], radius, norm;
  ToolColor *color;
  float tmpPairs[6];
  VisuRendering *method;
  GList *pairsLst;

  pairsLst = svgSetup_pairs(dataObj);
  *nPairs_out = nPairs = g_list_length(pairsLst);
  if (nPairs <= 0)
    return (GLfloat*)0;

  method = visu_object_getRendering(VISU_INSTANCE);

  DBG_fprintf(stderr, "Dump Cairo: found %d pairs to draw.\n", nPairs);
  pairs = g_malloc(sizeof(GLfloat) * nPairs * PAIRS_NBUFF);
  glFeedbackBuffer(nPairs * PAIRS_NBUFF, GL_3D, pairs);
  glRenderMode(GL_FEEDBACK);
  glPushMatrix();

  /* Render the list of pairs and free them each time. */
  for (tmpLst = pairsLst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      pairData = (struct _pairs*)tmpLst->data;

      visu_data_getNodePosition(dataObj, pairData->node1, xyz1);
      visu_data_getNodePosition(dataObj, pairData->node2, xyz2);
      u[0] = xyz2[0] - xyz1[0];
      u[1] = xyz2[1] - xyz1[1];
      u[2] = xyz2[2] - xyz1[2];
      norm = sqrt(u[0] * u[0] + u[1] * u[1] + u[2] * u[2]);
      u[0] /= norm;
      u[1] /= norm;
      u[2] /= norm;
      radius = visu_rendering_getSizeOfElement(method, pairData->ele1);
      xyz1[0] += radius * u[0];
      xyz1[1] += radius * u[1];
      xyz1[2] += radius * u[2];
      radius = visu_rendering_getSizeOfElement(method, pairData->ele2);
      xyz2[0] -= radius * u[0];
      xyz2[1] -= radius * u[1];
      xyz2[2] -= radius * u[2];
      glBegin(GL_POINTS);
      glVertex3fv(xyz1);
      glVertex3fv(xyz2);
      glEnd();
      /* We save colour channel as passthrough. */
      color = visu_pair_data_getColor(pairData->data);
      glPassThrough(color->rgba[0]);
      glPassThrough(color->rgba[1]);
      glPassThrough(color->rgba[2]);
      glPassThrough(pairData->alpha);
      /* We save the width and the printLength in a passthrough. */
      width = (int*)visu_pair_data_getProperty(pairData->data, "width");
      if (width)
	glPassThrough((float)*width);
      else
	glPassThrough((float)visu_glExt_box_getLineWidth());
      if (pairData->data->printLength)
	glPassThrough(norm);
      else
	glPassThrough(-1.f);
	  
      /* Free the data. */
#if GLIB_MINOR_VERSION > 9
      g_slice_free1(sizeof(struct _pairs), tmpLst->data);
#else
      g_free(tmpLst->data);
#endif
    }
  glPopMatrix();

  /* Free the list itself. */
  g_list_free(pairsLst);

  /* Analyse the OpenGL results. */
  nValues = glRenderMode(GL_RENDER);
  DBG_fprintf(stderr, " | OpenGL returns %d.\n", nValues);
  i = 0;
  nPairs = 0;
  while (i < nValues)
    {
	  
      if (pairs[i] == GL_POINT_TOKEN && 
	  pairs[i + PAIRS_NVALS] == GL_POINT_TOKEN &&
	  pairs[i + 2 * PAIRS_NVALS] == GL_PASS_THROUGH_TOKEN)
	{
	  /* Copy all these values into the beginning of the pairs
	     array. */
	  norm = (pairs[i + 3] + pairs[i + 7]) / 2.f;
	  pairs[nPairs * PAIRS_NCMPT +  0] = pairs[i + 1]; /* x1 */
	  pairs[nPairs * PAIRS_NCMPT +  1] = pairs[i + 2]; /* y1 */
	  pairs[nPairs * PAIRS_NCMPT +  2] = pairs[i + 5]; /* x2 */
	  pairs[nPairs * PAIRS_NCMPT +  3] = pairs[i + 6]; /* y2 */
	  pairs[nPairs * PAIRS_NCMPT +  4] = norm;         /* altitude */
	  pairs[nPairs * PAIRS_NCMPT +  5] = pairs[i + PAIRS_XBUFF +  7]; /* alpha */
	  pairs[nPairs * PAIRS_NCMPT +  6] = pairs[i + PAIRS_XBUFF +  1]; /* red   */
	  pairs[nPairs * PAIRS_NCMPT +  7] = pairs[i + PAIRS_XBUFF +  3]; /* green */
	  pairs[nPairs * PAIRS_NCMPT +  8] = pairs[i + PAIRS_XBUFF +  5]; /* blue  */
	  pairs[nPairs * PAIRS_NCMPT +  9] = pairs[i + PAIRS_XBUFF +  9]; /* width */
	  pairs[nPairs * PAIRS_NCMPT + 10] = pairs[i + PAIRS_XBUFF + 11]; /* prtLg */
	  i += PAIRS_NBUFF;
	  nPairs += 1;
	}
      else if (pairs[i] == GL_POINT_TOKEN &&
	       pairs[i + PAIRS_NVALS] == GL_PASS_THROUGH_TOKEN)
	{
	  DBG_fprintf(stderr, "| uncomplete pair for i=%d\n", i);
	  i += PAIRS_NVALS + 2 * PAIRS_NPASS;
	}
      else if (pairs[i] == GL_PASS_THROUGH_TOKEN)
	{
	  DBG_fprintf(stderr, "| no pair for i=%d\n", i);
	  i += 2 * PAIRS_NPASS;
	}
    }
  sort_by_z(pairs, tmpPairs, PAIRS_NCMPT, 4, 0, nPairs - 1);
  DBG_fprintf(stderr, " | will draw %d pairs.\n", nPairs);
  *nPairs_out = nPairs;

  return pairs;
}

static GLfloat* svgCompute_coordinates(VisuData *dataObj, int *nNodes_out)
{
  VisuDataIter iter;
  VisuRendering *method;
  GLfloat *coordinates;
  GLint nValues;
  float tmpFloat[NBUFF];
  float modelView[16];
  float xyz[3], radius, rgba[4];
  int i, nNodes;

  method = visu_object_getRendering(VISU_INSTANCE);
  visu_data_iterNew(dataObj, &iter);

  /* We create a feedback mode to get node coordinates. */
  coordinates = g_malloc(sizeof(GLfloat) * (iter.nAllStoredNodes * NBUFF));
  glFeedbackBuffer((iter.nAllStoredNodes * NBUFF), GL_3D_COLOR, coordinates);
  glRenderMode(GL_FEEDBACK);
  glGetFloatv(GL_MODELVIEW_MATRIX, modelView);

  /* First thing is to order nodes along z axes. */
  for (visu_data_iterStartVisible(dataObj, &iter); iter.node;
       visu_data_iterNextVisible(dataObj, &iter))
    {
      glBegin(GL_POINTS);
      visu_data_getNodePosition(dataObj, iter.node, xyz);
      if (visu_data_getUserColor(dataObj, iter.element, iter.node, rgba))
	openGLSet_color(iter.element->material, rgba);
      /* We compute the node position in the eyes coordinates. */
      glVertex3fv(xyz);
      /* We compute the node apparent radius using the real radius in
	 X direction in ModelView. */
      radius = visu_rendering_getSizeOfElement(method, iter.element);
      glVertex3f(xyz[0] + modelView[0] * radius,
		 xyz[1] + modelView[4] * radius,
		 xyz[2] + modelView[8] * radius);
      glEnd();

      /* We store the number of the VisuElement. */
      glPassThrough(iter.iElement);
    }

  /* We sort the coordinates along z. */
  nValues = glRenderMode(GL_RENDER);
  /* We compact coordinates to keep complete vertex list. */
  i = 0;
  nNodes = 0;
  while (i < nValues)
    {
      if (coordinates[i] == GL_POINT_TOKEN &&
	  coordinates[i + NVALS] == GL_POINT_TOKEN)
	{
	  /* 	  fprintf(stderr, "Found a complete node %d, at %gx%g.\n", */
	  /* 		  i, coordinates[i + 1], coordinates[i + 2]); */
	  /* 	  fprintf(stderr, " | move it from %d to %d.\n", i, nNodes); */
	  /* A complete set, copying it to nNodes location. */
	  if (nNodes != i)
	    memcpy(coordinates + nNodes, coordinates + i, sizeof(GLfloat) * NBUFF);
	  i += NBUFF;
	  nNodes += NBUFF;
	}
      else
	{
	  /* 	  fprintf(stderr, "Found a uncomplete node at %d.\n", i); */
	  /* Incomplete set, go on till the GL_PASS_THROUGH_TOKEN. */
	  while (coordinates[i] != GL_PASS_THROUGH_TOKEN)
	    i += 1;
	  /* Remove the GL_POINT_TOKEN. */
	  i += 2;
	  /* 	  fprintf(stderr, " | jump to %d.\n", i); */
	}
    }
  sort_by_z(coordinates, tmpFloat, NBUFF, 3, 0, nNodes / NBUFF - 1);

  *nNodes_out = nNodes;
  return coordinates;
}

static cairo_pattern_t* svgGet_pattern(gboolean flat, float rgba[4], float alpha)
{
  cairo_pattern_t *pat;
  float hsl[3], rgb[3], lum;
  if (flat)
    pat = cairo_pattern_create_rgba(rgba[0], rgba[1], rgba[2], rgba[3] * alpha);
  else
    {
      pat = cairo_pattern_create_radial(.4f, .4f, .1f, 0.f, 0.f, 1.f);
      /* We get the Element colour in HSL. */
      tool_color_convertRGBtoHSL(hsl, rgba);
      lum = hsl[2];
      hsl[2] = CLAMP(lum + 0.2f, 0.f, 1.f);
      tool_color_convertHSLtoRGB(rgb, hsl);
      cairo_pattern_add_color_stop_rgba(pat, 0, rgb[0], rgb[1], rgb[2], alpha);
      hsl[2] = CLAMP(lum + 0.05f, 0.f, 1.f);
      tool_color_convertHSLtoRGB(rgb, hsl);
      cairo_pattern_add_color_stop_rgba(pat, 0.3, rgb[0], rgb[1], rgb[2], alpha);
      hsl[2] = CLAMP(lum - 0.05f, 0.f, 1.f);
      tool_color_convertHSLtoRGB(rgb, hsl);
      cairo_pattern_add_color_stop_rgba(pat, 0.7, rgb[0], rgb[1], rgb[2], alpha);
      hsl[2] = CLAMP(lum - 0.2f, 0.f, 1.f);
      tool_color_convertHSLtoRGB(rgb, hsl);
      cairo_pattern_add_color_stop_rgba(pat, 1, rgb[0], rgb[1], rgb[2], alpha);
    }
  return pat;
}
static cairo_pattern_t** svgSetup_patterns(VisuData *dataObj, gboolean flat)
{
  cairo_pattern_t **pat;
  guint i;

  pat = g_malloc(sizeof(cairo_pattern_t*) * dataObj->ntype);
  for (i = 0; i < dataObj->ntype; i++)
    pat[i] = svgGet_pattern(flat, dataObj->fromIntToVisuElement[i]->rgb, 1.f);
  return pat;
}

static void svgSetup_scale(cairo_t *cr, VisuData *dataObj)
{
  VisuOpenGLView *view;
  int viewport[4];
  cairo_matrix_t scale = {1., 0., 0., -1., 0., 0.};

  view = visu_data_getOpenGLView(dataObj);
  glGetIntegerv(GL_VIEWPORT, viewport);
  scale.y0 = (double)viewport[3];
  cairo_set_matrix(cr, &scale);
}

static GLfloat* svgCompute_box(int *nBox_out)
{
  GLfloat *box;
  GLint nValuesBox;

  box = g_malloc(sizeof(GLfloat) * 7000);
  glFeedbackBuffer(7000, GL_3D, box);
  glRenderMode(GL_FEEDBACK);
  visuExtensions_callList(VISU_GLEXT_BOX_ID, FALSE);
  nValuesBox = glRenderMode(GL_RENDER);

  *nBox_out = (int)nValuesBox;
  return box;
}
static void svgDraw_line(cairo_t *cr, float x0, float y0, float x1, float y1,
			 float z0, float z1, float *rgb, float *fog)
{
  float d[3], alpha;
  cairo_pattern_t *pat;

  pat = (cairo_pattern_t*)0;
  if (visu_glExt_fog_getOn())
    {
      if (z0 != z1)
	{
	  pat = cairo_pattern_create_linear(x0, y0, x1, y1);
	  alpha = CLAMP(visu_glExt_fog_getStart() +
			(visu_glExt_fog_getEnd() - z0) /
			(visu_glExt_fog_getEnd() - visu_glExt_fog_getStart()), 0.f, 1.f);
	  d[0] = alpha * rgb[0] + (1.f - alpha) * fog[0];
	  d[1] = alpha * rgb[1] + (1.f - alpha) * fog[1];
	  d[2] = alpha * rgb[2] + (1.f - alpha) * fog[2];
	  cairo_pattern_add_color_stop_rgb(pat, 0, d[0], d[1], d[2]);
	  alpha = CLAMP(visu_glExt_fog_getStart() +
			(visu_glExt_fog_getEnd() - z1) /
			(visu_glExt_fog_getEnd() - visu_glExt_fog_getStart()), 0.f, 1.f);
	  d[0] = alpha * rgb[0] + (1.f - alpha) * fog[0];
	  d[1] = alpha * rgb[1] + (1.f - alpha) * fog[1];
	  d[2] = alpha * rgb[2] + (1.f - alpha) * fog[2];
	  cairo_pattern_add_color_stop_rgb(pat, 1, d[0], d[1], d[2]);
	  cairo_set_source(cr, pat);
	}
      else
	{
	  alpha = CLAMP(visu_glExt_fog_getStart() +
			(visu_glExt_fog_getEnd() - z1) /
			(visu_glExt_fog_getEnd() - visu_glExt_fog_getStart()), 0.f, 1.f);
	  d[0] = alpha * rgb[0] + (1.f - alpha) * fog[0];
	  d[1] = alpha * rgb[1] + (1.f - alpha) * fog[1];
	  d[2] = alpha * rgb[2] + (1.f - alpha) * fog[2];
	  cairo_set_source_rgb(cr, d[0], d[1], d[2]);
	}	
    }
  else
    cairo_set_source_rgb(cr, rgb[0], rgb[1], rgb[2]);
  cairo_move_to(cr, x0, y0);
  cairo_line_to(cr, x1, y1);
  cairo_stroke(cr);
  if (pat)
    cairo_pattern_destroy(pat);
}
static void svgDraw_boxBack(cairo_t *cr, GLfloat *box, int nValuesBox, GLfloat val)
{
  int i;
  float *rgb, fog[4];

  cairo_set_line_width(cr, (double)visu_glExt_box_getLineWidth());
  rgb = visu_glExt_box_getRGBValues();
  svgGet_fogRGBA(fog);
  /* We draw the lines that have a boundary hidden by elements. */
  for (i = 0; i < nValuesBox; i += 7)
    if (box[i + 3] >= val && box[i + 6] >= val)
      svgDraw_line(cr, box[i + 1], box[i + 2], box[i + 4], box[i + 5],
		   box[i + 3], box[i + 6], rgb, fog);
}
static void svgDraw_boxFront(cairo_t *cr, GLfloat *box, int nValuesBox, GLfloat val)
{
  int i;
  float *rgb, fog[4];

  cairo_set_line_width(cr, (double)visu_glExt_box_getLineWidth());
  rgb = visu_glExt_box_getRGBValues();
  svgGet_fogRGBA(fog);
  /* We draw the lines that have a boundary hidden by elements. */
  for (i = 0; i < nValuesBox; i += 7)
    if (box[i + 3] < val || box[i + 6] < val)
      svgDraw_line(cr, box[i + 1], box[i + 2], box[i + 4], box[i + 5],
		   box[i + 3], box[i + 6], rgb, fog);
}

static void svgGet_fogRGBA(float rgba[4])
{
  /* We get the fog color. */
  if (visu_glExt_fog_getOn())
    {
      if (visu_glExt_fog_getUseSpecificColor())
	visu_glExt_fog_getValues(rgba);
      else
	visu_glExt_bg_getValues(rgba);
    }
  else
    {
      rgba[0] = 0.;
      rgba[1] = 0.;
      rgba[2] = 0.;
      rgba[3] = 0.;
    }
}

static void svgDraw_nodesAndPairs(cairo_t *cr, GLfloat *coordinates, int nNodes,
				  GLfloat *pairs, int nPairs, cairo_pattern_t **pat,
				  gboolean flat)
{
  int iPairs, i;
  float alpha, radius, rgbaFog[4];
  cairo_matrix_t scaleFont = {FONT_SMALL, 0., 0., -FONT_SMALL, 0., 0.};
  cairo_pattern_t *pat_;

  svgGet_fogRGBA(rgbaFog);
  cairo_set_line_width(cr, 1.);
  cairo_set_font_matrix(cr, &scaleFont);
  iPairs = 0;
  alpha = 1.f;
  for (i = 0; i < nNodes; i+= NBUFF)
    {
      /* We draw the pairs in between. */
      svgDraw_pairs(cr, pairs, nPairs, &iPairs, coordinates[i + 3]);

      /* Compute the alpha for the fog. */
      alpha = (visu_glExt_fog_getOn())?CLAMP(visu_glExt_fog_getStart() +
				    (visu_glExt_fog_getEnd() - coordinates[i + 3]) /
				    (visu_glExt_fog_getEnd() - visu_glExt_fog_getStart()), 0.f, 1.f):1.f;
      
      radius = (coordinates[i + NVALS + 1] - coordinates[i + 1]) *
	(coordinates[i + NVALS + 1] - coordinates[i + 1]) +
	(coordinates[i + NVALS + 2] - coordinates[i + 2]) *
	(coordinates[i + NVALS + 2] - coordinates[i + 2]);
      radius = sqrt(radius);

/*       fprintf(stderr, "%gx%g %g %g %g\n", coordinates[i + 1], coordinates[i + 2], */
/*  	      radius, coordinates[i + 3], coordinates[i + NBUFF - 1]); */
      
      cairo_new_path(cr);
      cairo_save(cr);
      if (alpha > 0.f)
	{
	  cairo_translate(cr, coordinates[i + 1], coordinates[i + 2]);
	  cairo_arc(cr, 0.f, 0.f, radius, 0., 2 * G_PI);
	  cairo_save(cr);
	  cairo_scale(cr, radius, radius);
      
	  if (pat)
	    cairo_set_source(cr, pat[(int)coordinates[i + NVERT * NVALS + 1]]);
	  else
	    {
	      pat_ = svgGet_pattern(flat, coordinates + i + 4, 1.f);
	      cairo_set_source(cr, pat_);
	      cairo_pattern_destroy(pat_);
	    }
	  cairo_fill_preserve(cr);
	  if (alpha < 1.f)
	    {
	      cairo_set_source_rgba(cr, rgbaFog[0], rgbaFog[1], rgbaFog[2],
				    1.f - alpha);
	      cairo_fill_preserve(cr);
	    }
	  cairo_restore(cr);
	  cairo_set_source_rgb(cr, (1.f - alpha) * rgbaFog[0],
			       (1.f - alpha) * rgbaFog[1],
			       (1.f - alpha) * rgbaFog[2]);
	  cairo_stroke(cr);
	}

      cairo_restore(cr);
    }
  /* We draw the remaining pairs. */
  svgDraw_pairs(cr, pairs, nPairs, &iPairs, -1);
}
static void svgDraw_pairs(cairo_t *cr, GLfloat *pairs, int nPairs,
			  int *iPairs, GLfloat val)
{
  char distStr[8];
  float fog[4];

  svgGet_fogRGBA(fog);

  if (pairs && *iPairs < PAIRS_NCMPT * nPairs)
    {
      while (pairs[*iPairs + 4] > val && *iPairs < PAIRS_NCMPT * nPairs)
	{
	  DBG_fprintf(stderr, " | pair %d at (%f;%f) - (%f;%f) %g,%g\n",
		      *iPairs / PAIRS_NCMPT,
		      pairs[*iPairs + 0], pairs[*iPairs + 1],
		      pairs[*iPairs + 2], pairs[*iPairs + 3],
		      pairs[*iPairs + 4], pairs[*iPairs + 5]);
	  cairo_set_line_width(cr, pairs[*iPairs + 9]);
	  svgDraw_line(cr, pairs[*iPairs + 0], pairs[*iPairs + 1],
		       pairs[*iPairs + 2], pairs[*iPairs + 3],
		       pairs[*iPairs + 4], pairs[*iPairs + 4],
		       pairs + *iPairs + 6, fog);
	  if (pairs[*iPairs + 10] > 0)
	    {
	      cairo_move_to(cr, (pairs[*iPairs + 0] + pairs[*iPairs + 2]) * 0.5,
			    (pairs[*iPairs + 1] + pairs[*iPairs + 3]) * 0.5);
	      sprintf(distStr, "%7.3f", pairs[*iPairs + 10]);
	      cairo_show_text(cr, distStr);
	    }
	  *iPairs += PAIRS_NCMPT;
	}
      cairo_set_line_width(cr, 1.);
    }
}


#endif
#endif
