/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "visu_basic.h"
#include "visu_object.h"
#include "visu_rendering.h"
#include "visu_extension.h"
#include "visu_data.h"
#include "visu_elements.h"
#include "visu_commandLine.h"
#include "opengl.h"
#include "visu_dump.h"
#include "visu_pairs.h"
#include "visu_tools.h"
#include "visu_plugins.h"
#include "visu_configFile.h"
#include "visu_pickMesure.h"
#include "extraFunctions/dataFile.h"
#include "extraFunctions/plane.h"
#include "extraFunctions/pot2surf.h"
#include "extraFunctions/extraNode.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/text.h"
#include "OSOpenGL/visu_openGL.h"
#include "extensions/fogAndBGColor.h"
#include "extensions/scale.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/geometry.h"

#include "renderingMethods/renderingAtomic.h"
#include "renderingMethods/renderingSpin.h"

#include <stdio.h>
#include <locale.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */

/* For the GdkPixbuf, to be removed later. */
#include <gtk/gtk.h>

/**
 * SECTION:visu_basic
 * @short_description: Main functions of V_Sim (except graphical
 * ones).
 *
 * <para>There are here the main functions of V_Sim (except for
 * graphical methods) such as open file.</para>
 */

/* Valeurs par dfaut pour les chemins utiles dans V_Sim. */
#if SYSTEM_X11 == 1
#define V_SIM_DATA_DIR_DEFAULT    "/usr/local/share/v_sim"
#define V_SIM_LEGAL_DIR_DEFAULT   "/usr/local/share/v_sim"
#define V_SIM_PIXMAPS_DIR_DEFAULT "/usr/local/share/v_sim/pixmaps"
#define V_SIM_PLUGINS_DIR_DEFAULT "/usr/local/lib/v_sim/plug-ins"
#define V_SIM_LOCALE_DIR_DEFAULT  "/usr/local/share/locale"
#endif
#if SYSTEM_WIN32 == 1
#define V_SIM_DATA_DIR_DEFAULT    "C:\\PROGRAM FILES\\V_Sim\\share\\v_sim"
#define V_SIM_LEGAL_DIR_DEFAULT   "C:\\PROGRAM FILES\\V_Sim\\share\\doc\\v_sim"
#define V_SIM_PIXMAPS_DIR_DEFAULT "C:\\PROGRAM FILES\\V_Sim\\share\\v_sim\\pixmaps"
#define V_SIM_PLUGINS_DIR_DEFAULT "C:\\PROGRAM FILES\\V_Sim\\lib\\v_sim\\plug-ins"
#define V_SIM_LOCALE_DIR_DEFAULT  "C:\\PROGRAM FILES\\V_Sim\\share\\locale"
#endif
static gchar *v_sim_data_dir;
static gchar *v_sim_legal_dir;
static gchar *v_sim_pixmaps_dir;
static gchar *v_sim_local_conf_dir;
static gchar *v_sim_old_local_conf_dir;
static gchar *v_sim_plugins_dir;
static gchar *v_sim_locale_dir;
static VisuUnits preferedUnit;

/* Local methods. */
static gboolean dumpData(gpointer data);
static gboolean visuBasicDump(VisuData *data, DumpImage *dumpData,
			      int width, int height);
static void freeExtension(gpointer data);
static void onNewData(GObject* obj, VisuData *dataObj, gpointer data);
static void onNodesSet(VisuData *data, VisuNodeArray *nodes, gpointer user_data);

#define FLAG_PARAMETER_UNIT   "main_unit"
#define DESC_PARAMETER_UNIT   "Define the prefered unit to display files ; string"
static gboolean readUnit(gchar **lines, int nbLines, int position,
			 VisuData *dataObj, GError **error);
static void exportParameters(GString *data, VisuData *dataObj);

gboolean visuBasicLoad_dataFromFile(VisuData *data, FileFormat *format,
				    int nSet, GError **error)
{
  visuRenderingLoadFunc loadMethod;
  gboolean res;
  VisuUnits unit;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  loadMethod = visuRenderingClassGet_currentLoadFunc();
  if (!loadMethod)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			   _("No load method exists for the"
			     " specified rendering method.\n"));
      return FALSE;
    }

#if DEBUG == 1
  g_mem_profile();
#endif
  DBG_fprintf(stderr, "Visu Basic: load dataset %d.\n", nSet);
  res = loadMethod(data, format, nSet, error);
  DBG_fprintf(stderr, "Visu Basic: load OK from rendering method,"
	      " continue with basic parts.\n");
#if DEBUG == 1
  g_mem_profile();
#endif

  /* Set flags... */
  if (res)
    {
      visuDataSet_setId(data, nSet);
      visuDataSet_changeElementFlag(data, FALSE);
      unit = visuDataGet_unit(data);
      if (preferedUnit != unit_undefined && unit != unit_undefined &&
	  unit != preferedUnit)
	visuDataSet_unit(data, preferedUnit);
    }

  DBG_fprintf(stderr, "Visu Basic: emitting 'dataLoaded' signal for %p (%d).\n",
	      (gpointer)data, res);
  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATALOADED_SIGNAL],
		0, (res)?(gpointer)data:(gpointer)0, NULL);
  DBG_fprintf(stderr, "Visu Basic: emittion done.\n");

  return res;
}


void visuBasicInit()
{
  int res;
  VisuConfigFileEntry *entry;

  DBG_fprintf(stderr,"--- Initialising variables ---\n");
  /* We want to read . as floating point separator : in french it is , */
  setlocale(LC_NUMERIC, "C");

  /* Initialise le stockage des couleurs */
  toolColorInit_storage();

  /* Force the creation of the main object class. */
  g_type_class_ref(visuObject_get_type());
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onNewData), (gpointer)0);

  preferedUnit = unit_undefined;

  /* initialise la gestion des resources et des paramtres. */
  DBG_fprintf(stderr,"--- Initialising the VisuConfigFile part ---\n");
  res = visuConfigFile_init();
  if (!res)
    exit(1);

  /* Initialise the rendering methods. */
  DBG_fprintf(stderr,"--- Initialising the VisuRendering part ---\n");
  renderingAtomicInit();
  rspinInit();

  /* initialise les fonctions d'export. */
  DBG_fprintf(stderr,"--- Initialising dump modules ---\n");
  res = visuDumpInit();
  if (!res)
    exit(1);

  /* initialise OpenGL. */
  DBG_fprintf(stderr,"--- Initialising OpenGL ---\n");
  res = initOpengl();
  if (!res)
    exit(1);

  /* initialise les ajouts OpenGL. */
  DBG_fprintf(stderr,"--- Initialising OpenGL extensions ---\n");
  res = initOpenGLExtensions();
  if (!res)
    exit(1);

  /* initialise the VisuElement part. */
  DBG_fprintf(stderr,"--- Initialising the elements and data part of visu ---\n");
  res = initVisuElements();
  if (!res)
    exit(1);

  /* initialise la gestion des liaisons. */
  DBG_fprintf(stderr,"--- Initialising pairs modules ---\n");
  res = initPairsModule();
  if (!res)
    exit(1);

  /* initialise la gestion des sessions pick. */
  DBG_fprintf(stderr,"--- Initialising pick module ---\n");
  initPick_module();

  DBG_fprintf(stderr,"--- Initialising dataFile modules ---\n");
  res = initDataFileModule();
  if (!res)
    exit(1);

  /* initialise the scalar field handler. */
  DBG_fprintf(stderr,"--- Initialising scalar field module ---\n");
  scalarFieldInit();

  /* Charge toutes les extensions OpenGL. */
  DBG_fprintf(stderr,"--- Initialising external OpenGL Extensions ---\n");
  loadExtensions();

  /* Charge les plugins. */
  DBG_fprintf(stderr,"--- Initialising plugins ---\n");
  visuPluginsInit();

  /* Initialise extra functions. */
  DBG_fprintf(stderr,"--- Initialising extra functions ---\n");
  isosurfacesInit();
  matrixInit();
  geometryInit();

  /* Force the creation of the VisuData class. */
  g_type_class_ref(visu_data_get_type());
  /* Force the creation of the Scale class. */
  g_type_class_ref(scale_get_type());

  entry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
				  FLAG_PARAMETER_UNIT,
				  DESC_PARAMETER_UNIT,
				  1, readUnit);
  visuConfigFileSet_version(entry, 3.5f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);
}

GString* visuBasicParse_configFiles()
{
  GString *message;
  char* path;
  GError *error;
  gboolean res;
  char *resFile;

  message = (GString*)0;
  DBG_fprintf(stderr,"--- Initialising the parameters ---\n");
  /* Look for parameters file */
  path = visuConfigFileGet_validPath(VISU_CONFIGFILE_PARAMETER, R_OK, 0);
  if (path)
    {
      error = (GError*)0;
      res = visuConfigFileLoad(VISU_CONFIGFILE_PARAMETER, path, (VisuData*)0, &error);
      if (error)
	{
	  message = g_string_new("");
	  g_string_append_printf(message,
				 _("While parsing parameter file '%s':\n\n%s"),
				 path, error->message);
	  g_error_free(error);
	}
      g_free(path);
    }
  else
    message = g_string_new("Unable to find a valid parameters file."
			   " Starting with defaults.\n");

  DBG_fprintf(stderr,"--- Initialising resources ---\n");
  /* Look for resources file */
  resFile = commandLineGet_resourcesFile();
  if (!resFile)
    path = visuConfigFileGet_validPath(VISU_CONFIGFILE_RESOURCE, R_OK, 0);
  else
    path = g_strdup(resFile);
  if (path)
    {
      error = (GError*)0;
      res = visuConfigFileLoad(VISU_CONFIGFILE_RESOURCE, path, (VisuData*)0, &error);
      if (error)
	{
	  if (message)
	    g_string_append(message, "\n\n");
	  else
	    message = g_string_new("");
	  g_string_append_printf(message,
				 _("While parsing resource file '%s':\n\n%s"),
				 path, error->message);
	  g_error_free(error);
	}
      g_free(path);
    }

  return message;
}

VisuData* visuBasicSet_dataFromCommandLine()
{
  char* filename, *spin_filename;
  VisuData *newData;

  newData = (VisuData*)0;

  filename = commandLineGet_ArgFilename();
  spin_filename = commandLineGet_ArgSpinFileName();
  if (filename && !spin_filename)
    {
      newData = visuDataNew();
      if (!newData)
	return (VisuData*)0;

      visuRenderingClassSet_currentByName(VISU_RENDERING_ATOMIC);

      visuDataAdd_file(newData, filename, FILE_KIND_POSITION, (FileFormat*)0);
    }
  else if(filename && spin_filename)
    {
      newData = visuDataNew();
      if (!newData)
	return (VisuData*)0;

      visuRenderingClassSet_currentByName(VISU_RENDERING_SPIN);

      visuDataAdd_file(newData, filename, FILE_KIND_POSITION, (FileFormat*)0);
      visuDataAdd_file(newData, spin_filename, FILE_KIND_SPIN, (FileFormat*)0);
    }
  return newData;
}

static void onNewData(GObject* obj _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDefined",
		   G_CALLBACK(onNodesSet), (gpointer)0);
}
static void onNodesSet(VisuData *data, VisuNodeArray *nodes, gpointer user_data _U_)
{
  /* We add the label node property to each VisuData with new nodes. */
  if (nodes)
    extraNodeAdd_label(data);
}

struct _dump
{
  VisuData *data;
  int width, height;
  DumpImage *dumpData;

  GMainLoop *loop;
};
int visuBasicExport_main(void)
{
  GString *message;
  VisuData *newData;
  GError *error;
  gboolean res;
  struct _dump dt;

  dt.loop = g_main_loop_new(NULL, FALSE);
  g_type_init();

  visuBasicInit();

  /* We need to initialise the OpenGL surface as soon as possible to
     avoid crash in some OpenGL implementations when OpenGL routines
     may be called while parsing files. */
  commandLineGet_XWindowGeometry((int*)&(dt.width), (int*)&(dt.height));
  dt.dumpData = visuOpenGLNew_pixmapContext((guint)dt.width, (guint)dt.height);
  if (!dt.dumpData)
    {
      g_error("can't create off-screen rendering. Command line"
	      " exportation is not available.");
    }
  /* We set the glViewport of this new context. */
  glViewport(0, 0, dt.width, dt.height);
  /* We call the given draw method. */
  openGLInit_context();

  message = visuBasicParse_configFiles();
  if (message)
    {
      g_warning(message->str);
      g_string_free(message, TRUE);
    }
  newData = visuBasicSet_dataFromCommandLine();
  if (newData)
    {
      error = (GError*)0;
      res = visuBasicLoad_dataFromFile(newData, (FileFormat*)0, 0, &error);
      if (!res)
	{
	  /* We free the pixmap context. */
	  visuOpenGLFree_pixmapContext(dt.dumpData);
	  
	  g_object_unref(newData);

	  if (error)
	    {
	      g_error(error->message);
	      g_error_free(error);
	    }
	  else
	    {
	      g_error("No error message!");
	    }
	}

      DBG_fprintf(stderr, "Visu Basic: emitting the 'dataReadyForRendering' signal.\n");
      g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATAREADYFORRENDERING_SIGNAL],
		    0, (gpointer)newData, NULL);
      DBG_fprintf(stderr, "Visu Basic: emitting done.\n");

      DBG_fprintf(stderr, "Visu Basic: adding the load call back in the queue.\n");
      dt.data = newData;
      g_idle_add(dumpData, (gpointer)&dt);
    }
  else
    g_error(_("a file to render is mandatory with the '--export' option."));
  
  g_main_loop_run(dt.loop);
  return 0;
}

void visuBasicSet_extInfos(VisuData *dataObj, DrawMethod method,
			   int *nodes, DataNode *data)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(dataObj));
  g_return_if_fail((method == DRAW_METH_OTHER && IS_DATA_NODE_TYPE(data)) ||
		   method != DRAW_METH_OTHER);

  DBG_fprintf(stderr, "VisuBasic: set the extension infos with method %d.\n", method);
  switch (method)
    {
    case DRAW_METH_NONE:
      extInfosSet_used(dataObj, FALSE);
      break;
    case DRAW_METH_ID:
      extInfosSet_used(dataObj, TRUE);
      extInfosSet_number(dataObj, nodes);
      break;
    case DRAW_METH_TYPE:
      extInfosSet_used(dataObj, TRUE);
      extInfosSet_element(dataObj, nodes);
      break;
    default:
      extInfosSet_used(dataObj, TRUE);
      extInfosSet_data(dataObj, data, nodes);
    }
}

static int planesId, surfId, mapId;

static void setBgImage(const gchar *filename)
{
  GError *error;
  GdkPixbuf *pixbuf;
  gchar *title;
  gboolean fit;

  DBG_fprintf(stderr, "Visu Basic: set the background image to '%s'.\n",
	      filename);
  error = (GError*)0;
  pixbuf = gdk_pixbuf_new_from_file(filename, &error);
  if (!pixbuf)
    {
      g_warning(error->message);
      g_error_free(error);
      return;
    }

  fit = TRUE;
  title = g_path_get_basename(filename);
  if (!strcmp(title, "logo_grey.png"))
    {
      fit = FALSE;
      g_free(title);
      title = (gchar*)0;
    }
  bgSet_image(gdk_pixbuf_get_pixels(pixbuf),
	      gdk_pixbuf_get_width(pixbuf),
	      gdk_pixbuf_get_height(pixbuf),
	      gdk_pixbuf_get_has_alpha(pixbuf),
	      title, fit);
  g_object_unref(pixbuf);
  g_free(title);
}

static void rebuildPlanes(VisuData *dataObj)
{
  VisuOptionSet *set;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);

  DBG_fprintf(stderr, "Visu Basic: create OpenGL list for planes.\n");
  planesDraw_list(set->planesList, planesId);
  DBG_fprintf(stderr, " | OK\n");
}
static void rebuildSurf(VisuData *dataObj)
{
  SurfacesOrder *surfaceOrder;
  VisuOptionSet *set;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);
  /* Create the surface list. */
  surfaceOrder = isosurfacesOrder_new();
  isosurfacesDraw_surfaces(surfId + 1, set->surfsList, surfaceOrder);
  isosurfacesDuplicate(surfId, surfId + 1, dataObj, TRUE);
  isosurfacesOrder_free(surfaceOrder);
}
static float drawnMinMax[2];
static void rebuildMap(VisuData *dataObj)
{
  VisuOptionSet *set;
  int i;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);
  /* Create the map list. */
  glNewList(mapId, GL_COMPILE);
  for (i = 1; i <= set->mapPlaneId[0]; i++)
    scalarFieldDraw_map((ScalarField*)set->fieldsList->data,
			set->planesList[set->mapPlaneId[i]],
			set->shade, visuDataGet_openGLView(dataObj),
			(float)commandLineGet_mapPrecision(),
			drawnMinMax, set->logScale, set->nIsolines,
			set->isoLinesColor, (float*)0);
  glEndList();
}
static void rebuildLegend(VisuData *dataObj)
{
  double minMax[2];
  float fMinMax[2], smin;
  VisuOptionSet *set;

  set = g_object_get_data(G_OBJECT(dataObj), "optionSet");
  g_return_if_fail(set);
  /* Create the map list. */
  scalarFieldGet_minMax((ScalarField*)set->fieldsList->data, minMax);
  fMinMax[0] = minMax[0];
  fMinMax[1] = minMax[1];
  smin = scalarFieldGet_secondaryMin((ScalarField*)set->fieldsList->data);
  
  openGLText_rebuildFontList();

  glNewList(mapId + 1, GL_COMPILE);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_ALPHA_TEST);
  shadeDraw_legend(set->shade, .05, -1., fMinMax,
		   drawnMinMax, 2, set->logScale, &smin);
  glEndList();
}

void visuBasicCreate_extensions(VisuData *data, VisuOptionSet *set, gboolean rebuild)
{
  OpenGLExtension *plane, *surfExt, *map;

  g_object_set_data_full(G_OBJECT(data), "optionSet", set, freeExtension);

  /* We create the plane extension if necessary. */
  if (set->planesList)
    {
      planesId = openGLObjectList_new(1);
      plane = OpenGLExtension_new("Planes", _("Planes"), (char*)0,
				  planesId, rebuildPlanes);
      plane->used = 1;
      OpenGLExtensionSet_sensitiveToRenderingMode(plane, TRUE);
      OpenGLExtensionSet_priority(plane, OPENGL_EXTENSION_PRIORITY_LOW + 1);
      OpenGLExtensionRegister(plane);
      if (rebuild)
	rebuildPlanes(data);
    }

  /* We create the surface extension. */
  if (set->surfsList)
    {
      surfId = openGLObjectList_new(2);
      surfExt = OpenGLExtension_new("Isosurfaces", _("Isosurfaces"), (char*)0,
				    surfId, rebuildSurf);
      surfExt->used = 1;
      OpenGLExtensionSet_sensitiveToRenderingMode(surfExt, TRUE);
      OpenGLExtensionSet_saveOpenGLState(surfExt, TRUE);
      OpenGLExtensionSet_priority(surfExt, OPENGL_EXTENSION_PRIORITY_LOW + 2); 
      OpenGLExtensionRegister(surfExt);
      if (rebuild)
	rebuildSurf(data);
    }

  /* We create the map extension. */
  if (set->mapPlaneId)
    {
      mapId = openGLObjectList_new(2);
      map = OpenGLExtension_new("Map", _("Coloured map"), (char*)0,
				mapId, rebuildMap);
      map->used = 1;
      OpenGLExtensionRegister(map);
      if (rebuild)
	rebuildMap(data);

      map = OpenGLExtension_new("MapLegend", _("Map legend"), (char*)0,
				mapId + 1, rebuildLegend);
      OpenGLExtensionSet_priority(map, OPENGL_EXTENSION_PRIORITY_LAST);
      OpenGLExtensionSet_saveOpenGLState(map, TRUE);
      map->used = 1;
      OpenGLExtensionRegister(map);
      if (rebuild)
	rebuildLegend(data);
    }

  /* We set the background image. */
  if (set->bgImage)
    setBgImage(set->bgImage);
}

static gboolean dumpData(gpointer data)
{
  GMainContext *context;
  GError *error;
  VisuOptionSet *set;
  struct _dump *dt;

  dt = (struct _dump*)data;
  set = g_malloc(sizeof(VisuOptionSet));
  error = (GError*)0;
  if (!visuBasicApply_commandLine(dt->data, set, &error))
    {
      g_warning(error->message);
      g_error_free(error);
    }
  else
    {
      visuBasicCreate_extensions(dt->data, set, FALSE);

      context = g_main_loop_get_context(dt->loop);
      while(g_main_context_pending(context))
	g_main_context_iteration(context, FALSE);

      visuBasicDump(dt->data, dt->dumpData, dt->width, dt->height);

      g_object_unref(G_OBJECT(dt->data));
    }

  g_main_loop_quit(dt->loop);

  return FALSE;
}

gboolean visuBasicApply_commandLine(VisuData *visuData, VisuOptionSet *set,
				    GError **error)
{
  gchar *planesFile, *surfFile, *fieldFile, *valueFile;
  GString *message;
  int errors, i, j, presetShade, nb, surfNb;
  gboolean somethingIsLoaded, res, new;
  int *colUsed;
  float *translations, *extension;
  float *values;
  gchar **names;
  guint width, height;
  GList *tmplst, *list;
  OptionTable *table;
  int *nodes;
  DrawItem mode;
  guint info;
  DataNode *dt;
  DrawMethod method;
  float vertices[8][3];

  g_return_val_if_fail(visuData && set, FALSE);
  g_return_val_if_fail(error && !*error, FALSE);

  set->colorFile     = (gchar*)0;
  set->planesList    = (Plane**)0;
  set->surfsList     = (Surfaces**)0;
  set->shade         = (Shade*)0;
  set->mapPlaneId    = (gint*)0;
  set->logScale      = linear;
  set->nIsolines     = 0;
  set->isoLinesColor = (float*)0;
  set->fieldsList    = (GList*)0;
  set->bgImage       = (gchar*)0;

  surfNb = 0;

  commandLineGet_XWindowGeometry((int*)&width, (int*)&height);
  visuDataSet_sizeOfView(visuData, width, height);

  /* Post loading sequence, corresponding to all other arguments given to command line. */
  /* The value File, maybe used later. */
  valueFile = commandLineGet_valueFile();
  /* translate argument */
  translations = commandLineGet_translation();
  if (translations)
    {
      visuDataSet_XYZtranslation(visuData, translations);
      visuData_constrainedInTheBox(visuData);
    }
  /* expand argument */
  extension = commandLineGet_extension();
  if (extension)
    {
      if (!translations)
	visuData_constrainedInTheBox(visuData);
      visuData_replicate(visuData, extension, (gboolean*)0);
    }

  /* Post loading sequence, corresponding to all other arguments given to command line. */
  /* The shade option. */
  presetShade = commandLineGet_presetColor();
  list        = toolShadeBuild_presetList();
  if (presetShade >= 0)
    {
      if (presetShade >= (int)g_list_length(list))
	{
	  g_warning(_("unknown shade id (max is %d)."), g_list_length(list) - 1);
	  set->shade = (Shade*)g_list_nth_data(list, 0);
	}
      else
	set->shade = (Shade*)g_list_nth_data(list, presetShade);
      DBG_fprintf(stderr, "Visu Basic: load a shade from command line %p.\n",
		  (gpointer)set->shade);
    }
  else
    set->shade = (Shade*)g_list_nth_data(list, 0);
  /* colorize argument */
  set->colorFile = commandLineGet_colorizeFileName();
  if (set->colorFile)
    {
      DBG_fprintf(stderr, "Visu basic : loading a new data file '%s'.\n",
		  set->colorFile);
      message = g_string_new(_("Loading data file...\n"));
      somethingIsLoaded = dataFileSet_file(visuData, set->colorFile, &new,
					   message, &errors);

      /* Print errors if necessary. */
      if (errors)
	g_warning(message->str);
      g_string_free(message, TRUE);

      if (somethingIsLoaded)
	{
	  colUsed = commandLineGet_colorizeColUsed();
	  if (colUsed)
	    for (i = 0; i < 3; i++)
	      dataFileSet_colUsed(visuData, colUsed[i] - 1, i);
	  dataFileSet_shade(visuData, set->shade);
	  dataFileSet_used(visuData, TRUE);
	}
    }
  /* Planes argument. */
  planesFile = (valueFile)?valueFile:commandLineGet_planesFileName();
  if (planesFile)
    {
      somethingIsLoaded = planesParse_XMLFile(planesFile, &set->planesList, error);
      if (!somethingIsLoaded)
	return FALSE;
      else
	{
	  visuDataGet_boxVertices(visuData, vertices, TRUE);
	  for (i = 0; set->planesList[i]; i++)
	    planeSet_box(set->planesList[i], vertices);
	  /* Hide nodes if required. */
	  res = planeShowHide_all(visuData, set->planesList);
	  if (res)
	    visuDataEmit_nodeRenderedChange(visuData);
	}
    }
  /* Iso-surfaces arguments. */
  surfFile = commandLineGet_isoSurfacesFileName();
  if (surfFile)
    {
      surfNb = 1;
      set->surfsList = g_malloc(sizeof(Surfaces*) * (surfNb + 1));
      somethingIsLoaded = isosurfacesLoad_file(surfFile, set->surfsList, error);
      if (!somethingIsLoaded)
	{
	  surfNb = 0;
	  return FALSE;
	}
      else if (commandLineGet_fitToBox())
	isoSurfacesSet_fitToBox(visuData, set->surfsList[0]);
      isosurfacesSet_showAll(set->surfsList[0], TRUE);
      set->surfsList[surfNb] = (Surfaces*)0;
    }
  /* Scalar-field arguments. */
  fieldFile = commandLineGet_scalarFieldFileName();
  if (fieldFile)
    {
      /* Create an option table for possible spin or complex
	 modifiers. */
      table = commandLineGet_options();
      /* Call the load method. */
      somethingIsLoaded = scalarFieldLoad_fromFile(fieldFile, &set->fieldsList,
						   error, table);
      if (!somethingIsLoaded)
	{
	  surfNb = 0;
	  return FALSE;
	}
      else
	{
	  if (commandLineGet_fitToBox())
	    {
	      tmplst = set->fieldsList;
	      while(tmplst)
		{
		  scalarFieldSet_fitToBox(visuData, (ScalarField*)tmplst->data);
		  tmplst = g_list_next(tmplst);
		}
	    }

	  nb             = g_list_length(set->fieldsList);
	  set->surfsList = g_malloc0(sizeof(Surfaces*) * (nb + 1));
	  surfNb         = 0;
	  values         = commandLineGet_isoValues(&nb);
	  names          = commandLineGet_isoNames(&nb);
	  if (values)
	    {
	      tmplst = set->fieldsList;
	      while(tmplst)
		{
		  for (i = 0; i < nb; i++)
		    res = pot2surfCreate(set->surfsList, (ScalarField*)tmplst->data,
					 values[i], i, names[i]);
		  isosurfacesSet_showAll(set->surfsList[surfNb++], TRUE);
		  tmplst = g_list_next(tmplst);
		}
	    }
	  
	  /* We may add the iso-values from a file. */
	  if (valueFile)
	    {
	      somethingIsLoaded =
		pot2surfParse_XMLFile(valueFile, set->surfsList,
				      (ScalarField*)set->fieldsList->data, error);
	      if (!somethingIsLoaded)
		return FALSE;
	    }
	}
    }
  /* We apply the masking effect of planes, if necessary. */
  if (set->surfsList && set->planesList)
    for (i = 0; set->surfsList[i]; i++)
      isosurfacesHide(set->surfsList[i], set->planesList);
  /* The coloured map argument. */
  set->mapPlaneId = (gint*)commandLineGet_coloredMap();
  if (set->mapPlaneId)
    {
      if (!set->planesList)
	g_warning(_("option '--build-map' has been given but"
		    " no plane is available (use '--planes')."));
      if (!set->fieldsList)
	g_warning(_("option '--build-map' has been given but"
		    " no scalar field is available (use '--scalar-field')."));
      if (!set->shade)
	g_warning(_("option '--build-map' has been given but"
		    " no shade is available (use '--color-preset')."));
      set->logScale = commandLineGet_logScale();
      set->nIsolines = commandLineGet_nIsoLines();
      set->isoLinesColor = commandLineGet_isoLinesColor();
      if (set->planesList && set->fieldsList && set->shade)
	{
	  /* We change the visibility of the selected planes. */
	  for (i = 0; set->planesList[i]; i++)
	    for (j = 1; j <= set->mapPlaneId[0]; j++)
	      if (i == set->mapPlaneId[j])
		{
		  planeSet_rendered(set->planesList[i], FALSE);
		  break;
		}
	}
    }
  /* The bg image. */
  set->bgImage = commandLineGet_bgImage();
  /* The pick information. */
  if (valueFile)
    {
      if (!pickMesureParse_XMLFile(valueFile, visuData, &list, &mode, &info, error))
	return FALSE;
      nb = g_list_length(list) + 1;
      if (mode == DRAW_SELECTED)
	{
	  nodes = g_malloc(sizeof(int) * nb);
	  nodes[nb] = -1;
	  i = 0;
	  for (tmplst = list; tmplst; tmplst = g_list_next(tmplst))
	    {
	      if (GPOINTER_TO_INT(tmplst->data) == PICK_SELECTED)
		{
		  tmplst = g_list_next(tmplst);
		  nodes[i++] = GPOINTER_TO_INT(tmplst->data) - 1;
		}
	      else if (GPOINTER_TO_INT(tmplst->data) == PICK_REFERENCE_1)
		{
		  tmplst = g_list_next(tmplst);
		  /* We add the last selection. */
		  tmplst = g_list_next(tmplst);
		  nodes[i++] = GPOINTER_TO_INT(tmplst->data) - 1;
		}
	      else if (GPOINTER_TO_INT(tmplst->data) == PICK_HIGHLIGHT)
		tmplst = g_list_next(tmplst);
	    }
	  g_list_free(list);
	  nodes[i] = -1;
	}
      else
	nodes = (int*)0;
      if (mode != DRAW_NEVER)
	{
	  dt = (DataNode*)0;
	  switch (info)
	    {
	    case 0:
	      method = DRAW_METH_ID; break;
	    case 1:
	      method = DRAW_METH_TYPE; break;
	    default:
	      method = DRAW_METH_OTHER;
	      i = 2;
	      for (tmplst = nodeDataGet_list(); tmplst; tmplst = g_list_next(tmplst))
		{
		  if (nodeDataGet_used(DATA_NODE(tmplst->data), visuData))
		    {
		      if (i == (int)info)
			break;
		      i += 1;
		    }
		}
	      if (tmplst)
		dt = DATA_NODE(tmplst->data);
	      break;
	    }
	  visuBasicSet_extInfos(visuData, method, nodes, dt);
	}
    }

  return TRUE;
}
static void freeExtension(gpointer data)
{
  int i;
  GList *tmplst;
  VisuOptionSet *set;

  set = (VisuOptionSet*)data;
  /* Free allocated memory. */
  if (set->fieldsList)
    {
      tmplst = set->fieldsList;
      while(tmplst)
	{
	  scalarFieldFree((ScalarField*)tmplst->data);
	  tmplst = g_list_next(tmplst);
	}
	g_list_free(set->fieldsList);
    }
  if (set->planesList)
    {
      for (i = 0; set->planesList[i]; i++)
	g_object_unref(G_OBJECT(set->planesList[i]));
      g_free(set->planesList);
    }
  if (set->surfsList)
    {
      for (i = 0; set->surfsList[i]; i++)
	isosurfacesFree(set->surfsList[i]);
      g_free(set->surfsList);
    }
  g_free(set);
}

static void setFileFormatOption(OptionTable *opts, FileFormat *format,
				const gchar *name)
{
  Option *opt;
  gboolean status;
  int val;
  FileFormatProperty *prop;

  opt  = toolOptionsGet_optionFromTable(opts, name);
  prop = fileFormatGet_propertyByName(format, name);
  if (opt && !prop)
    g_warning("No exportation option called '%s' for the file format '%s'",
	      name, fileFormatGet_label(format));

  if (!prop || !opt)
    return;

  if (toolOptionsGet_type(opt) == OPTION_BOOLEAN)
    {
      if (toolOptionsGet_valueBoolean(opt, &status))
	fileFormatSet_propertyBoolean(prop, status);
    }
  else if (toolOptionsGet_type(opt) == OPTION_INTEGER)
    {
      if (toolOptionsGet_valueInteger(opt, &val))
	fileFormatSet_propertyInteger(prop, val);
    }
}
static gboolean setFileFormatOptionHelp(OptionTable *opts, FileFormat *format)
{
  GList *lst;
  FileFormatProperty *prop;

  if (toolOptionsGet_optionFromTable(opts, "list"))
    {
      for (lst = fileFormatGet_propertiesList(format); lst; lst = g_list_next(lst))
	{
	  prop = (FileFormatProperty*)lst->data;
	  fprintf(stdout, " - '%25s'", fileFormatGet_propertyName(prop));
	  switch (fileFormatGet_propertyType(prop))
	    {
	    case fileFormat_type_integer:
	      fprintf(stdout, " %10s (%5d): ", _("integer"),
		      fileFormatGet_propertyInteger(prop));
	      break;
	    case fileFormat_type_boolean:
	      fprintf(stdout, " %10s (%5d): ", _("boolean"),
		      fileFormatGet_propertyBoolean(prop));
	      break;
	    case fileFormat_type_string:
	      fprintf(stdout, " %10s: ", _("string"));
	      break;
	    default:
	      break;
	    }
	  fprintf(stdout, "%s.\n", fileFormatGet_propertyLabelUTF8(prop));
	}
      return TRUE;
    }
  else
    return FALSE;
}
static gboolean visuBasicDump(VisuData *data, DumpImage *dumpData,
			      int width, int height)
{
  DumpType *format;
  GList *pnt;
  char *exportFileName;
  guchar* image;
  GError *error;
  gboolean status;
  OptionTable *opts;

  exportFileName = (char*)commandLineGet_ExportFileName();
  if (!exportFileName)
    {
      g_error("This method should be called with"
	      " an argument that is the file name to export to.\n");
    }
  
  pnt = visuDumpGet_allModules();
  while (pnt && !fileFormatUse_match( ((DumpType*)pnt->data)->fileType,
				     exportFileName))
    pnt = g_list_next(pnt);
  if (!pnt)
    {
      g_warning("The format can't be found from the"
		" filename '%s' entered.\n", exportFileName);
      return FALSE;
    }
  format = (DumpType*)pnt->data;

  /* We transfer some options to file format. */
  opts = commandLineGet_options();
  if (opts)
    {
      /* Help command. */
      if (setFileFormatOptionHelp(opts, format->fileType))
	return TRUE;
      /* Ascii options. */
      setFileFormatOption(opts, format->fileType, "comment_hidden_nodes");
      setFileFormatOption(opts, format->fileType, "expand_box");
      setFileFormatOption(opts, format->fileType, "delete_hidden_nodes");
      setFileFormatOption(opts, format->fileType, "reduced_coordinates");
      setFileFormatOption(opts, format->fileType, "angdeg_box");
      /* PS options. */
      setFileFormatOption(opts, format->fileType, "reduced_colormap");
      /* SVG options. */
      setFileFormatOption(opts, format->fileType, "use_flat_rendering");
      /* XYZ options. */
      setFileFormatOption(opts, format->fileType, "expand_box");
      /* Jpg options. */
      setFileFormatOption(opts, format->fileType, "quality");
    }

  DBG_fprintf(stderr, "Visu Basic: Preparing to dump the image to '%s' as"
	      " format '%s'.\n", exportFileName, format->fileType->description);

  visuDataSet_sizeOfView(data, (guint)width, (guint)height);
  openGLViewCompute_matrixAndView(visuDataGet_openGLView(data)); 
  rebuildAllExtensionsLists(data);
  if (format->bitmap)
    {
      openGL_reDraw((const char**)0, data);
      /* We copy the pixmap into generic data. */
      image = visuOpenGLGet_pixmapData((guint)width, (guint)height, format->hasAlpha);
    }
  else
    image = (guchar*)0;

  error = (GError*)0;
  if (!format->writeFunc(format->fileType, exportFileName, width, height,
			 data, image, &error, (voidDataFunc)0, (gpointer)0) && error)
    {
      g_warning(error->message);
      g_error_free(error);
      status = FALSE;
    }
  else
    status = TRUE;
  
  if (image)
    g_free(image);
  /* We free the pixmap context. */
  visuOpenGLFree_pixmapContext(dumpData);

  return status;
}

static gchar* setDir(const gchar* const *sysDirs, const gchar *prefix,
		     const gchar* subDir, const gchar* defaultDir)
{
  gchar *dir;
  int i;

  dir = g_build_filename(prefix, subDir, NULL);
  if (!g_file_test(dir, G_FILE_TEST_IS_DIR))
    {
      g_free(dir);
      dir = (gchar*)0;
      /* We try the XDG stuff. */
      for (i = 0; sysDirs[i]; i++)
	{
	  dir = g_build_filename(sysDirs[i], subDir, NULL);
	  if (g_file_test(dir, G_FILE_TEST_IS_DIR))
	    break;
	  else
	    g_free(dir);
	  dir = (gchar*)0;
	}
    }
  if (!dir)
    dir = g_strdup(defaultDir);
  return dir;
}

void visuBasicSet_paths(const char *exeLocation)
{
#if SYSTEM_WIN32 == 1
  #define V_SIM_INI_FILE PACKAGE".ini"
  GIOChannel *iniFile;
  gchar *iniPath, *tmp, *me, *prefix;
  gchar *buffer, *line;
  GIOStatus res;
  GError *err;
  gchar **tokens;
  gsize length;

  v_sim_data_dir    = g_strdup(V_SIM_DATA_DIR_DEFAULT);
  v_sim_legal_dir   = g_strdup(V_SIM_LEGAL_DIR_DEFAULT);
  v_sim_pixmaps_dir = g_strdup(V_SIM_PIXMAPS_DIR_DEFAULT);
  v_sim_plugins_dir = g_strdup(V_SIM_PLUGINS_DIR_DEFAULT);
  v_sim_locale_dir  = g_strdup(V_SIM_LOCALE_DIR_DEFAULT);
  iniPath = g_strdup(V_SIM_INI_FILE);
  /* Try to find the INI file from cwd. */
  prefix = (gchar*)0;
  iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
  if (iniFile)
    prefix = g_get_current_dir();
  /* Try to find the INI file from the name of the executable. */
  if (!iniFile)
    {
      g_free(iniPath);

      if (g_file_test(exeLocation, G_FILE_TEST_IS_SYMLINK))
	tmp = g_file_read_link(exeLocation, (GError**)0);
      else
	tmp = g_strdup(exeLocation);
      me = normalize_path(tmp);
      g_free(tmp);
    
      DBG_fprintf(stderr, "Visu Basic: running program is '%s'.\n", me);
      /* From the location of the executable, we take the base dir. */
      prefix = g_path_get_dirname(me);
      g_free(me);

      iniPath = g_build_filename(prefix, V_SIM_INI_FILE, NULL);
      iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
    }  
  /* Try to find the INI file in the Windows directory. */
  if (!iniFile)
    {
      g_free(iniPath);
      prefix = g_strdup("C:\\WINDOWS");
      iniPath = g_build_filename(prefix, V_SIM_INI_FILE, NULL);
      iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
    }
  if (iniFile)
    {
      buffer = (gchar*)0;
      err = (GError*)0;
      do
	{
	  res = g_io_channel_read_line(iniFile, &line, &length, NULL, &err);
	  if (line && res == G_IO_STATUS_NORMAL)
	    {
	      tokens = g_strsplit(line, "=", 2);
	      if (!strcmp(g_strstrip(tokens[0]), "data_dir") && tokens[1])
		{
		  g_free(v_sim_data_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_data_dir = g_strdup(tmp);
		  else
		    v_sim_data_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "legal_dir") && tokens[1])
		{
		  g_free(v_sim_legal_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_legal_dir = g_strdup(tmp);
		  else
		    v_sim_legal_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "pixmaps_dir") && tokens[1])
		{
		  g_free(v_sim_pixmaps_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_pixmaps_dir = g_strdup(tmp);
		  else
		    v_sim_pixmaps_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "plugins_dir") && tokens[1])
		{
		  g_free(v_sim_plugins_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_plugins_dir = g_strdup(tmp);
		  else
		    v_sim_plugins_dir = g_build_filename(prefix, tmp, NULL);
		}
	      if (!strcmp(g_strstrip(tokens[0]), "locale_dir") && tokens[1])
		{
		  g_free(v_sim_locale_dir);
		  tmp = g_strstrip(tokens[1]);
		  if (g_path_is_absolute(tmp))
		    v_sim_locale_dir = g_strdup(tmp);
		  else
		    v_sim_locale_dir = g_build_filename(prefix, tmp, NULL);
		}
	      g_strfreev(tokens);
	      g_free(line);
	    }
	}
      while (res != G_IO_STATUS_EOF);
      g_io_channel_shutdown (iniFile, FALSE, (GError**)0);
      g_io_channel_unref (iniFile);
    }
  g_free(iniPath);
#endif
#if SYSTEM_X11 == 1
  const gchar* const *sysDirs;
  gchar *me, *prefix, *tmp;
#if DEBUG == 1
  int i;
#endif

  /* We try to get the dirs from XDG specs, otherwise we fall back to
     hard coded values at compilation time. */
#if GLIB_MINOR_VERSION > 5
  sysDirs = g_get_system_data_dirs();
#if DEBUG == 1
  fprintf(stderr, "Visu Basic: available data dirs:\n");
  for (i = 0; sysDirs[i]; i++)
    fprintf(stderr, " | '%s'.\n", sysDirs[i]);
#endif
#else
  sysDirs = g_malloc(sizeof(const gchar*));
  sysDirs[0] = (const gchar*)0;
#endif

  if (g_file_test(exeLocation, G_FILE_TEST_IS_SYMLINK))
    tmp = g_file_read_link(exeLocation, (GError**)0);
  else
    tmp = g_strdup(exeLocation);
  me = normalize_path(tmp);
  g_free(tmp);
    
  DBG_fprintf(stderr, "Visu Basic: running program is '%s'.\n", me);
  /* From the location of the executable, we take the base dir
     or its parents if the basedir is bin. */
  prefix = g_path_get_dirname(me);
  g_free(me);
  me = g_path_get_basename(prefix);
  if (!strcmp(me, "bin"))
    {
      g_free(me);
      me = prefix;
      prefix = g_path_get_dirname(me);
    }
  g_free(me);
  DBG_fprintf(stderr, " | prefix is '%s'.\n", prefix);

  v_sim_data_dir = setDir(sysDirs, prefix, "share/" PACKAGE, DATA_DIR);
  v_sim_legal_dir = setDir(sysDirs, prefix, "share/doc/" PACKAGE, LEGAL_DIR);
  v_sim_pixmaps_dir = setDir(sysDirs, prefix, "share/" PACKAGE "/pixmaps", PIXMAPS_DIR);
  v_sim_plugins_dir = setDir(sysDirs, prefix, "lib/" PACKAGE "/plug-ins", PLUGINS_DIR);
  v_sim_locale_dir = setDir(sysDirs, prefix, "share/locale", LOCALE_DIR);

  g_free(prefix);
#endif

  /* Create the local dirs. */
#if GLIB_MINOR_VERSION > 5
  v_sim_local_conf_dir = g_build_filename(g_get_user_config_dir(), "v_sim", NULL);
#else
  v_sim_local_conf_dir = g_build_filename(g_get_home_dir(), ".config/v_sim", NULL);
#endif
  if (!v_sim_local_conf_dir)
    g_warning("WARNING! Impossible to get the default"
	      " path $XDG_CONFIG_HOME/v_sim.\n");
  v_sim_old_local_conf_dir = g_build_filename(g_get_home_dir(), ".v_sim", NULL);

  DBG_fprintf(stderr, "Visu Basic: data directory      : '%s'.\n", v_sim_data_dir);
  DBG_fprintf(stderr, "Visu Basic: local conf directory: '%s'.\n", v_sim_local_conf_dir);
  DBG_fprintf(stderr, "Visu Basic: legal directory     : '%s'.\n", v_sim_legal_dir);
  DBG_fprintf(stderr, "Visu Basic: pixmaps directory   : '%s'.\n", v_sim_pixmaps_dir);
  DBG_fprintf(stderr, "Visu Basic: plug-ins directory  : '%s'.\n", v_sim_plugins_dir);
  DBG_fprintf(stderr, "Visu Basic: locale directory    : '%s'.\n", v_sim_locale_dir);
}
const gchar* visuBasicGet_dataDir()
{
  return v_sim_data_dir;
}
const gchar* visuBasicGet_legalDir()
{
  return v_sim_legal_dir;
}
const gchar* visuBasicGet_pixmapsDir()
{
  return v_sim_pixmaps_dir;
}
const gchar* visuBasicGet_localDir()
{
  return v_sim_local_conf_dir;
}
const gchar* visuBasicGet_oldLocalDir()
{
  return v_sim_old_local_conf_dir;
}
const gchar* visuBasicGet_pluginsDir()
{
  return v_sim_plugins_dir;
}
const gchar* visuBasicGet_localeDir()
{
  return v_sim_locale_dir;
}
void visuBasicFree_all()
{
  DBG_fprintf(stderr, "Visu Basic: free all.\n - the paths\n");
  g_free(v_sim_data_dir);
  g_free(v_sim_legal_dir);
  g_free(v_sim_pixmaps_dir);
  g_free(v_sim_plugins_dir);
  g_free(v_sim_locale_dir);
  g_free(v_sim_local_conf_dir);
  g_free(v_sim_old_local_conf_dir);

  DBG_fprintf(stderr, " - the color storage\n");
  toolColorFree_storage();
}

VisuUnits visuBasicGet_preferedUnit()
{
  return preferedUnit;
}
gboolean visuBasicSet_preferedUnit(VisuUnits unit)
{
  if (unit == preferedUnit)
    return FALSE;

  preferedUnit = unit;
  return TRUE;
}
static gboolean readUnit(gchar **lines, int nbLines, int position,
			 VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  VisuUnits unit;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_string(lines[0], position, &tokens, 1, FALSE, error))
    return FALSE;

  unit = visuToolsGet_unitFromName(tokens[0]);
  if (unit == unit_undefined)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: unit '%s' is unknown.\n"),
			   position, tokens[0]);
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  preferedUnit = unit;
  return TRUE;
}
static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  const gchar **units;

  if (preferedUnit != unit_undefined)
    {
      units = visuToolsGet_unitNames();

      g_string_append_printf(data, "# %s\n", DESC_PARAMETER_UNIT);
      g_string_append_printf(data, "%s: %s\n\n", FLAG_PARAMETER_UNIT,
			     units[preferedUnit]);
    }
}
