/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Jrmy BLANC et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BLANC, 
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Jrmy BLANC et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "vibration.h"

#include <math.h> /*for sinus*/
#include <coreTools/toolMatrix.h> /* for spherical coordonates */

#include <visu_rendering.h>
#include <renderingMethods/renderingSpin.h> /* for SPIN_VALUE*/
#include "opengl.h" /* for highlight methods*/
#include <openGLFunctions/objectList.h> /* permit to draw arrow */
#include <visu_extension.h>
#include <visu_basic.h>

#define OMEGA_COEFF 0.05f
#define AMPL_COEFF 0.5f
#define PHONON "Vibration"

/**
 * SECTION:vibration
 */

/* Displacement vector per atom is (U+iV) = (Ux + iVx, ...).
   Current applied displacement on one atom is (dx, dy, dz).
   qr is the Q.R part of the exponential. */
enum
  {
    Ux,
    Uy,
    Uz,
    Vx,
    Vy,
    Vz,
    dx,
    dy,
    dz,
    qr,
    nData
  };

/* Local variables. */
OpenGLExtension *extSpin = NULL;

/*Local Method */
static void freeVib(gpointer data);
static void spinCreate_openGLExtension();
static void freeVibe(gpointer obj, gpointer data _U_);
static gpointer newOrCopyVibe(gconstpointer obj, gpointer data _U_);
static void vibrationIncr(VisuData *data, gboolean time, gboolean phase);
static void onPopulationChanged(VisuData *dataObj, gint *ids, gpointer data);
static void onPositionChanged(VisuData *dataObj, gpointer data);
static gboolean move(gpointer data);
static gboolean xyzspin(VisuData *data);

typedef struct vibration_
{
  /* Parameters for the available phonons. */
  guint n;
  float *q;
  float *omega;
  float *en;
  /* Array of all the displacement vectors. */
  gboolean *loaded;
  guint nUs;
  float *u;
  float *norm;

  /* Current applied frequency and amplitude, phonon id, time and delta time. */
  float freq, ampl;
  gint iph;
  gulong t;

  /* Internal parameters. */
  gboolean haveTime, havePhase;
  gulong signal;
} Vibration;

/******************************************************************************************************/

guint vibrationStart_play(VisuData *data)
{
  Vibration *vib;
  guint timeoutID = 0;

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib, 0);

  timeoutID = g_timeout_add(50, (GSourceFunc)move, data);
 
  return timeoutID;
}

gboolean vibrationStop_play(guint timeoutID)
{
  /* to stop the timer */
  return g_source_remove(timeoutID);
}

/******************************************************************************************************/

static void freeVibe(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(float) * nData, obj);
#else
  g_free(obj);
#endif
}

static gpointer newOrCopyVibe(gconstpointer obj, gpointer data _U_)
{
  float *spinData;

#if GLIB_MINOR_VERSION > 9
  spinData = g_slice_alloc(sizeof(float) * nData);
#else
  spinData = g_malloc(sizeof(float) * nData);
#endif
  if (obj)
    {      
      spinData[Ux] = ((float*)obj)[0];
      spinData[Uy] = ((float*)obj)[1];
      spinData[Uz] = ((float*)obj)[2];
      spinData[Vx] = ((float*)obj)[3];
      spinData[Vy] = ((float*)obj)[4];
      spinData[Vz] = ((float*)obj)[5];
      spinData[dx] = 0.f;
      spinData[dy] = 0.f;
      spinData[dz] = 0.f;
      spinData[qr] = 0.f;
    }
  else
    memset(spinData, 0, sizeof(float) * nData);
    
  return (gpointer)spinData;
}

gboolean vibrationInit(VisuData *data, guint n, guint nNodes)
{
  Vibration *vib;
  gboolean new;
  VisuNodeArray *nodes;
  guint i;

  nodes = visuDataGet_nodeArray(data);

  new = FALSE;
  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  if (!vib)
    {
      new = TRUE;
      vib = g_malloc(sizeof(Vibration));
      g_object_set_data_full(G_OBJECT(data), PHONON, (gpointer)vib, freeVib);
      vib->q     = g_malloc(sizeof(float) * n * 3);
      vib->omega = g_malloc(sizeof(float) * n);
      vib->en    = g_malloc(sizeof(float) * n);
      vib->loaded= g_malloc(sizeof(gboolean) * n);
      vib->nUs   = 6 * nNodes;
      vib->u     = g_malloc(sizeof(float) * n * vib->nUs);
      vib->norm  = g_malloc(sizeof(float) * n);
      g_signal_connect(G_OBJECT(data), "NodePopulationIncrease",
		       G_CALLBACK(onPopulationChanged), (gpointer)0);
      vib->signal = g_signal_connect(G_OBJECT(data), "NodePositionChanged",
				     G_CALLBACK(onPositionChanged), (gpointer)0);
    }
  else if (n != vib->n || vib->nUs != 6 * nNodes)
    {
      vib->q     = g_realloc(vib->q,     sizeof(float) * n * 3);
      vib->omega = g_realloc(vib->omega, sizeof(float) * n);
      vib->en    = g_realloc(vib->en,    sizeof(float) * n);
      vib->loaded= g_realloc(vib->loaded,sizeof(gboolean) * n);
      vib->nUs   = 6 * nNodes;
      vib->u     = g_realloc(vib->u,     sizeof(float) * n * vib->nUs);
      vib->norm  = g_realloc(vib->norm,  sizeof(float) * n);
    }
  for (i = 0; i < n; i++)
    vib->loaded[i] = FALSE;

  vib->n    = n;
  vib->iph  = -1;
  vib->t    = 0;
  vib->freq = 5.f;
  vib->ampl = 1.f;
  vib->haveTime  = FALSE;
  vib->havePhase = FALSE;

  DBG_fprintf(stderr, "Vibration: set %d vibration properties for data %p (%d).\n",
	      n, (gpointer)data, nNodes);

  return new;
}
static void freeVib(gpointer data)
{
  Vibration *vib;

  vib = (Vibration*)data;
  g_free(vib->q);
  g_free(vib->omega);
  g_free(vib->en);
  g_free(vib->u);
  g_free(vib->norm);
  g_free(vib->loaded);
}

gboolean vibrationSet_displacements(VisuData *data, guint iph,
				    float *vibes, gboolean complex)
{
  float *tmpVibeValues, max;
  Vibration *vib;
  int i, nSet;

  g_return_val_if_fail(data && vibes, FALSE);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib && iph < vib->n, FALSE);

  DBG_fprintf(stderr, "Vibration: set vibration data for phonon %d (%d).\n",
	      iph, vib->nUs / 6);

  nSet = (complex)?6:3;

  /* We copy the displacement information into the Vibration
     structure for the given phonon. */
  max = 0.f;
  for (i = 0; i < (int)vib->nUs / 6; i++)
    {
      tmpVibeValues = vib->u + iph * vib->nUs + i * 6;
      tmpVibeValues[0] = vibes[i * nSet + 0];
      tmpVibeValues[1] = vibes[i * nSet + 1];
      tmpVibeValues[2] = vibes[i * nSet + 2];
      tmpVibeValues[3] = (complex)?vibes[i * nSet + 3]:0.f;
      tmpVibeValues[4] = (complex)?vibes[i * nSet + 4]:0.f;
      tmpVibeValues[5] = (complex)?vibes[i * nSet + 5]:0.f;
      max = MAX(tmpVibeValues[0] * tmpVibeValues[0] +
		tmpVibeValues[1] * tmpVibeValues[1] +
		tmpVibeValues[2] * tmpVibeValues[2] +
		tmpVibeValues[3] * tmpVibeValues[3] +
		tmpVibeValues[4] * tmpVibeValues[4] +
		tmpVibeValues[5] * tmpVibeValues[5], max);
      DBG_fprintf(stderr, "| %d -> (%g,%g,%g) + i (%g,%g,%g)\n", i,
		  tmpVibeValues[Ux], tmpVibeValues[Uy], tmpVibeValues[Uz],
		  tmpVibeValues[Vx], tmpVibeValues[Vy], tmpVibeValues[Vz]);
    }
  vib->norm[iph] = sqrt(max);
  DBG_fprintf(stderr, "Vibration: set norm to %f.\n", vib->norm[iph]);

  vib->loaded[iph] = TRUE;

  return TRUE;
}

gboolean vibrationSet_currentMode(VisuData *data, guint iph, GError **error)
{
  float *tmpVibeValues, xyz[3], *vals;
  VisuDataIter iter;
  VisuNodeProperty *nodevibe;
  GValue vibeValue = {0.0, {{0.0}, {0.0}}};
  Vibration *vib;
  VisuNodeArray *nodes;
  gboolean new, ok;
  int i, iNode;

  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Vibration: set current mode to %d.\n", iph);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib && iph < vib->n, FALSE);

  if (!vib->loaded[iph])
    {
      visuDataFree_population(data);
      ok = visuBasicLoad_dataFromFile(data, (FileFormat*)0, iph, error);
      if (!ok)
	return FALSE;
    }
  g_return_val_if_fail(vib->loaded[iph], FALSE);

  /* We copy for all the nodes the displacements into the PHONON
     node property. */
  nodes = visuDataGet_nodeArray(data);
  nodevibe = visuNodeGet_property(nodes, PHONON);
  new = (nodevibe == (VisuNodeProperty*)0);

  /*Associate the table to a VisuNodeArray as a new property*/
  if (new)
    nodevibe = visuNodeNew_pointerProperty(visuDataGet_nodeArray(data),
					   PHONON, freeVibe, newOrCopyVibe,
					   (gpointer)0 );

  /*for each node*/
  i   = 0;
  g_value_init(&vibeValue, G_TYPE_POINTER);
  visuDataIter_new(data, &iter);
  for(visuDataIter_start(data, &iter); iter.node;
      visuDataIter_next(data, &iter))
    {
      iNode = visuNodeGet_original(nodes, iter.node->number);
      iNode = (iNode >= 0)?iNode:(int)iter.node->number;
      vals = vib->u + iph * vib->nUs + iNode * 6;
      if (new)
	{
	  tmpVibeValues = newOrCopyVibe(vals, (gpointer)0);
	  g_value_set_pointer(&vibeValue, tmpVibeValues);
	  visuNodePropertySet_value(nodevibe, iter.node, &vibeValue);
	}
      else
	{
	  visuNodePropertyGet_value(nodevibe, iter.node, &vibeValue);
	  tmpVibeValues = (float*)g_value_get_pointer(&vibeValue);
	  tmpVibeValues[Ux] = vals[0];
	  tmpVibeValues[Uy] = vals[1];
	  tmpVibeValues[Uz] = vals[2];
	  tmpVibeValues[Vx] = vals[3];
	  tmpVibeValues[Vy] = vals[4];
	  tmpVibeValues[Vz] = vals[5];
	  tmpVibeValues[dx] = 0.f;
	  tmpVibeValues[dy] = 0.f;
	  tmpVibeValues[dz] = 0.f;
	}
      /* Set the phase. */
      visuDataGet_reducedNodePosition(data, iter.node, xyz);
      tmpVibeValues[qr] = 2.f * G_PI *
	(xyz[0] * vib->q[3 * iph + 0] +
	 xyz[1] * vib->q[3 * iph + 1] +
	 xyz[2] * vib->q[3 * iph + 2]);
      DBG_fprintf(stderr, "| %d -> (%g,%g,%g) + i (%g,%g,%g) | (%g,%g,%g) %g\n", i,
		  tmpVibeValues[Ux], tmpVibeValues[Uy], tmpVibeValues[Uz],
		  tmpVibeValues[Vx], tmpVibeValues[Vy], tmpVibeValues[Vz],
		  tmpVibeValues[dx], tmpVibeValues[dy], tmpVibeValues[dz],
		  tmpVibeValues[qr]);

      i += 1;
    }

  vib->iph = iph;

  return new;
}

gboolean vibrationSet_characteristic(VisuData *data, guint n,
				     float q[3], float en, float omega)
{
  Vibration *vib;

  g_return_val_if_fail(data, FALSE);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib || vib->n <= n, FALSE);

  vib->q[n * 3 + 0] = q[0];
  vib->q[n * 3 + 1] = q[1];
  vib->q[n * 3 + 2] = q[2];
  vib->omega[n]     = omega;
  vib->en[n]        = en;

  return TRUE;
}
gboolean vibrationGet_characteristic(VisuData *data, guint n,
				     float q[3], float *en, float *omega)
{
  Vibration *vib;

  g_return_val_if_fail(data && en && omega, FALSE);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib || vib->n <= n, FALSE);

  q[0]   = vib->q[n * 3 + 0];
  q[1]   = vib->q[n * 3 + 1];
  q[2]   = vib->q[n * 3 + 2];
  *omega = vib->omega[n];
  *en    = vib->en[n];

  return TRUE;
}
gboolean vibrationGet_nPhonons(VisuData *data, guint *n)
{
  Vibration *vib;

  g_return_val_if_fail(data && n, FALSE);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib, FALSE);

  *n = vib->n;

  return TRUE;
}
void vibrationSet_userFrequency(VisuData *data, float freq)
{
  Vibration *vib;
  float f;

  g_return_if_fail(data);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_if_fail(vib);

  f = vib->freq;
  if (freq == 0.f)
    vib->freq = vib->omega[vib->iph];
  else
    vib->freq = freq;
  /* We update t to keep the same position. */
  vib->t = (gulong)((float)vib->t * f / vib->freq);
}
gboolean vibrationSet_amplitude(VisuData *data, float ampl)
{
  Vibration *vib;

  g_return_val_if_fail(data, FALSE);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib, FALSE);

  if (vib->ampl == ampl)
    return FALSE;

  vib->ampl = ampl;
  return TRUE;
}
guint vibrationGet_currentMode(VisuData *data)
{
  Vibration *vib;

  g_return_val_if_fail(data, 0);

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib, 0);

  return vib->iph;
}

static gboolean getVibration(VisuNodeProperty **nodevibe, Vibration **vib,
			     GValue *vibeValue, VisuData *data)
{
  *vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(*vib, FALSE);

  g_value_init(vibeValue, G_TYPE_POINTER);
  *nodevibe = visuNodeGet_property(visuDataGet_nodeArray(data), PHONON);

  return TRUE;
}

static void vibrationIncr(VisuData *data, gboolean time, gboolean phase)
{
  GValue vibeValue = {0.0, {{0.0}, {0.0}}};
  float *dxyz, t, c, wdtr, wdti, dU[3];
  VisuDataIter iter;
  VisuNodeProperty *nodevibe;
  Vibration *vib;
  gboolean set;

  set = getVibration(&nodevibe, &vib, &vibeValue, data);
  g_return_if_fail(set);

  if (vib->iph < 0)
    return;

  DBG_fprintf(stderr, "Vibration: incr %d %d\n", time, phase);
  DBG_fprintf(stderr, " | t = %ld, w = %g.\n", vib->t, vib->freq);

  /* update the timer every time the method is call */
  if (time)
    vib->t += 1;
  else
    vib->t = 0;

  c =  AMPL_COEFF  * vib->ampl / vib->norm[vib->iph];
  t = -OMEGA_COEFF * vib->freq * vib->t;

  DBG_fprintf(stderr, " | c = %g, t = %g.\n", c, t);

  visuDataIter_new(data, &iter);
  for(visuDataIter_start(data, &iter); iter.node;
      visuDataIter_next(data, &iter))
    {
      /* Get the vector of vibration of the node (GValue) */
      visuNodePropertyGet_value(nodevibe, iter.node, &vibeValue );
      /* Convert GValue to Float[3]  */
      dxyz = (float*)g_value_get_pointer(&vibeValue);

      /* calculate xyz(t) */
      wdtr = c * sin(((time)?t:0.f) + ((phase)?dxyz[qr]:0.f));
      wdti = -c * cos(((time)?t:0.f) + ((phase)?dxyz[qr]:G_PI_2));
      dU[0] = dxyz[Ux] * wdtr + dxyz[Vx] * wdti;
      dU[1] = dxyz[Uy] * wdtr + dxyz[Vy] * wdti;
      dU[2] = dxyz[Uz] * wdtr + dxyz[Vz] * wdti;
      iter.node->translation[0] += dU[0] - dxyz[dx];
      iter.node->translation[1] += dU[1] - dxyz[dy];
      iter.node->translation[2] += dU[2] - dxyz[dz];
      dxyz[dx] = dU[0];
      dxyz[dy] = dU[1];
      dxyz[dz] = dU[2];

      DBG_fprintf(stderr, " | dx = %f, dy = %f, dz = %f\n",
		  dU[0], dU[1], dU[2]);
    }
  vib->haveTime  = time;
  vib->havePhase = phase;
     
  /*emit signal for pairs and other things */   
  g_signal_handler_block(G_OBJECT(data), vib->signal);
  visuDataEmit_nodePositionChanged(data);
  g_signal_handler_unblock(G_OBJECT(data), vib->signal);
  /* re-create the nodes list */
  visuData_createAllNodes(data);
  /* redraw with the new values */
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static gboolean move(gpointer data)
{
  vibrationIncr(VISU_DATA(data), TRUE, TRUE);
  return TRUE;
}
void vibrationReset_position(VisuData *data)
{
  vibrationIncr(data, FALSE, FALSE);
}
void vibrationSet_zeroTime(VisuData *data)
{
  vibrationIncr(data, FALSE, TRUE);
}

static gboolean xyzspin(VisuData *data)
{
  GValue vibeValue = {0.0, {{0.0}, {0.0}}};
  float xyz[3], *dxyz;
  float vals[3], sph[3], fact, eleSize;
  VisuDataIter iter;
  VisuNodeProperty *nodevibe;
  GLUquadricObj *obj;
  Vibration *vib;

  vib = (Vibration*)g_object_get_data(G_OBJECT(data), PHONON);
  g_return_val_if_fail(vib, FALSE);

  if (vib->iph < 0)
    return TRUE;

  obj = gluNewQuadric();

  /* We compute the maximum size of the arrow tail. */
  fact = 2.f * AMPL_COEFF * vib->ampl / vib->norm[vib->iph];

  g_value_init(&vibeValue, G_TYPE_POINTER);
  visuDataIter_new(data, &iter);
  nodevibe = visuNodeGet_property(visuDataGet_nodeArray(data), PHONON);
  g_return_val_if_fail(nodevibe, FALSE);

  for(visuDataIter_start(data, &iter); iter.element;
      visuDataIter_nextElement(data, &iter))
    {
      /* change spin color*/
      openGLSet_highlightColor(iter.element->material,
			       iter.element->rgb, 1.f);
      eleSize = visuRenderingGet_sizeOfElement(visuRenderingClassGet_current(),
					       iter.element);

      for(visuDataIter_restartNode(data, &iter); iter.node;
	  visuDataIter_nextNode(data, &iter))
	{
	  /* Get the vector of vibration of the node (GValue) */
	  visuNodePropertyGet_value(nodevibe, iter.node, &vibeValue );
	  /* Convert GValue to Float[3]  */
	  dxyz = (float*)g_value_get_pointer(&vibeValue);

	  /* Spherical coordonates of dxyz */
	  matrix_cartesianToSpherical(sph, dxyz);
	  vals[SPIN_MODULUS] = sph[0];
	  vals[SPIN_THETA]   = sph[1];
	  vals[SPIN_PHI]     = sph[2];

	  if (vals[SPIN_MODULUS] > 0.05f * vib->norm[vib->iph])
	    {
	      /* get the nodes position in the box */
	      visuDataGet_nodePosition(data, iter.node, xyz);
  
	      DBG_fprintf(stderr, "Vibration: drawing an arrow for the node \n");

	      glPushMatrix();

	      /* translate and rotate the arrow */
	      glTranslated(xyz[0] - dxyz[dx],
			   xyz[1] - dxyz[dy],
			   xyz[2] - dxyz[dz]);
	      glRotated(vals[SPIN_PHI], 0, 0, 1);
	      glRotated(vals[SPIN_THETA], 0, 1, 0);

	      /* draw a smooth arrow */
	      openGLObjectListDraw_smoothArrow
		(obj, -1, TRUE, vals[SPIN_MODULUS] * fact + 2.5f * eleSize, eleSize * 0.2f, 10, FALSE,
		 0.5f * eleSize, eleSize * 0.3f, 10, FALSE);

	      glPopMatrix();
	    }
	}
    }

  gluDeleteQuadric(obj);

  return TRUE;
}

void vibrationBuild_listSpin(VisuData *data)
{

  DBG_fprintf(stderr, "Vibration: rebuilding object list for visuData %p.\n",
	      (gpointer)data);
  if (!extSpin)
    spinCreate_openGLExtension();

  glNewList(extSpin->objectListId, GL_COMPILE);
  xyzspin(data);
  glEndList();
}

static void spinCreate_openGLExtension()
{
  int id;

  /* Get an OpenGL identifier to store the arrows and labels. */
  id      = openGLObjectList_new(1);
  extSpin = OpenGLExtension_new("vibration", "vibration",
				"arrows representing node moves.",
				id, vibrationBuild_listSpin);
  OpenGLExtensionSet_priority(extSpin, OPENGL_EXTENSION_PRIORITY_FIRST + 1);
  extSpin->used = 0;
  OpenGLExtensionRegister(extSpin);
}

gboolean vibrationSet_useSpin(gboolean value)
{
  if (!extSpin)
    spinCreate_openGLExtension();

  if (value == extSpin->used)
    return FALSE;

  extSpin->used = value;

  return TRUE;
}

gboolean vibrationIsSet(VisuData *data)
{
  return (g_object_get_data(G_OBJECT(data), PHONON) != (Vibration*)0);
}
static void onPopulationChanged(VisuData *dataObj, gint *ids, gpointer data _U_)
{
  int i;
  VisuNode *node;
  VisuNodeProperty *nodevibe;
  GValue vibeValue = {0.0, {{0.0}, {0.0}}};
  float *dxyz ,xyz[3];
  Vibration *vib;

  vib = (Vibration*)g_object_get_data(G_OBJECT(dataObj), PHONON);
  g_return_if_fail(vib);

  if (vib->iph < 0)
    return;

  g_value_init(&vibeValue, G_TYPE_POINTER);
  nodevibe = visuNodeGet_property(visuDataGet_nodeArray(dataObj), PHONON);

  DBG_fprintf(stderr, "Vibration: on population increase recalculate pahse.\n");
  /* We need to recalculate the phase of the new ones. */
  for (i = 2; ids[i] >= 0; i++)
    {
      node = visuDataGet_nodeFromNumber(dataObj, ids[i]);
      /* Get the vector of vibration of the node (GValue) */
      visuNodePropertyGet_value(nodevibe, node, &vibeValue );
      /* Convert GValue to Float[3]  */
      dxyz = (float*)g_value_get_pointer(&vibeValue);
      
      visuDataGet_reducedNodePosition(dataObj, node, xyz);
      DBG_fprintf(stderr, " | node %d, %f -> %f.\n", ids[i], dxyz[3],
		  xyz[0] * vib->q[3 * vib->iph + 0] +
		  xyz[1] * vib->q[3 * vib->iph + 1] +
		  xyz[2] * vib->q[3 * vib->iph + 2]);
      dxyz[qr] = 2.f * G_PI *
	(xyz[0] * vib->q[3 * vib->iph + 0] +
	 xyz[1] * vib->q[3 * vib->iph + 1] +
	 xyz[2] * vib->q[3 * vib->iph + 2]);
    }
}
static void onPositionChanged(VisuData *dataObj, gpointer data _U_)
{
  VisuDataIter iter;
  VisuNodeProperty *nodevibe;
  GValue vibeValue = {0.0, {{0.0}, {0.0}}};
  Vibration *vib;
  gboolean set;
  float *dxyz ,xyz[3];

  DBG_fprintf(stderr, "Vibration: on position changed recalculate phase.\n");

  set = getVibration(&nodevibe, &vib, &vibeValue, dataObj);
  g_return_if_fail(set);

  if (vib->iph < 0)
    return;

  /* We need to recalculate the phase of the new ones. */
  visuDataIter_new(dataObj, &iter);
  for(visuDataIter_start(dataObj, &iter); iter.node;
      visuDataIter_next(dataObj, &iter))
    {
      /* Get the vector of vibration of the node (GValue) */
      visuNodePropertyGet_value(nodevibe, iter.node, &vibeValue );
      /* Convert GValue to Float[3]  */
      dxyz = (float*)g_value_get_pointer(&vibeValue);
      
      visuDataGet_reducedNodePosition(dataObj, iter.node, xyz);
      DBG_fprintf(stderr, " | node %d, %f -> %f.\n", iter.node->number, dxyz[qr],
		  xyz[0] * vib->q[3 * vib->iph + 0] +
		  xyz[1] * vib->q[3 * vib->iph + 1] +
		  xyz[2] * vib->q[3 * vib->iph + 2]);
      dxyz[qr] = 2.f * G_PI *
	(xyz[0] * vib->q[3 * vib->iph + 0] +
	 xyz[1] * vib->q[3 * vib->iph + 1] +
	 xyz[2] * vib->q[3 * vib->iph + 2]);
    }
}
