/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_DATAFILE_H
#define VISU_DATAFILE_H

#include <glib.h>
#include <visu_data.h>
#include <coreTools/toolShade.h>

/**
 * DataFileInputScaleId:
 * @dataFile_normalize: input data are converted into [0;1] using input min/max values.
 * @dataFile_minMax: input data are converted into [0;1] using user defined min/max values.
 *
 * Control how input data are converted into [0;1], after conversion,
 * values are clamped if needed.
 */
typedef enum
  {
    dataFile_normalize,
    dataFile_minMax
  } DataFileInputScaleId;

/**
 * initDataFileModule:
 *
 * Called by V_Sim on startup, should not be called again.
 *
 * Returns: 1 if eveything goes right.
 */
int initDataFileModule();

/**
 * dataFileReDraw:
 * @data: a #VisuData object that holds the data file informations.
 *
 * This methods is a short-cut to call visuData_createAllNodes() and then emit the
 * 'OpenGLAskForReDraw' signal.
 *
 * May become deprecated soon.
 */
void dataFileReDraw(VisuData *data);

/**
 * dataFileSet_used:
 * @data: a #VisuData object to set the colorisation tool or not ;
 * @val: a boolean integer.
 * 
 * When TRUE, rendering is modified by applying a colorization
 * method to normal nodes. The color used depend on input data. See 
 * dataFileSet_file() to choose them.
 *
 * Returns: TRUE if val is true and if a valid input file is already
 *          in memory.
 */
gboolean dataFileSet_used(VisuData *data, int val);
/**
 * dataFileGet_used:
 * @data: a #VisuData object to get if the colorisation tool is set or not ;
 * 
 * This method retrieve the used flag, see dataFileSet_used() to set it.
 *
 * Returns: 1 if the used flag is set.
 */
int dataFileGet_used(VisuData *data);

/**
 * dataFileSet_file:
 * @attachedVisuData: a #VisuData object to attach the data file to ;
 * @filename: the path to find the data file on the disk ;
 * @new: return TRUE if some data are associated for the first time to
 *       the given #VisuData ;
 * @message: an allocated #GString to store some IO messages ;
 * @errors: an allocated integer to store if some errors occured.
 *
 * Call this method to parse a data file and associate its values to the given #VisuData object.
 *
 * Returns: 0 if nothing is loaded, 1 some data have been loaded.
 */
int dataFileSet_file(VisuData *attachedVisuData, char* filename, gboolean *new,
		     GString *message, int *errors);
/**
 * dataFileGet_file:
 * @visuData: a #VisuData object.
 *
 * If the given @visuData has an input data file already loaded, it returns its name.
 *
 * Returns: the name of the input data file if set. This name is own by
 *          V_Sim and should not be freed.
 */
gchar* dataFileGet_file(VisuData *visuData);
/**
 * dataFileGet_fileMinMaxFromColumn:
 * @visuData: the #VisuData object which the colour data are associated to ;
 * @minMax: an allocated array of two floating point values ;
 * @column: an integer.
 *
 * This method is used to retrieve the minimum and the maximum
 * values of the column designed by the @column argument. Column
 * are numbered beginning at 0.
 *
 * Returns: FALSE if @column < 0 or if @column is greater than the number
 *          of read column or if no file has been set.
 */
gboolean dataFileGet_fileMinMaxFromColumn(VisuData *visuData, float minMax[2], int column);

/**
 * dataFileSet_scaleType:
 * @visuData: a #VisuData object ;
 * @scale: an integer.
 *
 * This method is used to change the scale method used on input data.
 * See #DataFileInputScaleId for further informations. This method raises
 * a error if no input file has already been associated to the give @visuData.
 *
 * Returns: 1 if visuData_createAllNodes() should be called.
 */
gboolean dataFileSet_scaleType(VisuData *visuData, DataFileInputScaleId scale);
/**
 * dataFileGet_scaleType:
 * @visuData: a #VisuData object.
 *
 * Retrieve the scaling method of input data associated to the given @visuData.
 *
 * Returns: the scaling method if @visuData has an input data file associated
 *          or the default value if not.
 */
DataFileInputScaleId dataFileGet_scaleType(VisuData *visuData);

/**
 * dataFileSet_min:
 * @visuData: a #VisuData object ;
 * @min: a floating point value.
 *
 * When the scaling method is #dataFile_minMax (see #DataFileInputScaleId)
 * min and max value for convert input data are user defined. Use this method
 * to choose the minimum bound. This method raises
 * a error if no input file has already been associated to the give @visuData.
 *
 * Returns: 1 if visuData_createAllNodes() should be called.
 */
gboolean dataFileSet_min(VisuData *visuData, float min);
/**
 * dataFileSet_max:
 * @visuData: a #VisuData object ;
 * @max: a floating point value.
 *
 * When the scaling method is #dataFile_minMax (see #DataFileInputScaleId)
 * min and max value for convert input data are user defined. Use this method
 * to choose the maximum bound. This method raises
 * a error if no input file has already been associated to the give @visuData.
 *
 * Returns: 1 if visuData_createAllNodes() should be called.
 */
gboolean dataFileSet_max(VisuData *visuData, float max);
/**
 * dataFileGet_min:
 * @visuData: a #VisuData object.
 *
 * Retrieve the minimum value used when scaling is user defined.
 *
 * Returns: the minimum bound if @visuData has an input data file associated
 *          or the default value if not.
 */
float dataFileGet_min(VisuData *visuData);
/**
 * dataFileGet_max:
 * @visuData: a #VisuData object.
 *
 * Retrieve the maximum value used when scaling is user defined.
 *
 * Returns: the maximum bound if @visuData has an input data file associated
 *          or the default value if not.
 */
float dataFileGet_max(VisuData *visuData);

/**
 * dataFileSet_colUsed:
 * @visuData: a #VisuData object ;
 * @val: 0 or 1 ;
 * @pos: an integer in [0;2].
 *
 * Choose if the loaded value should change the given channel of the colour.
 *
 * Returns: 1 if visuData_createAllNodes() should be called.
 */
gboolean dataFileSet_colUsed(VisuData *visuData, int val, int pos);
/**
 * dataFileGet_colUsed:
 * @visuData: a #VisuData object.
 *
 * This method is used to retrieve the vector used to adapt or not the colour
 * to the value of the loaded data.
 *
 * Returns: a three value array, own by V_Sim. It should not be freed.
 */
int* dataFileGet_colUsed(VisuData *visuData);

/**
 * dataFileGet_nbColumns:
 * @visuData: a #VisuData object.
 *
 * This method is used to retrieve the number of columns of data read in
 * the loaded file.
 *
 * Returns: this number of columns.
 */
int dataFileGet_nbColumns(VisuData *visuData);

/**
 * dataFileSet_shade:
 * @visuData: the #VisuData object which the colour data are associated to ;
 * @shade: a valid #Shade object.
 *
 * Apply all caracteristic of the given shade to the colorization the
 * the given #VisuObject.
 *
 * Returns: TRUE if visuData_createAllNodes() should be called.
 */
gboolean dataFileSet_shade(VisuData *visuData, Shade *shade);
/**
 * dataFileGet_shade:
 * @visuData: the #VisuData object which the colour data are associated to.
 *
 * Return the shade used to colourise the nodes.
 *
 * Returns: the Shade used (own by V_Sim).
 */
Shade* dataFileGet_shade(VisuData *visuData);
gboolean dataFileSet_scalingUsed(VisuData *visuData, int val);
int dataFileGet_scalingUsed(VisuData *visuData);

/**
 * dataFileApply_hideOnMinValue:
 * @visuData: the #VisuData object which the colour data are associated to ;
 * @column: a integer corresponding to a column id ;
 * @value: a floating point value.
 *
 * Use this method hide nodes whose data from column @column is
 * lower than the given @value. If @column is -1, then all previous
 * masking are cancelled.
 *
 * Returns: TRUE if visuData_createAllNodes() should be called and
 *          "NodeRenderedChanged" signal should be emitted (with a
 *          NULL #VisuElement parameter).
 */
gboolean dataFileApply_hideOnMinValue(VisuData *visuData, int column, float value);

#endif
