/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "renderingAtomic_ascii.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "renderingAtomic.h"
#include <visu_rendering.h>
#include <visu_data.h>

/**
 * SECTION:renderingAtomic_ascii
 * @short_description: Method to load ascii position file.
 *
 * <para>Ascii format is a plain text format to store elements and
 * their position. The first line is a commentary and is not
 * read. Then, the next two lines must contains each three numbers,
 * describing the box containing the data. The first number is the
 * length on x direction. The next two numbers are on one hand the
 * component on the same x direction and on the other hand the other
 * component on a direction orthogonal to the x direction, setting
 * thus the y direction. The three other numbers are the component of
 * the z direction on first, the x direction, then the y direction and
 * finally in a direction orthogonal both to x and y directions. After
 * these three first lines, all other lines can be blank, beginning
 * with a '#' character or containing three numbers and a label name
 * for the element.</para>
 *
 * <para>The read_ascii_file() method is used to try to load an ascii
 * file in memory. Other methods are quite private and just put here
 * for other loader which wants to share the code of this loader (see
 * read_xyz_file() for instance).</para>
 */

static gboolean loadAscii(VisuData *data, const gchar* filename,
			  FileFormat *format, int nSet, GError **error);
static int read_ascii_file(VisuData *data, FILE *flux, GError **error);


RenderingFormatLoad* atomicAsciiInit()
{
  char *typeASCII[] = {"*.ascii", (char*)0};
  char *descrASCII = _("'x y z Element' format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "Plain text native format of V_Sim.";
  meth->fmt = fileFormatNew(descrASCII, typeASCII);
  if (!meth->fmt)
    {
      g_error("Can't initialize the rendering atomic method, aborting...");
    }
  meth->priority = 50;
  meth->load = loadAscii;

  return meth;
}

/******************************************************************************/

int readFile_is_comment(char *str) {
   int n = strlen(str);
   register int i;
   
   if(n == 0) return 1;
   else if(str[0] == '#') return 1;
   else if(str[0] == '!') return 1;
   else {
      /* n-2 to skip terminal \n */
      for(i=n-2; i>= 0; i--) {
         if(str[i] != ' ') return 0;
      }
   }
   return 1;
}

/******************************************************************************/

void freeDataAscii(gpointer data)
{
  g_free((struct dataAscii*)data);
}
void putValTypeInAsciiType(gpointer key _U_, gpointer value, gpointer data)
{
  VisuElement **tab;
  struct dataAscii *infos;
  
  tab = *(VisuElement***)data;
  infos = (struct dataAscii *)value;
  tab[infos->pos] = infos->ele;
}
void putValNbInAsciiType(gpointer key _U_, gpointer value, gpointer data)
{
  int *tab;
  struct dataAscii *infos;
  
  tab = *(int**)data;
  infos = (struct dataAscii *)value;
  tab[infos->pos] = infos->nbNodes;
}
/******************************************************************************/

static gboolean loadAscii(VisuData *data, const gchar* filename,
			  FileFormat *format _U_, int nSet _U_, GError **error)
{
  int res;
  FILE *readFrom;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  readFrom = fopen(filename, "r");
  if (!readFrom)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("Impossible to open this file.\n"));
      return FALSE;
    }
  /* Do the job of parsing. */
  res = read_ascii_file(data, readFrom, error);
  fclose(readFrom);

  if (res < 0)
    {
      /* The file is not an ascii file. */
      return FALSE;
    }
  else if (res > 0)
    {
      /* The file is an ascii file but some errors occured. */
      return TRUE;
    }
  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}


static int read_ascii_file(VisuData *data, FILE *flux, GError **error)
{
  char line[MAX_LINE_LENGTH], info[MAX_LINE_LENGTH];
  double xu, yu, zu;
  unsigned int i;
  int res;
  char nomloc[9], *ptChar;
  VisuElement *type;
  VisuElement **types;
  unsigned int *nattyp;
  unsigned int ntype, nbLine;
  GHashTable *elements;
  struct dataAscii *infos;
  float boxGeometry[6], xyz[3];
  gchar *infoUTF8;
    
  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  /* The first line is a commentry. */
  (void)fgets(info, MAX_LINE_LENGTH, flux);
  if(feof(flux))
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The ASCII file should"
			     " contain more than 3 lines.\n"));
      /* The file doesn't conform to ascii norm, we exit with a wrong type flag. */
      return -1;
    }
  /* The second line contains xx, xy and yy */
  (void)fgets(line, MAX_LINE_LENGTH, flux);
  if(feof(flux))
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The ASCII file should"
			     " contain more than 3 lines.\n"));
      /* The file doesn't conform to ascii norm, we exit with a wrong type flag. */
      return -1;
    }
  if(sscanf(line, "%f %f %f", &(boxGeometry[0]), &(boxGeometry[1]), &(boxGeometry[2])) != 3)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("Cannot read dxx dyx dyy"
			     " on 2nd line of ASCII file.\n"));
      /* The file doesn't conform to ascii norm, we exit with a wrong type flag. */
      return -1;
    }
  /* The third line contains xz, yz and zz */
  (void)fgets(line, MAX_LINE_LENGTH, flux);
  if(feof(flux))
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The ASCII file should"
			     " contain more than 3 lines.\n"));
      /* The file doesn't conform to ascii norm, we exit with a wrong type flag. */
      return -1;
    }
  if(sscanf(line, "%f %f %f", &boxGeometry[3], &boxGeometry[4], &boxGeometry[5]) != 3)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("Cannot read dzx dzy dzz"
			     " on 3rd line of ASCII file.\n"));
      /* The file doesn't conform to ascii norm, we exit with a wrong type flag. */
      return -1;
    }
   
  /* Ok, from this point we assert that the file is in ascii format.
     All following errors are treated as syntax errors and the corresponding
     flag is return. */

  /* 1st pass to count ntype */
  nbLine = 4;
  ntype = 0;
  elements = g_hash_table_new_full(g_direct_hash, g_direct_equal,
				   NULL, freeDataAscii);
  g_return_val_if_fail(elements, -1);
  for(;;)
    {
      (void)fgets(line, MAX_LINE_LENGTH, flux);
      nbLine += 1;
      if(feof(flux))
	{
	  (void)rewind(flux);
	  break;
	}
      if(readFile_is_comment(line)) continue;
      if(sscanf(line, "%lf %lf %lf %s", 
            &xu, &yu, &zu, nomloc) != 4)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Cannot read x, y, z,"
				 " name in ASCII file at line %d.\n\n"
				 " Quoting '%s'.\n"), nbLine, line);
	  return 1;
	}
      nomloc[8] = '\0';
      /* adding nomloc to the hashtable */
      type = visuElementGet_fromName(nomloc);
      if (!type)
	{
	  type = visuElementNew_withName(nomloc);
	  if (!type)
	    {
	      g_warning("Cannot create a new type for '%s'.", nomloc);
	      return 1;
	    }
	  res = visuElementAdd(type);
	  if (res)
	    {
	      g_warning("Cannot add new element, maximum has been reached.");
	      return 1;
	    }
	}
      infos = (struct dataAscii*)g_hash_table_lookup(elements, (gconstpointer)type);
      if (!infos)
	{
	  infos = g_malloc(sizeof(struct dataAscii));
	  infos->ele = type;
	  infos->pos = ntype;
	  infos->nbNodes = 1;
	  g_hash_table_insert(elements, (gpointer)type, (gpointer)infos);
	  ntype++;
	}
      else
	infos->nbNodes += 1;
    }
  /* Allocate the space for the nodes. */
  types = g_malloc(sizeof(VisuElement*) * ntype);
  nattyp = g_malloc(sizeof(int) * ntype);
  g_hash_table_foreach(elements, (GHFunc)putValTypeInAsciiType, (gpointer)&types);
  g_hash_table_foreach(elements, (GHFunc)putValNbInAsciiType, (gpointer)&nattyp);
  g_hash_table_destroy(elements);
  if (ntype <= 0)
    {
      g_free(nattyp);
      g_free(types);
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The file contains no atom coordinates.\n"));
      return 1;
    }

  /* Set the commentary. */
  info[MAX_LINE_LENGTH - 1] = '\0';
  ptChar = strchr(info, '\n');
  if (ptChar)
    *ptChar = '\0';
  if (info[0] == '#')
    infoUTF8 = g_locale_to_utf8(info + 1, -1, NULL, NULL, NULL);
  else
    infoUTF8 = g_locale_to_utf8(info, -1, NULL, NULL, NULL);
  if (infoUTF8)
    {
      visuDataSet_fileCommentary(data, infoUTF8, 0);
      g_free(infoUTF8);
    }
  else
    g_warning("Can't convert '%s' to UTF8.\n", info);

  res = visuDataSet_population(data, ntype, nattyp, types);
  if (!res)
    {
      g_error("Can't store the nodes in the VisuData object.");
    }
  g_free(nattyp);
  g_free(types);
         
  /* reread the file to store the coordinates */
  
  (void)fgets(line, MAX_LINE_LENGTH, flux);
  (void)fgets(line, MAX_LINE_LENGTH, flux);
  (void)fgets(line, MAX_LINE_LENGTH, flux);
   
  i = 0;
  for(;;)
    {
      (void)fgets(line, MAX_LINE_LENGTH, flux);
      if(feof(flux))
	break;
      if(readFile_is_comment(line)) continue;
      
      (void)sscanf(line, "%lf %lf %lf %s", 
		   &xu, &yu, &zu, nomloc);
      nomloc[8] = '\0';
      type = visuElementGet_fromName(nomloc);
      if(!type)
	{
	  g_warning("INTERNAL ERROR! The input file must"
		    " have been modified when loading since"
		    " at the second reading the element '%s'"
		    " seems to be new.\n", nomloc);
	  return 1;
	}
      xyz[0] = (float)xu;
      xyz[1] = (float)yu;
      xyz[2] = (float)zu;
      visuDataAdd_nodeFromElement(data, type, xyz);
      i++;
    }

  /* We finish with the box description. */
  visuDataSet_boxGeometry(data, boxGeometry, TRUE);
   
  return 0;
}

