/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "cylinder.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>

#include <opengl.h>
#include <visu_tools.h>
#include <visu_data.h>
#include <visu_object.h>
#include <visu_configFile.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolConfigFile.h>

/**
 * SECTION:cylinder
 * @short_description: Gives methods to draw cylinders as OpenGl
 * objects pairing two elements.
 *
 * <para>The cylinders have two characteristics: their radius and
 * their colour. Their radius are in the units of the rendered
 * box. They can be specific to each kind of pairs (e.g. Si-Si) or
 * they have a default value. Only the default value is stored in the
 * resource file with the flag "pairCylinder_radius". Their color is
 * herited by the colour of the pair. The lighting values
 * (i.e. emi., shi., spe. ...) are not movable and are fixed to amb =
 * 0.5, dif = 0.5, shi = 0, spe = 0, emi = 0.</para>
 */

static int listCylinder;

#define FLAG_RESOURCES_PAIR_RADIUS "pairCylinder_pairRadius"
#define DESC_RESOURCES_PAIR_RADIUS "This value is the radius for specific pairs drawn as cylinders ; element1 elemen2 0 < real < 10"
#define FLAG_RESOURCES_LINK_RADIUS "pairCylinder_linkRadius"
#define DESC_RESOURCES_LINK_RADIUS "This value is the radius for specific drawn link as cylinders ; [element1] [element2] [min] [max] [0 < real < 10]"
#define FLAG_RESOURCES_CYLINDER_RADIUS "pairCylinder_radius"
#define DESC_RESOURCES_CYLINDER_RADIUS "This value is the default radius of the pairs drawn as cylinders ; 0 < real < 10"
#define RESOURCES_CYLINDER_RADIUS_DEFAULT 0.1
static float cylinderRadius;

#define FLAG_RESOURCES_CYLINDER_COLOR_TYPE "cylinder_colorType"
#define DESC_RESOURCES_CYLINDER_COLOR_TYPE "It chooses the colors of the cylinders according differents criterion ;"
#define RESOURCES_CYLINDER_COLOR_TYPE_DEFAULT cylinder_user_color
static int cylinderColorType;

static gboolean readCylinderDefaultRadius(gchar **lines, int nbLines, int position,
					  VisuData *dataObj, GError **error);
static gboolean readCylinderRadius(gchar **lines, int nbLines,
				   int position, VisuData *dataObj, GError **error);
static gboolean readLinkRadius(gchar **lines, int nbLines,
			       int position, VisuData *dataObj, GError **error);
static gboolean readCylinderColorType(gchar **lines, int nbLines, int position,
				      VisuData *dataObj, GError **error);
static gboolean exportResourcesCylinder(GString *data, int *nbLinesWritten,
					VisuData *dataObj);


int setCylinderGeneralRadius(float val)
{
  DBG_fprintf(stderr, "Pairs Cylinder : set the general cylinder radius to %f.\n", val);
  val = CLAMP(val, RESOURCES_CYLINDER_RADIUS_MIN, RESOURCES_CYLINDER_RADIUS_MAX);

  if (val == cylinderRadius)
    return 0;

  visuPairSet_outOfDate();
  cylinderRadius = val;
  return 1;
}
float getCylinderGeneralRadius()
{
  return cylinderRadius;
}

int setCylinderRadius(VisuPairData *data, float val)
{
  float *radius;

  if (!data)
    return 0;

  DBG_fprintf(stderr, "Pairs Cylinder : set the cylinder radius to %f.\n", val);
  val = CLAMP(val, RESOURCES_CYLINDER_RADIUS_MIN, RESOURCES_CYLINDER_RADIUS_MAX);

  radius = (float*)visuPairGet_linkProperty(data, "radius");
  if (!radius)
    {
      radius = g_malloc(sizeof(float));
      visuPairSet_linkProperty(data, "radius", (gpointer)radius);
      *radius = val;
      visuPairSet_outOfDate();
      return 1;
    }
  else
    if (*radius != val)
      {
	*radius = val;
	visuPairSet_outOfDate();
	return 1;
      }
  return 0;
}
float getCylinderRadius(VisuPairData *data)
{
  float *radius;

  if (!data)
    return -1.;

  radius = (float*)visuPairGet_linkProperty(data, "radius");
  if (radius)
    return *radius;
  else
    return cylinderRadius;
}
int setCylinderColorType(int val)
{
  DBG_fprintf(stderr, "Pairs Cylinder : set the cylinder color type to %d.\n", val);
  g_return_val_if_fail(val >= 0 && val < cylinder_nb_color, 0);

  cylinderColorType = val;
  visuPairSet_outOfDate();
  return 1;
}
int getCylinderColorType()
{
  return cylinderColorType;
}

void setColorAndWidthForCylinder(VisuElement *ele1 _U_, VisuElement *ele2 _U_,
				 VisuPairData *data)
{
  float rgba[4], mm[5] = {0.5, 0.5, 0., 0. , 0.};
  Color *color;

  switch (cylinderColorType)
    {
    case cylinder_user_color:
      color = (Color*)visuPairGet_linkProperty(data, "color");
      if (!color)
	return;
      rgba[0] = color->rgba[0];
      rgba[1] = color->rgba[1];
      rgba[2] = color->rgba[2];
      rgba[3] = 1.;
      setOpenGlMaterial(mm, rgba);
      break;
    case cylinder_element_color:
    default:
      break;
    }
  visuPairSet_outOfDate();
}

void drawCylinderPairs(VisuElement *ele1, VisuElement *ele2, VisuPairData *data,
		       OpenGLView *view, double x1, double y1, double z1,
		       double x2, double y2, double z2, float d2, float alphaColour)
{
  float rgba[4], mm[5] = {0.5, 0.5, 0., 0. , 0.};
  Color *color;
  int mat1, mat2, nlat;
  double vNorm[3]; /* vecteur normal aux vecteurs (0,0,1) et (x2-x1, y2-y1, z2-z1) */
  double vDest[3]; /* vecteur (x2-x1, y2-y1, z2-z1) */
  double cosAlpha; /* cosinus entre (0,0,1) et Vdest */
  double alpha;
  #define RADTODEG 57.29577951
  float radius, *tmpRad;
  GLUquadricObj *obj;


  tmpRad = visuPairGet_linkProperty(data, "radius");
  if (tmpRad)
    radius = *tmpRad;
  else
    radius = cylinderRadius;
  nlat = OpenGLViewGet_numberOfFacettes(view, radius);
  vDest[0] = x2 - x1;
  vDest[1] = y2 - y1;
  vDest[2] = z2 - z1;
  if (vDest[0] != 0 || vDest[1] != 0)
    {
      vNorm[0] = - vDest[1];
      vNorm[1] = vDest[0];
      vNorm[2] = 0.;
      cosAlpha = sqrt((vDest[2] * vDest[2]) / d2);
      if (vDest[2] < 0.)
	cosAlpha = - cosAlpha;
      alpha = acos(cosAlpha) * RADTODEG;
    }
  else
    {
      vNorm[0] = 1.;
      vNorm[1] = 0.;
      vNorm[2] = 0.;
      if (vDest[2] < 0.)
	alpha = 180.;
      else
	alpha = 0.;
    }
  obj = gluNewQuadric();
  glPushMatrix();
  switch (cylinderColorType)
    {
    case cylinder_user_color:
      color = (Color*)visuPairGet_linkProperty(data, "color");
      if (!color)
	return;
      rgba[0] = color->rgba[0];
      rgba[1] = color->rgba[1];
      rgba[2] = color->rgba[2];
      rgba[3] = alphaColour;
      setOpenGlMaterial(mm, rgba);

      glTranslated(x1, y1, z1);
      glRotated(alpha, vNorm[0], vNorm[1], vNorm[2]);
      gluCylinder(obj, (GLdouble)radius, (GLdouble)radius,
		  (GLdouble)sqrt(d2), (GLint)nlat, (GLint)1);
      break;
    case cylinder_element_color:
      mat1 = visuElementGet_identifierMaterial(ele1);
      mat2 = visuElementGet_identifierMaterial(ele2);
      if (mat1 <= 0 || mat2 <= 0)
	g_warning("Can't draw cylinders because either ele1"
		  "or ele2 has no identifier for material.\n");
      glTranslated(x1, y1, z1);
      glRotated(alpha, vNorm[0], vNorm[1], vNorm[2]);
      glCallList(mat1);
      gluCylinder(obj, (GLdouble)radius, (GLdouble)radius,
		  (GLdouble)sqrt(d2) * 0.5f, (GLint)nlat, (GLint)1);
      glPopMatrix();
      glPushMatrix();
      glTranslated(x2, y2, z2);
      glRotated(alpha - 180., vNorm[0], vNorm[1], vNorm[2]);
      glCallList(mat2);
      gluCylinder(obj, (GLdouble)radius, (GLdouble)radius,
		  (GLdouble)sqrt(d2) * 0.5f, (GLint)nlat, (GLint)1);
      break;
    default:
      break;
    }
  glPopMatrix();
}

PairsExtension* initPairsCylinder()
{
  char *name = _("Cylinder pairs");
  char *desc = _("Pairs are rendered by cylinders."
		 " The color and the width can by chosen.");
  PairsExtension *extension;
  VisuConfigFileEntry *resourceEntry, *oldEntry;

  extension = visuPairExtensionNew("Cylinder pairs", name, desc, TRUE,
				   (initEndOpenGlPairsFunc)0,
				   (initEndOpenGlPairsFunc)0,
				   setColorAndWidthForCylinder, (startEndPairsFunc)0,
				   drawCylinderPairs);

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_CYLINDER_COLOR_TYPE,
					  DESC_RESOURCES_CYLINDER_COLOR_TYPE,
					  1, readCylinderColorType);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_CYLINDER_RADIUS,
					  DESC_RESOURCES_CYLINDER_RADIUS,
					  1, readCylinderDefaultRadius);
  oldEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				     FLAG_RESOURCES_PAIR_RADIUS,
				     DESC_RESOURCES_PAIR_RADIUS,
				     1, readCylinderRadius);
  visuConfigFileSet_version(resourceEntry, 3.1f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_LINK_RADIUS,
					  DESC_RESOURCES_LINK_RADIUS,
					  1, readLinkRadius);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileSet_replace(resourceEntry, oldEntry);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesCylinder);


  listCylinder = openGLObjectList_new(2);

  cylinderRadius = RESOURCES_CYLINDER_RADIUS_DEFAULT;
  cylinderColorType = cylinder_user_color;

  pointerToPairExtension_cylinder = extension;
  return extension;
}


/*****************************************/
/* Dealing with parameters and resources */
/*****************************************/

static gboolean readLinkRadius(gchar **lines, int nbLines, int position,
			       VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  int id;
  float radius;
  VisuPairData *data;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  tokens = g_strsplit_set(lines[0], " \n", MAX_LINE_LENGTH);
  id = 0;
  if (!visuPairRead_linkFromTokens(tokens, &id, &data, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  /* Read the associated radius. */
  if (!configFileRead_floatFromTokens(tokens, &id, &radius, 1, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  /* Check. */
  if (radius < RESOURCES_CYLINDER_RADIUS_MIN ||
      radius > RESOURCES_CYLINDER_RADIUS_MAX)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: radius must be in %g-%g.\n"),
			   position, RESOURCES_CYLINDER_RADIUS_MIN,
			   RESOURCES_CYLINDER_RADIUS_MAX);
      return FALSE;
    }
  /* Set the value. */
  setCylinderRadius(data, radius);

  return TRUE;
}
static gboolean readCylinderDefaultRadius(gchar **lines, int nbLines, int position,
					  VisuData *dataObj _U_, GError **error)
{
  float radius;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &radius, 1, error))
    return FALSE;
  if (radius < RESOURCES_CYLINDER_RADIUS_MIN ||
      radius > RESOURCES_CYLINDER_RADIUS_MAX)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: radius must be in %g-%g.\n"),
			   position, RESOURCES_CYLINDER_RADIUS_MIN,
			   RESOURCES_CYLINDER_RADIUS_MAX);
      return FALSE;
    }
  setCylinderGeneralRadius(radius);
  return TRUE;
}
static gboolean readCylinderColorType(gchar **lines, int nbLines, int position,
				      VisuData *dataObj _U_, GError **error)
{
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_integer(lines[0], position, &val, 1, error))
    return FALSE;
  if (val < 0 || val >= cylinder_nb_color)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 1 integer value must"
			     " appear after the %s markup.\n"),
			   position, FLAG_RESOURCES_CYLINDER_COLOR_TYPE);
      return FALSE;
    }
  setCylinderColorType(val);

  return TRUE;
}
static void exportPairsRadius(VisuElement *ele1, VisuElement *ele2,
			      VisuPairData *data, gpointer userData)
{
  struct foreachFuncExport_struct *str;
  float *radius;

  radius = (float*)visuPairGet_linkProperty(data, "radius");
  if (radius)
    {
      str = ( struct foreachFuncExport_struct*)userData;
      /* We export the resource only if the elements are
	 part of the given VisuData. */
      if (str->dataObj)
	{
	  /* We test the first element. */
	  if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt, ele1))
	    return;
	  /* We test the second element. */
	  if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt, ele2))
	    return;
	}

      g_string_append_printf(str->data,
			     "%s:\n"
			     "    %s %s  %4.3f %4.3f  %4.3f\n", FLAG_RESOURCES_LINK_RADIUS,
			     ele1->name, ele2->name, data->minMax[PAIRS_MIN],
			     data->minMax[PAIRS_MAX], *radius);
      *str->nbLinesWritten += 2;
    }
}
static gboolean exportResourcesCylinder(GString *data, int *nbLinesWritten,
				 VisuData *dataObj)
{
  struct foreachFuncExport_struct str;

  g_string_append_printf(data, "# %s 0 <= integer < %d\n", DESC_RESOURCES_CYLINDER_COLOR_TYPE,
			 cylinder_nb_color);
  g_string_append_printf(data, "%s:\n    %d\n",
			 FLAG_RESOURCES_CYLINDER_COLOR_TYPE,
			 cylinderColorType);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_CYLINDER_RADIUS);
  g_string_append_printf(data, "%s:\n    %f\n",
			 FLAG_RESOURCES_CYLINDER_RADIUS, cylinderRadius);
  *nbLinesWritten = 6;
  str.data           = data;
  str.nbLinesWritten = nbLinesWritten;
  str.dataObj        = dataObj;
  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_PAIR_RADIUS);
  visuPairForeach(exportPairsRadius, &str);
  g_string_append_printf(data, "\n");
  *nbLinesWritten += 2;

  return TRUE;
}



/* OBSOLETE function kept for backward compatibility. */
static gboolean readCylinderRadius(gchar **lines, int nbLines, int position _U_,
				   VisuData *dataObj _U_, GError **error)
{
  int token;
  float radius;
  VisuElement* ele[2];
  VisuPairData *data;
  gchar **tokens;

  g_return_val_if_fail(nbLines == 1, FALSE);

  /* Tokenize the line of values. */
  tokens = g_strsplit_set(g_strchug(lines[0]), " \n", MAX_LINE_LENGTH);
  token = 0;

  /* Get the two elements. */
  if (!configFileRead_elementFromTokens(tokens, &token, ele, 2, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }

  data = visuPairGet_linkFromId(ele[0], ele[1], 0);
  g_return_val_if_fail(data, FALSE);

  /* Read 1 float. */
  if (!configFileRead_floatFromTokens(tokens, &token, &radius, 1, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  radius = CLAMP(radius, RESOURCES_CYLINDER_RADIUS_MIN,
		 RESOURCES_CYLINDER_RADIUS_MAX);
  setCylinderRadius(data, radius);

  g_strfreev(tokens);
  return TRUE;
}
