/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "dumpToAscii.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <visu_tools.h>

/**
 * SECTION:dumpToAscii
 * @short_description: add an export capability of current positions.
 *
 * This provides a write routine to export V_Sim current
 * coordinates. It has several options to output or not hiddden nodes
 * or replicated nodes.
 */

static gboolean writeDataInAscii(FileFormat *format, const char* filename,
				 int width, int height, VisuData *dataObj,
				 guchar* imageData, GError **error,
				 voidDataFunc functionWait, gpointer data);
static gpointer waitData;
static voidDataFunc waitFunc;

DumpType* dumpToAscii_init()
{
  DumpType *ascii;
  char *typeASCII[] = {"*.ascii", (char*)0};
#define descrASCII _("Ascii file (current positions)")
  FileFormat* fmt;

  ascii = g_malloc(sizeof(DumpType));
  fmt = fileFormatNew(descrASCII, typeASCII);
  if (!fmt)
    {
      g_error("Can't initialize the ASCII dump module, aborting.\n");
    }

  ascii->bitmap = FALSE;
  ascii->fileType = fmt;
  ascii->writeFunc = writeDataInAscii;
  
  fileFormatAdd_propertyBoolean(fmt, "delete_hidden_nodes",
				_("Don't output hidden nodes"), FALSE);
  fileFormatAdd_propertyBoolean(fmt, "comment_hidden_nodes",
				_("Comment hidden nodes (if output)"), TRUE);
  fileFormatAdd_propertyBoolean(fmt, "expand_box",
				_("Expand the bounding box"), TRUE);

  waitData = (gpointer)0;
  waitFunc = (voidDataFunc)0;

  return ascii;
}

static gboolean writeDataInAscii(FileFormat *format, const char* filename,
				 int width _U_, int height _U_, VisuData *dataObj,
				 guchar* imageData _U_, GError **error,
				 voidDataFunc functionWait, gpointer data)
{
  FILE *file;
  gchar *nom, *prevFile;
  OpenGLView *view;
  char tmpChr;
  GList *prop;
  gchar* name;
  gboolean suppr, comment, expand;
  float xyz[3], ext[3];
  VisuDataIter iter;

  g_return_val_if_fail(error && !*error, FALSE);

  comment = TRUE;
  expand  = FALSE;
  suppr   = FALSE;
  prop = fileFormatGet_propertiesList(format);
  while (prop)
    {
      name = fileFormatGet_propertyName((FileFormatProperty*)(prop->data));
      if (strcmp(name, "comment_hidden_nodes") == 0)
	comment = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      if (strcmp(name, "expand_box") == 0)
	expand  = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      if (strcmp(name, "delete_hidden_nodes") == 0)
	suppr   = fileFormatGet_propertyBoolean((FileFormatProperty*)(prop->data));
      prop = g_list_next(prop);
    }

  waitData = data;
  waitFunc = functionWait;

  DBG_fprintf(stderr, "Dump ASCII : begin export of current positions...\n");

  file = fopen(filename, "w");
  if (!file)
    {
      *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_FILE,
			   _("Cannot open file (to write in)."));
      return FALSE;
    }                  

  nom = visuDataGet_file(dataObj, 0, (FileFormat**)0);
  if (nom)
    {
      prevFile = g_path_get_basename(nom);
      fprintf(file, "# V_Sim export to ascii from '%s'\n", prevFile);
      g_free(prevFile);
    }
  else
    {
      g_warning("Can't get the name of the file to export.");
      fprintf(file, "# V_Sim export to ascii\n");
    }

  view = visuDataGet_openGLView(dataObj);
  if (expand)
    visuDataGet_extension(dataObj, ext);
  else
    {
      ext[0] = 0.;
      ext[1] = 0.;
      ext[2] = 0.;
    }
  fprintf(file, "%15g %15g %15g\n", view->box->p2[0] * (1. + 2. * ext[0]),
	  view->box->p4[0] * (1. + 2. * ext[1]),
	  view->box->p4[1] * (1. + 2. * ext[1]));
  fprintf(file, "%15g %15g %15g\n", view->box->p5[0] * (1. + 2. * ext[2]),
	  view->box->p5[1] * (1. + 2. * ext[2]),
	  view->box->p5[2] * (1. + 2. * ext[2]));

  if (ext[0] != 0. || ext[1] != 0. || ext[2] != 0.)
    {
      fprintf(file, "# Box is expanded, previous box size:\n");
      fprintf(file, "# %15g %15g %15g\n", view->box->p2[0], view->box->p4[0],
	      view->box->p4[1]);
      fprintf(file, "# %15g %15g %15g\n", view->box->p5[0], view->box->p5[1],
	      view->box->p5[2]);
      fprintf(file, "# and box extension:\n");
      fprintf(file, "# %15g %15g %15g\n", ext[0], ext[1], ext[2]);
    }
  if (comment)
    {
      fprintf(file, "# Statistics are valid for all nodes (hidden or not).\n");
      fprintf(file, "# Hidden nodes are printed, but commented.\n");
    }
  visuDataIter_new(dataObj, &iter);
  fprintf(file, "# Box contains %d element(s).\n", iter.nElements);
  fprintf(file, "# Box contains %d nodes.\n", iter.nAllStoredNodes);
  for (visuDataIter_start(dataObj, &iter); iter.element;
       visuDataIter_nextElement(dataObj, &iter))
    fprintf(file, "#  | %d nodes for element '%s'.\n",
	    iter.nStoredNodes[iter.iElement], iter.element->name);
  if (suppr)
    fprintf(file, "# Hidden nodes have been suppressed.\n");

  for (visuDataIter_startNumber(dataObj, &iter); iter.node && iter.element;
       visuDataIter_nextNodeNumber(dataObj, &iter))
    {
      if ((iter.element->rendered && iter.node->rendered) || !comment)
	tmpChr = ' ';
      else
	tmpChr = '#';
      visuDataGet_nodePosition(dataObj, iter.node, xyz);
      if ((iter.element->rendered && iter.node->rendered) || !suppr)
	fprintf(file, "%c%15g %15g %15g %s\n", tmpChr,
		xyz[0] + ext[0] * view->box->p2[0] + ext[1] * view->box->p4[0] + ext[2] * view->box->p5[0] ,
		xyz[1] + ext[1] * view->box->p4[1] + ext[2] * view->box->p5[1],
		xyz[2] + ext[2] * view->box->p5[2], iter.element->name);
    }
      
  fclose(file);

  return TRUE;
}
