/*
 * This program converts the logfile of twlog to ADIF format
 */

/* for STANDALONE use */
//#define STANDALONE 

#ifndef STANDALONE
#include "twlog.h"
#endif

#include "adif.h"

/* 
 * If you just use 80, 40, etc. for the Band value, set FIXBAND
 * to 1.  This will produce the proper adif format (80M and 40M)
 * when converting to adif format
 */
#define FIXBAND 0

FILE *fpin, *fpout;

/* For STANDALONE:
 * Uncomment the #define STANDALONE above
 * Compile it with -  gcc adif.c -o adif
 * Put adif and your log file in the same directory
 * File names must be logfile (twlog format) and adiflog (adif format)
 * Use "adif a" to convert logfile to adiflog
 * or "adif l" to convert adiflog to logfile
 */
#ifdef STANDALONE
int main (int argc, char *argv[])
{
   int rtn;

   if (argc < 2)
   {
      fprintf (stderr, "Error:  %s a | l\n", argv[0]);
      fprintf (stderr, "Error:  Use \"%s a\" to convert to adif\n", argv[0]);
      fprintf (stderr, "Error:  or \"%s l\" to convert to logfile\n", argv[0]);
      return 0;
   }

   switch (argv[1][0])
   {
      case 'a':
         fprintf (stderr, "Do to_adif\n");
         rtn = to_adif("logfile", "adiflog");
         break;

      case 'l':
         fprintf (stderr, "Do de_adif\n");
         rtn = de_adif ("adiflog", "logfile");
         break;

      default:
         fprintf (stderr, "Error:  The command line argument "
                  "must be 'a' or 'l'\n");
         return 1;
   }

   if (rtn == 0)
      return 0;
   else
      return 1;
}
#endif


/*
 * to_adif - Converts logfile (twlog format) to adif format.
 * Results are in outlog.
 */
int to_adif (char *inlog, char *outlog)
{

   /* open input file */
   if ((fpin = open_input (inlog)) == NULL)
   {
      printf ("to_adif: open_input failed\n");
      return -1;
   }

   /* open output file*/
   if ((fpout = open_output (outlog)) == NULL)
   {
      printf ("to_adif: open_output failed\n");
      return -1;
   }

   /* process the logfile to adif */
   make_adif (fpin, fpout);

   /* close things */
   fclose (fpin);
   fclose (fpout);
   return 0;
}


/*
 * de_adif - Convert inlog (adif format) to twlog format.
 * Results are in outlog.
 */
int de_adif (char inlog[], char outlog[])
{

   /* open input file */
   if ((fpin = open_input (inlog)) == NULL)
   {
      printf ("de_adif: open_input failed\n");
      return -1;
   }

   /* open output file */
   if ((fpout = open_output (outlog)) == NULL)
   {
      printf ("de_adif: open_output failed\n");
      return -1;
   }

   /* call make_logfile() */
   make_logfile(fpin, fpout);
   fclose (fpin);
   fclose (fpout);
   return 0;
}


/*
 * open_input ()
 * opens input log
 */
FILE *open_input (char *in)
{
   FILE *fp;

   if ((fp = fopen (in, "r")) == NULL)
   {
      fprintf (stderr, "Failed to open %s\n", in);
      return NULL;
   }
   return fp;
}


/*
 * open_output
 * opens output log
 */
FILE *open_output (char *out)
{
   FILE *fp;

   if ((fp = fopen (out, "w")) == NULL)
   {
      fprintf (stderr, "Failed to open %s\n", out);
      return NULL;
   }
   return fp;
}


/*
 * make_adif
 */
void make_adif (FILE *fpin, FILE *fpout)
{
   int i;
   char buf[300];
   char mo[3];
   int qso_date_l, time_on_l, call_l, band_l, mode_l, power_l;
   int hisRST_l, myRST_l, time_off_l, comment_l;

   char qso_date[L_QSO_DATE];
   char time_on[L_TIME_ON];
   char call[L_CALL];
   char band[L_BAND];
   char mode[L_MODE];
   char power[L_POWER];
   char hisRST[L_HISRST];
   char myRST[L_MYRST];
   char time_off[L_TIME_OFF];
   char comment[L_COMMENT];

   /* get data and reformat */
   while (fgets (buf, 300, fpin) != NULL)
   {
      /* special handling for date */
      strncpy (qso_date, &buf[7], 4);   /* copy YYYY to date[] */
      qso_date[4] = '\0';

      /* convert Mon to MM */
      for (i = 1; i <= 12; i++)
      {
         if (strncmp (month[i], &buf[3], 3) == 0)
         {
            sprintf (mo, "%02d", i);
            strncat (&qso_date[4], mo, 2);      /* cat MM to date[] */
         }
      }
      qso_date[6] = buf[0];
      qso_date[7] = buf[1];
      qso_date[8] = '\0';                      /* QSO_DATE */
      qso_date_l = 8;                          /* always 8 */

      /* special handling for time on */
      strncpy (time_on, &buf[15], L_TIME_ON);
      time_on[4] = '\0'; 
      time_on_l = 4;

      /* get call */
      call_l = getcols (buf, 25, call, L_CALL);            /* CALL */
      /* get band */
      band_l = getcols (buf, 40, band, L_BAND);            /* BAND */
      #if FIXBAND == 1
      strcat (band, "M");
      band_l++;
      #endif
      /* get mode */
      mode_l = getcols (buf, 55, mode, L_MODE);            /* MODE */
      /* get power */
      power_l = getcols (buf, 65, power, L_POWER);         /* TX_PWR */
      /* get hisRST */
      hisRST_l = getcols (buf, 75, hisRST, L_HISRST);      /* RST_SENT */
      /* get myRST */
      myRST_l = getcols (buf, 85, myRST, L_MYRST);         /* RST_RCVD */
      /* get time_off */

      /* special handling for time off */
      strncpy (time_off, &buf[95], L_TIME_OFF);            /* TIME_OFF */
      time_off[4] = '\0'; 
      time_off_l = 4;

      /* handle comment special */
      strncpy (comment, &buf[105], L_COMMENT);             /* COMMENT */
      comment_l = strlen (comment) - 1;
      comment[comment_l] = '\0';

      fprintf (fpout,"<QSO_DATE:%d>%s<TIME_ON:%d>%s<CALL:%d>%s<BAND:%d>%s"
              "<MODE:%d>%s\n<TX_PWR:%d>%s<RST_SENT:%d>%s<RST_RCVD:%d>%s"
              "<TIME_OFF:%d>%s\n<COMMENT:%d>%s<EOR>\n",
              qso_date_l, qso_date, time_on_l, time_on, call_l, call,
              band_l,  band, mode_l, mode, power_l, power, hisRST_l, hisRST,
              myRST_l, myRST, time_off_l, time_off, comment_l, comment);
   }
   return;
}


/*
 * getcols - passed buff address and index to start of column
 */
int getcols(char *buf, int i, char *col, int len)
{
   int j;

   for (j=0; j<len; j++)
      col[j] = ' ';

   strncpy (col, &buf[i], len-1);

   j = len - 1;
   while (col[j] == ' ')
   {
      col[j] = '\0';
      j--;
   }

   return strlen (col);
}


/*
 * make_logfile
 */
void make_logfile(FILE *fpin, FILE *fpout)
{
#define NUMFIELDS 15
   static char *fields[] = {
                     "<QSO_DATE:", "<TIME_ON:", "<CALL:", "<BAND:", "<MODE:",
                     "<TX_PWR:", "<RST_SENT:", "<RST_RCVD:", "<TIME_OFF:",
                     "<COMMENT:", "QTH:", "STATE:", "<NAME:", "NOTES:",
                     "FREQ:"
                    };

   char abox[10][200];
   char buf[1000];
   char *ptr;
   char mon[4];
   char lenstr[5];
   int  len, mnum;
   int  i, j, n;

   for (i=0; i<NUMFIELDS; i++)
   {
      abox[i][0] = '\0';
   }
   buf[0] = '\0';

   while (fgets (buf + strlen(buf), 1000, fpin) != NULL)
   {
      if (buf[0] != '<')          /* skip any header or blank lines */
      {
         continue;
      }

      do_case (buf);              /* convert fields to all caps */

      if (strstr (buf, "<EOR>") == NULL)
      {
         continue;
      }

      for (j=0; j<NUMFIELDS; j++)               /* scan for fields */
      {
         if ((ptr = strstr (buf, fields[j])) != NULL)
         {
            ptr = ptr + strlen (fields[j]);     /* point to field size */

            for (n=0; *ptr != '>'; n++,ptr++)   /* pick out the len field */
            {                                   /* and convert to int */
               lenstr[n] = *ptr;
            }
            lenstr[n] = '\n';
            ptr++;
            len = atoi (lenstr);

            switch (j)                          /* switch on field name  */
            {
               /* date */
               case 0:
                  mon[0] = ptr[4];
                  mon[1] = ptr[5];
                  mon[2] = '\0';
                  mnum = atoi (mon);

                  // TJW sprintf (abox[j], "%c%c %s %c%c%c%c\0",
                  sprintf (abox[j], "%c%c %s %c%c%c%c",
                    ptr[6], ptr[7], month[mnum],
                    ptr[0], ptr[1], ptr[2], ptr[3]);
                  break;

               /* time_on and time_off */
               case 1:
               case 8:
                  strncpy (abox[j], ptr, 4);
                  abox[j][4] = '\0';
                  strcat (abox[j], " GMT");
                  break;

               /* band and freq - freq overrides band */
               case 3:
               case 14:
                  strncpy (abox[3], ptr, len);
                  abox[3][len] = '\0';
                  break;

               /* comment, qth, state, name, notes */
               /* these go to the logfile comment field */
               case 9:
               case 10:
               case 11:
               case 12:
               case 13:
                  if (strlen (abox[9]) > 0)  /* avoid creep of comments len */
                  {
                     strcat (abox[9], " ");
                  }
                  strncat (abox[9], ptr, len);
                  break;

               /* and for the everything else */
               default:
                  strncat (abox[j], ptr, len);
                  break;
            }
         }
      }

      fprintf (fpout, "%-15s%-10s%-15s%-15s%-10s%-10s%-10s%-10s%-10s%-s\n",
         abox[0], abox[1], abox[2], abox[3], abox[4],
         abox[5], abox[6], abox[7], abox[8], abox[9]);

      for (i=0; i<NUMFIELDS; i++)
         abox[i][0] = '\0';
      buf[0] = '\0';
   }
}


/*
 * do_case - convert all fields to upper case
 */
void do_case (char *ptr)
{
   int i, doit;

   i = 0;
   doit = 0;
   while (ptr[i] != '\0')      /* set all flag fields to upper case */
   {
      if (doit == 1)
         ptr[i] = toupper (ptr[i]);
      if (ptr[i] == '<')
         doit = 1;
      if (ptr[i] == '>')
         doit = 0;
      i++;
   }
}
