# -*- python -*-

# Copyright 2005-2008, W. Martin Borgert <debacle@debian.org>
# GPL v3 or later
# Complete BNF: http://people.debian.org/~debacle/ttcn-bnf.html

import collections
import sys
import pyparsing as pp
pp.ParserElement.enablePackrat()

# Thanks to Paul McGuire for pyparsing and this progress indicator
def showProgress(instring, loc, toks):
    sys.stderr.write("\r%6.2f%%" % (loc*100.0/len(instring)))
    sys.stderr.flush()

def checkOctPair(s, loc, toks):
    flag = False
    for c in toks[0][1:-2]:
        if c in "0123456789abcdefABCDEF":
            flag = not flag
    if flag:
        raise ValueError, "Octetstring not well-formed"

known_modules = {}

def register_gmi(s,l,t):
    known_modules[str(t)] = 0

def check_gmi(s, l, t):
    if not str(t) in known_modules.keys():
        raise pp.ParseException(s, l, "GlobalModuleId unknown")

# This is inspired by the original pyparsing.Word class
class ppWord(pp.Word):
    def __init__(self, initChars, bodyChars):
        pp.Word.__init__(self, initChars, bodyChars)
        self.keywords = {}

    def parseImpl(self, instring, loc, doActions=True):
        if not(instring[loc] in self.initChars):
            exc = self.myException
            exc.loc = loc
            exc.pstr = instring
            raise exc
        start = loc
        loc += 1
        bodychars = self.bodyChars
        maxloc = start + self.maxLen
        maxloc = min(maxloc, len(instring))
        while loc < maxloc and instring[loc] in bodychars:
            loc += 1

        if loc - start < self.minLen:
            exc = self.myException
            exc.loc = loc
            exc.pstr = instring
            raise exc

        if instring[start:loc] in self.keywords:
            exc = self.myException
            exc.loc = loc
            exc.pstr = instring
            raise exc
        return loc, [instring[start:loc]]

_Keywords = {
    # [2]
    "module": "TTCN3ModuleKeyword",
    # [8]
    "language": "LanguageKeyword",
    # [14]
    "type": "TypeDefKeyword",
    # [17]
    "record": "RecordKeyword",
    # [31]
    "optional": "OptionalKeyword",
    # [33]
    "union": "UnionKeyword",
    # [37]
    "set": "SetKeyword",
    # [39]
    "of": "OfKeyword",
    # [43]
    "enumerated": "EnumKeyword",
    # [55]
    "length": "LengthKeyword",
    # [59]
    "port": "PortKeyword",
    # [65]
    "message": "MessageKeyword",
    # [67]
    "all": "AllKeyword",
    # [70]
    "procedure": "ProcedureKeyword",
    # [75]
    "mixed": "MixedKeyword",
    # [80]
    "component": "ComponentKeyword",
    # [81]
    "extends": "ExtendsKeyword",
    # [92]
    "const": "ConstKeyword",
    # [96]
    "template": "TemplateKeyword",
    # [99]
    "modifies": "ModifiesKeyword",
    # [126]
    "pattern": "PatternKeyword",
    # [128]
    "complement": "ComplementKeyword",
    # [131]
    "subset": "SubsetKeyword",
    # [133]
    "superset": "SupersetKeyword",
    # [135] (not in the reserved words list of the standard)
    "permutation": "PermutationKeyword",
    # [142]
    "ifpresent": "IfPresentKeyword",
    # [146]
    "infinity": "InfinityKeyword",
    # [156]
    "match": "MatchKeyword",
    # [158]
    "valueof": "ValueofKeyword",
    # [160]
    "function": "FunctionKeyword",
    # [165]
    "return": "ReturnKeyword",
    # [167]
    "runs": "RunsKeyword",
    # [168]
    "on": "OnKeyword",
    # [169]
    "mtc": "MTCKeyword",
    # [182]
    "signature": "SignatureKeyword",
    # [187]
    "exception": "ExceptionKeyword",
    # [189]
    "noblock": "NoBlockKeyword",
    # [192]
    "testcase": "TestcaseKeyword",
    # [198]
    "system": "SystemKeyword",
    # [200]
    "execute": "ExecuteKeyword",
    # [205]
    "altstep": "AltstepKeyword",
    # [213]
    "import": "ImportKeyword",
    # [216]
    "except": "ExceptKeyword",
    # [231] (not in the reserved words list of the standard)
    "recursive": "RecursiveKeyword",
    # [266]
    "group": "GroupKeyword",
    # [269]
    "external": "ExtKeyword",
    # [275]
    "modulepar": "ModuleParKeyword",
    # [282]
    "control": "ControlKeyword",
    # [291]
    "var": "VarKeyword",
    # [300]
    "timer": "TimerKeyword",
    # [308]
    "self": "SelfOp",
    # [313]
    "done": "DoneKeyword",
    # [314]
    "killed": "KilledKeyword",
    # [316]
    "running": "RunningKeyword",
    # [318]
    "create": "CreateKeyword",
    # [319]
    "alive": "AliveKeyword",
    # [321]
    "connect": "ConnectKeyword",
    # [330]
    "disconnect": "DisconnectKeyword",
    # [332]
    "map": "MapKeyword",
    # [334]
    "unmap": "UnmapKeyword",
    # [336]
    "start": "StartKeyword",
    # [341]
    "kill": "KillKeyword",
    # [346]
    "send": "SendOpKeyword",
    # [350]
    "to": "ToKeyword",
    # [354]
    "call": "CallOpKeyword",
    # [357]
    "nowait": "NowaitKeyword",
    # [365]
    "reply": "ReplyKeyword",
    # [369]
    "raise": "RaiseKeyword",
    # [373]
    "receive": "ReceiveOpKeyword",
    # [376]
    "from": "FromKeyword",
    # [380]
    "value": "ValueKeyword",
    # [382]
    "sender": "SenderKeyword",
    # [385]
    "trigger": "TriggerOpKeyword",
    # [388]
    "getcall": "GetCallOpKeyword",
    # [392]
    "param": "ParamKeyword",
    # [403]
    "getreply": "GetReplyOpKeyword",
    # [407]
    "check": "CheckOpKeyword",
    # [414]
    "catch": "CatchOpKeyword",
    # [419]
    "clear": "ClearOpKeyword",
    # [424]
    "stop": "StopKeyword",
    # [425]
    "any": "AnyKeyword",
    # [432]
    "read": "ReadKeyword",
    # [436]
    "timeout": "TimeoutKeyword",
    # [439]
    "bitstring": "BitStringKeyword",
    # [440]
    "boolean": "BooleanKeyword",
    # [441]
    "integer": "IntegerKeyword",
    # [442]
    "octetstring": "OctetStringKeyword",
    # [443]
    "hexstring": "HexStringKeyword",
    # [444]
    "verdicttype": "VerdictTypeKeyword",
    # [445]
    "float": "FloatKeyword",
    # [446]
    "address": "AddressKeyword",
    # [447]
    "default": "DefaultKeyword",
    # [448]
    "anytype": "AnyTypeKeyword",
    # [449]
    "charstring": "CharStringKeyword",
    # [451]
    "universal": "UniversalKeyword",
    # [469]
    "char": "CharKeyword",
    # [499]
    "null": "AddressValue",
    # [501]
    "omit": "OmitKeyword",
    # [502]
    "in": "InParKeyword",
    # [503]
    "out": "OutParKeyword",
    # [504]
    "inout": "InOutParKeyword",
    # [514]
    "with": "WithKeyword",
    # [519]
    "encode": "EncodeKeyword",
    # [520]
    "variant": "VariantKeyword",
    # [521]
    "display": "DisplayKeyword",
    # [522]
    "extension": "ExtensionKeyword",
    # [523]
    "override": "OverrideKeyword",
    # [534]
    "setverdict": "SetVerdictKeyword",
    # [535]
    "getverdict": "GetLocalVerdict",
    # [537]
    "action": "ActionKeyword",
    # [541]
    "alt": "AltKeyword",
    # [548]
    "interleave": "InterleavedKeyword",
    # [554]
    "label": "LabelKeyword",
    # [557]
    "goto": "GotoKeyword",
    # [558]
    "repeat": "RepeatStatement",
    # [560]
    "activate": "ActivateKeyword",
    # [562]
    "deactivate": "DeactivateKeyword",
    # [605]
    "log": "LogKeyword",
    # [609]
    "for": "ForKeyword",
    # [614]
    "while": "WhileKeyword",
    # [616]
    "do": "DoKeyword",
    # [618]
    "if": "IfKeyword",
    # [620]
    "else": "ElseKeyword",
    # [623]
    "select": "SelectKeyword",
    # [626]
    "case": "CaseKeyword",
    }

def keywords():
    global _Keywords
    l = _Keywords.keys()
    l.extend("and xor or".split(" "))
    l.extend("and4b xor4b or4b not4b".split(" "))
    l.extend("mod rem".split(" "))
    l.extend("pass fail inconc none error".split(" "))
    l.extend("true false".split(" "))
    l.extend(
        ("bit2hex bit2int bit2oct bit2str char2int char2oct float2int " \
        + "hex2bit hex2int hex2oct hex2str int2bit int2char int2float " \
        + "int2hex int2oct int2str int2unichar ischosen ispresent lengthof " \
        + "oct2bit oct2char oct2hex oct2int oct2str regexp replace rnd " \
        + "sizeof sizeoftype str2float str2int str2oct substr unichar2int"
        ).split(" "))
    l.sort()
    for kw in l:
        print kw,
    print

kwTally = collections.defaultdict(lambda : [0, 0, 0])

def tallyKeyword(kw):
    def test(*args):
        kwTally[kw][0] += 1
    def succ(*args):
        kwTally[kw][2] += 1
    def fail(*args):
        kwTally[kw][1] += 1
    return test, succ, fail

def ParseTally():
    return kwTally
    
def BNF(progress, debug):
    global _Keywords
    progressaction = ""
    if progress:
        progressaction = ".addParseAction(showProgress)"
    for kw in _Keywords.keys():
        exec '%s = pp.Keyword("%s")%s("_")' % (_Keywords[kw], kw, progressaction)
    # some symbols
    # [378]
    PortRedirectSymbol = pp.Literal("->").suppress()
    # [627]
    Dot = pp.Literal(".").suppress()
    # [628]
    Dash = pp.Literal("-")
    # [629]
    Minus = pp.Group(Dash)("Minus")
    # [630]
    SemiColon = pp.Literal(";").suppress()
    # [631]
    Colon = pp.Literal(":").suppress()
    # [632]
    Underscore = pp.Literal("_")
    # [633]
    AssignmentChar = pp.Literal(":=").suppress()
    # [116]
    NotUsedSymbol = (Dash)("NotUsed")
    LBrace = pp.Literal("{").suppress()
    RBrace = pp.Literal("}").suppress()
    LParen = pp.Literal("(").suppress()
    RParen = pp.Literal(")").suppress()
    LBrack = pp.Literal("[").suppress()
    RBrack = pp.Literal("]").suppress()
    ToSymbol = pp.Literal("..").suppress()
    Comma = pp.Literal(",").suppress()

    # Identifiers
    # [492]
    Identifier = ppWord(pp.alphas, pp.alphanums + "_")
    Identifier.keywords = _Keywords
    # [6]
    ModuleIdentifier = (Identifier)("ModuleIdentifier")
    # [19]
    StructTypeIdentifier = (Identifier)("StructTypeIdentifier")
    # [30]
    StructFieldIdentifier = (Identifier)("StructFieldIdentifier")
    # [44]
    EnumTypeIdentifier = (Identifier)("EnumTypeIdentifier")
    # [47]
    EnumerationIdentifier = (Identifier)("EnumerationIdentifier")
    # [49]
    SubTypeIdentifier = (Identifier)("SubTypeIdentifier")
    # [60]
    PortTypeIdentifier = (Identifier)("PortTypeIdentifier")
    # [83]
    ComponentTypeIdentifier = (Identifier)("ComponentTypeIdentifier")
    # [88]
    PortIdentifier = (Identifier)("PortIdentifier")
    # [93]
    ConstIdentifier = (Identifier)("ConstIdentifier")
    # [97]
    TemplateIdentifier = (Identifier)("TemplateIdentifier")
    # [161]
    FunctionIdentifier = (Identifier)("FunctionIdentifier")
    # [183]
    SignatureIdentifier = (Identifier)("SignatureIdentifier")
    # [193]
    TestcaseIdentifier = (Identifier)("TestcaseIdentifier")
    # [206]
    AltstepIdentifier = (Identifier)("AltstepIdentifier")
    # [267]
    GroupIdentifier = (Identifier)("GroupIdentifier")
    # [270]
    ExtFunctionIdentifier = (Identifier)("ExtFunctionIdentifier")
    # [273]
    ExtConstIdentifier = (Identifier)("ExtConstIdentifier")
    # [280]
    ModuleParIdentifier = (Identifier)("ModuleParIdentifier")
    # [292]
    VarIdentifier = (Identifier)("VarIdentifier")
    # [301]
    TimerIdentifier = (Identifier)("TimerIdentifier")
    # [506]
    ValueParIdentifier = (Identifier)("ValueParIdentifier")
    # [508]
    PortParIdentifier = Identifier
    # [510]
    TimerParIdentifier = (Identifier)("TimerParIdentifier")
    # [512]
    TemplateParIdentifier = (Identifier)("TemplateParIdentifier")
    # [555]
    LabelIdentifier = (Identifier)("LabelIdentifier")

    # Grammar
    # [626] CaseKeyword ::= "case"
    # [625]
    TemplateInstance = pp.Forward()
    StatementBlock = pp.Forward()
    SelectCase = pp.Group(CaseKeyword + (
        (LParen \
         + pp.delimitedList(TemplateInstance) \
         + RParen) \
        | ElseKeyword) + StatementBlock)("SelectCase")
    # [624]
    SelectCaseBody = pp.Group(LBrace + pp.OneOrMore(SelectCase) \
                              + RBrace)("SelectCaseBody")
    # [623] SelectKeyword ::= "select"
    # [622]
    SingleExpression = pp.Forward()
    SelectCaseConstruct = pp.Group(SelectKeyword + LParen \
                                   + SingleExpression + RParen \
                                   + SelectCaseBody)
    # [621]
    ElseClause = pp.Group(ElseKeyword + StatementBlock)("ElseClause")
    # [620] ElseKeyword ::= "else"
    # [619]
    BooleanExpression = pp.Forward()
    ElseIfClause = pp.Group(ElseKeyword + IfKeyword + LParen \
                            + BooleanExpression + RParen \
                            + StatementBlock)("ElseIfClause")
    # [618] IfKeyword ::= "if"
    # [617]
    ConditionalConstruct = pp.Group(
        IfKeyword + LParen \
        + BooleanExpression + RParen \
        + StatementBlock \
        + pp.ZeroOrMore(ElseIfClause) \
        + pp.Optional(ElseClause))("ConditionalConstruct")
    # [616] DoKeyword ::= "do"
    # [615]
    DoWhileStatement = pp.Group(DoKeyword + StatementBlock + WhileKeyword \
                                + LParen + BooleanExpression \
                                + RParen)("DoWhileStatement")
    # [614] WhileKeyword ::= "while"
    # [613]
    WhileStatement = pp.Group(WhileKeyword + LParen \
                              + BooleanExpression + RParen \
                              + StatementBlock)("WhileStatement")
    # [612]
    Assignment = pp.Forward()
    Step = Assignment
    # [611]
    Final = BooleanExpression
    # [610]
    VarInstance = pp.Forward()
    Initial = (VarInstance | Assignment)
    # [609] ForKeyword ::= "for"
    # [608]
    ForStatement = pp.Group(ForKeyword + LParen + Initial \
                            + SemiColon + Final + SemiColon + Step \
                            + RParen + StatementBlock)("ForStatement")
    # [607]
    LoopConstruct = pp.Group(
        ForStatement | WhileStatement | DoWhileStatement)("LoopConstruct")
    # [606]
    FreeText = (pp.dblQuotedString)("FreeText")
    LogItem = pp.Group(FreeText | TemplateInstance)("LogItem")
    # [605] LogKeyword ::= "log"
    # [604]
    LogStatement = pp.Group(
        LogKeyword + LParen \
        + pp.delimitedList(LogItem) + RParen)("LogStatement")
    # [603]
    #ShiftOp = pp.oneOf("<< >> <@ @>")
    ShiftOp = pp.Group(pp.Literal("<<") | pp.Literal(">>") \
                       | pp.Literal("<@") | pp.Literal("@>"))("ShiftOp")
    # [602]
    StringOp = pp.Literal("&")("StringOp")
    # [601]
    EqualOp = pp.oneOf("== !=")("EqualOp")
    # [600]
    RelOp = pp.oneOf("<= >= > <")("RelOp")
    # [599]
    UnaryOp = pp.oneOf("+ -")("UnaryOp")
    # [598]
    MultiplyOp = pp.oneOf("* / mod rem")("MultiplyOp")
    # [597]
    AddOp = pp.oneOf("+ - &")("AddOp") # & = StringOp
    # [596]
    TestcaseInstance = pp.Forward()
    TemplateOps = pp.Forward()
    FunctionInstance = pp.Forward()
    ConfigurationOps = pp.Forward()
    VerdictOps = pp.Forward()
    ActivateOp = pp.Forward()
    TimerOps = pp.Forward()
    OpCall = pp.Group(ConfigurationOps | ActivateOp | VerdictOps \
                      | FunctionInstance \
                      | TestcaseInstance | TemplateOps \
                      | TimerOps)("OpCall")
    # [595]
    TypeDefIdentifier = pp.Forward()
    ArrayOrBitRef = pp.Forward()
    # TODO: Identifier == StructFieldIdentifier or TypeDefIdentifier
    ExtendedFieldReference = pp.OneOrMore(
        (Dot + Identifier("Identifier"))
        | ArrayOrBitRef)("ExtendedFieldReference")
    # [594]
    Value = pp.Forward()
    _single_exp_in_primary = pp.Group(
        LParen + SingleExpression \
        + RParen)("_SingleExpression")
    Primary = pp.Group(OpCall \
                       | _single_exp_in_primary \
                       | Value)("Primary")
    # [593]
    UnaryExpression = pp.Group(pp.Optional(UnaryOp) \
                               + Primary)("UnaryExpression")
    # [592]
    MulExpression = pp.Group(
        UnaryExpression + pp.ZeroOrMore(MultiplyOp \
                                        + UnaryExpression))("MulExpression")
    # [591]
    AddExpression = pp.Group(
        MulExpression + pp.ZeroOrMore(AddOp + MulExpression))("AddExpression")
    # [590]
    BitNotExpression = pp.Group(
        pp.Optional("not4b") + AddExpression)("BitNotExpression")
    # [589] BitAndExpression ::= BitNotExpression { "and4b" BitNotExpression }
    # [588] BitXorExpression ::= BitAndExpression { "xor4b" BitAndExpression }
    # [587] BitOrExpression ::= BitXorExpression { "or4b" BitXorExpression }
    _BitOpExpression = pp.Group(BitNotExpression \
        + pp.ZeroOrMore(pp.oneOf("and4b xor4b or4b") \
                        + BitNotExpression))("BitOpExpression")
    # [586]
    ShiftExpression = pp.Group(
        _BitOpExpression \
        + pp.ZeroOrMore(ShiftOp + _BitOpExpression))("ShiftExpression")
    # [585]
    RelExpression = pp.Group(
        ShiftExpression \
        + pp.Optional((EqualOp | RelOp) + ShiftExpression))("RelExpression")
    # [584] EqualExpression ::= RelExpression { EqualOp RelExpression }
    # [583]
    NotExpression = pp.Group(
        pp.Optional(pp.Literal("not")("not")) + RelExpression)("NotExpression")
    # [582] AndExpression ::= NotExpression { "and" NotExpression }
    # [581] XorExpression ::= AndExpression { "xor" AndExpression }
    # [580]
    _logicalOp = pp.oneOf("and xor or")("logicalOp")
    SingleExpression << pp.Group(NotExpression \
        + pp.ZeroOrMore(_logicalOp + NotExpression))("SingleExpression")
    # [579]
    VariableRef = pp.Forward()
    Expression = pp.Forward()
    TemplateBody = pp.Forward()
    Assignment << pp.Group(VariableRef + AssignmentChar \
                           + (TemplateBody | Expression))("Assignment")
    # [578]
    ConstantExpression = pp.Forward()
    ArrayElementConstExpressionList = pp.delimitedList(ConstantExpression)
    # [577]
    ArrayConstExpression = pp.Group(
        LBrace + pp.Optional(ArrayElementConstExpressionList) \
        + RBrace)
    # [576]
    FieldReference = pp.Forward()
    FieldConstExpressionSpec = pp.Group(FieldReference + AssignmentChar \
                                        + ConstantExpression)
    # [575]
    FieldConstExpressionList = pp.Group(
        LBrace \
        + pp.delimitedList(FieldConstExpressionSpec) \
        + RBrace)
    # [574]
    CompoundConstExpression = (FieldConstExpressionList \
                               | ArrayConstExpression)
    # [573]
    BooleanExpression << SingleExpression
    # [572]
    SingleConstExpression = SingleExpression
    # [571]
    ConstantExpression << pp.Group(
        SingleConstExpression \
        | CompoundConstExpression)("ConstantExpression")
    # [570]
    NotUsedOrExpression = pp.Group(
        NotUsedSymbol | Expression)("NotUsedOrExpression")
    # [569]
    ArrayElementExpressionList = pp.Group(
        pp.delimitedList(NotUsedOrExpression))("ArrayElementExpressionList")
    # [568]
    ArrayExpression = pp.Group(LBrace \
                               + pp.Optional(ArrayElementExpressionList) \
                               + RBrace)("ArrayExpression")
    # [567]
    FieldExpressionSpec = pp.Group(
        FieldReference + AssignmentChar \
        + NotUsedOrExpression)("FieldExpressionSpec")
    # [566]
    FieldExpressionList = pp.Group(LBrace \
                                   + pp.delimitedList(FieldExpressionSpec) \
                                   + RBrace)("FieldExpressionList")
    # [565]
    CompoundExpression = pp.Group(
        FieldExpressionList | ArrayExpression)("CompoundExpression")
    # [564]
    Expression << pp.Group(
        SingleExpression | CompoundExpression)("Expression")
    # [563]
    BasicStatements = pp.Group(
        LogStatement | LoopConstruct \
        | ConditionalConstruct | SelectCaseConstruct \
        | Assignment)("BasicStatements")
    # [562] DeactivateKeyword ::= "deactivate"
    # [561]
    ComponentOrDefaultReference = pp.Forward()
    DeactivateStatement = pp.Group(
        DeactivateKeyword \
        + pp.Optional(LParen \
                      + ComponentOrDefaultReference \
                      + RParen))("DeactivateStatement")
    # [560] ActivateKeyword ::= "activate"
    # [559]
    AltstepInstance = pp.Forward()
    ActivateOp << pp.Group(ActivateKeyword + LParen \
                          + AltstepInstance + RParen)("ActivateOp")
    # [558] RepeatStatement ::= "repeat"
    # [557] GotoKeyword ::= "goto"
    # [556]
    GotoStatement = pp.Group(GotoKeyword + LabelIdentifier)("GotoStatement")
    # [555] LabelIdentifier ::= Identifier
    # [554] LabelKeyword ::= "label"
    # [553]
    LabelStatement = pp.Group(LabelKeyword + LabelIdentifier)("LabelStatement")
    # [552]
    InterleavedAction = pp.Group(StatementBlock)("InterleavedAction")
    # [551]
    GuardOp = pp.Forward()
    InterleavedGuard = pp.Group(LBrack + RBrack + GuardOp)("InterleavedGuard")
    # [550]
    InterleavedGuardElement = pp.Group(
        InterleavedGuard + InterleavedAction)("InterleavedGuardElement")
    # [549]
    _opt_semicolon = pp.Optional(SemiColon).suppress()
    InterleavedGuardList = pp.Group(
        pp.OneOrMore(InterleavedGuardElement \
                     + _opt_semicolon))("InterleavedGuardList")
    # [548] InterleavedKeyword ::= "interleave"
    # [547]
    InterleavedConstruct = pp.Group(
        InterleavedKeyword + LBrace \
        + InterleavedGuardList + RBrace)("InterleavedConstruct")
    # [546]
    TimeoutStatement = pp.Forward()
    ReceiveStatement = pp.Forward()
    TriggerStatement = pp.Forward()
    GetCallStatement = pp.Forward()
    CatchStatement = pp.Forward()
    CheckStatement = pp.Forward()
    GetReplyStatement = pp.Forward()
    DoneStatement = pp.Forward()
    KilledStatement = pp.Forward()
    GuardOp << pp.Group(
        TimeoutStatement | ReceiveStatement | TriggerStatement \
        | GetCallStatement | CatchStatement | CheckStatement \
        | GetReplyStatement | DoneStatement | KilledStatement)("GuardOp")
    # [545]
    AltGuardChar = pp.Group(LBrack \
                            + pp.Optional(BooleanExpression) \
                            + RBrack)("AltGuardChar")
    # [544]
    ElseStatement = pp.Group(LBrack + ElseKeyword + RBrack \
                             + StatementBlock)("ElseStatement")
    # [543]
    GuardStatement = pp.Group(AltGuardChar \
                              + ((AltstepInstance \
                                  + pp.Optional(StatementBlock)) \
                                 | (GuardOp \
                                    + StatementBlock)))("GuardStatement")
    # [542]
    AltGuardList = pp.Group(pp.ZeroOrMore(GuardStatement \
                                          + _opt_semicolon) \
                            + pp.Optional(ElseStatement \
                                          + _opt_semicolon))("AltGuardList")
    # [541] AltKeyword ::= "alt"
    # [540]
    AltConstruct = pp.Group(AltKeyword + LBrace + AltGuardList \
                            + RBrace)("AltConstruct")
    # [539]
    ReturnStatement = pp.Group(
        ReturnKeyword + pp.Optional(Expression))("ReturnStatement")
    # [538]
    ActionText = pp.Group(FreeText | Expression)("ActionText")
    # [537] ActionKeyword ::= "action"
    # [536]
    SUTStatements = pp.Group(ActionKeyword + LParen \
                             + pp.Optional(ActionText) \
                             + pp.ZeroOrMore(StringOp + ActionText) \
                             + RParen)("SUTStatements")
    # [535] GetLocalVerdict ::= "getverdict"
    # [534] SetVerdictKeyword ::= "setverdict"
    # [533]
    SetLocalVerdict = pp.Group(
        SetVerdictKeyword + LParen \
        + SingleExpression + RParen)("SetLocalVerdict")
    # [532]
    VerdictOps << pp.Group(GetLocalVerdict)("VerdictOps")
    # [531]
    VerdictStatements = pp.Group(SetLocalVerdict)("VerdictStatements")
    # [530]
    BehaviourStatements = pp.Group(
        TestcaseInstance | FunctionInstance \
        | ReturnStatement | AltConstruct \
        | InterleavedConstruct | LabelStatement \
        | GotoStatement | RepeatStatement \
        | DeactivateStatement | AltstepInstance \
        | ActivateOp)("BehaviourStatements")
    # [529]
    AttribSpec = FreeText("AttribSpec")
    # [528]
    GroupRefList = pp.Forward()
    TypeRefList = pp.Forward()
    TemplateRefList = pp.Forward()
    ConstRefList = pp.Forward()
    AltstepRefList = pp.Forward()
    TestcaseRefList = pp.Forward()
    FunctionRefList = pp.Forward()
    SignatureRefList = pp.Forward()
    ModuleParRefList = pp.Forward()
    AllRef = pp.Group((GroupKeyword + AllKeyword \
                       + pp.Optional(ExceptKeyword \
                                     + LBrace + GroupRefList \
                                     + RBrace)) \
                      | (TypeDefKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + TypeRefList \
                                       + RBrace)) \
                      | (TemplateKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + TemplateRefList \
                                       + RBrace)) \
                      | (ConstKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + ConstRefList \
                                       + RBrace)) \
                      | (AltstepKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + AltstepRefList \
                                       + RBrace)) \
                      | (TestcaseKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + TestcaseRefList \
                                       + RBrace)) \
                      | (FunctionKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + FunctionRefList \
                                       + RBrace)) \
                      | (SignatureKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + SignatureRefList \
                                       + RBrace)) \
                      | (ModuleParKeyword + AllKeyword \
                         + pp.Optional(ExceptKeyword \
                                       + LBrace + ModuleParRefList \
                                       + RBrace)))
    # [527]
    FullGroupIdentifier = pp.Forward()
    DefinitionRef = pp.Group(
        FullGroupIdentifier | Identifier("Identifier"))("DefinitionRef")
    # TODO: parse action needed to verify this:
    #DefinitionRef = (StructTypeIdentifier | EnumTypeIdentifier
    #                 | PortTypeIdentifier | ComponentTypeIdentifier \
    #                 | SubTypeIdentifier | ConstIdentifier \
    #                 | TemplateIdentifier | AltstepIdentifier \
    #                 | TestcaseIdentifier | FunctionIdentifier \
    #                 | SignatureIdentifier | VarIdentifier \
    #                 | TimerIdentifier | PortIdentifier \
    #                 | ModuleParIdentifier | FullGroupIdentifier)
    # [526]
    DefOrFieldRef = pp.Group(
        DefinitionRef | FieldReference | AllRef)("DefOrFieldRef")
    # [525]
    DefOrFieldRefList = pp.Group(
        pp.delimitedList(DefOrFieldRef))("DefOrFieldRefList")
    # [524]
    AttribQualifier = pp.Group(LParen + DefOrFieldRefList \
                               + RParen)("AttribQualifier")
    # [523] OverrideKeyword ::= "override"
    # [522] ExtensionKeyword ::= "extension"
    # [521] DisplayKeyword ::= "display"
    # [520] VariantKeyword ::= "variant"
    # [519] EncodeKeyword ::= "encode"
    # [518]
    AttribKeyword = pp.Group(
        EncodeKeyword | VariantKeyword | DisplayKeyword \
        | ExtensionKeyword)("AttribKeyword")
    # [517]
    SingleWithAttrib = pp.Group(AttribKeyword \
                                + pp.Optional(OverrideKeyword) \
                                + pp.Optional(AttribQualifier) \
                                + AttribSpec)("SingleWithAttrib")
    # [516]
    MultiWithAttrib = pp.Group(pp.OneOrMore(
        SingleWithAttrib + _opt_semicolon))("MultiWithAttrib")
    # [515]
    WithAttribList = pp.Group(LBrace + MultiWithAttrib \
                              + RBrace)("WithAttribList")
    # [514] WithKeyword :: "with"
    # [513]
    WithStatement = pp.Group(WithKeyword + WithAttribList)("WithStatement")
    # [512] TemplateParIdentifier ::= Identifier
    # [511]
    Type = pp.Forward()
    _optDirection = pp.Group(
        pp.Optional(
        InParKeyword | OutParKeyword | InOutParKeyword))("Direction")
    FormalTemplatePar = pp.Group(
        _optDirection \
        + TemplateKeyword \
        + Type + TemplateParIdentifier)("FormalTemplatePar")
    # [510] TimerParIdentifier ::= Identifier
    # [509]
    FormalTimerPar = pp.Group(pp.Optional(InOutParKeyword) + TimerKeyword \
                              + TimerParIdentifier)("FormalTimerPar")
    # [508] PortParIdentifier ::= Identifier
    # [507]
    FormalPortPar = pp.Group(pp.Optional(InOutParKeyword) \
                             + PortTypeIdentifier \
                             + PortParIdentifier)
    # [506] ValueParIdentifier ::= Identifier
    # [505]
    FormalValuePar = pp.Group(_optDirection \
                              + Type + ValueParIdentifier)("FormalValuePar")
    # [504] InOutParKeyword ::= "inout"
    # [503] OutParKeyword ::= "out"
    # [502] InParKeyword ::= "in"
    # [501] OmitKeyword ::= "omit"
    # [500]
    OmitValue = pp.Group(OmitKeyword)("OmitValue")
    # [499] AddressValue ::= "null"
    # [498] forward
    # FreeText << pp.dblQuotedString
    # [497] ExtendedAlphaNum
    # [496] LowerAlpha
    # [495] UpperAlpha
    # [494] AlphaNum
    # [493] Alpha
    # [492] Identifier
    # [491] Char
    # [490]
    Cstring = pp.Combine(pp.Literal('"') \
                         + pp.ZeroOrMore(pp.Literal('""') \
                                         | pp.CharsNotIn('"\n\r')) \
                         + pp.Literal('"'))("Cstring")
    # [489] Oct ::= Hex Hex
    # [488]
    Ostring = pp.Regex("'[0-9a-fA-F]*'O")("Ostring")
    Ostring.setParseAction(checkOctPair)
    # [487] Hex ::= Num | "A" | "B" | "C" | "D" | "E" | "F" (or lower case)
    # [486]
    Hstring = pp.Regex("'[0-9a-fA-F]*'H")
    # [485] Bin ::= "0" | "1"
    # [484]
    Bstring = pp.Regex("'[01]*'B")("Bstring")
    # [483] Num ::= "0" | NonZeroNum
    # [482]
    DecimalNumber = pp.Word("0123456789")
    # [481] NonZeroNum ::= "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9"
    # [480]
    Number = (pp.Literal("0") \
              | pp.Word("123456789", "0123456789"))("Number")

    # helper
    GlobalModuleId = pp.Forward()
    _GlobalModuleIdDot = pp.Optional(Identifier + pp.Suppress(Dot))
    _GlobalModuleIdDot.setParseAction(check_gmi)
    _ModulePrefix = pp.Optional(_GlobalModuleIdDot)

    # [479]
    ValueReference = pp.Group(
        _ModulePrefix + Identifier("Identifier"))("ValueReference")
    # TODO: parse action needed to verify this:
    #ValueReference = pp.Group(pp.Optional(GlobalModuleId + Dot) + \
    #                          (ConstIdentifier | ExtConstIdentifier \
    #                           | ModuleParIdentifier | ValueParIdentifier \
    #                           | VarIdentifier))
    # [478]
    ReferencedValue = pp.Group(
        ValueReference \
        + pp.Optional(ExtendedFieldReference))("ReferencedValue")
    # [477]
    Exponential = pp.Literal("E")
    # [476] Number [Dot DecimalNumber] Exponential [Minus] Number
    FloatENotation = pp.Regex("[0-9]+(\.[0-9]+)?E(-)?[0-9]+")
    # [475] Number Dot DecimalNumber
    FloatDotNotation = pp.Regex("[0-9]+\.[0-9]+")
    # [474]
    FloatValue = (FloatDotNotation | FloatENotation)("FloatValue")
    # [473]
    Cell = Number
    # [472]
    Row = Number
    # [471]
    Plane = Number
    # [470]
    Group = Number
    # [469] CharKeyword ::= "char"
    # [468]
    Quadruple = pp.Group(CharKeyword + LParen + Group \
                         + Comma + Plane + Comma + Row \
                         + Comma + Cell + RParen)
    # [467]
    CharStringValue = pp.Group(Cstring | Quadruple)("CharStringValue")
    # [466]
    EnumeratedValue = (EnumerationIdentifier)("EnumeratedValue")
    # [465]
    VerdictTypeValue = pp.oneOf(
        "pass fail inconc none error")("VerdictTypeValue")
    # [464]
    HexStringValue = Hstring
    # [463]
    OctetStringValue = Ostring
    # [462]
    IntegerValue = pp.Group(Number)("IntegerValue")
    # [461]
    BooleanValue = pp.oneOf("true false")("BooleanValue")
    # [460]
    BitStringValue = pp.Group(Bstring)("BitStringValue")
    # [459]
    PredefinedValue = pp.Group(FloatValue \
        | BitStringValue | BooleanValue | CharStringValue \
        | IntegerValue | OctetStringValue | HexStringValue \
        | VerdictTypeValue | EnumeratedValue \
        | AddressValue | OmitValue)("PredefinedValue")
    # [458]
    Value << pp.Group(PredefinedValue ^ ReferencedValue)("Value")
    # [457]
    ArrayBounds = pp.Group(SingleConstExpression)("ArrayBounds")
    # [456]
    ArrayDef = pp.Group(
        pp.OneOrMore(LBrack + ArrayBounds \
                     + pp.Optional(ToSymbol + ArrayBounds) \
                     + RBrack))("ArrayDef")
    # [455]
    TypeActualPar = ConstantExpression
    # [454]
    TypeActualParList = pp.Group(LParen \
                                 + pp.delimitedList(TypeActualPar) \
                                 + RParen)
    # [453]
    TypeReference = pp.Group(StructTypeIdentifier \
                              + pp.Optional(TypeActualParList) \
                              | Identifier("Identifier"))("TypeReference")
    # TODO: parse action needed to verify this:
    #TypeReference = (pp.Group(StructTypeIdentifier \
    #                          + pp.Optional(TypeActualParList) \
    #                          ^ EnumTypeIdentifier \
    #                          ^ SubTypeIdentifier \
    #                          ^ ComponentTypeIdentifier))
    # [452]
    ReferencedType = pp.Group(
        TypeReference \
        + pp.Optional(ExtendedFieldReference))("ReferencedType")
    # [451] UniversalKeyword ::= "universal"
    # [450]
    UniversalCharString = pp.Group(UniversalKeyword + CharStringKeyword)
    # [449] CharStringKeyword ::= "charstring"
    # [448] AnyTypeKeyword ::= "anytype"
    # [447] DefaultKeyword ::= "default"
    # [446] AddressKeyword ::= "address"
    # [445] FloatKeyword ::= "float"
    # [444] VerdictTypeKeyword ::= "verdicttype"
    # [443] HexStringKeyword ::= "hexstring"
    # [442] OctetStringKeyword ::= "octetstring"
    # [441] IntegerKeyword ::= "integer"
    # [440[ BooleanKeyword ::= "boolean"
    # [439] BitStringKeyword ::= "bitstring"
    # [438]
    # Note: CharKeyword is not in the standard for PredefinedType!
    PredefinedType = (BitStringKeyword | BooleanKeyword \
                      | CharStringKeyword | UniversalCharString \
                      | IntegerKeyword | OctetStringKeyword \
                      | HexStringKeyword | VerdictTypeKeyword \
                      | FloatKeyword | AddressKeyword \
                      | DefaultKeyword | AnyTypeKeyword \
                      | CharKeyword)("PredefinedType")
    # [437]
    Type << (PredefinedType | ReferencedType)("Type")
    # [436] TimeoutKeyword ::= "timeout"
    # [435]
    TimerRef = pp.Forward()
    TimerRefOrAny = pp.Group(
        (AnyKeyword + TimerKeyword) | TimerRef)("TimerRefOrAny")
    # [434]
    TimeoutStatement << pp.Group(TimerRefOrAny + Dot + TimeoutKeyword)
    # [433]
    RunningTimerOp = pp.Group(TimerRefOrAny + Dot + RunningKeyword)
    # [432] ReadKeyword ::= "read"
    # [431]
    ReadTimerOp = pp.Group(TimerRef + Dot + ReadKeyword)
    # [430]
    TimerRefOrAll = pp.Group(
        TimerRef | (AllKeyword + TimerKeyword))("TimerRefOrAll")
    # [429]
    StopTimerStatement = pp.Group(TimerRefOrAll + Dot + StopKeyword)
    # [428]
    TimerValue = pp.Forward()
    StartTimerStatement = pp.Group(TimerRef + Dot + StartKeyword \
                                   + pp.Optional(LParen \
                                                 + TimerValue \
                                                 + RParen))
    # [427]
    TimerOps << (ReadTimerOp | RunningTimerOp)
    # [426]
    TimerStatements = (StartTimerStatement | StopTimerStatement \
                       | TimeoutStatement)
    # [425] AnyKeyword ::= "any"
    # [424] StopKeyword ::= "stop"
    # [423]
    PortStopOp = StopKeyword
    # [422]
    Port = pp.Forward()
    PortOrAll = pp.Group( # forward
        (AllKeyword + PortKeyword) | Port)("PortOrAll")
    StopStatement = pp.Group(PortOrAll + Dot + PortStopOp)
    # [421]
    PortStartOp = StartKeyword
    # [420]
    StartStatement = pp.Group(PortOrAll + Dot + PortStartOp)
    # [419] ClearOpKeyword ::= "clear"
    # [418]
    PortClearOp = ClearOpKeyword
    # [417] forward
    # PortOrAll << (pp.Group(AllKeyword + PortKeyword) | Port)
    # [416]
    ClearStatement = pp.Group(PortOrAll + Dot + PortClearOp)
    # [415]
    Signature = pp.Forward()
    CatchOpParameter = pp.Group((Signature + Comma \
                                 + TemplateInstance) \
                                | TimeoutKeyword)("CatchOpParameter")
    # [414] CatchOpKeyword ::= "catch"
    # [413]
    PortRedirect = pp.Forward()
    FromClause = pp.Forward()
    PortCatchOp = pp.Group(CatchOpKeyword + pp.Optional(LParen \
                                                        + CatchOpParameter \
                                                        + RParen) \
                           + pp.Optional(FromClause) \
                           + pp.Optional(PortRedirect))("PortCatchOp")
    # [412]
    PortOrAny = pp.Forward()
    CatchStatement << pp.Group(PortOrAny + Dot + PortCatchOp)
    # [411]
    PortReceiveOp = pp.Forward()
    PortGetCallOp = pp.Forward()
    PortGetReplyOp = pp.Forward()
    CheckPortOpsPresent = pp.Group(
        PortReceiveOp | PortGetCallOp | PortGetReplyOp \
        | PortCatchOp)("CheckPortOpsPresent")
    # [410]
    SenderSpec = pp.Forward()
    RedirectPresent = pp.Group(
        PortRedirectSymbol + SenderSpec)("RedirectPresent")
    # [409]
    FromClausePresent = pp.Group(
        FromClause + pp.Optional(PortRedirectSymbol \
                                 + SenderSpec))("FromClausePresent")
    # [408]
    CheckParameter = pp.Group(
        CheckPortOpsPresent | FromClausePresent \
        | RedirectPresent)("CheckParameter")
    # [407] CheckOpKeyword ::= "check"
    # [406]
    PortCheckOp = pp.Group(CheckOpKeyword \
                           + pp.Optional(LParen \
                                         + CheckParameter \
                                         + RParen))("PortCheckOp")
    # [405]
    CheckStatement << pp.Group(
        PortOrAny + Dot + PortCheckOp)("CheckStatement")
    # [404]
    ValueMatchSpec = pp.Group(
        ValueKeyword + TemplateInstance)("ValueMatchSpec")
    # [403] GetReplyOpKeyword ::= "getreply"
    # [402]
    ValueSpec = pp.Forward()
    ParamSpec = pp.Forward()
    RedirectWithParamSpec = pp.Forward()
    RedirectWithValueAndParamSpec = pp.Group(
        (ValueSpec \
         + pp.Optional(ParamSpec) \
         + pp.Optional(SenderSpec)) \
        | RedirectWithParamSpec)("RedirectWithValueAndParamSpec")
    # [401]
    PortRedirectWithValueAndParam = pp.Group(
        PortRedirectSymbol \
        + RedirectWithValueAndParamSpec)("PortRedirectWithValueAndParam")
    # [400]
    ReceiveParameter = pp.Forward()
    PortGetReplyOp << pp.Group(
        GetReplyOpKeyword \
        + pp.Optional(LParen \
                      + ReceiveParameter \
                      + pp.Optional(ValueMatchSpec) \
                      + RParen) \
        + pp.Optional(FromClause) \
        + pp.Optional(PortRedirectWithValueAndParam))("PortGetReplyOp")
    # [399]
    GetReplyStatement << pp.Group(
        PortOrAny + Dot + PortGetReplyOp)("GetReplyStatement")
    # [398]
    VariableEntry = pp.Group(NotUsedSymbol | VariableRef)("VariableEntry")
    # [397]
    VariableList = pp.delimitedList(VariableEntry)
    # [396]
    ParameterIdentifier = ValueParIdentifier
    # [395]
    VariableAssignment = pp.Group(
        VariableRef + AssignmentChar \
        + ParameterIdentifier)("VariableAssignment")
    # [394]
    AssignmentList = pp.Group(
        pp.delimitedList(VariableAssignment))("AssignmentList")
    # [393]
    ParamAssignmentList = pp.Group(LParen \
                                   + (AssignmentList \
                                      | VariableList) \
                                   + RParen)("ParamAssignmentList")
    # [392] ParamKeyword ::= "param"
    # [391]
    ParamSpec << pp.Group(ParamKeyword + ParamAssignmentList)("ParamSpec")
    # [390]
    RedirectWithParamSpec << pp.Group(
        (ParamSpec + pp.Optional(SenderSpec)) \
        | SenderSpec)("RedirectWithParamSpec")
    # [389]
    PortRedirectWithParam = pp.Group(
        PortRedirectSymbol \
        + RedirectWithParamSpec)("PortRedirectWithParam")
    # [388] GetCallOpKeyword ::= "getcall"
    # [387]
    PortGetCallOp << pp.Group(
        GetCallOpKeyword \
        + pp.Optional(LParen \
                      + ReceiveParameter \
                      + RParen) \
        + pp.Optional(FromClause) \
        + pp.Optional(PortRedirectWithParam))("PortGetCallOp")
    # [386]
    GetCallStatement << pp.Group(
        PortOrAny + Dot + PortGetCallOp)("GetCallStatement")
    # [385] TriggerOpKeyword ::= "trigger"
    # [384]
    PortTriggerOp = pp.Group(
        TriggerOpKeyword \
        + pp.Optional(LParen \
                      + ReceiveParameter \
                      + RParen) \
        + pp.Optional(FromClause) \
        + pp.Optional(PortRedirect))("PortTriggerOp")
    # [383]
    TriggerStatement << pp.Group(
        PortOrAny + Dot + PortTriggerOp)("TriggerStatement")
    # [382] SenderKeyword ::= "sender"
    # [381]
    SenderSpec << pp.Group(SenderKeyword + VariableRef)("SenderSpec")
    # [380] ValueKeyword ::= "value"
    # [379]
    ValueSpec << pp.Group(ValueKeyword + VariableRef)("ValueSpec")
    # [378] PortRedirectSymbol ::= "->"
    # [377]
    PortRedirect << pp.Group(PortRedirectSymbol \
                             + ((ValueSpec \
                                 + pp.Optional(SenderSpec)) \
                                | SenderSpec))("PortRedirect")
    # [376] FromKeyword ::= "from"
    # [375]
    AddressRef = pp.Forward()
    FromClause << pp.Group(FromKeyword + AddressRef)("FromClause")
    # [374]
    ReceiveParameter << TemplateInstance
    # [373] ReceiveOpKeyword ::= "receive"
    # [372]
    PortReceiveOp << pp.Group(ReceiveOpKeyword \
                              + pp.Optional(LParen \
                                            + ReceiveParameter \
                                            + RParen) \
                              + pp.Optional(FromClause) \
                              + pp.Optional(PortRedirect))("PortReceiveOp")
    # [371]
    PortOrAny << pp.Group((AnyKeyword + PortKeyword) | Port)("PortOrAny")
    # [370]
    ReceiveStatement << pp.Group(PortOrAny \
                                 + Dot + PortReceiveOp)("ReceiveStatement")
    # [369] RaiseKeyword ::= "raise"
    # [368]
    ToClause = pp.Forward()
    PortRaiseOp = pp.Group(RaiseKeyword + LParen + Signature \
                           + Comma + TemplateInstance \
                           + RParen + pp.Optional(ToClause))("PortRaiseOp")
    # [367]
    RaiseStatement = pp.Group(Port + Dot + PortRaiseOp)("RaiseStatement")
    # [366]
    ReplyValue = pp.Group(ValueKeyword + Expression)("ReplyValue")
    # [365] ReplyKeyword ::= "reply"
    # [364]
    PortReplyOp = pp.Group(ReplyKeyword + LParen + TemplateInstance \
                           + pp.Optional(ReplyValue) + RParen \
                           + pp.Optional(ToClause))("PortReplyOp")
    # [363]
    ReplyStatement = pp.Group(Port + Dot + PortReplyOp)("ReplyStatement")
    # [362]
    CallBodyOps = pp.Group(GetReplyStatement | CatchStatement)("CallBodyOps")
    # [361]
    CallBodyGuard = pp.Group(AltGuardChar + CallBodyOps)("CallBodyGuard")
    # [360]
    CallBodyStatement = pp.Group(
        CallBodyGuard + StatementBlock)("CallBodyStatement")
    # [359]
    CallBodyStatementList = pp.Group(
        pp.OneOrMore(CallBodyStatement \
                     + _opt_semicolon))("CallBodyStatementList")
    # [358]
    PortCallBody = pp.Group(LBrace + CallBodyStatementList \
                            + RBrace)("PortCallBody")
    # [357] NowaitKeyword ::= "nowait"
    # [356]
    CallTimerValue = (NowaitKeyword | TimerValue)
    # [355]
    CallParameters = pp.Group(
        TemplateInstance \
        + pp.Optional(Comma \
                      + CallTimerValue))("CallParameters")
    # [354] CallOpKeyword ::= "call"
    # [353]
    PortCallOp = pp.Group(CallOpKeyword + LParen + CallParameters \
                          + RParen + pp.Optional(ToClause))("PortCallOp")
    # [352]
    CallStatement = pp.Group(Port + Dot + PortCallOp \
                             + pp.Optional(PortCallBody))("CallStatement")
    # [351]
    AddressRef << TemplateInstance
    # [350] ToKeyword ::= "to"
    # [349]
    AddressRefList = pp.delimitedList(AddressRef)
    # [348]
    ToClause << pp.Group(ToKeyword \
                         + (AddressRefList | AddressRef \
                            | (AllKeyword \
                               + ComponentKeyword)))("ToClause")
    # [347]
    SendParameter = pp.Group(TemplateInstance)("SendParameter")
    # [346] SendOpKeyword ::= "send"
    # [345]
    PortSendOp = pp.Group(SendOpKeyword + LParen \
                          + SendParameter + RParen \
                          + pp.Optional(ToClause))("PortSendOp")
    # [344]
    SendStatement = pp.Group(Port + Dot + PortSendOp)("SendStatement")
    # [343]
    CommunicationStatements = pp.Group(
        SendStatement | CallStatement \
        | ReplyStatement | RaiseStatement \
        | ReceiveStatement | TriggerStatement \
        | GetCallStatement | GetReplyStatement \
        | CatchStatement | CheckStatement \
        | ClearStatement | StartStatement \
        | StopStatement)("CommunicationStatements")
    # [342]
    Port << pp.Group(PortIdentifier + pp.Optional(ArrayOrBitRef))("Port")
    # [341] KillKeyword ::= "kill"
    # [340]
    ComponentOrDefaultReference << pp.Group(
        VariableRef | FunctionInstance)("ComponentOrDefaultReference")
    # [339]
    MTCOp = MTCKeyword # forward
    ComponentReferenceOrLiteral = pp.Group( # forward
        ComponentOrDefaultReference \
        | MTCOp | SelfOp)("ComponentReferenceOrLiteral")
    KillTCStatement = pp.Group(KillKeyword \
                               | (ComponentReferenceOrLiteral + Dot \
                                          + KillKeyword) \
                               | (AllKeyword + ComponentKeyword \
                                  + Dot \
                                  + KillKeyword))
    # [338] forward
    #ComponentReferenceOrLiteral << (ComponentOrDefaultReference | MTCOp \
    #                                | SelfOp)
    # [337]
    StopTCStatement = pp.Group(StopKeyword \
                               | (ComponentReferenceOrLiteral + Dot \
                                  + StopKeyword) \
                               | (AllKeyword + ComponentKeyword \
                                  + Dot \
                                  + StopKeyword))("StopTCStatement")
    # [336] StartKeyword ::= "start"
    # [335]
    StartTCStatement = pp.Group(ComponentOrDefaultReference + Dot \
                                + StartKeyword \
                                + LParen + FunctionInstance \
                                + RParen)
    # [334] UnmapKeyword ::= "unmap"
    # [333]
    SingleOrMultiConnectionSpec = pp.Forward()
    UnmapStatement = pp.Group(
        UnmapKeyword + SingleOrMultiConnectionSpec)("UnmapStatement")
    # [332] MapKeyword ::= "map"
    # [331]
    SingleConnectionSpec = pp.Forward()
    MapStatement = pp.Group(MapKeyword + SingleConnectionSpec)("MapStatement")
    # [330] DisconnectKeyword ::= "disconnect"
    # [329]
    AllCompsAllPortsSpec = pp.Group(LParen + AllKeyword \
                                    + ComponentKeyword + Colon \
                                    + AllKeyword + PortKeyword \
                                    + RParen)
    # [328]
    ComponentRef = pp.Forward()
    AllPortsSpec = pp.Group(LParen + ComponentRef \
                            + Colon + AllKeyword + PortKeyword \
                            + RParen)
    # [327]
    PortRef = pp.Forward()
    AllConnectionsSpec = pp.Group(LParen + PortRef \
                                    + RParen)("AllConnectionsSpec")
    # [326]
    SingleOrMultiConnectionSpec << pp.Group(
        SingleConnectionSpec \
        | AllConnectionsSpec \
        | AllPortsSpec | AllCompsAllPortsSpec)("SingleOrMultiConnectionSpec")
    # [325]
    DisconnectStatement = pp.Group(
        DisconnectKeyword \
        + pp.Optional(SingleOrMultiConnectionSpec))("DisconnectStatement")
    # [324]
    SystemOp = SystemKeyword # forward
    ComponentRef << pp.Group(
        ComponentOrDefaultReference | SystemOp \
        | SelfOp | MTCOp)("ComponentRef")
    # [323]
    PortRef << pp.Group(ComponentRef + Colon + Port)("PortRef")
    # [322]
    SingleConnectionSpec << pp.Group(LParen \
                                     + PortRef + Comma + PortRef \
                                     + RParen)("SingleConnectionSpec")
    # [321] ConnectKeyword ::= "connect"
    # [320]
    ConnectStatement = pp.Group(
        ConnectKeyword + SingleConnectionSpec)("ConnectStatement")
    # [319] AliveKeyword ::= "alive"
    # [318] CreateKeyword ::= "create"
    # [317]
    ComponentId = pp.Forward()
    AliveOp = pp.Group(ComponentId + Dot + AliveKeyword)
    # [316] RunningKeyword ::= "running"
    # [315]
    RunningOp = pp.Group(ComponentId + Dot + RunningKeyword)
    # [314] KilledKeyword ::= "killed"
    # [313] DoneKeyword ::= "done"
    # [312]
    ComponentId << pp.Group(ComponentOrDefaultReference \
                            | ((AnyKeyword | AllKeyword) \
                               + ComponentKeyword))("ComponentId")
    # [311]
    KilledStatement << pp.Group(ComponentId + Dot + KilledKeyword)
    # [310]
    DoneStatement << pp.Group(ComponentId + Dot + DoneKeyword)
    # [309] forward
    # MTCOp << MTCKeyword
    # [308] SelfOp ::= "self"
    # [307] forward
    # SystemOp << SystemKeyword
    # [306]
    ComponentType = pp.Forward()
    CreateOp = pp.Group(ComponentType + Dot + CreateKeyword \
                        + pp.Optional(LBrace \
                                      + SingleExpression \
                                      + RBrace) \
                        + pp.Optional(AliveKeyword))
    # [305]
    ConfigurationOps << (SelfOp | SystemOp | CreateOp | RunningOp | MTCOp \
                         | AliveOp)
    # [304]
    ConfigurationStatements = pp.Group(
        ConnectStatement | MapStatement \
        | DisconnectStatement | UnmapStatement \
        | DoneStatement | KilledStatement \
        | StartTCStatement | StopTCStatement \
        | KillTCStatement)("ConfigurationStatements")
    # [303]
    # TODO: parse action needed to verify this:
    #TimerRef << pp.Group((TimerIdentifier | TimerParIdentifier) \
    #                     + pp.Optional(ArrayOrBitRef))
    TimerRef << pp.Group(Identifier("Identifier") \
                         + pp.Optional(ArrayOrBitRef))("TimerRef")
    # [302]
    TimerValue << pp.Group(Expression)("TimerValue")
    # [301] TimerIdentifier ::= Identifier
    # [300] TimerKeyword ::= "timer"
    # [299]
    SingleTimerInstance = pp.Group(
        TimerIdentifier + pp.Optional(ArrayDef) \
        + pp.Optional(AssignmentChar \
                      + TimerValue))("SingleTimerInstance")
    # [298]
    TimerList = pp.Group(
        pp.delimitedList(SingleTimerInstance))("TimerList")
    # [297]
    TimerInstance = pp.Group(TimerKeyword + TimerList)("TimerInstance")
    # [296]
    # TODO: parse action needed to verify this:
    #VariableRef = pp.Group((VarIdentifier ^ ValueParIdentifier) \
    #                        + pp.Optional(ExtendedFieldReference))
    VariableRef << pp.Group(
        Identifier("Identifier") \
        + pp.Optional(ExtendedFieldReference))("VariableRef")
    # [295]
    TempVarInitialValue = TemplateBody
    # [294]
    SingleTempVarInstance = pp.Group(VarIdentifier + pp.Optional(ArrayDef) \
                                     + pp.Optional(AssignmentChar \
                                                   + TempVarInitialValue))
    # [293]
    TempVarList = pp.delimitedList(SingleTempVarInstance)
    # [292] VarIdentifier ::= Identifier
    # [291] VarKeyword ::= "var"
    # [290]
    VarInitialValue = pp.Group(Expression)("VarInitialValue")
    # [289]
    SingleVarInstance = pp.Group(
        VarIdentifier + pp.Optional(ArrayDef) \
        + pp.Optional(AssignmentChar \
                      + VarInitialValue))("SingleVarInstance")
    # [288]
    VarList = pp.delimitedList(SingleVarInstance)
    # [287]
    VarInstance << pp.Group(VarKeyword \
                            + ((TemplateKeyword + Type \
                                + TempVarList) \
                               | (Type + VarList)))("VarInstance")
    # [286]
    ControlStatement = pp.Group(
        TimerStatements | BasicStatements \
        | BehaviourStatements | SUTStatements \
        | StopKeyword)("ControlStatement")
    # [285]
    FunctionLocalDef = pp.Forward()
    FunctionLocalInst = pp.Forward()
    ControlStatementOrDef = pp.Group(
        FunctionLocalDef | FunctionLocalInst \
        | ControlStatement)("ControlStatementOrDef")
    # [284]
    ControlStatementOrDefList = pp.Group(
        pp.OneOrMore(ControlStatementOrDef \
                     + _opt_semicolon))("ControlStatementOrDefList")
    # [283]
    ModuleControlBody = pp.Group(
        pp.Optional(ControlStatementOrDefList))("ModuleControlBody")
    # [282] ControlKeyword ::= "control"
    # [281]
    ModuleControlPart = pp.Group(ControlKeyword + LBrace \
                                 + ModuleControlBody + RBrace \
                                 + pp.Optional(WithStatement) \
                                 + _opt_semicolon)("ModuleControlPart")
    # [280] ModuleParIdentifier ::= Identifier
    # [279]
    ModuleParList = pp.Group(
        ModuleParIdentifier \
        + pp.Optional(AssignmentChar \
                      + ConstantExpression) \
        + pp.ZeroOrMore(Comma \
                        + ModuleParIdentifier \
                        + pp.Optional(AssignmentChar \
                                      + ConstantExpression)))("ModuleParList")
    # [278]
    ModuleParType = pp.Group(Type)("ModuleParType")
    # [277]
    ModulePar = pp.Group(ModuleParType + ModuleParList)("ModulePar")
    # [276]
    MultitypedModuleParList = pp.Group(pp.ZeroOrMore(
        ModulePar \
        + _opt_semicolon)("MultitypedModuleParList"))
    # [275] ModuleParKeyword ::= "modulepar"
    # [274]
    ModuleParDef = pp.Group(ModuleParKeyword \
                            + (ModulePar \
                               | (LBrace \
                                  + MultitypedModuleParList \
                                  + RBrace)))("ModuleParDef")
    # [273] ExtConstIdentifier ::= Identifier
    # [272]
    ExtConstIdentifierList = pp.delimitedList(
        ExtConstIdentifier)("ExtConstIdentifierList")
    # [271]
    ExtConstDef = pp.Group(ExtKeyword + ConstKeyword + Type \
                           + ExtConstIdentifierList)("ExtConstDef")
    # [270] ExtFunctionIdentifier ::= Identifier
    # [269] ExtKeyword ::= "external"
    # [268]
    FunctionFormalParList = pp.Forward()
    ReturnType = pp.Forward()
    ExtFunctionDef = pp.Group(
        ExtKeyword + FunctionKeyword \
        + ExtFunctionIdentifier + LParen \
        + pp.Optional(FunctionFormalParList) \
        + RParen + pp.Optional(ReturnType))("ExtFunctionDef")
    # [267] GroupIdentifier ::= Identifier
    # [266] GroupKeyword ::= "group"
    # [265]
    ModuleDefinitionsPart = pp.Forward()
    GroupDef = pp.Group(GroupKeyword + GroupIdentifier + LBrace \
                        + pp.Optional(ModuleDefinitionsPart) \
                        + RBrace)("GroupDef")
    # [264]
    AllModuleParWithExcept = pp.Group(AllKeyword \
                                      + pp.Optional(ExceptKeyword \
                                                    + ModuleParRefList))
    # [263]
    ModuleParRefList << pp.delimitedList(ModuleParIdentifier)
    # [262]
    ImportModuleParSpec = pp.Group(ModuleParKeyword \
                                   + (ModuleParRefList \
                                      | AllModuleParWithExcept))
    # [261]
    AllSignaturesWithExcept = pp.Group(AllKeyword \
                                       + pp.Optional(ExceptKeyword \
                                                     + SignatureRefList))
    # [260]
    SignatureRefList << pp.delimitedList(SignatureIdentifier)
    # [259]
    ImportSignatureSpec = pp.Group(SignatureKeyword \
                                   + (SignatureRefList \
                                      | AllSignaturesWithExcept))
    # [258]
    AllFunctionsWithExcept = pp.Group(AllKeyword \
                                      + pp.Optional(ExceptKeyword \
                                                    + FunctionRefList))
    # [257]
    FunctionRefList << pp.delimitedList(FunctionIdentifier)
    # [256]
    ImportFunctionSpec = pp.Group(
        FunctionKeyword \
        + (FunctionRefList \
           | AllFunctionsWithExcept))("ImportFunctionSpec")
    # [255]
    AllTestcasesWithExcept = pp.Group(AllKeyword \
                                      + pp.Optional(ExceptKeyword \
                                                    + TestcaseRefList))
    # [254]
    TestcaseRefList << pp.delimitedList(TestcaseIdentifier)
    # [253]
    ImportTestcaseSpec = pp.Group(TestcaseKeyword \
                                  + (TestcaseRefList \
                                     | AllTestcasesWithExcept))
    # [252]
    AllAltstepsWithExcept = pp.Group(AllKeyword \
                                     + pp.Optional(ExceptKeyword \
                                                   + AltstepRefList))
    # [251]
    AltstepRefList << pp.delimitedList(AltstepIdentifier)("AltstepRefList")
    # [250]
    ImportAltstepSpec = pp.Group(AltstepKeyword \
                                 + (AltstepRefList \
                                    | AllAltstepsWithExcept))
    # [249]
    AllConstsWithExcept = pp.Group(AllKeyword + pp.Optional(ExceptKeyword \
                                                            + ConstRefList))
    # [248]
    ConstRefList << pp.delimitedList(ConstIdentifier)
    # [247]
    ImportConstSpec = pp.Group(ConstKeyword + (ConstRefList \
                                               | AllConstsWithExcept))
    # [246]
    AllTemplsWithExcept = pp.Group(
        AllKeyword \
        + pp.Optional(ExceptKeyword \
                      + TemplateRefList))("AllTemplsWithExcept")
    # [245]
    TemplateRefList << pp.delimitedList(TemplateIdentifier)
    # [244]
    ImportTemplateSpec = pp.Group(
        TemplateKeyword \
        + (TemplateRefList | AllTemplsWithExcept))("ImportTemplateSpec")
    # [243]
    TypeDefIdentifier << (Identifier)("TypeDefIdentifier")
    # TODO: parse action needed to verify this:
    #TypeDefIdentifier = (StructTypeIdentifier ^ EnumTypeIdentifier \
    #                     ^ PortTypeIdentifier ^ ComponentTypeIdentifier \
    #                     ^ SubTypeIdentifier)
    # [242]
    AllTypesWithExcept = pp.Group(AllKeyword + pp.Optional(ExceptKeyword \
                                                           + TypeRefList))
    # [241]
    TypeRefList << pp.Group(
        pp.delimitedList(TypeDefIdentifier))("TypeRefList")
    # [240]
    ImportTypeDefSpec = pp.Group(
        TypeDefKeyword + (TypeRefList \
                          | AllTypesWithExcept))("ImportTypeDefSpec")
    # [239]
    ExceptFullGroupIdentifier = FullGroupIdentifier
    # [238]
    ExceptGroupRefList = pp.delimitedList(ExceptFullGroupIdentifier)
    # [237]
    ExceptsDef = pp.Forward()
    FullGroupIdentifierWithExcept = pp.Group(FullGroupIdentifier \
                                             + pp.Optional(ExceptsDef))
    # [236]
    FullGroupIdentifier << pp.delimitedList(GroupIdentifier, delim=".")
    # [235]
    AllGroupsWithExcept = pp.Group(AllKeyword + pp.Optional(ExceptKeyword \
                                                            + GroupRefList))
    # [234]
    GroupRefListWithExcept = pp.Group(
        pp.delimitedList(
            FullGroupIdentifierWithExcept))("GroupRefListWithExcept")
    # [233]
    GroupRefList << pp.delimitedList(FullGroupIdentifier)
    # [232]
    ImportGroupSpec = pp.Group(
        GroupKeyword \
        + (GroupRefListWithExcept \
           | AllGroupsWithExcept))("ImportGroupSpec")
    # [231] RecursiveKeyword ::= "recursive"
    # [230]
    ModuleId = pp.Forward()
    ImportFromSpec = pp.Group(
        FromKeyword + ModuleId \
        + pp.Optional(RecursiveKeyword))("ImportFromSpec")
    # [229]
    ImportElement = pp.Group(
        ImportGroupSpec | ImportTypeDefSpec \
        | ImportTemplateSpec \
        | ImportConstSpec | ImportTestcaseSpec \
        | ImportAltstepSpec | ImportFunctionSpec \
        | ImportSignatureSpec | ImportModuleParSpec)("ImportElement")
    # [228]
    ImportSpec = pp.Group(
        pp.ZeroOrMore(ImportElement + _opt_semicolon))("ImportSpec")
    # [227]
    ExceptModuleParSpec = pp.Group(ModuleParKeyword \
                                   + (ModuleParRefList | AllKeyword))
    # [226]
    ExceptSignatureSpec = pp.Group(SignatureKeyword \
                                   + (SignatureRefList | AllKeyword))
    # [225]
    ExceptFunctionSpec = pp.Group(FunctionKeyword \
                                  + (FunctionRefList | AllKeyword))
    # [224]
    ExceptAltstepSpec = pp.Group(AltstepKeyword \
                                 + (AltstepRefList | AllKeyword))
    # [223]
    ExceptTestcaseSpec = pp.Group(TestcaseKeyword \
                                  + (TestcaseRefList | AllKeyword))
    # [222]
    ExceptConstSpec = pp.Group(ConstKeyword \
                               + (ConstRefList | AllKeyword))
    # [221]
    ExceptTemplateSpec = pp.Group(TemplateKeyword \
                                  + (TemplateRefList | AllKeyword))
    # [220]
    ExceptTypeDefSpec = pp.Group(TypeDefKeyword \
                                 + (TypeRefList | AllKeyword))
    # [219]
    ExceptGroupSpec = pp.Group(GroupKeyword \
                               + (ExceptGroupRefList | AllKeyword))
    # [218]
    ExceptElement = (ExceptGroupSpec | ExceptTypeDefSpec \
                     | ExceptTemplateSpec \
                     | ExceptConstSpec | ExceptTestcaseSpec \
                     | ExceptAltstepSpec | ExceptFunctionSpec \
                     | ExceptSignatureSpec | ExceptModuleParSpec)
    # [217]
    ExceptSpec = pp.ZeroOrMore(ExceptElement + _opt_semicolon)
    # [216] ExceptKeyword ::= "except"
    # [215]
    ExceptsDef << pp.Group(ExceptKeyword + LBrace \
                           + ExceptSpec + RBrace)("ExceptsDef")
    # [214]
    AllWithExcepts = pp.Group(
        AllKeyword + pp.Optional(ExceptsDef))("AllWithExcepts")
    # [213] ImportKeyword ::= "import"
    # [212]
    ImportDef = pp.Group(ImportKeyword + ImportFromSpec \
                         + (AllWithExcepts \
                            | (LBrace + ImportSpec \
                               + RBrace)))("ImportDef")
    # [211]
    AltstepRef = pp.Group(_ModulePrefix + AltstepIdentifier)("AltstepRef")
    # [210]
    FunctionActualParList = pp.Forward()
    AltstepInstance << pp.Group(AltstepRef + LParen \
                                + pp.Optional(FunctionActualParList) \
                                + RParen)("AltstepInstance")
    # [209]
    ConstDef = pp.Forward()
    TemplateDef = pp.Forward()
    AltstepLocalDef = pp.Group(VarInstance | TimerInstance \
                               | ConstDef | TemplateDef)("AltstepLocalDef")
    # [208]
    AltstepLocalDefList = pp.ZeroOrMore(
        AltstepLocalDef \
        + _opt_semicolon)("AltstepLocalDefList")
    # [207]
    AltstepFormalParList = pp.Group(
        FunctionFormalParList)("AltstepFormalParList")
    # [206] AltstepIdentifier ::= Identifier
    # [205] AltstepKeyword ::= "altstep"
    # [204]
    RunsOnSpec = pp.Forward()
    AltstepDef = pp.Group(AltstepKeyword + AltstepIdentifier \
                          + LParen \
                          + pp.Optional(AltstepFormalParList) \
                          + RParen + pp.Optional(RunsOnSpec) \
                          + LBrace + AltstepLocalDefList \
                          + AltGuardList + RBrace)("AltstepDef")
    # [203]
    TestcaseActualPar = TemplateInstance
    # [202]
    TestcaseActualParList = pp.delimitedList(TestcaseActualPar)
    # [201]
    TestcaseRef = pp.Group(_ModulePrefix \
                           + TestcaseIdentifier)("TestcaseRef")
    # [200] ExecuteKeyword ::= "execute"
    # [199]
    TestcaseInstance << pp.Group(ExecuteKeyword + LParen \
                                 + TestcaseRef + LParen \
                                 + pp.Optional(TestcaseActualParList) \
                                 + RParen \
                                 + pp.Optional(Comma + TimerValue) \
                                 + RParen)("TestcaseInstance")
    # [198] SystemKeyword ::= "system"
    # [197]
    SystemSpec = pp.Group(SystemKeyword + ComponentType)("SystemSpec")
    # [196]
    ConfigSpec = pp.Group(RunsOnSpec + pp.Optional(SystemSpec))("ConfigSpec")
    # [195]
    TestcaseFormalPar = pp.Group(FormalValuePar \
                                 | FormalTemplatePar)("TestcaseFormalPar")
    # [194]
    TestcaseFormalParList = pp.Group(
        pp.delimitedList(TestcaseFormalPar))("TestcaseFormalParList")
    # [193] TestcaseIdentifier ::= Identifier
    # [192] TestcaseKeyword ::= "testcase"
    # [191]
    TestcaseDef = pp.Group(TestcaseKeyword + TestcaseIdentifier \
                           + LParen \
                           + pp.Optional(TestcaseFormalParList) \
                           + RParen \
                           + ConfigSpec + StatementBlock)("TestcaseDef")
    # [190]
    Signature << pp.Group(_ModulePrefix \
                          + SignatureIdentifier)("Signature")
    # [189] NoBlockKeyword ::= "noblock"
    # [188]
    ExceptionTypeList = pp.Group(pp.delimitedList(Type))("ExceptionTypeList")
    # [187] ExceptionKeyword ::= "exception"
    # [186]
    ExceptionSpec = pp.Group(ExceptionKeyword + LParen \
                             + ExceptionTypeList + RParen)("ExceptionSpec")
    # [185]
    SignatureFormalPar = FormalValuePar
    # [184]
    SignatureFormalParList = pp.delimitedList(SignatureFormalPar)
    # [183] SignatureIdentifier ::= Identifier
    # [182] SignatureKeyword ::= "signature"
    # [181]
    SignatureDef = pp.Group(SignatureKeyword + SignatureIdentifier \
                            + LParen \
                            + pp.Optional(SignatureFormalParList) \
                            + RParen \
                            + pp.Optional(ReturnType | NoBlockKeyword) \
                            + pp.Optional(ExceptionSpec))("SignatureDef")
    # [180]
    FunctionActualPar = pp.Group(TimerRef ^ TemplateInstance \
                                 ^ Port ^ ComponentRef)("FunctionActualPar")
    # [179]
    FunctionActualParList << pp.Group(pp.delimitedList(
        FunctionActualPar))("FunctionActualParList")
    # [178]
    PreDefFunctionIdentifier = pp.oneOf(
        "bit2hex bit2int bit2oct bit2str char2int char2oct float2int " \
        + "hex2bit hex2int hex2oct hex2str int2bit int2char int2float " \
        + "int2hex int2oct int2str int2unichar ischosen ispresent lengthof " \
        + "oct2bit oct2char oct2hex oct2int oct2str regexp replace rnd " \
        + "sizeof sizeoftype str2float str2int str2oct substr unichar2int"
        )("PreDefFunctionIdentifier")
    # [177]
    FunctionRef = pp.Group(
        PreDefFunctionIdentifier \
        | (_ModulePrefix \
           + (FunctionIdentifier \
              | ExtFunctionIdentifier)))("FunctionRef")
    # [176]
    FunctionInstance << pp.Group(FunctionRef + LParen \
                                 + pp.Optional(FunctionActualParList) \
                                 + RParen)("FunctionInstance")
    # [175]
    FunctionStatement = pp.Group(
        ConfigurationStatements | TimerStatements \
        | CommunicationStatements \
        | BasicStatements | BehaviourStatements \
        | VerdictStatements | SUTStatements)("FunctionStatement")
    # [174]
    FunctionLocalDef << pp.Group(ConstDef | TemplateDef)("FunctionLocalDef")
    # [173]
    FunctionLocalInst << pp.Group(VarInstance \
                                  | TimerInstance)("FunctionLocalInst")
    # [172]
    FunctionStatementOrDef = pp.Group(
        FunctionLocalDef | FunctionLocalInst \
        | FunctionStatement)("FunctionStatementOrDef")
    # [171]
    FunctionStatementOrDefList = pp.OneOrMore(
        FunctionStatementOrDef \
        + _opt_semicolon)("FunctionStatementOrDefList")
    # [170]
    StatementBlock << pp.Group(LBrace.suppress() \
                              + pp.Optional(FunctionStatementOrDefList) \
                              + RBrace.suppress())("StatementBlock")
    _StatementBlock = pp.Group(LBrace.suppress() \
                               + Assignment \
                               + RBrace.suppress())("StatementBlock")
    # [169] MTCKeyword ::= "mtc"
    # [168] OnKeyword ::= "on"
    # [167] RunsKeyword ::= "runs"
    # [166]
    RunsOnSpec << pp.Group(RunsKeyword + OnKeyword \
                           + ComponentType)("RunsOnSpec")
    # [165] ReturnKeyword ::= "return"
    # [164]
    ReturnType << pp.Group(ReturnKeyword + Type)("ReturnType")
    # [163]
    FunctionFormalPar = pp.Group(
        FormalValuePar | FormalTimerPar \
        | FormalTemplatePar | FormalPortPar)("FunctionFormalPar")
    # [162]
    FunctionFormalParList << pp.delimitedList(
        FunctionFormalPar)("FunctionFormalParList")
    # [161] FunctionIdentifier ::= Identifier
    # [160] FunctionKeyword ::= "function"
    # [159]
    FunctionDef = pp.Group(FunctionKeyword + FunctionIdentifier \
                           + LParen \
                           + pp.Optional(FunctionFormalParList) \
                           + RParen \
                           + pp.Optional(RunsOnSpec) \
                           + pp.Optional(ReturnType) \
                           + StatementBlock)("FunctionDef")
    # [158] ValueofKeyword ::= "valueof"
    # [157]
    ValueofOp = pp.Group(ValueofKeyword + LParen \
                       + TemplateInstance + RParen)("ValueofOp")
    # [156] MatchKeyword ::= "match"
    # [155]
    MatchOp = pp.Group(MatchKeyword + LParen + Expression \
                       + Comma + TemplateInstance \
                       + RParen)("MatchOp")
    # [154]
    TemplateOps << pp.Group(MatchOp | ValueofOp)("TemplateOps")
    # [153]
    TemplateActualPar = TemplateInstance
    # [152]
    TemplateActualParList = pp.Group(LParen \
                                     + pp.delimitedList(TemplateActualPar) \
                                     + RParen)
    # [151]
    TemplateRefWithParList = pp.Forward()
    DerivedRefWithParList = pp.Group(
        ModifiesKeyword + TemplateRefWithParList)("DerivedRefWithParList")
    # [150]
    InLineTemplate = pp.Group(pp.Optional((Type | Signature) \
                                          + Colon) \
                              + pp.Optional(DerivedRefWithParList \
                                            + AssignmentChar) \
                              + TemplateBody)("InLineTemplate")
    # [149]
    TemplateRef = pp.Group((_ModulePrefix \
                            + TemplateIdentifier) \
                           | TemplateParIdentifier)("TemplateRef")
    # TODO: parse action needed to verify this:
    #TemplateRef = (TemplateIdentifier ^ TemplateParIdentifier)
    # [148]
    TemplateRefWithParList << pp.Group(
        _ModulePrefix \
        + TemplateIdentifier \
        + pp.Optional(TemplateActualParList) \
        | TemplateParIdentifier)("TemplateRefWithParList")
    # [147]
    TemplateInstance << pp.Group(InLineTemplate)("TemplateInstance")
    # [146] InfinityKeyword ::= "infinity"
    # [145]
    UpperBound = pp.Group(InfinityKeyword \
                          | SingleConstExpression)("UpperBound")
    # [144]
    LowerBound = pp.Group(pp.Combine(Minus + InfinityKeyword) \
                          | SingleConstExpression)("LowerBound")
    # [143]
    Range = pp.Group(LParen + LowerBound + ToSymbol \
                     + UpperBound + RParen)("Range")
    # [142] IfPresentKeyword ::= "ifpresent"
    # [141]
    IfPresentMatch = IfPresentKeyword
    # [140]
    StringLength = pp.Forward()
    LengthMatch = StringLength
    # [139] standard says: at least two TemplateBodies - really?
    ValueOrAttribList = pp.Group(LParen \
                                 + pp.delimitedList(TemplateBody) \
                                 + RParen)("ValueOrAttribList")
    # [138]
    AnyOrOmit = pp.Literal("*")("AnyOrOmit")
    # [137]
    AnyValue = pp.Literal("?")("AnyValue")
    # [136]
    PermutationList = pp.Group(LParen \
                               + pp.delimitedList(TemplateBody) \
                               + RParen)
    # [135] PermutationKeyword ::= "permutation"
    # [134]
    PermutationMatch = pp.Group(PermutationKeyword + PermutationList)
    # [133] SupersetKeyword ::= "superset"
    # [132]
    ValueList = pp.Group( # forward
        LParen \
        + pp.delimitedList(ConstantExpression) \
        + RParen)("ValueList")
    SupersetMatch = pp.Group(SupersetKeyword + ValueList)
    # [131] SubsetKeyword ::= "subset"
    # [130]
    SubsetMatch = pp.Group(SubsetKeyword + ValueList)
    # [129] forward
    #ValueList << pp.Group(LParen \
    #                      + pp.delimitedList(ConstantExpression) \
    #                      + RParen)
    # [128] ComplementKeyword ::= "complement"
    # [127]
    Complement = pp.Group(ComplementKeyword + ValueList)("Complement")
    # [126] PatternKeyword ::= "pattern"
    # [125]
    CharStringMatch = pp.Group(PatternKeyword + Cstring)("CharStringMatch")
    # [124] OctOrMatch ::= Oct | AnyValue | AnyOrOmit
    # [123]
    OctetStringMatch = pp.Regex("'[0-9a-fA-F\?\*]*'O")("OctetStringMatch")
    OctetStringMatch.setParseAction(checkOctPair)
    # [122] HexOrMatch ::= Hex | AnyValue | AnyOrOmit
    # [121]
    HexStringMatch = pp.Regex("'[0-9a-fA-F\?\*]*'H")
    # [120] BinOrMatch ::= Bin | AnyValue | AnyOrOmit
    # [119]
    BitStringMatch = pp.Regex("'[01\?\*]*'B'")
    # [118]
    ExtraMatchingAttributes = pp.Group(
        LengthMatch | IfPresentMatch \
        | (LengthMatch + IfPresentMatch))("ExtraMatchingAttributes")
    # [117]
    MatchingSymbol = pp.Group(
        Complement | AnyValue | AnyOrOmit | ValueOrAttribList \
        | Range | BitStringMatch | HexStringMatch \
        | OctetStringMatch | CharStringMatch | SubsetMatch \
        | SupersetMatch)("MatchingSymbol")
    # [116] NotUsedSymbol :== Dash
    # [115]
    ArrayElementSpec = (NotUsedSymbol | PermutationMatch | TemplateBody)
    # [114]
    ArrayElementSpecList = pp.delimitedList(ArrayElementSpec)
    # [113]
    ArrayValueOrAttrib = pp.Group(LBrace + ArrayElementSpecList \
                                  + RBrace)("ArrayValueOrAttrib")
    # [112]
    SingleValueOrAttrib = pp.Group(
        SingleExpression | MatchingSymbol \
        | TemplateRefWithParList)("SingleValueOrAttrib")
    # [111]
    FieldOrBitNumber = pp.Group(SingleExpression)("FieldOrBitNumber")
    # [110]
    ArrayOrBitRef << pp.Group(LBrack + FieldOrBitNumber \
                              + RBrack)("ArrayOrBitRef")
    # [109]
    SignatureParIdentifier = ValueParIdentifier
    # [108]
    ParRef = SignatureParIdentifier
    # [107]
    StructFieldRef = pp.Group(
        StructFieldIdentifier | PredefinedType \
        | TypeReference)("StructFieldRef")
    # [106]
    FieldReference << pp.Group(
        StructFieldRef | ArrayOrBitRef | ParRef)("FieldReference")
    # [105]
    FieldSpec = pp.Group(FieldReference + AssignmentChar \
                         + TemplateBody)
    # [104]
    FieldSpecList = pp.Group(LBrace \
                             + pp.Optional(pp.delimitedList(FieldSpec)) \
                             + RBrace)("FieldSpecList")
    # [103]
    SimpleSpec = pp.Group(SingleValueOrAttrib)("SimpleSpec")
    # [102]
    # Note: standard says "| ExtraMatchingAttributes"!
    TemplateBody << pp.Group(
        (SimpleSpec | FieldSpecList \
                 | ArrayValueOrAttrib) \
        + pp.Optional(ExtraMatchingAttributes))("TemplateBody")
    # [101]
    TemplateFormalPar = pp.Group(FormalValuePar \
                                 | FormalTemplatePar)("TemplateFormalPar")
    # [100]
    TemplateFormalParList = pp.Group(pp.delimitedList(
        TemplateFormalPar))("TemplateFormalParList")
    # [99] ModifiesKeyword ::= "modifies"
    # [98]
    DerivedDef = pp.Group(ModifiesKeyword + TemplateRef)("DerivedDef")
    # [97] TemplateIdentifier ::= Identifier
    # [96] TemplateKeyword ::= "template"
    # [95]
    BaseTemplate = pp.Group(Type + TemplateIdentifier \
                            + pp.Optional(LParen \
                                          + TemplateFormalParList \
                                          + RParen))("BaseTemplate")
    # [94]
    TemplateDef << pp.Group(TemplateKeyword + BaseTemplate \
                            + pp.Optional(DerivedDef) + AssignmentChar \
                            + TemplateBody)("TemplateDef")
    # [93] ConstIdentifier ::= Identifier
    # [92] ConstKeyword ::= "const"
    # [91]
    SingleConstDef = pp.Group(
        ConstIdentifier + pp.Optional(ArrayDef) \
        + AssignmentChar + ConstantExpression)("SingleConstDef")
    # [90]
    ConstList = pp.delimitedList(SingleConstDef)("ConstList")
    # [89]
    ConstDef << pp.Group(ConstKeyword + Type + ConstList)("ConstDef")
    # [88] PortIdentifier ::= Identifier
    # [87]
    PortElement = pp.Group(PortIdentifier \
                           + pp.Optional(ArrayDef))("PortElement")
    # [86]
    PortType = pp.Forward()
    PortInstance = pp.Group(PortKeyword + PortType \
                            + pp.delimitedList(PortElement))("PortInstance")
    # [85]
    ComponentElementDef = pp.Group(
        PortInstance | VarInstance \
        | TimerInstance | ConstDef)("ComponentElementDef")
    # [84]
    ComponentDefList = pp.ZeroOrMore(ComponentElementDef \
                                     + _opt_semicolon)("ComponentDefList")
    # [83] ComponentTypeIdentifier ::= Identifier
    # [82]
    ComponentType << pp.Group(_ModulePrefix \
                              + ComponentTypeIdentifier)("ComponentType")
    # [81] ExtendsKeyword ::= "extends"
    # [80] ComponentKeyword ::= "component"
    # [79]
    ComponentDef = pp.Group(ComponentKeyword + ComponentTypeIdentifier \
                            + pp.Optional(ExtendsKeyword \
                                          + pp.delimitedList(ComponentType)) \
                            + LBrace \
                            + pp.Optional(ComponentDefList) \
                            + RBrace)("ComponentDef")
    # [78]
    ProcOrType = (Signature | Type)
    # [77]
    ProcOrTypeList = (AllKeyword | pp.delimitedList(ProcOrType))
    # [76]
    Direction = pp.Group( # forward
        InParKeyword | OutParKeyword | InOutParKeyword)("Direction")
    MixedList = pp.Group(Direction + ProcOrTypeList)("MixedList")
    # [75] MixedKeyword ::= "mixed"
    # [74]
    MixedAttribs = pp.Group(MixedKeyword + LBrace \
                            + pp.OneOrMore(MixedList \
                                           + _opt_semicolon) \
                            + RBrace)("MixedAttribs")
    # [73]
    SignatureList = pp.delimitedList(Signature)
    # [72]
    AllOrSignatureList = pp.Group(
        AllKeyword | SignatureList)("AllOrSignatureList")
    # [71]
    ProcedureList = pp.Group(
        Direction + AllOrSignatureList)("ProcedureList")
    # [70] ProcedureKeyword ::= "procedure"
    # [69]
    ProcedureAttribs = pp.Group(ProcedureKeyword + LBrace \
                                + pp.OneOrMore(ProcedureList \
                                               + _opt_semicolon) \
                                + RBrace)("ProcedureAttribs")
    # [68]
    TypeList = pp.delimitedList(Type)
    # [67] AllKeyword ::= "all"
    # [66]
    AllOrTypeList = pp.Group(AllKeyword | TypeList)("AllOrTypeList")
    # [65] MessageKeyword ::= "message"
    # [64] forward
    #Direction << pp.oneOf("in out inout")
    # [63]
    MessageList = pp.Group(Direction + AllOrTypeList)("MessageList")
    # [62]
    MessageAttribs = pp.Group(MessageKeyword + LBrace \
                              + pp.OneOrMore(MessageList \
                                             + _opt_semicolon) \
                              + RBrace)("MessageAttribs")
    # [61]
    PortDefAttribs = pp.Group(MessageAttribs | ProcedureAttribs \
                              | MixedAttribs)("PortDefAttribs")
    # [60] PortTypeIdentifier ::= Identifier
    # [59] PortKeyword ::= "port"
    # [58]
    PortDefBody = pp.Group(PortTypeIdentifier + PortDefAttribs)("PortDefBody")
    # [57]
    PortDef = pp.Group(PortKeyword + PortDefBody)("PortDef")
    # [56]
    PortType << pp.Group(_ModulePrefix \
                         + PortTypeIdentifier)("PortType")
    # [55] LengthKeyword ::= "length"
    # [54]
    StringLength << pp.Group(LengthKeyword + LParen \
                             + SingleConstExpression \
                             + pp.Optional(ToSymbol + UpperBound) \
                             + RParen)("StringLength")
    # [53]
    RangeDef = pp.Group(LowerBound + ToSymbol \
                        + UpperBound)("RangeDef")
    # [52]
    ValueOrRange = pp.Group(RangeDef | SingleConstExpression)("ValueOrRange")
    # [51]
    AllowedValues = pp.Group(LParen \
                             + (pp.delimitedList(ValueOrRange) \
                                | CharStringMatch ) \
                             + RParen)("AllowedValues")
    # [49] SubTypeIdentifier ::= Identifier
    # [50]
    SubTypeSpec = pp.Group(AllowedValues + pp.Optional(StringLength) \
                           | StringLength)("SubTypeSpec")
    # [48]
    SubTypeDef = pp.Group(Type + (AddressKeyword \
                                  | SubTypeIdentifier) \
                          + pp.Optional(ArrayDef) \
                          + pp.Optional(SubTypeSpec))("SubTypeDef")
    # [47] EnumerationIdentifier ::= Identifier
    # [46]
    Enumeration = pp.Group(EnumerationIdentifier \
                           + pp.Optional(LParen \
                                         + pp.Optional(Minus) \
                                         + Number \
                                         + RParen))("Enumeration")
    # [45]
    EnumerationList = pp.Group(
        pp.delimitedList(Enumeration))("EnumerationList")
    # [44] EnumTypeIdentifier ::= Identifier
    # [43] EnumKeyword ::= "enumerated"
    # [42]
    EnumDef = pp.Group(EnumKeyword + (AddressKeyword \
                                      | EnumTypeIdentifier) \
                       + LBrace + EnumerationList + RBrace)("EnumDef")
    # [41]
    StructOfDefBody = pp.Forward()
    SetOfDef = pp.Group(SetKeyword + pp.Optional(StringLength) + OfKeyword \
                        + StructOfDefBody)("SetOfDef")
    # [40]
    NestedTypeDef = pp.Forward()
    StructOfDefBody << pp.Group(
        (Type | NestedTypeDef) \
        + (AddressKeyword \
           | StructTypeIdentifier) \
        + pp.Optional(SubTypeSpec))("StructOfDefBody")
    # [39] OfKeyword ::= "of"
    # [38]
    RecordOfDef = pp.Group(RecordKeyword + pp.Optional(StringLength) \
                           + OfKeyword + StructOfDefBody)("RecordOfDef")
    # [37] SetKeyword ::= "set"
    # [36]
    StructDefBody = pp.Forward()
    SetDef = pp.Group(SetKeyword + StructDefBody)("SetDef")
    # [35]
    UnionFieldDef = pp.Group((Type | NestedTypeDef) \
                             + StructFieldIdentifier \
                             + pp.Optional(ArrayDef) \
                             + pp.Optional(SubTypeSpec))("UnionFieldDef")
    # [34]
    StructDefFormalParList = pp.Forward()
    UnionDefBody = pp.Group(
        (AddressKeyword \
         | (StructTypeIdentifier \
            + pp.Optional(StructDefFormalParList))) \
        + LBrace + pp.delimitedList(UnionFieldDef) \
        + RBrace)("UnionDefBody")
    # [33] UnionKeyword ::= "union"
    # [32]
    UnionDef = pp.Group(UnionKeyword + UnionDefBody)("UnionDef")
    # [31] OptionalKeyword ::= "optional"
    # [30] StructFieldIdentifier ::= Identifier
    # [29]
    NestedEnumDef = pp.Group(EnumKeyword + LBrace \
                             + EnumerationList + RBrace)
    # [28]
    NestedSetOfDef = pp.Group(SetKeyword + pp.Optional(StringLength) \
                              + OfKeyword + (Type | NestedTypeDef))
    # [27]
    NestedRecordOfDef = pp.Group(RecordKeyword + pp.Optional(StringLength) \
                                 + OfKeyword + (Type | NestedTypeDef))
    # [26]
    StructFieldDef = pp.Forward()
    NestedSetDef = pp.Group(SetKeyword + LBrace \
                            + pp.Optional(pp.delimitedList(StructFieldDef)) \
                            + RBrace)
    # [25]
    NestedUnionDef = pp.Group(UnionKeyword + LBrace \
                              + pp.Optional(pp.delimitedList(UnionFieldDef)) \
                              + RBrace)
    # [24]
    NestedRecordDef = pp.Group(
        RecordKeyword + LBrace \
        + pp.Optional(pp.delimitedList(StructFieldDef)) \
        + RBrace)("NestedRecordDef")
    # [23]
    NestedTypeDef << pp.Group(
        NestedRecordDef | NestedUnionDef | NestedSetDef \
        | NestedRecordOfDef | NestedSetOfDef | NestedEnumDef)("NestedTypeDef")
    # [22]
    StructFieldDef << pp.Group(
        (Type | NestedTypeDef) \
        + StructFieldIdentifier \
        + pp.Optional(ArrayDef) \
        + pp.Optional(SubTypeSpec) \
        + pp.Optional(OptionalKeyword))("StructFieldDef")
    # [21]
    StructDefFormalPar = FormalValuePar
    # [20]
    StructDefFormalParList << pp.Group(
        LParen \
        + pp.delimitedList(StructDefFormalPar) \
        + RParen)
    # [19] StructTypeIdentifier ::= Identifier
    # [18]
    StructDefBody << pp.Group(
        (AddressKeyword \
         | (StructTypeIdentifier + \
            pp.Optional(StructDefFormalParList))) \
        + LBrace + pp.Optional(pp.delimitedList(StructFieldDef))
        + RBrace)("StructDefBody")
    # [17] RecordKeyword ::= "record"
    # [16]
    RecordDef = pp.Group(RecordKeyword + StructDefBody)("RecordDef")
    # [15]
    StructuredTypeDef = pp.Group(
        RecordDef | UnionDef | SetDef \
        | RecordOfDef | SetOfDef \
        | EnumDef | PortDef | ComponentDef)("StructuredTypeDef")
    # [14] TypeDefKeyword ::= "type"
    # [13]
    TypeDefBody = pp.Group(StructuredTypeDef | SubTypeDef)("TypeDefBody")
    # [12]
    TypeDef = pp.Group(TypeDefKeyword + TypeDefBody)("TypeDef")
    # [11]
    ModuleDefinition = pp.Group(
        (TypeDef | ConstDef | TemplateDef | ModuleParDef \
         | FunctionDef | SignatureDef | TestcaseDef \
         | AltstepDef | ImportDef | GroupDef | ExtFunctionDef \
         | ExtConstDef) \
        + pp.Optional(WithStatement))("ModuleDefinition")
    # [10]
    ModuleDefinitionsList = pp.Group(
        pp.OneOrMore(ModuleDefinition \
                     + _opt_semicolon))("ModuleDefinitionsList")
    # [9]
    ModuleDefinitionsPart << pp.Group(
        ModuleDefinitionsList)("ModuleDefinitionsPart")
    # [8] LanguageKeyword ::= "language"
    # [7]
    LanguageSpec = pp.Group(LanguageKeyword + FreeText)("LanguageSpec")
    # [6] ModuleIdentifier ::= Identifier
    # [5]
    GlobalModuleId << pp.Group(ModuleIdentifier)("GlobalModuleId")
    GlobalModuleId.setParseAction(register_gmi)
    # [4]
    ModuleId << pp.Group(GlobalModuleId \
                         + pp.Optional(LanguageSpec))("ModuleId")
    # [3]
    TTCN3ModuleId = pp.Group(ModuleId)("TTCN3ModuleId")
    # [2] TTCN3ModuleKeyword ::= "module"
    # [1]
    TTCN3Module = (TTCN3ModuleKeyword + TTCN3ModuleId \
                   + LBrace \
                   + pp.Optional(ModuleDefinitionsPart) \
                   + pp.Optional(ModuleControlPart) \
                   + RBrace \
                   + pp.Optional(WithStatement) \
                   + _opt_semicolon)("TTCN3Module")
    bnf = (pp.OneOrMore(TTCN3Module) + pp.StringEnd())
    singleLineComment = "//" + pp.restOfLine
    allComments = pp.FollowedBy("/") + (pp.cStyleComment | singleLineComment)
    bnf.ignore(allComments)

    # add tally by expression
    # capitalizedWord = pp.Word(pp.alphas.upper(), pp.alphanums)
    if debug:
        for exprName in locals().keys():
            if isinstance(locals()[exprName], pp.ParserElement):
                expr = locals()[exprName]
                expr.setDebugActions(*tallyKeyword(exprName))

    return bnf
