/* ownerstring.c -- view/change the owner string on a Toshiba laptop.
 *
 * Copyright (c) 1998-2001  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log:$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static const char rcsid[]="$Id:$";

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif
#include<signal.h>
#include<paths.h>

#include"sci.h"
#include"hci.h"

#define PID_FILE _PATH_VARTMP ".ownerstring.pid"

#define USAGE \
"This utility can be used to set and change the owner string.\n\n\
Usage: ownerstring [-p|r|s]\n\
    -p : Display the current owner string on stdout.\n\
    -r : remove the current owener string\n\
    -s : Set the owener string using the input on stdin.\n"


/*
 * Catch any signals and exit the program in nicely.
 */
void Catch(int x)
{
	fprintf(stderr, "ownerstring: caught signal %d, cleaning up...\n", x);
	unlink(PID_FILE);
	exit(1);

	return;
}


/*
 * check the laptop supports the owner string feature
 */
int OwnerStringSupport(void)
{
	SMMRegisters reg;

	reg.eax = HCI_GET;
	reg.ebx = HCI_OWNERSTRING;
	reg.esi = 0x0000;

	return HciFunction(&reg);
}


/*
 * Read owner string into a null terminated string, return NULL on error
 */
char *GetOwnerString(void)
{
	SMMRegisters reg;
	char *owner,*string;
	unsigned long characters;
	int i,j,c,length;

	/* how big is the current owner string */

	reg.eax = HCI_GET;
	reg.ebx = HCI_OWNERSTRING;
	reg.esi = 0x0000;
	HciFunction(&reg);
	length = (reg.ecx & 0xffff0000)>>16;
	owner = calloc(length, sizeof(char));
	if (owner==NULL) {
		fprintf(stderr, "UNABLE TO MALLOC SPACE");
		return NULL;
	}

	/* read the current owner string four characters at a time */

	string = owner;
	for (i=0;i<length;i+=4) {
		reg.eax = HCI_GET;
		reg.ebx = 0x29;
		reg.esi = (unsigned long) i;
		if (HciFunction(&reg)!=0x00) {
			fprintf(stdout, "ERROR : reading ownerstring\n");
			break;
		}

		/* add non NULL characters to the string */

		characters = reg.edx;
		for (j=0;j<4;j++) {
			c = (char) characters & 0xff;
			characters = characters >> 8;
		        if ((c>=0x20) && (c<=0x80))
				*(string++) = c;
			if (c==0x0d)
				*(string++) = c;
			if (c==0x00)
				*(string++) = c;
		}
	}
	*string = '\0';

	return owner;
}


/*
 * Store a new owner string.
 */
int SetOwnerString(char *owner)
{
	SMMRegisters reg;
	char string[513];
	int i;

	/* copy at most 512 characters of the input string */

	memset(string, 0, sizeof(char));
	strncpy(string, owner, 512);

	/* set the new string */

	for (i=0;i<512;i+=8) {
		reg.eax = HCI_SET;
		reg.ebx = HCI_OWNERSTRING;
		reg.ecx = 8;
		reg.esi = i;

		reg.edx = (int) string[i];
		reg.edx |= ((int) string[i+1]) << 8;
		reg.edx |= ((int) string[i+2]) << 16;
		reg.edx |= ((int) string[i+3]) << 24;
		
		reg.edi = (int) string[i+4];
		reg.edi |= ((int) string[i+5]) << 8;
		reg.edi |= ((int) string[i+6]) << 16;
		reg.edi |= ((int) string[i+7]) << 24;

		if (HciFunction(&reg)!=0x00) {
			fprintf(stdout, "ERROR : writing ownerstring\n");
			return 1;
		}
	}

	return 0;
}



int main(int argc, char *argv[])
{
	FILE *str;
	int i,pid,version,print,remove,set;
	char c,*owner,new[513];


	/* this program *must* be run as root */

	if (getuid()) {
		fprintf(stderr, "ownerstring: must be run as root.\n");
		return 1;
	}

	/* is this a Toshiba laptop? */

	if (SciSupportCheck(&version)==SCI_FAILURE) {
		fprintf(stderr, "ownerstring: this computer is not supported "
			"or the kernel module is not installed.\n");
		return 1;
	}

	/* check this model supports an owner string */

	if (OwnerStringSupport()!=0x00) {
		fprintf(stderr, "ownerstring: this computer does not support "
			"the ownerstring feature.\n");
		return 1;
	} 

	/* process command line arguments */

	print = 0;
	set = 0;
	remove = 0;
	if (argc==1) {
		printf(USAGE);
		return 0;
	}

	if (!strcmp(argv[1], "-p") || !strcmp(argv[1], "--print")) {
		print = 1;
	} else if (!strcmp(argv[1], "-r") || !strcmp(argv[1], "--remove")) {
		remove = 1;
	} else if (!strcmp(argv[1], "-s") || !strcmp(argv[1], "--set")) {
		set = 1;
	} else {
		printf("ownerstring: illegal option %s\n", argv[1]);
		return 1;
	}

	/* if we recieve a signal, exit cleanly */

	signal(SIGHUP, Catch);
	signal(SIGINT, Catch);
	signal(SIGQUIT, Catch);
	signal(SIGILL, Catch);
	signal(SIGTRAP, Catch);
	signal(SIGABRT, Catch);
	signal(SIGIOT, Catch);
	signal(SIGFPE, Catch);
	signal(SIGKILL, Catch);
	signal(SIGSEGV, Catch);
	signal(SIGPIPE, Catch);
 	signal(SIGTERM, Catch);
	signal(SIGCHLD, Catch);
	signal(SIGCONT, Catch);
	signal(SIGSTOP, Catch);
	signal(SIGTSTP, Catch);
	signal(SIGTTIN, Catch);
	signal(SIGTTOU, Catch);

	/* check to see if a copy of ownerstring is already running */

	if (!access(PID_FILE, R_OK)) {
		if ((str = fopen(PID_FILE, "r" ))) {
			fscanf(str, "%d", &pid);
			fclose(str);
			fprintf(stderr, "ownerstring: Already running as "
				"process %d.\n", pid);
			return 1;
		}
	}

	/* create the pid file */

	pid = getpid();
	if ((str = fopen(PID_FILE, "w"))) {
		fprintf(str, "%d\n", pid);
		fclose(str);
	}
	
	if (print==1) {
		owner = GetOwnerString();
		printf("%s\n", owner);
	} else if (remove==1) {
		memset(new, 0, sizeof(char)*513);
		SetOwnerString(new);
	} else if (set==1) {
		i = 0;
		memset(new, 0, sizeof(char)*513);

		/* read the new owner string from stdin */

		while ((i<512) && ((c=getchar())!=EOF)) {
			new[i] = c;
			i++;
		}
		
		SetOwnerString(new);
	}

	unlink(PID_FILE);

	return 0;
}
