// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#include "ParzenDistribution.h"
#include "log_add.h"
#include "random.h"
#include "SeqDataSet.h"

namespace Torch {

ParzenDistribution::ParzenDistribution(SeqDataSet* data_, real var_) : Distribution()
{
  data = data_;

  n_observations = data->n_observations;
  n_inputs = data->n_inputs;

  setVar(var_);

  n_real_examples = 0;
  real_examples = NULL;
}

void ParzenDistribution::setVar(real var_)
{
  var = var_;

  sum_log_var_plus_n_obs_log_2_pi = -0.5 * n_observations*(LOG_2_PI + log(var));
  minus_half_over_var = -0.5 / var;
}

void ParzenDistribution::allocateMemory()
{
  max_n_frames = 1;
  n_params = numberOfParams();
  addToList(&outputs,n_outputs,(real*)xalloc(sizeof(real)*n_outputs));
  log_probabilities = (real*)xalloc(sizeof(real)*max_n_frames);
}

void ParzenDistribution::freeMemory()
{
  freeList(&outputs,true);
  free(log_probabilities);
}

int ParzenDistribution::numberOfParams()
{
  return 0;
}

void ParzenDistribution::reset()
{
  n_real_examples = data->n_examples;
  real_examples = (int*)xrealloc(real_examples,n_real_examples*sizeof(int));
  for (int i=0;i<data->n_examples;i++) {
    real_examples[i] = data->selected_examples[i];
  }
}

void ParzenDistribution::eMSequenceInitialize(List* inputs)
{
  if (!inputs)
    return;
  SeqExample* ex = (SeqExample*)inputs->ptr;
  if (ex->n_real_frames > max_n_frames) {
    max_n_frames = ex->n_real_frames;
    log_probabilities = (real*)xrealloc(log_probabilities,sizeof(real)*max_n_frames);
  }
}

void ParzenDistribution::sequenceInitialize(List* inputs)
{
  eMSequenceInitialize(inputs);
}

real ParzenDistribution::frameLogProbability(real *observations, real *inputs, int t)
{
  // first keep the current pointers...
  int current_example = data->current_example;
  int current_frame = data->current_frame;

  // then compute the likelihood...
  real lp = LOG_ZERO;
  tot_n_frames = 0;
  int *i_ptr = real_examples;
  for (int i=0;i<n_real_examples;i++) {
    data->setRealExample(*i_ptr++);
    tot_n_frames += data->n_frames;
    for (int j=0;j<data->n_frames;j++) {
      data->setFrame(j);
      real lp_ij = frameLogProbabilityOneFrame(observations,data->examples[data->current_example].observations[data->current_frame]);
      lp = log_add(lp, lp_ij);
    }
  }
  lp -= log((real)tot_n_frames);
  log_probabilities[t] = lp;

  // restore the dataset status
  data->setRealExample(current_example);
  data->setFrame(current_frame);

  return lp;
}

real ParzenDistribution::frameLogProbabilityOneFrame(real *observations, real *mean)
{
  real sum_xmu = 0.;
  real *x = observations;
  for(int j = 0; j < n_observations; j++) {
    real xmu = (*x++ - *mean++);
    sum_xmu += xmu*xmu;
  }
  real lp = sum_xmu*minus_half_over_var + sum_log_var_plus_n_obs_log_2_pi;
  return lp;
}

void ParzenDistribution::frameExpectation(real *observations, real *inputs, int t)
{
  real *obs = observations;
  for (int i=0;i<n_observations;i++) {
    *obs++ = 0;
  }
}


ParzenDistribution::~ParzenDistribution()
{
  free(real_examples);
  freeMemory();
}

}

