#
# TCL Library for TkCVS
#

#
# $Id: modules.tcl,v 1.24 2002/01/23 07:01:38 dorothyr Exp $
#
# Procedures to parse the CVS modules file and store whatever is
# read into various associative arrays, sorted, and unsorted lists.
#

#
# Global variables:
#
# modval
#   The string that specifies or defines the module.
# modtitle
#   The descriptive title of the module.  If not specified, modval is used.
# cvscfg
#   General configuration variables (array)
# filenames
#   For each module, the list of files that it contains.

proc read_modules {root} {
#
# Read one pass through the modules file.
#
  global cvs
  global modval
  global modtitle
  global cvsglb
  global cvscfg

  gen_log:log T "ENTER ($root)"

  # Clear the arrays
  catch {unset modval}
  catch {unset modtitle}

  # You can get a list of the modules, sans comments, by doing
  # cvs -d $CVSROOT checkout -c.  Cvs will do whatever parsing,
  # such as line-continuation, it does.
  set commandline "$cvs -d $root checkout -c"
  gen_log:log C $commandline
  set ret [catch {eval "exec $commandline"} view_this]
  if {$ret} {
    gen_log:log D "cvs checkout -c returned $ret"
    cvsfail "$view_this"
  }
  # checkout -c output may be line-wrapped.  Un-wrap it here.
  regsub -all {\n\s+} $view_this " " view_this

  set mod_lines [split $view_this "\n"]
  foreach line $mod_lines {
    set text [split $line]
    set checkname [lindex $text 0]
    if {[string length $checkname]} {
      set modname $checkname
      set modstring [string trim [join [lrange $text 1 end]]]
      set modtitle($modname) $modstring
      # Remove flags except for -a.  Luckily alias modules can't have
      # any other options.
      #gen_log:log D "{$modname} {$modstring}"
      while {[regexp -- {^-[^a]} $modstring]} {
        regsub -- {-[ioestudl]\s+\S+\s*} $modstring {} modstring
      }
      gen_log:log D "{$modname} {$modstring}"
      set modval($modname) $modstring
    } else {
      set modtitle($modname) $modstring
      append modtitle($modname) "..."
      #gen_log:log D "{$modname} {$modtitle($modname)}"
    }
  }

  read_module_extensions $root

  gen_log:log T "LEAVE"
}

proc read_module_extensions {root} {
# Read another pass through the modules file to get the extensions.
  global modtitle
  global cvs
  global cvscfg

  gen_log:log T "ENTER ($root)"

  # We need to see the actual modules file
  set modulefile $cvscfg(tmpdir)/modules-[pid]
  set commandline "$cvs -d $root checkout -p CVSROOT/modules > $modulefile"
  gen_log:log C $commandline
  set ret [catch {eval "exec $commandline"} view_this]
  # should only fail if the checkout has been aborted
  if {$ret} {
    gen_log:log D "cvs checkout returned $ret"
    if {[string match "* aborted*" $view_this]} {
      cvsfail $view_this
    }
  }

  gen_log:log F "OPEN $modulefile"
  set mf [open $modulefile]
  while {[gets $mf line] >= 0} {
    # Split and parse the line
    if {[regexp {^#[DM]} $line]} {
      regsub -all {\s+} $line " " line
      set text [split $line]
      set dname [lindex $text 1]
      set modtitle($dname) [lrange $text 2 end]
      #gen_log:log D "Directory: {$dname} {$modtitle($dname)}"
    }
  }
  close $mf
  file delete -force $modulefile
  gen_log:log F "DELETE $modulefile"
  gen_log:log T "LEAVE"
}

proc gather_mod_index {} {
#
# Creates a new global list called modlist for the report printouts
#
  global cvscfg
  global modtitle
  global dcontents
  global dparent
  global modlist
  global modlist_sorted

  gen_log:log T "ENTER ()"
  set modlist {}
  set dlist {}
  if {! [info exists modtitle]} {
    gen_log:log T "LEAVE (no modtitle array)"
    return
  }
  foreach d [array names dcontents] {
    #gen_log:log D "dcontents($d) is $dcontents($d)"
    foreach i $dcontents($d) {
      lappend dlist $i
      set path [file join $d $i]
      set dparent($path) $d
      #gen_log:log D "dparent($path) is $d"
    }
  }
  foreach mcode [array names modtitle] {
    # Skip aliases
    if {[string match "-a *" $modtitle($mcode)]} {
      continue
    }
    # Dont add subdirs to the list
    set match 0
    foreach i $dlist {
      if {$i == $mcode} {
        set match 1
      }
    }
    if {! $match} {
      lappend modlist "$mcode\t$modtitle($mcode)"
    }
  }

  set modlist_sorted [lsort $modlist]
  if {$cvscfg(logging) && [regexp -nocase {d} $cvscfg(log_classes)]} {
    foreach idx $modlist_sorted {
      gen_log:log D "$idx"
      set dname [lindex $idx 0]
      if {[info exists dparent($dname)]} {
        gen_log:log D "   PARENT: $dparent($dname)"
      }
      if {[info exists dcontents($dname)]} {
        gen_log:log D "   CHILDREN: $dcontents($dname)"
      }
      set desc [find_subdirs $dname 0]
      if {$desc != ""} {
        gen_log:log D "   SUBDIRS: $desc"
      }
    }
  }
  gen_log:log T "LEAVE"
}

proc find_filenames {mcode} {
#
# This does the work of setting up the filenames array for a module,
# containing the list of file names within it.
#
  global filenames
  global cwd
  global cvs
  global cvsglb
  global cvscfg
  global checkout_version
  global mod_tagA
  global feedback

  gen_log:log T "ENTER ($mcode)"

  if {[info exists filenames($mcode)]} {
    set filenames($mcode) ""
  }

  feedback_cvs $feedback(mod) "Building file list, please wait!"

  if {[info exists checkout_version] && $checkout_version != {} } {
    set rev $checkout_version
  } elseif {[info exists mod_tagA] && $mod_tagA != {} } {
    set rev $mod_tagA
  } else {
    set rev HEAD
  }

  # Trick of using rdiff to list files without checking them out
  # derived from "cvsls" by Eugene Kramer
  # cvs 1.9:
  #  Need to use -f with pserver, or it skips files that havent
  #  changed.  With local repository, it reports them as new.
  # But without pserver, it skips them with -f but not without!
  # cvs 1.10.8:
  #  Both pserver and local act like 1.9 local, that is, -f makes
  #  it skip new files.
  set commandline \
     "$cvs -d $cvscfg(cvsroot) rdiff -s -D 01/01/1971 -r $rev $mcode"
  gen_log:log C  $commandline
  catch {eval "exec $commandline"} view_this
   
  set view_lines [split $view_this "\n"]
  foreach line $view_lines {
    gen_log:log D "$line"
    if {[string match "File *" $line]} {
      set lst [split $line]
      set cut [expr {[llength $lst] - 6}]
      set dname [join [lrange $lst 1 $cut]]
      gen_log:log D "$dname"
      lappend filenames($mcode) $dname
    }
  }
  feedback_cvs $feedback(mod) ""
  gen_log:log T "LEAVE"
}

proc find_subdirs {mname level} {
  global dcontents
  global subdirs

  #gen_log:log T "ENTER ($mname $level)"
  if {$level == 0} {
    set subdirs {}
  }
  if {[info exists dcontents($mname)]} {
    #gen_log:log D "$mname contents: {$dcontents($mname)}"
    foreach d $dcontents($mname) {
      set path [file join $mname $d]
      if {[info exists dcontents($path)]} {
        lappend subdirs $path
      }
      find_subdirs $path 1
    }
  }
  #gen_log:log T "LEAVE ($subdirs)"
  return $subdirs
}
