/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>

#include <iterator>

#include <gtk--/alignment.h>
#include <gtk--/button.h>
#include <gtk--/frame.h>
#include <gtk--/label.h>
#include <gtk--/pixmap.h>
#include <gtk--/scrolledwindow.h>
#include <gtk--/table.h>
#include <gtk--/toolbar.h>
#include <gtk--/tooltips.h>

#include "thoughttracker.h"
#include "app.h"
#include "database.h"
#include "menubar.h"
#include "querydlg.h"
#include "search.h"
#include "xpm.h"

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTSearchWidget::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

using SigC::slot;

using namespace Gtk;

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTSearchWidget::TTSearchWidget(TTApplication *application)
  : VBox(), app(application), selected(-1)
{
  // configure tool bar
  {
    using namespace Toolbar_Helpers;

    // build it backwards
    Toolbar *tbar = manage(new Toolbar(GTK_ORIENTATION_HORIZONTAL,
      GTK_TOOLBAR_BOTH));
    tbar->set_button_relief(GTK_RELIEF_NONE);
    tbar->set_space_style(GTK_TOOLBAR_SPACE_LINE);
    pack_start(*tbar, false, false, false);

    tbar->tools().push_front(ButtonElem(_("Clear"),
      *manage(new Pixmap(icon_clear_xpm)),
      slot(this, &TTSearchWidget::btn_clear),
      _("clear the entire form")));

    tbar->tools().push_front(Space());
    Pixmap *icon = manage(new Pixmap(icon_del_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Delete"), *icon,
      slot(this, &TTSearchWidget::btn_del),
      _("delete selected entry")));
    w.b.del = (*tbar->tools().begin())->get_widget();
    tbar->tools().push_front(ButtonElem(_("New..."),
      *manage(new Pixmap(icon_add_xpm)),
      slot(this, &TTSearchWidget::btn_add),
      _("add a new entry")));

    tbar->tools().push_front(Space());
    icon = manage(new Pixmap(icon_link_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Link"), *icon,
      slot(this, &TTSearchWidget::btn_link),
      _("link active entry to selected one")));
    w.b.link =(*tbar->tools().begin())->get_widget();

    tbar->tools().push_front(Space());
    icon = manage(new Pixmap(icon_show_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Show..."), *icon,
      slot(this, &TTSearchWidget::btn_show),
      _("show the selected entry")));
    w.b.show = (*tbar->tools().begin())->get_widget();
    tbar->tools().push_front(ButtonElem(_("Search"),
      *manage(new Pixmap(icon_search_xpm)),
      slot(this, &TTSearchWidget::btn_search),
      _("perform the search")));
  }

  // configure frame
  Frame *frame = manage(new Frame(_("Search")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  frame->set_border_width(5);
  pack_start(*frame, true, true, false);

  // configure vbox in frame
  VBox *vbox = manage(new VBox);
  vbox->set_border_width(5);
  vbox->set_spacing(5);
  frame->add(*vbox);

  // configure search string entry field plus label in vbox
  Box *box = manage(new HBox(0, 3));
  vbox->pack_start(*box, false, false, false);
  Label *label = manage(new Label(_("Search string:")));
  box->pack_start(*label, false, false, false);
  w.text = manage(new Entry);
  w.text->key_press_event.connect(slot(this,
    &TTSearchWidget::handle_ss_return));
  box->pack_start(*w.text, true, true, false);

  // configure table with subframes and radio button groups for search options
  Table *table = manage(new Table(3, 2, 0));
  vbox->pack_start(*table, false, false, false);

  // configure frame with radio button group for search contents selection
  frame = manage(new Frame(_("Contents")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  table->attach(*frame, 0, 1, 0, 1, GTK_EXPAND, 0, 3, 0);
  box = manage(new VBox);
  box->set_border_width(5);
  frame->add(*box);
  RadioButton *rbtn = manage(new RadioButton(contents,
    _("summaries and details")));
  w.b.c_all = rbtn;
  box->pack_start(*rbtn, false, false, false);
  rbtn = manage(new RadioButton(contents,
    _("summaries only")));
  box->pack_start(*rbtn, false, false, false);

  // configure frame with radio button group for search logic selection
  frame = manage(new Frame(_("Logic")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  table->attach(*frame, 1, 2, 0, 1, GTK_EXPAND, 0, 3, 0);
  box = manage(new VBox);
  box->set_border_width(5);
  frame->add(*box);
  w.b.l_and = rbtn = manage(new RadioButton(logic,
    _("conjunctive (AND)")));
  box->pack_start(*rbtn, false, false, false);
  rbtn = manage(new RadioButton(logic,
    _("disjunctive (OR)")));
  box->pack_start(*rbtn, false, false, false);

  // configure frame with radio button group for attributes selection
  frame = manage(new Frame(_("Attributes")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  table->attach(*frame, 2, 3, 0, 2, GTK_EXPAND, GTK_FILL, 3, 0);
  box = manage(new VBox);
  box->set_border_width(5);
  frame->add(*box);
  Alignment *align = manage(new Alignment(0.0, 0.5, 0.0, 0.0));
  box->pack_start(*align, false, false, false);
  w.b.t_hub = manage(new CheckButton(_("hubs")));
  align->add(*w.b.t_hub);
  w.b.t_hub->toggled.connect(slot(this, &TTSearchWidget::handle_t_hub));
  w.b.t_hub->set_active(true);
  align = manage(new Alignment(0.0, 0.5, 0.0, 0.0));
  box->pack_start(*align, false, false, false);
  w.b.t_ent = manage(new CheckButton(_("regular entries")));
  align->add(*w.b.t_ent);
  w.b.t_ent->toggled.connect(slot(this, &TTSearchWidget::handle_t_ent));
  w.b.t_ent->set_active(true);
  align = manage(new Alignment(0.0, 0.5, 0.0, 0.0));
  box->pack_start(*align, false, false, false);
  w.b.t_lnk = manage(new CheckButton(_("linked")));
  align->add(*w.b.t_lnk);
  w.b.t_lnk->toggled.connect(slot(this, &TTSearchWidget::handle_t_lnk));
  w.b.t_lnk->set_active(true);
  align = manage(new Alignment(0.0, 0.5, 0.0, 0.0));
  box->pack_start(*align, false, false, false);
  w.b.t_iso = manage(new CheckButton(_("isolated")));
  align->add(*w.b.t_iso);
  w.b.t_iso->toggled.connect(slot(this, &TTSearchWidget::handle_t_iso));
  w.b.t_iso->set_active(true);

  // create search results list label
  align = manage(new Alignment(0.5, 1.0, 0.0, 0.0));
  table->attach(*align, 0, 2, 1, 2, GTK_EXPAND | GTK_FILL,
    GTK_EXPAND | GTK_FILL, 0, 3);
  label = manage(new Label(_("Results:")));
  align->add(*label);

  // configure search results list
  ScrolledWindow *swin = manage(new ScrolledWindow);
  swin->set_policy(GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  vbox->pack_start(*swin, true, true, false);
  w.list = manage(new TTLinkList);
  w.list->select_row.connect(slot(this, &TTSearchWidget::handle_select_entry));
  w.list->unselect_row.connect(slot(this,
    &TTSearchWidget::handle_unselect_entry));
  w.list->key_press_event.connect(slot(this,
    &TTSearchWidget::handle_srl_return));
  swin->add(*w.list);

  // configure active entry field and button
  box = manage(new HBox);
  vbox->pack_start(*box, false, false, false);
  w.b.active = manage(new Button(_("Active entry:")));
  w.b.active->set_relief(GTK_RELIEF_NONE);
  manage(new Tooltips)->set_tip(*w.b.active, _("show the active entry"));
  w.b.active->clicked.connect(slot(this, &TTSearchWidget::btn_active));
  box->pack_start(*w.b.active, false, false, false);
  w.active = manage(new Entry);
  w.active->set_editable(false);
  box->pack_start(*w.active, true, true, false);
  w.b.set_active = manage(new Button);
  Pixmap *icon = manage(new Pixmap(set_active_xpm));
  icon->set_build_insensitive(true);
  w.b.set_active->add(*icon);
  w.b.set_active->set_relief(GTK_RELIEF_NONE);
  manage(new Tooltips)->set_tip(*w.b.set_active,
    _("set active to selected entry"));
  w.b.set_active->clicked.connect(slot(this, &TTSearchWidget::btn_set_active));
  box->pack_start(*w.b.set_active, false, false, false);

  // set up menu bar interaction
  w.text->focus_in_event.connect(bind(slot(app->w.mbar,
    &TTMenuBar::set_cb_widget), (Editable*) w.text));
  w.text->focus_out_event.connect(bind(slot(app->w.mbar,
    &TTMenuBar::set_cb_widget), (Editable*) 0));

  // show everything but us in our entirety :-)
  show_all();
  hide();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTSearchWidget::update_buttons()
{
  // update tool bar buttons' states
  if (selected < 0) {  // no entry selected
    w.b.show->set_sensitive(false);
    w.b.del->set_sensitive(false);
    w.b.link->set_sensitive(false);
    w.b.set_active->set_sensitive(false);
  } else {  // entry selected
    w.b.show->set_sensitive(true);
    w.b.del->set_sensitive(true);
    w.b.link->set_sensitive(app->get_active() >= 0 &&
      !app->db.linked(app->get_active(), selected));  // order! (efficiency!)
    w.b.set_active->set_sensitive(app->get_active() != selected);
  }

  // update "active entry" elements
  static int active = -2;  // != -1 to force update on first call
  int id = app->get_active();
  if (active != id) {  // only update if necessary
    if (id < 0) {  // no active entry
      w.active->set_text("");
      w.b.active->set_sensitive(false);
    } else {  // active entry
      dbf_t flags;
      string summary, text;
      if (app->db.read(id, flags, summary, text) == DB_FAILURE)
        internal_error_popup(app, TTQD_ERROR_DB_READ);
      w.active->set_text(summary);
      w.b.active->set_sensitive(true);
    }
    active = id;
  }
}

//.............................................................................
void
TTSearchWidget::clear()
{
  w.text->set_text("");
  w.list->clear();
  w.active->set_text("");
  selected = -1;
  app->set_active(-1);
}

//.............................................................................
void
TTSearchWidget::btn_search()
{
  static const unsigned int max_results = 50;

  w.list->clear();
  list<dbid_t> results;
  unsigned sflags = 0;
  if (!w.b.c_all->get_active()) sflags |= DB_SEARCH_SUMMARIES;
  if (!w.b.l_and->get_active()) sflags |= DB_SEARCH_OR;
  if (w.b.t_hub->get_active())  sflags |= DB_SEARCH_HUB;
  if (w.b.t_ent->get_active())  sflags |= DB_SEARCH_ENT;
  if (w.b.t_lnk->get_active())  sflags |= DB_SEARCH_LNK;
  if (w.b.t_iso->get_active())  sflags |= DB_SEARCH_ISO;
  int res = app->db.search(w.text->get_text(), sflags, results,
    app->opts.nolistlimit ? 0 : max_results);
  if (res == DB_FAILURE)
    internal_error_popup(app, TTQD_ERROR_SEARCH);
  else {
    if (results.size() == 1)
      app->sbar_msg(_("Found 1 matching record."));
    else {
      char *s = new char[70];  // should suffice
      sprintf(s, app->opts.nolistlimit || results.size() < max_results ?
        _("Found %d matching records.") :
        _("Search results list truncated at %d records.  "
          "Try to be more specific."), results.size());
      app->sbar_msg(s);
      delete[] s; 
    }

    w.list->freeze();
    for (list<dbid_t>::iterator i = results.begin(); i != results.end(); i++)
    {
      dbf_t flags;
      string summary, text;
      if (app->db.read(*i, flags, summary, text) == DB_FAILURE) {
        internal_error_popup(app, TTQD_ERROR_DB_READ);
        break;
      }
      w.list->append_link(*i, summary, flags & DB_FLAG_HUB, app->db.linked(*i));
    }
    if (results.size()) {  // make the list grab the focus if there are results
      w.list->gtkobj()->focus_row = 0;  // evil kludge, I know
#if GTKMM_MINOR_VERSION <= 1 && GTKMM_MAJOR_VERSION < 2
      w.list->cause_select_row(0, 0);
#else
      w.list->select_row(0, 0);
#endif
      w.list->grab_focus();
    }
    w.list->thaw();
  }
  update_buttons();

  // show single search results immediately
  if (app->opts.quicksearch && results.size() == 1 && app->get_active() < 0) {
    app->set_active(*results.begin());
    app->w.entry->load_record(app->get_active());
    app->run_entry();
  }
}

//.............................................................................
void
TTSearchWidget::btn_show()
{
  if (selected < 0) return;
  app->w.entry->load_record(selected);
  app->run_entry();
}

//.............................................................................
void
TTSearchWidget::btn_link()
{
  dbid_t active = app->get_active();
  if (active < 0 || selected < 0) return;

  // link them
  if (app->db.link(active, selected) == DB_SUCCESS) {
    update_list_for(active);
    update_list_for(selected);
    update_buttons();
    app->sbar_msg(_("Entries linked."));
  } else
    internal_error_popup(app, TTQD_ERROR_DB_WRITE);
}

//.............................................................................
void
TTSearchWidget::btn_add()
{
  app->run_entry();
  app->w.entry->btn_clear();
}

//.............................................................................
void
TTSearchWidget::btn_del()
{
  // ask for confirmation
  if (selected < 0 || query_popup(app, _("Delete entry..."),
    _("Are you sure you want to delete the selected entry?"),
    TTQD_BTN_YES | TTQD_BTN_NO, TTQD_BTN_NO) != TTQD_BTN_YES)
  {
    app->sbar_msg(_("The entry has NOT been deleted."));
    return;
  }

  // delete the record
  list<dbid_t> l;  // get linked entries to update list later on
  app->db.links(selected, l);
  if (app->db.del(selected) == DB_SUCCESS) {
    if (selected == app->get_active()) app->set_active(-1);
    update_buttons();
    app->w.mbar->update_bookmark(selected);
    // update list for entries possibly linked to the deleted one
    for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
      update_list_for(*i);
    app->sbar_msg(_("Entry deleted."));
  } else
    internal_error_popup(app, TTQD_ERROR_DB_WRITE);
  w.list->delete_link(selected);  // remove entry from list
}

//.............................................................................
void
TTSearchWidget::btn_clear()
{
  clear();
  update_buttons();
  w.text->grab_focus();
  app->sbar_msg(_("Specify search criteria..."));
}

//.............................................................................
void
TTSearchWidget::btn_active()
{
  if (app->get_active() < 0) return;
  app->w.entry->load_record(app->get_active());
  app->run_entry();
}

//.............................................................................
void
TTSearchWidget::btn_set_active()
{
  if (selected < 0) return;
  app->set_active(selected);
  update_buttons();
}

//.............................................................................
void
TTSearchWidget::update_list_for(dbid_t id)
{
  if (id < 0) return;

  dbf_t flags;
  string summary, text;
  if (app->db.read(id, flags, summary, text) == DB_SUCCESS) {  // exists, update
    w.list->update_link(id, summary, flags & DB_FLAG_HUB, app->db.linked(id));
    w.list->sort();  // re-sort list
  } else
    w.list->delete_link(id);
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTSearchWidget::handle_select_entry(gint row, gint col, GdkEvent *event)
{
  selected = w.list->id_of_row(row);
#ifdef ENABLE_DEBUG
  DMSG << "selected id = " << selected << endl;
#endif
  update_buttons();
  if (event && event->type == GDK_2BUTTON_PRESS) btn_show();
}

//.............................................................................
void
TTSearchWidget::handle_unselect_entry(gint row, gint col, GdkEvent *event)
{
  selected = -1;
#ifdef ENABLE_DEBUG
  DMSG << "selection cleared" << endl;
#endif
  update_buttons();
}

//.............................................................................
void
TTSearchWidget::handle_t_hub()
{
  if (!w.b.t_hub->get_active()) w.b.t_ent->set_active(true);
}

//.............................................................................
void
TTSearchWidget::handle_t_ent()
{
  if (!w.b.t_ent->get_active()) w.b.t_hub->set_active(true);
}

//.............................................................................
void
TTSearchWidget::handle_t_lnk()
{
  if (!w.b.t_lnk->get_active()) w.b.t_iso->set_active(true);
}

//.............................................................................
void
TTSearchWidget::handle_t_iso()
{
  if (!w.b.t_iso->get_active()) w.b.t_lnk->set_active(true);
}

