/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <fstream>

#include <gdk/gdkx.h>

#include "thoughttracker.h"
#include "app.h"
#include "optionsdlg.h"
#include "querydlg.h"
#include "util.h"
#include "xpm.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTApplication::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

const char *TTApplication::cfgfile_name = "/config";

//.............................................................................
/* just exits on fatal errors received via a callback */
static void
fatal_callback(const string &msg)
{
  cerr << msg << endl << _("Fatal error, exiting.") << endl;
  exit(EXIT_FAILURE);
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTApplication::TTApplication(const string &data_directory)
  : Gtk::Window(GTK_WINDOW_TOPLEVEL), data_dir(data_directory),
    db(data_dir, &fatal_callback), prev_active(-1), active(-1)
{
  // configure main window
  set_name("main_window");
  ensure_style();
  set_border_width(0);
  set_title("ThoughtTracker " VERSION);
  set_policy(0, 1, 0);
  set_wmclass(WM_CLASS_NAME, "main_window");
  realize();
  // set app icon
  static GdkBitmap *mask = 0;
  static GdkPixmap *icon = gdk_pixmap_create_from_xpm_d(
    GTK_WIDGET(gtkobj())->window, &mask,
    &GTK_WIDGET(gtkobj())->style->bg[GTK_STATE_NORMAL], logo_xpm);
/*
  glong data[2];
  data[0] = GDK_WINDOW_XWINDOW(icon);
  data[1] = GDK_WINDOW_XWINDOW(mask);
  Atom icon_atom = gdk_atom_intern("KWM_WIN_ICON", 0);
  gdk_property_change(GTK_WIDGET(gtkobj())->window, icon_atom, icon_atom, 16,
    GDK_PROP_MODE_REPLACE, (guchar*) data, 2);
*/
  gdk_window_set_icon(GTK_WIDGET(gtkobj())->window, 0, icon, mask);
  gdk_window_set_icon_name(GTK_WIDGET(gtkobj())->window, "ThoughtTracker "
    VERSION);

  // configure main box widget
  w.mainbox = manage(new Gtk::VBox);
  add(*w.mainbox);

  // configure menu bar
  w.mbar = manage(new TTMenuBar(this));
  w.mainbox->pack_start(*w.mbar, false, false, false);

  // pack widgets
  w.entry = manage(new TTEntryWidget(this));
  w.mainbox->pack_start(*w.entry, true, true, false);
  w.search = manage(new TTSearchWidget(this));
  w.mainbox->pack_start(*w.search, true, true, false);

  // configure status bar
  w.sbar = manage(new Gtk::Statusbar);
  w.mainbox->pack_start(*w.sbar, false, true, false);

  // set options to default values, then read the config file
  opts.fixedfont = "-*-courier-medium-r-normal--*";
  opts.fixeddefault = false;
  opts.autolink = false;
  opts.quicksearch = true;
  opts.nolistlimit = false;
  opts.savesize = false;
  opts.panepos = 2*APP_WIN_HEIGHT/5;
  load_config();
  w.entry->set_fixed_font(opts.fixedfont);

  // prepare window for display
  GdkWindowHints hints = GDK_HINT_MIN_SIZE;
  GdkGeometry geometry;
  geometry.min_width = APP_WIN_WIDTH;
  geometry.min_height = APP_WIN_HEIGHT;
  set_geometry_hints(*w.mainbox, &geometry, hints);

  w.mbar->show();
  w.mainbox->show();
  w.sbar->show();
  run_search();
  w.search->btn_clear();  // i.a., set keyboard focus to the first entry field
  show();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTApplication::load_config()
{
  string cfgfile = data_dir + cfgfile_name;
#ifdef ENABLE_DEBUG
  DMSG << "loading configuration from " << cfgfile << endl;
#endif
  ifstream cfg(cfgfile.c_str(), ios::nocreate | ios::skipws);
  if (cfg && cfg.good()) {  // open succeeded
    int number = 0;
    while (!cfg.eof()) {
      // read an entire line
      char *buf;
      cfg.gets(&buf);
      string line = buf;
      delete[] buf;
      number++;
      // parse line
      if (line.length()) {
        unsigned int i, j;
        // skip any white space
        for (i = 0; i < line.length() && isspace(line[i]); i++);
        if (i < line.length()) {  // continue if not at end of line
          // i = start of keyword: now find end of keyword
          for (j = i; j < line.length() && !isspace(line[j]) && line[j] != '=';
            j++);
          // temporarily save keyword
          string keyword = line.substr(i, j-i);
          // skip any white space
          for (i = j; i < line.length() && isspace(line[i]); i++);
          // if there's no `=' here, we've got an error
          if (i < line.length() && line[i] == '=') {  // syntax OK so far
            // continue skipping `=' and any trailing white space
            for (++i; i < line.length() && isspace(line[i]); i++);
            if (i < line.length()) {  // OK, found value after `='
              // is the value quoted?
              bool quoted = line[i] == '"';
              // find end of value
              for (j = i + quoted; j < line.length() &&
                (quoted ? (line[j] != '"') : !isspace(line[j]));
                j += 1 + (quoted && line[j] == '\\' && j < line.length()-1));
              if (!quoted || line[j] == '"') {  // OK, found end o'value
                if (quoted) j++;  // skip rhs quote character
                // temporarily save value
                string value = line.substr(i, j-i);
                // if quoted, unquote the string
                cfg_unquote(value);
                // check if rest of line is empty
                for (i = j; i < line.length() && isspace(line[i]); i++);
                if (i == line.length()) {  // OK: rest of line is blank
                  // syntax checks done; now check semantics
                  if (keyword == "bookmarks") {
                    w.mbar->parse_bm_string(value);
                    continue;
                  } else if (keyword == "fixedfont") {
                    opts.fixedfont = value;
                    continue;
                  } else if (keyword == "fixeddefault") {
                    if (parse_bool_opt(opts.fixeddefault, value)) continue;
                  } else if (keyword == "autolink") {
                    if (parse_bool_opt(opts.autolink, value)) continue;
                  } else if (keyword == "quicksearch") {
                    if (parse_bool_opt(opts.quicksearch, value)) continue;
                  } else if (keyword == "nolistlimit") {
                    if (parse_bool_opt(opts.nolistlimit, value)) continue;
                  } else if (keyword == "size") {
                    opts.savesize = true;
                    const char *ptr = value.c_str();
                    char *ptr2;
                    gint x = strtol(ptr, &ptr2, 10);
                    if (ptr2 != ptr && tolower(*ptr2++) == 'x' && *ptr2) {
                      ptr = ptr2;
                      gint y = strtol(ptr, &ptr2, 10);
                      if (ptr2 != ptr) {
                        if (*ptr2++ == '/' && *ptr2) {
                          ptr = ptr2;
                          opts.panepos = strtol(ptr, &ptr2, 10);
                        } else
                          opts.panepos = 2*y/5;
                        set_usize(x, y);
                        continue;
                      }
                    }
                  } else {  // unknown keyword
                    cerr << _("Error in ") << cfgfile << _(", line ") <<
                      number << ":\n  " << _("unknown keyword: ") << '`' <<
                      keyword << "'." << endl;
                    continue;
                  }
                  // if we get till here, the value is bad
                  cerr << _("Error in ") << cfgfile << _(", line ") << number <<
                    ":\n  " << _("bad value for keyword ") << '`' << keyword <<
                    "'." << endl;
                  continue;
                } else  // rest of line is NOT blank: set j for warning below
                  for (j = i; j < line.length() && !isspace(line[j]); j++);
              }
            }
          } else  // no `='
            j = i+1;  // for warning below
          // if we get till here, there was a parse error
          cerr << _("Parse error in ") << cfgfile << ", line " << number <<
            ",\n  " << (i < line.length() ? (string(_("near ")) + '`' +
            line.substr(i, j-i) + "'") : _(" at end of line")) << '.' << endl;
        }  // non-empty line
      }  // if line.length()
    }  // while !eof()
  }  // config file open failed: remain silent (1st run -> doesn't exist?)
}

//.............................................................................
void
TTApplication::save_config()
{
  static const char *bools[2] = { "off", "on" };

  string cfgfile = data_dir + cfgfile_name;
  string tmpfile = cfgfile + ".tmp";  // write to temp. file, then rename it
#ifdef ENABLE_DEBUG
  DMSG << "saving configuration to " << cfgfile << endl;
#endif
  ofstream cfg(tmpfile.c_str(), ios::out | ios::trunc);
  if (cfg && cfg.good()) {
    string bm = w.mbar->get_bm_string();
    cfg_quote(bm);
    if (bm.length()) cfg << "bookmarks = " << bm << endl;
    string ff = opts.fixedfont;
    cfg_quote(ff);
    cfg << "fixedfont = " << ff << endl;
    // note: `!!' just to be sure (if bool is #defined as int)
    cfg << "fixeddefault = " << bools[!!opts.fixeddefault] << endl <<
           "autolink = "     << bools[!!opts.autolink] << endl <<
           "quicksearch = "  << bools[!!opts.quicksearch] << endl <<
           "nolistlimit = "  << bools[!!opts.nolistlimit] << endl;
    if (opts.savesize) {
      cfg << "size = " << width() << 'x' << height() << '/' <<
        (w.entry->is_mapped() ? w.entry->get_pane_pos() : opts.panepos) <<
        endl;
    }
    cfg.close();
    unlink(cfgfile.c_str());
    if (!rename(tmpfile.c_str(), cfgfile.c_str())) return;
  }
  cerr << _("Error writing to config file") << " (" << cfgfile << ")!" << endl;
}

//.............................................................................
void
TTApplication::set_active(dbid_t id)
{ 
  if (active >= 0) prev_active = active;
  active = id;
  w.mbar->update();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTApplication::run_entry() const
{
  if (!w.entry->is_mapped())  // doing this isn't possible within the ctor
    w.entry->set_pane_pos(opts.panepos);  // init entry form pane position
  w.search->hide();
  w.entry->show();
  w.entry->prepare_for_display();
  w.mbar->update();
}

//.............................................................................
void
TTApplication::run_search() const
{
  w.entry->hide();
  w.search->show();
  w.search->prepare_for_display();
  w.mbar->update();
}

//.............................................................................
int
TTApplication::delete_event_impl(GdkEventAny *event)
{
  if (!w.entry->save_record(false)) return 1;
  save_config();  // savesize requires us to do that here, too
  Gtk::Main::quit();
  return 0;
}

//.............................................................................
void
TTApplication::die(const string &msg)
{
  cerr << msg << endl;
  exit(EXIT_FAILURE);
}

//.............................................................................
bool
TTApplication::parse_bool_opt(bool &opt, const string &val)
{
  static const char *no[] = { "no", "off", "false", "disabled", "0", 0 };
  static const char *yes[] = { "yes", "on", "true", "enabled", "1", 0 };

#ifdef HAVE_STRCASECMP
  char *s = new char[val.length()+1];
  strcpy(s, val.c_str());
  for (unsigned int i = 0; i < strlen(s); i++) s[i] = tolower(s[i]);
  // check for true
  for (int i = 0; yes[i]; i++)
    if (!strcmp(yes[i], s)) {
      opt = true;
      delete[] s;
      return true;
    }
  // check for false
  for (int i = 0; no[i]; i++)
    if (!strcmp(no[i], s)) {
      opt = false;
      delete[] s;
      return true;
    }
  delete[] s;
#else
  for (int i = 0; yes[i]; i++)
    if (!strcasecmp(yes[i], val.c_str())) return opt = true;
  for (int i = 0; no[i]; i++)
    if (!strcasecmp(no[i], val.c_str())) {
      opt = false;
      return true;
    }
#endif  // HAVE_STRCASECMP
  return false;
}

