
/*********************************************************************
 *                
 * Filename:      thinkpad.c
 * Description:   loadable kernel module that serves as a router of ioctl
 *                requests to other modules that control various hardware
 *                features of IBM ThinkPads
 * Author:        Thomas Hood <jdthood@mail.com>
 * Created:       19 July 1999 
 *
 * Please report bugs to the author ASAP.
 * 
 *     Copyright (c) 1999 J.D. Thomas Hood, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *     
 ********************************************************************/

#include "thinkpad_mod_defines.h"

#include <linux/module.h>
#include <linux/kmod.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/miscdevice.h>
#include <linux/fs.h>
#ifdef USE_PROC_FS
#include <linux/proc_fs.h>
#endif
#include <asm/uaccess.h>
#include <linux/wrapper.h>
#include "thinkpad_common.h"
#include "thinkpad.h"
#include "smapi.h"
#include "superio.h"
#include "rtcmosram.h"

/****** defines ******/

#define SZ_THINKPAD_VERSION "1.0"
/* must be no longer than LEN_VERSION_MAX */

#define SZ_THINKPAD_NAME "thinkpad"

#define MINOR_THINKPAD 170

/****** forward declarations ******/
#ifdef USE_PROC_FS
#ifndef NEW_PROC
static int thinkpad_read_proc(
	char *pchBuf,
	char **ppchStart,
	off_t offThe,
	int intLen,
	int intUnused
);
#endif
#endif
static int thinkpad_open(
	struct inode * pinodeThe,
	struct file * pfileThe
);
static int thinkpad_release(
	struct inode * pinodeThe,
	struct file * pfileThe 
);
static int thinkpad_ioctl(
	struct inode * pinodeThe,
	struct file * pfileThe,
	unsigned int uintIoctlNum,
	unsigned long ulongIoctlArg
);
void cleanup_module( void );
int init_module( void );

typedef enum _mod_t {
	MOD_SMAPI,
	MOD_SUPERIO,
	MOD_RTCMOSRAM
} mod_t;
typedef int (* pxdo_t)( unsigned long, flag_t );
typedef void (* pxregs_t)( void );

/****** variables ******/

/*** module paramters ***/
int enable_smapi = 1;
int enable_superio = 1;
int enable_rtcmosram = 1;
MODULE_PARM( enable_smapi, "i" );
MODULE_PARM( enable_superio, "i" );
MODULE_PARM( enable_rtcmosram, "i" );
MODULE_AUTHOR( "Thomas Hood" );
MODULE_DESCRIPTION( "A driver for IBM ThinkPad configuration" );

/*** local ***/
static int cOpen  = 0;  /* count of opens */
static char _szThinkpadName[] = SZ_THINKPAD_NAME;
static char _szThinkpadVersion[] = SZ_THINKPAD_VERSION;
static char _szSmapiName[] = "smapi";
static char _szSuperioName[] = "superio";
static char _szRtcmosramName[] = "rtcmosram";
static flag_t _fThinkpadReady = 0;

static struct file_operations fileopsThinkpad = {
	NULL,  /* seek */
	NULL, /* read */
	NULL, /* write */
	NULL, /* readdir */
	NULL, /* select */
	thinkpad_ioctl,
	NULL, /* mmap */
	thinkpad_open,
	NULL, /* flush */
	thinkpad_release
};

#ifdef USE_PROC_FS
#ifndef NEW_PROC
/*
 * Used by all the modules:
 */
struct proc_dir_entry _proc_dir_entryThinkpadDir = {
	0,                  /* low_ino: the inode--dynamic */
	8, "thinkpad",      /* len of name, name */
	S_IFDIR | S_IRUGO | S_IXUGO,  /* mode */
	1, 0, 0,            /* nlinks, owner, group */
	0,                  /* size--unused */
	NULL,               /* point to operations--use default */
	NULL
};
static flag_t _fCreatedProcThinkpadDir, _fCreatedProcThinkpad;
static struct proc_dir_entry _proc_dir_entryThinkpad = {
	0,                  /* low_ino: the inode--dynamic */
	8, "thinkpad",      /* len of name, name */
	S_IFREG | S_IRUGO,  /* mode */
	1, 0, 0,            /* nlinks, owner, group */
	0,                  /* size--unused */
	NULL,               /* point to operations--use default */
	&thinkpad_read_proc
};
#endif
#endif

/****** functions ******/

/*
 * Set up pointers to the service module ioctl handlers that are loaded
 * and also, if necessary and if enabled, load them.
 *
 * I believe we have to do this for each ioctl since we never know
 * when a module will get unloaded.  I presume, though, that we can
 * count on a module not being unloaded between locate_do_func() and
 * the function call!
 */
static pxdo_t locate_do_func( mod_t modWhich )
{
	pxdo_t pxdoThe;

	if ( modWhich == MOD_SMAPI ) {
		if ( enable_smapi ) {
			pxdoThe = (pxdo_t)get_module_symbol( _szSmapiName, "smapi_do" );
			if ( pxdoThe != NULL ) return pxdoThe;
			request_module( _szSmapiName );
			pxdoThe = (pxdo_t)get_module_symbol( _szSmapiName, "smapi_do" );
			return pxdoThe;
		} else return (pxdo_t)NULL;
	} else if ( modWhich == MOD_SUPERIO ) {
		if ( enable_superio ) {
			pxdoThe = (pxdo_t)get_module_symbol( _szSuperioName, "superio_do" );
			if ( pxdoThe != NULL ) return pxdoThe;
			request_module( _szSuperioName );
			pxdoThe = (pxdo_t)get_module_symbol( _szSuperioName, "superio_do" );
			return pxdoThe;
		} else return (pxdo_t)NULL;
	} else if ( modWhich == MOD_RTCMOSRAM ) {
		if ( enable_rtcmosram ) {
			pxdoThe = (pxdo_t)get_module_symbol( _szRtcmosramName, "rtcmosram_do" );
			if ( pxdoThe != NULL ) return pxdoThe;
			request_module( _szRtcmosramName );
			pxdoThe = (pxdo_t)get_module_symbol( _szRtcmosramName, "rtcmosram_do" );
			return pxdoThe;
		} else return (pxdo_t)NULL;
	}

	return (pxdo_t) NULL;
}

#ifdef USE_PROC_FS
static int thinkpad_read_proc(
	char *pchBuf, 
	char **ppchStart, 
	off_t offThe,
	int intLen, 
#ifdef NEW_PROC
	int *eof, 
	void *data
#else
	int intUnused
#endif
) {

	if ( !_fThinkpadReady ) {
		return sprintf(
			pchBuf,
			"%s version %s not ready\n",
			_szThinkpadName, _szThinkpadVersion
		);
	}

	return sprintf(
		pchBuf,
		"%s version %s enabled to access modules: %s %s %s.\n",
		_szThinkpadName, _szThinkpadVersion,
		enable_smapi ? _szSmapiName : "",
		enable_superio ? _szSuperioName : "",
		enable_rtcmosram ? _szRtcmosramName : ""
	);
}
#endif

static int thinkpad_open(
	struct inode * pinodeThe,
	struct file * pfileThe
) {

	if ( cOpen ) return -EBUSY;

	cOpen++;
	MOD_INC_USE_COUNT;

	return 0;
}


static int thinkpad_release(
	struct inode * pinodeThe,
	struct file * pfileThe
) {

	cOpen--;
	MOD_DEC_USE_COUNT;

	return 0;
}



static flag_t caller_has_w( struct file * pfileThe )
{
	return ((pfileThe->f_mode) & FMODE_WRITE) ? 1 : 0;
}


static int thinkpad_ioctl(
	struct inode * pinodeThe,
	struct file * pfileThe,
	unsigned int uintIoctlNum,
	unsigned long ulongIoctlArg
) {
	unsigned long ulRtnCopy;

	if ( ! _fThinkpadReady )
		return -ETHINKPAD_EXECUTION;

#ifdef DEBUG_VERBOSE
	printk( "%s: doing ioctl number 0x%x\n", _szThinkpadName, uintIoctlNum );
#endif

	switch ( uintIoctlNum ) {
		/* should  return  at the end of each case block */

		case IOCTL_THINKPAD_GETVER: 
			ulRtnCopy = copy_to_user(
				(byte *)ulongIoctlArg,
				_szThinkpadVersion,
				strlen( _szThinkpadVersion ) + 1
			);
			if ( ulRtnCopy ) return -EFAULT;
			return 0;

		case IOCTL_THINKPAD_ENABLE:
		case IOCTL_THINKPAD_DISABLE: {
			long lRes;
			char szBuf[LEN_NAME_MAX+1];
			flag_t fEnable;

			fEnable = ( uintIoctlNum == IOCTL_THINKPAD_ENABLE ) ? 1 : 0;


			if ( !caller_has_w( pfileThe ) ) return -EACCES;

			lRes = strncpy_from_user(
				szBuf,
				(char *)ulongIoctlArg,
				LEN_NAME_MAX
			);
			szBuf[LEN_NAME_MAX] = '\0'; /* ensure termination */

			printk(
				KERN_INFO
				"%s: %sabling %s\n", _szThinkpadName, fEnable?"en":"dis", szBuf
			);

			if ( strnicmp( szBuf, _szSmapiName, LEN_NAME_MAX ) == 0 ) {
				enable_smapi = fEnable; return 0;
			} else if ( strnicmp( szBuf, _szSuperioName, LEN_NAME_MAX ) == 0 ) {
				enable_superio = fEnable; return 0;
			} else if ( strnicmp( szBuf, _szRtcmosramName, LEN_NAME_MAX ) == 0 ) {
				enable_rtcmosram = fEnable; return 0; 
			} else return -EINVAL;
		}

		case IOCTL_SMAPI_REQUEST: {
			pxdo_t pxdoSmapi;
			if ( ! enable_smapi ) return -ETHINKPAD_MODULE_DISABLED;
			pxdoSmapi = locate_do_func( MOD_SMAPI );
			if ( pxdoSmapi == NULL ) return -ETHINKPAD_MODULE_NOT_FOUND;
			return (* pxdoSmapi)(
				ulongIoctlArg, 
				caller_has_w( pfileThe )
			);
		}

		case IOCTL_SUPERIO_REQUEST:  {
			pxdo_t pxdoSuperio;
			if ( ! enable_superio ) return -ETHINKPAD_MODULE_DISABLED;
			pxdoSuperio = locate_do_func( MOD_SUPERIO );
			if ( pxdoSuperio == NULL ) return -ETHINKPAD_MODULE_NOT_FOUND;
			return (* pxdoSuperio)(
				ulongIoctlArg, 
				caller_has_w( pfileThe )
			);
		}

		case IOCTL_RTCMOSRAM_REQUEST: {
			pxdo_t pxdoRtcmosram;
			if ( ! enable_rtcmosram ) return -ETHINKPAD_MODULE_DISABLED;
			pxdoRtcmosram = locate_do_func( MOD_RTCMOSRAM );
			if ( pxdoRtcmosram == NULL ) return -ETHINKPAD_MODULE_NOT_FOUND;
			return (* pxdoRtcmosram)(
				ulongIoctlArg, 
				caller_has_w( pfileThe )
			);
		}

		default:
			/* ioctl number not recognized -- do nothing */
			return -ENOTTY;

	} /* switch */
			
	return -ETHINKPAD_PROGRAMMING; /* We should never arrive here */
}


static struct miscdevice structmiscdeviceThinkpad = {
	MINOR_THINKPAD,
	SZ_THINKPAD_NAME,
	&fileopsThinkpad
};


int __init init_module( void )
{
	int intRtnMiscRegister;

	/*** register ***/
	intRtnMiscRegister = misc_register( &structmiscdeviceThinkpad );
	if ( intRtnMiscRegister ) {
		if ( intRtnMiscRegister == -EBUSY ) {
			printk(
				KERN_ERR
				"%s: error %d was returned by misc_register(): device is busy\n",
				_szThinkpadName, intRtnMiscRegister
			);
		} else {
			printk(
				KERN_ERR
				"%s: error %d was returned by misc_register()\n",
				_szThinkpadName, intRtnMiscRegister
			);
		}
		return intRtnMiscRegister;
	}

	/* this module has been registered successfully */

	printk(
		KERN_INFO
		"%s (version %s): I have registered to handle major: %d minor: %d.\n",
		_szThinkpadName, _szThinkpadVersion, 10, MINOR_THINKPAD
	);

	_fThinkpadReady = 1;

#ifdef USE_PROC_FS
	/*** Add /proc entries ***/
#ifdef NEW_PROC
 	if ( !proc_mkdir( "thinkpad", 0 ) ) {
		printk( KERN_ERR "%s: could not proc_mkdir() /proc/thinkpad/\n", _szThinkpadName );
	} else { /* /proc/thinkpad was created. */
		if ( !create_proc_read_entry( "thinkpad/thinkpad", S_IFREG | S_IRUGO, NULL, thinkpad_read_proc, NULL ) ) {
			printk( KERN_ERR "%s: could not create_proc_read_entry() /proc/thinkpad/thinkpad\n", _szThinkpadName );
		}
	}
#else
	if ( proc_register( &proc_root, &_proc_dir_entryThinkpadDir ) ) {
		printk( KERN_ERR "%s: could not register /proc/thinkpad/\n", _szThinkpadName );
	} else {
		_fCreatedProcThinkpadDir = 1;
		if ( proc_register( &_proc_dir_entryThinkpadDir, &_proc_dir_entryThinkpad ) ) {
			printk( KERN_ERR "%s: could not register /proc/thinkpad/thinkpad\n", _szThinkpadName );
		} else {
			_fCreatedProcThinkpad = 1;
		}
	}
#endif
#endif

	return 0;
}
		

void cleanup_module( void )
{
	int intRtn;

	intRtn = misc_deregister( &structmiscdeviceThinkpad );

	if ( intRtn != 0 ) printk( KERN_ERR "%s: error reported by misc_deregister: %d\n", _szThinkpadName, intRtn );

	_fThinkpadReady = 0;

#ifdef USE_PROC_FS
	/*** Remove /proc entries ***/
#ifdef NEW_PROC
 	remove_proc_entry("thinkpad/thinkpad", NULL);
 	remove_proc_entry("thinkpad", NULL);
#else
	if ( _fCreatedProcThinkpad ) {
		_fCreatedProcThinkpad = 0;
		intRtn = proc_unregister( &_proc_dir_entryThinkpadDir, _proc_dir_entryThinkpad.low_ino );
		if ( intRtn ) {
			printk( KERN_ERR "%s: error returned by proc_unregister()\n", _szThinkpadName );
		} else {
			if ( _fCreatedProcThinkpadDir ) {
				_fCreatedProcThinkpadDir = 0;
				intRtn = proc_unregister( &proc_root, _proc_dir_entryThinkpadDir.low_ino );
				if ( intRtn ) printk( KERN_ERR "%s: error returned by proc_unregister()\n", _szThinkpadName );
			}
		}
	}
#endif
#endif

	printk(
		KERN_INFO
		"%s: I have cleaned up.\n",
		_szThinkpadName
	);

	return;
}

