/*

    File: analyse.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "fnctdsk.h" /* get_LBA_part */
#include "analyse.h"
#include "bfs.h"
#include "swap.h"
#include "bsd.h"
#include "fat.h"
#include "ntfs.h"
#include "ext2.h"
#include "netware.h"
#include "intrf.h"
#include "rfs.h"
#include "lvm.h"
#include "md.h"
#include "jfs_superblock.h"
#include "jfs.h"
#include "savehdr.h"
#include "cramfs.h"
#include "xfs.h"
#include "ufs.h"
#include "sysv.h"

int search_NTFS_backup(t_param_disk *disk_car,t_partition *partition, const int debug, const int dump_ind)
{
  unsigned char buffer[DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
    return -1;
  /* NTFS recovery using backup sector */
  if(recover_NTFS(disk_car,(struct ntfs_boot_sector*)&buffer,partition,debug,dump_ind,1)==0)
  {
    strncpy(partition->info,"NTFS found using backup sector!",sizeof(partition->info));
    return 1;
  }
  return 0;
}

int search_FAT_backup(t_param_disk *disk_car,t_partition *partition, const int debug, const int dump_ind)
{
  unsigned char buffer[3*disk_car->sector_size];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
    return -1;
  /* FAT32 recovery using backup sector */
  if(recover_FAT32(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind,1)==0)
  {
    strncpy(partition->info,"FAT found using backup sector!",sizeof(partition->info));
    return 1;
  }
  return 0;
}

int search_type_0(t_param_disk *disk_car,t_partition *partition, const int debug, const int dump_ind)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
    return -1;
  if(recover_Linux_SWAP(disk_car,(const union swap_header *)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_LVM(disk_car,(const pv_disk_t*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_FAT12(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_FAT16(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_FAT32(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind,0)==0) return 1;
  if(recover_HPFS(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_OS2MB(disk_car,(const struct fat_boot_sector*)&buffer,partition,debug,dump_ind)==0) return 1;
  if(recover_NTFS(disk_car,(const struct ntfs_boot_sector*)&buffer,partition,debug,dump_ind,0)==0) return 1;
  if(recover_netware(disk_car,(const struct disk_netware *)&buffer,partition)==0) return 1;
  if(recover_xfs(disk_car,(const struct xfs_sb*)&buffer,partition,debug,dump_ind)==0) return 1;
  return 0;
}

int search_type_1(t_param_disk *disk_car,t_partition *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_1 lba=%lu\n",(long unsigned)(partition->part_offset/disk_car->sector_size));
  }
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
    return -1;
  if(recover_BSD(disk_car,(const struct disklabel *)&buffer[0x200],partition,debug,dump_ind)==0) return 1;
  if(recover_BeFS(disk_car,(const struct disk_super_block *)&buffer[0x200],partition,debug,dump_ind)==0) return 1;
  if(recover_cramfs(disk_car,(const struct cramfs_super*)&buffer[0x200],partition,debug,dump_ind)==0) return 1;
  if(recover_sysv(disk_car,&buffer[0x200],partition,debug,dump_ind)==0) return 1;
  return 0;
}

int search_type_2(t_param_disk *disk_car,t_partition *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_2 lba=%lu\n",(long unsigned)(partition->part_offset/disk_car->sector_size));
  }
  if(disk_car->read(disk_car,1024, &buffer[0x400], partition->part_offset+1024)!=0)
    return -1;
  if(recover_EXT2(disk_car,(const struct ext2_super_block*)&buffer[0x400],partition,debug,dump_ind)==0)
  {
    return 1;
  }
  return 0;
}

int search_type_16(t_param_disk *disk_car,t_partition *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_64 lba=%lu\n",(long unsigned)(partition->part_offset/disk_car->sector_size));
  }
  /* Test UFS */
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+16*512)!=0) /* 8k offset */
    return -1;
  if(recover_ufs(disk_car,buffer,partition,debug,dump_ind)==0) return 1;
  return 0;
}

int search_type_64(t_param_disk *disk_car,t_partition *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_64 lba=%lu\n",(long unsigned)(partition->part_offset/disk_car->sector_size));
  }
  /* Test JFS */
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+64*512)!=0) /* 32k offset */
    return -1;
  if(recover_JFS(disk_car,(const struct jfs_superblock*)&buffer,partition,debug,dump_ind)==0) return 1;
  return 0;
}

int search_type_128(t_param_disk *disk_car,t_partition *partition,const int debug, const int dump_ind)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(debug>1)
  {
    ecrit_rapport("search_type_128 lba=%lu\n",(long unsigned)(partition->part_offset/disk_car->sector_size));
  }
  /* Test ReiserFS */
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+128*512)!=0) /* 64k offset */
    return -1;
  if(recover_rfs(disk_car,(const struct reiserfs_super_block*)&buffer,partition,debug,dump_ind)==0) return 1;
  return 0;
}


