/************************************************************************/
/*									*/
/*  Print Dialog.							*/
/*									*/
/************************************************************************/

#   include	"appFrameConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>

#   include	<appDebugon.h>

#   include	<appFrame.h>
#   include	"appUnit.h"
#   include	"appGeoString.h"
#   include	"appPaperChooser.h"
#   include	"appPaper.h"

#   define	DRH_MM	45

/************************************************************************/
/*									*/
/*  Represents a print dialog.						*/
/*									*/
/************************************************************************/

#   define	FILEL	400

#   define	PRINTselAll_PAGES	0
#   define	PRINTselPAGE_RANGE	1
#   define	PRINTselCURRENT_PAGE	2

#   define	PRINTselCOUNT		3

typedef struct PlacementSpecification
    {
    int		psNup;
    int		psHorizontal;
    char *	psOptionText;
    APP_WIDGET	psOptionWidget;
    } PlacementSpecification;

#   define	PRINTps_1_UP	0
#   define	PRINTps_2_UP	1
#   define	PRINTps_4H_UP	2
#   define	PRINTps_4V_UP	3

#   define	PRINTpsCOUNT	4

static PlacementSpecification	APP_Placements[]=
    {
	{ 1, 0, "1 Up"		},
	{ 2, 1, "2 Up"		},
	{ 4, 1, "4 Up Hor"	},
	{ 4, 0, "4 Up Ver"	},
    };

typedef struct AppPrintDialog
    {
    AppDialog			apdDialog;

    APP_WIDGET			apdPrintButton;
    APP_WIDGET			apdCancelButton;

    AppOptionmenu		apdPrinterOptionmenu;
    AppOptionmenu		apdPlacementOptionmenu;
    AppOptionmenu		apdSelectionOptionmenu;

    APP_WIDGET			apdCustomTransformRow;
    APP_WIDGET			apdRotate90Toggle;
    APP_WIDGET			apdCenterHToggle;

    int				apdUnitType;
    PaperChooser		apdPaperChooser;

    APP_WIDGET			apdFaxNumberText;
    APP_WIDGET			apdPrintToFileOption;

    APP_WIDGET			apdPageDrawing;
    AppDrawingData		apdDrawingData;
    int				apdDrawingDataAllocated;
    APP_FONT *			apdFontStruct;

    APP_WIDGET			apdSelectionOptions[PRINTselCOUNT];
    APP_WIDGET			apdPageFromTextWidget;
    APP_WIDGET			apdPageToTextWidget;

    int				apdPrinterCount;
    int				apdPrintOptionCount;
    int				apdDestinationChosen;

    int				apdActivateFaxOptions;
    int				apdFileDestination;
    int				apdFaxDestination;

    DocumentGeometry		apdDocumentGeometry;
    DocumentGeometry		apdPrinterGeometry;
    PlacementSpecification	apdPlacements[PRINTpsCOUNT];
    int				apdPlacementChosen;
    int				apdRotate90Chosen;
    int				apdCenterHChosen;

    int				apdPageCount;
    int				apdFirstSelectedPage;
    int				apdLastSelectedPage;
    int				apdSelectionChosen;
    int				apdFirstPageChosen;
    int				apdLastPageChosen;

    void *			apdTarget;

    char *			apdPrinterText;
    char *			apdNoneText;
    char *			apdCancelText;
    char *			apdPrintText;

    char *			apdFaxText;
    char *			apdPrintToFileText;

    char *			apdPaperSizeText;
    char *			apdCustomPaperSizeText;
    char *			apdPlacementText;
    char *			apdRotate90Text;
    char *			apdCenterHText;

    char *			apdSelectionText;
    char *			apdAllPagesText;
    char *			apdPageRangeText;
    char *			apdCurrentPageText;
    char *			apdFromPageText;
    char *			apdToPageText;

    char *			apdDefaultPrinter;

    APP_WIDGET			apdPrinterOptions[1]; /* LAST! */

    } AppPrintDialog;

static AppConfigurableResource APP_PrintDialogresourceTable[]=
    {
	APP_RESOURCE( "printDialogPrinter",
		    offsetof(AppPrintDialog,apdPrinterText),
		    "Printer" ),
	APP_RESOURCE( "printDialogNone",
		    offsetof(AppPrintDialog,apdNoneText),
		    "None" ),
	APP_RESOURCE( "printDialogCancel",
		    offsetof(AppPrintDialog,apdCancelText),
		    "Cancel" ),
	APP_RESOURCE( "printDialogPrint",
		    offsetof(AppPrintDialog,apdPrintText),
		    "Print" ),
	APP_RESOURCE( "printDialogPrintToFile",
		    offsetof(AppPrintDialog,apdPrintToFileText),
		    "Print to File" ),

	APP_RESOURCE( "printDialogFax",
		    offsetof(AppPrintDialog,apdFaxText),
		    "Fax" ),
	APP_RESOURCE( "printDialogPaperSize",
		    offsetof(AppPrintDialog,apdPaperSizeText),
		    "Paper Size" ),
	APP_RESOURCE( "printDialogCustomPaperSize",
		    offsetof(AppPrintDialog,apdCustomPaperSizeText),
		    "Custom" ),

	/***/
	APP_RESOURCE( "printDialogPlacement",
		    offsetof(AppPrintDialog,apdPlacementText),
		    "Placement" ),

	APP_RESOURCE( "printDialog_1_UpText",
		    offsetof(AppPrintDialog,
				apdPlacements[PRINTps_1_UP].psOptionText),
		    "1 Page/Sheet" ),
	APP_RESOURCE( "printDialog_2_UpText",
		    offsetof(AppPrintDialog,
				apdPlacements[PRINTps_2_UP].psOptionText),
		    "2 Pages/Sheet" ),
	APP_RESOURCE( "printDialog_4H_UpText",
		    offsetof(AppPrintDialog,
				apdPlacements[PRINTps_4H_UP].psOptionText),
		    "4 Pages/Sheet Horizontal" ),
	APP_RESOURCE( "printDialog_4V_UpText",
		    offsetof(AppPrintDialog,
				apdPlacements[PRINTps_4V_UP].psOptionText),
		    "4 Pages/Sheet Vertical" ),

	APP_RESOURCE( "printDialogRotate90",
		    offsetof(AppPrintDialog,apdRotate90Text),
		    "Rotate 90\260" ),
	APP_RESOURCE( "printDialogCenterH",
		    offsetof(AppPrintDialog,apdCenterHText),
		    "Center Horizontally" ),

	/***/
	APP_RESOURCE( "printDialogSelection",
		    offsetof(AppPrintDialog,apdSelectionText),
		    "Selection" ),

	APP_RESOURCE( "printDialogAllPages",
		    offsetof(AppPrintDialog,apdAllPagesText),
		    "All Pages" ),
	APP_RESOURCE( "printDialogPageRange",
		    offsetof(AppPrintDialog,apdPageRangeText),
		    "Page Range" ),
	APP_RESOURCE( "printDialogCurrentPage",
		    offsetof(AppPrintDialog,apdCurrentPageText),
		    "Current Page" ),

	APP_RESOURCE( "printDialogFromPage",
		    offsetof(AppPrintDialog,apdFromPageText),
		    "From" ),
	APP_RESOURCE( "printDialogToPage",
		    offsetof(AppPrintDialog,apdToPageText),
		    "To" ),

	APP_RESOURCE( "defaultPrinter",
		    offsetof(AppPrintDialog,apdDefaultPrinter),
		    "" ),
    };

/************************************************************************/
/*									*/
/*  Show a range of pages in the Print Dialog.				*/
/*									*/
/************************************************************************/

static void appPrintDialogShowFromTo(	AppPrintDialog *	apd,
					int			fromPage,
					int			toPage,
					int			enabled )
    {
    appIntegerToTextWidget( apd->apdPageFromTextWidget, fromPage );
    appIntegerToTextWidget( apd->apdPageToTextWidget, toPage );

    appEnableText( apd->apdPageFromTextWidget, enabled );
    appEnableText( apd->apdPageToTextWidget, enabled );

    return;
    }

/************************************************************************/
/*									*/
/*  Fax number changed.							*/
/*									*/
/************************************************************************/

static void appPrintCheckFaxNumber(	AppPrintDialog *	apd )
    {
    char *	faxNumber;
    char *	s;

    s= faxNumber= appGetStringFromTextWidget( apd->apdFaxNumberText );

    while( *s == ' ' )
	{ s++;	}

    appGuiEnableWidget( apd->apdPrintButton, *s != '\0' );

    appFreeStringFromTextWidget( faxNumber );

    return;
    }

static void appPrintFaxChanged(	APP_WIDGET	w,
				void *		voidapd,
				void *		call_data )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;

    appPrintCheckFaxNumber( apd );

    return;
    }

/************************************************************************/
/*									*/
/*  A printer was chosen.						*/
/*									*/
/************************************************************************/

static void appPrinterAdaptToDestination(	AppPrintDialog *	apd )
    {
    if  ( apd->apdDestinationChosen == apd->apdFaxDestination )
	{
	if  ( apd->apdFaxNumberText )
	    {
	    appEnableText( apd->apdFaxNumberText, 1 );

	    appPrintCheckFaxNumber( apd );
	    }
	}
    else{
	if  ( apd->apdFaxNumberText )
	    {
	    appEnableText( apd->apdFaxNumberText, 0 );
	    }

	appGuiEnableWidget( apd->apdPrintButton, 1 );

	appGuiSetDefaultButtonForDialog( &(apd->apdDialog),
						    apd->apdPrintButton );
	}
    }

static void appDestinationChosen(	APP_WIDGET	w,
					void *		voidapd,
					void *		voidpbcs )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;
    short		n= -1;

    for ( n= 0; n < apd->apdPrintOptionCount; n++ )
	{
	if  ( apd->apdPrinterOptions[n] == w )
	    { break;	}
	}

    if  ( n < 0 || n >= apd->apdPrintOptionCount )
	{ LLDEB(n,apd->apdPrintOptionCount); return;	}

    apd->apdDestinationChosen= n;

    appPrinterAdaptToDestination( apd );

    return;
    }

/************************************************************************/
/*									*/
/*  A placement was chosen.						*/
/*									*/
/************************************************************************/

static void appPrintDialogPlacementChosen(	APP_WIDGET	w,
						void *		voidapd,
						void *		voidpbcs )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;
    AppDrawingData *	add= &(apd->apdDrawingData);
    int			i;

    for ( i= 0; i < PRINTpsCOUNT; i++ )
	{
	if  ( apd->apdPlacements[i].psOptionWidget == w )
	    { break;	}
	}

    if  ( i < 0 || i >= PRINTpsCOUNT )
	{ LLDEB(i,PRINTpsCOUNT); return;	}

    apd->apdPlacementChosen= i;

    if  ( apd->apdCustomTransformRow )
	{
	int	rotate90= apd->apdRotate90Chosen;
	int	centerHorizontally= apd->apdCenterHChosen;

	if  ( apd->apdPlacementChosen != 0 )
	    { rotate90= centerHorizontally= 0;	}

	appGuiEnableWidget( apd->apdRotate90Toggle,
				    apd->apdPlacementChosen == 0 );
	appGuiEnableWidget( apd->apdCenterHToggle,
				    apd->apdPlacementChosen == 0 );

	appGuiSetToggleState( apd->apdRotate90Toggle, rotate90 );
	appGuiSetToggleState( apd->apdCenterHToggle, centerHorizontally );
	}

    appExposeRectangle( add, 0, 0, 0, 0 );

    return;
    }

/************************************************************************/
/*									*/
/*  A selection mode was chosen.					*/
/*									*/
/************************************************************************/

static void appSelectionChosen(		APP_WIDGET	w,
					void *		voidapd,
					void *		voidpbcs )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;
    int			n;

#   ifdef USE_MOTIF
    XSetInputFocus( XtDisplay( apd->apdDialog.adTopWidget ),
				    XtWindow( apd->apdDialog.adTopWidget ),
				    RevertToNone, CurrentTime );
#   endif

    for ( n= 0; n < PRINTselCOUNT; n++ )
	{
	if  ( apd->apdSelectionOptions[n] == w )
	    { break;	}
	}

    if  ( n < 0 || n >= PRINTselCOUNT )
	{ LLDEB(n,PRINTselCOUNT); return;	}

    switch( n )
	{
	case PRINTselAll_PAGES:
	    apd->apdFirstPageChosen= 0;
	    apd->apdLastPageChosen= apd->apdPageCount- 1;

	    appPrintDialogShowFromTo( apd, 1, apd->apdPageCount, 0 );
	    apd->apdSelectionChosen= n;
	    return;

	case PRINTselPAGE_RANGE:
	    appEnableText( apd->apdPageFromTextWidget, 1 );
	    appEnableText( apd->apdPageToTextWidget, 1 );
	    apd->apdSelectionChosen= n;

#	    ifdef USE_MOTIF
	    XmProcessTraversal( apd->apdPageFromTextWidget,
							XmTRAVERSE_CURRENT );
#	    endif
	    return;

	case PRINTselCURRENT_PAGE:
	    apd->apdFirstPageChosen= apd->apdFirstSelectedPage;
	    apd->apdLastPageChosen= apd->apdLastSelectedPage;

	    appPrintDialogShowFromTo( apd, apd->apdFirstSelectedPage+ 1,
					    apd->apdLastSelectedPage+ 1, 0 );
	    apd->apdSelectionChosen= n;
	    return;

	default:
	    LDEB(n); return;
	}
    }

/************************************************************************/
/*									*/
/*  The buttons have been pushed.					*/
/*									*/
/************************************************************************/

static void appPrintDialogCancelPushed(	APP_WIDGET	w,
					void *		voidapd,
					void *		voidpbcs )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;

    appGuiBreakDialog( &(apd->apdDialog), AQDrespCANCEL );

    return;
    }

static void appPrintDialogPrintPushed(	APP_WIDGET	w,
					void *		voidapd,
					void *		voidpbcs )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;

    if  ( apd->apdDestinationChosen < 0	)
	{ LDEB(apd->apdDestinationChosen); return;	}

    if  ( apd->apdSelectionChosen == PRINTselPAGE_RANGE )
	{
	int		from;
	int		to;

	if  ( appGetIntegerFromTextWidget( apd->apdPageFromTextWidget,
					&from, 1, 0, apd->apdPageCount, 0 ) )
	    { return;	}

	if  ( appGetIntegerFromTextWidget( apd->apdPageToTextWidget,
					&to, from, 0, apd->apdPageCount, 1 ) )
	    { return;	}

	apd->apdFirstPageChosen= from- 1;
	apd->apdLastPageChosen= to- 1;
	}

    appGuiBreakDialog( &(apd->apdDialog), AQDrespOK );

    return;
    }

/************************************************************************/
/*									*/
/*  A paper size was chosen.						*/
/*									*/
/************************************************************************/

static void appPrintDialogSizeChosen(	APP_WIDGET	w,
					void *		voidapd,
					void *		voidpbcs )
    {
    AppPrintDialog *		apd= (AppPrintDialog *)voidapd;
    AppDrawingData *		add= &(apd->apdDrawingData);

    appPaperSizeChosen( w, &(apd->apdPaperChooser),
						&(apd->apdPrinterGeometry),
						apd->apdUnitType );

    appExposeRectangle( add, 0, 0, 0, 0 );

    return;
    }

static void appPrintDialogPaperSizeChanged(	APP_WIDGET	w,
						void *		voidapd,
						void *		voidcbs	 )
    {
    AppPrintDialog *		apd= (AppPrintDialog *)voidapd;
    AppDrawingData *		add= &(apd->apdDrawingData);
    const int			set= 0;

    appPaperSizeChanged( &(apd->apdPaperChooser),
						&(apd->apdPrinterGeometry),
						apd->apdUnitType );

    appPaperChooserAdaptToGeometry( &(apd->apdPaperChooser),
			apd->apdUnitType, &(apd->apdPrinterGeometry), set );

    appExposeRectangle( add, 0, 0, 0, 0 );

    return;
    }

/************************************************************************/
/*									*/
/*  The custom placement toggles were activated.			*/
/*									*/
/************************************************************************/

static void appPrintDialogRotate90Toggled(	APP_WIDGET	w,
						void *		voidapd,
						void *		voidtbcs )
    {
    AppPrintDialog *		apd= (AppPrintDialog *)voidapd;
    AppDrawingData *		add= &(apd->apdDrawingData);
    int				set;

    set= appGuiGetToggleStateFromCallback( w, voidtbcs );

    apd->apdRotate90Chosen= ( set != 0 );

    appExposeRectangle( add, 0, 0, 0, 0 );

    return;
    }

static void appPrintDialogCenterHToggled(	APP_WIDGET	w,
						void *		voidapd,
						void *		voidtbcs )
    {
    AppPrintDialog *		apd= (AppPrintDialog *)voidapd;
    AppDrawingData *		add= &(apd->apdDrawingData);
    int				set;

    set= appGuiGetToggleStateFromCallback( w, voidtbcs );

    apd->apdCenterHChosen= ( set != 0 );

    appExposeRectangle( add, 0, 0, 0, 0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a schematic view of the page placement.			*/
/*									*/
/************************************************************************/

static void appPrintDialogPrintGeometry(
				    PrintGeometry *		pg,
				    const AppPrintDialog *	apd )
    {
    const PlacementSpecification *	ps;

    ps= apd->apdPlacements+ apd->apdPlacementChosen;

    pg->pgSheetGeometry= apd->apdPrinterGeometry;

    pg->pgRotate90= apd->apdRotate90Chosen;
    pg->pgCenterHorizontally= apd->apdCenterHChosen;

    if  ( apd->apdPlacementChosen != 0 )
	{ pg->pgRotate90= pg->pgCenterHorizontally= 0; }

    pg->pgNup= ps->psNup;
    pg->pgHorizontal= ps->psHorizontal;

    return;
    }

static APP_EVENT_HANDLER( appPrintDialogDrawPage, w, voidapd, exposeEvent )
    {
    AppPrintDialog *		apd= (AppPrintDialog *)voidapd;
    AppDrawingData *		add= &(apd->apdDrawingData);

    PrintGeometry		pg;

    if  ( apd->apdPlacementChosen < 0			||
	  apd->apdPlacementChosen >= PRINTpsCOUNT	)
	{ LLDEB(apd->apdPlacementChosen,PRINTpsCOUNT); return;	}

    appPrintDialogPrintGeometry( &pg, apd );

    appDrawNupDiagram( apd->apdPageDrawing, add, apd->apdFontStruct, DRH_MM,
					    &(apd->apdDocumentGeometry), &pg );
    }

/************************************************************************/
/*									*/
/*  Fill the list of printers.						*/
/*									*/
/************************************************************************/

static void appPrintDialogFillPrinterMenu( AppPrintDialog *	apd,
					int			printerCount,
					int			defaultPrinter,
					PrintDestination *	pd )
    {
    int			i;
    int			destinationChosen= -1;

    appEmptyOptionmenu( &(apd->apdPrinterOptionmenu) );

    for ( i= 0; i < printerCount; pd++, i++ )
	{
	apd->apdPrinterOptions[i]=
			appAddItemToOptionmenu( &(apd->apdPrinterOptionmenu),
			pd->pdPrinterName, appDestinationChosen,
			(void *)apd );

	if  ( i == defaultPrinter )
	    { destinationChosen= i;	}
	if  ( i == 0 )
	    { destinationChosen= i;	}
	}

    apd->apdPrinterOptions[i]= appAddItemToOptionmenu(
		    &(apd->apdPrinterOptionmenu),
		    apd->apdFaxText, appDestinationChosen, (void *)apd );

    if  ( ! apd->apdActivateFaxOptions )
	{ appGuiEnableWidget( apd->apdPrinterOptions[i], 0 ); }

    apd->apdFaxDestination= i++;
    
    apd->apdPrinterOptions[i]= apd->apdPrintToFileOption=
	appAddItemToOptionmenu( &(apd->apdPrinterOptionmenu),
	    apd->apdPrintToFileText, appDestinationChosen, (void *)apd );

    if  ( destinationChosen < 0 )
	{ destinationChosen= printerCount+ 1;	}

    apd->apdFileDestination= i++;
    apd->apdPrintOptionCount= i;
    
    appSetOptionmenu( &(apd->apdPrinterOptionmenu), destinationChosen );

    apd->apdPrinterCount= printerCount;

    if  ( defaultPrinter >= 0 )
	{ apd->apdDestinationChosen= defaultPrinter;	}
    else{ apd->apdDestinationChosen= destinationChosen; }

    appOptionmenuRefreshWidth( &(apd->apdPrinterOptionmenu) );
    }

/************************************************************************/
/*									*/
/*  Fill the list of possible page placements.				*/
/*									*/
/************************************************************************/

static void appPrintDialogFillPlacementMenu( AppPrintDialog *	apd )
    {
    int				i;

    PlacementSpecification *	ps;

    appEmptyOptionmenu( &(apd->apdPlacementOptionmenu) );

    ps= apd->apdPlacements;
    for ( i= 0; i < PRINTpsCOUNT; ps++, i++ )
	{
	ps->psOptionWidget= appAddItemToOptionmenu(
				&(apd->apdPlacementOptionmenu),
				ps->psOptionText,
				appPrintDialogPlacementChosen, (void *)apd );
	}

    apd->apdPlacementChosen= 0;
    apd->apdRotate90Chosen= 0;
    apd->apdCenterHChosen= 0;

    appSetOptionmenu( &(apd->apdPlacementOptionmenu), 0 );

    appOptionmenuRefreshWidth( &(apd->apdPlacementOptionmenu) );
    }

/************************************************************************/
/*									*/
/*  Fill the list of selection options.					*/
/*									*/
/************************************************************************/

#   define	PRINTselAll_PAGES	0
#   define	PRINTselPAGE_RANGE	1
#   define	PRINTselCURRENT_PAGE	2

static void appPrintDialogFillSelectionMenu( AppPrintDialog *	apd )
    {
    appEmptyOptionmenu( &(apd->apdSelectionOptionmenu) );

    apd->apdSelectionOptions[PRINTselAll_PAGES]= appAddItemToOptionmenu(
					    &(apd->apdSelectionOptionmenu),
					    apd->apdAllPagesText,
					    appSelectionChosen, (void *)apd );
    apd->apdSelectionOptions[PRINTselPAGE_RANGE]= appAddItemToOptionmenu(
					    &(apd->apdSelectionOptionmenu),
					    apd->apdPageRangeText,
					    appSelectionChosen, (void *)apd );
    apd->apdSelectionOptions[PRINTselCURRENT_PAGE]= appAddItemToOptionmenu(
					    &(apd->apdSelectionOptionmenu),
					    apd->apdCurrentPageText,
					    appSelectionChosen, (void *)apd );

    apd->apdSelectionChosen= PRINTselAll_PAGES;

    appSetOptionmenu( &(apd->apdSelectionOptionmenu),
						    apd->apdSelectionChosen );

    appOptionmenuRefreshWidth( &(apd->apdSelectionOptionmenu) );
    }

/************************************************************************/
/*									*/
/*  Protocol callback: The print dialog is closed through the window	*/
/*  manager. Interpret this as 'Cancel'					*/
/*									*/
/************************************************************************/

static APP_CLOSE_CALLBACK( appClosePrintDialog, w, voidapd )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;

    appGuiBreakDialog( &(apd->apdDialog), AQDrespCANCEL );

    return;
    }

static APP_DESTROY_CALLBACK( appDestroyPrintDialog, w, voidapd )
    {
    AppPrintDialog *	apd= (AppPrintDialog *)voidapd;

    appCleanPaperChooser( &(apd->apdPaperChooser) );

    return;
    }

/************************************************************************/
/*									*/
/*  Make the frame for selecting a printer.				*/
/*									*/
/************************************************************************/

static APP_WIDGET appPrintDialogMakePrinterFrame( APP_WIDGET	parent,
					AppPrintDialog *	apd )
    {
    APP_WIDGET	printerFrame;
    APP_WIDGET	frameColumn;

    APP_WIDGET	row;

    const int	heightResizable= 0;
    const int	textEnabled= 0;
    const int	textColumn= 1;
    const int	textColspan= 1;

    appMakeColumnFrameInColumn( &printerFrame, &frameColumn,
						parent, apd->apdPrinterText );

    row= appMakeRowInColumn( frameColumn, 2, heightResizable );

    appMakeOptionmenuInRow( &(apd->apdPrinterOptionmenu), row, 0, 1 );

    appMakeTextInRow( &(apd->apdFaxNumberText), row, textColumn,
					    textColspan, 15, textEnabled );

    appGuiSetTypingCallbackForText( apd->apdFaxNumberText,
					    appPrintFaxChanged, (void *)apd );

    return printerFrame;
    }

/************************************************************************/
/*									*/
/*  Make the frame for giving page placement.				*/
/*									*/
/************************************************************************/

static APP_WIDGET appPrintDialogMakePlacementFrame( APP_WIDGET	parent,
					EditApplication *	ea,
					AppPrintDialog *	apd )
    {
    APP_WIDGET		placementFrame;
    APP_WIDGET		frameColumn;

    const int		heightResizable= 0;

    appMakeColumnFrameInColumn( &placementFrame, &frameColumn, parent,
						    apd->apdPlacementText );

    appMakeOptionmenuInColumn( &(apd->apdPlacementOptionmenu), frameColumn );

    apd->apdCustomTransformRow=
			appMakeRowInColumn( frameColumn, 2, heightResizable );
    apd->apdRotate90Toggle= appMakeToggleInRow( apd->apdCustomTransformRow,
				apd->apdRotate90Text,
				appPrintDialogRotate90Toggled, (void *)apd, 0 );
    apd->apdCenterHToggle= appMakeToggleInRow( apd->apdCustomTransformRow,
				apd->apdCenterHText,
				appPrintDialogCenterHToggled, (void *)apd, 1 );

    apd->apdPageDrawing= appMakePageDrawing( parent, ea, DRH_MM,
					appPrintDialogDrawPage, (void *)apd );

    return placementFrame;
    }

/************************************************************************/
/*									*/
/*  Make the frame to select what pages are to be printed.		*/
/*									*/
/************************************************************************/

static APP_WIDGET appPrintDialogMakeSelectionFrame( APP_WIDGET	parent,
					AppPrintDialog *	apd )
    {
    APP_WIDGET		selectionFrame;
    APP_WIDGET		frameColumn;

    APP_WIDGET		pageSelectionRow;
    APP_WIDGET		row;
    APP_WIDGET		leftColumn;
    APP_WIDGET		rightColumn;
    APP_WIDGET		fromLabel;
    APP_WIDGET		toLabel;

    const int		heightResizable= 0;
    const int		textEnabled= 0;

    appMakeColumnFrameInColumn( &selectionFrame, &frameColumn, parent,
						    apd->apdSelectionText );

    appMakeOptionmenuInColumn( &(apd->apdSelectionOptionmenu), frameColumn );

    pageSelectionRow= appMakeRowInColumn( frameColumn, 2, heightResizable );

    appMakeColumnInRow( &leftColumn, pageSelectionRow, 0 );
    appMakeColumnInRow( &rightColumn, pageSelectionRow, 1 );

    appMakeLabelAndTextRow( &row, &fromLabel, &(apd->apdPageFromTextWidget),
			    leftColumn, apd->apdFromPageText, 8, textEnabled );

    appMakeLabelAndTextRow( &row, &toLabel, &(apd->apdPageToTextWidget),
			    rightColumn, apd->apdToPageText, 8, textEnabled );

    return selectionFrame;
    }

/************************************************************************/
/*									*/
/*  Make the form with the two buttons.					*/
/*									*/
/************************************************************************/

static APP_WIDGET appPrintDialogMakeButtonRow(	APP_WIDGET		parent,
						AppPrintDialog *	apd )
    {
    APP_WIDGET		row;
    const int		heightResizable= 0;

    row= appMakeRowInColumn( parent, 2, heightResizable );

    appMakeButtonInRow( &(apd->apdPrintButton), row, apd->apdPrintText,
			appPrintDialogPrintPushed, (void *)apd, 0, 1 );

    appMakeButtonInRow( &(apd->apdCancelButton), row, apd->apdCancelText,
			appPrintDialogCancelPushed, (void *)apd, 1, 0 );

    appGuiSetCancelButtonForDialog( &(apd->apdDialog),
						    apd->apdCancelButton );
    return row;
    }

/************************************************************************/
/*									*/
/*  make a page tool.							*/
/*									*/
/************************************************************************/

static AppPrintDialog * appMakePrintDialog( EditApplication *	ea,
					EditDocument *		ed,
					APP_WIDGET		printOption,
					const char *		pixmapName )
    {
    AppPrintDialog *	apd;
    AppDrawingData *	add;
    
    APP_WIDGET		printerFrame;
    APP_WIDGET		selectionFrame;
    APP_WIDGET		paned;
    APP_WIDGET		buttonRow;

    int			i;

    int			high;
    int			wide;

    APP_BITMAP_IMAGE	iconPixmap= (APP_BITMAP_IMAGE)0;
    APP_BITMAP_MASK	iconMask= (APP_BITMAP_MASK)0;
    
    char		name[128];

    const double	magnification= 1.0;

    if  ( appGetImagePixmap( ea, pixmapName, &iconPixmap, &iconMask )  )
	{ SDEB(pixmapName); return (AppPrintDialog *)0;	}

    if  ( ea->eaPrintDestinationCount == 0				&&
	  utilPrinterGetPrinters( &(ea->eaPrintDestinationCount),
				    &(ea->eaDefaultPrintDestination),
				    &(ea->eaPrintDestinations),
				    ea->eaCustomPrintCommand,
				    ea->eaCustomPrinterName )		)
	{ LDEB(1); return (AppPrintDialog *)0;	}

    apd= (AppPrintDialog *)malloc(
		    sizeof(AppPrintDialog)+
		    ( ea->eaPrintDestinationCount+ 2 )* sizeof(APP_WIDGET) );
    if  ( ! apd )
	{
	LXDEB(ea->eaPrintDestinationCount,apd);
	return (AppPrintDialog *)0;
	}

    apd->apdCustomTransformRow= (APP_WIDGET)0;
    apd->apdRotate90Toggle= (APP_WIDGET)0;
    apd->apdCenterHToggle= (APP_WIDGET)0;

    add= &(apd->apdDrawingData);

    appInitDrawingData( add );

    for ( i= 0; i < PRINTpsCOUNT; i++ )
	{ apd->apdPlacements[i]= APP_Placements[i];	}

    appGuiGetResourceValues( ea, apd,
				    APP_PrintDialogresourceTable,
				    sizeof(APP_PrintDialogresourceTable)/
				    sizeof(AppConfigurableResource) );

    if  ( ea->eaFaxCommand )
	{ apd->apdActivateFaxOptions= 1;	}
    else{ apd->apdActivateFaxOptions= 0;	}

    if  ( apd->apdDefaultPrinter && apd->apdDefaultPrinter[0] )
	{
	PrintDestination *	pd= ea->eaPrintDestinations;

	for ( i= 0; i < ea->eaPrintDestinationCount; pd++, i++ )
	    {
	    if  ( ! strcmp( pd->pdPrinterName, apd->apdDefaultPrinter ) )
		{ ea->eaDefaultPrintDestination= i; break;	}
	    }
	}

    apd->apdTarget= (void *)ea;
    apd->apdPrinterCount= 0;
    apd->apdPrintOptionCount= 0;
    apd->apdFileDestination= -1;
    apd->apdFaxDestination= -1;
    apd->apdUnitType= ea->eaUnitInt;

    appInitPaperChooser( &(apd->apdPaperChooser) );

    appMakeVerticalDialog( &(apd->apdDialog), &paned, ea,
						appClosePrintDialog,
						appDestroyPrintDialog,
						(void *)apd,
						ea->eaPrintDialogName );

    appSetShellTitle( apd->apdDialog.adTopWidget,
					printOption, ea->eaApplicationName );

    printerFrame= appPrintDialogMakePrinterFrame( paned, apd );

    appMakePaperChooserWidgets( paned, apd->apdPaperSizeText,
					    &(apd->apdPaperChooser),
					    appPrintDialogPaperSizeChanged,
					    (void *)apd );

    appPrintDialogMakePlacementFrame( paned, ea, apd );
    selectionFrame= appPrintDialogMakeSelectionFrame( paned, apd );
    buttonRow= appPrintDialogMakeButtonRow( paned, apd );

    appPrintDialogFillPrinterMenu( apd,
				    ea->eaPrintDestinationCount,
				    ea->eaDefaultPrintDestination,
				    ea->eaPrintDestinations );

    appPrintDialogFillPlacementMenu( apd );
    appPrintDialogFillSelectionMenu( apd );

    appPaperChooserFillMenu( &(apd->apdPaperChooser),
				apd->apdCustomPaperSizeText,
				appPrintDialogSizeChosen, (void *)apd );

    appGuiShowDialog( &(apd->apdDialog), ed->edToplevel.atTopWidget );

    appPaperChooserRetreshMenuWidth( &(apd->apdPaperChooser) );

    appOptionmenuRefreshWidth( &(apd->apdPrinterOptionmenu) );
    appOptionmenuRefreshWidth( &(apd->apdPlacementOptionmenu) );
    appOptionmenuRefreshWidth( &(apd->apdSelectionOptionmenu) );

    if  ( appSetDrawingDataForWidget( apd->apdPageDrawing,
						    magnification, add ) )
	{ LDEB(1);	}

    appDrawGetSizeOfWidget( &wide, &high, apd->apdPageDrawing );

    sprintf( name, ea->eaPrintDialogFont, high/ 12 );
    apd->apdFontStruct= appDrawOpenFont( add, name );

    if  ( ! apd->apdFontStruct )
	{ SXDEB(ea->eaPrintDialogFont,apd->apdFontStruct);	}
    else{
	appDrawSetFont( add, apd->apdFontStruct );
	}

    return apd;
    }

/************************************************************************/
/*									*/
/*  Show the 'Print...' dialog.						*/
/*									*/
/*  1)	Make or just show it.						*/
/*  2)	Set the default destination.					*/
/*  3)	Make Fax View by Martin Vermeer happy.				*/
/*									*/
/************************************************************************/

void appRunPrintDialog(			EditDocument *		ed,
					const DocumentGeometry * dgDocument,
					int			pageCount,
					int			firstSelected,
					int			lastSelected,
					APP_WIDGET		printOption,
					const char *		pixmapName )
    {
    EditApplication *		ea= ed->edApplication;
    AppPrintDialog *		apd= (AppPrintDialog *)ea->eaPrintDialog;

    PrintJob			pj;
    PrintGeometry		pg;

    appPrintJobForEditDocument( &pj, ed );

    /*  1  */
    if  ( ! apd )
	{
	apd= appMakePrintDialog( ea, ed, printOption, pixmapName );

	if  ( ! apd )
	    { XDEB(apd); return;	}

	ea->eaPrintDialog= (void *)apd;

	/*  2  */
	if  ( ea->eaDefaultPrintDestination >= 0			  &&
	      ea->eaDefaultPrintDestination < ea->eaPrintDestinationCount )
	    { apd->apdDestinationChosen= ea->eaDefaultPrintDestination; }
	else{
	    if  ( apd->apdPrinterCount > 0 )
		{ apd->apdDestinationChosen= 0;	}
	    else{ apd->apdDestinationChosen= 1; }
	    }

	/*  1  */
	if  ( apd->apdActivateFaxOptions )
	    {
	    const char *	faxNumber= getenv( "APP_FAX_TO" );

	    if  ( faxNumber )
		{
		appStringToTextWidget( apd->apdFaxNumberText, faxNumber );
		apd->apdDestinationChosen= apd->apdFaxDestination;
		}
	    }
	}
    else{
	appSetShellTitle( apd->apdDialog.adTopWidget, printOption,
						    ea->eaApplicationName );

	appGuiShowDialog( &(apd->apdDialog), ed->edToplevel.atTopWidget );
	}

    apd->apdPrinterGeometry= ea->eaDefaultDocumentGeometry;
    apd->apdDocumentGeometry= *dgDocument;

    apd->apdPageCount= pageCount;
    apd->apdFirstSelectedPage= firstSelected;
    apd->apdLastSelectedPage= lastSelected;
    apd->apdFirstPageChosen= 0;
    apd->apdLastPageChosen= apd->apdPageCount- 1;

    apd->apdSelectionChosen= PRINTselAll_PAGES;
    appSetOptionmenu( &(apd->apdSelectionOptionmenu),
						apd->apdSelectionChosen );
    appPrintDialogShowFromTo( apd, 1, apd->apdPageCount, 0 );

    appGuiEnableWidget( apd->apdSelectionOptions[PRINTselCURRENT_PAGE],
			    ( apd->apdFirstSelectedPage >= 0	&&
			      apd->apdLastSelectedPage >= 0	)	);

    appSetOptionmenu( &(apd->apdPrinterOptionmenu),
						apd->apdDestinationChosen );
    appPrinterAdaptToDestination( apd );

    appPaperChooserAdaptToGeometry( &(apd->apdPaperChooser),
			apd->apdUnitType, &(apd->apdPrinterGeometry), 1 );

    appGuiRunDialog( &(apd->apdDialog), AQDrespNONE, ea );

    if  ( apd->apdPlacementChosen < 0			||
	  apd->apdPlacementChosen >= PRINTpsCOUNT	)
	{
	LLDEB(apd->apdPlacementChosen,PRINTpsCOUNT);
	appGuiHideDialog( &(apd->apdDialog) );
	return;
	}

    appPrintDialogPrintGeometry( &pg, apd );

    switch( apd->apdDialog.adResponse )
	{
	case AQDrespOK:
	    if  ( apd->apdDestinationChosen == apd->apdFileDestination )
		{
		appDocPrintToFile( printOption, apd->apdDialog.adTopWidget,
						ed, &pg,
						apd->apdFirstPageChosen,
						apd->apdLastPageChosen );
		}
	    else{
		if  ( apd->apdDestinationChosen == apd->apdFaxDestination )
		    {
		    char *	faxNumber;
		    char *	s;

		    s= faxNumber=
			appGetStringFromTextWidget( apd->apdFaxNumberText );

		    while( *s == ' ' )
			{ s++;	}

		    if  ( ! *s )
			{ SDEB(faxNumber);	}
		    else{
			if  ( appFaxDocument( ed, s, &pg,
						apd->apdFirstPageChosen,
						apd->apdLastPageChosen ) )
			    { SDEB(faxNumber);	}
			}

		    appFreeStringFromTextWidget( faxNumber );
		    }
		else{
		    if  ( appPrintDocument( apd->apdDestinationChosen, &pj, &pg,
			    apd->apdFirstPageChosen, apd->apdLastPageChosen ) )
			{ LDEB(apd->apdDestinationChosen); }
		    }
		}

	    appGuiHideDialog( &(apd->apdDialog) );
	    return;

	default:
	    LDEB(apd->apdDialog.adResponse);
	    /*FALLTHROUGH*/
	case AQDrespCANCEL:
	    appGuiHideDialog( &(apd->apdDialog) );
	    return;
	}

    }
