/************************************************************************/
/*									*/
/*  Buffer administration routines relating to the text particules in a	*/
/*  text paragraph.							*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"

/*
# define DEB_PARTICULES
*/

/************************************************************************/
/*									*/
/*  Add a Particule to a paragraph.					*/
/*									*/
/************************************************************************/

TextParticule * docCopyParticule(	BufferItem *		bi,
					int			part,
					int			off,
					int			len,
					int			kind,
					const TextParticule *	from	)
    {
    TextParticule *	tp;

    TextParticule	scratch;

    scratch= *from;

    tp= docInsertTextParticule( bi, part, off, len, kind,
						    from->tpTextAttribute );

    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    tp->tpPhysicalFont= scratch.tpPhysicalFont;

    return tp;
    }

TextParticule *	docInsertParticule(	BufferItem *	bi,
					int		n,
					int		off,
					int		len,
					int		kind )
    {
    TextParticule *	tp;

    tp= (TextParticule *)realloc( bi->biParaParticules,
		( bi->biParaParticuleCount + 1 ) * sizeof( TextParticule ) );
    if  ( ! tp )
	{ LLDEB(bi->biParaParticuleCount,tp); return tp; }
    bi->biParaParticules= tp;

    if  ( n == -1 )
	{ n= bi->biParaParticuleCount;	}
    else{
	int		i;

	for ( i= bi->biParaParticuleCount; i > n; i-- )
	    { tp[i]= tp[i-1];	}
	}

    tp += n;

    docInitTextAttribute( &tp->tpTextAttribute );

    tp->tpStroff= off;
    tp->tpStrlen= len;
    tp->tpKind= kind;
    tp->tpPhysicalFont= -1;

    tp->tpX0= 0;
    tp->tpPixelsWide= 0;

    bi->biParaParticuleCount++;

    return tp;
    }

/************************************************************************/
/*									*/
/*  Insert a text particule.						*/
/*									*/
/************************************************************************/

TextParticule *	docInsertTextParticule(	BufferItem *	bi,
					int		n,
					int		off,
					int		len,
					int		kind,
					TextAttribute	ta )
    {
    TextParticule *	tp;

    tp= docInsertParticule( bi, n, off, len, kind );
    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    tp->tpTextAttribute= ta;

    return tp;
    }

/************************************************************************/
/*									*/
/*  Delete a series of particules.					*/
/*									*/
/************************************************************************/

void docDeleteParticules(	BufferItem *	bi,
				int		first,
				int		count )
    {
    if  ( first > bi->biParaParticuleCount )
	{
	LLDEB(first,bi->biParaParticuleCount);
	first= bi->biParaParticuleCount;
	}

    if  ( first+ count > bi->biParaParticuleCount )
	{
	LLDEB(first+count,bi->biParaParticuleCount);
	count= bi->biParaParticuleCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biParaParticuleCount -= count;

    while( first < bi->biParaParticuleCount )
	{
	bi->biParaParticules[first]= bi->biParaParticules[first+ count];
	first++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Copy particules plus contents from one paragraph to another.	*/
/*									*/
/*  1)  Claim sufficient memory for the data and for the particules.	*/
/*  2)  Insert the new text into the paragraph.				*/
/*  3)  Insert the new particules into the paragraph.			*/
/*									*/
/*  4)  As this routine is the engine of the 'Paste' mechanism, make	*/
/*	links to the text itself relative.				*/
/*									*/
/************************************************************************/

int docCopyParticules(	BufferDocument *	bdTo,
			const BufferDocument *	bdFrom,
			int *			fieldMap,
			unsigned int *		pFieldUpd,
			BufferItem *		biTo,
			const BufferItem *	biFrom,
			int			partTo,
			int			partFrom,
			int			countFrom,
			int *			pParticulesInserted,
			int *			pCharactersCopied,
			const char *		refFileName )
    {
    TextParticule *		tp;
    TextParticule *		tpTo;
    const TextParticule *	tpFrom;

    int				size;
    int				stroffTo;
    int				stroffShift;

    int				i;

    int				replacedEmpty= 0;

    int				refFileSize= 0;

    if  ( refFileName )
	{ refFileSize= strlen( refFileName );	}

    /*  1  */
    size= ( biTo->biParaParticuleCount+ countFrom ) * sizeof( TextParticule );
    tp= (TextParticule *)realloc( biTo->biParaParticules, size );
    if  ( ! tp )
	{ LXDEB(size,tp); return -1;	}
    biTo->biParaParticules= tp;

    /*  2  */
    if  ( biTo->biParaStrlen == 0 && biTo->biParaParticuleCount == 1 )
	{ biTo->biParaParticuleCount--; replacedEmpty= 1;	}

    tpFrom= biFrom->biParaParticules+ partFrom;
    size= tpFrom[countFrom-1].tpStroff+ tpFrom[countFrom-1].tpStrlen-
							    tpFrom->tpStroff;

    if  ( partTo >= biTo->biParaParticuleCount )
	{
	partTo= biTo->biParaParticuleCount;
	stroffTo= biTo->biParaStrlen;
	}
    else{
	tp= biTo->biParaParticules+ partTo;
	stroffTo= tp->tpStroff;
	}

    if  ( docParaStringReplace( &stroffShift, biTo, stroffTo, stroffTo,
			    biFrom->biParaString+ tpFrom->tpStroff, size ) )
	{ LDEB(size); return -1;	}
			    
    /*  3  */
    tp= biTo->biParaParticules;
    for ( i= biTo->biParaParticuleCount- 1; i >= partTo; i-- )
	{
	tp[i+countFrom]= tp[i];
	tp[i+countFrom].tpStroff += size;
	}

    tpTo= tp+ partTo;
    for ( i= 0; i < countFrom; tpTo++, tpFrom++, i++ )
	{
	*(tpTo)= *(tpFrom);
	tpTo->tpStroff= stroffTo;

	/*  4  */
	if  ( ! refFileName || tpFrom->tpKind != DOCkindFIELDSTART )
	    {
	    if  ( docCopyParticuleData(
				bdTo, bdFrom, fieldMap, pFieldUpd,
				biTo, biFrom, tpTo, tpFrom ) )
		{ LLDEB(partTo,i); return -1;	}
	    }
	else{
	    if  ( docCopyFieldRelative( bdTo, bdFrom, fieldMap, pFieldUpd,
					    biTo, biFrom, tpTo, tpFrom,
					    refFileName, refFileSize ) )
		{ LLDEB(partTo,i); return -1;	}
	    }

	stroffTo += tpTo->tpStrlen;
	}

    biTo->biParaParticuleCount += countFrom;

    *pParticulesInserted += countFrom- replacedEmpty;
    *pCharactersCopied += stroffShift;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save paragraph contents for readers.				*/
/*									*/
/************************************************************************/

int docSaveParticules(	BufferItem *		bi,
			TextAttribute		ta,
			const unsigned char *	text,
			int			len	)
    {
    while( len > 0 )
	{
	const unsigned char *	particule= text;
	int			particuleLength= 0;
	unsigned char *		to;
	int			i;

	int	off= bi->biParaStrlen;

	while( len > 0 && text[0] != ' ' )
	    { text++; particuleLength++; len--; }
	while( len > 0 && text[0] == ' ' )
	    { text++; particuleLength++; len--; }

	/*
	appDebug( "%s(%3d): \"%.*s\"\n",
			    __FILE__, __LINE__, particuleLength, particule );
	*/

	if  ( docInflateTextString( bi, particuleLength ) )
	    { LLDEB(bi->biParaStrlen,particuleLength); return -1; }

	to= bi->biParaString+ off;
	for ( i= 0; i < particuleLength; i++ )
	    { *(to++)= *(particule++);	}
	*to= '\0';

	bi->biParaStrlen += particuleLength;

	if  ( ! docInsertTextParticule( bi, -1, off, bi->biParaStrlen- off,
							DOCkindTEXT, ta ) )
	    { LDEB(bi->biParaParticuleCount); return -1;	}
	}

    return 0;
    }

int docSaveSpecialParticule(		BufferItem *		bi,
					TextAttribute		ta,
					int			kind )
    {
    if  ( docInflateTextString( bi, 1 ) )
	{ LDEB(bi->biParaStrlen); return -1;	}

    if  ( ! docInsertTextParticule( bi, -1, bi->biParaStrlen, 1,
							    kind, ta ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    bi->biParaString[bi->biParaStrlen++]= ' ';
    bi->biParaString[bi->biParaStrlen  ]= '\0';

    return 0;
    }

/************************************************************************/
/*									*/
/*  Redivide a piece of a paragraph in particules.			*/
/*									*/
/************************************************************************/

int docRedivideStringInParticules(	BufferItem *	bi,
					int		strOff,
					int		strLen,
					int		part,
					int		partsFree,
					TextAttribute	ta )
    {
    int			bytesDone= 0;
    int			partsDone= 0;

    TextParticule *	tp= bi->biParaParticules+ part;

    while( bytesDone < strLen )
	{
	int	len= 0;
#	ifdef DEB_PARTICULES
	char *	label= "???";
#	endif

	while( bytesDone+ len < strLen			&&
	       bi->biParaString[strOff+ len] != ' '	)
	    { len++;	}
	while( bytesDone+ len < strLen			&&
	       bi->biParaString[strOff+ len] == ' '	)
	    { len++;	}

	if  ( partsDone < partsFree )
	    {
	    tp->tpStroff= strOff;
	    tp->tpStrlen= len;
	    tp->tpKind= DOCkindTEXT;
	    tp->tpTextAttribute= ta;

	    tp->tpPhysicalFont= -1;
	    tp->tpX0= -1;
	    tp->tpPixelsWide= 0;

#	    ifdef DEB_PARTICULES
	    label= "NW.";
#	    endif
	    }
	else{
	    tp= docInsertTextParticule( bi, part,
					    strOff, len, DOCkindTEXT, ta );
	    if  ( ! tp )
		{ XDEB(tp); return -1;	}
#	    ifdef DEB_PARTICULES
	    label= "NW+";
#	    endif
	    }

#	ifdef DEB_PARTICULES
	appDebug( "%s %3d: [%4d..%4d] %s <%s> \"%.*s\" len= %d\n", label, part,
		    tp->tpStroff,
		    tp->tpStroff+ tp->tpStrlen,
		    docKindStr( tp->tpKind ),
		    docAttributeStr( tp->tpTextAttribute ),
		    (int)tp->tpStrlen,
		    bi->biParaString+ tp->tpStroff,
		    tp->tpStrlen );
#	endif

	strOff += len; bytesDone += len;
	partsDone++; part++, tp++;
	}

    return partsDone;
    }

/************************************************************************/
/*									*/
/*  Shift the particules in a paragraph after an insertion or a		*/
/*  deletion.								*/
/*									*/
/************************************************************************/

int docShiftParticuleOffsets(	BufferDocument *	bd,
				BufferItem *		paraBi,
				int			partFrom,
				int			partUpto,
				int			stroffShift )
    {
    int			part;
    TextParticule *	tp;

    if  ( stroffShift < 0 )
	{
	tp= paraBi->biParaParticules+ partFrom;
	for ( part= partFrom; part < partUpto; tp++, part++ )
	    {
	    if  ( tp->tpKind == DOCkindNOTE )
		{
		DocumentNote *	dn;

		if  ( docGetNote( &dn, bd, paraBi, tp->tpStroff ) < 0 )
		    { LDEB(tp->tpStroff);		}
		else{ dn->dnStroff += stroffShift;	}
		}

	    tp->tpStroff += stroffShift;
	    }
	}

    if  ( stroffShift > 0 )
	{
	tp= paraBi->biParaParticules+ partUpto- 1;
	for ( part= partUpto- 1; part >= partFrom; tp--, part-- )
	    {
	    if  ( tp->tpKind == DOCkindNOTE )
		{
		DocumentNote *	dn;

		if  ( docGetNote( &dn, bd, paraBi, tp->tpStroff ) < 0 )
		    { LDEB(tp->tpStroff);		}
		else{ dn->dnStroff += stroffShift;	}
		}

	    tp->tpStroff += stroffShift;
	    }
	}

    return 0;
    }

