/*
** Copyright (C) 1995, 1996, 1997, 1998 Hewlett-Packard Company
** Originally by Kevin Hughes, kev@kevcom.com, 3/11/94
**
** This program and library is free software; you can redistribute it and/or
** modify it under the terms of the GNU (Library) General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU (Library) General Public License for more details.
**
** You should have received a copy of the GNU (Library) General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**---------------------------------------------------------
** Added support for METADATA
** G. Hill ghill@library.berkeley.edu 3/18/97
**
** Added printing of common words
** G.Hill 4/7/97 ghill@library.berkeley.edu
**
** change sprintf to snprintf to avoid corruption
** added safestrcpy() macro to avoid corruption from strcpy overflow
** added INDEX_READ_ONLY option to make "safe" exe that can only READ index files
**   (INDEX_READ_ONLY may be defined in the Makefile or by uncommenting define)
** SRE 11/17/99
**
** fixed cast to int and unused variable problems pointed out by "gcc -Wall"
** fixed variable declarations for INDEX_READ_ONLY to avoid unused variables
** SRE 2/22/00
*/

#define MAIN_FILE
#include "swish.h"

#include "error.h"
#include "list.h"
#include "search.h"
#include "index.h"
#include "string.h"
#include "file.h"
#include "http.h"
#include "merge.h"
#include "docprop.h"
#include "mem.h"


/*
** This array has pointers to all the indexing data source
** structures
*/
extern struct _indexing_data_source_def *data_sources[];

int main(argc, argv)
int argc;
char **argv;
{
char c;
char *word=NULL, *wordlist=NULL, *maxhitstr=NULL, *structstr=NULL, *beginhitstr=NULL;
int lenword=0, lenwordlist=0, lenmaxhitstr=0, lenstructstr=0, lenbeginhitstr=0;
int i, hasindex, hasdir, hasconf, hasverbose, structure, stopwords, index, decode, merge, hasMetaName;
long plimit, flimit;
FILE *fp1;
struct swline *conflist, *tmplist, *tmplist2;
int totalfiles, pos, j;
long offsetstart, hashstart, starttime, stoptime;
int lentmpindex1=0;
char *tmpindex1=NULL;
int lentmpindex2=0;
char *tmpindex2=NULL;
int lenindex1=0;
char *index1=NULL;
int lenindex2=0;
char *index2=NULL;
int lenindex3=0;
char *index3=NULL;
int lenindex4=0;
char *index4=NULL;
FILE *fp2;
struct file *filep;
struct entryarray *entryp;
int INDEX_READ_ONLY;
char *tmp;

	starttime=0L;

	if(!lenindex1)index1=emalloc((lenindex1=MAXSTRLEN)+1);index1[0]='\0';
	if(!lenindex2)index2=emalloc((lenindex2=MAXSTRLEN)+1);index2[0]='\0';
	if(!lenindex3)index3=emalloc((lenindex3=MAXSTRLEN)+1);index3[0]='\0';
	if(!lenindex4)index4=emalloc((lenindex4=MAXSTRLEN)+1);index4[0]='\0';
	if(!lentmpindex1)tmpindex1=emalloc((lentmpindex1=MAXSTRLEN)+1);tmpindex1[0]='\0';
	if(!lentmpindex2)tmpindex2=emalloc((lentmpindex2=MAXSTRLEN)+1);tmpindex2[0]='\0';
	
	
	/* If argv[0] is swish-search then you cannot index and/or merge */
	if(!(tmp=strrchr(argv[0],DIRDELIMITER))) 
		tmp=argv[0];
	else tmp++;
		/* We must ignore case for WIN 32 */
	if(strcasecmp(tmp,"swish-search")==0)
		INDEX_READ_ONLY=1;
	else
		INDEX_READ_ONLY=0;
	index = decode = merge = 0;
	hasindex = hasdir = hasconf = hasverbose = hasMetaName = 0;
	followsymlinks = stopwords = stopPos = 0;
	totalwords = totalfiles = stopwords = 0;
	stopList = NULL;
	applyStemmingRules = 0;				/* added 11/24/98 */
	applySoundexRules = 0;			/* added DN 09/01/99 */
	useCustomOutputDelimiter = 0;		/* added 11/24/98 */
	ignoreTotalWordCountWhenRanking = 0; /* added 11/24/98 */
	filelist = NULL;
	entrylist = NULL;
	dirlist = indexlist = conflist = tmplist = NULL;
	replacelist = NULL;
	metaEntryList = NULL;
	maxhits = -1;
	beginhits = 0;
	verbose = VERBOSE;
	plimit = PLIMIT;
	flimit = FLIMIT;
	minwordlimit = MINWORDLIMIT;
	maxwordlimit = MAXWORDLIMIT;
	indexComments = 1;
	nocontentslist = NULL;
	suffixlist = NULL;
	applyautomaticmetanames = 0;
	makeallstringlookuptables();
	allocatedefaults();
	wordchars = SafeStrCopy(wordchars,WORDCHARS,&lenwordchars);
	sortstring(wordchars);  /* Sort chars and remove dups */
	makelookuptable(wordchars,wordcharslookuptable);
	beginchars = SafeStrCopy(beginchars,BEGINCHARS,&lenbeginchars);
	sortstring(beginchars);  /* Sort chars and remove dups */
	makelookuptable(beginchars,begincharslookuptable);
	endchars = SafeStrCopy(endchars,ENDCHARS,&lenendchars);
	sortstring(endchars);  /* Sort chars and remove dups */
	makelookuptable(endchars,endcharslookuptable);
	ignorelastchar = SafeStrCopy(ignorelastchar,IGNORELASTCHAR,&lenignorelastchar);
	sortstring(ignorelastchar);  /* Sort chars and remove dups */
	makelookuptable(ignorelastchar,ignorelastcharlookuptable);
	ignorefirstchar = SafeStrCopy(ignorefirstchar,IGNOREFIRSTCHAR,&lenignorefirstchar);
	sortstring(ignorefirstchar);  /* Sort chars and remove dups */
	makelookuptable(ignorefirstchar,ignorefirstcharlookuptable);
	structure = 1;
	if(!lenwordlist)wordlist=(char *)emalloc((lenwordlist=MAXSTRLEN) + 1);
	wordlist[0] = '\0';
	if(!lenstructstr)structstr=(char *)emalloc((lenstructstr=MAXSTRLEN)+1);
	structstr[0] = '\0';
	setlocale(LC_CTYPE,"");
        filterdir[0] ='\0';
        filterlist = NULL;

	/* Initialize tmpdir */
	tmpdir = SafeStrCopy(tmpdir,TMPDIR,&lentmpdir);

	/* Initialize spider directory */
	spiderdirectory = SafeStrCopy(spiderdirectory,SPIDERDIRECTORY,&lenspiderdirectory);

	/* By default we are set up to use the first data source in the list */
 	IndexingDataSource = data_sources[0];
	
	if (argc == 1)
		usage();
	while (--argc > 0) 
	{
		++argv;
		if ((*argv)[0] != '-')
			usage();
		c = (*argv)[1];
		if ((*argv)[2] != '\0' && isalpha((int)(*argv)[2])) /* cast to int, 2/22/00 */
			usage();
		if (c == 'i') 
		{
			index = 1;
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') {
				dirlist = (struct swline *)
					addswline(dirlist, (++argv)[0]);
				argc--;
			}
		}
		else if (c == 'w') 
		{
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') 
			{
				word = SafeStrCopy(word, (++argv)[0],&lenword);
				argc--;
				if((int)(strlen(wordlist) +  strlen(" ") + strlen(word)) >= lenwordlist) {
					lenwordlist =strlen(wordlist) +  strlen(" ") + strlen(word) + 200;
					wordlist = (char *) erealloc(wordlist,lenwordlist + 1);
				}
				sprintf(wordlist,"%s%s%s", wordlist, (wordlist[0] == '\0') ? "" : " ", word);
			}
		}
		else if (c == 'S')
		{
		  struct _indexing_data_source_def **data_source;
		  const char* opt = (++argv)[0];
		  argc--;
		  for (data_source = data_sources; *data_source != 0; data_source++ )

		    {
		      if (strcmp(opt, (*data_source)->IndexingDataSourceId) == 0)
			{
			  break;
			}
		    }
		  
		  if (!*data_source)
		    {
		      snprintf(errorstr, MAXSTRLEN, "Unknown -S option \"%s\"", opt);
		      progerr(errorstr);
		    }
		  else
		    {
		      IndexingDataSource = *data_source;
		    }  
		}
		else if (c == 'p') {
		  /* -p <property_to_display> [<property_to_display>]* */
		  while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') 
			{
				addSearchResultDisplayProperty((++argv)[0]);
				argc--;
			}
		}
		else if (c == 's') {
		  /* -s <property_to_sort> [<property_to_sort>]* */
		  while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') 
			{
				addSearchResultSortProperty((++argv)[0]);
				argc--;
			}
		}
		else if (c == 'f') 
		{
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') 
			{
				indexlist = (struct swline *)
					addswline(indexlist, (++argv)[0]);
				argc--;
			}
		}
		else if (c == 'c') 
		{
			index = 1;
			hasconf = 1;
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') {
				conflist = (struct swline *)
					addswline(conflist, (++argv)[0]);
				argc--;
			}
		}
		else if (c == 'C') {
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-')
			{
				conflist = (struct swline *)
					addswline(conflist, (++argv)[0]);
				argc--;
			}
			if (conflist == NULL)
				progerr("Specifiy the configuration file.");
			else
				hasMetaName = 1;
		      }
		else if (c == 'l') {
			followsymlinks = 1;
			argc--;
		}
		else if (c == 'b') {
			if ((argv + 1)[0] == '\0')
				beginhits = 0;
			else {
				beginhitstr = SafeStrCopy(beginhitstr, (++argv)[0],&lenbeginhitstr);
				if (isdigit((int)beginhitstr[0]))
					beginhits = atoi(beginhitstr);
				else
					beginhits = 0;
				argc--;
			}
		}
		else if (c == 'm') {
			if ((argv + 1)[0] == '\0')
				maxhits = -1;
			else {
				maxhitstr = SafeStrCopy(maxhitstr, (++argv)[0],&lenmaxhitstr);
				if (lstrstr(maxhitstr, "all"))
					maxhits = -1;
				else if (isdigit((int)maxhitstr[0])) /* cast to int, 2/22/00 */
					maxhits = atoi(maxhitstr);
				else
					maxhits = -1;
				argc--;
			}
		}
		else if (c == 't') {
			if ((argv + 1)[0] == '\0')
				progerr("Specify tag fields (HBtheca).");
			else {
				structure = 0;
				structstr = SafeStrCopy(structstr,(++argv)[0],&lenstructstr);
				argc--;
			}
		}
		else if (c == 'v') {
			hasverbose = 1;
			if ((argv + 1)[0] == '\0') {
				verbose = 3;
				break;
			}
			else if (!isdigit((int)(argv + 1)[0][0])) /* cast to int, 2/22/00 */
				verbose = 3;
			else
				verbose = atoi((++argv)[0]);
			argc--;
		}
		else if (c == 'V')
			printversion();
		else if (c == 'z' || c == 'h' || c == '?')
			usage();
		else if (c == 'M') {
			merge = 1;
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') {
				indexlist = (struct swline *)
					addswline(indexlist, (++argv)[0]);
				argc--;
			}
		}
		else if (c == 'D') {
			decode = 1;
			while ((argv + 1)[0] != '\0' && *(argv + 1)[0] != '-') {
				indexlist = (struct swline *)
					addswline(indexlist, (++argv)[0]);
				argc--;
			}
		}
		else if (c == 'd')
		{
			/* added 11/24/98 MG */
			if (((argv + 1)[0] != '\0') && (*(argv + 1)[0] != '-'))
			{
				useCustomOutputDelimiter = 1;
				customOutputDelimiter=SafeStrCopy(customOutputDelimiter, (++argv)[0],&lencustomOutputDelimiter);
				if (strcmp(customOutputDelimiter, "dq") == 0)
					{ customOutputDelimiter=SafeStrCopy(customOutputDelimiter, "\"",&lencustomOutputDelimiter); } /* double quote is cool */
				argc--;
			}
		}
		else
			usage();
		if (argc == 0)
			break;
	}

	hasdir = (dirlist == NULL) ? 0 : 1;
	hasindex = (indexlist == NULL) ? 0 : 1;
	
	if (index && merge)
		index = 0;
	
	if (decode) {
		
		if (!hasindex)
			progerr("Specify the index file to decode.");
		
		while (indexlist != NULL) {
			
			if ((fp1 = openIndexFileForRead(indexlist->line)) == NULL) {
				snprintf(errorstr,
					MAXSTRLEN, "Couldn't open the index file \"%s\".",
					indexlist->line);
				progerr(errorstr);
			}
			if (!isokindexheader(fp1)) {
				snprintf(errorstr,
					MAXSTRLEN, "\"%s\" has an unknown format.",
					indexlist->line);
				progerr(errorstr);
			}
			
			decompress(fp1);
			putchar('\n');
			fclose(fp1);
			
			indexlist = indexlist->next;
		}
		exit(0);
		
	}
	else if (index) 
	{
		if(INDEX_READ_ONLY) {
			/* new "safe" mode, added 11/99 to optionally build exe that cannot write index files */
			printf("CANNOT INDEX Data Source: \"%s\"\nREAD-ONLY MODE\n", IndexingDataSource->IndexingDataSourceName);
			exit(1);
		}
		printf("Indexing Data Source: \"%s\"\n", IndexingDataSource->IndexingDataSourceName);

		if (hasconf)
		{
			while (conflist != NULL) {
				getdefaults(conflist->line, &hasdir, &hasindex,
					&plimit, &flimit, hasverbose);
				conflist = conflist->next;
			}
		}

		if (!hasindex)
			indexlist = (struct swline *) addswline(indexlist, INDEXFILE);

		if (!hasdir)
			progerr("Specify directories or files to index.");
		
		if (verbose < 0)
			verbose = 0;
		if (verbose > 4)
			verbose = 4;
		if (verbose)
			starttime = getthetime();

		while (dirlist != NULL) {
			if (verbose) {
				printf("Indexing %s..\n",dirlist->line);
				fflush(stdout);
			}
			indexpath(dirlist->line);
			dirlist = dirlist->next;
		}

		if ((fp1 = openIndexFileForWrite(indexlist->line)) == NULL) {
			snprintf(errorstr,
				MAXSTRLEN, "Couldn't write the index file \"%s\".",
				indexlist->line);
			progerr(errorstr);
		}

		if (verbose > 1)
			putchar('\n');
		if (verbose)
			printf("Removing very common words...\n");
		fflush(stdout);

		filep = filelist;
		totalfiles = getfilecount(filep);

		entryp = entrylist;
		stopwords = removestops(entryp, totalfiles, plimit, flimit);
		
		if (verbose) {
			if (stopwords) {
				/* 05/00 Jose Ruiz
				** Adjust totalwords
				*/
				totalwords -= stopwords;
				if(totalwords<0) totalwords=0;
				printf("%d word%s removed.\n",
					stopwords, (stopwords == 1) ? "" : "s");
				printf("%d words removed not in common words array:\n", stopPos);
				for (pos = 0; pos < stopPos; pos++) 
					printf("%s, ", stopList[pos]);
				printf("\n");
			}
			else
				printf("no words removed.\n");
			printf("Writing main index...\n");
		}

			/* Compute hash table */
		if (verbose)
			printf("Computing hash table ...\n");

		for (i=0; i<SEARCHHASHSIZE; i++) hashentries[i] = NULL;
		computeHash(entrylist);

		if (verbose)
			printf("Writing header ...\n");
		printheader(fp1, indexlist->line, totalwords, totalfiles, 0);
		
		offsetstart = ftell(fp1);
		for (i = 0; i < MAXCHARS; i++)
			printlong(fp1,(long)0);
		fputc('\n', fp1);
		
		hashstart = ftell(fp1);
		for (i = 0; i < SEARCHHASHSIZE; i++)
			printlong(fp1,(long)0);
		fputc('\n', fp1);

		if (verbose)
			printf("Writing index entries ...\n");
		printindex(entrylist, fp1);
		if (verbose)
			printf("Writing stopwords ...\n");
		printstopwords(fp1);
		
		if (verbose) {
			if (totalwords)
				printf("%d unique word%s indexed.\n",
				totalwords, (totalwords == 1) ? "" : "s");
			else
				printf("no unique words indexed.\n");
			printf("Writing file index...\n");
		}
		
		if(verbose)
			printf("Writing file list ...\n");
		printfilelist(filelist, fp1);

		if(verbose)
			printf("Writing file offsets ...\n");
		printfileoffsets(fp1);

		if(verbose)
			printf("Writing MetaNames ...\n");
		printMetaNames(fp1);
		fclose(fp1);
		if(verbose)
			printf("Writing offsets (2)...\n");
		fp2 = openIndexFileForReadAndWrite(indexlist->line);
		fseek(fp2, offsetstart, 0);
		for (i = 0; i < MAXCHARS; i++)
			printlong(fp2,offsets[i]);

		fseek(fp2, hashstart, 0);
		for (i = 0; i < SEARCHHASHSIZE; i++)
			printlong(fp2,hashoffsets[i]);
		printhash(fp2);
		fclose(fp2);
	
		if (verbose) 
		{
			if (totalfiles)
				printf("%d file%s indexed.\n", totalfiles,
				(totalfiles == 1) ? "" : "s");
			else
				printf("no files indexed.\n");
			
			stoptime = getthetime();
			printrunning(starttime, stoptime);
			printf("Indexing done!\n");
		}
#ifdef INDEXPERMS
		chmod(indexlist->line, INDEXPERMS);
#endif

#ifdef DEBUGMEMORY
        checkmem();
#endif
		exit(0);
	}
	else if (merge) 
	{
		if(INDEX_READ_ONLY) {
			/* new "safe" mode, added 11/99 to optionally build exe that cannot write index files */
			printf("CANNOT MERGE Data Source: \"%s\"\nREAD-ONLY MODE\n", IndexingDataSource->IndexingDataSourceName);
			exit(1);
		}
		if (indexlist == NULL)
			progerr("Specify index files and an output file.");
		if (hasconf)
		{
			while (conflist != NULL) 
			{
				getdefaults(conflist->line, &hasdir, &hasindex,
					&plimit, &flimit, hasverbose);
				conflist = conflist->next;
			}
		}
		
		tmplist = indexlist;
		for (i = 0; tmplist != NULL; i++) {
			index4=SafeStrCopy(index4, tmplist->line, &lenindex4);
			tmplist = tmplist->next;
		}
		j = i - 2;
		if (i < 3)
			progerr("Specify index files and an output file.");
		
		tmpindex1=SafeStrCopy(tmpindex1, tmpnam(NULL),&lentmpindex1);
		tmpindex2=SafeStrCopy(tmpindex2, tmpnam(NULL),&lentmpindex2);
		
		i = 1;
		index1=SafeStrCopy(index1, indexlist->line,&lenindex1);
		indexlist = indexlist->next;
		while (i <= j) {
			index2=SafeStrCopy(index2, indexlist->line,&lenindex2);
			if (i % 2) {
				if (i != 1)
					{ index1=SafeStrCopy(index1, tmpindex2, &lenindex1); }
				index3=SafeStrCopy(index3,tmpindex1,&lenindex3);
			}
			else {
				index1=SafeStrCopy(index1,tmpindex1,&lenindex1);
				index3=SafeStrCopy(index3,tmpindex2,&lenindex3);
			}
			if (i == j)
				{index3=SafeStrCopy(index3,index4,&lenindex3);}
			readmerge(index1, index2, index3);
			indexlist = indexlist->next;
			i++;
		}
#ifdef INDEXPERMS
		chmod(index3, INDEXPERMS);
#endif
		if (isfile(tmpindex1))
			remove(tmpindex1);
		if (isfile(tmpindex2))
			remove(tmpindex2);
	
#ifdef DEBUGMEMORY
        checkmem();
#endif
	
	}
	else 
	{
		
		for (i = 0; structstr[i] != '\0'; i++)
		{
			switch (structstr[i]) 
			{
			case 'H':
				structure |= IN_HEAD;
				break;
			case 'B':
				structure |= IN_BODY;
				break;
			case 't':
				structure |= IN_TITLE;
				break;
			case 'h':
				structure |= IN_HEADER;
				break;
			case 'e':
				structure |= IN_EMPHASIZED;
				break;
			case 'c':
				structure |= IN_COMMENTS;
				break;
			default:
				structure |= IN_FILE;
				break;
			}
		}
		
		if (maxhits <= 0)
			maxhits = -1;
		if (!hasindex)
			indexlist = (struct swline *)
			addswline(indexlist, INDEXFILE);
		if (hasMetaName)
			while (conflist != NULL) {
			getdefaults(conflist->line, &hasdir, &hasindex,
				&plimit, &flimit, hasverbose);
			conflist = conflist->next;
		}
		search(wordlist, indexlist, structure);
			/* Free indexlist */
        	tmplist = indexlist;
        	while (tmplist) {
                	tmplist2 = tmplist->next;
                	efree(tmplist->line);
                	efree(tmplist);
                	tmplist = tmplist2;
        	}
			/* Free conflist */
        	tmplist = conflist;
        	while (tmplist) {
                	tmplist2 = tmplist->next;
                	efree(tmplist->line);
                	efree(tmplist);
                	tmplist = tmplist2;
		}
		if(lenword) efree(word);
		if(lenwordlist) efree(wordlist);
		if(lenmaxhitstr) efree(maxhitstr);
		if(lenstructstr) efree(structstr);
		if(lenbeginhitstr) efree(beginhitstr);
		if(lenindexn) efree(indexn);
		if(lenindexa) efree(indexa);
		if(lenindexp) efree(indexp);
		if(lenindexd) efree(indexd);
		if(lencustomOutputDelimiter) efree(customOutputDelimiter);
	}

#ifdef DEBUGMEMORY
	checkmem();
#endif	
	exit(0);
	return 0;
}

/* Gets the current time in seconds since the epoch.
*/

long getthetime()
{
	long thetime;
	time_t tp;
	
	thetime = (long) time(&tp);
	return thetime;
}

/* Prints the running time (the time it took for indexing).
*/

void printrunning(starttime, stoptime)
     long starttime;
     long stoptime;
{
	int minutes, seconds;
	
	minutes = (stoptime - starttime) / SECSPERMIN;
	seconds = (stoptime - starttime) % SECSPERMIN;
	printf("Running time: ");
	if (minutes)
		printf("%d minute%s", minutes, (minutes == 1) ? "" : "s");
	if (minutes && seconds)
		printf(", ");
	if (seconds)
		printf("%d second%s", seconds, (seconds == 1) ? "" : "s");
	if (!minutes && !seconds)
		printf("Less than a second");
	printf(".\n");
}

/* Prints the SWISH usage.
*/

void usage()
{
	const char* defaultIndexingSystem = "";

	printf("  usage: swish [-i dir file ... ] [-S system] [-c file] [-f file] [-l] [-v (num)]\n");
	printf("         swish -w word1 word2 ... [-f file1 file2 ...] [-p prop1 ...] [-s sortprop] [-m num] [-t str] [-d delim]\n");
	printf("         swish -M index1 index2 ... outputfile\n");
	printf("         swish -D file\n");
	printf("         swish -V\n");
	putchar('\n');
	printf("options: defaults are in brackets\n");


	printf("         -S : specify which indexing system to use.\n");
	printf("              Valid options are:\n");
	#ifdef ALLOW_FILESYSTEM_INDEXING_DATA_SOURCE
	printf("              \"fs\" - index local files in your File System\n");
	if (!*defaultIndexingSystem)
		defaultIndexingSystem = "fs";
	#endif
	#ifdef ALLOW_HTTP_INDEXING_DATA_SOURCE
	printf("              \"http\" - index web site files using a web crawler\n");
	if (!*defaultIndexingSystem)
		defaultIndexingSystem = "http";
	#endif
	printf("              The default value is: \"%s\"\n", defaultIndexingSystem);

	printf("         -i : create an index from the specified files\n");
	printf("         -w : search for words \"word1 word2 ...\"\n");
	printf("         -t : tags to search in - specify as a string\n");
	printf("              \"HBthec\" - in head, body, title, header,\n");
	printf("              emphasized, comments\n");
	printf("         -f : index file to create or search from [%s]\n", INDEXFILE);
	printf("         -c : configuration file to use for indexing\n");
	printf("         -v : verbosity level (0 to 3) [%d]\n", VERBOSE);
	printf("         -l : follow symbolic links when indexing\n");
	printf("         -b : begin results at this number\n");
	printf("         -m : the maximum number of results to return [%d]\n", MAXHITS);
	printf("         -M : merges index files\n");
	printf("         -D : decodes an index file\n");
	printf("         -p : include these document properties in the output \"prop1 prop2 ...\"\n");
	printf("         -s : sort by these document properties in the output \"prop1 prop2 ...\"\n");
	printf("         -d : next param is delimiter. use \"-d dq\" to use a double quote\n");
	printf("         -V : prints the current version\n\n");
	printf("version: %s\n", VERSION);
	printf("   docs: http://sunsite.berkeley.edu/SWISH-E/\n");
	exit(1);
}

void printversion()
{
	printf("SWISH-E %s\n", VERSION);
	exit(0);
}

/*
 * Binary files must be open with the "b" option under Win32, so all
 * fopen() calls to index files have to go through these routines to 
 * keep the code portable. 
 * Note: text files should be opened normally, without the "b" option,
 * otherwise end-of-line processing is not done correctly (on Win32).
 */
#ifdef _WIN32
#define FILEMODE_READ		"rb"
#define FILEMODE_WRITE		"wb"
#define FILEMODE_READWRITE	"rb+"
#else
#define FILEMODE_READ		"r"
#define FILEMODE_WRITE		"w"
#define FILEMODE_READWRITE	"r+"
#endif

FILE* openIndexFileForWrite(filename)
char* filename;
{
	return fopen(filename, FILEMODE_WRITE);
}

FILE* openIndexFileForRead(filename)
char* filename;
{
	return fopen(filename, FILEMODE_READ);
}

FILE* openIndexFileForReadAndWrite(filename)
char* filename;
{
	return fopen(filename, FILEMODE_READWRITE);
}


/*
 * Invoke the methods of the current Indexing Data Source
 */
void indexpath(path)
char *path;
{
	/* invoke routine to index a "path" */
	(*IndexingDataSource->indexpath_fn)(path);
}

int vgetc(vp)
void *vp;
{
	/* invoke routine to get char from "file" */
	return (*IndexingDataSource->vgetc_fn)(vp);
}

int vsize(vp)
void *vp;
{
	/* invoke routine to get size of "file" */
	return (*IndexingDataSource->vsize_fn)(vp);
}

int vtell(vp)
void *vp;
{
        /* invoke routine to get position in "file" */
        return (*IndexingDataSource->vtell_fn)(vp);
}

int vseek(vp,pos)
void *vp;
long pos;
{
        /* invoke routine to set pointer in "file" */
        return (*IndexingDataSource->vseek_fn)(vp,pos);
}

int parseconfline(line)
char *line;
{
	/* invoke routine to parse config file lines */
	return (*IndexingDataSource->parseconfline_fn)(line);
}

void allocatedefaults()
{
	lenwordchars=lenbeginchars=lenendchars=lenignorelastchar=lenignorefirstchar=lentranslatechars1=lentranslatechars2=MAXCHARDEFINED;
	wordchars = (char *)emalloc(lenwordchars + 1);
	wordchars[0]='\0';
	beginchars = (char *)emalloc(lenbeginchars + 1);
	beginchars[0]='\0';
	endchars = (char *)emalloc(lenendchars + 1);
	endchars[0]='\0';
	ignorelastchar = (char *)emalloc(lenignorelastchar + 1);
	ignorelastchar[0]='\0';
	ignorefirstchar = (char *)emalloc(lenignorefirstchar + 1);
	ignorefirstchar[0]='\0';
	translatechars1 = (char *)emalloc(lentranslatechars1 + 1);
	translatechars1[0]='\0';
	translatechars2 = (char *)emalloc(lentranslatechars2 + 1);
	translatechars2[0]='\0';

	lenindexedon=lensavedasheader=lenindexn=lenindexd=lenindexp=lenindexa=MAXSTRLEN;
	indexn = (char *)emalloc(lenindexn + 1);indexn[0]='\0';
	indexd = (char *)emalloc(lenindexd + 1);indexd[0]='\0';
	indexp = (char *)emalloc(lenindexp + 1);indexp[0]='\0';
	indexa = (char *)emalloc(lenindexa + 1);indexa[0]='\0';
	savedasheader = (char *)emalloc(lensavedasheader + 1);savedasheader[0]='\0';
	indexedon = (char *)emalloc(lenindexedon + 1);indexedon[0]='\0';

	lenfilterdir=lentmpdir=lenspiderdirectory=MAXSTRLEN;
	filterdir = (char *)emalloc(lenfilterdir + 1);filterdir[0]='\0';
	tmpdir = (char *)emalloc(lentmpdir + 1);tmpdir[0]='\0';
	spiderdirectory = (char *)emalloc(lenspiderdirectory + 1);spiderdirectory[0]='\0';
	
	lencustomOutputDelimiter=5;
	customOutputDelimiter=(char *)emalloc(lencustomOutputDelimiter+1); customOutputDelimiter[0]='\0';
	
}
