/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SubtitleSubRip.h"
#include <iostream>
#include <fstream>
#include <glibmm/ustring.h>
#include "utility.h"
#include "RegEx.h"

/*
 *
 */
Glib::ustring SubtitleSubRip::get_name()
{
	return "SubRip";
}

/*
 *
 */
Glib::ustring SubtitleSubRip::get_extension()
{
	return "srt";
}

/*
 *
 */
bool SubtitleSubRip::check(const std::string &line)
{
	static RegEx ex(
				"([0-9]{2}|[0-9]):[0-9]{2}:[0-9]{2},[0-9]{3}"
				" --> "
				"([0-9]{2}|[0-9]):[0-9]{2}:[0-9]{2},[0-9]{3}");

	return ex.exec(line);
}

/*
 *
 */
SubtitleSubRip::SubtitleSubRip(Document* doc)
:SubtitleFormat(doc)
{
	se_debug(SE_DEBUG_LOADER | SE_DEBUG_SAVER);
}

/*
 *
 */
SubtitleSubRip::~SubtitleSubRip()
{
	se_debug(SE_DEBUG_LOADER | SE_DEBUG_SAVER);
}


/*
 *
 */
bool SubtitleSubRip::open(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	se_debug(SE_DEBUG_LOADER);

	SubtitleFormat::open(filename,encoding);
	
	std::ifstream file(filename.c_str());
	if(!file)
	{
		throw SubtitleException("SubtitleSubRip", _("I can't open this file."));
	}

	std::string line;
	while(!file.eof() && std::getline(file, line))
	{
		unsigned int num = 0;

		// lecture du numero
		if(sscanf(line.c_str(), "%d", &num) > 0)
		{
			// lecture du temps
			if(std::getline(file, line))
			{
				int start[4], end[4];

				if(sscanf(line.c_str(), "%d:%d:%d,%d --> %d:%d:%d,%d",
							&start[0], &start[1], &start[2], &start[3], 
							&end[0], &end[1], &end[2], &end[3]) == 8)
				{
					Glib::ustring text;
					int count = 0; // permet de compter le nombre de ligne

					while(std::getline(file, line))
					{
						line = check_end_char(charset_to_utf8(line));

						if(line.size()==0)
							break;
						else
						{
							if(count > 0) // s'il y a deja une ligne alors on utlise le | pour le retour a la ligne
								text += get_newline();

							text += line;
							++count;
						}
					}

					SubtitleModifier subtitle(m_subtitleModel->append());

					subtitle.set_text(text);
					subtitle.set_start(SubtitleTime(start[0], start[1], start[2], start[3]));
					subtitle.set_end(SubtitleTime(end[0], end[1], end[2], end[3]));
				}
			}
		}
	}

	file.close();

	return true;
}

/*
 *
 */
bool SubtitleSubRip::save(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	se_debug(SE_DEBUG_SAVER);

	SubtitleFormat::save(filename,encoding);

	std::ofstream file(filename.c_str());
	if(!file)
	{
		throw SubtitleException("SubtitleSubRip", _("I can't open this file."));
	}

	Glib::ustring text;

	Gtk::TreeNodeChildren rows = m_subtitleModel->children();
	for(Gtk::TreeIter it = rows.begin(); it; ++it)
	{
		SubtitleModifier subtitle(*it);

		file << subtitle.get_num() 
			<< std::endl 
			<< subtitletime_to_subrip_time(subtitle.get_start().str()) 
			<< " --> " 
			<< subtitletime_to_subrip_time(subtitle.get_end().str()) 
			<< std::endl;

		text = subtitle.get_text();

		// pas besoin on utilise déjà \n
		//newline_to_characters(text, "\n");

		file << utf8_to_charset(text) << std::endl;
		file << std::endl;
	}
	
	file.close();
	return true;
}


Glib::ustring SubtitleSubRip::subtitletime_to_subrip_time(const Glib::ustring &time)
{
	SubtitleTime t(time);

	char *tmp = g_strdup_printf("%02i:%02i:%02i,%03i",
			t.hours, t.mins, t.secs, t.msecs);

	Glib::ustring str(tmp);

	g_free(tmp);

	return str;
}

