/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <glib/gi18n.h>
#include "sg-util.h"
#include "sgtk-util.h"
#include "st-plugins-preferences-page.h"
#include "st-plugins.h"
#include "st-util-api.h"
#include "st-util.h"
#include "st-settings.h"
#include "st-stock.h"

/*** type definitions ********************************************************/

enum {
  COLUMN_PLUGIN,

  COLUMN_ENABLED,
  COLUMN_ICON,
  COLUMN_LABEL,

  N_COLUMNS
};
    
/*** function declarations ***************************************************/

static void st_plugins_preferences_page_init (STPluginsPreferencesPage *page);
static void st_plugins_preferences_page_toggled_h (GtkCellRendererToggle *renderer,
						   const char *path,
						   gpointer user_data);

/*** implementation **********************************************************/

GType
st_plugins_preferences_page_get_type (void)
{
  static GType type = 0;
  
  if (! type)
    {
      static const GTypeInfo info = {
	sizeof(STPluginsPreferencesPageClass),
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	sizeof(STPluginsPreferencesPage),
	0,
	(GInstanceInitFunc) st_plugins_preferences_page_init,
      };

      type = g_type_register_static(ST_TYPE_PREFERENCES_PAGE,
				    "STPluginsPreferencesPage",
				    &info,
				    0);
    }

  return type;
}

static void
st_plugins_preferences_page_init (STPluginsPreferencesPage *page)
{
  STPreferencesPage *ppage = ST_PREFERENCES_PAGE(page);
  GtkWidget *label;
  GtkListStore *store;
  GSList *l;
  GtkWidget *view;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GtkWidget *scrolled;

  st_preferences_page_set_name(ppage, "plugins");
  st_preferences_page_set_stock_id(ppage, ST_STOCK_PLUGIN);
  st_preferences_page_set_label(ppage, _("Plugins"));
  st_preferences_page_set_help_link_id(ppage, "preferences-plugins");

  label = gtk_label_new(_("Choose the plugins to load at startup."));
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(page), label, FALSE, FALSE, 0);

  store = gtk_list_store_new(N_COLUMNS,
			     ST_TYPE_PLUGIN,
			     G_TYPE_BOOLEAN,
			     GDK_TYPE_PIXBUF,
			     G_TYPE_STRING);

  SG_LIST_FOREACH(l, st_plugins_list)
    {
      STPlugin *plugin = l->data;
      GdkPixbuf *pixbuf;
      GdkPixbuf *scaled;
      GtkTreeIter iter;

      pixbuf = st_plugin_get_pixbuf(plugin);
      scaled = pixbuf ? sgtk_pixbuf_scale(pixbuf, GTK_ICON_SIZE_MENU) : NULL;

      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter,
			 COLUMN_PLUGIN, plugin,
			 COLUMN_ENABLED, ! sg_str_slist_contains(st_settings.disabled_plugins, st_plugin_get_filename(plugin)),
			 COLUMN_ICON, scaled,
			 COLUMN_LABEL, st_plugin_get_label(plugin),
			 -1);

      if (scaled)
	g_object_unref(scaled);
    }

  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store),
				       COLUMN_LABEL,
				       GTK_SORT_ASCENDING);

  view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);
  gtk_tree_view_set_search_column(GTK_TREE_VIEW(view), COLUMN_LABEL);

  renderer = gtk_cell_renderer_toggle_new();
  g_signal_connect(renderer, "toggled", G_CALLBACK(st_plugins_preferences_page_toggled_h), store);

  column = gtk_tree_view_column_new_with_attributes(NULL, renderer,
						    "active", COLUMN_ENABLED,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);

  column = gtk_tree_view_column_new();

  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_set_attributes(column, renderer, "pixbuf", COLUMN_ICON, NULL);

  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_set_attributes(column, renderer, "text", COLUMN_LABEL, NULL);

  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  g_object_unref(store);

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled), GTK_SHADOW_IN);

  gtk_container_add(GTK_CONTAINER(scrolled), view);
  gtk_widget_show_all(scrolled);
  gtk_box_pack_start(GTK_BOX(page), scrolled, TRUE, TRUE, 0);
}

static void
st_plugins_preferences_page_toggled_h (GtkCellRendererToggle *renderer,
				       const char *path,
				       gpointer user_data)
{
  GtkTreeModel *model = user_data;
  STPlugin *plugin;
  const char *filename;
  GtkTreeIter iter;
  gboolean status;
  GSList *l;
  gboolean disabled = FALSE;
  
  status = gtk_tree_model_get_iter_from_string(model, &iter, path);
  g_return_if_fail(status == TRUE);

  gtk_tree_model_get(model, &iter, COLUMN_PLUGIN, &plugin, -1);
  filename = st_plugin_get_filename(plugin);

  SG_LIST_FOREACH(l, st_settings.disabled_plugins)
    {
      char *disabled_filename = l->data;

      if (! strcmp(disabled_filename, filename))
	{
	  disabled = TRUE;
	  g_free(disabled_filename);
	  st_settings.disabled_plugins = g_slist_delete_link(st_settings.disabled_plugins, l);
	  break;
	}
    }

  if (! disabled)
    st_settings.disabled_plugins = g_slist_append(st_settings.disabled_plugins, g_strdup(filename));

  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_ENABLED, disabled, -1);
  g_object_unref(plugin);
}

STPreferencesPage *
st_plugins_preferences_page_new (void)
{
  return g_object_new(ST_TYPE_PLUGINS_PREFERENCES_PAGE, NULL);
}
