/* 
 * Sony Programmable I/O Control Device (SPIC) driver for VAIO. 
 * Userspace Control Utility
 *
 * Copyright 2001 Stelian Pop, Alcove
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <getopt.h>
#include <errno.h>
#include <limits.h>
#include <linux/types.h>

#include "spicctrl.h"

static char *version = "version 1.2, Jan 23, 2001";

/* Function prototypes */
int main(int, char *[]);
static void usage(char *);
static int spic_ioctl(int, void *);

/*
 * Main function.
 */
int main(int argc, char *argv[]) {
	int option_index = 0;
	char c, choice = 0;
	__u32 value = ULONG_MAX;
	__u8 value8;
	__u16 value16;
	static struct option long_options[] = {
		{"getacstatus", 0, 0, 'a'},
		{"setbrightness",  1, 0, 'b'},
		{"getbrightness", 0, 0, 'B'},
		{"getbat1capacity", 0, 0, 'c'},
		{"getbat2capacity", 0, 0, 'C'},
		{"setbluetoothpower", 1, 0, 'l'},
		{"getbluetoothpower", 0, 0, 'L'},
		{"powerstatus", 0, 0, 'p'},
		{"getbat1remaining", 0, 0, 'r'},
		{"getbat2remaining", 0, 0, 'R'},
		{"getbat1status", 0, 0, 's'},
		{"getbat2status", 0, 0, 'S'},
		{0, 0, 0, 0}
	};

	while (1) {
		c = getopt_long(argc, argv, "ab:BcCl:LprRsS", 
				long_options, &option_index);
		if (c == -1)
			break;

		switch (c) {
			case 'b':
			case 'l':
				choice = c;
				value = strtoul(optarg, NULL, 0);
				if (value == ULONG_MAX) {
					fprintf(stderr, 
						"bad value for option: %s\n",
						strerror(errno));
					usage(argv[0]);
					return 1;
				}
				break;
			case 'a':
			case 'B':
			case 'c':
			case 'C':
			case 'L':
			case 'p':
			case 'r':
			case 'R':
			case 's':
			case 'S':
				choice = c;
				break;
			default:
				fprintf(stderr, "unknown option %c\n", c);
				break;
		}
	}

	switch (choice) {
		case 'a':
			if (spic_ioctl(SONYPI_IOCGBATFLAGS, &value8) < 0)
				return -1;
			printf("%d\n", (value8 & SONYPI_BFLAGS_AC) != 0);
			break;
		case 'b':
			value8 = (__u8) value;
			return spic_ioctl(SONYPI_IOCSBRT, &value8);
			break;
		case 'B':
			if (spic_ioctl(SONYPI_IOCGBRT, &value8) < 0)
				return -1;
			printf("%d\n", value8);
			break;
		case 'c':
			if (spic_ioctl(SONYPI_IOCGBAT1CAP, &value16) < 0)
				return -1;
			printf("%d\n", value16);
			break;
		case 'C':
			if (spic_ioctl(SONYPI_IOCGBAT2CAP, &value16) < 0)
				return -1;
			printf("%d\n", value16);
			break;
		case 'l':
			value8 = (__u8) value;
			return spic_ioctl(SONYPI_IOCSBLUE, &value8);
			break;
		case 'L':
			if (spic_ioctl(SONYPI_IOCGBLUE, &value8) < 0)
				return -1;
			printf("%d\n", value8);
			break;
		case 'p':
			if (spic_ioctl(SONYPI_IOCGBATFLAGS, &value8) < 0)
				return -1;
			if (value8 & SONYPI_BFLAGS_B1) {
				__u16 cap, rem;
				if (spic_ioctl(SONYPI_IOCGBAT1CAP, &cap) < 0)
					return -1;
				if (spic_ioctl(SONYPI_IOCGBAT1REM, &rem) < 0)
					return -1;
				printf("BAT1: %d/%d %.2f%% ", rem, cap, 100*rem/(double)cap);
			}
			if (value8 & SONYPI_BFLAGS_B2) {
				__u16 cap, rem;
				if (spic_ioctl(SONYPI_IOCGBAT2CAP, &cap) < 0)
					return -1;
				if (spic_ioctl(SONYPI_IOCGBAT2REM, &rem) < 0)
					return -1;
				printf("BAT2: %d/%d %.2f%% ", rem, cap, 100*rem/(double)cap);
			}
			if (value8 & SONYPI_BFLAGS_AC)
				printf("AC");
			printf("\n");
			break;	
		case 'r':
			if (spic_ioctl(SONYPI_IOCGBAT1REM, &value16) < 0)
				return -1;
			printf("%d\n", value16);
			break;
		case 'R':
			if (spic_ioctl(SONYPI_IOCGBAT2REM, &value16) < 0)
				return -1;
			printf("%d\n", value16);
			break;
		case 's':
			if (spic_ioctl(SONYPI_IOCGBATFLAGS, &value8) < 0)
				return -1;
			printf("%d\n", (value8 & SONYPI_BFLAGS_B1) != 0);
			break;
		case 'S':
			if (spic_ioctl(SONYPI_IOCGBATFLAGS, &value8) < 0)
				return -1;
			printf("%d\n", (value8 & SONYPI_BFLAGS_B2) != 0);
			break;
		default:
			usage(argv[0]);
			return 1;
	}

	return 0;
}

/*
 * Prints the program usage.
 *
 *	program: program filename
 */
static void usage(char *program) {

	fprintf(stderr,
		"Sony Vaio SPIC control program %s\n\n"
		"Usage: %s [COMMAND] [OPTION]...\n"
		"\nCommands:\n"
		"\t-a, --getacstatus\t\tget AC adaptor status\n"
		"\t-b, --setbrightness=NUM\t\tset lcd screen brightness\n"
		"\t-B, --getbrightness\t\tget lcd screen brightness\n"
		"\t-c, --getbat1capacity\t\tget first battery capacity\n"
		"\t-C, --getbat2capacity\t\tget second battery capacity\n"
		"\t-l, --setbluetoothpower=NUM\tset bluetooth device power state\n"
		"\t-L, --getbluetoothpower\t\tget bluetooth device power state\n"
		"\t-p, --powerstatus\t\tprint out battery summary\n"
		"\t-r, --getbat1remaining\t\tget first battery remaining capacity\n"
		"\t-R, --getbat2remaining\t\tget second battery remaining capacity\n"
		"\t-s, --getbat1status\t\tget first battery status\n"
		"\t-S, --getbat2status\t\tget second battery status\n",
		version, program);
}

/*
 * Sends a ioctl command to the SPIC driver.
 *
 * Return: 0 if success, -1 if an error occured.
 */
static int spic_ioctl(int ioctlno, void *param) {
	int fd;

	if ( (fd = open("/dev/sonypi", O_RDWR)) == -1) {
		fprintf(stderr, "Failed to open /dev/sonypi: %s\n", 
			strerror(errno));
		return -1;
	}
	if (ioctl(fd, ioctlno, param) < 0) {
		fprintf(stderr, "ioctl failed: %s\n", strerror(errno));
		return -1;
	}
	close(fd);
	return 0;
}

