-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.Walk_Expression_P)
procedure Wf_Named_Argument_Association
  (Node       : in     STree.SyntaxNode;
   Scope      : in     Dictionary.Scopes;
   E_Stack    : in out Exp_Stack.Exp_Stack_Type;
   Heap_Param : in out Lists.List_Heap) is
   Exp_Result, Fun_Info                    : Sem.Exp_Record;
   Fun_Sym, Param_Sym                      : Dictionary.Symbol;
   Ident_Node                              : STree.SyntaxNode;
   Ident_Str                               : LexTokenManager.Lex_String;
   Already_Present, Name_Is_Parameter_Name : Boolean;
   Unused_Value                            : Maths.Value;
   Error_Found                             : Boolean := False;

   --------------------------------------------------------------

   function Find_Identifier (Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in STree.Table;
   --# pre STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.named_argument_association or
   --#   STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.annotation_named_argument_association;
   --# return Return_Node => STree.Syntax_Node_Type (Return_Node, STree.Table) = SP_Symbols.identifier;
   is
      Ident_Node : STree.SyntaxNode;
   begin
      Ident_Node := STree.Child_Node (Current_Node => Node);
      -- ASSUME Ident_Node = annotation_named_argument_association OR annotation_simple_name OR
      --                     named_argument_association OR simple_name
      if STree.Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.simple_name
        or else STree.Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.annotation_simple_name then
         -- ASSUME Ident_Node = annotation_simple_name OR simple_name
         Ident_Node := STree.Child_Node (Current_Node => Ident_Node);
      elsif STree.Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.named_argument_association
        or else STree.Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.annotation_named_argument_association then
         -- ASSUME Ident_Node = named_argument_association OR annotation_named_argument_association
         Ident_Node := STree.Child_Node (Current_Node => STree.Next_Sibling (Current_Node => Ident_Node));
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = annotation_named_argument_association OR annotation_simple_name OR " &
              "named_argument_association OR simple_name in Find_Identifier");
      end if;
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => STree.Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Find_Identifier");
      return Ident_Node;
   end Find_Identifier;

   --------------------------------------------------------------

   function Expression_Location (Node : STree.SyntaxNode) return LexTokenManager.Token_Position
   --# global in STree.Table;
   --# pre STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.named_argument_association or
   --#   STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.annotation_named_argument_association;
   is
      Result : STree.SyntaxNode;
   begin
      Result := STree.Child_Node (Current_Node => Node);
      -- ASSUME Result = annotation_named_argument_association OR annotation_simple_name
      --                 named_argument_association OR simple_name
      if STree.Syntax_Node_Type (Node => Result) = SP_Symbols.simple_name
        or else STree.Syntax_Node_Type (Node => Result) = SP_Symbols.annotation_simple_name then
         -- ASSUME Result = annotation_simple_name OR simple_name
         Result := STree.Next_Sibling (Current_Node => Result);
      elsif STree.Syntax_Node_Type (Node => Result) = SP_Symbols.named_argument_association
        or else STree.Syntax_Node_Type (Node => Result) = SP_Symbols.annotation_named_argument_association then
         -- ASSUME Result = named_argument_association OR annotation_named_argument_association
         Result := STree.Next_Sibling (Current_Node => STree.Next_Sibling (Current_Node => Result));
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Result = annotation_named_argument_association OR annotation_simple_name OR " &
              "named_argument_association OR simple_name in Expression_Location");
      end if;
      -- ASSUME Result = annotation_expression OR expression
      SystemErrors.RT_Assert
        (C       => STree.Syntax_Node_Type (Node => Result) = SP_Symbols.annotation_expression
           or else STree.Syntax_Node_Type (Node => Result) = SP_Symbols.expression,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Result = annotation_expression OR expression in Expression_Location");
      return STree.Node_Position (Node => Result);
   end Expression_Location;

   --------------------------------------------------------------

   procedure Range_Check (A_Range     : in     Boolean;
                          Node        : in     STree.SyntaxNode;
                          Error_Found : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from A_Range,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found                from *,
   --#                                         A_Range;
   --# pre STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.named_argument_association or
   --#   STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.annotation_named_argument_association;
   is
   begin
      if A_Range then
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 341,
            Reference => ErrorHandler.No_Reference,
            Position  => Expression_Location (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end Range_Check;

begin -- Wf_Named_Argument_Association
   Exp_Stack.Pop (Item  => Exp_Result,
                  Stack => E_Stack);
   Exp_Stack.Pop (Item  => Fun_Info,
                  Stack => E_Stack);
   Fun_Sym := Fun_Info.Other_Symbol;

   Find_Named_Argument_Association_Parameter
     (Node                   => Node,
      Subprog_Sym            => Fun_Sym,
      Name_Is_Parameter_Name => Name_Is_Parameter_Name,
      Param_Sym              => Param_Sym);

   Ident_Node := Find_Identifier (Node => Node);
   Ident_Str  := STree.Node_Lex_String (Node => Ident_Node);

   if Name_Is_Parameter_Name then
      -- Seed syntax tree with expected type for run-time check;
      -- but, don't do this for instantiation of unchecked_conversion
      -- because we don't want any RTCs for association of those parameters
      -- (provided the function parameter subtype and actual subtype match)
      if not (Dictionary.IsAnUncheckedConversion (Fun_Sym) and then Exp_Result.Type_Symbol = Dictionary.GetType (Param_Sym)) then
         STree.Add_Node_Symbol (Node => Node,
                                Sym  => Dictionary.GetType (Param_Sym));
      end if;

      -- There is a special case involving functions an stream variables.  We allow a stream
      -- variable to be a parameter to an Unchecked_Conversion but need to ensure that
      -- the function inherits the restrictions associated with referencing a stream
      -- (e.g. cannot be used in gernal expression).  We can do this here by checking
      -- the StreamSymbol of the parameter expression (there will only be one if we are
      -- talking about an unchecked conversion) and if it is non-null then setting the
      -- stream symbol of the function result record (now an object) to the function symbol.
      -- Note that this clause will only be executed for an unchecked conversion because
      -- a parameter which is a stream would hav ebeen rejected at wf_primary in all other
      -- cases
      if Exp_Result.Stream_Symbol /= Dictionary.NullSymbol then
         Fun_Info.Stream_Symbol := Fun_Sym;
      end if;

      Add_Name (Name       => Ident_Str,
                List       => Fun_Info.Param_List,
                Heap_Param => Heap_Param,
                Present    => Already_Present);
      if Already_Present then
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 4,
            Reference => ErrorHandler.No_Reference,
            Position  => STree.Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      else -- not already present so do further checks
         Range_Check (A_Range     => Exp_Result.Is_ARange,
                      Node        => Node,
                      Error_Found => Error_Found);

         -- function is deemed constant if it is predefined and all its parameters
         -- are constant.
         Fun_Info.Is_Constant := Fun_Info.Is_Constant and then Exp_Result.Is_Constant;

         if (Fun_Info.Tagged_Parameter_Symbol = Exp_Result.Type_Symbol
               or else (Fun_Info.Tagged_Parameter_Symbol = Dictionary.NullSymbol
                          and then Dictionary.CompatibleTypes (Scope, Dictionary.GetType (Param_Sym), Exp_Result.Type_Symbol))
               or else (not Dictionary.IsAnExtensionOf (Exp_Result.Type_Symbol, Fun_Info.Tagged_Parameter_Symbol)
                          and then Dictionary.CompatibleTypes (Scope, Dictionary.GetType (Param_Sym), Exp_Result.Type_Symbol))) then
            Tagged_Actual_Must_Be_Object_Check
              (Node_Pos         => Expression_Location (Node => Node),
               Formal_Type      => Dictionary.GetType (Param_Sym),
               Actual_Type      => Exp_Result.Type_Symbol,
               Controlling_Type => Dictionary.GetSubprogramControllingType (Fun_Sym),
               Is_A_Variable    => Exp_Result.Is_AVariable,
               Is_A_Constant    => Exp_Result.Is_Constant,
               Error_Found      => Error_Found);
            -- Following call will deal with scalar value constraint checking
            --# accept Flow, 10, Unused_Value, "Expected ineffective assignment";
            Sem.Constraint_Check
              (Val           => Exp_Result.Value,
               New_Val       => Unused_Value,
               Is_Annotation => STree.Syntax_Node_Type (Node => Node) = SP_Symbols.annotation_named_argument_association,
               Typ           => Dictionary.GetType (Param_Sym),
               Position      => Expression_Location (Node => Node));
            --# end accept;
            -- Check array bounds etc.
            if Dictionary.TypeIsArray (Dictionary.GetType (Param_Sym))
              and then not Dictionary.IsUnconstrainedArrayType (Dictionary.GetType (Param_Sym)) then
               -- Formal is a constrained subtype of an unconstrained array
               if Dictionary.IsUnconstrainedArrayType (Exp_Result.Type_Symbol) then
                  -- Actual is unconstrained.  In SPARK95 or 2005, this is OK if
                  -- the actual is a static String expression, but illegal
                  -- otherwise.
                  if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
                    and then Dictionary.IsPredefinedStringType (Exp_Result.Type_Symbol) then
                     -- Formal must be a constrained String subtype, so we need
                     -- to check the upper bound of the actual against the expected
                     -- upper bound of the formal.
                     if Exp_Result.Range_RHS = Maths.NoValue then
                        -- Actual is not static, so must be illegal
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 39,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Expression_Location (Node => Node),
                           Id_Str    => LexTokenManager.Null_String);
                     else
                        -- Actual is static, so check upper-bound against that expected
                        if Exp_Result.Range_RHS /=
                          Maths.ValueRep
                          (Dictionary.GetScalarAttributeValue
                             (False,
                              LexTokenManager.Last_Token,
                              Dictionary.GetType (Param_Sym))) then
                           ErrorHandler.Semantic_Error
                             (Err_Num   => 418,
                              Reference => ErrorHandler.No_Reference,
                              Position  => Expression_Location (Node => Node),
                              Id_Str    => LexTokenManager.Null_String);

                        end if;
                     end if;
                  else
                     -- SPARK83 or not a String type, so illegal
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 39,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Expression_Location (Node => Node),
                        Id_Str    => LexTokenManager.Null_String);
                  end if;
               elsif Sem.Illegal_Unconstrained
                 (Left_Type  => Exp_Result.Type_Symbol,
                  Right_Type => Dictionary.GetType (Param_Sym)) then
                  -- Although both formal and actual are constrained their bounds don't match
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 418,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Expression_Location (Node => Node),
                     Id_Str    => LexTokenManager.Null_String);
               end if;
            end if;
            -- To help the VCG with generating checks involving unconstrained formal parameters, we
            -- seed the syntax tree with a constraining type mark.  The positional_argument_association
            -- node is already used for RTC purposes, so we seed the expression node instead.
            if STree.Syntax_Node_Type (Node => Node) = SP_Symbols.named_argument_association then
               -- ASSUME Node = named_argument_association
               Sem.Plant_Constraining_Type
                 (Expression_Type => Exp_Result.Type_Symbol,
                  String_Length   => Exp_Result.Range_RHS,
                  Actual_Node     => STree.Expression_From_Named_Argument_Association (Node => Node));
            end if;
         else
            Error_Found := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Expression_Location (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end if;
   else
      Error_Found := True;
      ErrorHandler.Semantic_Error_Lex1_Sym1
        (Err_Num   => 2,
         Reference => ErrorHandler.No_Reference,
         Position  => STree.Node_Position (Node => Ident_Node),
         Id_Str    => Ident_Str,
         Sym       => Fun_Sym,
         Scope     => Scope);
   end if;
   Fun_Info.Errors_In_Expression := Error_Found or else Fun_Info.Errors_In_Expression or else Exp_Result.Errors_In_Expression;
   Exp_Stack.Push (X     => Fun_Info,
                   Stack => E_Stack);
   --# accept Flow, 33, Unused_Value, "Expected to be neither referenced nor exported";
end Wf_Named_Argument_Association;
