-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (ErrorHandler.Conversions.ToString)
procedure IneffectiveStatement
  (Err_Num          : in     Error_Types.NumericError;
   With_Explanation : in     Boolean;
   E_Str            : in out E_Strings.T) is
   procedure IneffectiveStatementExpl (E_Str : in out E_Strings.T)
   --# global in Err_Num;
   --# derives E_Str from *,
   --#                    Err_Num;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an Append_String for the
   --      explanatory text that is delineated by --! comments.

   procedure Append_Explanation
   --# global in     Err_Num;
   --#        in     With_Explanation;
   --#        in out E_Str;
   --# derives E_Str from *,
   --#                    Err_Num,
   --#                    With_Explanation;
   is
      Explanation_String : E_Strings.T := E_Strings.Empty_String;
   begin
      if With_Explanation then
         -- we need to at least look for an explanation
         IneffectiveStatementExpl (E_Str => Explanation_String);
         if E_Strings.Get_Length (E_Str => Explanation_String) > 0 then
            -- there actually is one
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Prefix);
            E_Strings.Append_Examiner_String (E_Str1 => E_Str,
                                              E_Str2 => Explanation_String);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Postfix);
         end if;
      end if;
   end Append_Explanation;

begin

   -- HTML Directives
   --! <NameFormat> <"flow-"><Name>
   --! <ErrorFormat> <"!!! Flow Error : "><Name><" : "><Error>

   if Err_Num.Name1 = Error_Types.NoName then
      --! <Name> 10
      E_Strings.Append_String (E_Str => E_Str,
                               Str   => "Ineffective statement");
      --! Execution of this statement cannot affect the final value of any exported
      --! variable of the subprogram in which it occurs. The cause may be a
      --! data-flow anomaly (i.e. the statement could be an assignment to a
      --! variable, which is always updated again before it is read. However,
      --! statements may be ineffective for other reasons - see Section 4.1 of
      --! Appendix A.
   else
      --! <Name> 10
      E_Strings.Append_String (E_Str => E_Str,
                               Str   => "Assignment to ");
      Append_Name (E_Str => E_Str,
                   Name  => Err_Num.Name1,
                   Scope => Err_Num.Scope);
      E_Strings.Append_String (E_Str => E_Str,
                               Str   => " is ineffective");
      --! This message always relates to a procedure call or an assignment to a
      --! record. The variable XXX may be an actual parameter corresponding to a
      --! formal one that is exported;
      --! otherwise XXX is an exported global variable of the procedure.
      --! The message indicates that
      --! the updating of XXX, as a result of the procedure call, has no effect on
      --! any final values of exported variables of the calling subprogram.
      --! Where the ineffective assignment is expected (e.g. calling a supplied
      --! procedure that returns more parameters than are needed for the immediate purpose),
      --! it can be a useful convention to choose a distinctive name, such as &quot;Unused&quot; for
      --! the actual parameter concerned.  The message &quot;Assignment to Unused is ineffective&quot;
      --! is then self-documenting.
   end if;

   Append_Explanation;
   E_Strings.Append_String (E_Str => E_Str,
                            Str   => ".");
end IneffectiveStatement;
